/*
 * gpn.c - support functions for gpm-Linux
 *
 * Copyright 1993        ajh@gec-mrc.co.uk (Andrew Haylett)
 * Copyright 1994,1995   rubini@ipvvis.unipv.it (Alessandro Rubini)
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 ********/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>        /* strerror(); ?!? memcpy() */
#include <signal.h>
#include <stdarg.h>        /* Log uses it */
#include <errno.h>
#include <unistd.h>        /* getopt(),symlink() */
#include <sys/stat.h>      /* mkdir()  */
#include <sys/param.h>
#include <sys/time.h>      /* timeval */
#include <sys/wait.h>      /* wait() */
#include <sys/types.h>     /* socket() */
#include <sys/socket.h>    /* socket() */
#include <sys/un.h>        /* struct sockaddr_un */

#ifdef	SIGTSTP		/* true if BSD system */
#include <sys/file.h>
#include <sys/ioctl.h>
#endif

#include "gpmInt.h"
#include "gpm.h"        /* this includes socket and internet headers */

extern int	errno;


/*===================================================================*
 *     ==>    the logging utility
 *
 * This procedure is used to log info (mostly debugging ones)
 * the log file is quite large, actually, so be careful....
 *-------------------------------------------------------------------*/
#ifdef CONFIG_GPM_LOG      /* only if enabled */
int Log(char *fmt,...)
{
  va_list args;
  static time_t mytime;
  static char str_time[15];
  static FILE *f;
  static int pid;
  static char temp[80];

  if (!f)
    {
    pid=getpid();
    f=fopen(GPM_NODE_LOG,"a");
    if (!f) f=fopen("/dev/null","w");      /* not null, hopefully */
    setvbuf(f,NULL,_IONBF,0);
    }

  time(&mytime);
  strftime(str_time,15,"%H:%M:%S",localtime(&mytime));
  sprintf(temp,"%s - %i: %s\n",str_time,pid,fmt);

  va_start(args,fmt);           /* but the second argument is not used... */
  vfprintf(f,temp,args);
  return 1;
}
#endif


/*===================================================================*/
int _oops(char *f, int n, char *s, int err)
{
  LOG(("%s(%i): Exiting: \"%s: %s\"",f,n,s,strerror(err) ));
  DEBUG((stderr,"oops() invoked from %s(%i)\n",f,n));
  fprintf(stderr,"%s: %s: %s\n",prgname,s,strerror(err));
  exit(1);
}

/*===================================================================*/
/* octal digit */
static int isodigit(const unsigned char c)
{
  return ((c & ~7) == '0');
}

/* routine to convert digits from octal notation (Andries Brouwer) */
static int getsym(const unsigned char *p0, unsigned char *res)
{
  const unsigned char *p = p0;
  char c;

  c = *p++;
  if (c == '\\' && *p) 
    {
    c = *p++;
    if (isodigit(c)) 
      {
      c -= '0';
      if (isodigit(*p)) c = 8*c + (*p++ - '0');
      if (isodigit(*p)) c = 8*c + (*p++ - '0');
      }
    }
  *res = c;
  return (p - p0);
}

static int loadlut(char *charset)
{
  int i, c, fd;
  unsigned char this, next;
  static unsigned long long_array[9]={
    0x05050505, /* ugly, but preserves alignment */
    0x00000000, /* control chars     */
    0x00000000, /* digits            */
    0x00000000, /* uppercase and '_' */
    0x00000000, /* lowercase         */
    0x00000000, /* Latin-1 control   */
    0x00000000, /* Latin-1 misc      */
    0x00000000, /* Latin-1 uppercase */
    0x00000000  /* Latin-1 lowercase */
    };


#define inwordLut (long_array+1)

  for (i=0; charset[i]; )
    {
    i += getsym(charset+i, &this);
    if (charset[i] == '-' && charset[i + 1] != '\0')
      i += getsym(charset+i+1, &next) + 1;
    else
      next = this;
    for (c = this; c <= next; c++)
      inwordLut[c>>5] |= 1 << (c&0x1F);
    }
  
  if ((fd=open("/dev/console", O_WRONLY)) < 0)
    oops("/dev/console");
  if ((fd<0) || (ioctl(fd, TIOCLINUX, &long_array)<0))
    {
    if (errno==EPERM && getuid())
      fprintf(stderr,"%s: you should probably be root\n",prgname);
    else if (errno==EINVAL)
      fprintf(stderr,"%s: is your kernel compiled with CONFIG_SELECTION on?\n",
	      prgname);
    oops("loadlut");
    }
  close(fd);

  return 0;
}

/*===================================================================*/
static int usage(void)
{
 printf(GPM_NAME " " GPM_RELEASE ", " GPM_DATE "\n"
        "Usage: %s [options]\n",prgname);
 printf("  Valid options are (not all of them are implemented)\n"
 "    -a accel         sets the acceleration (default %d)\n"
 "    -b baud-rate     sets the baud rate (default %d)\n"
 "    -B sequence      allows changing the buttons (default '%s')\n"
 "    -d delta         sets the delta value (default %d)\n"
 "    -D               dirty operation (debug only)\n"
 "    -i interval      max time interval for multiple clicks (default %i)\n"
 "    -k               kill a running gpm, to start X with a busmouse\n"
 "    -l charset       loads the inword() LUT (default '%s')\n"
 "    -L charset       load LUT and exit\n"
 "    -m mouse-device  sets mouse device\n"
 "    -o modem-lines   toggle modem lines (\"dtr\", \"rts\", \"both\")\n"
 "    -p               draw the pointer while striking a selection\n"
 "    -P               do it only with word and line drags\n"
 "    -r number        sets the responsiveness (default %i)\n"
 "    -s sample-rate   sets the sample rate (default %d)\n"
 "    -t mouse-type    sets mouse type (default '%s')\n"
 "                     Microsoft = `ms', Mouse Systems Corp = `msc',\n"
 "                     MM Series = `mm', Logitech = `logi', BusMouse = `bm',\n"
 "                     MSC 3-bytes = `sun', PS/2 = `ps2')\n"
 "                     MouseMan = `mman', oldest 2-button serial = `bare'\n"
 "    -T               test: read mouse, no clients\n"
 "    -v               print version info and exit\n",
        DEF_ACCEL, DEF_BAUD, DEF_SEQUENCE, DEF_DELTA, DEF_TIME, DEF_LUT,
	DEF_SCALE, DEF_SAMPLE, DEF_TYPE);
  return 1;
}

/*===================================================================*/
int cmdline(int argc, char **argv)
{
char options[]="a:b:B:d:Dhi:kl:L:m:o:pPr:s:t:Tv23";
int opt, fd;
static struct {char *in; char *out;} seq[] = {
  {"123","01234567"},
  {"132","02134657"},
  {"213","01452367"},
  {"231","02461357"},
  {"312","04152637"},
  {"321","04261537"},
  {NULL,NULL}
};

  while ((opt = getopt(argc, argv, options)) != -1)
    {
    switch (opt)
      {
      case 'a': opt_accel = atoi(optarg); break;
      case 'b': opt_baud = atoi(optarg); break;
      case 'B': opt_sequence = optarg; break;
      case 'd': opt_delta = atoi(optarg); break;
      case 'D': opt_dirty++; break;
      case 'h': exit(usage());
      case 'i': opt_time=atoi(optarg); break;
      case 'k': opt_kill++; break;
      case 'l': opt_lut=optarg; break;
      case 'L': loadlut(optarg); exit(0);
      case 'm': opt_dev = optarg; break;
      case 'o':
          if (!strcmp(optarg,"dtr"))       opt_toggle=TIOCM_DTR;
          else if (!strcmp(optarg,"dtr"))  opt_toggle=TIOCM_RTS;
          else if (!strcmp(optarg,"both")) opt_toggle=TIOCM_DTR | TIOCM_RTS;
          else exit(usage());
      case 'p': opt_ptrdrag=0; break;
      case 'P': opt_ptrdrag=1; break;
      case 'r':
	  /* being called responsiveness, I must take the inverse */
	  opt_scale=atoi(optarg);
          if (!opt_scale) opt_scale=100; /* a maximum */
	  else opt_scale=100/opt_scale;
	  break;
      case 's': opt_sample = atoi(optarg); break;
      case 't': opt_type=optarg; break;
      case 'T': opt_test++; break;
      case 'v': printf(GPM_NAME " " GPM_RELEASE ", " GPM_DATE "\n"); exit(0);
      case '2': opt_three=-1; break;
      case '3': opt_three=1; break;
      default:
          exit(usage());
      }
    }

  if (opt_accel<2) exit(usage());
  if (opt_delta<2) exit(usage());
  if (strlen(opt_sequence)!=3 || atoi(opt_sequence)<100) exit(usage());

  /* look for the type */
  for (m_type=mice; m_type->fun; m_type++)
    if (!strcmp(opt_type,m_type->name))
      break;
  if (!(m_type->fun)) /* not found */
    exit(usage());


  /* check for uniqueness */
  {
  struct sockaddr_un ctladdr;
  int ctlfd, pid, len; FILE *f;
  
  bzero((char *)&ctladdr,sizeof(ctladdr));
  ctladdr.sun_family=AF_UNIX;
  strcpy(ctladdr.sun_path, GPM_NODE_CTL);
  len=sizeof(ctladdr.sun_family)+strlen(GPM_NODE_CTL);

  if ( (ctlfd=socket(AF_UNIX,SOCK_STREAM,0))<0 )
	oops("socket()");

  if ( connect(ctlfd,(struct sockaddr *)(&ctladdr),len)>=0 )
	{
	/* another gpm is runnin, get its pid */
	f=fopen(GPM_NODE_PID,"r");
	if (f && fscanf(f,"%i",&pid)==1)
	  {
	  if (opt_dirty || opt_kill) 
	    {
	    if (kill(pid,SIGUSR1)==-1) oops("kill()");
	    if (opt_kill) exit(0);
	    }
	  else
	    {
	    fprintf(stderr,"gpm-Linux is already running as pid %i\n",pid);
	    exit(1);
	    }
	  }
	else
	  {
	  fprintf(stderr,"%s: fatal error (should't happen)\n",prgname);
	  exit(1);
	  }
	}

  close(ctlfd);

#ifndef DEBUGGING

  /* go to background and become a session leader (Stefan Giessler) */
  switch(fork())
    {
    case -1: oops("fork()"); /* error */
    case  0: break;          /* child */
    default: exit(0);        /* parent */
    }

  if (!freopen("/dev/console","w",stderr)) /* the currently current console */
    {
    printf("gpm: freopen(stderr) failed\n"); exit(1);
    }
  if (setsid()<0) oops("setsid()");

#endif

  /* open the device */
  if (opt_dev)
    {
    if (!strcmp(opt_dev,"-"))
      fd=0;
    else if ((fd=open(opt_dev,O_RDWR))<0)
      oops(opt_dev); /* user failed */
    }
  else /* use "/dev/mouse" */
    {
    opt_dev = "/dev/mouse";
    if ((fd=open(opt_dev,O_RDWR))<0)
      oops("/dev/mouse");
    }

  /* chdir */
  if (chdir(GPM_NODE_DIR) && mkdir(GPM_NODE_DIR,GPM_NODE_DIR_MODE))
    oops(GPM_NODE_DIR);
  if (chdir(GPM_NODE_DIR))
    oops(GPM_NODE_DIR);      /* well, I must create my directory first */

  /* now sign */
  f=fopen(GPM_NODE_PID,"w");
  if (!f)
    {
    if (getuid()) fprintf(stderr,"%s: you're not root, can you write to %s?\n",
			  prgname,GPM_NODE_DIR);
    oops(GPM_NODE_PID);
    }
  fprintf(f,"%i\n",getpid());
  fclose(f);
  LOG(("Signed"));
  }

  /*
   * well, now create a symlink in the /tmp dir to be compliant with old
   * executables
   */
  symlink(GPM_NODE_CTL,"/tmp/gpmctl");

  /* init the device, and change mouse type */
  if (m_type->init)
    m_type=(m_type->init)(fd, m_type->flags, m_type);

  if (opt_toggle)
    {
    unsigned int modem_lines;

    ioctl(fd, TIOCMGET, &modem_lines);
    modem_lines &= ~opt_toggle;
    ioctl(fd, TIOCMSET, &modem_lines);
    }
 
  loadlut(opt_lut);

  /* choose the sequence */
  for (opt=0; seq[opt].in && strcmp(seq[opt].in,opt_sequence); opt++)
    ;
  if (!seq[opt].in) exit(usage());
  opt_sequence=seq[opt].out;

  return fd;
}
  

/*
 * This is because Linus uses 4-wide tabstops,
 * forcing me to use the same default to manage kernel sources
 */

/* Local Variables: */
/* tab-width:8      */
/* End:             */


