/* Dialog managing.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>	/* For free() */
#include <stdarg.h>
#include <sys/types.h>
#include <string.h>
#include "mad.h"
#include "global.h"
#include "dialog.h"
#include "input.h"
#include "color.h"
#include "util.h"
#include "win.h"
#include "mouse.h"
#include "main.h"
#include "key.h"	/* For mi_getch() */
#include "dlg.h"	/* draw_box, yes I know, it's silly */

static char rcsid [] = "$Id: dialog.c,v 1.14 1995/01/27 02:34:31 miguel Exp $";
static char buffer [4096];

static Dialog *current_dialog = 0;
WINDOW *top_window;

Refresh *refresh_list = 0;

void push_refresh (void (*new_refresh)(void *), void *parameter, int flags)
{
    Refresh *new;

    new = xmalloc (sizeof (Refresh), "push_refresh");
    new->next = (struct Refresh *) refresh_list;
    new->refresh_fn = new_refresh;
    new->parameter = parameter;
    new->flags     = flags;
    refresh_list = new;
}

void pop_refresh (void)
{
    Refresh *old;
    
    if (!refresh_list)
	fprintf (stderr, "\n\n\nrefresh stack underflow!\n\n\n");
    else {
	old = refresh_list;
	refresh_list = refresh_list->next;
	free (old);
    }
}

static void do_complete_refresh (Refresh *refresh_list)
{
    if (!refresh_list)
	return;

    if (refresh_list->flags != REFRESH_COVERS_ALL)
	do_complete_refresh (refresh_list->next);
    
    (*(refresh_list->refresh_fn))(refresh_list->parameter);
}

void do_refresh (void)
{
    if (!refresh_list)
	fprintf (stderr, "\n\nNo refresh handlers defined!\n\n");
    else {
	if (fast_refresh)
	    (*(refresh_list->refresh_fn))(refresh_list->parameter);
	else {
	    do_complete_refresh (refresh_list);
	}
    }
}

/* Poor man's window puts, it doesn't handle auto-wrap */
void my_wputs (WINDOW *w, int y, int x, char *text)
{
    char p;

    wmove (w, y, x);
    while ((p = *text++) != 0){
	if (p == '\n')
	    wmove (w, ++y, x);
	else 
	    waddch (w, p);
    }
}

void create_dialog (int cols, int rows, char *header, char *text, int error)
{
    int base = LINES/3;
    int xpos, ypos;
    Dialog *newd;
    
    if (rows+1 > LINES/3+2)
	base = LINES/3+2;

    newd = (Dialog *) xmalloc (sizeof (Dialog), "create_dialog");
    newd->backpointer = current_dialog;
    current_dialog = newd;

    xpos = COLS/2-(cols+5)/2;
    ypos = base-(rows+1)/2;

    /* Setup window */
    newd->d = newwin (rows+4, cols+4, ypos, xpos);
    leaveok (newd->d, TRUE);
    if ((error & D_ERROR))
	wattron (newd->d, ERROR_COLOR | A_BOLD);
    else
	wattron (newd->d, REVERSE_COLOR);
    wclr (newd->d);
    
    /* Mouse events */
    push_event (1, 1, COLS, LINES, (mouse_h) null_event, 0, event_use_frame);
    push_frame (xpos, ypos, 0);

    /* Draw the contents */
    draw_box (newd->d, 1, 1, rows+2, cols+2);
    mvwprintw (newd->d, 1, 1+(cols-strlen (header)+2)/2, header);
    my_wputs (newd->d, 2, 2, text); 
    touchwin (newd->d);
    
    wrefresh (newd->d);

    /* Export the top_window */
    top_window = newd->d;
}

WINDOW *get_top_text (void)
{
    return current_dialog->d;
}

int run_dialog (void)
{
    int key;

    push_event (1, 1, COLS, LINES, click_may_break_loop,
		quit_event, event_use_frame);
    key = mi_getch ();
    pop_event ();
    do_refresh ();
    return key;
}

void destroy_dialog (void)
{
    Dialog *p;
    
    if (!current_dialog){
	fprintf (stderr, "Trying to destroy non-existant dialog\n");
	exit (1);
    }
    delwin (current_dialog->d);
    p = current_dialog;
    current_dialog = p->backpointer;
    if (current_dialog)
	top_window = current_dialog->d;
    else
	top_window = 0;
    free (p);
    pop_frame ();
    pop_event ();
}

int message (int error, char *header, char *text,  ...)
{
    va_list  args;
    int      len, lines;
    char     *buf;
    int      v;

    buf = buffer;

    strcpy (buffer, "\n");
    va_start (args, text);
    vsprintf (&buffer [1], text, args);
    strcat (buf, "\n");
    va_end (args);
    len = max (strlen (header), msglen (buf, &lines));
    create_dialog (len, lines, header, buf, error);
    if (!(error & D_INSERT)){
	v = run_dialog ();
	destroy_dialog ();
	return v;
    }
    return 0; 
}

int error (int status, int errnum, char *text,  ...)
{
    va_list  args;
    int      len, lines;
    char     *buf;
    int      v;
    char     *header = " Error ";
    
    buf = buffer;

    strcpy (buffer, "\n");
    va_start (args, text);
    vsprintf (&buffer [1], text, args);
    strcat (buf, "\n");
    
    len = max (strlen (header), msglen (buf, &lines));
    create_dialog (len, lines, header, buf, 1);
    v = run_dialog ();
    destroy_dialog ();
    va_end (args);
    return 0;
}

char *input_dialog (char *header, char *text, char *def_text)
{
    int    len, lines;
    int    abort = 0;
    Input  *in;
    int    c;
    char   *result;

    if (strlen (text) >= sizeof (buffer))
	text [sizeof (buffer)-1] = 0;

    strcpy (buffer, text);
    
    len = max (strlen (header), msglen (buffer, &lines));
    len = max (len, 60);
    create_dialog (len, lines+1, header, buffer, 0);
    in = create_input (3, lines+2, current_dialog->d, INPUT_COLOR, 58,
		       def_text);
    while ((c = mi_getch ()) != '\n'){
	if (is_abort_char (c)){
	    abort = 1;
	    break;
	}
	handle_char (in, c);
    }
    destroy_dialog ();
    do_refresh ();
    result = in->buffer;
    destroy_input (in, IN_KEEP_BUFFER);
    if (abort){
	free (result);
	return 0;
    } else
	return result;
}

char *input_expand_dialog (char *header, char *text, char *def_text)
{
    char *result;
    char *expanded;

    result = input_dialog (header, text, def_text);
    if (result){
	expanded = tilde_expand (result);
	free (result);
	return expanded;
    }
    return result;
}
