#include <OI/oi.H>
#include <string.h>


/*
 *	callbacks - member functions as callback functions.
 *
 *	This program demonstrates setting an entry check callback on an entry field.
 *	The program demonstrates how to set up member functions as callback functions.
 *	The program simply creates multiple entry fields and lets the user enter names
 *	of fruits and vegetables.
 *
 *	The reader should refer to the OI documentation for
 *	information on the following member functions.
 *		- OIIntro
 *			OI_callback, base class.
 *			OI_init()
 *			OI_begin_interaction()
 *			OI_fini()
 *		- OI_d_tech
 *			layout_associated_object()
 *			set_associated_object()
 *			set_layout()
 *		- OI_app_window
 *			oi_create_app_window()
 *		- OI_entry_field
 *			oi_create_entry_field()
 *			set_entry_chk()
 *				OI_EF_ENTRY_CHK_OK
 *				OI_EF_ENTRY_CHK_BAD
 *
 */



/*
 *	define the FOOD class
 *
 *	class OI_callback is the ultimate base class of OI_d_tech, and must also
 *	be the ultimate base class for any user defined classes which are to have
 *	their member functions used as callback routines.
 *
 */
class FOOD: public OI_callback {
public:
	FOOD();
	OI_ef_entry_chk_status	validate(OI_entry_field*, void*);
virtual	OI_ef_entry_chk_status	valid(char*) = 0;
};
FOOD::FOOD() { }	/* out-of-line'd for the compiler */


/*
 *	FRUIT class definition,
 *		derived from FOOD class.
 *
 *	contains public member function for entry_check validation.
 */
class FRUIT: public FOOD {
public:
	OI_ef_entry_chk_status	valid(char*);
};


/*
 *	VEGETABLE class definition,
 *		derived from FOOD class.
 *
 *	contains public member function for entry_check validation.
 */
class VEGETABLE: public FOOD {
public:
	OI_ef_entry_chk_status	valid(char*);
};


/*
 *	FOOD::validate
 *
 *	validate the food name that was just entered.
 *	if text was entered, then ok = valid(text)
 */
OI_ef_entry_chk_status
FOOD::validate(OI_entry_field *efp, void*)
{
	OI_ef_entry_chk_status	ok;
	char			*cp;

	ok = OI_EF_ENTRY_CHK_BAD;
	if (cp = efp->part_text())
		ok = valid(cp);
	return (ok);
}
	

/*
 *	FRUIT:valid
 *
 *	Check entered string against list of valid fruits.
 */
OI_ef_entry_chk_status
FRUIT::valid(char *strp)
{
	static	char		*valid_str[] = { "apple", "orange", "pear", "peach" };
	const	int		N_STR = (sizeof(valid_str)/sizeof(char *));
	int			i;
	OI_ef_entry_chk_status	ok;

	ok = OI_EF_ENTRY_CHK_BAD;
	for (i = 0; i < N_STR; i++) {
		if (! strcmp(strp, valid_str[i])) {
			ok = OI_EF_ENTRY_CHK_OK;
			break;
		}
	}
	return (ok);
}


/*
 *	VEGETABLE:valid
 *
 *	Check entered string against list of valid vegetables.
 */
OI_ef_entry_chk_status
VEGETABLE::valid(char *strp)
{
	static	char		*valid_str[] = { "pea", "bean", "corn", "beet" };
	const	int		N_STR = (sizeof(valid_str)/sizeof(char *));
	int			i;
	OI_ef_entry_chk_status	ok;

	ok = OI_EF_ENTRY_CHK_BAD;
	for (i = 0; i < N_STR; i++) {
		if (! strcmp(strp, valid_str[i])) {
			ok = OI_EF_ENTRY_CHK_OK;
			break;
		}
	}
	return (ok);
}


void
main (int argc, char **argv)
{
	OI_connection	*conp;				/* the connection to the server */
	OI_app_window	*appWindow;			/* the enclosing app window */
	OI_entry_field	*entryField;			/* the entry field */
	FOOD		*fp;

	/*
	 *	Open a connection.
	 */
	if ((conp = OI_init(&argc, argv, "callback", "Callback"))) {

		/*
		 *	Create the application window.
		 *	Make it row layout.
		 */
		appWindow = oi_create_app_window("myapp", 10, 10, "C++ callback");
		appWindow->set_layout(OI_layout_row);

		/*
		 *	Create a Fruit object.
		 *	Create an entry field into which the user can type fruit names.
		 *	Set the entry check for this entry field to be the
		 *		C++ `fruit' objects `validate' member function.
		 *	Layout the entry field into the application window.
		 */
		fp = new FRUIT();
		entryField = oi_create_entry_field("fruit", 8, "    Fruit: ", NULL, 8);
		entryField->set_entry_chk(fp, (OI_ef_entry_chk_memfnp)&FOOD::validate);
		entryField->layout_associated_object(appWindow, 0, 1, OI_ACTIVE);

		/*
		 *	Create a Vegetable object.
		 *	Create an entry field into which the user can type vegetable names.
		 *	Set the entry check for this entry field to be the
		 *		C++ `vegetable' objects `validate' member function.
		 *	Layout the entry field into the application window.
		 */
		fp = new VEGETABLE();
		entryField = oi_create_entry_field("fruit", 8, "Vegetable: ", NULL, 8);
		entryField->set_entry_chk(fp, (OI_ef_entry_chk_memfnp)&FOOD::validate);
		entryField->layout_associated_object(appWindow, 0, 2, OI_ACTIVE);

		/*
		 *	Display main window.
		 *	Begin interaction.
		 */
		appWindow->set_associated_object(conp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 *	Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}
