#ifndef ComBas_H
#define ComBas_H

/*
 ******************************************************************************
 *
 *	ParcPlace Systems, Inc.
 *	Copyright (c) 1990, ParcPlace Systems, Inc. USA   All rights reserved.
 *
 ****************************************************************************
 *
 *	File:	ComBas.H
 *
 *	Description:
 *		Header for the following classes:
 *			CommonBase
 *			CommonBaseWithName
 *			CommonBaseWithId
 *
 *		All private member functions are prefixed with cb_
 *
 *	CommonBase:
 *
 *		CommonBase's sole purpose is to provide an ancestor class for
 *		user defined classes such that arbitrary user defined objects
 *		can have their member functions used as callbacks.
 *
 *		This requires that the virtual tabble be allocated in a known place,
 *		so CommonBase defines at least one virtual function.
 *
 *		Two virtuals are defined; a virtual destructor, and one to ask if the
 *		object is of a particular class, where class is identified by a unique id
 *		which is assigned when the class is initialized.
 *
 *		A class is initialized by calling the CommonBase static member function
 *		CommonBase::unique_id, which returns a unique integer which may be
 *		used to identify the class.  The class should save this in a private
 *		static member, and override the virtual are_you_a as follows:
 *
 *		class	Derived : public CommonBase {
 *		 public:
 *			int		Derived::are_you_a(unsigned short i)	{ return(i == id); }
 *		 private:
 *		static	unsigned short	id ;		// unique id for Derived class
 *		}
 *
 *		The virtual are_you_a is not strictly required to be overridden;
 *		CommonBase provides a default which always returns 0.
 *
 *		CommonBase::unique_id should be called only once for a class directly
 *		derived from CommonBase or CommonBaseWithName, and no further derived
 *		classes should call unique_id.
 *
 *	CommonBaseWithName:
 *
 *		CommonBaseWithName is a virtual base class designed for those classes
 *		which desire some means of naming objects.
 *
 *		It assumes an object name may be converted by some mechanism into a CommonBaseNameId
 *		(long integer).  The CommonBaseNameId need not be unique.  If the class desires
 *		some means of locating objects based on name, then the class is responsible
 *		for implementing a sufficiently unique name to CommonBaseNameId converter and
 *		associated lookup mechanism.
 *
 *		CommonBaseWithName does not actually store the CommonBaseNameId for an instance,
 *		since some classes of objects may be able to generate a name and CommonBaseNameId from other
 *		information.
 *
 *	CommonBaseWithId:
 *
 *		CommonBaseWithId is a base class designed for those classes which wish to
 *		store the CommonBaseNameId in the object instance.
 *
 *	RCSid = "$Id: ComBas.H,v 4.1.1.1 1993/06/01 22:52:01 mth Exp $"
 *
 ****************************************************************************
 */

class	CommonBase;
class	CommonBaseWithName;
class	CommonBaseWithId;

class	CommonBase	{							// Common Base class for all objects
	static		unsigned short		_cb_id_counter;
 public:
						CommonBase();
	virtual					~CommonBase();
	static		unsigned short		unique_id()		{ return (++_cb_id_counter); }
	virtual		int			are_you_a(unsigned short);
} ;

typedef	unsigned long	CommonBaseNameId ;

class	CommonBaseWithName : public CommonBase 	{				// Objects with a Name
 public:
						CommonBaseWithName();
	virtual					~CommonBaseWithName();
	virtual		CommonBaseNameId	id() = 0;			// Get long int corresponding to object name
	virtual	const	char*			name() = 0;			// Get object name
	// Note!
	// Deriving a subclass requires implementing the virtuals id() and name()
} ;

class	CommonBaseWithId : public CommonBaseWithName 	{			// Named objects with id stored internally
			CommonBaseNameId	_cb_id;				// short, efficient form for object name
 public:
						CommonBaseWithId(CommonBaseNameId i)	{ _cb_id = i; }
	virtual					~CommonBaseWithId();
	virtual		CommonBaseNameId	id() ;				// Get long int corresponding to object name
			void			set_id(CommonBaseNameId i)		{ _cb_id = i; }
	// Note!
	// Deriving a subclass requires implementing the virtual name()
} ;

#endif /* ComBas_H */
