# Pixel-Graphik fr CLISP
# Bruno Haible 2.9.1993

#include "lispbibl.c"
#include "arilev0.c" # wegen dreisatz


#ifdef GRAPHICS


# Low-Level-Funktionen der Pixel-Grafik:
# gr_init(width,height,colors);  erzeugt einen Grafik-Bildschirm mit
#                                (andeutungsweise) gegebener Gre.
# gr_show();                     zeigt den Grafik-Bildschirm an.
# gr_colors()                    liefert einen Array von benannten Farben.
# gr_clear(color);               lscht den gesamten Grafik-Bildschirm.
# gr_xdim                        Breite des Grafik-Bildschirms in Pixeln.
# gr_ydim                        Hhe des Grafik-Bildschirms in Pixeln.
# gr_get(x,y)                    liefert die Farbe des Punktes (x,y)
# gr_dot(color,x,y);             zeichnet einen Punkt (x,y)
# gr_box(color,x1,y1,x2,y2);     zeichnet ein Rechteck [x1,x2] x [y1,y2]
# gr_line(color,x1,y1,x2,y2);    zeichnet eine Linie von (x1,y1) bis (x2,y2)

struct named_color
{
    int color;
    char* name;
};


# Hilfsfunktionen frs Clipping:

# dot_clipping(&clip,&args)   modifiziert die Argumente passend fr gr_dot
# box_clipping(&clip,&args)   modifiziert die Argumente passend fr gr_box
# line_clipping(&clip,&args)  modifiziert die Argumente passend fr gr_line
# Rckgabewert ist jeweils 0, wenn berhaupt nichts zu zeichnen ist, sonst 1.

struct clip
{
    sintL x1, x2;  # Bereich x1 <= x <= x2 (x1,x2 >= 0)
    sintL y1, y2;  # Bereich y1 <= y <= y2 (y1,y2 >= 0)
    # Bei x1 > x2 oder y1 > y2 ist der Bereich leer.
};

struct dot_args
{
    sintL x, y;    # Pixel (x,y)
};

struct box_args
{
    sintL xa, ya, xb, yb; # alle Pixel im Rechteck mit Ecken (xa,ya) und (xb,yb)
};

struct line_args
{
    sintL xa, ya, xb, yb; # Linie von (xa,ya) bis (xb,yb)
};

int dot_clipping (const struct clip * clip, const struct dot_args * args)
{
    return    (clip->x1 <= args->x) && (args->x <= clip->x2)
           && (clip->y1 <= args->y) && (args->y <= clip->y2);
}

int box_clipping (const struct clip * clip, struct box_args * args)
{
    sintL xa, ya, xb, yb;
    # Clipping in x-Richtung:
    if (args->xa <= args->xb)
      { xa = args->xa; xb = args->xb; }
      else
      { xa = args->xb; xb = args->xa; }
    if (xa < clip->x1)
      xa = clip->x1;
    if (xb > clip->x2)
      xb = clip->x2;
    if (xa > xb)
      return 0;
    args->xa = xa; args->xb = xb;
    # Clipping in y-Richtung:
    if (args->ya <= args->yb)
      { ya = args->ya; yb = args->yb; }
      else
      { ya = args->yb; yb = args->ya; }
    if (ya < clip->y1)
      ya = clip->y1;
    if (yb > clip->y2)
      yb = clip->y2;
    if (ya > yb)
      return 0;
    args->ya = ya; args->yb = yb;
    # Fertig.
    return 1;
}

# Liefert zu a,b>=0, c>0 (alle <2^31):  round(a * b / c) = floor(2*a*b+c,2*c)
static sintL dreisatz (uintL a, uintL b, uintL c)
{
    uintL hi, lo;
    mulu32(a,2*b, hi=,lo=); # 2^32*hi + lo = 2*a*b
    if ((lo += c) < c) { hi += 1; } # 2^32*hi + lo = 2*a*b + c
    if (hi < c) # avoid overflow
      { uintL q;
        divu_6432_3232(hi,lo,2*c, q=,);
        if ((sintL)q >= 0)
          return q;
      }
    return bit(31)-1;
}

int line_clipping (const struct clip * clip, struct line_args * args)
{
    sintL xa, ya, xb, yb;
    sintL x1 = clip->x1;
    sintL x2 = clip->x2;
    sintL y1 = clip->y1;
    sintL y2 = clip->y2;
    # Check whether x1 <= x2 and y1 <= y2.
    if ((x1 > x2) || (y1 > y2))
      return 0;
    # Ensure xa <= xb.
    if (args->xa <= args->xb)
      { xa = args->xa; ya = args->ya; xb = args->xb; yb = args->yb; }
      else
      { xa = args->xb; ya = args->yb; xb = args->xa; yb = args->ya; }
    # Line entirely outside the window, in the same sector?
    if (   (xa > x2) # implies xb > x2 too
        || (xb < x1) # implies xa < x1 too
        || (ya < y1 && yb < y1)
        || (ya > y2 && yb > y2)
       )
      return 0;
    # Line entirely inside the window?
    if (xa >= x1 && xb <= x2 && ya >= y1 && ya <= y2 && yb >= y1 && yb <= y2)
      goto ok;
    # Special case horizontal line
    if (ya == yb)
      { if (xa < x1) xa = x1;
        if (xb > x2) xb = x2;
        goto ok;
      }
    # Special case vertical line
    if (xa == xb)
      { if (ya < y1) ya = y1;
        elif (ya > y2) ya = y2;
        if (yb < y1) yb = y1;
        elif (yb > y2) yb = y2;
        goto ok;
      }
    # Move left starting point into the clip rectangle.
    if (xa < x1 || ya < y1 || ya > y2)
      { # try to intersect line with left border
        sintL y;
        if (ya < yb)
          { if ((xa < x1)
                && ((y = ya + dreisatz(x1-xa, yb-ya, xb-xa)) >= y1)
               )
              { if (y > y2)
                  return 0;
                xa = x1; ya = y;
              }
            elif (ya < y1)
              { xa = xa + dreisatz(y1-ya, xb-xa, yb-ya); ya = y1; }
          }
          else # ya > yb
          { if ((xa < x1)
                && ((y = ya - dreisatz(x1-xa, ya-yb, xb-xa)) <= y2)
               )
              { if (y < y1)
                  return 0;
                xa = x1; ya = y;
              }
            elif (ya > y2)
              { xa = xa + dreisatz(ya-y2, xb-xa, ya-yb); ya = y2; }
          }
      }
    # Move right starting point into the clip rectangle.
    if (xb > x2 || yb < y1 || yb > y2)
      { # try to intersect line with right border
        sintL y;
        if (ya < yb)
          { if ((xb > x2)
                && ((y = yb - dreisatz(xb-x2, yb-ya, xb-xa)) <= y2)
               )
              { if (y < y1)
                  return 0;
                xb = x2; yb = y;
              }
            elif (yb > y2)
              { xb = xb - dreisatz(yb-y2, xb-xa, yb-ya); yb = y2; }
          }
          else # ya > yb
          { if ((xb > x2)
                && ((y = yb + dreisatz(xb-x2, ya-yb, xb-xa)) >= y1)
               )
              { if (y > y2)
                  return 0;
                xb = x2; yb = y;
              }
            elif (yb < y1)
              { xb = xb - dreisatz(y1-yb, xb-xa, ya-yb); yb = y1; }
          }
      }
    ok:
    args->xa = xa; args->ya = ya; args->xb = xb; args->yb = yb;
    return 1;
}


#ifdef GRAPHICS_SWITCH

# Ein Text-Bildschirm und ein Grafik-Bildschirm knnen sichtbar sein, jedoch
# nicht gleichzeitig.
# switch_text_mode(); wechselt auf den Text-Bildschirm.
# switch_graphics_mode(); wechselt auf den Grafik-Bildschirm.
extern void switch_text_mode (void);
extern void switch_graphics_mode (void);


#ifdef EMUNIX_PORTABEL


#include <graph.h>
extern unsigned char *_g_mem;

# Only two modes: G_MODE_OFF (text) and G_MODE_VGA_L (320x200x256).

#define GRAPH_SIZE  (320*200)

int cur_mode = G_MODE_OFF;   /* current video mode       */
int flip_mode = G_MODE_OFF;  /* flipped video mode       */
unsigned char * graph_buf;   /* saves graphics data during flip */
static struct clip cur_clip; /* current whole-screen clipping */

static int initialized = 0;

static int initialize (void)
{
    if ((graph_buf = malloc(GRAPH_SIZE)) == NULL)
        { errno = ENOMEM; return -1; }
    initialized = 1;
    return 0;
}

void switch_text_mode (void)
{
    if (cur_mode == G_MODE_OFF)
        return;
    /* cur_mode != G_MODE_OFF, so flip_mode == G_MODE_OFF,
       and initialize() has already been called. */
    memcpy(graph_buf,_g_mem,GRAPH_SIZE);
    flip_mode = cur_mode;
    g_mode(G_MODE_OFF); cur_mode = G_MODE_OFF;
}

void switch_graphics_mode (void)
{
    if (flip_mode == G_MODE_OFF)
        return;
    /* flip_mode != G_MODE_OFF, so cur_mode == G_MODE_OFF,
       and initialize() has already been called. */
    g_mode(flip_mode); cur_mode = flip_mode;
    cur_clip.x1 = 0; cur_clip.x2 = g_xsize-1;
    cur_clip.y1 = 0; cur_clip.y2 = g_ysize-1;
    memcpy(_g_mem,graph_buf,GRAPH_SIZE);
    flip_mode = G_MODE_OFF;
}

int gr_init (sintL width, sintL height, sintL colors)
{
    if (!initialized)
        if (initialize())
            { OS_error(); }
    if (!g_mode(G_MODE_VGA_L))
        return -1;
    flip_mode = G_MODE_OFF; cur_mode = G_MODE_VGA_L;
    cur_clip.x1 = 0; cur_clip.x2 = g_xsize-1;
    cur_clip.y1 = 0; cur_clip.y2 = g_ysize-1;
    return 0;
}

void gr_show (void)
{
    if (cur_mode == G_MODE_OFF)
      { switch_graphics_mode();
        if (cur_mode == G_MODE_OFF)
          { fehler(
                   DEUTSCH ? "Grafik nicht initialisiert." :
                   ENGLISH ? "graphics not initialized" :
                   ""
                  );
      }   }
}

static struct named_color EGA_colors []
  = { { G_BLACK,                 "BLACK"         },
      { G_BLUE,                  "BLUE"          },
      { G_GREEN,                 "GREEN"         },
      { G_CYAN,                  "CYAN"          },
      { G_RED,                   "RED"           },
      { G_MAGENTA,               "MAGENTA"       },
      { G_BROWN,                 "BROWN"         },
      { G_WHITE,                 "LIGHT-GRAY"    },
      { G_INTENSITY | G_BLACK,   "DARK-GRAY"     },
      { G_INTENSITY | G_BLUE,    "LIGHT-BLUE"    },
      { G_INTENSITY | G_GREEN,   "LIGHT-GREEN"   },
      { G_INTENSITY | G_CYAN,    "LIGHT-CYAN"    },
      { G_INTENSITY | G_RED,     "LIGHT-RED"     },
      { G_INTENSITY | G_MAGENTA, "LIGHT-MAGENTA" },
      { G_INTENSITY | G_YELLOW,  "YELLOW"        },
      { G_INTENSITY | G_WHITE,   "WHITE"         },
      { 0,                       NULL            }
    };

struct named_color * gr_colors (void)
{
    return &!EGA_colors;
}

#define gr_xdim  g_xsize
#define gr_ydim  g_ysize

int gr_get (sintL x, sintL y)
{
    int color;
    gr_show();
    color = g_get(x,y);
    if (color < 0)
        color = G_BLACK;
    return color;
}

void gr_dot (sintL color, sintL x, sintL y)
{
    gr_show();
    if (color >= 0 && color < g_colors)
        g_set(x,y,color);
}

void gr_box (sintL color, sintL x1, sintL y1, sintL x2, sintL y2)
{
    gr_show();
    if (color >= 0 && color < g_colors)
        g_box(x1,y1,x2,y2,color,G_FILL);
}

void gr_clear (sintL color)
{
    gr_show();
    if (color >= 0 && color < g_colors)
        g_clear(color);
}

void gr_line (sintL color, sintL x1, sintL y1, sintL x2, sintL y2)
{
    # Clipping selber durchfhren; dem EMX-Clipping trauen wir nicht.
    struct clip clip;
    struct line_args args;
    gr_show();
    args.xa = x1; args.ya = y1; args.xb = x2; args.yb = y2;
    if (color >= 0 && color < g_colors)
      if (line_clipping(&cur_clip,&args))
        g_line(args.xa,args.ya,args.xb,args.yb,color);
}


#else # (EMUNIX && !EMUNIX_PORTABEL) || UNIX_LINUX


#ifdef UNIX

# Um Zugriff auf I/O-Ports und den Bildschirmspeicher zu haben, mu das
# Programm mit "setuid root"-Privileg installiert werden.

global uid_t root_uid; # wird von SPVW initialisiert

#define BEGIN_ROOT_PRIV  \
  set_break_sem_1(); \
  setreuid(user_uid,root_uid);

#define END_ROOT_PRIV  \
  setreuid(root_uid,user_uid); \
  clr_break_sem_1();

#endif


# =========================== vgalib/vga.h =================================== #

/* VGAlib version 1.2 - (c) 1993 Tommy Frandsen                    */
/*                                                                 */
/* This library is free software; you can redistribute it and/or   */
/* modify it without any restrictions. This library is distributed */
/* in the hope that it will be useful, but without any warranty.   */

#define TEXT         0
#define G320x200x16  1
#define G640x200x16  2
#define G640x350x16  3
#define G640x480x16  4
#define G320x200x256 5
#define G320x240x256 6
#define G320x400x256 7
#define G360x480x256 8
#define G640x480x2   9

#define G640x480x256  10
#define G800x600x256  11
#define G1024x768x256 12

extern int vga_setmode (int mode);
extern int vga_hasmode (int mode);

extern void vga_clear (void);

extern int vga_getxdim (void);
extern int vga_getydim (void);
extern int vga_getcolors (void);

extern void vga_screenoff (void);
extern void vga_screenon (void);

extern void switch_text_mode (void);
extern void switch_graphics_mode (void);

extern void vga_setcolor (int color);
extern void vga_drawpixel (int x, int y);
extern void vga_drawhline (int x1, int x2, int y);
extern void vga_drawbox (int x1, int y1, int x2, int y2);
extern void vga_drawline (int x1, int y1, int x2, int y2);
#if 0
extern void vga_drawscanline (int line, unsigned char* colors);
extern void vga_drawscansegment (unsigned char* colors, int x, int y, int length);
#endif


# =========================== vgalib/vga.c =================================== #

/* VGAlib version 1.2 - (c) 1993 Tommy Frandsen                    */
/*                                                                 */
/* This library is free software; you can redistribute it and/or   */
/* modify it without any restrictions. This library is distributed */
/* in the hope that it will be useful, but without any warranty.   */

#ifdef UNIX
  #include <sys/kd.h>
#endif

#define uchar unsigned char

#define GRAPH_BASE 0xA0000
#define GRAPH_SIZE 0x10000
#define FONT_BASE  0xA0000
#define FONT_SIZE  0x2000
#define NULL_SIZE  0x1000

#define MAX_REGS 100

/* VGA index register ports */
#define CRT_IC  0x3D4   /* CRT Controller Index - color emulation */
#define CRT_IM  0x3B4   /* CRT Controller Index - mono emulation */
#define ATT_IW  0x3C0   /* Attribute Controller Index & Data Write Register */
#define GRA_I   0x3CE   /* Graphics Controller Index */
#define SEQ_I   0x3C4   /* Sequencer Index */
#define PEL_IW  0x3C8   /* PEL Write Index */
#define PEL_IR  0x3C7   /* PEL Read Index */

/* VGA data register ports */
#define CRT_DC  0x3D5   /* CRT Controller Data Register - color emulation */
#define CRT_DM  0x3B5   /* CRT Controller Data Register - mono emulation */
#define ATT_R   0x3C1   /* Attribute Controller Data Read Register */
#define GRA_D   0x3CF   /* Graphics Controller Data Register */
#define SEQ_D   0x3C5   /* Sequencer Data Register */
#define MIS_R   0x3CC   /* Misc Output Read Register */
#define MIS_W   0x3C2   /* Misc Output Write Register */
#define IS1_RC  0x3DA   /* Input Status Register 1 - color emulation */
#define IS1_RM  0x3BA   /* Input Status Register 1 - mono emulation */
#define PEL_D   0x3C9   /* PEL Data Register */

/* VGA indexes max counts */
#define CRT_C   24      /* 24 CRT Controller Registers */
#define ATT_C   21      /* 21 Attribute Controller Registers */
#define GRA_C   9       /* 9  Graphics Controller Registers */
#define SEQ_C   5       /* 5  Sequencer Registers */
#define MIS_C   1       /* 1  Misc Output Register */

/* VGA registers saving indexes */
#define CRT     0               /* CRT Controller Registers start */
#define ATT     CRT+CRT_C       /* Attribute Controller Registers start */
#define GRA     ATT+ATT_C       /* Graphics Controller Registers start */
#define SEQ     GRA+GRA_C       /* Sequencer Registers */
#define MIS     SEQ+SEQ_C       /* General Registers */
#define EXT     MIS+MIS_C       /* SVGA Extended Registers */

#define SEG_SELECT 0x3CD

#define ABS(a) (((a)<0) ? -(a) : (a))

/* variables used to shift between monchrome and color emulation */
static int CRT_I;               /* current CRT index register address */
static int CRT_D;               /* current CRT data register address */
static int IS1_R;               /* current input status register address */
static int color_text;          /* true if color text emulation */


/* graphics mode information */
struct info {
    int xdim;
    int ydim;
    int colors;
    int xbytes;
};


/* BIOS mode 0Dh - 320x200x16 */
static uchar g320x200x16_regs[60] = {
  0x2D,0x27,0x28,0x90,0x2B,0x80,0xBF,0x1F,0x00,0xC0,0x00,0x00,
  0x00,0x00,0x00,0x00,0x9C,0x8E,0x8F,0x14,0x00,0x96,0xB9,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F,0xFF,
  0x03,0x09,0x0F,0x00,0x06,
  0x63
};
static struct info g320x200x16_info = { 320, 200, 16, 40 };


/* BIOS mode 0Eh - 640x200x16 */
static uchar g640x200x16_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0xBF,0x1F,0x00,0xC0,0x00,0x00,
  0x00,0x00,0x00,0x00,0x9C,0x8E,0x8F,0x28,0x00,0x96,0xB9,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0x63
};
static struct info g640x200x16_info = { 640, 200, 16, 80 };


/* BIOS mode 10h - 640x350x16 */
static uchar g640x350x16_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0xBF,0x1F,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0x83,0x85,0x5D,0x28,0x0F,0x63,0xBA,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0xA3
};
static struct info g640x350x16_info = { 640, 350, 16, 80 };


/* BIOS mode 12h - 640x480x16 */
static uchar g640x480x16_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0x28,0x00,0xE7,0x04,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0xE3
};
static struct info g640x480x16_info = { 640, 480, 16, 80 };


/* BIOS mode 13h - 320x200x256 */
static uchar g320x200x256_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0xBF,0x1F,0x00,0x41,0x00,0x00,
  0x00,0x00,0x00,0x00,0x9C,0x8E,0x8F,0x28,0x40,0x96,0xB9,0xA3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x41,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0x63
};
static struct info g320x200x256_info = { 320, 200, 256, 320 };


/* non-BIOS mode - 320x240x256 */
static uchar g320x240x256_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0x0D,0x3E,0x00,0x41,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0xAC,0xDF,0x28,0x00,0xE7,0x06,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x41,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0xE3
};
static struct info g320x240x256_info = { 320, 240, 256, 80 };


/* non-BIOS mode - 320x400x256 */
static uchar g320x400x256_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0xBF,0x1F,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0x9C,0x8E,0x8F,0x28,0x00,0x96,0xB9,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x41,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0x63
};
static struct info g320x400x256_info = { 320, 400, 256, 80 };


/* non-BIOS mode - 360x480x256 */
static uchar g360x480x256_regs[60] = {
  0x6B,0x59,0x5A,0x8E,0x5E,0x8A,0x0D,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0xAC,0xDF,0x2D,0x00,0xE7,0x06,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x41,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0xE7
};
static struct info g360x480x256_info = { 360, 480, 256, 90 };


/* monochrome mode based on BIOS mode 12h - 640x480x2 */
static uchar g640x480x2_regs[60] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0x28,0x00,0xE7,0x04,0xE3,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x0F,0x00,0x20,0x00,0x00,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x06,
  0xE3
};
static struct info g640x480x2_info = { 640, 480, 2, 80 };


/* ET4000 BIOS mode 2Eh - 640x480x256 */
static uchar g640x480x256_regs[70] = {
  0x5F,0x4F,0x50,0x82,0x54,0x80,0x0B,0x3E,0x00,0x40,0x00,0x00,
  0x00,0x00,0x00,0x00,0xEA,0x8C,0xDF,0x50,0x60,0xE7,0x04,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xE3,
  0x70,0x00,0x08,0x00,0x43,0x1F,0xBC,0x00,0x00,0x00
};
static struct info g640x480x256_info = { 640, 480, 256, 640 };


/* ET4000 BIOS mode 30h - 800x600x256 */
static uchar g800x600x256_regs[70] = {
  0x7A,0x63,0x64,0x1D,0x68,0x9A,0x78,0xF0,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x5C,0x8E,0x57,0x64,0x60,0x5B,0x75,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0xEF,
  0x70,0x00,0x08,0x00,0x43,0x1F,0xBC,0x00,0x00,0x00
};
static struct info g800x600x256_info = { 800, 600, 256, 800 };


/* ET4000 BIOS mode 38h - 1024x768x256 */
static uchar g1024x768x256_regs[70] = {
  0x99,0x7F,0x7F,0x1D,0x83,0x17,0x2F,0xF5,0x00,0x60,0x00,0x00,
  0x00,0x00,0x00,0x00,0x08,0x80,0xFF,0x80,0x60,0xFF,0x30,0xAB,
  0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,
  0x0C,0x0D,0x0E,0x0F,0x01,0x00,0x0F,0x00,0x00,
  0x00,0x00,0x00,0x00,0x00,0x40,0x05,0x0F,0xFF,
  0x03,0x01,0x0F,0x00,0x0E,
  0x27,
  0x70,0x00,0x0A,0x80,0x43,0x1F,0xBC,0x00,0x00,0x00
};
static struct info g1024x768x256_info = { 1024, 768, 256, 1024 };


/* default palette values */
static uchar default_red[256]
             = { 0, 0, 0, 0,42,42,42,42,21,21,21,21,63,63,63,63,
                 0, 5, 8,11,14,17,20,24,28,32,36,40,45,50,56,63,
                 0,16,31,47,63,63,63,63,63,63,63,63,63,47,31,16,
                 0, 0, 0, 0, 0, 0, 0, 0,31,39,47,55,63,63,63,63,
                63,63,63,63,63,55,47,39,31,31,31,31,31,31,31,31,
                45,49,54,58,63,63,63,63,63,63,63,63,63,58,54,49,
                45,45,45,45,45,45,45,45, 0, 7,14,21,28,28,28,28,
                28,28,28,28,28,21,14, 7, 0, 0, 0, 0, 0, 0, 0, 0,
                14,17,21,24,28,28,28,28,28,28,28,28,28,24,21,17,
                14,14,14,14,14,14,14,14,20,22,24,26,28,28,28,28,
                28,28,28,28,28,26,24,22,20,20,20,20,20,20,20,20,
                 0, 4, 8,12,16,16,16,16,16,16,16,16,16,12, 8, 4,
                 0, 0, 0, 0, 0, 0, 0, 0, 8,10,12,14,16,16,16,16,
                16,16,16,16,16,14,12,10, 8, 8, 8, 8, 8, 8, 8, 8,
                11,12,13,15,16,16,16,16,16,16,16,16,16,15,13,12,
                11,11,11,11,11,11,11,11, 0, 0, 0, 0, 0, 0, 0, 0};
static uchar default_green[256]
             = { 0, 0,42,42, 0, 0,21,42,21,21,63,63,21,21,63,63,
                 0, 5, 8,11,14,17,20,24,28,32,36,40,45,50,56,63,
                 0, 0, 0, 0, 0, 0, 0, 0, 0,16,31,47,63,63,63,63,
                63,63,63,63,63,47,31,16,31,31,31,31,31,31,31,31,
                31,39,47,55,63,63,63,63,63,63,63,63,63,55,47,39,
                45,45,45,45,45,45,45,45,45,49,54,58,63,63,63,63,
                63,63,63,63,63,58,54,49, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 7,14,21,29,28,28,28,28,28,28,28,28,21,14, 7,
                14,14,14,14,14,14,14,14,14,17,21,24,28,28,28,28,
                28,28,28,28,28,24,21,17,20,20,20,20,20,20,20,20,
                20,22,24,26,28,28,28,28,28,28,28,28,28,26,24,22,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 8,12,16,16,16,16,
                16,16,16,16,16,12, 8, 4, 8, 8, 8, 8, 8, 8, 8, 8,
                 8,10,12,14,16,16,16,16,16,16,16,16,16,14,12,10,
                11,11,11,11,11,11,11,11,11,12,13,15,16,16,16,16,
                16,16,16,16,16,15,13,12, 0, 0, 0, 0, 0, 0, 0, 0};
static uchar default_blue[256]
             = { 0,42, 0,42, 0,42, 0,42,21,63,21,63,21,63,21,63,
                 0, 5, 8,11,14,17,20,24,28,32,36,40,45,50,56,63,
                63,63,63,63,63,47,31,16, 0, 0, 0, 0, 0, 0, 0, 0,
                 0,16,31,47,63,63,63,63,63,63,63,63,63,55,47,39,
                31,31,31,31,31,31,31,31,31,39,47,55,63,63,63,63,
                63,63,63,63,63,58,54,49,45,45,45,45,45,45,45,45,
                45,49,54,58,63,63,63,63,28,28,28,28,28,21,14, 7,
                 0, 0, 0, 0, 0, 0, 0, 0, 0, 7,14,21,28,28,28,28,
                28,28,28,28,28,24,21,17,14,14,14,14,14,14,14,14,
                14,17,21,24,28,28,28,28,28,28,28,28,28,26,24,22,
                20,20,20,20,20,20,20,20,20,22,24,26,28,28,28,28,
                16,16,16,16,16,12, 8, 4, 0, 0, 0, 0, 0, 0, 0, 0,
                 0, 4, 8,12,16,16,16,16,16,16,16,16,16,14,12,10,
                 8, 8, 8, 8, 8, 8, 8, 8, 8,10,12,14,16,16,16,16,
                16,16,16,16,16,15,13,12,11,11,11,11,11,11,11,11,
                11,12,13,15,16,16,16,16, 0, 0, 0, 0, 0, 0, 0, 0};


/* used to decompose color value into bits (for fast scanline drawing) */
union bits {
    struct {
        uchar bit3;
        uchar bit2;
        uchar bit1;
        uchar bit0;
    } b;
    unsigned int i;
};

/* color decompositions */
static union bits color16[16] = {{0,0,0,0},
                                 {0,0,0,1},
                                 {0,0,1,0},
                                 {0,0,1,1},
                                 {0,1,0,0},
                                 {0,1,0,1},
                                 {0,1,1,0},
                                 {0,1,1,1},
                                 {1,0,0,0},
                                 {1,0,0,1},
                                 {1,0,1,0},
                                 {1,0,1,1},
                                 {1,1,0,0},
                                 {1,1,0,1},
                                 {1,1,1,0},
                                 {1,1,1,1}};

/* display plane buffers (for fast scanline drawing) */
static uchar plane0[256];
static uchar plane1[256];
static uchar plane2[256];
static uchar plane3[256];


static uchar text_regs[MAX_REGS];   /* VGA registers for saved text mode */

/* saved text mode palette values */
static uchar text_red[256];
static uchar text_green[256];
static uchar text_blue[256];

/* saved graphics mode palette values */
static uchar graph_red[256];
static uchar graph_green[256];
static uchar graph_blue[256];

static int         cur_mode  = TEXT;     /* current video mode       */
static int         flip_mode = TEXT;     /* flipped video mode       */
static struct info cur_info;             /* current video parameters */
static int         cur_color;            /* current color            */
static struct clip cur_clip;             /* current whole-screen clipping */

static int initialized = 0;   /* flag: initialize() called ?  */
static int et4000      = 0;   /* flag: ET4000 or standard VGA */

#ifdef UNIX
static int   tty0_fd;    /* /dev/tty0 file descriptor                 */
#endif
static uchar* graph_mem; /* dummy buffer for mmapping graphics memory */
static uchar* graph_buf; /* saves graphics data during flip           */

static uchar font_buf1[FONT_SIZE];  /* saved font data - plane 2 */
static uchar font_buf2[FONT_SIZE];  /* saved font data - plane 3 */
static uchar null_buf[NULL_SIZE];   /* used to speed up clear */


static inline void port_out (uchar value, unsigned short port)
{
        __asm__ volatile ("outb %0,%1"
                          :
                          : "a" ((uchar) value), "d" ((unsigned short) port));
}

static inline uchar port_in (unsigned short port)
{
        uchar _v;
        __asm__ volatile ("inb %1,%0"
                          : "=a" (_v)
                          : "d" ((unsigned short) port));
        return _v;
}


# There is a big problem:
#   Text output may not occur while the VGA hardware is in graphics mode.
#
# We distinguish
# a. text output from CLISP itself,
# b. text output from the OS, for example when the user types something,
# c. text output from other programs.
#
# a. Solved: We switch to text mode every time we are about to output
#    something to stdout. (See stream.d.)
# b. Solved: DOS/EMX doesn't output anything (local echo is not done until
#    the next read() call), Linux consoles are protected by KD_GRAPHICS.
# c. Partially solved: On exit, CLISP switches to text mode (see spvw.d).
#    Signals SIGTSTP, SIGTTIN, SIGTTOU are ignored, and signal SIGSTOP is
#    de facto disabled, using termios ioctl's.
#    But when other programs are called by CLISP (using SHELL, EXECUTE or
#    MAKE-PIPE-OUTPUT-STREAM), it is up to the user to ensure they don't do
#    any output!

#ifdef UNIX

static char suspend_char = 0; # the suspend character (normally Ctrl-Z)

static void set_graphmode_signals (void)
{
    begin_system_call();
    SIGNAL(SIGTSTP,SIG_IGN);
    SIGNAL(SIGTTIN,SIG_IGN);
    SIGNAL(SIGTTOU,SIG_IGN);
    { struct termios the_termio;
      if ( tcgetattr(stdin_handle,&the_termio) ==0)
        { if (!suspend_char && the_termio.c_cc[VSUSP])
            suspend_char = the_termio.c_cc[VSUSP];
          the_termio.c_cc[VSUSP] = 0; # disable suspend character
          TCSETATTR(stdin_handle,TCSAFLUSH,&the_termio);
    }   }
    end_system_call();
}

static void set_textmode_signals (void)
{
    begin_system_call();
    SIGNAL(SIGTSTP,SIG_DFL);
    SIGNAL(SIGTTIN,SIG_DFL);
    SIGNAL(SIGTTOU,SIG_DFL);
    if (suspend_char)
      { struct termios the_termio;
        if ( tcgetattr(stdin_handle,&the_termio) ==0)
          { # Be careful not to enable Ctrl-Z when we are in raw mode!
            if ((the_termio.c_lflag & ICANON) && !the_termio.c_cc[VSUSP])
              { the_termio.c_cc[VSUSP] = suspend_char; # re-enable suspend character
                TCSETATTR(stdin_handle,TCSAFLUSH,&the_termio);
      }   }   }
    end_system_call();
}

#endif

#ifdef EMUNIX

# EMX doesn't have these signals.

static inline void set_graphmode_signals (void) {}
static inline void set_textmode_signals (void) {}

#endif


# When poking around in the VGA hardware, we need to disable the most critical
# interrupts. Well, we can't disable the scheduler, but we can block nearly
# all the signals and disable SIGSTOP.
# The macros disable_interrupts() and enable_interrupts() must be called in
# pairs, in the same block.

#ifdef UNIX
  #define disable_interrupts()  \
    { sigset_t old_sigmask, sigblock_mask;                 \
      sigfillset(&sigblock_mask);                          \
      sigprocmask(SIG_BLOCK,&sigblock_mask,&old_sigmask);  \
      { struct termios the_termio;                         \
        if ( tcgetattr(stdin_handle,&the_termio) ==0)      \
          { the_termio.c_lflag &= ~ISIG;                   \
            TCSETATTR(stdin_handle,TCSAFLUSH,&the_termio); \
      }   }
  #define enable_interrupts()  \
      { struct termios the_termio;                         \
        if ( tcgetattr(stdin_handle,&the_termio) ==0)      \
          { the_termio.c_lflag |= ISIG;                    \
            TCSETATTR(stdin_handle,TCSAFLUSH,&the_termio); \
      }   }                                                \
      sigprocmask(SIG_SETMASK,&old_sigmask,NULL);          \
    }
#endif
#ifdef EMUNIX
  # EMX doesn't have these signals.
  #define disable_interrupts()
  #define enable_interrupts()
#endif


# The next six functions should only be called with interrupts disabled.


# Get I/O port permissions.
static int get_perm()
{
    static int done = 0;

    if (!done)
        {

          #ifdef UNIX
          {
            int err = 0;

            BEGIN_ROOT_PRIV;

            /* get I/O permissions for VGA registers */
            if (ioperm(CRT_IC, 1, 1)
             || ioperm(CRT_IM, 1, 1)
             || ioperm(ATT_IW, 1, 1)
             || ioperm(GRA_I,  1, 1)
             || ioperm(SEQ_I,  1, 1)
             || ioperm(PEL_IW, 1, 1)
             || ioperm(PEL_IR, 1, 1)
             || ioperm(CRT_DC, 1, 1)
             || ioperm(CRT_DM, 1, 1)
             || ioperm(ATT_R,  1, 1)
             || ioperm(GRA_D,  1, 1)
             || ioperm(SEQ_D,  1, 1)
             || ioperm(MIS_R,  1, 1)
             || ioperm(MIS_W,  1, 1)
             || ioperm(IS1_RC, 1, 1)
             || ioperm(IS1_RM, 1, 1)
             || ioperm(PEL_D,  1, 1)

            /* ET4000 registers */
             || ioperm(0x3bf,  1, 1)
             || ioperm(0x3cc,  1, 1)
             || ioperm(0x3d8,  1, 1)
             || ioperm(0x3b8,  1, 1)
             || ioperm(0x3c3,  1, 1)
             || ioperm(0x3cd,  1, 1)
               )
                err = 1;

            END_ROOT_PRIV;

            if (err)
                return -1; /* can't get I/O permissions */
          }
          #endif

          #ifdef EMUNIX

            if (_portaccess(0x3B4,0x3DA))
                { errno = EPERM; return -1; } /* can't get I/O permissions */

          #endif

            /* color or monochrome text emulation? */
            color_text = port_in(MIS_R) & 0x01;

            /* chose registers for color/monochrome emulation */
            if (color_text) {
                CRT_I = CRT_IC;
                CRT_D = CRT_DC;
                IS1_R = IS1_RC;
            } else {
                CRT_I = CRT_IM;
                CRT_D = CRT_DM;
                IS1_R = IS1_RM;
            }

            done = 1;
        }

    return 0;
}


# Returns 1 if an ET4000 card is present, else 0.

static int et4000_test (void)
{
    static int result = -1;
    uchar new, old, val;
    int  base;

    /* test already done? */
    if (result >= 0)
        return result;

    /* test for Tseng clues */
    old = port_in(0x3cd);
    port_out(0x55, 0x3cd);
    new = port_in(0x3cd);
    port_out(old, 0x3cd);

    /* return false if not Tseng */
    if (new != 0x55)
        return result = 0;

    /* test for ET4000 clues */
    if (port_in(0x3cc) & 1)
        base = 0x3d4;
    else
        base = 0x3b4;
    port_out(0x33, base);
    old = port_in(base+1);
    new = old ^ 0xf;
    port_out(new, base+1);
    val = port_in(base+1);
    port_out(old, base+1);

    /* return true if ET4000 */
    return result = (val == new);
}


static void et4000_save_regs (uchar regs[])
{
    int i;

    /* save extended CRT registers */
    for (i = 0; i < 6; i++) {
         port_out(0x32+i, CRT_I);
         regs[EXT+i] = port_in(CRT_D);
    }

    /* save extended sequencer register */
    port_out(7, SEQ_I);
    regs[EXT+6] = port_in(SEQ_D);

    /* save some other ET4000 specific registers */
    regs[EXT+7] = port_in(0x3c3);
    regs[EXT+8] = port_in(0x3cd);

    /* save extended attribute register */
    port_in(IS1_R);    /* reset flip flop */
    port_out(0x16, ATT_IW);
    regs[EXT+9] = port_in(ATT_R);
}


static void et4000_set_regs (uchar regs[])
{
    int i;

    /* write some ET4000 specific registers */
    port_out(regs[EXT+7], 0x3c3);
    port_out(regs[EXT+8], 0x3cd);

    /* write extended sequencer register */
    port_out(7, SEQ_I);
    port_out(regs[EXT+6], SEQ_D);

    /* write extended CRT registers */
    for (i = 0; i < 6; i++) {
         port_out(0x32+i, CRT_I);
         port_out(regs[EXT+i], CRT_D);
    }

    /* write extended attribute register */
    port_in(IS1_R);    /* reset flip flop */
    port_out(0x16, ATT_IW);
    port_out(regs[EXT+9], ATT_IW);
}


static void save_regs (uchar regs[])
{
    int i;

    /* save VGA registers */
    for (i = 0; i < CRT_C; i++) {
         port_out(i, CRT_I);
         regs[CRT+i] = port_in(CRT_D);
    }
    for (i = 0; i < ATT_C; i++) {
         port_in(IS1_R);
         port_out(i, ATT_IW);
         regs[ATT+i] = port_in(ATT_R);
    }
    for (i = 0; i < GRA_C; i++) {
         port_out(i, GRA_I);
         regs[GRA+i] = port_in(GRA_D);
    }
    for (i = 0; i < SEQ_C; i++) {
         port_out(i, SEQ_I);
         regs[SEQ+i] = port_in(SEQ_D);
    }
    regs[MIS] = port_in(MIS_R);

    if (et4000)
        et4000_save_regs(regs);
}


static void set_regs (uchar regs[], int mode)
{
    int i;

    /* disable video */
    port_in(IS1_R);
    port_out(0x00, ATT_IW);

    /* update misc output register */
    port_out(regs[MIS], MIS_W);

    /* synchronous reset on */
    port_out(0x00,SEQ_I);
    port_out(0x01,SEQ_D);

    /* write sequencer registers */
    for (i = 1; i < SEQ_C; i++) {
        port_out(i, SEQ_I);
        port_out(regs[SEQ+i], SEQ_D);
    }

    /* synchronous reset off */
    port_out(0x00, SEQ_I);
    port_out(0x03, SEQ_D);

    /* deprotect CRT registers 0-7 */
    port_out(0x11, CRT_I);
    port_out(port_in(CRT_D)&0x7F, CRT_D);

    /* write CRT registers */
    for (i = 0; i < CRT_C; i++) {
        port_out(i, CRT_I);
        port_out(regs[CRT+i], CRT_D);
    }

    /* write graphics controller registers */
    for (i = 0; i < GRA_C; i++) {
        port_out(i, GRA_I);
        port_out(regs[GRA+i], GRA_D);
    }

    /* write attribute controller registers */
    for (i = 0; i < ATT_C; i++) {
        port_in(IS1_R);   /* reset flip-flop */
        port_out(i, ATT_IW);
        port_out(regs[ATT+i],ATT_IW);
    }

    if (et4000)
        if (mode == G640x480x256 || mode == G800x600x256 || mode == G1024x768x256)
            et4000_set_regs(regs);
}


static void vga_setpalette (int index, int red, int green, int blue)
{
    int i;

    /* select palette register */
    port_out(index, PEL_IW);

    /* write RGB components */
    for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
    port_out(red, PEL_D);
    for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
    port_out(green, PEL_D);
    for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
    port_out(blue, PEL_D);
}


static void vga_getpalette (int index, int *red, int *green, int *blue)
{
    int i;

    /* select palette register */
    port_out(index, PEL_IR);

    /* read RGB components */
    for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
    *red = (int) port_in(PEL_D);
    for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
    *green = (int) port_in(PEL_D);
    for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
    *blue = (int) port_in(PEL_D);
}

#if 0

static void vga_setpalvec (int start, int num, int *pal)
{
    int i, j;

    /* select palette register */
    port_out(start, PEL_IW);

    for(j = 0; j < num; j++) {
        for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
        port_out(*(pal++), PEL_D);
        for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
        port_out(*(pal++), PEL_D);
        for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
        port_out(*(pal++), PEL_D);
    }
}


static void vga_getpalvec (int start, int num, int *pal)
{
    int i, j;

    /* select palette register */
    port_out(start, PEL_IR);

    for(j = 0; j < num; j++) {
        for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
        *(pal++) = (int) port_in(PEL_D);
        for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
        *(pal++) = (int) port_in(PEL_D);
        for(i = 0; i < 10; i++) ;   /* delay (minimum 240ns) */
        *(pal++) = (int) port_in(PEL_D);
    }
}

#endif

void vga_setcolor (int color)
{
    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
            /* update set/reset register */
            port_out(0x00, GRA_I );
            port_out(color, GRA_D );
            break;
        case G640x480x2:
            if (color != 0)
                color = 15;
            /* update set/reset register */
            port_out(0x00, GRA_I );
            port_out(color, GRA_D );
            break;
        case G320x200x256:
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
            cur_color = color;
            break;
    }
}


void vga_screenoff()
{
    /* turn off screen for faster VGA memory access */
    port_out(0x01, SEQ_I);
    port_out(port_in(SEQ_D)|0x20, SEQ_D);
}


void vga_screenon()
{
    /* turn screen back on */
    port_out(0x01, SEQ_I);
    port_out(port_in(SEQ_D)&0xDF, SEQ_D);
}


void vga_clear()
{
    int i, j;

    vga_screenoff();

    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
        case G640x480x2:
            vga_setcolor(0);

            /* write to all bits */
            port_out(0x08, GRA_I );
            port_out(0xFF, GRA_D );

            /* write dummy values to clear video memory */
            for(i = 0; i < 16; i++)
                memcpy(graph_mem + i*NULL_SIZE, null_buf, NULL_SIZE);

            break;
        case G320x200x256:
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
            /* write to all planes */
            port_out(0x02, SEQ_I );
            port_out(0x0F, SEQ_D );

            /* clear video memory */
            for(i = 0; i < 16; i++)
                memcpy(graph_mem + i*NULL_SIZE, null_buf, NULL_SIZE);

            break;
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
            for(i = 0; i < 16; i++) {
                /* select segment */
                port_out(i, SEG_SELECT);

                /* clear video memory */
                for(j = 0; j < 16; j++)
                    memcpy(graph_mem + j*NULL_SIZE, null_buf, NULL_SIZE);
            }
            break;
    }

    vga_setcolor(15);

    vga_screenon();
}


/* set cur_mode to TEXT */
static void set_textmode (void)
{
    int i;

    /* disable video */
    port_in(IS1_R);
    port_out(0x00, ATT_IW);

    if (et4000 && cur_mode == G1024x768x256)
        set_regs(g640x480x256_regs, G640x480x256);

    cur_mode = TEXT;

    /* restore font data - first select a 16 color graphics mode */
    set_regs(g640x480x16_regs, G640x480x16);

    /* disable Set/Reset Register */
    port_out(0x01, GRA_I );
    port_out(0x00, GRA_D );

    /* restore font data in plane 2 - necessary for all VGA's */
    port_out(0x02, SEQ_I );
    port_out(0x04, SEQ_D );
    memcpy(graph_mem, font_buf1, FONT_SIZE);

    /* restore font data in plane 3 - necessary for Trident VGA's */
    port_out(0x02, SEQ_I );
    port_out(0x08, SEQ_D );
    memcpy(graph_mem, font_buf2, FONT_SIZE);

    /* change register adresses if monochrome text mode */
    if (!color_text) {
        CRT_I = CRT_IM;
        CRT_D = CRT_DM;
        IS1_R = IS1_RM;
        port_out(port_in(MIS_R)&0xFE, MIS_W);
    }

    /* restore saved palette */
    for(i = 0; i < 256; i++)
        vga_setpalette(i, text_red[i], text_green[i], text_blue[i]);

    /* restore text mode VGA registers */
    set_regs(text_regs, TEXT);

    /* enable video */
    port_in(IS1_R);
    port_out(0x20, ATT_IW);

    #ifdef UNIX
    /* enable text output - restores the screen contents */
    ioctl(tty0_fd, KDSETMODE, KD_TEXT);
    #endif

    /* restore text mode termio */
    set_textmode_signals();
}


/* set cur_mode to mode != TEXT, maybe clearing the screen */
static void set_graphmode (int mode, int clear)
{
    int i;

    #ifdef UNIX
    /* disable text output */
    ioctl(tty0_fd, KDSETMODE, KD_GRAPHICS);
    #endif

    /* disable video */
    port_in(IS1_R);
    port_out(0x00, ATT_IW);

    if (et4000 && cur_mode == G1024x768x256)
        set_regs(g640x480x256_regs, G640x480x256);

    cur_mode = mode;

    /* shift to color emulation */
    CRT_I = CRT_IC;
    CRT_D = CRT_DC;
    IS1_R = IS1_RC;
    port_out(port_in(MIS_R)|0x01, MIS_W);

    switch (mode) {
        case G320x200x16:
            set_regs(g320x200x16_regs, G320x200x16);
            cur_info = g320x200x16_info;
            break;
        case G640x200x16:
            set_regs(g640x200x16_regs, G640x200x16);
            cur_info = g640x200x16_info;
            break;
        case G640x350x16:
            set_regs(g640x350x16_regs, G640x350x16);
            cur_info = g640x350x16_info;
            break;
        case G640x480x16:
            set_regs(g640x480x16_regs, G640x480x16);
            cur_info = g640x480x16_info;
            break;
        case G320x200x256:
            set_regs(g320x200x256_regs, G320x200x256);
            cur_info = g320x200x256_info;
            break;
        case G320x240x256:
            set_regs(g320x240x256_regs, G320x240x256);
            cur_info = g320x240x256_info;
            break;
        case G320x400x256:
            set_regs(g320x400x256_regs, G320x400x256);
            cur_info = g320x400x256_info;
            break;
        case G360x480x256:
            set_regs(g360x480x256_regs, G360x480x256);
            cur_info = g360x480x256_info;
            break;
        case G640x480x2:
            set_regs(g640x480x2_regs, G640x480x2);
            cur_info = g640x480x2_info;
            break;
        case G640x480x256:
            set_regs(g640x480x256_regs, G640x480x256);
            cur_info = g640x480x256_info;
            break;
        case G800x600x256:
            set_regs(g800x600x256_regs, G800x600x256);
            cur_info = g800x600x256_info;
            break;
        case G1024x768x256:
            set_regs(g1024x768x256_regs, G1024x768x256);
            cur_info = g1024x768x256_info;
            break;
    }
    cur_clip.x1 = 0; cur_clip.x2 = cur_info.xdim-1;
    cur_clip.y1 = 0; cur_clip.y2 = cur_info.ydim-1;

    if (clear) {
        /* set default palette */
        for(i = 0; i < 256; i++)
            vga_setpalette(i, default_red[i],default_green[i],default_blue[i]);

        /* clear screen (sets current color to 15) */
        vga_clear();
    }

    /* enable video */
    port_in(IS1_R);
    port_out(0x20, ATT_IW);

    /* set graphics mode termio */
    set_graphmode_signals();
}


/* Global initialization. */
/* Only called once, by the first call to vga_hasmode() or vga_setmode(). */

static int initialize (void)
{
    int result;
    int mem_fd;
    uchar* mem_area;
    int i, j;

    disable_interrupts();

    if (get_perm())
        { asciz_out("VGAlib: can't get I/O permissions" CRLFstring);
          result = -1; goto done;
        }

    et4000 = et4000_test();

    if (et4000) {
        /* get access to extended registers */
        port_out(3, 0x3bf);
        if (port_in( 0x3cc ) & 1)
            port_out(0xa0, 0x3d8);
        else
            port_out(0xa0, 0x3b8);
    }

    /* allocate memory for saved graphics data */
    if ((graph_buf = malloc(4*GRAPH_SIZE)) == NULL)
        { asciz_out("VGAlib: not enough memory" CRLFstring);
          errno = ENOMEM; return -1;
        }

    #ifdef UNIX

    BEGIN_ROOT_PRIV;

    /* open /dev/tty0 - current virtual console */
    if ((tty0_fd = open("/dev/console", O_RDONLY) ) < 0)
        if ((tty0_fd = open("/dev/tty0", O_RDONLY) ) < 0)
            { END_ROOT_PRIV;
              free(graph_buf);
              asciz_out("VGAlib: can't open /dev/console" CRLFstring);
              result = -1; goto done;
            }

    /* open /dev/mem */
    if ((mem_fd = open("/dev/mem", O_RDWR) ) < 0)
        { END_ROOT_PRIV;
          close(tty0_fd); free(graph_buf);
          asciz_out("VGAlib: can't open /dev/mem" CRLFstring);
          result = -1; goto done;
        }

    END_ROOT_PRIV;

    /* mmap graphics memory */
    if ((mem_area = malloc(GRAPH_SIZE + getpagesize()-1)) == NULL)
        { close(tty0_fd); close(mem_fd); free(graph_buf);
          asciz_out("VGAlib: not enough memory" CRLFstring);
          errno = ENOMEM; result = -1; goto done;
        }
    graph_mem = mem_area + ((-(unsigned long)mem_area) & (getpagesize()-1));
    graph_mem = (uchar *) mmap((caddr_t)graph_mem, GRAPH_SIZE,
                               PROT_READ|PROT_WRITE, MAP_SHARED|MAP_FIXED,
                               mem_fd, GRAPH_BASE);
    if ((long)graph_mem < 0)
        { close(tty0_fd); close(mem_fd); free(mem_area); free(graph_buf);
          asciz_out("VGAlib: couldn't mmap graphics memory" CRLFstring);
          result = -1; goto done;
        }

    #endif

    #ifdef EMUNIX

    graph_mem = _memaccess (GRAPH_BASE, GRAPH_BASE+GRAPH_SIZE-1, 1);
    if (graph_mem == NULL)
        { asciz_out("VGAlib: couldn't get access to graphics memory" CRLFstring);
          errno = EINVAL; result = -1; goto done;
        }

    #endif

    /* Here's the point of no return. */

    #ifdef UNIX
    /* disable text output to console */
    ioctl(tty0_fd, KDSETMODE, KD_GRAPHICS);
    #endif

    /* disable video */
    port_in(IS1_R);
    port_out(0x00, ATT_IW);

    save_regs(text_regs);

    /* save text mode palette - first select palette index 0 */
    port_out(0, PEL_IR);

    /* read RGB components - index is autoincremented */
    for(i = 0; i < 256; i++) {
        for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */
        text_red[i] = port_in(PEL_D);
        for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */
        text_green[i] = port_in(PEL_D);
        for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */
        text_blue[i] = port_in(PEL_D);
    }

    /* shift to color emulation */
    CRT_I = CRT_IC;
    CRT_D = CRT_DC;
    IS1_R = IS1_RC;
    port_out(port_in(MIS_R)|0x01, MIS_W);

    /* save font data - first select a 16 color graphics mode */
    set_regs(g640x480x16_regs, G640x480x16);

    /* save font data in plane 2 */
    port_out(0x04, GRA_I);
    port_out(0x02, GRA_D);
    memcpy(font_buf1, graph_mem, FONT_SIZE);

    /* save font data in plane 3 */
    port_out(0x04, GRA_I);
    port_out(0x03, GRA_D);
    memcpy(font_buf2, graph_mem, FONT_SIZE);

    /* enable video */
    port_in(IS1_R);
    port_out(0x20, ATT_IW);

    /* initialize buffer used when clearing in 256 color modes */
    for(i = 0; i < NULL_SIZE; i++)
        null_buf[i] = 0;

    /* we have modified the VGA registers -- back to text mode for now */
    set_textmode();

    initialized = 1;

    result = 0;

    done:
    enable_interrupts();

    return result;
}


int vga_hasmode (int mode)
{
    switch (mode) {
        case TEXT:
            return 1;
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
        case G640x480x2:
        case G320x200x256:
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
            if (!initialized)
                if (initialize())
                    return 0;
            return 1;
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
            if (!initialized)
                if (initialize())
                    return 0;
            return et4000;
        default:
            return 0;
    }
}

int vga_setmode (int mode)
{
    if (!initialized)
        if (initialize())
            return -1;

    disable_interrupts();

    if (mode == TEXT)
        set_textmode();
    else
        set_graphmode(mode,1);

    enable_interrupts();

    return 0;
}


# We do `lazy switching' between text mode and graphics mode.
# This switching doesn't lose screen's contents. It is therefore also called
# `flipping'.
# flip_mode is the mode of the "screen behind the scenes".
# At least one of cur_mode and flip_mode is TEXT.

void switch_text_mode (void)
{
    int i, j;

    if (cur_mode == TEXT)
        return;

    /* cur_mode != TEXT, so flip_mode == TEXT,
       and initialize() has already been called. */

    disable_interrupts();

    /* disable video */
    port_in(IS1_R);
    port_out(0x00, ATT_IW);

    /* save all four planes - first select a 16 color graphics mode */
    set_regs(g640x480x16_regs, G640x480x16);

    for(i = 0; i < 4; i++) {
        /* save plane i */
        port_out(0x04, GRA_I);
        port_out(   i, GRA_D);
        memcpy(graph_buf + i*GRAPH_SIZE, graph_mem, GRAPH_SIZE);
    }

    /* save graphics mode palette - first select palette index 0 */
    port_out(0, PEL_IR);

    /* read RGB components - index is autoincremented */
    for(i = 0; i < 256; i++) {
        for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */
        graph_red[i] = port_in(PEL_D);
        for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */
        graph_green[i] = port_in(PEL_D);
        for(j = 0; j < 10; j++) ;   /* delay (minimum 240ns) */
        graph_blue[i] = port_in(PEL_D);
    }

    flip_mode = cur_mode;

    set_textmode();

    enable_interrupts();
}

void switch_graphics_mode (void)
{
    int i, j;

    if (flip_mode == TEXT)
        return;

    /* flip_mode != TEXT, so cur_mode == TEXT,
       and initialize() has already been called. */

    disable_interrupts();

    #ifdef UNIX
    /* disable text output */
    ioctl(tty0_fd, KDSETMODE, KD_GRAPHICS);
    #endif

    /* disable video */
    port_in(IS1_R);
    port_out(0x00, ATT_IW);

    /* restore all four planes - first select a 16 color graphics mode */
    set_regs(g640x480x16_regs, G640x480x16);

    /* disable Set/Reset Register */
    port_out(0x01, GRA_I );
    port_out(0x00, GRA_D );

    for(i = 0; i < 4; i++) {
        /* restore plane i */
        port_out(0x02, SEQ_I );
        port_out(1<<i, SEQ_D );
        memcpy(graph_mem, graph_buf + i*GRAPH_SIZE, GRAPH_SIZE);
    }

    /* restore saved palette */
    for(i = 0; i < 256; i++)
        vga_setpalette(i, graph_red[i], graph_green[i], graph_blue[i]);

    set_graphmode(flip_mode,0);

    flip_mode = TEXT;

    enable_interrupts();
}


void vga_drawpixel (int x, int y)
{
    unsigned long offset;

    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
        case G640x480x2:
            /* select bit */
            port_out(8, GRA_I);
            port_out(0x80 >> (x & 7), GRA_D);

            /* read into latch and write dummy back */
            offset = y*cur_info.xbytes + (x>>3);
            graph_mem[offset] = graph_mem[offset];
            break;
        case G320x200x256:
            /* write color to pixel */
            graph_mem[y*320 + x] = cur_color;
            break;
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
            /* select plane */
            port_out(0x02, SEQ_I);
            port_out(1 << (x & 3), SEQ_D);

            /* write color to pixel */
            graph_mem[y*cur_info.xbytes + (x>>2)] = cur_color;
            break;
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
            offset = y*cur_info.xbytes+x;

            /* select segment */
            port_out(offset >> 16, SEG_SELECT);

            /* write color to pixel */
            graph_mem[offset & 0xFFFF] = cur_color;
            break;
    }
}


/* Get pixel's color. */
int vga_getpixel (int x, int y)
{
    unsigned long offset;

    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
        case G640x480x2:
            # Not yet tested!!
            { int i;
              int color = 0;
              uchar mask = 0x80 >> (x & 7);
              offset = y*cur_info.xbytes + (x>>3);
              for(i = 0; i < 4; i++) {
                  /* select plane i */
                  port_out(0x04, GRA_I);
                  port_out(   i, GRA_D);
                  if (graph_mem[offset] & mask)
                      color |= (1<<i);
              }
              return color;
            }
        case G320x200x256:
            /* get pixel color */
            return graph_mem[y*320 + x];
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
            /* select plane */
            port_out(0x02, SEQ_I);
            port_out(1 << (x & 3), SEQ_D);

            /* get pixel color */
            return graph_mem[y*cur_info.xbytes + (x>>2)];
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
            offset = y*cur_info.xbytes+x;

            /* select segment */
            port_out(offset >> 16, SEG_SELECT);

            /* get pixel color */
            return graph_mem[offset & 0xFFFF];
        default:
            return -1;
    }
}

#if 0

void vga_drawscansegment (uchar* colors, int x, int y, int length)
{
    /* both length and x must divide with 8 */

    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
            {
                int i, j, k, first, last;
                union bits bytes;
                uchar* address;

                k = 0;
                for(i = 0; i < length; i += 8) {
                    bytes.i = 0;
                    first = i;
                    last  = i+8;
                    for(j = first; j < last; j++)
                       bytes.i = (bytes.i<<1) | color16[colors[j]].i;
                    plane0[k]   = bytes.b.bit0;
                    plane1[k]   = bytes.b.bit1;
                    plane2[k]   = bytes.b.bit2;
                    plane3[k++] = bytes.b.bit3;
                }

                address = graph_mem + (y*cur_info.xdim+x)/8;

                /* disable Set/Reset Register */
                port_out(0x01, GRA_I );
                port_out(0x00, GRA_D );

                /* write to all bits */
                port_out(0x08, GRA_I );
                port_out(0xFF, GRA_D );

                /* select map mask register */
                port_out(0x02, SEQ_I );

                /* write plane 0 */
                port_out(0x01, SEQ_D );
                memcpy(address, plane0, length/8);

                /* write plane 1 */
                port_out(0x02, SEQ_D );
                memcpy(address, plane1, length/8);

                /* write plane 2 */
                port_out(0x04, SEQ_D );
                memcpy(address, plane2, length/8);

                /* write plane 3 */
                port_out(0x08, SEQ_D );
                memcpy(address, plane3, length/8);

                /* restore map mask register */
                port_out(0x0F, SEQ_D );

                /* enable Set/Reset Register */
                port_out(0x01, GRA_I );
                port_out(0x0F, GRA_D );
            }
            break;
        case G640x480x2:
            {
                /* disable Set/Reset Register */
                port_out(0x01, GRA_I );
                port_out(0x00, GRA_D );

                /* write to all bits */
                port_out(0x08, GRA_I );
                port_out(0xFF, GRA_D );

                /* write to all planes */
                port_out(0x02, SEQ_I );
                port_out(0x0F, SEQ_D );

                memcpy(graph_mem + (y*cur_info.xdim+x)/8, colors, length);

                /* restore map mask register */
                port_out(0x0F, SEQ_D );

                /* enable Set/Reset Register */
                port_out(0x01, GRA_I );
                port_out(0x0F, GRA_D );
            }
            break;
        case G320x200x256:
            /* linear addressing - easy and fast */
            memcpy(graph_mem + y*cur_info.xdim+x, colors, length);
            break;
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
            {
                int first, last, offset, pixel, plane;

                /* select map mask register */
                port_out(0x02, SEQ_I);

                for(plane = 0; plane < 4; plane++) {
                    /* select plane */
                    port_out(1 << plane, SEQ_D);

                    pixel = plane;
                    first = (y*cur_info.xdim+x)/4;
                    last  = (y*cur_info.xdim+x+length)/4;
                    for(offset = first; offset < last; offset++) {
                        graph_mem[offset] = colors[pixel];
                        pixel += 4;
                    }
                }
            }
            break;
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
            {
                unsigned long offset;
                int segment, free;

                offset  = y*cur_info.xbytes+x;
                segment = offset >> 16;
                free    = ((segment+1)<<16)-offset;

                if (free < length) {
                    port_out(segment, SEG_SELECT);
                    memcpy(graph_mem + (offset & 0xFFFF), colors, free);
                    port_out(segment+1, SEG_SELECT);
                    memcpy(graph_mem, colors+free, length-free);
                } else {
                    port_out(segment, SEG_SELECT);
                    memcpy(graph_mem + (offset & 0xFFFF), colors, length);
                }

            }
            break;
    }
}


void vga_drawscanline (int line, uchar* colors)
{
    if (cur_mode == G640x480x2)
        vga_drawscansegment(colors, 0, line, cur_info.xbytes); # xbytes = xdim/8
    else
        vga_drawscansegment(colors, 0, line, cur_info.xdim);
}


void vga_drawbox (int x1, int y1, int x2, int y2)
{
    /* we may assume 0 <= x1 <= x2 and 0 <= y1 <= y2 */
    int x1_8 = round_up((unsigned int) x1, 8);
    int x2_8 = round_down((unsigned int) (x2+1), 8);
    if (x1_8 < x2_8)
      { unsigned int scansegment_length = x2_8 - x1_8;
        if (cur_mode == G640x480x2) scansegment_length = scansegment_length/8;
       {uchar colors[scansegment_length];
        { int i;
          int color = cur_color;
          if (cur_mode == G640x480x2) color = (color==0 ? 0x00 : 0xFF);
          for (i=0; i<scansegment_length; i++)
              colors[i] = color;
        }
        { int x, y;
          for (y = y1; y <= y2; y++) {
              for (x = x1; x < x1_8; x++)
                  vga_drawpixel(x, y);
              vga_drawscansegment(colors,x1_8,y,scansegment_length);
              for (x = x2_8; x <= x2; x++)
                  vga_drawpixel(x, y);
        } }
      }}
      else
      { int x, y;
        for (y = y1; y <= y2; y++)
            for (x = x1; x <= x2; x++)
                vga_drawpixel(x, y);
      }
}

# Leider gehen damit die 16-Farben-Modi nicht.
# Test:
# (progn
#   (graph-init 320 200 16) bzw.
#   (graph-init 640 200 16) bzw.
#   (graph-init 640 350 16) bzw.
#   (graph-init 640 480 16)
#   (graph-clear)
#   (graph-clear 10)
#   (graph-box 17 100 54 130 14)
#   (sleep 2)
# )

#endif

void vga_drawhline (int x1, int x2, int y)
{
    /* we may assume 0 <= x1 <= x2 */

    switch (cur_mode) {
        case G320x200x16:
        case G640x200x16:
        case G640x350x16:
        case G640x480x16:
        case G640x480x2:
          { volatile uchar* p1 = &graph_mem[y*cur_info.xbytes + (x1>>3)];
            volatile uchar* p2 = &graph_mem[y*cur_info.xbytes + (x2>>3)];
            volatile uchar* p;

            x1 &= 7;
            x2 &= 7;
            if (p1 == p2)
              { /* select bits */
                port_out(8, GRA_I);
                port_out((0x100 >> x1) - (0x80 >> x2), GRA_D);
                /* read into latch and write dummy back */
                *p2 = *p2;
              }
              else # p1 < p2
              { p = p1;
                /* select bits 7-x1..0 */
                port_out(8, GRA_I);
                port_out((0x100 >> x1) - 1, GRA_D);
                /* read into latch and write dummy back */
                *p = *p;
                if (++p < p2)
                  { /* select bits 7..0 */
                    port_out(8, GRA_I);
                    port_out(0x100 - 1, GRA_D);
                    do { /* read into latch and write dummy back */
                         *p = *p;
                       }
                       while (++p < p2);
                  }
                /* select bits 7..7-x2 */
                port_out(8, GRA_I);
                port_out(0x100 - (0x80 >> x2), GRA_D);
                /* read into latch and write dummy back */
                *p = *p;
              }
          }
          break;
        case G320x200x256:
          { volatile uchar* p1 = &graph_mem[y*320 + x1];
            volatile uchar* p2 = &graph_mem[y*320 + x2];
            volatile uchar* p;

            /* write color to pixels */
            for (p=p1; p<=p2; p++)
                *p = cur_color;
          }
          break;
        case G320x240x256:
        case G320x400x256:
        case G360x480x256:
          { volatile uchar* p1 = &graph_mem[y*cur_info.xbytes + (x1>>2)];
            volatile uchar* p2 = &graph_mem[y*cur_info.xbytes + (x2>>2)];
            volatile uchar* p;
            int i;

            x1 &= 3;
            x2 &= 3;
            if (p1 == p2)
                for (i = x1; i <= x2; i++) {
                    /* select plane */
                    port_out(0x02, SEQ_I);
                    port_out(1 << i, SEQ_D);

                    /* write color to pixel */
                    *p2 = cur_color;
                }
              else # p1 < p2
                for (i = 0; i < 4; i++) {
                    /* select plane */
                    port_out(0x02, SEQ_I);
                    port_out(1 << i, SEQ_D);

                    /* write color to pixels */
                    p = p1;
                    if (i >= x1)
                        *p = cur_color;
                    while (++p < p2)
                        *p = cur_color;
                    if (i <= x2)
                        *p = cur_color;
                }
          }
          break;
        case G640x480x256:
        case G800x600x256:
        case G1024x768x256:
          { unsigned long offset1 = y*cur_info.xbytes + x1;
            unsigned long offset2 = y*cur_info.xbytes + x2;
            volatile uchar* p;
            volatile uchar* p2;
            unsigned long offset;

            if ((offset1 >> 16) == (offset2 >> 16))
              { /* select segment */
                port_out(offset1 >> 16, SEG_SELECT);
                /* write color to pixels */
                offset1 &= 0xFFFF; offset2 &= 0xFFFF;
                for (p = &graph_mem[offset1], p2 = &graph_mem[offset2]; p<=p2; p++)
                    *p = cur_color;
              }
              else
              { /* select segment */
                port_out(offset1 >> 16, SEG_SELECT);
                /* write color to pixels */
                offset1 &= 0xFFFF;
                for (p = &graph_mem[offset1], p2 = &graph_mem[0x10000]; p<p2; p++)
                    *p = cur_color;
                /* select segment */
                port_out(offset2 >> 16, SEG_SELECT);
                /* write color to pixels */
                offset2 &= 0xFFFF;
                for (p = &graph_mem[0], p2 = &graph_mem[offset2]; p<=p2; p++)
                    *p = cur_color;
              }
          }
          break;
    }
}


void vga_drawbox (int x1, int y1, int x2, int y2)
{
    /* we may assume 0 <= x1 <= x2 and 0 <= y1 <= y2 */
    int y;
    for (y = y1; y <= y2; y++)
        vga_drawhline(x1,x2,y);
}


void vga_drawline (int x1, int y1, int x2, int y2)
{
    int dx = x2 - x1;
    int dy = y2 - y1;
    int ax = ABS(dx) << 1;
    int ay = ABS(dy) << 1;
    int sx = (dx >= 0) ? 1 : -1;
    int sy = (dy >= 0) ? 1 : -1;

    int x  = x1;
    int y  = y1;

    if (ax > ay) {
        int d = ay - (ax >> 1);
        while (x != x2) {
            vga_drawpixel(x, y);

            if (d > 0 || d == 0 && sx == 1) {
                y += sy;
                d -= ax;
            }
            x += sx;
            d += ay;
        }
    } else {
        int d = ax - (ay >> 1);
        while (y != y2) {
            vga_drawpixel(x, y);

            if (d > 0 || d == 0 && sy == 1) {
                x += sx;
                d -= ay;
            }
            y += sy;
            d += ax;
        }
    }
    vga_drawpixel(x, y);
}


int vga_getxdim (void)
{
    return cur_info.xdim;
}


int vga_getydim (void)
{
    return cur_info.ydim;
}


int vga_getcolors (void)
{
    return cur_info.colors;
}


# ============================================================================ #


int gr_init (sintL width, sintL height, sintL colors)
{
    int mode = TEXT;

    if (vga_hasmode(G320x200x16))
      { mode = G320x200x16; }
    if (colors <= 16)
      { # choose a 16-color mode if possible
        if (width > 320)
          { if (vga_hasmode(G640x200x16))
              { mode = G640x200x16; }
            if (height > 200 && vga_hasmode(G640x350x16))
              { mode = G640x350x16; }
            if (height > 350 && vga_hasmode(G640x480x16))
              { mode = G640x480x16; }
          }
          else
          { if (height > 200 && vga_hasmode(G320x240x256))
              { mode = G320x240x256; }
            if (height > 240 && vga_hasmode(G320x400x256))
              { mode = G320x400x256; }
            if (height > 400 && vga_hasmode(G360x480x256))
              { mode = G360x480x256; }
            if (width > 360 && vga_hasmode(G640x480x16))
              { mode = G640x480x16; }
      }   }
      else
      { # choose a 256-color mode
        if (vga_hasmode(G320x200x256))
          { mode = G320x200x256; }
        if (height > 200 && vga_hasmode(G320x240x256))
          { mode = G320x240x256; }
        if (height > 240 && vga_hasmode(G320x400x256))
          { mode = G320x400x256; }
        if ((width > 320 || height > 400) && vga_hasmode(G360x480x256))
          { mode = G360x480x256; }
        if (width > 360 && vga_hasmode(G640x480x256))
          { mode = G640x480x256; }
      }
    if ((width > 640 || height > 480) && vga_hasmode(G800x600x256))
      { mode = G800x600x256; }
    if ((width > 800 || height > 600) && vga_hasmode(G1024x768x256))
      { mode = G1024x768x256; }

    if (mode == TEXT)
      return -1;

    flip_mode = TEXT;
    if (vga_setmode(mode) < 0)
      { OS_error(); }

    return 0;
}

void gr_show (void)
{
    if (cur_mode == TEXT)
      { switch_graphics_mode();
        if (cur_mode == TEXT)
          { fehler(
                   DEUTSCH ? "Grafik nicht initialisiert." :
                   ENGLISH ? "graphics not initialized" :
                   ""
                  );
      }   }
}

static struct named_color mono_colors []
  = { {  0, "BLACK" },
      { 15, "WHITE" },
      { 0,  NULL    }
    };
static struct named_color EGA_colors []
  = { {  0, "BLACK"         },
      {  1, "BLUE"          },
      {  2, "GREEN"         },
      {  3, "CYAN"          },
      {  4, "RED"           },
      {  5, "MAGENTA"       },
      {  6, "BROWN"         },
      {  7, "LIGHT-GRAY"    },
      {  8, "DARK-GRAY"     },
      {  9, "LIGHT-BLUE"    },
      { 10, "LIGHT-GREEN"   },
      { 11, "LIGHT-CYAN"    },
      { 12, "LIGHT-RED"     },
      { 13, "LIGHT-MAGENTA" },
      { 14, "YELLOW"        },
      { 15, "WHITE"         },
      { 0,  NULL            }
    };
static struct named_color VGA_colors []
  = { {  0, "BLACK"         },
      {  1, "DARK-BLUE"     },
      {  2, "DARK-GREEN"    },
      {  3, "DARK-CYAN"     },
      {  4, "DARK-RED"      },
      {  5, "DARK-MAGENTA"  },
      {  9, "BLUE"          },
      { 10, "GREEN"         },
      { 11, "CYAN"          },
      { 12, "RED"           },
      { 13, "MAGENTA"       },
      { 14, "YELLOW"        },
      { 15, "WHITE"         },
      { 43, "BROWN"         }, # or 66
      { 26, "DARK-GRAY"     },
      { 29, "LIGHT-GRAY"    },
      { 0,  NULL            }
    };

struct named_color * gr_colors (void)
{
    switch (cur_info.colors)
      { case 2: return &!mono_colors;
        case 16: return &!EGA_colors;
        case 256: return &!VGA_colors;
        default: NOTREACHED
      }
}

#define gr_xdim  cur_info.xdim
#define gr_ydim  cur_info.ydim

int gr_get (sintL x, sintL y)
{
    struct dot_args args;
    gr_show();
    args.x = x; args.y = y;
    if (dot_clipping(&cur_clip,&args))
      return vga_getpixel(args.x,args.y);
      else
      return 0;
}

void gr_dot (sintL color, sintL x, sintL y)
{
    struct dot_args args;
    gr_show();
    args.x = x; args.y = y;
    if (color >= 0 && color < cur_info.colors)
      if (dot_clipping(&cur_clip,&args))
        { if (!(color == cur_color)) vga_setcolor(color);
          vga_drawpixel(args.x,args.y);
        }
}

void gr_box (sintL color, sintL x1, sintL y1, sintL x2, sintL y2)
{
    struct box_args args;
    gr_show();
    args.xa = x1; args.ya = y1; args.xb = x2; args.yb = y2;
    if (color >= 0 && color < cur_info.colors)
      if (box_clipping(&cur_clip,&args))
        { if (!(color == cur_color)) vga_setcolor(color);
          vga_drawbox(args.xa,args.ya,args.xb,args.yb);
        }
}

void gr_clear (sintL color)
{
    gr_show();
    if (color >= 0 && color < cur_info.colors)
      { if (color==0)
          { vga_clear(); }
          else
          { if (!(color == cur_color)) vga_setcolor(color);
            vga_drawbox(cur_clip.x1,cur_clip.y1,cur_clip.x2,cur_clip.y2);
      }   }
}

void gr_line (sintL color, sintL x1, sintL y1, sintL x2, sintL y2)
{
    struct line_args args;
    gr_show();
    args.xa = x1; args.ya = y1; args.xb = x2; args.yb = y2;
    if (color >= 0 && color < cur_info.colors)
      if (line_clipping(&cur_clip,&args))
        { if (!(color == cur_color)) vga_setcolor(color);
          vga_drawline(args.xa,args.ya,args.xb,args.yb);
        }
}

#endif # EMUNIX_PORTABEL

#endif # GRAPHICS_SWITCH

#ifdef GRAPHICS_X

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

static Display* display = NULL;  # X server connection
static int screen;               # the default screen number
#define COLOR_WHITE  WhitePixel(display,screen)
#define COLOR_BLACK  BlackPixel(display,screen)
static Window window;            # window ID
static int x_xdim;               # width of window
static int x_ydim;               # height of window
static GC draw_gc;               # graphics context for drawing
static sintL draw_color;         # current drawing color

int x_fatal_error_handler (Display* display)
{
    fehler("X I/O error");
}

int x_init (sintL width, sintL height, sintL colors)
{
    # Open X display:
    if (display==NULL)
      { display = XOpenDisplay(NULL);
        if (display==NULL)
          { return -1; }
      }
    # Get its default screen number:
    screen = DefaultScreen(display);
    # Set the error handler:
    XSetIOErrorHandler(&x_fatal_error_handler);

    # Open the window:
    {   XFontStruct* fontstruct;   # font descriptor
        XSizeHints*  sizehints;    # size hints for window manager
        XWMHints*    wmhints;      # hints for window manager

        # Load default font:
        fontstruct = XLoadQueryFont(display,"fixed");
        if (fontstruct==NULL)
          { return -1; }

        # Select colors:
        {
            #define NUMCOLORS  16
            static char* colorname [NUMCOLORS] =
              { "WHITE",  "BLACK",    "PINK",    "RED",
                "ORANGE", "YELLOW",   "GREEN",   "TURQUOISE",
                "BLUE",   "LAVENDER", "MAGENTA", "PURPLE",
                "GOLD",   "BROWN",    "CYAN",    "LIGHT GRAY"
              };
            Xcolor xcolor [NUMCOLORS];  # the colors
            Xcolor unused_color;
            int numcolors = (colors <= 2 ? 2 : NUMCOLORS);
            int i;
            for (i = 0; i < numcolors; i++)
                XAllocNamedColor (display, DefaultColormap(display,screen),
                                           colorname[i],
                                           &xcolor[i],   # color on screen
                                           &unused_color # exact color: unused
                                 );
        }

        # Size hints for the Window Manager:
        sizehints = XAllocSizeHints();
        if (sizehints==NULL)
          return -1;
        sizehints->flags = PSize | PPosition;
        # size: width, height
        sizehints->width = width; sizehints->height = height;
        # position: centered
        sizehints->x = DisplayWidth(display,screen) - sizehints->width;
        if (sizehints->x < 0) sizehints->x = 0;
        sizehints->x >>= 1;
        sizehints->y = DisplayHeight(display,screen) - sizehints->height;
        if (sizehints->y < 0) sizehints->y = 0;
        sizehints->y >>= 1;

        # Get a window.
        # Background will be COLOR_WHITE, Foreground will be COLOR_BLACK.
        window = XCreateSimpleWindow (display, RootWindow(display,screen),
                                      sizehints->x, sizehints->y,
                                      sizehints->width, sizehints->height,
                                      1, # border width
                                      COLOR_BLACK, # border color
                                      COLOR_WHITE  # background color
                                     );

        # Attach a label to the window.
        # XSetStandardProperties oder XSetWMProperties ??
        XSetStandardProperties(display,window,
                               "CLISP graphics", # window label
                               "CLISP graphics", # icon label
                               None,
                               NULL, 0, # argv and argc
                               sizehints # size hints
                              );

        # window manager hints
        wmhints = XAllocWMHints();
        if (wmhints==NULL)
          { return -1; }
        wmhints->flags = InputHint | StateHint;
        wmhints->input = FALSE; wmhints->state = NormalState;
        XSetWMHints(display,window, wmhints);

        # Set up colors.
        {   var XSetWindowAttributes xswa;
            xswa.colormap = DefaultColormap(display,screen);
            XChangeWindowAttributes (display,window, CWColormap, &xswa);
        }

        # Set up graphics context.
        {   var XGCValues gcv;
            gcv.font = fontstruct->fid;
            gcv.background = COLOR_WHITE;
            gcv.foreground = draw_color = COLOR_BLACK;
            gcv.plane_mask = AllPlanes;
            draw_gc = XCreateGC (display,window,
                                 GCPlaneMask | GCFont | GCForeground | GCBackground,
                                 &gcv
                                );
        }

        XSelectInput (display,window, StructureNotifyMask | ExposureMask);

        # Finally, make the window visible.
        XMapWindow (display,window);

        # Don't laugh, but we have to wait until the window really gets visible.
        loop # a small event-dispatch loop
          { var XEvent event;
            XWindowEvent (display,window, StructureNotifyMask, &event);
            if (event.type == MapNotify)
              { XClearWindow(display,window); break; }
          }

    }
    return 0;
}

# NB: XCloseDisplay(display); ist nicht ntig. Das erledigt der Window-Manager
# von selbst, wenn unser Programm beendet wird.
# XFreeGC(display,draw_gc);
# XUnloadFont(display,fontstruct->fid);
# XCloseDisplay(display);

void x_show (void)
{ # Do nothing, even if the window is iconified.
}

void x_dot (sintL color, sintL x, sintL y)
{
    if (!(color==draw_color))
      { XSetForeground(display,draw_gc,color); draw_color = color; }
    XDrawPoint(display,window, draw_gc, x,y);
}

void x_box (sintL color, sintL x1, sintL y1, sintL x2, sintL y2)
{
    if (!(color==draw_color))
      { XSetForeground(display,draw_gc,color); draw_color = color; }
    if (x1 > x2) swap(sintL,x1,x2);
    if (y1 > y2) swap(sintL,y1,y2);
    XDrawRectangle(display,window, draw_gc, x1,y1, x2-x1,y2-y1);
}

void x_line (sintL color, sintL x1, sintL y1, sintL x2, sintL y2)
{
    if (!(color==draw_color))
      { XSetForeground(display,draw_gc,color); draw_color = color; }
    XDrawLine(display,window, draw_gc, x1,y1, x2,y2);
}

Fehlen:
# x_colors()                    liefert einen Array von benannten Farben.
# x_clear(color);               lscht den gesamten Grafik-Bildschirm.
# x_get(x,y)                    liefert die Farbe des Punktes (x,y)

x_text() { XDrawString (display,window, ...) }

#endif # GRAPHICS_X

#ifdef GRAPHICS_SUN

#include <suntool/sunview.h>
#include <suntool/canvas.h>
#include <suntool/textsw.h>
#include <suntool/panel.h>

static Frame screen;
static Canvas canvas;
static Pixwin* pw;
static Pixfont* font;

void sun_init (sintL width, sintL height, sintL colors)
{
    screen = window_create (NULL,
                            FRAME,
                            FRAME_LABEL, "CLISP graphics",
                          # WIN_ERROR_MSG, "must be in SunView/SunTools",
                            0);
    canvas = window_create (screen,
                            CANVAS,
                            WIN_WIDTH, width,
                            WIN_HEIGHT, height,
                            0);
    window_fit(screen);
    pw = canvas_pixwin(canvas);
    font = pw_pfsysopen();
    pw_vector(...);
    pw_char(pw,?,?,PIX_SRC|PIX_DST,font,ch);
    window_main_loop(screen);
}

#endif # GRAPHICS_SUN


# Lisp-Funktionen:

# (SYS::GRAPH-INIT [width [height [colors]]])
LISPFUN(gr_init,0,3,norest,nokey,0,NIL)
{ { var reg2 object width = STACK_2;
    var reg4 object height = STACK_1;
    var reg5 object colors = STACK_0;
    var reg1 sintL w = (eq(width,unbound) ? 400 : I_to_L(width));
    var reg3 sintL h = (eq(height,unbound) ? 3*(w>>2) : I_to_L(height));
    if (gr_init(w, h, (eq(colors,unbound) ? 16 : I_to_L(colors))) < 0)
      { pushSTACK(S(gr_init));
        fehler(
               DEUTSCH ? "~: Kann nicht in Grafik-Modus schalten." :
               ENGLISH ? "~: cannot switch to graphics mode" :
               ""
              );
      }
    skipSTACK(3);
  }
  # Aliste der Farben als Wert:
  { var reg1 struct named_color * p = gr_colors();
    var reg3 uintC count = 0;
    while (p->name) { p++; count++; }
    pushSTACK(NIL);
    dotimesC(count,count,
      {  p--;
         pushSTACK(allocate_cons());
         pushSTACK(allocate_cons());
       { var reg4 object name = intern_keyword(asciz_to_string(p->name));
         var reg2 object acons = popSTACK();
         Car(acons) = name; Cdr(acons) = fixnum(p->color);
        {var reg3 object lcons = popSTACK();
         Car(lcons) = acons; Cdr(lcons) = STACK_0;
         STACK_0 = lcons;
      }}});
    value1 = popSTACK(); mv_count=1;
} }

# (SYS::GRAPH-SHOW)
LISPFUNN(gr_show,0)
{ gr_show();
  value1 = NIL; mv_count=0;
}

# (SYS::GRAPH-CLEAR [color])
LISPFUN(gr_clear,0,1,norest,nokey,0,NIL)
{ var reg1 object color = popSTACK();
  gr_clear(eq(color,unbound) ? 0 : I_to_L(color));
  value1 = NIL; mv_count=0;
}

# (SYS::GRAPH-DIMS)
LISPFUNN(gr_dims,0)
{ value1 = fixnum(gr_xdim); value2 = fixnum(gr_ydim); mv_count=2; }

# (SYS::GRAPH-DOT x y [color])
LISPFUN(gr_dot,2,1,norest,nokey,0,NIL)
{ var reg1 object color = STACK_0;
  var reg2 sintL x = I_to_L(STACK_2);
  var reg3 sintL y = I_to_L(STACK_1);
  if (eq(color,unbound))
    { value1 = fixnum(gr_get(x,y)); }
    else
    { gr_dot(I_to_L(color),x,y); value1 = color; }
  mv_count=1;
  skipSTACK(3);
}

# (SYS::GRAPH-BOX x1 y1 x2 y2 color)
LISPFUNN(gr_box,5)
{ var reg1 object color = STACK_0;
  var reg2 sintL x1 = I_to_L(STACK_4);
  var reg3 sintL y1 = I_to_L(STACK_3);
  var reg4 sintL x2 = I_to_L(STACK_2);
  var reg5 sintL y2 = I_to_L(STACK_1);
  gr_box(I_to_L(color),x1,y1,x2,y2);
  value1 = NIL; mv_count=0;
  skipSTACK(5);
}

# (SYS::GRAPH-LINE x1 y1 x2 y2 color)
LISPFUNN(gr_line,5)
{ var reg1 object color = STACK_0;
  var reg2 sintL x1 = I_to_L(STACK_4);
  var reg3 sintL y1 = I_to_L(STACK_3);
  var reg4 sintL x2 = I_to_L(STACK_2);
  var reg5 sintL y2 = I_to_L(STACK_1);
  gr_line(I_to_L(color),x1,y1,x2,y2);
  value1 = NIL; mv_count=0;
  skipSTACK(5);
}

#endif # GRAPHICS

