/*
 *	complex_ls.C -- A more complex directory listing program.
 */

/*
 *	complex_ls takes simple_ls and expands it to sort the directories, use X selections, and allow one to view text
 *	files.
 *
 * Functions used:
 *
 */

/* Header Files */

#ifdef ultrix
extern "C" {
	char *strdup (const char *);
}
#endif

# include <OI/oi.H>
# include <sys/types.h>
# include <sys/stat.h>
# include <string.h>
#if defined(ultrix) || defined(SVR4) || defined(hpux)
#include <unistd.h>
#else
#include <sysent.h>
#endif
# include <dirent.h>
# include <stdlib.h>
	
/* Macro Definitions */

/* Variable Definitions */

/* Function Declarations */

void make_dir_box (OI_scroll_box *, char *);

/*
 *	File
 *
 *	The file class lets me store information about the file, such as name and mode.
 */
class File {
 public:
				File (char *fn, u_short mod) : filename (strdup (fn)), mode (mod) {}
				~File()				{ free ((char *) filename); }
	const char		*filename;
	const u_short		mode;
};

/*
 *	FileArray
 *
 *	This is just an array of File.  It lets me insert, sort, and layout the icons.  Mostly, so I do not have
 *	all this code in line.
 */
class FileArray {
 public:
				FileArray(int);
				~FileArray();
	void			sort();
	void			layout_files(OI_box *);
	void			insert (File *);
 private:
	int			num_elem;
	int			num_alloc;
	File			**files;
};

/*
 *	Icon
 *
 *	The icon class is a box that has a glyph and a static text label in it.  We use this as a base
 *	class for icons we really display.
 */
class Icon : public OI_box {
 public:
			Icon (const char *);
	virtual		~Icon();
 protected:
	void		create_icon (char *, char *, OI_number, OI_number, char *);
	virtual void	click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number);
	char		*pathname;
};
void Icon::click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number) {}

/*
 *	DirIcon
 *
 *	DirIcon is used to display directories.  It is derived from icon and adds a static directory icon that gets
 *	cloned for every directory that we have.
 */
class DirIcon : public Icon {
 public:
			DirIcon (const char *dir_name);
			~DirIcon();
 private:
	void		click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number);
};
DirIcon::~DirIcon() { }

/*
 *	ExecIcon
 *
 *	ExecIcon is used to display executable files.  It is derived from icon and adds a static directory icon that gets
 *	cloned for every executable that we have.
 */
class ExecIcon : public Icon {
 public:
			ExecIcon (const char *exec);
			~ExecIcon();
 private:
	void	click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number);
};
ExecIcon::~ExecIcon() { }

/*
 *	AsciiIcon
 *
 *	AsciiIcon is used to display normal files.  It is derived from icon and adds a static directory icon that gets
 *	cloned for every normal file that we have.
 */
class AsciiIcon : public Icon {
 public:
			AsciiIcon (const char *ascii);
			~AsciiIcon ();
 private:
	void	click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number);
};
AsciiIcon::~AsciiIcon () { }

/*
 *	FileArray::FileArray
 *
 *	Constructor for FileArray.  Preallocate the number of entries that the user wants.
 */
FileArray::FileArray (int to_alloc)
{
	num_elem = 0;
	num_alloc = to_alloc;
	files = (File **) malloc (sizeof (File *) * num_alloc);
	for (int i = 0; i < num_alloc; i++)
		files[i] = 0;
}

/*
 *	FileArray::~FileArray
 *
 *	Cleans up memory that we have in use.
 */
FileArray::~FileArray()
{
	for (int i = 0; i < num_elem; i++)
		delete files[i];
	free ((char *)files);
}

/*
 *	file_comp
 *
 *	Helper function for FileArray::sort.  Called by qsort to compare two files.
 */
int file_comp (const void *a, const void *b)
{
	File *f1 = *(File **) a;
	File *f2 = *(File **) b;

	return strcmp (f1->filename, f2->filename);
}

/*
 *	FileArray::sort
 *
 *	Just does a qsort of the allocated elements of the array.
 */ 
void FileArray::sort()
{
	qsort (files, num_elem, sizeof (File *), &file_comp);
}

/*
 *	FileArray::insert
 *
 *	Inserts an element into the array.  It will reallocate the array if we grow too much.
 */
void FileArray::insert (File *f)
{
	if (num_elem == num_alloc) {
		files = (File **) realloc ((char *)files, sizeof (File *) * num_alloc * 2);
		num_alloc *= 2;
	}

	files[num_elem++] = f;
}

/*
 *	FileArray::layout_files
 *
 *	I'm not sure this belongs in the FileArray class, but it is here for now.  It lays out all of the files in icons
 *	that we create.
 */
void FileArray::layout_files(OI_box *bp)
{
	int			row;
	int			col;
	int			i;
	Icon			*icon;
	int			nrows;
	
	row = 1;
	col = 1;
	nrows = (num_elem + 3) / 4;
	for (i = 0; i < num_elem; i++) {
		if ((files[i]->mode & S_IFMT) == S_IFDIR)
			icon = new DirIcon (files[i]->filename);
		else if (files[i]->mode & S_IEXEC)
			icon = new ExecIcon (files[i]->filename);
		else
			icon = new AsciiIcon (files[i]->filename);
		icon->layout_associated_object (bp, col, row, OI_ACTIVE);
		row++;
		if (row > nrows) {
			row = 1;
			col++;
		}
	}
}

/*
 *	Icon::Icon
 *
 *	Constructor for Icon class.  Makes a box, puts an icon and label in it.
 */
Icon::Icon (const char *lab) : OI_box ("iconBox", 1, 1)
{
	OI_static_text	*stp;

	/*
	 *	Set frame width to 0 and make this layout.
	 */
	set_layout (OI_layout_row);
	set_frame_width (0);
	
	/*
	 *	Create a centered label.
	 */
	stp = oi_create_static_text ("label", lab);
	stp -> set_gravity (OI_grav_center);
	stp -> disallow_cut_paste();
	stp -> layout_associated_object (this, 2, 1, OI_ACTIVE);


	/*
	 *	Set the pathname.
	 */
	pathname = strdup ( lab );

	/*
	 *	Set the click functions.
	 */
	set_click (this, (OI_click_memfnp) &Icon::click);
	stp->set_click (this, (OI_click_memfnp) &Icon::click);
}

/*
 *	Icon::~Icon
 *
 *	Cleanup.
 */
Icon::~Icon()
{
	free (pathname);
}

/*
 *	Icon::create_icon
 *
 *	Creates an icon of the specifed color, using the bits bm and bm_bg.  Both of these bitmaps must be the same
 *	size, or the glyph code hickups.
 */
void Icon::create_icon (char *bm, char *bm_bg, OI_number width, OI_number height, char *color)
{
	OI_glyph *gp;
	OI_pic_spec_mask *masks[2];
	PIXEL bg_pxl;
	
	/*
	 *	If the connection is color, allocate the named color.  Otherwise just use the background color.
	 */
	if (connection()->is_color())
		connection()->str_color (color, &bg_pxl);
	else
		bg_pxl = bkg_pixel();
		
	/*
	 *	Create the masks, then create the glyph.
	 */
	masks[0] = oi_create_pic_data_mask (bm, OI_PIC_FG, OI_PIC_NONE, 0, bg_pxl);
	masks[1] = oi_create_pic_data_mask (bm_bg, OI_PIC_NONE, OI_PIC_FG, bg_pxl, 0);
	gp = oi_create_glyph ("icon", 2, masks, "", width, height, OI_YES, OI_NO);

	gp->set_click (this, (OI_click_memfnp) &Icon::click);
	gp->layout_associated_object (this, 1, 1, OI_ACTIVE);
}

/*
 *	DirIcon::DirIcon
 *
 *	Create the icon for this guy.  Most of the work is done by Icon.
 */
DirIcon::DirIcon (const char *name) : Icon (name)
{
#include "../bitmaps/dir.xbm"
#include "../bitmaps/dir-bg.xbm"
	create_icon (dir_bits, dir_bg_bits, dir_width, dir_height, "MediumPurple");
}

/*
 *	DirIcon::click
 *
 *	Overrides the icon click function.  Here we set the directory.  One thing we might want to do is to set the title
 *	of the application to the directory.
 */
void DirIcon::click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number)
{
	OI_app_window	*top_level;
	OI_d_tech	*objp;
	OI_d_tech	*kid;
	OI_d_tech	*tmp;
	OI_scroll_box	*dir_box;
	
	/*
	 *	Find our app window.  Then, set the working cursor on it to give the user feedback.  Also find icon box.
	 */
	dir_box = (OI_scroll_box *) ancestor ("dirBox");
	top_level = (OI_app_window *) ancestor("topLevel");
	top_level -> set_working();

	/*
	 *	Find the scroll box's interior box.  We set it (and by implication all its children) to NOT_DISPLAYED
	 *	so the user won't get confused.
	 */
	objp = dir_box->object_box();
	objp->set_state (OI_NOT_DISPLAYED);

	/*
	 *	Now the magic.  We walk through all the children of the interior box.  This should be all of the icons that
	 *	were there when the user clicked.  Take special notice of test tmp == this.  We need to do a delete_all()
	 *	on all the icons, except "this" one.  "This" one needs to have a delete_all_delayed().  One might think that
	 *	you could do a delete_all_delayed() on all the icons, but that doesn't work like you'd think.  Only the first
	 *	delete_all_delayed() is honored, so we have to make this test.
	 */
	kid = objp -> next_child (NULL);
	while (kid) {
		tmp = kid;
		kid = objp->next_child(kid);
		if (tmp == this)
			tmp->delete_all_delayed();
		else
			tmp->delete_all();
	}

	make_dir_box (dir_box, pathname);

	/*
	 *	Turn the icons back on.  Then remove the working cursor (and re-enable clicks to the app window).
	 */
	objp->set_state (OI_ACTIVE);
	top_level -> clear_working();
}

/*
 *	ExecIcon::ExecIcon
 *
 *	Create the icon for this guy.  Most of the work is done by Icon.
 */
ExecIcon::ExecIcon (const char *name) : Icon (name)
{
#include "../bitmaps/exec.xbm"
#include "../bitmaps/exec-bg.xbm"
	create_icon (exec_bits, exec_bg_bits, exec_width, exec_height, "Plum");
}

/*
 *	ExecIcon::click
 *
 *	Overrides the icon click function.  Just execute this command.
 */
void ExecIcon::click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number)
{
	char		command[MAXPATHLEN + 4];

	sprintf (command, "%s &", pathname);
	system (command);
}
 
/*
 *	AsciiIcon::AsciiIcon
 *
 *	Create the icon for this guy.  Most of the work is done by Icon.
 */
AsciiIcon::AsciiIcon (const char *name) : Icon (name)
{
#include "../bitmaps/ascii.xbm"
#include "../bitmaps/ascii-bg.xbm"
	create_icon (ascii_bits, ascii_bg_bits, ascii_width, ascii_height, "PaleTurquoise");
}

/*
 *	AsciiIcon::click
 *
 *	Overrides the icon click function.  Create a scroll text in a dialog box to display the file to the user.
 */
void AsciiIcon::click (OI_d_tech *, void *, OI_number, OI_number, OI_number, OI_number, OI_number)
{
	OI_dialog_box		*dbp;
	OI_scroll_text		*stp;

	dbp = oi_create_dialog_box ("fileDB", 1, 1);
	dbp->set_layout (OI_layout_row);
	dbp->disallow_pushpin();
	dbp->set_title("File");
	dbp->set_longterm(pathname);
	dbp->buttons()->numbered_cell(1)->del();
	dbp->buttons()->numbered_cell(0)->set_label ("Dismiss");
	dbp->buttons()->numbered_cell(0)->change_action (dbp, (OI_action_memfnp)&OI_dialog_box::delete_all_delayed, NULL);
		
	stp = oi_create_scroll_text ("fileViewer", OI_SCROLL_BAR_BOTH, 24, 80, 24, 80);
	stp->allow_auto_controller_visibility();
	stp->set_text_to_file (pathname);
	stp->set_size_track(OI_size_track_full);
	stp->disallow_kb_input();
	stp->layout_associated_object (dbp, 1, 1, OI_ACTIVE);

	dbp->set_associated_object (dbp->root(), OI_DEF_LOC, OI_DEF_LOC, OI_ACTIVE);
}

/*
 *	make_dir_box
 *
 *	Creates a box that has all the entries in the directory in it.  No sorting of the directory is done just yet,
 *	as that would obscure the OI techniques being demonstrated.
 */
void make_dir_box (OI_scroll_box *boxp, char *dir)
{
	struct stat		sb;
	DIR			*dirp;
	struct dirent		*dirent;
	FileArray		*files;
	char			path[MAXPATHLEN];

	
	/*
	 *	Stat the directory to see if it is there.
	 */
	if (stat (dir, &sb)) {
		fprintf (stderr, "Can't stat directory %s\n", dir);
		exit (1);
	}

	/*
	 *	Check to see if it is a directory.  If not fail.
	 */
	if ((sb.st_mode & S_IFMT) != S_IFDIR) {
		fprintf (stderr, "%s is not a directory", dir);
		exit (1);
	}

	/*
	 *	chdir do the directory and set the title of this app window to be the current directory.
	 */
	if (chdir (dir)) {
		perror (dir);
		exit (1);
	}	
	boxp->app_window()->set_longterm (path);

	/*
	 *	A suspend layout is done to avoid O(n^2) layout behaviour.
	 */
	boxp->suspend_layout();

	/*
	 *	Open the directory.  While this shouldn't fail, I'll be paranoid just in case...
	 */
	if ((dirp = opendir (".")) == NULL) {
		fprintf (stderr, "Logic says this can't happen, but I can't opendir %s\n", dir);
		exit (1);
	}

	/*
	 *	Read the directory in.  For each file, make a new icon specific to each type of file.  Then layout the new icon.
	 *	Be really dumb about placement of icons.  A more advanced program would sort them and arrange them into n
	 *	columns.
	 */
	files = new FileArray(100);
	readdir (dirp);		// Skip .
	while (dirent = readdir (dirp)) {
		if (stat (dirent->d_name, &sb))
			continue;
		files->insert(new File (dirent->d_name, sb.st_mode));
	}
	closedir (dirp);

	/*
	 *	Now sort and layout all these glyphs.
	 */
	files->sort();
	files->layout_files(boxp->object_box());
	delete files;
	
	/*
	 *	The resume layout here is supposed to be faster since we only layout each object once, not n times.
	 */
	boxp->resume_layout();
}

/*
 *	Main
 *
 *	Main program.
 */
void
main(int argc, char *argv[])
{
	OI_connection	*conp;
	OI_app_window	*top_level;
	OI_scroll_box	*dir_box;

	/*
	 *	Open a connection.
	 */
	if (conp = OI_init (&argc, argv, "Complex Directory Listing program"))
	{
		/*
		 *	Create the OI_tests main window.  Make it row layout.
		 */
		top_level = oi_create_app_window("topLevel",1,1,"Directory");
		top_level->set_layout (OI_layout_row);

		/*
		 *	Create a box to hold the directory.  We use a scroll box that size tracks for the box.
		 */
		dir_box = oi_create_scroll_box ("dirBox", OI_SCROLL_BAR_BOTH, 30, 30, 1, 1, 10, 10);
		dir_box->set_layout (OI_layout_column);
		dir_box->set_size_track (OI_size_track_full);

		/*
		 *	Create a new directory object.
		 */
		dir_box->layout_associated_object (top_level, 1, 1, OI_ACTIVE);
		make_dir_box (dir_box, argc > 1 ? argv[1] : ".");
		
		/*
		 *	OK, display main window.
		 */
		top_level->set_associated_object(top_level->root(),OI_DEF_LOC, OI_DEF_LOC,OI_ACTIVE);
		OI_begin_interaction();
	}

	/*
	 * Cleanup.  Make sure that we cleanup the library.
	 */
	OI_fini();
}
