#ifndef _SECLIB_H_
#define _SECLIB_H_

/*
    Windows NT Security functions library.
    Copyright (C) 1995  Jeremy R. Allison

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Log:	seclib.h,v $
 * Revision 1.2  95/10/17  18:15:40  18:15:40  jra (Jeremy Allison)
 * Added lsa code.
 * 
 * Revision 1.1  1995/06/30  18:32:55  jra
 * Initial revision
 *^M
*/

#include <windows.h>
#include <tchar.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

// Kernel object security utility functions.

/* Add access allowed ACE to a kernel object. */
extern BOOL AddAllowedSIDKernel( HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL AddAllowedNameKernel( HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );
/* Add access denied ACE to a kernel object */
extern BOOL AddDeniedSIDKernel( HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL AddDeniedNameKernel( HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );

// File security utility functions.

/* Add access allowed ACE to a file. */
extern BOOL AddAllowedSIDFile( const TCHAR *, PSID, ACCESS_MASK, BYTE );
extern BOOL AddAllowedNameFile( const TCHAR *, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );
/* Add access denied ACE to a file */
extern BOOL AddDeniedSIDFile( const TCHAR *, PSID, ACCESS_MASK, BYTE );
extern BOOL AddDeniedNameFile( const TCHAR *, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );

// User Object security utility functions.

/* Add access allowed ACE to a user object. */
extern BOOL AddAllowedSIDUser( HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL AddAllowedNameUser( HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );
/* Add access denied ACE to a user object */
extern BOOL AddDeniedSIDUser( HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL AddDeniedNameUser( HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );

// Registry key security utility functions.

/* Add access allowed ACE to a Registry key. */
extern BOOL AddAllowedSIDRegistry( HKEY, PSID, ACCESS_MASK, BYTE );
extern BOOL AddAllowedNameRegistry( HKEY, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );
/* Add access denied ACE to a Registry key */
extern BOOL AddDeniedSIDRegistry( HKEY, PSID, ACCESS_MASK, BYTE );
extern BOOL AddDeniedNameRegistry( HKEY, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );

/* File and directory permission manipulation routines */
extern BOOL AddDirectoryAccessPermissionsByName( const TCHAR *, const TCHAR *, const TCHAR *,
					BOOL, const TCHAR *, const TCHAR *);
extern BOOL AddDirectoryAccessPermissionsBySID( const TCHAR *, PSID, BOOL, const TCHAR *, const TCHAR *);

extern BOOL AddFileAccessPermissionsByName( const TCHAR *, const TCHAR *, const TCHAR *, const TCHAR *);
extern BOOL AddFileAccessPermissionsBySID( const TCHAR *, PSID, const TCHAR *);

/* #defines as convenience routines for manipulating directories in the same
	way file manager does */
#define DIRECTORY_LIST_PERM (_TEXT("RX"),0)
#define DIRECTORY_READ_PERM (_TEXT("RX"),_TEXT("RX"))
#define DIRECTORY_ADD_PERM (_TEXT("WX"),0)
#define DIRECTORY_ADD_AND_READ_PERM (_TEXT("RWX"),_TEXT("RX"))
#define DIRECTORY_CHANGE_PERM (_TEXT("RWXD"),_TEXT("RWXD"))
#define DIRECTORY_NO_ACCESS_PERM (_TEXT("N"),_TEXT("N"))
#define DIRECTORY_FULL_ACCESS_PERM (_TEXT("F"),_TEXT("F"))

#define FILE_NO_ACCESS_PERM (_TEXT("N"))
#define FILE_READ_PERM (_TEXT("RX"))
#define FILE_CHANGE_PERM (_TEXT("RWXD"))
#define FILE_FULL_ACCESS_PERM (_TEXT("F"))

/*
 * Type of a security descriptor handle.
 */
typedef PVOID SECURITY_DESCRIPTOR_HANDLE;

/* Create a SECURITY_DESCRIPTOR_HANDLE owned by the calling process */
extern BOOL CreateSecurityDescriptorHandle( SECURITY_DESCRIPTOR_HANDLE *);

/* Create a SECURITY_DESCRIPTOR_HANDLE owned by the calling process */
extern BOOL CreateSecurityDescriptorHandleFromSecurityDescriptor( SECURITY_DESCRIPTOR_HANDLE *, const SECURITY_DESCRIPTOR *, 
	const GENERIC_MAPPING *);

/* Functions to initialize a SECURITY_DESCRIPTOR_HANDLE from various
   securable NT objects */
extern BOOL CreateSecurityDescriptorHandleFromFile( SECURITY_DESCRIPTOR_HANDLE *, const TCHAR *, 
		const SECURITY_INFORMATION);
extern BOOL CreateSecurityDescriptorHandleFromKernelObject( SECURITY_DESCRIPTOR_HANDLE *, const HANDLE, 
		const SECURITY_INFORMATION);
extern BOOL CreateSecurityDescriptorHandleFromUserObject( SECURITY_DESCRIPTOR_HANDLE *, const HANDLE,
		const SECURITY_INFORMATION);
extern BOOL CreateSecurityDescriptorHandleFromRegistryKey( SECURITY_DESCRIPTOR_HANDLE *, const HKEY,
		const SECURITY_INFORMATION);

/* Change the owner of the SECURITY_DESCRIPTOR_HANDLE */
extern BOOL SetSecurityDescriptorOwnerByName(SECURITY_DESCRIPTOR_HANDLE, const TCHAR *, const TCHAR *);
extern BOOL SetSecurityDescriptorOwnerBySID(SECURITY_DESCRIPTOR_HANDLE, const SID *);

/* Change the PRIMARY_GROUP of the SECURITY_DESCRIPTOR_HANDLE */
extern BOOL SetSecurityDescriptorGroupByName(SECURITY_DESCRIPTOR_HANDLE, const TCHAR *, const TCHAR *);
extern BOOL SetSecurityDescriptorGroupBySID(SECURITY_DESCRIPTOR_HANDLE, const SID *);

/* Convert the SECURITY_DESCRIPTOR_HANDLE to an absolute SECURITY_DESCRIPTOR.
   This will allocate memory, and so must be paired with a call to
   FreeSecurityDescriptorFromHandle() 
*/
extern BOOL AllocateSecurityDescriptorFromHandle(SECURITY_DESCRIPTOR_HANDLE, SECURITY_DESCRIPTOR **);

/* Frees the SECURITY_DESCRIPTOR allocated by previous call to AllocateSecurityDescriptorFromHandle() */
extern void FreeSecurityDescriptorFromHandle(SECURITY_DESCRIPTOR_HANDLE);

/* Deletes the SECURITY_DESCRIPTOR_HANDLE */
extern void CloseSecurityDescriptorHandle(SECURITY_DESCRIPTOR_HANDLE);

/* Adds an allowed ACE to a SECURITY_DESCRIPTOR_HANDLE */
extern BOOL AddAllowedSID( SECURITY_DESCRIPTOR_HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL AddAllowedName( SECURITY_DESCRIPTOR_HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE);

/* Adds a denied ACE to a SECURITY_DESCRIPTOR_HANDLE */
extern BOOL AddDeniedSID( SECURITY_DESCRIPTOR_HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL AddDeniedName( SECURITY_DESCRIPTOR_HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );

/* Removes an allowed ACE from a SECURITY_DESCRIPTOR_HANDLE */
extern BOOL RemoveAllowedSID( SECURITY_DESCRIPTOR_HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL RemoveAllowedName( SECURITY_DESCRIPTOR_HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE);

/* Removes a denied ACE from a SECURITY_DESCRIPTOR_HANDLE */
extern BOOL RemoveDeniedSID( SECURITY_DESCRIPTOR_HANDLE, PSID, ACCESS_MASK, BYTE );
extern BOOL RemoveDeniedName( SECURITY_DESCRIPTOR_HANDLE, const TCHAR *, const TCHAR *, ACCESS_MASK, BYTE );

/* Remove all ACL entries from a SECURITY_DESCRIPTOR_HANDLE. FreeSecurityDescriptorFromHandle()
   must have been called first.
*/
extern BOOL SetSecurityDescriptorHandleACLToNoAccess( SECURITY_DESCRIPTOR_HANDLE );
extern BOOL SetSecurityDescriptorHandleACLToAllAccess( SECURITY_DESCRIPTOR_HANDLE );

/* Initialize a SECURITY_ATTRIBUTES structure from a SECURITY_DESCRIPTOR_HANDLE */
extern BOOL InitializeSecurityAttributesFromHandle(SECURITY_ATTRIBUTES *,
					SECURITY_DESCRIPTOR_HANDLE, BOOL);

extern BOOL TakePrivilege( const TCHAR *priv_name);
extern BOOL ReleasePrivilege( const TCHAR *priv_name);
extern BOOL ChangeFileOwner( const TCHAR *, const TCHAR *, const TCHAR *);
extern BOOL TakeFileOwnership( const TCHAR *file);

extern BOOL AddUserRightToName(const TCHAR *name, const TCHAR *mach, const TCHAR *privname);
extern BOOL AddUserRightToSid(PSID psid, const TCHAR *mach, const TCHAR *privname);
extern BOOL RemoveUserRightFromName(const TCHAR *name, const TCHAR *mach, const TCHAR *privname);
extern BOOL RemoveUserRightFromSid(PSID psid, const TCHAR *mach, const TCHAR *privname);
 
#ifdef __cplusplus
};
#endif

#endif
