/*
    Windows NT Security functions library.
    Copyright (C) 1995  Jeremy R. Allison

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Log: sechand.cpp,v $
// Revision 1.1  1995/06/30  18:32:55  jra
// Initial revision
//^M
*/

#include "secdesc.h"
#include "seclib.h"
#include "autoheap.h"
#include "objhandle.h"

//
// C Linkage routines for SECURITY_DESCRIPTOR_HANDLE objects.
//

extern "C" {

// Create a SECURITY_DESCRIPTOR_HANDLE owned by the calling thread
// and with the same primary group as the calling thread.

BOOL CreateSecurityDescriptorHandle( SECURITY_DESCRIPTOR_HANDLE *psh) 
{
	DWORD err = 0;
	*psh = 0;
	try {
		secDesc *psecd = new secDesc;
		*psh = (SECURITY_DESCRIPTOR_HANDLE)psecd;
	}
	catch (DWORD caerr) {
		err = caerr;
	}
	if(err)	{
		SetLastError(err);
		return FALSE;
	}
	return TRUE;
}

// Create a SECURITY_DESCRIPTOR_HANDLE from an existing SECURITY_DESCRIPTOR

BOOL CreateSecurityDescriptorHandleFromSecurityDescriptor( SECURITY_DESCRIPTOR_HANDLE *psh,
	const SECURITY_DESCRIPTOR *psd, const GENERIC_MAPPING *pgm) 
{
	DWORD err = 0;
	*psh = 0;
	try {
		secDesc *psecd = new secDesc( psd, pgm );
		*psh = (SECURITY_DESCRIPTOR_HANDLE)psecd;
	}
	catch (DWORD caerr) {
		err = caerr;
	}
	if(err)	{
		SetLastError(err);
		return FALSE;
	}
	return TRUE;
}

// Create a SECURITY_DESCRIPTOR_HANDLE from a FILE or DIRECTORY object

BOOL CreateSecurityDescriptorHandleFromFile( SECURITY_DESCRIPTOR_HANDLE *psh,
	const TCHAR *path, const SECURITY_INFORMATION si) 
{
	DWORD error = 0;
	*psh = 0;
	SECURITY_DESCRIPTOR *psd = 0;
	DWORD size;
	autoHeapChar sdptr;
	static GENERIC_MAPPING fgm = { FILE_GENERIC_READ, FILE_GENERIC_WRITE, 
			FILE_GENERIC_EXECUTE, FILE_ALL_ACCESS};
	
	file_obj_handle obj( path );
	obj.GetObjectSecurity(si, psd, 0, &size);
	if(GetLastError() != ERROR_INSUFFICIENT_BUFFER)
		return FALSE;
	sdptr = new char [ size ];
	if(!(psd = (SECURITY_DESCRIPTOR *)sdptr.getPtr())) {
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}
	if(!obj.GetObjectSecurity(si, psd, size, &size))
		return FALSE;
	
	return CreateSecurityDescriptorHandleFromSecurityDescriptor(psh, psd, &fgm);
}

// Create a SECURITY_DESCRIPTOR_HANDLE from a HANDLE object. (Private)
static BOOL CreateSecurityDescriptorHandleFromHandle( SECURITY_DESCRIPTOR_HANDLE *psh,
	obj_handle &obj, const SECURITY_INFORMATION si, const GENERIC_MAPPING *pgm) 
{
	*psh = 0;
	SECURITY_DESCRIPTOR *psd = 0;
	DWORD size;
	autoHeapChar sdptr;

	obj.GetObjectSecurity(si, psd, 0, &size);
	if(GetLastError() != ERROR_INSUFFICIENT_BUFFER)
		return FALSE;
	sdptr = new char [ size ];
	if(!(psd = (SECURITY_DESCRIPTOR *)sdptr.getPtr())) {
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}
	if(!obj.GetObjectSecurity(si, psd, size, &size))
		return FALSE;
	
	return CreateSecurityDescriptorHandleFromSecurityDescriptor(psh, psd, pgm);
}

// Create a SECURITY_DESCRIPTOR_HANDLE from a KERNEL object

BOOL CreateSecurityDescriptorHandleFromKernelObject( SECURITY_DESCRIPTOR_HANDLE *psh,
	const HANDLE hand, const SECURITY_INFORMATION si) 
{
	kernel_obj_handle obj( hand );
	return CreateSecurityDescriptorHandleFromHandle(psh, obj, si, 0);
}

// Create a SECURITY_DESCRIPTOR_HANDLE from a USER object

BOOL CreateSecurityDescriptorHandleFromUserObject( SECURITY_DESCRIPTOR_HANDLE *psh,
	const HANDLE hand, const SECURITY_INFORMATION si) 
{
	user_obj_handle obj( hand );
	return CreateSecurityDescriptorHandleFromHandle(psh, obj, si, 0);
}

// Create a SECURITY_DESCRIPTOR_HANDLE from a REGISTRY key

BOOL CreateSecurityDescriptorHandleFromRegistryKey( SECURITY_DESCRIPTOR_HANDLE *psh,
	const HKEY hkey, const SECURITY_INFORMATION si) 
{
	reg_obj_handle obj( hkey );
	return CreateSecurityDescriptorHandleFromHandle(psh, obj, si, 0);
}

/* Change the owner of the SECURITY_DESCRIPTOR_HANDLE */
BOOL SetSecurityDescriptorOwnerByName(SECURITY_DESCRIPTOR_HANDLE s, const TCHAR *name, const TCHAR *mach)
{	
	secDesc *psecd = (secDesc *)s;
	return psecd->setOwner(name, mach);
}

BOOL SetSecurityDescriptorOwnerBySID(SECURITY_DESCRIPTOR_HANDLE s, const SID *sid) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->setOwner(sid);
}

/* Change the PRIMARY_GROUP of the SECURITY_DESCRIPTOR_HANDLE */
BOOL SetSecurityDescriptorGroupByName(SECURITY_DESCRIPTOR_HANDLE s, const TCHAR *name, const TCHAR *mach)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->setGroup(name, mach);
}

BOOL SetSecurityDescriptorGroupBySID(SECURITY_DESCRIPTOR_HANDLE s, const SID *sid)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->setGroup(sid);
}

/* Convert the SECURITY_DESCRIPTOR_HANDLE to an absolute SECURITY_DESCRIPTOR.
   This will allocate memory, and so must be paired with a call to
   FreeSecurityDescriptorFromHandle() 
*/
BOOL AllocateSecurityDescriptorFromHandle(SECURITY_DESCRIPTOR_HANDLE s, SECURITY_DESCRIPTOR **ppsd)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->createSD(ppsd);
}

/* Frees the SECURITY_DESCRIPTOR allocated by previous call to AllocateSecurityDescriptorFromHandle() */
void FreeSecurityDescriptorFromHandle(SECURITY_DESCRIPTOR_HANDLE s)
{
	secDesc *psecd = (secDesc *)s;
	psecd->deleteSD();
}

/* Deletes the SECURITY_DESCRIPTOR_HANDLE */
void CloseSecurityDescriptorHandle(SECURITY_DESCRIPTOR_HANDLE s) 
{
	secDesc *psecd = (secDesc *)s;
	delete psecd;	
}

/* Adds an allowed ACE for a SID to a SECURITY_DESCRIPTOR_HANDLE */
BOOL AddAllowedSID( SECURITY_DESCRIPTOR_HANDLE s, PSID psid, ACCESS_MASK mask, BYTE flags) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->AddAllowedAce((SID *)psid, mask, flags);
}

/* Adds an allowed ACE for a name to a SECURITY_DESCRIPTOR_HANDLE */
BOOL AddAllowedName( SECURITY_DESCRIPTOR_HANDLE s, const TCHAR *name, const TCHAR *mach, 
					ACCESS_MASK mask, BYTE flags) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->AddAllowedAce(name, mach, mask, flags);
}

/* Adds a denied ACE for a SID to a SECURITY_DESCRIPTOR_HANDLE */
BOOL AddDeniedSID( SECURITY_DESCRIPTOR_HANDLE s, PSID psid, ACCESS_MASK mask, BYTE flags) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->AddDeniedAce((SID *)psid, mask, flags);
}

/* Adds a denied ACE for a name to a SECURITY_DESCRIPTOR_HANDLE */
BOOL AddDeniedName( SECURITY_DESCRIPTOR_HANDLE s, const TCHAR *name, const TCHAR *mach, 
					ACCESS_MASK mask, BYTE flags) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->AddDeniedAce(name, mach, mask, flags);
}

/* Removes an allowed ACE for a SID from a SECURITY_DESCRIPTOR_HANDLE */
BOOL RemoveAllowedSID( SECURITY_DESCRIPTOR_HANDLE s, PSID psid, ACCESS_MASK mask, BYTE flags) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->RemoveAllowedAce((SID *)psid, mask, flags);
}

/* Removes an allowed ACE for a name from a SECURITY_DESCRIPTOR_HANDLE */
BOOL RemoveAllowedName( SECURITY_DESCRIPTOR_HANDLE s, const TCHAR *name, const TCHAR *mach, 
					ACCESS_MASK mask, BYTE flags)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->RemoveAllowedAce(name, mach, mask, flags);
}

/* Removes a denied ACE for a SID from a SECURITY_DESCRIPTOR_HANDLE */
BOOL RemoveDeniedSID( SECURITY_DESCRIPTOR_HANDLE s, PSID psid, ACCESS_MASK mask, BYTE flags) 
{
	secDesc *psecd = (secDesc *)s;
	return psecd->RemoveDeniedAce((SID *)psid, mask, flags);
}

/* Removes a denied ACE for a SID from a SECURITY_DESCRIPTOR_HANDLE */
BOOL RemoveDeniedName( SECURITY_DESCRIPTOR_HANDLE s, const TCHAR *name, const TCHAR *mach, 
					ACCESS_MASK mask, BYTE flags)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->RemoveDeniedAce(name, mach, mask, flags);
}

/* Give a SECURITY_DESCRIPTOR_HANDLE an empty ACL. FreeSecurityDescriptorFromHandle()
   must have been called first.
*/
BOOL SetSecurityDescriptorHandleACLToNoAccess( SECURITY_DESCRIPTOR_HANDLE s)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->noAccessACL();
}

/* Give a SECURITY_DESCRIPTOR_HANDLE a NULL ACL. FreeSecurityDescriptorFromHandle()
   must have been called first.
*/
BOOL SetSecurityDescriptorHandleACLToAllAccess( SECURITY_DESCRIPTOR_HANDLE s)
{
	secDesc *psecd = (secDesc *)s;
	return psecd->allAccessACL();
}

/* Initialize a SECURITY_ATTRIBUTES structure from a SECURITY_DESCRIPTOR_HANDLE */
BOOL InitializeSecurityAttributesFromHandle(SECURITY_ATTRIBUTES *psa, SECURITY_DESCRIPTOR_HANDLE sh, 
				BOOL inherit)
{
	psa->nLength = sizeof(SECURITY_ATTRIBUTES);
	psa->bInheritHandle = inherit;
	return AllocateSecurityDescriptorFromHandle(sh, (SECURITY_DESCRIPTOR **)&psa->lpSecurityDescriptor);
}

}
