/*
    Windows NT Security functions library.
    Copyright (C) 1995  Jeremy R. Allison

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Log: secdesc.cpp,v $
// Revision 1.1  1995/06/30  18:32:55  jra
// Initial revision
//^M
*/

#include "secdesc.h"
#include "seclib.h"
#include "autoheap.h"
#include "objhandle.h"

//
// Open a thread token and get the SID of the TOKEN_OWNER.
//
BOOL secDesc::getThreadTokenOwnerInfo(SIDobj **ppown) {
	*ppown = 0;
	autoHeapChar tokptr;
	HANDLE h_tok = INVALID_HANDLE_VALUE;

	// Try and open the thread token first in case we are impersonating.
	BOOL ret = OpenThreadToken(GetCurrentThread(), TOKEN_QUERY, FALSE, &h_tok);
	// If the thread does not have a token it fails with an error of
	// ERROR_NO_TOKEN.
	if(ret == FALSE && (ERROR_NO_TOKEN != GetLastError()))
		return FALSE;

	if(!OpenProcessToken(GetCurrentProcess(), TOKEN_QUERY, &h_tok))
		return FALSE;

	// Get the size needed for the TOKEN_OWNER.
	DWORD own_size = 0;
	GetTokenInformation(h_tok, TokenOwner, 0, 0, &own_size);
	if(GetLastError() != ERROR_INSUFFICIENT_BUFFER)	{
		SetLastError(ERROR_OUTOFMEMORY);
		CloseHandle(h_tok);
		return FALSE;
	}

	tokptr = new char [ own_size ];
	if((char *)tokptr == 0) {
		CloseHandle(h_tok);
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}

	// Get the TOKEN_OWNER
	if(!GetTokenInformation(h_tok, TokenOwner, (TOKEN_OWNER *)tokptr.getPtr(), own_size, &own_size)) {
		CloseHandle(h_tok);
		return FALSE;
	}

	SID *pownersid = (SID *)((TOKEN_OWNER *)tokptr.getPtr())->Owner;
	// Copy the retrieved SID into the pointer.
	autoHeapPtr<SIDobj> ownptr = new SIDobj(pownersid);
	if(ownptr.getPtr() == 0) {
		CloseHandle(h_tok);
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}

	*ppown = ownptr.relinquishOwnership(); // *ppown now owns the memory allocated to ownptr
	CloseHandle(h_tok);
	return TRUE;
}

//
// Open a thread token and get the SID of the TOKEN_PRIMARY_GROUP.
//
BOOL secDesc::getThreadTokenGroupInfo(SIDobj **ppgrp) {
	*ppgrp = 0;
	autoHeapChar tokptr;
	HANDLE h_tok = INVALID_HANDLE_VALUE;

	// Try and open the thread token first in case we are impersonating.
	BOOL ret = OpenThreadToken(GetCurrentThread(), TOKEN_QUERY, FALSE, &h_tok);
	// If the thread does not have a token it fails with an error of
	// ERROR_NO_TOKEN.
	if(ret == FALSE && (ERROR_NO_TOKEN != GetLastError()))
		return FALSE;

	if(!OpenProcessToken(GetCurrentProcess(), TOKEN_QUERY, &h_tok))
		return FALSE;

	// Get the size needed for the TOKEN_PRIMARY_GROUP.
	DWORD grp_size = 0;
	GetTokenInformation(h_tok, TokenPrimaryGroup, 0, 0, &grp_size);
	if(GetLastError() != ERROR_INSUFFICIENT_BUFFER)	{
		CloseHandle(h_tok);
		return FALSE;
	}
	tokptr = new char [ grp_size ];
	if((char *)tokptr == 0) {
		CloseHandle(h_tok);
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}

	// Get the TOKEN_PRIMARY_GROUP
	if(!GetTokenInformation(h_tok, TokenPrimaryGroup, 
				(TOKEN_PRIMARY_GROUP *)tokptr.getPtr(), grp_size, &grp_size)) {
		CloseHandle(h_tok);
		return FALSE;
	}

	SID *pgroupsid = (SID *)((TOKEN_PRIMARY_GROUP *)tokptr.getPtr())->PrimaryGroup;

	// Copy the retrieved SID's into the class members.
	autoHeapPtr<SIDobj> grpptr = new SIDobj(pgroupsid);
	if(grpptr.getPtr() == 0)	{
		CloseHandle(h_tok);
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}

	*ppgrp = grpptr.relinquishOwnership(); // *ppgrp now owns the memory allocated to grpptr
	CloseHandle(h_tok);
	return TRUE;
}

//
// Ensure the pacl_list_ is valid.
//
BOOL secDesc::validAclList() {

	if(pacl_list_ != 0)
		return TRUE;

	pacl_list_ = new aclList;
	if(pacl_list_ == 0) {
		SetLastError(ERROR_OUTOFMEMORY);
		return FALSE;
	}
	return TRUE;
}

//
// Map an ACCESS_MASK containing specific rights to GENERIC rights.
//
void secDesc::mapMaskToGeneric(ACCESS_MASK smask, ACCESS_MASK *pdmask) {
	*pdmask = smask;
	ACCESS_MASK maskoutmask = 0;
	if(!pgm_)
		return;
	if( smask & pgm_->GenericRead) {
		*pdmask |= GENERIC_READ;
		maskoutmask |= pgm_->GenericRead;
	}
	if( smask & pgm_->GenericWrite) {
		*pdmask |= GENERIC_WRITE;
		maskoutmask |= pgm_->GenericWrite;
	}
	if( smask & pgm_->GenericExecute) {
		*pdmask |= GENERIC_EXECUTE;
		maskoutmask |= pgm_->GenericExecute;
	}
	if( smask & pgm_->GenericAll) {
		*pdmask |= GENERIC_ALL;
		maskoutmask |= pgm_->GenericAll;
	}
	*pdmask &= ~maskoutmask;
}

//
// Create a secDesc object given an owner and primary group.
//
secDesc::secDesc(const TCHAR *owner, const TCHAR *group, const TCHAR *mach)
	:pacl_(0), psidown_(0), psidgrp_(0), pacl_list_(0), pgm_(0) {	

	InitializeCriticalSection(&critsec_);
	if(!InitializeSecurityDescriptor(&sd_, SECURITY_DESCRIPTOR_REVISION))
		throw GetLastError();

	if(!setOwner(owner, mach))
		throw GetLastError();

	if(!setGroup(group, mach))
		throw GetLastError();
}

//
// Create a secDesc object given an owner and primary group SID.
//
secDesc::secDesc(const SID *owner, const SID *group) 
	: pacl_(0), psidown_(0), psidgrp_(0), pacl_list_(0), pgm_(0) {

	InitializeCriticalSection(&critsec_);
	if(!InitializeSecurityDescriptor(&sd_, SECURITY_DESCRIPTOR_REVISION))
		throw GetLastError();

	if(!setOwner(owner))
		throw GetLastError();

	if(!setGroup(group))
		throw GetLastError();
}

//
// Create a secDesc object from an already existing SECURITY_DESCRIPTOR.
// Copy the owner, group and DACL from the existing descriptor.
//
secDesc::secDesc(const SECURITY_DESCRIPTOR *psd, const GENERIC_MAPPING *pgm)
		:pacl_(0), psidown_(0), psidgrp_(0), pacl_list_(0), pgm_(0) {

	InitializeCriticalSection(&critsec_);
	if(!IsValidSecurityDescriptor((PSECURITY_DESCRIPTOR)psd))
		throw GetLastError();
	if(!InitializeSecurityDescriptor(&sd_, SECURITY_DESCRIPTOR_REVISION))
		throw GetLastError();

	autoHeapPtr<GENERIC_MAPPING> pgenmap;	
	if(pgm) {
		pgenmap = new GENERIC_MAPPING;
		if(pgenmap.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
			throw (DWORD)(ERROR_OUTOFMEMORY);
		}
		*pgenmap = *pgm;
	}
	SID *pownersid = 0;
	BOOL ownerDefaulted;
	autoHeapPtr<SIDobj> ownsidcopy;
	if(!GetSecurityDescriptorOwner((PSECURITY_DESCRIPTOR)psd, (PSID *)&pownersid, &ownerDefaulted ))
		throw GetLastError();
	if(pownersid) {
		// SECURITY_DESCRIPTOR has an owner, copy it.
		ownsidcopy = new SIDobj(pownersid);
		if(ownsidcopy.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
			throw (DWORD)(ERROR_OUTOFMEMORY);
		}
	}

	SID *pgroupsid = 0;
	autoHeapPtr<SIDobj> grpsidcopy;
	if(!GetSecurityDescriptorGroup((PSECURITY_DESCRIPTOR)psd, (PSID *)&pgroupsid, 
					&ownerDefaulted ))
		throw GetLastError();
	if(pgroupsid) {
		// SECURITY_DESCRIPTOR has a PRIMARY group, copy it.
		grpsidcopy = new SIDobj(pgroupsid);
		if(grpsidcopy.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
			throw (DWORD)(ERROR_OUTOFMEMORY);
		}
	}		 
 	// Now copy the ACL.
	BOOL daclPresent;
	ACL *pacl;
	if(!GetSecurityDescriptorDacl((PSECURITY_DESCRIPTOR)psd, &daclPresent,
			&pacl, &ownerDefaulted))
		throw GetLastError();
	if(daclPresent) {
		if(pacl == 0) {
			// DACL present but not specified - all access allowed.
			this->allAccessACL();
			psidown_ = ownsidcopy.relinquishOwnership();
			psidgrp_ = grpsidcopy.relinquishOwnership();
			return;
		}
		// Allocate a new aclList.
		autoHeapPtr<aclList> pacl_list(new aclList);
		if(pacl_list.getPtr() == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
			throw (DWORD)(ERROR_OUTOFMEMORY);
		}
		DWORD aceCount = pacl->AceCount;
		DWORD i;
		LPVOID pactace;
		ACE_HEADER *pacehdr;
		ACCESS_ALLOWED_ACE *pallowedace;
		ACCESS_DENIED_ACE *pdeniedace;
		for( i = 0; i < aceCount; i++) {
			if(!GetAce(pacl, i, &pactace))
				throw GetLastError();
			pacehdr = (ACE_HEADER *)pactace;

			switch(pacehdr->AceType) {
				case ACCESS_ALLOWED_ACE_TYPE:
					{
					pallowedace = (ACCESS_ALLOWED_ACE *)pactace;
					ACCESS_MASK mapped_mask;
					mapMaskToGeneric(pallowedace->Mask, &mapped_mask);
					SIDobj sid((SID *)&pallowedace->SidStart);
					pacl_list->AddAllowedAce(sid, mapped_mask, pacehdr->AceFlags);
					break;
					}
				case ACCESS_DENIED_ACE_TYPE:
					{
					pdeniedace = (ACCESS_DENIED_ACE *)pactace;
					ACCESS_MASK mapped_mask;
					mapMaskToGeneric(pallowedace->Mask, &mapped_mask);
					SIDobj sid((SID *)&pdeniedace->SidStart);
					pacl_list->AddDeniedAce(sid,mapped_mask, pacehdr->AceFlags);
					break;
					}
			}
		}	
		// Take ownership of the autoheap object.
		pacl_list_ = pacl_list.relinquishOwnership();
	}
	psidown_ = ownsidcopy.relinquishOwnership();
	psidgrp_ = grpsidcopy.relinquishOwnership();
	pgm_ = pgenmap.relinquishOwnership();
}

//
// Destructor to free all resources.
//
secDesc::~secDesc() {
	enter();
	delete psidown_;
	delete psidgrp_;
	delete [] (char *)pacl_;
	delete pacl_list_;
	delete pgm_;
	DeleteCriticalSection(&critsec_);
	leave();
}

//
// Set the owner of this SD given a name and machine.
//
BOOL secDesc::setOwner(const TCHAR *owner, const TCHAR *mach) {

	enter();

	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}

	BOOL ret = TRUE;
	DWORD err = 0;
	if(owner == 0) {
		ret = setOwner((SID *)0);
	} else {
		try {
			// This statement will generate an exception if if fails.
			autoHeapPtr<SIDobj> psid = new SIDobj(owner, mach);
			ret = setOwner(*psid);
		} catch (DWORD caerr) {
			err = caerr;
			ret = FALSE;
		}
	}

	if(err)
		SetLastError(err);
	leave();
	return ret;
}

//
// Set the group of this SD given a name and machine.
//
BOOL secDesc::setGroup(const TCHAR *group, const TCHAR *mach) {

	enter();

	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}

	BOOL ret = TRUE;
	DWORD err = 0;
	if(group == 0 ) {
		ret = setGroup((SID *)0);
	} else {
		try {
			// This statement will generate an exception if if fails.
			autoHeapPtr<SIDobj> psid = new SIDobj(group, mach);
			ret = setGroup(*psid);
		} catch (DWORD caerr) {
			err = caerr;
			ret = FALSE;
		}
	}

	if(err)
		SetLastError(err);
	leave();
	return ret;
}

//
// Set the owner of this SD given a SID.
//
BOOL secDesc::setOwner(const SID *owner) {

	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}

	delete psidown_;
	psidown_ = 0;

	BOOL ret = TRUE;
	DWORD err = 0;

	if(owner != 0) {
		try {
			psidown_ = new SIDobj(owner);
		} catch (DWORD caerr) {
			err = caerr;
			ret = FALSE;
		}
	} else {
		ret = getThreadTokenOwnerInfo( &psidown_ );
	}

 	if(ret == FALSE) {
		delete psidown_;
		psidown_ = 0;
		if(err)
			SetLastError(err);
		leave();
		return FALSE;
	}
	// If we get to here we *must* have a valid psidown_
	if(psidown_ == 0 ) {
		SetLastErrorEx(ERROR_INVALID_SID, SLE_ERROR);
		leave();
		return FALSE;
	}

	// if owner == 0 then owner was defaulted.
	if(!SetSecurityDescriptorOwner(&sd_, *psidown_, owner == 0)) {
		delete [] psidown_;
		psidown_ = 0;
		leave();
		return FALSE;
	}
	if(err)
		SetLastError(err);
	leave();
	return TRUE;
}

//
// Set the group of this SD given a SID.
//
BOOL secDesc::setGroup(const SID *group) {

	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}

	delete [] psidgrp_;
	psidgrp_ = 0;

	BOOL ret = TRUE;
	DWORD err = 0;
	if(group != 0) {
		try {
			psidgrp_ = new SIDobj(group);
		} catch (DWORD caerr) {
			err = caerr;
			ret = FALSE;
		}
	} else {
		ret = getThreadTokenGroupInfo( &psidgrp_ );
	}

	if(ret == FALSE) {
		delete psidgrp_;
		psidgrp_ = 0;
		if(err)
			SetLastError(err);
		leave();
		return FALSE;
	}

	// If we get to here we *must* have a valid psidgrp_
	if(psidgrp_ == 0) {
		SetLastErrorEx(ERROR_INVALID_SID, SLE_ERROR);
		leave();
		return FALSE;
	}

	// if group == 0 then group was defaulted.
	if(!SetSecurityDescriptorGroup(&sd_, *psidgrp_, group == 0)) {
		delete psidgrp_;
		psidgrp_ = 0;
		leave();
		return FALSE;
	}
	if(err)
		SetLastError(err);
	leave();
	return TRUE;

}

//
// Add an allowed ACE by name to this Security Descriptor.
//
BOOL secDesc::AddAllowedAce(const TCHAR *name, const TCHAR *mach, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret = pacl_list_->AddAllowedAce(name, mach, mask, flags);
	leave();
	return ret;
}

//
// Add an allowed ACE by SID to this Security Descriptor.
//
BOOL secDesc::AddAllowedAce(const SID *s, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(s);
		ret = pacl_list_->AddAllowedAce(sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	leave();
	return ret;
}

//
// Add a denied ACE by name to this Security Descriptor.
//
BOOL secDesc::AddDeniedAce(const TCHAR *name, const TCHAR *mach, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret = pacl_list_->AddDeniedAce(name, mach, mask, flags);
	leave();
	return ret;
}

//
// Add a denied ACE by SID to this Security Descriptor.
//
BOOL secDesc::AddDeniedAce(const SID *s, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret;
	DWORD err;
	try {
		SIDobj sid(s);
		ret = pacl_list_->AddDeniedAce(sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	leave();
	return ret;
}

//
// Remove an allowed ACE by name from this Security Descriptor.
//
BOOL secDesc::RemoveAllowedAce(const TCHAR *name, const TCHAR *mach, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret = pacl_list_->RemoveAllowedAce(name, mach, mask, flags);
	leave();
	return ret;
}

//
// Remove an allowed ACE by SID from this Security Descriptor.
//
BOOL secDesc::RemoveAllowedAce(const SID *s, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(s);
		ret = pacl_list_->RemoveAllowedAce(sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	leave();
	return ret;
}

//
// Remove a denied ACE by name from this Security Descriptor.
//
BOOL secDesc::RemoveDeniedAce(const TCHAR *name, const TCHAR *mach, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret = pacl_list_->RemoveDeniedAce(name, mach, mask, flags);
	leave();
	return ret;
}

//
// Remove a denied ACE by SID from this Security Descriptor.
//
BOOL secDesc::RemoveDeniedAce(const SID *s, ACCESS_MASK mask, BYTE flags) {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(!validAclList()) {
		leave();
		return FALSE;
	}
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid(s);
		ret = pacl_list_->RemoveDeniedAce(sid, mask, flags);
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	leave();
	return ret;
}

//
// Actually use the data in this secDesc to create and return a security descriptor
// with the given owner, group, and dacl.
//
BOOL secDesc::createSD(SECURITY_DESCRIPTOR **ppsd) {
	enter();
	*ppsd = 0;

	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}

	// Get the size of the pacl_ from the acl_list_ (if it exists).
	DWORD size = 0;
	autoHeapChar aclptr;

	if(pacl_list_) {
		pacl_list_->createACLfromList(0, &size);
		if(GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
			leave();
			return FALSE;
		}
		// allocate it.
		if(size) {
			// There is an ACL List.
			aclptr = new char [ size ];
			if((char *)aclptr == 0) {
				SetLastError(ERROR_OUTOFMEMORY);
				leave();
				return FALSE;
			}
			// Create the pacl_ from the acl_list_.
			if(!pacl_list_->createACLfromList((ACL *)aclptr.getPtr(), &size)) {
				leave();
				return FALSE;
			}
		}
	}
	if(!SetSecurityDescriptorDacl(&sd_, TRUE, (ACL *)aclptr.getPtr(), FALSE)) {
		leave();
		return FALSE;
	}
	/* The Owner and group should already be set */
	pacl_ = (ACL *)aclptr.relinquishOwnership();
	*ppsd = &sd_;
	leave();
	return TRUE;
}

//
// Mark the security descriptor as invalid by removing the acl
// createSD must have been previously called.
//
void secDesc::deleteSD() {
	enter();
	SetSecurityDescriptorDacl(&sd_, TRUE, 0, TRUE);
	delete [] (char *)pacl_;
	pacl_ = 0;
	leave();
}

// Clear the ACL in the secDesc	(ie. NO access allowed).
BOOL secDesc::noAccessACL() {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(pacl_list_)
		pacl_list_->clearACL();
	else {
		pacl_list_ = new aclList;
		if(pacl_list_ == 0) {
			SetLastError(ERROR_OUTOFMEMORY);
			leave();
			return FALSE;
		}
	}
	leave();
	return TRUE;
}

// Remove the ACL in the secDesc (ie. ALL access allowed).
BOOL secDesc::allAccessACL() {
	enter();
	if(pacl_ != 0) {
		SetLastError(ERROR_INVALID_SECURITY_DESCR);
		leave();
		return FALSE;
	}
	if(pacl_list_) {
		pacl_list_->clearACL();
		delete pacl_list_;
	}
	pacl_list_ = 0;
	leave();
	return TRUE;
}

