
/*
 * Windows NT Security functions library.
 * Copyright (C) 1995  Jeremy R. Allison
 *
 * Code to Add/Remove User rights.
 * Adapted from Microsoft's LSA Sample program, presented at the 
 * 1994 Windows NT PDC Conference.
 *
 */

#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <tchar.h>

#include "ntsecapi.h"
#include "unistring.h"
#include "seclib.h"
#include "sidobj.h"

/*
 * Construct a LsaString structure from a reference counted Unicode String.
 */

static PLSA_UNICODE_STRING MakeLsaString(PLSA_UNICODE_STRING LsaString, const UniString String)
{
    DWORD StringLength;
    if (String == (wchar_t *)0)
    {
        LsaString->Buffer = 0;
        LsaString->Length = 0;
        LsaString->MaximumLength = 0;
        return LsaString;
    }

    StringLength = wcslen(String);
    LsaString->Buffer = String;
    LsaString->Length = (USHORT) StringLength * sizeof(WCHAR);
    LsaString->MaximumLength = (USHORT) (StringLength + 1) * sizeof(WCHAR);
    return LsaString; 
}

/*
 * Open the Lsa Policy database.
 */

static BOOL OpenPolicy(UniString& ServerName, DWORD DesiredAccess, PLSA_HANDLE PolicyHandle)
{
    DWORD err;
    LSA_OBJECT_ATTRIBUTES ObjectAttributes;
    LSA_UNICODE_STRING ServerString;
    PLSA_UNICODE_STRING Server = 0;

	memset((void *)&ObjectAttributes, '\0', sizeof(ObjectAttributes));

    if (ServerName != (wchar_t *)0)
        Server = MakeLsaString(&ServerString, ServerName);

    err = LsaNtStatusToWinError(
    			LsaOpenPolicy(Server,&ObjectAttributes,DesiredAccess,PolicyHandle));
    if (err != 0) {
		SetLastError(err);
		return FALSE;
	}
    return(TRUE);


}

/*
 * Add or Remove the user right for the given SID, depending on the flag.
 */

static BOOL AdjustUserRight(PSID psid, const TCHAR *mach, const TCHAR *privname, BOOL allow)
{
    LSA_HANDLE polhand;
	DWORD err;

    if(OpenPolicy( UniString(mach), POLICY_ALL_ACCESS, &polhand) == FALSE)
		return FALSE;

	LSA_UNICODE_STRING UserRightString;
	if(allow) {
	    err = LsaNtStatusToWinError(
	    		LsaAddAccountRights(polhand,psid,
	    			MakeLsaString(&UserRightString, UniString(privname)),1));
	} else {
		err = LsaNtStatusToWinError(
	            LsaRemoveAccountRights(polhand,psid,FALSE,
	                MakeLsaString(&UserRightString, UniString(privname)),1));
	}

	if(err) {
		SetLastError(err);
		LsaClose(polhand);
		return FALSE;
	}
	LsaClose(polhand);
	return TRUE;
}

extern "C" {

/*
 * Add a User right to a user of a given name.
 */

BOOL AddUserRightToName(const TCHAR *name, const TCHAR *mach, const TCHAR *privname)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid( name, mach);
		ret =  AddUserRightToSid(sid, mach, privname); 
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

/*
 * Add a User right to a user of a given SID.
 */

BOOL AddUserRightToSid(PSID psid, const TCHAR *mach, const TCHAR *privname)
{
	return AdjustUserRight(psid, mach, privname, TRUE);
}

/*
 * Remove a User right to a user of a given name.
 */

BOOL RemoveUserRightFromName(const TCHAR *name, const TCHAR *mach, const TCHAR *privname)
{
	BOOL ret;
	DWORD err = 0;
	try {
		SIDobj sid( name, mach);
		ret =  RemoveUserRightFromSid(sid, mach, privname); 
	} catch (DWORD caerr) {
		err = caerr;
		ret = FALSE;
	}
	if(err)
		SetLastError(err);
	return ret;
}

/*
 * Remove a User right to a user of a given SID.
 */

BOOL RemoveUserRightFromSid(PSID psid, const TCHAR *mach, const TCHAR *privname)
{
	return AdjustUserRight(psid, mach, privname, FALSE);
}

}
