/* 
Unix file server for Pathworks for DOS. 
Version 0.5.
Copyright (C) Andrew Tridgell 1992.
Permission to use, copy and distribute this software is given to anyone
who wants to, for NON-PROFIT only. You may not charge for this software
or any derivatives of it without first contacting the Author.
*/




#include "local.h"
#include <stdio.h>
#include <strings.h>
#include <ctype.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <errno.h>
#include <sysexits.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <sys/mount.h>
#include <pwd.h>

#ifdef SUN
#include <unistd.h>
#include <sys/dirent.h>
#include <string.h>
#endif

#if (defined(ULTRIX) || defined(SEQUENT) || defined(SGI))
#include <sys/dir.h>
#endif

#ifdef AIX
#include <sys/dir.h>
#include <dirent.h>
#endif

#ifdef SEQUENT
char *strchr();
char *strrchr();
typedef int mode_t;
#define SEEK_SET 0
#endif

#define EXTERN
#include "server.h"

char *InBuffer = NULL;
char *OutBuffer = NULL;
int Client = 0;

char TimeBuf[100];

/****************************************************************************
return the date and time as a string
****************************************************************************/
char *timestring()
{
time_t t;
t = time(NULL);
strftime(TimeBuf,100,"%D %r",localtime(&t));
return(TimeBuf);
}


/*******************************************************************
byte swap an object - the byte order of the object is reversed
********************************************************************/
void object_byte_swap(obj,size)
void *obj;
int size;
{
int i;
char c;
char *p1 = (char *)obj;
char *p2 = p1 + size - 1;

size /= 2;

for (i=0;i<size;i++)
	{
	c = *p1;
	*p1 = *p2;
	*p2 = c;
	p1++;
	p2--;
	}
}

/****************************************************************************
byte swap a int16
****************************************************************************/
int16 int16_byte_swap(x)
int16 x;
{
int16 res;
res = x;
object_byte_swap(&res,sizeof(int16));
return(res);
}

/****************************************************************************
byte swap a int32
****************************************************************************/
int32 int32_byte_swap(x)
int32 x;
{
int32 res;
res = x;
object_byte_swap(&res,sizeof(int32));
return(res);
}

/*******************************************************************
check if a file exists
********************************************************************/
BOOL file_exist(fname)
char *fname;
{
struct stat st;
if (stat(fname,&st) != 0) 
  return(False);

return(S_ISREG(st.st_mode));
}

/*******************************************************************
check if a directory exists
********************************************************************/
BOOL directory_exist(dname)
char *dname;
{
struct stat st;
if (stat(dname,&st) != 0) 
  return(False);

return(S_ISDIR(st.st_mode));
}

/*******************************************************************
create a 32 bit dos packed date/time from some parameters
********************************************************************/
int32 make_dos_date(unixdate)
time_t unixdate;
{
int32 ret;
char *p;
struct tm *t;

t = localtime(&unixdate);
p = (char *)&ret;

p[0] = (t->tm_sec/2) + ((t->tm_min & 0x7) << 5);
p[1] = (t->tm_min >> 3) + (t->tm_hour << 3);
p[2] = t->tm_mday + (((t->tm_mon+1) & 0x7) << 5);
p[3] = ((t->tm_mon+1) >> 3) + ((t->tm_year-80) << 1);

return(ret);
}


/*******************************************************************
interpret a 32 bit dos packed date/time to some parameters
********************************************************************/
void interpret_dos_date(date,year,month,day,hour,minute,second)
int32 date;
int *year,*month,*day,*hour,*minute,*second;
{
int32 idate;
dosdate *d;

idate = int32_byte_swap(date);

d = (dosdate *)&idate;

*year = d->year + 1980;
*month = d->month;
*day = d->day;
*hour = d->hour;
*minute = d->minute;
*second = d->second * 2;
}

/*******************************************************************
true if the machine is big endian
********************************************************************/
BOOL big_endian()
{
int x = 2;
char *s;
s = (char *)&x;
return(s[0] == 0);
}

/*******************************************************************
compare 2 strings 
********************************************************************/
BOOL strequal(s1,s2)
char *s1;
char *s2;
{
if (!s1 || !s2) return(False);

return(strcasecmp(s1,s2)==0);
}


/*******************************************************************
convert a string to lower case
********************************************************************/
void strlower(s)
char *s;
{
while (*s)
	{
	*s = tolower(*s);
	s++;
	}
}

/*******************************************************************
convert a string to upper case
********************************************************************/
void strupper(s)
char *s;
{
while (*s)
	{
	*s = toupper(*s);
	s++;
	}
}

/****************************************************************************
string replace
****************************************************************************/
void string_replace(s,old,new)
char *s;
char old;
char new;
{
 while (*s)
{
  if (old == *s)
      *s = new;
  s++;
}
}

/****************************************************************************
make a file into unix format
****************************************************************************/
void unix_format(fname)
char *fname;
{
char namecopy[255];
string_replace(fname,'\\','/');
strlower(fname);
if (*fname == '/')
  {
    strcpy(namecopy,fname);
    strcpy(fname,".");
    strcat(fname,namecopy);
  }
}


/****************************************************************************
set a value at buf[pos] to integer val
****************************************************************************/
void SIVAL(buf,pos,val)
char *buf;
int pos;
int val;
{
val = ISWP(val);
memcpy(buf + pos,(char *)&val,sizeof(int));
}

/****************************************************************************
set a value at buf[pos] to int16 val
****************************************************************************/
void SSVAL(buf,pos,val)
char *buf;
int pos;
int16 val;
{
val = SSWP(val);
memcpy(buf + pos,(char *)&val,sizeof(int16));
}

/****************************************************************************
get a integer value
****************************************************************************/
int IVAL(buf,pos)
char *buf;
int pos;
{
int res;
memcpy((char *)&res,buf + pos,sizeof(int));
res = ISWP(res);
return(res);
}


/****************************************************************************
get a int16 value
****************************************************************************/
int16 SVAL(buf,pos)
char *buf;
int pos;
{
int16 res;
memcpy((char *)&res,buf + pos,sizeof(int16));
res = SSWP(res);
return(res);
}


/****************************************************************************
see if a name matches a mask. The mask takes the form of several characters,
with ? being a wild card.
****************************************************************************/
BOOL mask_match(name,mask,dodots)
char *name;
char *mask;
BOOL dodots;
{
char *p1,*p2;
char nbeg[255]; /* beginning of name */
char next[20]; /* extension of name */
char mext[20]; /* extension of mask */
char mbeg[20]; /* beg of mask */

strcpy(mbeg,mask);
if ((p1 = strchr(mbeg,'.')) != NULL)
  {*p1 = 0;p1++;strcpy(mext,p1);}
else
  {
    strcpy(mext,"");
    if (strlen(mbeg) > 8)
      {
      strcpy(mext,mbeg + 8);
      mbeg[8] = 0;
    }
  }

strcpy(nbeg,name);
if ((p1 = strchr(nbeg,'.')) != NULL)
  {*p1 = 0;p1++;strcpy(next,p1);}
else
  strcpy(next,"");


/* a couple of special cases */
if (strequal(name,".") || strequal(name,".."))
    return(dodots && strequal(mask,"????????.???"));

if (strlen(nbeg) == 0) return(False);
if (strlen(mbeg) == 0) return(False);
if (strlen(nbeg) > 8) return(False);
if (strlen(next) > 3) return(False);
if (strlen(mbeg) > 8) return(False);
if (strlen(mext) > 3) return(False);
if (strlen(nbeg) > strlen(mbeg)) return(False);
if (strlen(next) > strlen(mext)) return(False);

/* only accept lowercase names */
p1 = name;
while (*p1) 
  if (isupper(*p1++)) return(False);

p1 = nbeg;
p2 = mbeg;
while (*p2)
  {
    if ((*p2 != '?') && (*p1 != *p2)) 
      return(False);
    p2++;
    if (*p1) p1++;
  }

p1 = next;
p2 = mext;
while (*p2)
  {
    if ((*p2 != '?') && (*p1 != *p2)) 
      return(False);
    p2++;
    if (*p1) p1++;
  }
return(True);
}

/****************************************************************************
change a dos mode to a unix mode
****************************************************************************/
mode_t unix_mode(dosmode)
int dosmode;
{
mode_t result = 0444;

if ((dosmode & aRONLY) == 0)
 result |= 0222;
if ((dosmode & aHIDDEN) != 0)
 result |= 0100;
if ((dosmode & aDIR) != 0)
 result |= 0040000;
if ((dosmode & aSYSTEM) != 0)
 result |= 0010;
if ((dosmode & aARCH) != 0)
 result |= 0001;
return(result);
}

/****************************************************************************
change a unix mode to a dos mode
****************************************************************************/
int dos_mode(unixmode)
mode_t unixmode;
{
int result = 0;
if ((unixmode & 0222) == 0)
 result |= aRONLY;
if ((unixmode & 0040000) != 0)
 result |= aDIR;
if ((unixmode & 0100) != 0)
 result |= aHIDDEN;
if ((unixmode & 0010) != 0)
 result |= aSYSTEM;
if ((unixmode & 0001) != 0)
 result |= aARCH;
return(result);
}

/****************************************************************************
make a dir struct
****************************************************************************/
void make_dir_struct(buf,fname,size,mode,num,date)
char *buf;
char *fname;
int size;
int mode;
int num;
time_t date;
{
int32 dos_date;

dos_date = make_dos_date(date);

memset(buf,0,DIR_STRUCT_SIZE);
strcpy(buf + 29,fname);

strcpy(buf,"???????????");

SIVAL(buf,25,size);
PVAL(buf,20,char) = mode;
SIVAL(buf,11,num);

SIVAL(buf,21,ISWP(dos_date));

if ((mode & aDIR) != 0)
  {
  PVAL(buf,20,char) = 17;
  PVAL(buf,26,char) = 2;
  }

}

/****************************************************************************
end a directory listing
****************************************************************************/
void end_dir(cnum)
int cnum;
{
if (Connections[cnum].dirptr != NULL)
  {
  closedir(Connections[cnum].dirptr);
  Connections[cnum].dirptr = NULL;
  }
}


/****************************************************************************
start a directory listing
****************************************************************************/
BOOL start_dir(cnum,directory)
int cnum;
char *directory;
{
if (Connections[cnum].dirptr != NULL)
  end_dir(cnum);
Connections[cnum].dirptr = (void *)opendir(directory);
strcpy(Connections[cnum].dirpath,directory);

return(Connections[cnum].dirptr != NULL);

}

/****************************************************************************
get a directory entry
****************************************************************************/
BOOL get_dir_entry(cnum,mask,dirtype,fname,size,mode,entry,date)
int cnum;
char *mask;
int dirtype;
char *fname;
int *size;
int *mode;
int *entry;
time_t *date;
{
#ifdef SUN
struct dirent *readdir(); /* for some reason sun's header files have a
				problem with this function */
#endif

#if (defined(ULTRIX) || defined(SEQUENT) || defined(SGI))
struct direct *dptr;
#else
struct dirent *dptr;
#endif

BOOL found = False;
struct stat sbuf;
char path[255];
BOOL isrootdir;

isrootdir = (strequal(Connections[cnum].dirpath,"./") ||
	     strequal(Connections[cnum].dirpath,"."));


while (!found)
  {

    dptr = readdir(Connections[cnum].dirptr);

    if (dptr == NULL) 
	return(False);

    if (mask_match(dptr->d_name,mask,!isrootdir))
      {
	strcpy(fname,dptr->d_name);
	strcpy(path,Connections[cnum].dirpath);
	strcat(path,"/");
	strcat(path,fname);
	if (stat(path,&sbuf) != 0) continue;

	*mode = dos_mode(sbuf.st_mode);
	if ((*mode != 0) && (*mode != aARCH) && ((*mode & dirtype) == 0)) 
	  continue;
	*size = sbuf.st_size;
	*date = sbuf.st_mtime;

	if (lstat(path,&sbuf) != 0) continue;
	*entry = (sbuf.st_ino);

        found = True;
      }
  }

return(True);
}

/****************************************************************************
become the user of a connection number
****************************************************************************/
BOOL become_user(cnum)
int cnum;
{
if (!OPEN_CNUM(cnum))
	return(False);

if (setegid(Connections[cnum].gid) != 0)
  return(False);

if (seteuid(Connections[cnum].uid) != 0)
  return(False);

if (chdir(Connections[cnum].connectpath) != 0)
  {
    if (DEBUGLEVEL > 0)
      {
	fprintf(dbf,"%s chdir (%s) failed cnum=%d\n",timestring(),Connections[cnum].connectpath,cnum);
	fflush(dbf);
      }
    return(False);
  }

return(True);
}

/****************************************************************************
unbecome the user of a connection number
****************************************************************************/
BOOL unbecome_user()
{
seteuid(getuid());
setegid(getgid());
if (chdir("/") != 0)
  if (DEBUGLEVEL > 0)
    {
      fprintf(dbf,"%s chdir / failed\n",timestring());
      fflush(dbf);
    }
return(True);
}

/****************************************************************************
Signal handler for SIGPIPE (write on a disconnected socket) 
****************************************************************************/
void abort()
{
    exit(EX_IOERR);
}


/****************************************************************************
find a service entry
****************************************************************************/
int find_service(service)
char *service;
{
int i;
for (i=0;i<MAX_SERVICES;i++)
	 if (Services[i].available && strequal(Services[i].service,service))
		 return(i);
return(-1);
}

/****************************************************************************
find first available connection slot
****************************************************************************/
int find_free_connection()
{
int i;
for (i=1;i<MAX_CONNECTIONS;i++)
	 if (!Connections[i].open) 
		 return(i);
return(-1);
}

/****************************************************************************
find first available file slot
****************************************************************************/
int find_free_file()
{
int i;
for (i=0;i<MAX_OPEN_FILES;i++)
	 if (!Files[i].open) 
		 return(i);
return(-1);
}

/****************************************************************************
make a connection to a service
****************************************************************************/
int make_connection(service,user,password)
char *service;
char *user;
char *password;
{
int cnum;
int snum;
struct passwd *pass = NULL;
char key[50];
char salt[10];

strlower(user);
strlower(service);
strlower(password);

snum = find_service(service);
if (snum < 0)
	 {
	   if (DEBUGLEVEL > 0)
	     {
	       fprintf(dbf,"%s couldn't find service %s\n",timestring(),service);
	       fflush(dbf);
	     }
	 
	 return(-1);
	 }

cnum = find_free_connection();
if (cnum < 0)
	 {
	   if (DEBUGLEVEL > 0)
	     {
	       fprintf(dbf,"%s couldn't find free connection\n",timestring());
	       fflush(dbf);
	     }
	 
	 return(-1);
	 }

#ifdef GUEST_ACCOUNT
if (GUEST_OK(snum) && (*user == 0) && (strlen(GUEST_ACCOUNT) > 0))
{ /* we want guest account */
pass = getpwnam(GUEST_ACCOUNT);
if (pass == NULL)
	 {
	   if (DEBUGLEVEL > 0)
	     {
	       fprintf(dbf,"%s couldn't find guest account\n",timestring());
	       fflush(dbf);
	     }
	 
	 }		
}
else
#endif
{
pass = getpwnam(user);
if (pass == NULL)
	 {
	   if (DEBUGLEVEL > 0)
	     {
	       fprintf(dbf,"%s couldn't find account %s\n",user,timestring());
	       fflush(dbf);
	     }
	 
	 }
else
{
BOOL pwok = False;
#ifdef PWDAUTH
pwok = (pwdauth(user,password) == 0);
#else
strncpy(salt,pass->pw_passwd,2);
salt[2] = 0;
pwok = (strcmp(crypt(password,salt),pass->pw_passwd) == 0);
#endif
if (!pwok)
	 {
	   if (DEBUGLEVEL > 0)
	     {
	       fprintf(dbf,"%s invalid password\n",timestring());
	       fflush(dbf);
	     }
	 
	 pass = NULL;
	 }
}
}

if (pass == NULL)
	 return(-1);


Connections[cnum].uid = pass->pw_uid;
Connections[cnum].gid = pass->pw_gid;
Connections[cnum].connect_num = cnum;
Connections[cnum].service = snum;
Connections[cnum].dirptr = NULL;
Connections[cnum].dirpath[0] = 0;
strcpy(Connections[cnum].connectpath,Services[snum].path);
Connections[cnum].open = True;
return(cnum);
}


/****************************************************************************
parse service db structure
****************************************************************************/
BOOL read_service_entry(file,service,path,flags)
FILE *file;
char *service;
char *path;
char *flags;
{
char line[1000];
int nread;
*flags = 0;
*service = 0;
*path = 0;
*line = 0;

while (!feof(file))
  {
    *line = 0;
    fgets(line,1000,file);
    if ((*line == 0) || (strchr("#\n",*line) != NULL)) 
      continue;
    nread = sscanf(line,"%s %s %s",service,path,flags);
    if (nread == 0)
      continue;
    break;
  }

return((nread >= 2));
}


/****************************************************************************
load services database
****************************************************************************/
BOOL load_services(fname)
char *fname;
{
FILE *file;
int i;

file = fopen(fname,"r");
if (file == NULL) 
	 {
	   if (DEBUGLEVEL > 0)
	     {
	       fprintf(dbf,"%s couldn't open services file %s\n",timestring(),fname);
	       fflush(dbf);
	     }
	 
	 return(False);
	 }

for (i=0;i<MAX_SERVICES;i++)
	 {
	 char flags[100];
	 if (!read_service_entry(file,Services[i].service,Services[i].path,flags))
	   break;
	 strlower(flags);
	 Services[i].guest_ok = (strchr(flags,'g') != NULL);
	 Services[i].read_only = (strchr(flags,'w') == NULL);
	 Services[i].no_set_dir = (strchr(flags,'s') == NULL);
	 Services[i].available = True;
	 }

fclose(file);
return(True);
}



/****************************************************************************
open the socket communication
****************************************************************************/
void open_sockets()
{

/* We will abort gracefully when the client or remote system 
   goes away */
signal(SIGPIPE, abort);
Client = 0;

}

/****************************************************************************
close the socket communication
****************************************************************************/
void close_sockets()
{
/* nothing to do ! */
Client = 0;
}


/****************************************************************************
read data from the client
****************************************************************************/
int read_data(buffer,bufsize)
char *buffer;
int bufsize;
{
int ClientMask = 1<<Client;
int  SelectReadMask;/* select(2) mask modifiable by select(2) */
int  nready;        /* status return from select(2)           */
int nread;

do {
	 SelectReadMask = ClientMask;
	 nready = select(32,&SelectReadMask,(int *)0,(int *)0,0);
   } 
while( nready < 0  &&  errno == EINTR );


/* Query the system how many bytes are ready to be read */
ioctl(Client, FIONREAD, &nready);

/* Only try to get the smaller of nready and BUFSIZE */
nread = read(Client, buffer, nready < bufsize ? nready : bufsize);

/* return the number got */
return(nread);
}

/****************************************************************************
write data to the client
****************************************************************************/
void write_data(buffer,size)
char *buffer;
int size;
{
write(Client, buffer, size);
}


/****************************************************************************
reply to an init 
****************************************************************************/
int reply_init(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int outsize = 4;
memset(outbuf,0,outsize);
PVAL(outbuf,0,char) = 0202;

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s init\n",timestring());
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
create an error packet with dos error code enum.
****************************************************************************/
int error_packet(inbuf,outbuf,error_code)
char *inbuf;
char *outbuf;
int error_code;
{
int outsize = 39;
int cmd;
memset(outbuf,0,outsize);
cmd = PVAL(inbuf,8,char);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SSVAL(outbuf,28,SVAL(inbuf,28));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,30,SVAL(inbuf,30));
SSVAL(outbuf,9,1);
SSVAL(outbuf,11,error_code);

    
if (DEBUGLEVEL > 0)
  {
    fprintf(dbf,"%s error packet cmd=%d enum=%d\n",timestring(),(int)CVAL(inbuf,8),error_code);
    fflush(dbf);
  }
return(outsize);
}


/****************************************************************************
reply to a hello
****************************************************************************/
int reply_hello(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int outsize = 41;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = pHELLO;
SIVAL(outbuf,4,IVAL(inbuf,4));
SSVAL(outbuf,36,1);
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s hello\n",timestring());
    fflush(dbf);
  }
return(outsize);
}

/****************************************************************************
parse a connect packet
****************************************************************************/
void parse_connect(buf,service,user,password)
char *buf;
char *service;
char *user;
char *password;
{
char *p = buf + 40;
char *p2;

p2 = strrchr(p,'\\');
if (p2 == NULL)
	strcpy(service,p);
else
	strcpy(service,p2+1);

p += strlen(p) + 2;

strcpy(password,p);

*user = 0;
p = strchr(service,'%');
if (p != NULL)
	{
	*p = 0;
	strcpy(user,p+1);
	}
}


/****************************************************************************
reply to a connect
****************************************************************************/
int reply_connect(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char service[255];
char user[255];
char password[255];
int connection_num;
int outsize = 43;

parse_connect(inbuf,service,user,password);
connection_num = make_connection(service,user,password);

if (connection_num < 0)
  return(ERROR(eACCESS_DENIED));

memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
PVAL(outbuf,30,char) = PVAL(inbuf,30,char);
PVAL(outbuf,31,char) = PVAL(inbuf,31,char);
PVAL(outbuf,36,char) = 2;
PVAL(outbuf,37,char) = 'O';
PVAL(outbuf,38,char) = ' ';
SSVAL(outbuf,39,connection_num);
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));

if (DEBUGLEVEL > 0)
  {
    fprintf(dbf,"%s connect service=%s user=%s cnum=%d\n",timestring(),service,user,connection_num);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a fstat
****************************************************************************/
int reply_fstat(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int outsize;
int cnum,mode;
char name[255];
BOOL ok = False;
int len;
int dmode;

cnum = SVAL(inbuf,28);

strcpy(name,inbuf + 40);
unix_format(name);
mode = SVAL(inbuf,37);
len = strlen(name);

if (become_user(cnum))
  {
    ok = directory_exist(name);
    unbecome_user();
  }

if (!ok)
  return(ERROR(eACCESS_DENIED));

outsize = 39;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
PVAL(outbuf,30,char) = 19;
PVAL(outbuf,31,char) = '6';
SSVAL(outbuf,28,cnum);
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s fstat %s cnum=%d mode=%d\n",timestring(),name,cnum,mode);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a access
****************************************************************************/
int reply_access(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char fname[255];
int cnum;
int outsize;
struct stat sbuf;
BOOL ok;
int mode;
int size;

cnum = SVAL(inbuf,28);

strcpy(fname,inbuf + 40);
unix_format(fname);

if (become_user(cnum))
  {
    if (stat(fname,&sbuf) == 0)
      {
	mode = dos_mode(sbuf.st_mode);
	size = sbuf.st_size;
	ok = True;
      }
    unbecome_user();
  }

if (!ok)
  return(ERROR(eACCESS_DENIED));

outsize = 40;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));

PVAL(outbuf,28,char) = cnum;
PVAL(outbuf,37,char) = mode;
SIVAL(outbuf,43,size);

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s access name=%s mode=%d size=%d\n",timestring(),fname,mode,size);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a chmod
****************************************************************************/
int reply_chmod(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char fname[255];
int cnum;
int outsize;
BOOL ok;
int mode;

cnum = SVAL(inbuf,28);

strcpy(fname,inbuf + 56);
unix_format(fname);
mode = PVAL(inbuf,37,char);

if (!OPEN_CNUM(cnum) || Services[Connections[cnum].service].read_only)
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
  {
    ok =  (chmod(fname,unix_mode(mode)) == 0);
    unbecome_user();
  }

if (!ok)
	return(ERROR(eACCESS_DENIED));

outsize = 39;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));

PVAL(outbuf,28,char) = cnum;
PVAL(outbuf,30,char) = PVAL(inbuf,30,char);
PVAL(outbuf,31,char) = PVAL(inbuf,31,char);


if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s chmod name=%s mode=%d\n",timestring(),fname,mode);
    fflush(dbf);
  }

return(outsize);
}


/****************************************************************************
return number of 1K blocks available on a path and total number 
****************************************************************************/
void disk_free(path,dfree,dsize)
char *path;
long *dfree;
long *dsize;
{
#ifdef ULTRIX
struct fs_data buf;

getmnt(NULL,&buf,0,STAT_ONE,path);
*dfree = buf.fd_req.bfreen;
*dsize = buf.fd_req.btot;
#else
*dfree = 16000;
*dsize = 20000;
#endif
}

/****************************************************************************
reply to a dfree
****************************************************************************/
int reply_dfree(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int cnum;
int outsize;
long dfree,dsize;
BOOL ok=False;

cnum = SVAL(inbuf,28);

if (become_user(cnum))
  {
    disk_free("./",&dfree,&dsize);
    ok = True;
    unbecome_user();
  }

if (!ok)
	return(ERROR(eACCESS_DENIED));

outsize = 49;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SSVAL(outbuf,28,cnum);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
PVAL(outbuf,30,char) = 21;
PVAL(outbuf,31,char) = '6';
PVAL(outbuf,36,char) = 5;
PVAL(outbuf,37,char) = '\347';
PVAL(outbuf,38,char) = '%';
PVAL(outbuf,39,char) = 1;
PVAL(outbuf,42,char) = 4;
SSVAL(outbuf,43,dfree);
PVAL(outbuf,45,char) = '\370';

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s dfree cnum=%d dfree=%d\n",timestring(),cnum,dfree);
    fflush(dbf);
  }

return(outsize);
}


/****************************************************************************
reply to a dir
****************************************************************************/
int reply_dir(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char mask[255];
char directory[255];
char fname[255];
int size,mode,entry;
time_t date;
int dirtype,startcode;
int cnum;
int outsize;
int numentries = 0;
BOOL finished = False;
int maxentries = 22;
int i;
BOOL startdir;
char *p;
BOOL ok = False;

startcode = SVAL(inbuf,41);
startdir = (PVAL(inbuf,44,char) == '\\');

if (startdir)
  {
    strcpy(directory,inbuf+44);
    unix_format(directory);
    p = strrchr(directory,'/');
    if (p == NULL) 
      {strcpy(mask,directory);*directory = 0;}
    else
      {*p = 0;strcpy(mask,p+1);}
    if (strlen(directory) == 0)
      strcpy(directory,"./");
    strlower(mask);
  }
else
  {
    strncpy(mask,inbuf + 49,11);
    mask[11] = 0;
    strlower(mask);
  }

cnum = SVAL(inbuf,28);
dirtype = SVAL(inbuf,39);

if (become_user(cnum))
    {
      ok = True;
      if (startdir && !start_dir(cnum,directory)) 
	{
	  unbecome_user();
	  return(ERROR(eACCESS_DENIED));
	}

      if (dirtype == 8)
	{
	  make_dir_struct(outbuf + 45,"Remote",0,aVOLID,0,0);
	  numentries = 1;
	}
      else
	{
	  for (i=0;(i<maxentries) && !finished;i++)
	    {
	      p = outbuf+45+numentries*DIR_STRUCT_SIZE;
	      finished = !get_dir_entry(cnum,mask,dirtype,fname,&size,&mode,&entry,&date);
	      if (!finished)
		{
		  strupper(fname);
		  make_dir_struct(p,fname,size,mode,entry,date);
		  numentries++;
		}
	    }
	}
      unbecome_user();
    }

if (!ok || (numentries < 0))
	return(ERROR(eACCESS_DENIED));

outsize = 44;
memset(outbuf,0,outsize);
outsize += DIR_STRUCT_SIZE*numentries;
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
PVAL(outbuf,28,char) = cnum;

PVAL(outbuf,9,char) = 1;
PVAL(outbuf,11,char) = 18;
PVAL(outbuf,30,char) = 21;
PVAL(outbuf,31,char) = '6';
PVAL(outbuf,36,char) = 1;
SSVAL(outbuf,37,numentries);
PVAL(outbuf,39,char) = 3;
SSVAL(outbuf,41,5);
if (finished && (numentries > 0))
  {
  PVAL(outbuf,43,char) = '\260';
  PVAL(outbuf,44,char) = 2;
  }
if (finished && (numentries == 0))
  {
  PVAL(outbuf,43,char) = 0;
  PVAL(outbuf,44,char) = 0;
  }
if (!finished)
  {
  PVAL(outbuf,43,char) = '\262';
  PVAL(outbuf,44,char) = 3;
  }

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s dir mask=%s directory=%s cnum=%d dirtype=%d numentries=%d\n",timestring(),mask,directory,cnum,dirtype,numentries);
    fflush(dbf);
  }

return(outsize);
}


/****************************************************************************
reply to an open
****************************************************************************/
int reply_open(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char fname[255];
int cnum;
int fnum = -1;
int outsize;
int mode;
int openmode = 0;
BOOL ok = False;
int size = 0;

cnum = SVAL(inbuf,28);
mode = SVAL(inbuf,37);
strcpy(fname,inbuf+44);
unix_format(fname);

switch (mode)
  {
  case 0 : openmode = O_RDONLY; break;
  case 1 : openmode = O_WRONLY; break;
  case 2 : openmode = O_RDWR; break;
  default: openmode = O_RDONLY;
  }


if ((openmode != O_RDONLY) && !CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
  {
    fnum = find_free_file();
    if (fnum >= 0)
      {
	 Files[fnum].fd = open(fname,openmode,CREATE_FILE_MODE);
	 if (Files[fnum].fd >= 0)
	   {
	     struct stat sbuf;
	     if (fstat(Files[fnum].fd,&sbuf) == 0)
	       size = sbuf.st_size;
	     Files[fnum].open = True;
	     Files[fnum].cnum = cnum;
	   }
	 else
	   fnum = -1;
      }
    ok = True;
    unbecome_user();
  }

if ((fnum < 0) || !ok)
	return(ERROR(eFILE_NOT_FOUND));

outsize = 53;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,28,cnum);
SSVAL(outbuf,37,fnum);
SSVAL(outbuf,49,mode);
PVAL(outbuf,40,char) = '\267';
PVAL(outbuf,41,char) = '\212';
SSVAL(outbuf,36,7);

PVAL(outbuf,39,char) = ' ';
PVAL(outbuf,41,char) = '`';
PVAL(outbuf,42,char) = '\'';
PVAL(outbuf,43,char) = 'c';
PVAL(outbuf,44,char) = ')';
SIVAL(outbuf,45,size);
SSVAL(outbuf,30,SVAL(inbuf,30));

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s open %s fd=%d fnum=%d cnum=%d mode=%d omode=%d\n",timestring(),fname,Files[fnum].fd,fnum,cnum,mode,openmode);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a create
****************************************************************************/
int reply_create(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char fname[255];
int cnum;
int fnum = -1;
int outsize;
int createmode;
mode_t unixmode;
BOOL ok = False;

cnum = SVAL(inbuf,28);
createmode = PVAL(inbuf,37,char);
strcpy(fname,inbuf+46);
unix_format(fname);

unixmode = unix_mode(createmode);

unixmode |= 0200;

if (!CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
 {
   fnum = find_free_file();
   if (fnum >= 0)
     {
	Files[fnum].fd = creat(fname,unixmode);
	if (Files[fnum].fd >= 0)
	  {
	    Files[fnum].open = True;
	    Files[fnum].cnum = cnum;
	  }
	else
	  fnum = -1;
     }
   ok = True;
   unbecome_user();
 }

if (!ok || (fnum < 0))
	return(ERROR(eACCESS_DENIED));

outsize = 41;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,28,cnum);
SSVAL(outbuf,37,fnum);
PVAL(outbuf,30,char) = '\270';
PVAL(outbuf,31,char) = '\212';

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s create %s fd=%d fnum=%d cnum=%d dmode=%d umode=%o\n",timestring(),fname,Files[fnum].fd,fnum,cnum,createmode,unixmode);
    fflush(dbf);
  }


return(outsize);
}


/****************************************************************************
reply to a delete
****************************************************************************/
int reply_delete(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int outsize;
char name[255];
int cnum;
BOOL ok = False;

cnum = SVAL(inbuf,28);
strcpy(name,inbuf + 42);
unix_format(name);

if (!CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
  {
    ok = (unlink(name) == 0);
    unbecome_user();
  }

if (!ok)
	return(ERROR(eACCESS_DENIED));

outsize = 38;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SSVAL(outbuf,28,cnum);
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
PVAL(outbuf,30,char) = 19;
PVAL(outbuf,31,char) = '6';

return(outsize);
}

/****************************************************************************
reply to a read
****************************************************************************/
int reply_read(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int cnum,numtoread,fnum;
int nread = -1;
char *data;
int startpos;
int outsize;
BOOL ok = False;
memset(outbuf,0,52);

fnum = SVAL(inbuf,37);
cnum = SVAL(inbuf,28);
numtoread = SVAL(inbuf,39);
startpos = IVAL(inbuf,41);
data = outbuf + 52;

numtoread = MIN(bufsize-52,numtoread);

if (become_user(cnum))
 {
   if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
     {
        lseek(Files[fnum].fd,startpos,SEEK_SET);
	nread = read(Files[fnum].fd,data,numtoread);
      }
   ok = True;
   unbecome_user();
 }


if ((nread < 0) || !ok)
	return(ERROR(eACCESS_DENIED));

outsize = nread + 52;
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SSVAL(outbuf,28,cnum);
SSVAL(outbuf,37,nread);
SSVAL(outbuf,50,nread);

SSVAL(outbuf,30,SVAL(inbuf,30));


if (nread == numtoread)
 {
   PVAL(outbuf,47,char) = 3;
   PVAL(outbuf,48,char) = 4;
   PVAL(outbuf,49,char) = 1;
 }
else
 {
   PVAL(outbuf,47,char) = '\333';
   PVAL(outbuf,48,char) = 0;
   PVAL(outbuf,49,char) = 1;
 }

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s read fnum=%d cnum=%d num=%d nread=%d\n",timestring(),fnum,cnum,numtoread,nread);
    fflush(dbf);
  }


return(outsize);
}

/****************************************************************************
reply to a write
****************************************************************************/
int reply_write(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int cnum,numtowrite,fnum;
int nwritten = -1;
int outsize;
int startpos;
char *data;
BOOL ok = False;

fnum = SVAL(inbuf,37);
cnum = SVAL(inbuf,28);
numtowrite = SVAL(inbuf,39);
startpos = IVAL(inbuf,41);
data = inbuf + 52;

while (length - 52 < numtowrite)
  {
    int nread;
    nread = read_data(InBuffer + length,BUFFER_SIZE - length);
    if (nread == 0) break;
    length += nread;
  }

if (!CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
 {
   if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
     {
       lseek(Files[fnum].fd,startpos,SEEK_SET);
       nwritten = write(Files[fnum].fd,data,numtowrite);
     }
   unbecome_user();
   ok = True;
 }

if ((nwritten <= 0) || !ok)
	return(ERROR(eACCESS_DENIED));

outsize = 40;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));

SSVAL(outbuf,28,cnum);
SSVAL(outbuf,37,nwritten);

PVAL(outbuf,30,char) = 19;
PVAL(outbuf,31,char) = '6';
PVAL(outbuf,36,char) = 1;

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s write fnum=%d cnum=%d num=%d wrote=%d\n",timestring(),fnum,cnum,numtowrite,nwritten);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a close
****************************************************************************/
int reply_close(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int fnum,cnum;
int ret=0;
int outsize = 39;
BOOL ok = False;

memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = pCLOSE;
SIVAL(outbuf,4,IVAL(inbuf,4));

cnum = SVAL(inbuf,28);
fnum = SVAL(inbuf,37);


if (become_user(cnum))
 {
   if (OPEN_FNUM(fnum) && (Files[fnum].cnum == cnum))
	ret = close(Files[fnum].fd);
   Files[fnum].open = False;
   ok = True;
   unbecome_user();
 }

if ((ret < 0) || !ok)
	return(ERROR(eACCESS_DENIED));

SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,28,cnum);
PVAL(outbuf,30,char) = 19;
PVAL(outbuf,31,char) = '6';


if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s close fd=%d fnum=%d cnum=%d ret=%d\n",timestring(),Files[fnum].fd,fnum,cnum,ret);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a goodbye
****************************************************************************/
int reply_goodbye(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int cnum;
int outsize = 40;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = pGOODBYE;
SIVAL(outbuf,4,IVAL(inbuf,4));

cnum = SVAL(inbuf,28);
if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s Goodbye cnum=%d\n",timestring(),cnum);
    fflush(dbf);
  }


SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
return(outsize);
}

/****************************************************************************
reply to a mkdir
****************************************************************************/
int reply_mkdir(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char directory[255];
int cnum;
int outsize,ret;
BOOL ok = False;

strcpy(directory,inbuf + 40);
cnum = SVAL(inbuf,28);
unix_format(directory);

if (!CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
 {
   ret = mkdir(directory,CREATE_DIR_MODE);
   ok = True;
   unbecome_user();
 }

if ((ret < 0) || !ok)
	return(ERROR(eACCESS_DENIED));

outsize = 39;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,28,cnum);

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s mkdir %s cnum=%d ret=%d\n",timestring(),directory,cnum,ret);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a rmdir
****************************************************************************/
int reply_rmdir(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char directory[255];
int cnum;
int outsize;
BOOL ok = False;

cnum = SVAL(inbuf,28);
strcpy(directory,inbuf + 40);
unix_format(directory);


if (!CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
 {
   ok = (rmdir(directory) == 0);
   unbecome_user();
 }

if (!ok)
	return(ERROR(eINVALID_PATH));

outsize = 36;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SSVAL(outbuf,28,cnum);
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,30,SVAL(inbuf,30));

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s rmdir %s\n",timestring(),directory);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a rename
****************************************************************************/
int reply_rename(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
char oldname[255];
char newname[255];
int cnum;
int outsize,ret;
char *p;
BOOL ok = False;

strcpy(oldname,inbuf + 42);
p = inbuf+42;
p += strlen(p) + 2;
strcpy(newname,p);

cnum = SVAL(inbuf,28);
unix_format(oldname);
unix_format(newname);

if (!CAN_WRITE(cnum))
	return(ERROR(eACCESS_DENIED));

if (become_user(cnum))
 {
   ok = (rename(oldname,newname) == 0);
   unbecome_user();
 }

if (!ok)
	return(ERROR(eACCESS_DENIED));

outsize = 39;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,28,cnum);
SSVAL(outbuf,30,SVAL(inbuf,30));

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s rename %s to %s cnum=%d ret=%d\n",timestring(),oldname,newname,cnum,ret);
    fflush(dbf);
  }

return(outsize);
}

/****************************************************************************
reply to a setdir
****************************************************************************/
int reply_setdir(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int cnum,snum;
int outsize;
BOOL ok = False;
char newdir[200];
struct stat st;

cnum = SVAL(inbuf,28);
if (!OPEN_CNUM(cnum))
	return(ERROR(eACCESS_DENIED));

snum = Connections[cnum].service;
if (Services[snum].no_set_dir)
	return(ERROR(eACCESS_DENIED));

strcpy(newdir,inbuf + 40);
strlower(newdir);

if (strlen(newdir) == 0)
  ok = True;
else
  {
    ok = directory_exist(newdir);
    if (ok)
      strcpy(Connections[cnum].connectpath,newdir);
  }

if (!ok)
	return(ERROR(eINVALID_PATH));

outsize = 36;
memset(outbuf,0,outsize);
PVAL(outbuf,8,char) = PVAL(inbuf,8,char);
SIVAL(outbuf,4,IVAL(inbuf,4));
SIVAL(outbuf,0,int32_byte_swap(outsize - 4));
SSVAL(outbuf,28,cnum);
PVAL(outbuf,10,char) = PVAL(inbuf,10,char);

if (DEBUGLEVEL > 1)
  {
    fprintf(dbf,"%s setdir %s cnum=%d\n",timestring(),newdir,cnum);
    fflush(dbf);
  }

return(outsize);
}


/****************************************************************************
reply to an unknown type
****************************************************************************/
int reply_unknown(inbuf,outbuf,length,bufsize)
char *inbuf;
char *outbuf;
int length;
int bufsize;
{
int outsize;
int cnum;
int type;
cnum = SVAL(inbuf,28);
type = PVAL(inbuf,8,char);

if (DEBUGLEVEL > 0)
  {
    fprintf(dbf,"%s unknown command type: cnum=%d type=%d\n",timestring(),cnum,type);
    fflush(dbf);
  }


return(ERROR(eACCESS_DENIED));
}


/****************************************************************************
construct a reply to the incoming packet
****************************************************************************/
int construct_reply(inbuf,outbuf,size,bufsize)
char *inbuf;
char *outbuf;
int size;
int bufsize;
{
int type = PVAL(inbuf,8,char);
int init_code = PVAL(inbuf,0,char);
int sub_code = PVAL(inbuf,3,char);
int outsize;

if (init_code != 0)
	outsize = reply_init(inbuf,outbuf,size,bufsize);
else
switch (type)
	{
	case pHELLO:
		outsize = reply_hello(inbuf,outbuf,size,bufsize);
		break;
	case pACCESS:
		outsize = reply_access(inbuf,outbuf,size,bufsize);
		break;
	case pCHMOD:
		outsize = reply_chmod(inbuf,outbuf,size,bufsize);
		break;
	case pCONNECT:
		outsize = reply_connect(inbuf,outbuf,size,bufsize);
		break;
	case pFSTAT:
		outsize = reply_fstat(inbuf,outbuf,size,bufsize);
		break;
	case pDIR:
		outsize = reply_dir(inbuf,outbuf,size,bufsize);
		break;
	case pOPEN:
		outsize = reply_open(inbuf,outbuf,size,bufsize);
		break;
	case pCREATE:
		outsize = reply_create(inbuf,outbuf,size,bufsize);
		break;
	case pDELETE:
		outsize = reply_delete(inbuf,outbuf,size,bufsize);
		break;
	case pREAD:
		outsize = reply_read(inbuf,outbuf,size,bufsize);
		break;
	case pWRITE:
		outsize = reply_write(inbuf,outbuf,size,bufsize);
		break;
	case pCLOSE:
		outsize = reply_close(inbuf,outbuf,size,bufsize);
		break;		
	case pGOODBYE:
		outsize = reply_goodbye(inbuf,outbuf,size,bufsize);
		break;		
	case pMKDIR:
		outsize = reply_mkdir(inbuf,outbuf,size,bufsize);
		break;	
	case pRMDIR:
		outsize = reply_rmdir(inbuf,outbuf,size,bufsize);
		break;	
	case pDFREE:
		outsize = reply_dfree(inbuf,outbuf,size,bufsize);
		break;	
	case pRENAME:
		outsize = reply_rename(inbuf,outbuf,size,bufsize);
		break;	
	case pSETDIR:
		outsize = reply_setdir(inbuf,outbuf,size,bufsize);
		break;	
	default:
		outsize = reply_unknown(inbuf,outbuf,size,bufsize);
		break;
	}


return(outsize);
}


/****************************************************************************
process commands from the client
****************************************************************************/
void process()
{
int nread;

InBuffer = (char *)malloc(BUFFER_SIZE);
OutBuffer = (char *)malloc(BUFFER_SIZE);
if ((InBuffer == NULL) || (OutBuffer == NULL)) 
  return;

while (True)
	{
	nread = read_data(InBuffer,BUFFER_SIZE);
	if (nread <= 0) return;
	if (pcsain)
	  {
	    fprintf(pcsain,"\n%s Transaction %d\n",timestring(),trans_num);
	    fwrite(InBuffer,nread,1,pcsain);
	    fflush(pcsain);
	  }
	if (DEBUGLEVEL > 2)
	  {
	    fprintf(dbf,"%s Transaction %d\n",timestring(),trans_num);
	    fflush(dbf);
	  }
	
	nread = construct_reply(InBuffer,OutBuffer,nread,BUFFER_SIZE);

	if (pcsaout)
	  {
	    fprintf(pcsaout,"\n%s Transaction %d\n",timestring(),trans_num++);
	    fwrite(OutBuffer,nread,1,pcsaout);
	    fflush(pcsaout);
	  }
	
	write_data(OutBuffer,nread);
	trans_num++;
	}
}


/****************************************************************************
initialise connect, service and file structs
****************************************************************************/
void init_structs()
{
int i;
for (i=0;i<MAX_SERVICES;i++)
	Services[i].available = False;
for (i=0;i<MAX_CONNECTIONS;i++)
	Connections[i].open = False;
for (i=0;i<MAX_OPEN_FILES;i++)
	Files[i].open = False;
}


	

/****************************************************************************
main program
****************************************************************************/
int main()
{


NeedSwap = big_endian();

umask(!(CREATE_FILE_MODE | CREATE_DIR_MODE));
if (DEBUGLEVEL > 2)
  {
    pcsain = fopen(PCSAIN,"a");
    pcsaout = fopen(PCSAOUT,"a");
  }

if (DEBUGLEVEL > 0)
  {
    dbf = fopen(DEBUGFILE,"a");
    fprintf(dbf,"%s server started\n",timestring());
    fflush(dbf);
  }

init_structs();

if (!load_services(SERVICES))
  return(-1);	

if (DEBUGLEVEL > 1)
{
  fprintf(dbf,"%s loaded services\n",timestring());
  fflush(dbf);
}

	
open_sockets();
process();
close_sockets();
fclose(dbf);
return(0);
}




