/* 
   Unix SMB/Netbios implementation.
   Version 1.5.
   Copyright (C) Andrew Tridgell 1992,1993
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

int DEBUGLEVEL = 0;

BOOL passive = False;

/* these are some file handles where debug info will be stored */
FILE *dbf = NULL;
FILE *pcsain=NULL;
FILE *pcsaout=NULL;

/* the client file descriptor */
int Client = 0;

/* the last IP received from */
unsigned char lastip[4] = {0,0,0,0};


int trans_num = 0;


/* this is set to true on a big_endian machine (like a sun sparcstation)
this means that all shorts and ints must be byte swapped before being
put in the buffer */
BOOL NeedSwap=False;

/*******************************************************************
write an debug message on the debugfile. The first arg is the debuglevel.
********************************************************************/
void Debug(va_alist)
va_dcl
{
va_list ap;
char *format_str;
int debug_level;

if (!dbf) return;

va_start(ap);
debug_level = va_arg(ap,int);
format_str = va_arg(ap,char *);

if (DEBUGLEVEL >= debug_level)
  {
    vfprintf(dbf,format_str,ap);
    fflush(dbf);
  }
va_end(ap);
}


#ifdef STRING_DEBUG
#define LONG_LEN (MAXPATHLEN/3)
int mystrlen(char *s)
{
  int n=0;
  while (*s++)
    n++;
  if (n > LONG_LEN)
    Debug(0,"ERROR: long string\n");
  return n;
}

char *mystrchr(char *s,char c)
{
if (strlen(s) > LONG_LEN)
  Debug(0,"ERROR: long string\n");
while (*s)
  {
    if (*s == c) break;
    s++;
  }
if (*s == c)
  return s;
else
  return NULL;
}


char *mystrrchr(char *s,char c)
{
char *s2;
if (strlen(s) > LONG_LEN)
  Debug(0,"ERROR: long string\n");

s2 = s + strlen(s);

while (s != s2)
  {
    if (*s2 == c) break;
    s2--;
  }
if (*s2 == c)
  return s2;
else
  return NULL;
}

char *mystrcpy(char *d,char *s)
{
  if (strlen(s) > LONG_LEN)
    Debug(0,"ERROR: long string\n");
  while ((*d++  = *s++));
}

char *mystrncpy(char *d,char *s,int n)
{
  if (strlen(s) > LONG_LEN)
    Debug(0,"ERROR: long string\n");
  while ((*d++  = *s++) && n--);
}

char *mystrcat(char *d,char *s)
{
  if (strlen(s) > LONG_LEN || strlen(d)>LONG_LEN)
    Debug(0,"ERROR: long string\n");
  d+=strlen(d);
  while ((*d++  = *s++));
}

void mymemcpy(char *d,char *s,int n)
{
if (n > LONG_LEN)
  Debug(0,"ERROR: long copy\n");
while (n--)
  *d++ = *s++;
}

void mymemset(char *d,char c,int n)
{
if (n > LONG_LEN)
  Debug(0,"ERROR: long set\n");
while (n--)
  *d++ = c;
}
#endif


#ifndef SOLARIS
/****************************************************************************
exchange real and effective uids 
****************************************************************************/
void exchange_uids(void )
{
  if (geteuid() == 0 || getuid() == 0)
    {
      if (setreuid (geteuid (), getuid ()) ||
	  setregid (getegid (), getgid ()))
	{
	  Debug(0,"Cannot exchange real/effective uid or gid!\n");
	  exit(1);
	}
    }
}  
#endif

/****************************************************************************
  return the date and time as a string
****************************************************************************/
char *timestring(void )
{
  static char TimeBuf[100];
  time_t t;
  t = time(NULL);
  strftime(TimeBuf,100,"%D %r",localtime(&t));
  return(TimeBuf);
}

/****************************************************************************
interpret the weird netbios "name"
****************************************************************************/
void name_interpret(char *in,char *out)
{
char *p = in;
int len = (*in++) / 2;
while (len--)
  {
    *out = ((in[0]-'A')<<4) + (in[1]-'A');
    in += 2;
    out++;
  }
*out = 0;
strcat(out,in);
}

/****************************************************************************
mangle a name into netbios format
****************************************************************************/
void name_mangle(unsigned char *in,unsigned char *out)
{
  int len = 2*strlen(in);
  int pad = 0;

  if (len/2 < 16)
    pad = 16 - len;

  *out++ = 2*strlen(in);
  while (*in)
    {
      out[0] = (in[0]>>4) + 'A';
      out[1] = (in[0] & 0xF) + 'A';
      in++;
      out+=2;
    }
  
  while (pad--)
    {
      out[0] = 'C';
      out[1] = 'A';
      out+=2;
    }
  
  *out = 0;
}

/*******************************************************************
  byte swap an object - the byte order of the object is reversed
********************************************************************/
void object_byte_swap(void *obj,int size)
{
  int i;
  char c;
  char *p1 = (char *)obj;
  char *p2 = p1 + size - 1;
  
  size /= 2;
  
  for (i=0;i<size;i++)
    {
      c = *p1;
      *p1 = *p2;
      *p2 = c;
      p1++;
      p2--;
    }
}

/****************************************************************************
  byte swap a int16
****************************************************************************/
int16 int16_byte_swap(int16 x)
{
  int16 res;
  res = x;
  object_byte_swap(&res,sizeof(int16));
  return(res);
}

/****************************************************************************
  byte swap a int32
****************************************************************************/
int32 int32_byte_swap(int32 x)
{
  int32 res;
  res = x;
  object_byte_swap(&res,sizeof(int32));
  return(res);
}

/*******************************************************************
  check if a file exists
********************************************************************/
BOOL file_exist(char *fname)
{
  struct stat st;
  if (stat(fname,&st) != 0) 
    return(False);
  
  return(S_ISREG(st.st_mode));
}

/*******************************************************************
  check if a directory exists
********************************************************************/
BOOL directory_exist(char *dname)
{
  struct stat st;
  if (stat(dname,&st) != 0) 
    return(False);
  
  return(S_ISDIR(st.st_mode));
}

/*******************************************************************
returns the size in bytes of the named file
********************************************************************/
uint32 file_size(char *file_name)
{
  struct stat buf;
  buf.st_size = 0;
  stat(file_name,&buf);
  return(buf.st_size);
}


/*******************************************************************
  create a 32 bit dos packed date/time from some parameters
********************************************************************/
uint32 make_dos_date(time_t unixdate)
{
  int32 ret;
  unsigned char *p;
  struct tm *t;

  t = localtime(&unixdate);
  p = (unsigned char *)&ret;
  p[0] = (t->tm_sec/2) + ((t->tm_min & 0x7) << 5);
  p[1] = (t->tm_min >> 3) + (t->tm_hour << 3);
  p[2] = t->tm_mday + (((t->tm_mon+1) & 0x7) << 5);
  p[3] = ((t->tm_mon+1) >> 3) + ((t->tm_year-80) << 1);

  return(ret);
}



/*******************************************************************
  interpret a 32 bit dos packed date/time to some parameters
********************************************************************/
void interpret_dos_date(uint32 date,int *year,int *month,int *day,int *hour,int *minute,int *second)
{
  unsigned char *p = (unsigned char *)&date;

  *second = 2*(p[0] & 0x1F);
  *minute = (p[0]>>5) + ((p[1]&0x7)<<3);
  *hour = (p[1]>>3);
  *day = p[2]&0x1F;
  *month = (p[2]>>5) + ((p[3]&0x1)<<3) - 1;
  *year = (p[3]>>1) + 80;
}

/*******************************************************************
  create a unix date from a dos date
********************************************************************/
time_t make_unix_date(uint32 dos_date)
{
  struct tm t;
  
  interpret_dos_date(dos_date,&t.tm_year,&t.tm_mon,
		     &t.tm_mday,&t.tm_hour,&t.tm_min,&t.tm_sec);
  t.tm_wday = 1;
  t.tm_yday = 1;
  t.tm_isdst = 0;
  Debug(4,"year=%d month=%d day=%d hr=%d min=%d sec=%d\n",t.tm_year,t.tm_mon,
	 t.tm_mday,t.tm_hour,t.tm_sec);
  return(mktime(&t));
}

/*******************************************************************
  true if the machine is big endian
********************************************************************/
BOOL big_endian(void )
{
  int x = 2;
  char *s;
  s = (char *)&x;
  return(s[0] == 0);
}

/*******************************************************************
  compare 2 strings 
********************************************************************/
BOOL strequal(char *s1,char *s2)
{
  if (!s1 || !s2) return(False);
  
  return(strcasecmp(s1,s2)==0);
}


/*******************************************************************
  convert a string to lower case
********************************************************************/
void strlower(char *s)
{
  while (*s)
    {
      *s = tolower(*s);
      s++;
    }
}

/*******************************************************************
  convert a string to upper case
********************************************************************/
void strupper(char *s)
{
  while (*s)
    {
      *s = toupper(*s);
      s++;
    }
}

/****************************************************************************
  string replace
****************************************************************************/
void string_replace(char *s,char old,char new)
{
  while (*s)
    {
      if (old == *s)
	*s = new;
      s++;
    }
}

/****************************************************************************
  make a file into unix format
****************************************************************************/
void unix_format(char *fname)
{
  char namecopy[MAXPATHLEN]="";
  string_replace(fname,'\\','/');
  strlower(fname);
  if (*fname == '/')
    {
      strcpy(namecopy,fname);
      strcpy(fname,".");
      strcat(fname,namecopy);
    }
}


/****************************************************************************
  set a value at buf[pos] to integer val
****************************************************************************/
void SIVAL(char *buf,int pos,uint32 val)
{
  val = ISWP(val);
  memcpy(buf + pos,(char *)&val,sizeof(int));
}

/****************************************************************************
  set a value at buf[pos] to int16 val
****************************************************************************/
void SSVAL(char *buf,int pos,uint16 val)
{
  val = SSWP(val);
  memcpy(buf + pos,(char *)&val,sizeof(int16));
}

/****************************************************************************
  get a 32 bit integer value
****************************************************************************/
int IVAL(char *buf,int pos)
{
  int res;
  memcpy((char *)&res,buf + pos,sizeof(int));
  res = ISWP(res);
  return(res);
}


/****************************************************************************
  get a int16 value
****************************************************************************/
uint16 SVAL(char *buf,int pos)
{
  uint16 res;
  memcpy((char *)&res,buf + pos,sizeof(uint16));
  res = SSWP(res);
  return(res);
}


/*******************************************************************
  show a smb message structure
********************************************************************/
void show_msg(char *buf)
{
  int i;
  Debug(1,"size=%d\nsmb_com=0x%x\nsmb_rcls=%d\nsmb_reh=%d\nsmb_err=%d\n",
	  smb_len(buf),
	  (int)CVAL(buf,smb_com),
	  (int)CVAL(buf,smb_rcls),
	  (int)CVAL(buf,smb_reh),
	  (int)SVAL(buf,smb_err));
  Debug(1,"smb_tid=%d\nsmb_pid=%d\nsmb_uid=%d\nsmb_mid=%d\nsmt_wct=%d\n",
	  (int)SVAL(buf,smb_tid),
	  (int)SVAL(buf,smb_pid),
	  (int)SVAL(buf,smb_uid),
	  (int)SVAL(buf,smb_mid),
	  (int)CVAL(buf,smb_wct));
  for (i=0;i<(int)CVAL(buf,smb_wct);i++)
    Debug(1,"smb_vwv[%d]=%d\n",i,(int)SVAL(buf,smb_vwv+2*i));
  Debug(1,"smb_bcc=%d\n",(int)SVAL(buf,smb_vwv+2*(CVAL(buf,smb_wct))));
  
}

/*******************************************************************
  return the length of an smb packet
********************************************************************/
int smb_len(char *buf)
{
  int msg_flags = CVAL(buf,1);
  int len = (CVAL(buf,2) << 8) + CVAL(buf,3);

  if (msg_flags & 1)
    len += 1<<16;

  return len;
}

/*******************************************************************
  set the length of an smb packet
********************************************************************/
void smb_setlen(char *buf,int len)
{
  CVAL(buf,3) = len & 0xFF;
  CVAL(buf,2) = (len >> 8) & 0xFF;

  if (len >= (1 << 16))
    CVAL(buf,1) |= 1;
}

/*******************************************************************
  setup the word count and byte count for a smb message
********************************************************************/
int set_message(char *buf,int num_words,int num_bytes)
{
  CVAL(buf,smb_wct) = num_words;
  SSVAL(buf,smb_vwv + num_words*sizeof(WORD),num_bytes);  
  memset(buf + smb_size,0,num_words*2 + num_bytes);
  smb_setlen(buf,smb_size + num_words*2 + num_bytes - 4);
  return (smb_size + num_words*2 + num_bytes);
}

/*******************************************************************
  return a pointer to the smb_buf data area
********************************************************************/
char *smb_buf(char *buf)
{
  return (buf + smb_size + CVAL(buf,smb_wct)*2);
}

/*******************************************************************
return the absolute current directory path
********************************************************************/
char *GetWd(char *s)
{
#if (defined(LINUX) || defined(SOLARIS))
  return(getcwd(s,MAXPATHLEN));
#else
  return(getwd(s));
#endif
}

/*******************************************************************
reduce a file name, removing .. elements and checking that 
it is below dir in the heirachy
********************************************************************/
BOOL reduce_name(char *s,char *dir)
{
#ifndef REDUCE_PATHS
return True;
#else
  char dir2[MAXPATHLEN]="";
  char wd[MAXPATHLEN]="";
  char basename[MAXPATHLEN]="";
  char newname[MAXPATHLEN]="";
  char *p=NULL;
  BOOL relative = (*s != '/');

  Debug(2,"reduce_name [%s] [%s]\n",s,dir);

  /* remove any double slashes */
  while ((p = strstr(s,"//")) != NULL)
    {
      while (*p)
	{
	  p[0] = p[1];
	  p++;
	}
    }

  if (!GetWd(wd))
    {
      Debug(1,"couldn't getwd for %s %s\n",s,dir);
      return(False);
    }

  if (chdir(dir) != 0)
    {
      Debug(3,"couldn't chdir to %s\n",dir);
      return(False);
    }

  if (!GetWd(dir2))
    {
      Debug(1,"couldn't getwd for %s\n",dir);
      chdir(wd);
      return(False);
    }

  strcpy(basename,s);
  p = strrchr(basename,'/');
  if (p && (p != basename))
    {
      *p = 0;
      if (strcmp(p+1,".")==0)
	p[1]=0;
      if (strcmp(p+1,"..")==0)
	{
	  *p = '/';
	  p += 3;
	  p[1] = 0;
	}
    }

  if (chdir(basename) != 0)
    {
      Debug(3,"couldn't chdir for %s %s basename=%s\n",s,dir,basename);
      return(False);
    }

  if (!GetWd(newname))
    {
      chdir(wd);
      Debug(1,"couldn't get wd for %s %s\n",s,dir2);
      return(False);
    }

  if (p && (p != basename))
    {
      strcat(newname,"/");
      strcat(newname,p+1);
    }

  {
    int l = strlen(dir2);    
    if (dir2[l-1] == '/')
      l--;

    if (strncmp(newname,dir2,l) != 0)
      {
	Debug(2,"Bad access attempt? s=%s dir=%s newname=%s l=%d\n",s,dir2,newname,l);
	return(False);
      }

    if (relative)
      {
	if (newname[l] == '/')
	  strcpy(s,newname + l + 1);
	else
	  strcpy(s,newname+l);
      }
    else
      strcpy(s,newname);
  }

  chdir(wd);

  if (strlen(s) == 0)
    strcpy(s,"./");

  Debug(2,"reduced to %s\n",s);
  return(True);
#endif
}


/****************************************************************************
  see if a name matches a mask. The mask takes the form of several characters,
  with ? being a wild card.
****************************************************************************/
BOOL mask_match(char *name,char *mask,BOOL dodots)
{
  char *p1,*p2;
  char nbeg[MAXPATHLEN]=""; /* beginning of name */
  char next[MAXPATHLEN]=""; /* extension of name */
  char mext[MAXPATHLEN]=""; /* extension of mask */
  char mbeg[MAXPATHLEN]=""; /* beg of mask */  
    
  Debug(2,"mmatch [%s] [%s] %d\n",name,mask,dodots);

  strcpy(mbeg,mask);
  if ((p1 = strchr(mbeg,'.')) != NULL)
    {*p1 = 0;p1++;strcpy(mext,p1);}
  else
    {
      strcpy(mext,"");
      if (strlen(mbeg) > 8)
	{
	  strcpy(mext,mbeg + 8);
	  mbeg[8] = 0;
	}
    }

  if (*mbeg == 0)
    strcpy(mbeg,"????????");
  if (*mext == 0)
    strcpy(mext,"???");

  /* expand *'s */
  while ((p1 = strchr(mbeg,'*')) != NULL)
    {
      int lfill = 9 - strlen(mbeg);
      int l1= (p1 - mbeg);
      int i;
      char tmp[MAXPATHLEN]="";
      strcpy(tmp,mbeg);      
      for (i=0;i<lfill;i++)
	tmp[l1 + i] = '?';
      strcpy(tmp + l1 + lfill,mbeg + l1 + 1);	
      strcpy(mbeg,tmp);      
    }
  while ((p1 = strchr(mext,'*')) != NULL)
    {
      int lfill = 4 - strlen(mext);
      int l1= (p1 - mext);
      int i;
      char tmp[MAXPATHLEN]="";
      strcpy(tmp,mext);      
      for (i=0;i<lfill;i++)
	tmp[l1 + i] = '?';
      strcpy(tmp + l1 + lfill,mext + l1 + 1);	
      strcpy(mext,tmp);
    }
  
  strcpy(nbeg,name);
  if ((p1 = strchr(nbeg,'.')) != NULL)
    {*p1 = 0;p1++;strcpy(next,p1);}
  else
    strcpy(next,"");
  
  
  /* a couple of special cases */
  if (strequal(name,".") || strequal(name,".."))
    return(dodots && strequal(mbeg,"????????") && strequal(mext,"???"));
  
  if (strlen(nbeg) == 0) return(False);
  if (strlen(mbeg) == 0) return(False);
  if (strlen(nbeg) > 8) return(False);
  if (strlen(next) > 3) return(False);
  if (strlen(mbeg) > 8) return(False);
  if (strlen(mext) > 3) return(False);
  if (strlen(nbeg) > strlen(mbeg)) return(False);
  if (strlen(next) > strlen(mext)) return(False);
  
  /* only accept lowercase names */
  p1 = name;
  while (*p1) 
    if (isupper(*p1++)) return(False);

  Debug(5,"Matching [%8.8s.%3.3s] to [%8.8s.%3.3s]\n",nbeg,next,mbeg,mext);
  
  p1 = nbeg;
  p2 = mbeg;
  while (*p2)
    {
      if ((*p2 != '?') && (*p1 != *p2)) 
	return(False);
      p2++;
      if (*p1) p1++;
    }
  
  p1 = next;
  p2 = mext;
  while (*p2)
    {
      if ((*p2 != '?') && (*p1 != *p2)) 
	return(False);
      p2++;
      if (*p1) p1++;
    }
  return(True);
}


/****************************************************************************
  make a dir struct
****************************************************************************/
void make_dir_struct(char *buf,char *mask,char *fname,unsigned int size,int mode,unsigned int num,time_t date)
{  
  uint32 dos_date;
  char *p;
  char mask2[MAXPATHLEN]="";

  strcpy(mask2,mask);

  if ((mode & aDIR) != 0)
    size = 0;

  if ((p = strchr(mask2,'.')) != NULL)
    {
      *p = 0;
      strcpy(buf+1,mask2);
      memset(buf+1+strlen(mask2),' ',8-strlen(mask2));
      strncpy(buf+9,p+1,3);
      *p = '.';
    }
  else
    {
      memset(buf+1,' ',11);
      memcpy(buf+1,mask2,MIN(strlen(mask2),11));
    }

  memset(buf+21,0,DIR_STRUCT_SIZE-21);
  dos_date = make_dos_date(date);
  CVAL(buf,21) = mode;
  memcpy(buf+22,&dos_date,sizeof(dos_date));
  SSVAL(buf,26,size & 0xFFFF);
  SSVAL(buf,28,size >> 16);
  strncpy(buf+30,fname,12);
  buf[30+12] = 0;
}


/****************************************************************************
log a packet to pcsaout
****************************************************************************/
void log_out(char *buffer,int len)
{
  if (pcsaout)
    {
      fprintf(pcsaout,"\n%s Transaction %d\n",timestring(),trans_num++);
      fwrite(buffer,len,1,pcsaout);
      fflush(pcsaout);
    }      
}

/****************************************************************************
log a packet to pcsain
****************************************************************************/
void log_in(char *buffer,int len)
{
  if (pcsain)
    {
      fprintf(pcsain,"\n%s Transaction %d\n",timestring(),trans_num++);
      fwrite(buffer,len,1,pcsain);
      fflush(pcsain);
    }      
}

/****************************************************************************
write to a socket
****************************************************************************/
int write_socket(int fd,char *buf,int len)
{
if (passive)
  return(len);
return(write(fd,buf,len));
}

/****************************************************************************
read from a socket
****************************************************************************/
int read_socket(int fd,char *buf,int len)
{
#define NORECVFROM
#ifdef NORECVFROM
return(read(fd,buf,len));
#else
int ret;
struct sockaddr sock;
memset(&sock, 0, sizeof(sock));
ret = recvfrom(fd,buf,len,0,&sock,sizeof(sock));
if (ret <= 0)
  Debug(0,"read socket failed. ERRNO=%d\n",errno);
if (ret > 0)
  {
#if 0
    memcpy(lastip,sock.sa_data,4);
#endif
    Debug(1,"received %d bytes from (%d.%d.%d.%d)\n",
	  lastip[0],lastip[1],lastip[2],lastip[3]);
  }
return(ret);
#endif
}



/****************************************************************************
read data from the client. Maxtime is in 10ths of a sec
****************************************************************************/
int read_max_data(int fd,char *buffer,int bufsize,int maxtime)
{
  fd_set fds;
  int selrtn;
  int  nready;
  int nread;
  struct timeval timeout;
 
  FD_ZERO(&fds);
  FD_SET(fd,&fds);

  timeout.tv_sec = maxtime / 10;
  timeout.tv_usec = (maxtime % 10) * 100000;

  do {    
    if (maxtime > 0)
      selrtn = select(255,&fds,NULL,NULL,&timeout);
    else
      selrtn = select(255,&fds,NULL,NULL,NULL);
  } 
  while( selrtn < 0  &&  errno == EINTR );


  if (!FD_ISSET(fd,&fds))
    return 0;

  /* Query the system how many bytes are ready to be read */
  ioctl(fd, FIONREAD, &nready);

  /* Only try to get the smaller of nready and BUFSIZE */
  nread = read_socket(fd, buffer, nready < bufsize ? nready : bufsize);

  /* return the number got */
  return(nread);
}



/****************************************************************************
  read data from the client, reading exactly N bytes
****************************************************************************/
BOOL read_data(int fd,char *buffer,int N)
{
  int  nready;
  int nread = 0;
  
  while (nread < N)
    {
      nready = read_socket(fd,buffer + nread,N - nread);
      if (nready <= 0)
	return False;
      nread += nready;
    }
  return True;
}


/****************************************************************************
  read an smb from a fd and return it's length
****************************************************************************/
int receive_smb(char *buffer)
{
  int len;
  int fd = Client;
  memset(buffer,0,smb_size + 100);
  if (!read_data(fd,buffer,4))
    {
      Debug(0,"couldn't read from client\n");
      exit(1);
    }

  len = smb_len(buffer);

  if (!read_data(fd,buffer + 4,len))
    {
      Debug(0,"couldn't read %d bytes from client. Exiting\n",len);
      exit(1);
    }

  log_in(buffer,len);
  return(len + 4);
}

/****************************************************************************
  send an smb to a fd 
****************************************************************************/
BOOL send_smb(char *buffer)
{
  int fd = Client;
  int len;
  int ret,nwritten=0;
  len = smb_len(buffer) + 4;

  while (nwritten < len)
    {
      ret = write_socket(fd,buffer+nwritten,len - nwritten);
      if (ret <= 0)
	{
	  Debug(0,"Error writing %d bytes to client. %d. Exiting\n",len,ret);
	  exit(1);
	}
      nwritten += ret;
    }

  log_out(buffer,len);

  return True;
}


/****************************************************************************
word out the length of a nmb message
****************************************************************************/
int nmb_len(char *buf)
{
int i;
int ret = 12;
char *p = buf;
int qdcount = SVAL(buf,4);
int ancount = SVAL(buf,6);
int nscount = SVAL(buf,8);
int arcount = SVAL(buf,10);

for (i=0;i<qdcount;i++)
  {
    p = buf + ret;
    ret += name_len(p) + 4;
  }

for (i=0;i<(ancount + nscount + arcount);i++)
  {
    int rdlength;
    p = buf + ret;
    ret += name_len(p) + 8;
    p = buf + ret;
    rdlength = SVAL(p,0);
    ret += rdlength + 2;
  }

return(ret);
}


/****************************************************************************
receive a name message
****************************************************************************/
BOOL receive_nmb(char *buffer,int timeout)
{
  int ret = read_max_data(Client,buffer,BUFFER_SIZE,timeout);

  if (ret < 0)
    {
      Debug(0,"No bytes from client\n");
      exit(0);
    }

  if (ret == 0)
    return False;

  log_in(buffer,ret);

  Debug(1,"nmb_len=%d len=%d\n",nmb_len(buffer),ret);

  return(True);
}


/****************************************************************************
send a name message
****************************************************************************/
BOOL send_nmb(int fd,char *buffer)
{
  int len;
  int ret,nwritten=0;
  len = nmb_len(buffer);

  while (nwritten < len)
    {
      ret = write_socket(fd,buffer+nwritten,len - nwritten);
      if (ret <= 0)
	{
	  Debug(0,"Error writing %d bytes to client. %d. Exiting\n",len,ret);
	  return(False);
	}
      nwritten += ret;
    }

  log_out(buffer,len);

  return(True);
}



/****************************************************************************
find a pointer to a netbios name
****************************************************************************/
char *name_ptr(char *buf,int ofs)
{
  unsigned char c = *(unsigned char *)(buf+ofs);

  if ((c & 0xC0) == 0xC0)
    {
      uint16 l;
      char *p = (char *)&l;
      memcpy(&l,buf+ofs,2);
      p[0] &= ~0xC0;
      l = SVAL(p,0);
      Debug(5,"name ptr to pos %d from %d is %s\n",l,ofs,buf+l);
      return(buf + l);
    }
  else
    return(buf+ofs);
}  

/****************************************************************************
extract a netbios name from a buf
****************************************************************************/
void name_extract(char *buf,int ofs,char *name)
{
  strcpy(name,"");
  name_interpret(name_ptr(buf,ofs),name);
}  
  


/****************************************************************************
show a nmb message
****************************************************************************/
void show_nmb(char *inbuf)
{
  int i,l;
  int name_trn_id = SVAL(inbuf,0);
  int opcode = (CVAL(inbuf,2) >> 3) & 0xF;
  int nm_flags = ((CVAL(inbuf,2) & 0x7) << 4) + (CVAL(inbuf,3)>>4);
  int rcode = CVAL(inbuf,3) & 0xF;
  int qdcount = SVAL(inbuf,4);
  int ancount = SVAL(inbuf,6);
  int nscount = SVAL(inbuf,8);
  int arcount = SVAL(inbuf,10);
  char name[100];

  Debug(1,"\nPACKET INTERPRETATION\n");

  if (opcode == 5 && ((nm_flags & ~1) == 0x10) && rcode == 0)
    Debug(1,"NAME REGISTRATION REQUEST (%s)\n",nm_flags&1?"Broadcast":"Unicast");

  if (opcode == 5 && ((nm_flags & ~1) == 0x00) && rcode == 0)
    Debug(1,"NAME OVERWRITE REQUEST AND DEMAND (%s)\n",nm_flags&1?"Broadcast":"Unicast");
  
  if (opcode == 9 && ((nm_flags & ~1) == 0x00) && rcode == 0)
    Debug(1,"NAME REFRESH REQUEST (%s)\n",nm_flags&1?"Broadcast":"Unicast");
  
  if (opcode == 5 && nm_flags == 0x58 && rcode == 0)
    Debug(1,"POSITIVE NAME REGISTRATION RESPONSE\n");
  
  if (opcode == 5 && nm_flags == 0x58 && rcode != 0 && rcode != 7)
    Debug(1,"NEGATIVE NAME REGISTRATION RESPONSE\n");
  
  if (opcode == 5 && nm_flags == 0x50 && rcode == 0)
    Debug(1,"END-NODE CHALLENGE REGISTRATION RESPONSE\n");
  
  if (opcode == 5 && nm_flags == 0x58 && rcode != 0 && rcode == 7)
    Debug(1,"NAME CONFLICT DEMAND\n");
  
  if (opcode == 6 && (nm_flags&~1) == 0x00 && rcode == 0)
    Debug(1,"NAME RELEASE REQUEST & DEMAND (%s)\n",nm_flags&1?"Broadcast":"Unicast");
  
  if (opcode == 6 && (nm_flags&~1) == 0x40 && rcode == 0)
    Debug(1,"POSITIVE NAME RELEASE RESPONSE\n");
  
  if (opcode == 6 && (nm_flags&~1) == 0x40 && rcode != 0)
    Debug(1,"NEGATIVE NAME RELEASE RESPONSE\n");
  
  if (opcode == 0 && (nm_flags&~1) == 0x10 && rcode == 0)
    Debug(1,"NAME QUERY REQUEST (%s)\n",nm_flags&1?"Broadcast":"Unicast");
  
  if (opcode == 0 && (nm_flags&~0x28) == 0x50 && rcode == 0)
    Debug(1,"POSITIVE NAME QUERY RESPONSE\n");
  
  if (opcode == 0 && (nm_flags&~0x08) == 0x50 && rcode != 0)
    Debug(1,"NEGATIVE NAME QUERY RESPONSE\n");
  
  if (opcode == 0 && nm_flags == 0x10 && rcode == 0)
    Debug(1,"REDIRECT NAME QUERY RESPONSE\n");

  if (opcode == 7 && nm_flags == 0x80 && rcode == 0)
    Debug(1,"WAIT FOR ACKNOWLEDGEMENT RESPONSE\n");
  
  if (opcode == 0 && (nm_flags&~1) == 0x00 && rcode == 0)
    Debug(1,"NODE STATUS REQUEST (%s)\n",nm_flags&1?"Broadcast":"Unicast");

  if (opcode == 0 && nm_flags == 0x80 && rcode == 0)
    Debug(1,"NODE STATUS RESPONSE\n");
  
  
  Debug(1,"name_trn_id=0x%x\nopcode=0x%x\nnm_flags=0x%x\nrcode=0x%x\n",
	name_trn_id,opcode,nm_flags,rcode);
  Debug(1,"qdcount=%d\nancount=%d\nnscount=%d\narcount=%d\n",
	qdcount,ancount,nscount,arcount);

  l = 12;
  for (i=0;i<qdcount;i++)
    {
      int type,class;
      Debug(1,"QUESTION %d\n",i);
      name_extract(inbuf,l,name);
      l += name_len(inbuf+l);
      type = SVAL(inbuf+l,0);
      class = SVAL(inbuf+l,2);
      l += 4;
      Debug(1,"\t%s\n\ttype=0x%x\n\tclass=0x%x\n",name,type,class);
    }

  for (i=0;i<(ancount + nscount + arcount);i++)
    {
      int type,class,ttl,rdlength;
      Debug(1,"RESOURCE %d\n",i);
      name_extract(inbuf,l,name);
      l += name_len(inbuf + l);
      type = SVAL(inbuf+l,0);
      class = SVAL(inbuf+l,2);
      ttl = IVAL(inbuf+l,4);
      rdlength = SVAL(inbuf+l,8);
      l += 10 + rdlength;
      Debug(1,"\t%s\n\ttype=0x%x\n\tclass=0x%x\n",name,type,class);
      Debug(1,"\tttl=%d\n\trdlength=%d\n",ttl,rdlength);
    }

  Debug(1,"\n");
  
}

/****************************************************************************
return the total storage length of a mangled name
****************************************************************************/
int name_len(char *s)
{
  unsigned char c = *(unsigned char *)s;
  if ((c & 0xC0) == 0xC0)
    return(2);
  return(strlen(s) + 1);
}

/****************************************************************************
  close the socket communication
****************************************************************************/
void close_sockets(void )
{
  extern int Client;
  Client = 0;
}



/****************************************************************************
send a single packet to a port on another machine
****************************************************************************/
BOOL send_packet(char *buf,int len,unsigned char *ip,int port,int type)
{
  BOOL ret;
  int out_fd;
  struct sockaddr_in sock_out;
  int one=1;

  log_out(buf,len);

  if (passive)
    return(True);

  /* create a socket to write to */
  out_fd = socket(AF_INET, type, 0);
  if (out_fd == -1) 
    {
      Debug(0,"socket failed");
      return False;
    }

  /* allow broadcasts on it */
  setsockopt(out_fd,SOL_SOCKET,SO_BROADCAST,&one,sizeof(one));
		  
  /* set the address and port */
  memset(&sock_out, 0, sizeof(sock_out));
  memcpy(&sock_out.sin_addr, ip, 4);
  sock_out.sin_port = htons( port );
  sock_out.sin_family = AF_INET;
  
  /* log the packet */
  log_out(buf,len);


  Debug(1,"sending a packet of len %d to (%d.%d.%d.%d) on port %d of type %s\n",
	len,ip[0],ip[1],ip[2],ip[3],port,type==SOCK_DGRAM?"DGRAM":"STREAM");
	
  /* send it */
  ret = (sendto(out_fd,buf,len,0,(struct sockaddr *)&sock_out,sizeof(sock_out)) >= 0);

  if (!ret)
    Debug(1,"Send packet failed. ERRNO=%d\n",errno);

  close(out_fd);
  return(ret);
}
