/* 
   Unix SMB/Netbios implementation.
   Version 1.5.
   Copyright (C) Andrew Tridgell 1992,1993
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

char cur_dir[1000] = "\\";
char service[255];
char desthost[100];
char myname[100];

char debugf[255] = DEBUGFILE;

char *InBuffer = NULL;
char *OutBuffer = NULL;
int cnum = 0;
int pid = 0;
int uid = 0;
int mid = 0;

extern BOOL NeedSwap;

/****************************************************************************
change directory
****************************************************************************/
void cmd_cd(char *inbuf,char *outbuf )
{
  char *p;
  p = strtok(NULL," \t\n\r");
  if (p)
    {
      if (*p == '\\')
	strcpy(cur_dir,p);
      else
	strcat(cur_dir,p);
      strcat(cur_dir,"\\");
    }
  else
    printf("Current directory is %s\n",cur_dir);
}

/****************************************************************************
get a directory listing
****************************************************************************/
void cmd_dir(char *inbuf,char *outbuf )
{
  int attribute = aDIR | aSYSTEM | aHIDDEN;
  char mask[100];
  char *p;
  int received = 0;
  BOOL first = True;
  char status[21];

  strcpy(mask,cur_dir);
  strcat(mask,"\\");

  p = strtok(NULL," \t\n\r");
  if (p)
    {
      if (*p == '\\')
	strcpy(mask,p);
      else
	strcat(mask,p);
    }
  else
    strcat(mask,"*.*");

  while (1)
    {
      memset(outbuf,0,smb_size);
      set_message(outbuf,2,5 + strlen(mask) + (first?0:21));
      CVAL(outbuf,smb_com) = SMBsearch;
      SSVAL(outbuf,smb_tid,cnum);
      SSVAL(outbuf,smb_pid,pid);
      SSVAL(outbuf,smb_uid,uid);
      SSVAL(outbuf,smb_mid,mid);
      SSVAL(outbuf,smb_vwv0,(BUFFER_SIZE - 100)/DIR_STRUCT_SIZE);
      SSVAL(outbuf,smb_vwv1,attribute);
  
      p = smb_buf(outbuf);
      *p++ = 4;
      
      strcpy(p,mask);
      p += strlen(p) + 1;
      
      *p++ = 5;
      if (first)
	SSVAL(p,0,0);
      else
	{
	  SSVAL(p,0,21);
	  p += 2;
	  memcpy(p,status,21);
	}

      send_smb(outbuf);
      receive_smb(inbuf);
      first = False;

#if 0
      if (CVAL(inbuf,smb_rcls) != 0)
	break;
#endif

      received = SVAL(inbuf,smb_vwv0);
      if (received == 0)
	break;

      p = smb_buf(inbuf) + 3;

      while (received--)
	{
	  uint32 size;
	  uint32 Date;
	  time_t Time;
	  int attr;
	  char *name;
	  char attrstr[10]="";

	  memcpy(status,p,21);

	  attr = CVAL(p,21);
	  memcpy(&Date,p+22,4);
	  Time = make_unix_date(Date);
	  size = SVAL(p,26) + (SVAL(p,28)<<16);
	  name = p+30;

	  if (attr & aDIR) strcat(attrstr,"D");
	  if (attr & aARCH) strcat(attrstr,"A");
	  if (attr & aHIDDEN) strcat(attrstr,"H");
	  if (attr & aSYSTEM) strcat(attrstr,"S");
	  if (attr & aRONLY) strcat(attrstr,"R");

	  printf("%20.20s%7.7s%7d  %s",name,attrstr,size,asctime(localtime(&Time)));
	  p += DIR_STRUCT_SIZE;
	}
    }
}


/****************************************************************************
get a file
****************************************************************************/
void cmd_get(char *inbuf,char *outbuf )
{
  int handle,fnum;
  uint32 size,nread=0;
  char lname[100];
  char rname[100];
  char *p;

  strcpy(rname,cur_dir);
  strcat(rname,"\\");

  p = strtok(NULL," \t\n\r");
  if (!p)
    {
      printf("get <filename>\n");
      return;
    }
  strcat(rname,p); 
  strcpy(lname,p);

  p = strtok(NULL," \t\n\r");
  if (p)
    strcpy(lname,p);      

  memset(outbuf,0,smb_size);
  set_message(outbuf,2,2 + strlen(rname));

  CVAL(outbuf,smb_com) = SMBopen;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  SSVAL(outbuf,smb_vwv0,0);
  SSVAL(outbuf,smb_vwv1,0);
  
  p = smb_buf(outbuf);
  *p++ = 4;      
  strcpy(p,rname);


  handle = creat(lname,0644);
  if (handle < 0)
    {
      perror(lname);
      printf("Error opening local file %s\n",lname);
      return;
    }

  send_smb(outbuf);
  receive_smb(inbuf);

  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d opening remote file %s\n",SVAL(inbuf,smb_err),rname);
      close(handle);
      return;
    }

  fnum = SVAL(inbuf,smb_vwv0);
  size = SVAL(inbuf,smb_vwv4) + (SVAL(inbuf,smb_vwv5)<<16);

  printf("getting file %s of size %d bytes as %s\n",rname,size,lname);

  while (nread < size)
    {
      memset(outbuf,0,smb_size);
      set_message(outbuf,5,0);
      CVAL(outbuf,smb_com) = SMBread;
      SSVAL(outbuf,smb_tid,cnum);
      SSVAL(outbuf,smb_pid,pid);
      SSVAL(outbuf,smb_uid,uid);
      SSVAL(outbuf,smb_mid,mid);
      SSVAL(outbuf,smb_vwv0,fnum);
      SSVAL(outbuf,smb_vwv1,MIN(1024*((BUFFER_SIZE-1024)/1024),size - nread));
      SSVAL(outbuf,smb_vwv2,nread & 0xFFFF);
      SSVAL(outbuf,smb_vwv3,nread >> 16);
      SSVAL(outbuf,smb_vwv4,size - nread);

      send_smb(outbuf);
      receive_smb(inbuf);

      if (CVAL(inbuf,smb_rcls) != 0)
	{
	  printf("Error %d reading remote file\n",SVAL(inbuf,smb_err));
	  break;
	}

      if (write(handle,smb_buf(inbuf)+3,SVAL(inbuf,smb_vwv0)) !=
	  SVAL(inbuf,smb_vwv0))
	{
	  perror(lname);
	  printf("Error writing local file\n");
	  break;
	}
	  
      nread += SVAL(inbuf,smb_vwv0);
    }

  memset(outbuf,0,smb_size);
  set_message(outbuf,3,0);
  CVAL(outbuf,smb_com) = SMBclose;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  SSVAL(outbuf,smb_vwv0,fnum);
  SSVAL(outbuf,smb_vwv1,0);
  SSVAL(outbuf,smb_vwv2,0);

  send_smb(outbuf);
  receive_smb(inbuf);
  
  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d closing remote file\n",SVAL(inbuf,smb_err));
      close(handle);
      return;
    }

  close(handle);
}


/****************************************************************************
put a file
****************************************************************************/
void cmd_put(char *inbuf,char *outbuf )
{
  int handle,fnum;
  uint32 size,nread=0;
  char lname[100];
  char rname[100];
  char *p;
  
  strcpy(rname,cur_dir);
  strcat(rname,"\\");
  
  p = strtok(NULL," \t\n\r");
  if (!p)
    {
      printf("put <filename>\n");
      return;
    }
  strcpy(lname,p);
  
  p = strtok(NULL," \t\n\r");
  if (p)
    strcat(rname,p);      
  else
    strcat(rname,lname);

  memset(outbuf,0,smb_size);
  set_message(outbuf,3,2 + strlen(rname));
  
  CVAL(outbuf,smb_com) = SMBcreate;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  SSVAL(outbuf,smb_vwv0,0);
  SSVAL(outbuf,smb_vwv1,0);
  SSVAL(outbuf,smb_vwv2,0);
  
  p = smb_buf(outbuf);
  *p++ = 4;      
  strcpy(p,rname);
  
  send_smb(outbuf);
  receive_smb(inbuf);
  
  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d opening remote file %s\n",SVAL(inbuf,smb_err),rname);
      close(handle);
      return;
    }
  
  handle = open(lname,O_RDONLY);
  if (handle < 0)
    {
      perror(lname);
      printf("Error opening local file %s\n",lname);
      return;
    }

  
  fnum = SVAL(inbuf,smb_vwv0);
  size = file_size(lname);
  
  printf("putting file %s of size %d bytes as %s\n",lname,size,rname);
  
  while (nread < size)
    {
      int n = MIN(1024*((BUFFER_SIZE-1024)/1024),size - nread);
  
      memset(outbuf,0,smb_size);
      set_message(outbuf,5,n + 3);
      CVAL(outbuf,smb_com) = SMBwrite;
      SSVAL(outbuf,smb_tid,cnum);
      SSVAL(outbuf,smb_pid,pid);
      SSVAL(outbuf,smb_uid,uid);
      SSVAL(outbuf,smb_mid,mid);
      SSVAL(outbuf,smb_vwv0,fnum);
      SSVAL(outbuf,smb_vwv1,n);
      SSVAL(outbuf,smb_vwv2,nread & 0xFFFF);
      SSVAL(outbuf,smb_vwv3,nread >> 16);
      SSVAL(outbuf,smb_vwv4,size - nread);
      CVAL(smb_buf(outbuf),0) = 1;
      SSVAL(smb_buf(outbuf),1,n);

      if (read(handle,smb_buf(outbuf)+3,n) != n)
	{
	  perror(lname);
	  printf("Error reading local file\n");
	  break;
	}	  

      send_smb(outbuf);
      receive_smb(inbuf);

      if (CVAL(inbuf,smb_rcls) != 0)
	{
	  printf("Error %d writing remote file\n",SVAL(inbuf,smb_err));
	  break;
	}

      
      if (n != SVAL(inbuf,smb_vwv0))
	{
	  printf("Error: only wrote %d bytes\n",nread + SVAL(inbuf,smb_vwv0));
	  break;
	}

      nread += n;
    }

  memset(outbuf,0,smb_size);
  set_message(outbuf,3,0);
  CVAL(outbuf,smb_com) = SMBclose;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  SSVAL(outbuf,smb_vwv0,fnum);
  SSVAL(outbuf,smb_vwv1,0);
  SSVAL(outbuf,smb_vwv2,0);

  send_smb(outbuf);
  receive_smb(inbuf);
  
  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d closing remote file\n",SVAL(inbuf,smb_err));
      close(handle);
      return;
    }

  close(handle);
}


/****************************************************************************
delete some files
****************************************************************************/
void cmd_del(char *inbuf,char *outbuf )
{
  char mask[100];
  char *p;
  
  strcpy(mask,cur_dir);
  strcat(mask,"\\");
  
  p = strtok(NULL," \t\n\r");
  if (!p)
    {
      printf("del <filename>\n");
      return;
    }
  strcat(mask,p);


  memset(outbuf,0,smb_size);
  set_message(outbuf,1,2 + strlen(mask));
  
  CVAL(outbuf,smb_com) = SMBunlink;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  SSVAL(outbuf,smb_vwv0,0);
  
  p = smb_buf(outbuf);
  *p++ = 4;      
  strcpy(p,mask);
  
  send_smb(outbuf);
  receive_smb(inbuf);
  
  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d deleting remote file %s\n",SVAL(inbuf,smb_err),mask);
      return;
    }
  
}


/****************************************************************************
make a directory
****************************************************************************/
void cmd_mkdir(char *inbuf,char *outbuf )
{
  char mask[100];
  char *p;
  
  strcpy(mask,cur_dir);
  strcat(mask,"\\");
  
  p = strtok(NULL," \t\n\r");
  if (!p)
    {
      printf("mkdir <dirname>\n");
      return;
    }
  strcat(mask,p);


  memset(outbuf,0,smb_size);
  set_message(outbuf,0,2 + strlen(mask));
  
  CVAL(outbuf,smb_com) = SMBmkdir;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  
  p = smb_buf(outbuf);
  *p++ = 4;      
  strcpy(p,mask);
  
  send_smb(outbuf);
  receive_smb(inbuf);
  
  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d making remote directory file %s\n",SVAL(inbuf,smb_err),mask);
      return;
    }
  
}

/****************************************************************************
remove a directory
****************************************************************************/
void cmd_rmdir(char *inbuf,char *outbuf )
{
  char mask[100];
  char *p;
  
  strcpy(mask,cur_dir);
  strcat(mask,"\\");
  
  p = strtok(NULL," \t\n\r");
  if (!p)
    {
      printf("rmdir <dirname>\n");
      return;
    }
  strcat(mask,p);


  memset(outbuf,0,smb_size);
  set_message(outbuf,0,2 + strlen(mask));
  
  CVAL(outbuf,smb_com) = SMBrmdir;
  SSVAL(outbuf,smb_tid,cnum);
  SSVAL(outbuf,smb_pid,pid);
  SSVAL(outbuf,smb_uid,uid);
  SSVAL(outbuf,smb_mid,mid);
  
  p = smb_buf(outbuf);
  *p++ = 4;      
  strcpy(p,mask);
  
  send_smb(outbuf);
  receive_smb(inbuf);
  
  if (CVAL(inbuf,smb_rcls) != 0)
    {
      printf("Error %d removing remote directory file %s\n",SVAL(inbuf,smb_err),mask);
      return;
    }
  
}


/****************************************************************************
send a login command
****************************************************************************/
BOOL send_login(char *inbuf,char *outbuf )
{
  char *prot = "PC NETWORK PROGRAM 1.0";
  char *pass = NULL;  
  char *dev = "A:";
  char *p;
  int len = 4;

  /* send a session request (RFC 8002) */
  CVAL(outbuf,0) = 0x81;

  /* put in the destination name */
  p = outbuf+len;
  name_mangle(desthost,p);
  len += name_len(p);

  /* and my name */
  p = outbuf+len;
  name_mangle(myname,p);
  len += name_len(p);

  /* setup the packet length */
  smb_setlen(outbuf,len);

  send_smb(outbuf);
  receive_smb(inbuf);

  if (CVAL(inbuf,0) != 0x82)
    {
      Debug(0,"Session request failed\n");
      return(False);
    }      


  memset(outbuf,0,smb_size);
  set_message(outbuf,0,2 + strlen(prot));

  CVAL(outbuf,smb_com) = SMBnegprot;
  CVAL(smb_buf(outbuf),0) = 2;

  strcpy(smb_buf(outbuf)+1,prot);

  send_smb(outbuf);
  receive_smb(inbuf);

  if (CVAL(inbuf,smb_rcls) != 0)
    {
      Debug(0,"SMBnegprot failed error class 0x%x code 0x%x\n",
	    CVAL(inbuf,smb_rcls),
	    SVAL(inbuf,smb_err));
      return(False);
    }

  pass = getpass("Password: ");

  /* now we've got a connection - send a tcon message */
  memset(outbuf,0,smb_size);
  set_message(outbuf,0,6 + strlen(service) + strlen(pass) + strlen(dev));
  CVAL(outbuf,smb_com) = SMBtcon;
  p = smb_buf(outbuf);
  *p++ = 4;
  strcpy(p,service);
  p += strlen(p) + 1;
  *p++ = 4;
  strcpy(p,pass);
  p += strlen(p) + 1;
  *p++ = 4;
  strcpy(p,dev);

  send_smb(outbuf);
  receive_smb(inbuf);

  if (CVAL(inbuf,smb_rcls) != 0)
    {
      Debug(0,"SMBtcon failed error class 0x%x code 0x%x\n",
	    CVAL(inbuf,smb_rcls),
	    SVAL(inbuf,smb_err));
      return(False);
    }
  

  cnum = SVAL(inbuf,smb_vwv1);

  Debug(1,"Connected with cnum=%d\n",cnum);

  return True;

}


/****************************************************************************
send a logout command
****************************************************************************/
void send_logout(char *inbuf,char *outbuf )
{
  set_message(outbuf,0,0);

  CVAL(outbuf,smb_com) = SMBtdis;
  CVAL(outbuf,smb_tid) = cnum;
  send_smb(outbuf);
  receive_smb(inbuf);

  if (CVAL(inbuf,smb_rcls) != 0)
    {
      Debug(0,"SMBtdis failed error class 0x%x code 0x%x\n",
	    CVAL(inbuf,smb_rcls),
	    SVAL(inbuf,smb_err));
    }
  exit(0);
}


void cmd_help();

/* This defines the commands supported by this client */
struct
{
  char name[20];
  void (*fn)();
} commands[] = 
{
  {"ls",cmd_dir},
  {"dir",cmd_dir},
  {"cd",cmd_cd},
  {"get",cmd_get},
  {"put",cmd_put},
  {"del",cmd_del},
  {"mkdir",cmd_mkdir},
  {"rmdir",cmd_rmdir},
  {"quit",send_logout},
  {"help",cmd_help},
  {"?",cmd_help},
  {"",NULL}
};

/****************************************************************************
help
****************************************************************************/
void cmd_help(char *inbuf,char *outbuf )
{
  int i=0;
  while (commands[i].fn)
    {
      printf("%s\n",commands[i].name);
      i++;
    }
}

/****************************************************************************
open the client sockets
****************************************************************************/
BOOL open_sockets(int port )
{
  struct sockaddr_in sock_out;
  char *host;
  char service2[255];
  struct hostent *hp;
  extern int Client;

  strcpy(service2,service);
  host = strtok(service2,"\\/");
  strcpy(desthost,host);
  gethostname(myname,100);

  if ((hp = gethostbyname(host)) == 0) 
    {
      Debug(0,"Gethostbyname: Unknown host %s.\n",host);
      return False;
    }
  
  /* create a socket to write to */
  Client = socket(hp->h_addrtype, SOCK_STREAM, 0);
  if (Client == -1) 
    {
      perror("socket");
      return False;
    }
  
  memset(&sock_out, 0, sizeof(sock_out));
  memcpy(&sock_out.sin_addr, hp->h_addr, hp->h_length);
  
  sock_out.sin_port = htons( port );
  sock_out.sin_family = hp->h_addrtype;
  
  /* and connect it to the destination */
  if (connect(Client,(struct sockaddr *)&sock_out,sizeof(sock_out))<0)
    {
      perror("connect");
      close(Client);
      return False;
    }

  return True;
}


/****************************************************************************
  process commands from the client
****************************************************************************/
void process(void )
{
  static int trans_num = 0;
  int nread;
  char line[1000];

  InBuffer = (char *)malloc(BUFFER_SIZE);
  OutBuffer = (char *)malloc(BUFFER_SIZE);
  if ((InBuffer == NULL) || (OutBuffer == NULL)) 
    return;
  
  memset(OutBuffer,0,smb_size);
  if (!send_login(InBuffer,OutBuffer))
    return;

  while (!feof(stdin))
    {
      char *tok;
      int i;
      BOOL found = False;

      memset(OutBuffer,0,smb_size);

      /* display a prompt */
      printf("smb> ");

      /* and get a response */
      if (!fgets(line,1000,stdin))
	break;

      /* and get the first part of the command */
      tok = strtok(line," \t\n\r");
      
      i = 0;
      while (commands[i].fn != NULL)
	{
	  if (strequal(commands[i].name,tok))
	    {
	      found = True;
	      commands[i].fn(InBuffer,OutBuffer);
	    }
	  i++;
	}
      if (!found && tok)
	printf("%s: command not found\n",tok);
    }
  
  memset(OutBuffer,0,smb_size);
  send_logout(InBuffer,OutBuffer);
}


/****************************************************************************
usage on the program
****************************************************************************/
void usage(char *pname)
{
  printf("Usage: %s service [-p port] [-d debuglevel] [-l log]\n",pname);
  printf("\t-p port               listen on the specified port\n");
  printf("\t-d debuglevel         set the debuglevel\n");
  printf("\t-l log basename.      Basename for log/debug files\n");
  printf("\n");
}


/****************************************************************************
  main program
****************************************************************************/
int main(int argc,char *argv[])
{
  int port = 139;
  int opt;
  extern FILE *dbf;
  extern int DEBUGLEVEL;
  extern char *optarg;

  pid = getpid();
  uid = getuid();

  if (*argv[1] == '-')
    strcpy(service,"");  
  else
    {
      strcpy(service,argv[1]);  
      argc--;
      argv++;
    }

  while ((opt = getopt (argc, argv, "d:p:l:h")) != EOF)
    switch (opt)
      {
      case 'd':
	DEBUGLEVEL = atoi(optarg);
	break;
      case 'l':
	strcpy(debugf,optarg);
	break;
      case 'p':
	port = atoi(optarg);
	break;
      case 'h':
	usage(argv[0]);
	exit(0);
	break;
      default:
	usage(argv[0]);
	exit(1);
      }

  
  NeedSwap = big_endian();
  
  umask((~CREATE_MODE) & 0777);

  if (DEBUGLEVEL > 0)
    {
      dbf = stdout;
      Debug(1,"%s client started\n",timestring());
    }

  if (DEBUGLEVEL > 2)
    {
      extern FILE *pcsain,*pcsaout;
      char fname[255];
      sprintf(fname,"%s.client.in",debugf);
      pcsain = fopen(fname,"w"); 
      sprintf(fname,"%s.client.out",debugf);
      pcsaout = fopen(fname,"w");
    }
  
  if (open_sockets(port))
    {
      process();
      close_sockets();
    }
  return(0);
}

