/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Module name: namestat.c

   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   09 oct 96: lkcl@pires.co.uk
   created module namestat containing NetBIOS node status reply

*/

#include "includes.h"

extern int DEBUGLEVEL;

extern struct in_addr ipgrp;


/* this is used to sort names for a name status into a sensible order
   we put our own names first, then in alphabetical order */
static int status_compare(char *n1,char *n2)
{
  int l1,l2,l3;

  /* its a bit tricky because the names are space padded */
  for (l1=0;l1<15 && n1[l1] && n1[l1] != ' ';l1++) ;
  for (l2=0;l2<15 && n2[l2] && n2[l2] != ' ';l2++) ;
  l3 = strlen(lp_server_alias());

  if ((l1==l3) && strncmp(n1,lp_server_alias(),l3) == 0 && 
      (l2!=l3 || strncmp(n2,lp_server_alias(),l3) != 0))
    return -1;

  if ((l2==l3) && strncmp(n2,lp_server_alias(),l3) == 0 && 
      (l1!=l3 || strncmp(n1,lp_server_alias(),l3) != 0))
    return 1;

  return memcmp(n1,n2,18);
}


/****************************************************************************
  reply to a name status query

  combine the list of the local interface on which the query was made with
  the names registered via wins.
  ****************************************************************************/
void reply_name_status(struct packet_struct *p)
{
  struct nmb_packet *nmb = &p->packet.nmb;
  struct nmb_name *question   = &nmb->question.question_name;

  char rdata[MAX_DGRAM_SIZE];
  char *countptr, *buf, *bufend, *buf0;
  int names_added,idx;
  struct name_record *n;
  struct subnet_record *d = NULL;

  int token;

  BOOL bcast = nmb->header.nm_flags.bcast;
  BOOL wins  = nmb->header.nm_flags.recursion_desired;
  
  if (bcast) return; /* ignore broadcast name status requests */

  DEBUG(3,("Name status for name %s %s\n",namestr(question),inet_ntoa(p->ip)));

  if (wins && !bcast)
  {
    DEBUG(1,("nmbd received packet intended for winsd!\n"));
    return;
  }

  /* NOTE: we always treat a name status lookup as for a local interface */ 
  if (!(d = find_req_subnet(p->ip, False)))
  {
    DEBUG(3,("Name status req: %s not known\n", inet_ntoa(p->ip)));
    return;
  }

  n = find_name_search(&d, &nmb->question.question_name,
				FIND_WINS|FIND_SELF|FIND_LOCAL,p->ip, &idx);
  
  if (!n) return;

  /* select the right workgroup. all other workgroup names get excluded */
  if ((token = brse_domain_to_token(question->name)) == -1 &&
      (token = brse_alias_to_token (question->name)) == -1)
  {
    if ((token = brse_domain_to_token(lp_workgroup())) == -1) return;
  }
     
  /* end of buffer minus space for one last name minus space for info */
  bufend = &rdata[MAX_DGRAM_SIZE] - 18 - 64;

  countptr = buf = rdata;
  buf += 1;
  buf0 = buf;

  names_added = 0;

  n = d->namelist;

  while (buf < bufend) 
  {
    int i;
    for (i = 0; i < n->num_ips; i++)
    {
      if (n->ip_flgs[i].source == SELF)
      {
        int name_type = n->name.name_type;
        char *name = n->name.name;
      
        /* check if we want to exclude other workgroup names
	       from the response. if we don't exclude them, windows clients
	       get confused and will respond with an error for NET VIEW */
      
        int wg_tok = brse_domain_to_token(name);
        int nm_tok = brse_alias_to_token (name);

        if (!strequal(name,"*") &&
	        !strequal(name,"__SAMBA__") &&

            (((wg_tok == -1 || wg_tok == token) &&
              (nm_tok == -1 || nm_tok == token)) ||

	         (strequal(question->name, name) &&
              name_type == question->name_type)))
        {

          /* start with first bit of putting info in buffer: the name */
          bzero(buf,18);
	      sprintf(buf,"%-15.15s",n->name.name);
          strupper(buf);
        
          /* put name type and netbios flags in buffer */
          buf[15] = name_type;
          buf[16] = n->ip_flgs[i].nb_flags;
        
          buf += 18;
      
          names_added++;
        }
      }
    }

    /* remove duplicate names */
    qsort(buf0,names_added,18,QSORT_CAST status_compare);

    for (i = 1; i < names_added; i++)
    {
      if (memcmp(buf0 + 18*i,buf0 + 18*(i-1),16) == 0)
      {
	    names_added--;
	    if (names_added == i) break;

	    memmove(buf0 + 18*i, buf0 + 18*(i+1), 18*(names_added-i));
	    i--;
      }
    }
    
    buf = buf0 + 18*names_added;

    n = n->next;

    if (!n)
    {
      /* end of this name list: add wins names too? */
      struct subnet_record *w_d;

      if (!(w_d = find_subnet(ipgrp))) break;

      if (w_d != d)
      {
        d = w_d;
        n = d->namelist; /* start on the wins name list */
      }
	}
	if (!n) break;
  }
  
  SCVAL(countptr,0,names_added);
  
  /* XXXXXXX we should fill in more fields of the statistics structure */
  bzero(buf,64);
  {
    extern int num_good_sends,num_good_receives;
    SIVAL(buf,20,num_good_sends);
    SIVAL(buf,24,num_good_receives);
  }
  
  buf += 46;
  
  /* Send a POSITIVE NAME STATUS RESPONSE */
  reply_netbios_packet(p,nmb->header.name_trn_id,
			   0,NMB_STATUS,0,False,False,
		       &nmb->question.question_name,
		       RR_TYPE_STATUS, RR_CLASS_IP,
		       0,
		       rdata,PTR_DIFF(buf,rdata));
}

