/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   Module name: namelib.c

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   05 jul 96: lkcl@pires.co.uk
   created module namelib 

*/


#include "includes.h"

extern int ServerNMB;

extern int DEBUGLEVEL;


/****************************************************************************
  response from a name query server check. 
  ****************************************************************************/
void response_name_srv(time_t timestamp,
		void (*fn)(time_t , struct packet_struct *, struct response_record *),
		struct packet_struct *p, struct response_record *n)
{
	struct nmb_packet *nmb = &p->packet.nmb;
	struct nmb_name *ans_name = &nmb->answers->rr_name;

	struct nmb_query_status *nbd = (struct nmb_query_status*)n->nmb_data;

    /* issue another state: this time to do a name status check */

    /* initiate a name status check on the server that replied */
    struct nmb_query_status *nmb_data;

    nmb_data = (struct nmb_query_status *)(malloc(sizeof(*nmb_data)));

    if (nmb_data)
    {
		nmb_data->d = nbd->d;

        netbios_name_status(timestamp, ServerNMB, fn,
		        (void*)nmb_data, ans_name, 
                False,False,n->send_ip);
    }
}


/****************************************************************************
  interpret a node status response. this is pretty hacked: we need two bits of
  info. a) the name of the workgroup b) the name of the server. it will also
  add all the names it finds into the namelist.
****************************************************************************/
static BOOL interpret_node_status(struct subnet_record *d,
                char *p, struct nmb_name *name,int t,
               char *serv_name, struct in_addr ip, BOOL bcast)
{
  int level = t==0x20 ? 4 : 0;
  int numnames = CVAL(p,0);
  BOOL found = False;

  DEBUG(level,("received %d names\n",numnames));

  p += 1;

  if (serv_name) *serv_name = 0;

  while (numnames--)
  {
      struct nmb_name qname;
      fstring flags;
      int nb_flags;
      
      BOOL group = False;
      
      *flags = 0;
      
      make_nmb_name(&qname,p,CVAL(p,15),NULL);
      nb_flags = p[16];
      
      p += 18;
      
      if (NAME_GROUP    (nb_flags)) { strcat(flags,"<GROUP> "); group=True;}
      if (NAME_BFLAG    (nb_flags)) { strcat(flags,"B "); }
      if (NAME_PFLAG    (nb_flags)) { strcat(flags,"P "); }
      if (NAME_MFLAG    (nb_flags)) { strcat(flags,"M "); }
      if (NAME_HFLAG    (nb_flags)) { strcat(flags,"H "); }
      if (NAME_DEREG    (nb_flags)) { strcat(flags,"<DEREGISTERING> "); }
      if (NAME_CONFLICT (nb_flags)) { strcat(flags,"<CONFLICT> "); }
      if (NAME_ACTIVE   (nb_flags)) { strcat(flags,"<ACTIVE> "); }
      if (NAME_PERMANENT(nb_flags)) { strcat(flags,"<PERMANENT> "); }
      
      /* we want the server name */
      if (serv_name && !*serv_name && !group && t == 0)
      {
        StrnCpy(serv_name,qname.name,15);
        serv_name[15] = 0;
      }
      
      /* looking for a name and type? */
      if (name && !found && (t == qname.name_type))
      {
        /* take a guess at some of the name types we're going to ask for.
           evaluate whether they are group names or no... */
        if (((t == 0x1b || t == 0x1d             ) && !group) ||
            ((t == 0x20 || t == 0x1c || t == 0x1e) &&  group))
          {
            found = True;
            memcpy(&qname,name,sizeof(*name));
          }
      }
      
      DEBUG(level,("\t%s\t%s\n",namestr(&qname),flags));
  }
  DEBUG(level,("num_good_sends=%d num_good_receives=%d\n",
           IVAL(p,20),IVAL(p,24)));
  return found;
}


/****************************************************************************
  response from a name status check. states of type NAME_STATUS_DOM_SRV_CHK
  and NAME_STATUS_SRV_CHK dealt with here.
  ****************************************************************************/
void response_status_check(time_t timestamp, BOOL dmb, struct packet_struct *p,
        struct response_record *n)
{
	if (p)
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		BOOL bcast = nmb->header.nm_flags.bcast;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		struct nmb_query_status *nbd = (struct nmb_query_status*)n->nmb_data;

		/* NMB_STATUS arrives: contains workgroup name and server name required.
		   amongst other things. */

		struct nmb_name name;
		fstring serv_name;

		if (bcast || rcode != 0 || rdata == NULL)
		{
			return;
		}

		if (interpret_node_status(nbd->d,rdata,&name,name.name_type,
									serv_name,p->ip,bcast))
		{
			if (*serv_name)
			{
				sync_server(timestamp,dmb,
				          serv_name,name.name,name.name_type,n->send_ip);
			}
		}
		else
		{
			DEBUG(1,("No 0x1d name type in interpret_node_status()\n"));
		}
    }
}


/****************************************************************************
  report any problems with the fact that a response has been received.

  (responses for certain types of operations are only expected from one host)
  ****************************************************************************/
BOOL query_problem_check(struct response_record *n,
            struct nmb_packet *nmb, char *qname)
{
    if (n->num_msgs > 1)
    {
        if (nmb->header.rcode == 0 && nmb->answers->rdata)
        {
            int nb_flags = nmb->answers->rdata[0];

            if ((!NAME_GROUP(nb_flags)))
            {
              /* oh dear. more than one person responded to a unique name.
                 there is either a network problem, a configuration problem
                 or a server is mis-behaving */

              /* XXXX mark the name as in conflict, and then let the
                 person who just responded know that they must also mark it
                 as in conflict, and therefore must NOT use it.
                 see rfc1001.txt 15.1.3.5 */
                    
              DEBUG(3,("Unique Name conflict detected!\n"));
              return True;
            }
        }
        else
        {
             /* we have received a negative reply, having already received
                at least one response (pos/neg). something's really wrong! */

             DEBUG(3,("wierd name query problem detected!\n"));
             return True;
        }
    }
    return False;
}


