/* 
   Unix SMB/Netbios implementation.
   Version 3.0
   NBT netbios routines and daemon - version 3
   Copyright (C) Andrew Tridgell 1994-1996 Luke Leighton 1996
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   
   Revision History:

   14 jan 96: lkcl@pires.co.uk
   added multiple workgroup domain master support

   30 July 96: David.Chappell@mail.trincoll.edu
   Expanded multiple workgroup domain master browser support.

*/

#include "includes.h"

extern int DEBUGLEVEL;
extern BOOL CanRecurse;

extern int ClientDGRAM;
extern int ServerNMB;
extern int ClientPort;

/* this is our domain/workgroup/server database */
extern struct subnet_record *subnetlist;

extern int  updatecount;

extern struct in_addr ipgrp;                          

static void response_name_dom_srv_chk(time_t timestamp, struct packet_struct *p,
                                      struct response_record *n);

/****************************************************************************
  Send a announce request to the local net.
  
  This is called by become_master(). This purpose of this action is to
  encourage servers to send us host announcements right away.
  **************************************************************************/
void announce_request(struct work_record *work, struct in_addr ip)
{
  pstring outbuf;
  char *p;
  struct nmb_name from, to;
  char *my_name;

  if (!work) return;

  work->needannounce = True;

  DEBUG(2,("sending announce request to %s for workgroup %s\n",
       inet_ntoa(ip),work->work_group));

  my_name = brse_server_alias(work->token);

  bzero(outbuf,sizeof(outbuf));
  p = outbuf;
  CVAL(p,0) = ANN_AnnouncementRequest;
  p++;

  CVAL(p,0) = work->token; /* (local) unique workgroup token id */
  p++;
  StrnCpy(p,my_name,16);
  p = skip_string(p,1);
  
  /* XXXX note: if we sent the announcement request to 0x1d instead
     of 0x1e, then we could get the master browser to announce to
     us instead of the members of the workgroup. wha-hey! */

  make_nmb_name(&from, my_name         , 0x20, work->scope);
  make_nmb_name(&to  , work->work_group, 0x1e, work->scope);

  send_mailslot_reply(BROWSE_MAILSLOT,ClientDGRAM,outbuf,PTR_DIFF(p,outbuf),
              &from, *iface_ip(ip),
              &to  , ip);
}


/****************************************************************************
  request an announcement
  **************************************************************************/
void do_announce_request(char *info, struct nmb_name *from_name, 
                                     struct nmb_name *to_name,
             int announce_type, struct in_addr dest_ip)
{
  pstring outbuf;
  char *p;
  
  bzero(outbuf,sizeof(outbuf));
  p = outbuf;
  CVAL(p,0) = announce_type; 
  p++;
  
  DEBUG(2,("sending announce type %d: info %s to %s - server %s\n",
       announce_type, info, inet_ntoa(dest_ip),namestr(to_name)));
  
  StrnCpy(p,info,16);
  strupper(p);
  p = skip_string(p,1);
  
  send_mailslot_reply(BROWSE_MAILSLOT,ClientDGRAM,outbuf,PTR_DIFF(p,outbuf),
				from_name, *iface_ip(dest_ip),
				to_name  , dest_ip);
}


/****************************************************************************
  find a server responsible for a workgroup, and sync browse lists
  control ends up back here via response_name_query.
  **************************************************************************/
void sync_server(time_t time_now, BOOL dmb, char *serv_name, char *work_name, 
         int name_type,
         struct in_addr ip)
{
  int token = brse_domain_to_token(work_name);
  struct nmb_name from, to;
  char *work_scope = brse_domain_scope(token);

  /* with a domain master we can get the whole list (not local only list) */
  BOOL local_only = !dmb;

  add_browser_entry(time_now,serv_name,name_type,
                    work_name,work_scope,0,ip,local_only);

  if (dmb && brse_local_master(token))
  {
    char *my_name = brse_server_alias(token);
    make_nmb_name(&from, my_name  , 0x20, work_scope);
    make_nmb_name(&to  , serv_name, 0x0 , work_scope);

    /* announce ourselves as a master browser to serv_name */
    do_announce_request(my_name, /* info */
                        &from, /* from */
                        &to  , /* to */
                        ANN_MasterAnnouncement, ip);
  }
}


/****************************************************************************
  send a host announcement packet
  **************************************************************************/
void do_announce_host(int command,
        struct nmb_name *from_name, struct in_addr from_ip,
        struct nmb_name *to_name  , struct in_addr to_ip,
        time_t announce_interval,
        char *server_name, int server_type, 
		char major_version, char minor_version,
        uint16 browse_version, uint16 browse_sig,
		char *server_comment)
{
    pstring outbuf;
    char *p;

    bzero(outbuf,sizeof(outbuf));
    p = outbuf+1;

    /* command type */
    CVAL(outbuf,0) = command;

    /* announcement parameters */
    CVAL(p,0) = updatecount;
    SIVAL(p,1,announce_interval*1000); /* ms - despite the spec */

    StrnCpy(p+5,server_name,16);
    strupper(p+5);

    CVAL(p,21) = major_version; 
    CVAL(p,22) = minor_version; 

    SIVAL(p,23,server_type);

    SSVAL(p,27,browse_version);
    SSVAL(p,29,browse_sig);

    strcpy(p+31,server_comment);
    p += 31;
    p = skip_string(p,1);

    debug_browse_data(outbuf, PTR_DIFF(p,outbuf));

    /* send the announcement */
    send_mailslot_reply(BROWSE_MAILSLOT,ClientDGRAM,outbuf,
                      PTR_DIFF(p,outbuf),
                      from_name, from_ip,
                      to_name  , to_ip);
}


/****************************************************************************
  remove all samba's server entries
  ****************************************************************************/
void remove_my_servers(time_t time_now)
{
    struct subnet_record *d; 
    for (d = subnetlist; d; d = d->next)
    {
        struct work_record *work;
        for (work = d->workgrouplist; work; work = work->next)
        {
            struct server_record *s;
            for (s = work->serverlist; s; s = s->next)
            {
				struct nmb_name serv_name;
                if (!strequal(brse_server_alias(work->token),s->serv.name))
                {
                   continue;
                }
				make_nmb_name(&serv_name, s->serv.name, 0x0, work->scope);
                announce_server(time_now,d, work, &serv_name, s->serv.comment, 0, 0);
            }
        }
    }
}


/* a name query announce host structure */
struct nmb_query_annce
{
	time_t ttl; /* ms */

	struct in_addr myip;
	uint32 sv_type; /* server type */
	char name   [17]; /* server name */
	char comment[50]; /* server comment */
};

static void response_announce_host(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n);

/****************************************************************************
  announce a server entry
  ****************************************************************************/
void announce_server(time_t time_now,
				struct subnet_record *d, struct work_record *work,
             struct nmb_name *from, char *comment, time_t ttl, int server_type)
{
    /* domain type cannot have anything in it that might confuse
       a client into thinking that the domain is in fact a server.
       (SV_TYPE_SERVER_UNIX, for example)
     */
	uint32 domain_type = SV_TYPE_DOMAIN_ENUM|SV_TYPE_NT;

	BOOL wins_iface = ip_equal(d->bcast_ip, ipgrp);

	struct nmb_name to;

	if (wins_iface && server_type != 0)
	{
		/* wins pseudo-ip interface */
		if (!AM_MASTER(work))
		{
			/* non-master announce by unicast to the domain master */
			if (*lp_wins_server())
			{
              struct nmb_query_annce *nmb_data;

              nmb_data = (struct nmb_query_annce *)(malloc(sizeof(*nmb_data)));

              if (nmb_data)
              {
                char *name = from->name;

                make_nmb_name(&to,work->work_group, 0x1b, work->scope);

				nmb_data->ttl = ttl * 1000;
				nmb_data->sv_type = server_type;
				StrnCpy(nmb_data->name   ,name   ,sizeof(nmb_data->name   )-1);
				StrnCpy(nmb_data->comment,comment,sizeof(nmb_data->comment)-1);

				/* look up the domain master with the WINS server */
				/* response_announce_host() deals with results of query */
				netbios_name_query(time_now, ServerNMB, response_announce_host,
                     (void*)nmb_data, &to, 
					 False, False, ipgrp);
              }
			}
			else
			{
				/* we are the WINS server, but not the domain master.  */
				/* XXXX we need to look up the domain master in our
				   WINS database list, and do_announce_host(). maybe
				   we could do a name query on the unsuspecting domain
				   master just to make sure it's awake. */
			}
		}

		/* XXXX any other kinds of announcements we need to consider here?
		   e.g local master browsers... no. local master browsers do
		   local master announcements to their domain master. 
		 */
	}
	else
	{
		if (AM_MASTER(work))
		{
            make_nmb_name(&to,work->work_group, 0x1e, work->scope);

			DEBUG(3,("sending local master announce to %s %s\n",
							inet_ntoa(d->bcast_ip),namestr(&to)));

			do_announce_host(ANN_LocalMasterAnnouncement,
				from, d->myip,
				&to , d->bcast_ip,
				ttl*1000,
				from->name, server_type, 
				HOST_MAJOR_VERSION, HOST_MINOR_VERSION,
				HOST_BROWSE_VERSION, HOST_BROWSE_SIGNATURE,
                comment);

			DEBUG(3,("sending domain announce to %s %s\n",
							inet_ntoa(d->bcast_ip), namestr(&to)));

			/* XXXX should we do a domain-announce-kill? */
			if (server_type != 0)
			{
            	make_nmb_name(&to,MSBROWSE, 0x01, work->scope);

				do_announce_host(ANN_DomainAnnouncement,
					from, d->myip,
					&to , d->bcast_ip,
					ttl*1000,
					work->work_group, server_type ? domain_type : 0,
					WG_MAJOR_VERSION, WG_MINOR_VERSION,
					WG_BROWSE_VERSION, WG_BROWSE_SIGNATURE,
					from->name);
			}
		}
		else
		{
            make_nmb_name(&to,work->work_group, 0x1d, work->scope);

			DEBUG(3,("sending host announce to %s %s\n",
							inet_ntoa(d->bcast_ip),namestr(&to)));

			do_announce_host(ANN_HostAnnouncement,
				from, d->myip,
				&to , d->bcast_ip,
				ttl*1000,
				from->name, server_type, 
				HOST_MAJOR_VERSION, HOST_MINOR_VERSION,
				HOST_BROWSE_VERSION, HOST_BROWSE_SIGNATURE,
				comment);
		}
	}
}


/****************************************************************************
  response from a name query announce host
  NAME_QUERY_ANNOUNCE_HOST is dealt with here
  ****************************************************************************/
static void response_announce_host(time_t timestamp, struct packet_struct *p,
                                   struct response_record *n)
{
	if (!p)
	{
		return;
	}
	else
	{
		struct nmb_packet *nmb = &p->packet.nmb;
		struct nmb_name *ans_name = &nmb->answers->rr_name;
		int rcode = nmb->header.rcode;
		char *rdata = nmb->answers->rdata;

		struct nmb_ip found;
		struct nmb_ip *data = NULL;
			
		DEBUG(4,("NAME_QUERY_ANNCE_HOST\n"));

		if (rcode == 0 && rdata)
		{
			/* copy the netbios flags and the ip address out of reply data */
			found.nb_flags = rdata[0];
			putip((char*)&found.ip,&rdata[2]);
				
			data = &found;
		}

		DEBUG(4, ("Name query at %s ip %s - ",
			  namestr(&n->name), inet_ntoa(n->send_ip)));

		if (!name_equal(&n->name, ans_name))
		{
			/* someone gave us the wrong name as a reply. oops. */
			/* XXXX should say to them 'oi! release that name!' */

			DEBUG(4,("unexpected name received: %s\n", namestr(ans_name)));
			return;
		}

		if (data)
		{
			struct nmb_query_annce *nmb_data;
			struct nmb_name to;

			nmb_data = (struct nmb_query_annce*)n->nmb_data;

			/* we had sent out a name query to the current owner
			   of a name because someone else wanted it. now they
			   have responded saying that they still want the name,
			   so the other host can't have it.
			 */

			make_nmb_name(&to, nmb_data->name, 0x0, n->name.scope);

			/* do an announce host */
			do_announce_host(ANN_HostAnnouncement,
					&n->name, nmb_data->myip,
					&to    , data->ip,
					nmb_data->ttl,
					nmb_data->name, nmb_data->sv_type, 
					HOST_MAJOR_VERSION, HOST_MINOR_VERSION,
					HOST_BROWSE_VERSION, HOST_BROWSE_SIGNATURE,
					nmb_data->comment);
		}
		else
		{
			/* XXXX negative name query response. no master exists. oops */
		}
    }
}


/****************************************************************************
  construct a host announcement unicast
  **************************************************************************/
void announce_host(time_t time_now)
{
  struct subnet_record *d;
  pstring comment;

  for (d = subnetlist; d; d = d->next)
    {
      struct work_record *work;
      
      if (ip_equal(d->bcast_ip, ipgrp)) continue;

      for (work = d->workgrouplist; work; work = work->next)
    {
      uint32 stype = work->ServerType;
      struct server_record *s;
      
      char *my_name    = brse_server_alias  (work->token);
      char *my_comment = brse_server_comment(work->token);

      StrnCpy(comment, my_comment, 43);

      /* must work on the code that does announcements at up to
         30 seconds later if a master browser sends us a request
         announce.
       */

      if (work->needannounce) {
        /* drop back to a max 3 minute announce - this is to prevent a
           single lost packet from stuffing things up for too long */
        work->announce_interval = MIN(work->announce_interval,
                        CHECK_TIME_MIN_HOST_ANNCE*60);
        work->lastannounce_time = time_now - (work->announce_interval+1);
      }
      
      /* announce every minute at first then progress to every 12 mins */
      if (work->lastannounce_time && 
          (time_now - work->lastannounce_time) < work->announce_interval)
        continue;
      
      if (work->announce_interval < CHECK_TIME_MAX_HOST_ANNCE * 60) 
        work->announce_interval += 60;
      
      work->lastannounce_time = time_now;

      for (s = work->serverlist; s; s = s->next) {
        if (strequal(my_name, s->serv.name))
        { 
          struct nmb_name serv_name;
          make_nmb_name(&serv_name, my_name, 0x0, work->scope);

          DEBUG(4,("call to annnounce_server: %s %d %x\n",
                   namestr(&serv_name), work->announce_interval, stype));

          announce_server(time_now, d,work,&serv_name,comment,
                work->announce_interval,stype);
          break; 
        }
      }
      
      if (work->needannounce)
      {
          work->needannounce = False;
          break;
          /* sorry: can't do too many announces. do some more later */
      }
    }
  }
}


/****************************************************************************
  announce samba as a master to all other primary domain controllers.

  this actually gets done in search_and_sync_workgroups() via the
  NAME_QUERY_DOM_SRV_CHK state, if there is a response from the
  name query initiated here.  see response_name_query()
  **************************************************************************/
void announce_master(time_t time_now)
{
  struct subnet_record *d;
  static time_t last=0;
  BOOL am_master = False; /* are we a master of some sort? :-) */
  int token;

  if (!last) last = time_now;
  if (time_now-last < CHECK_TIME_MST_ANNOUNCE * 60) return;

  last = time_now;

  for (d = subnetlist; d; d = d->next)
  {
    struct work_record *work;
    for (work = d->workgrouplist; work; work = work->next)
    {
      if (AM_MASTER(work) || AM_DMBRSE(work))
      {
          am_master = True;
      }
    }
  }
  
  if (!am_master) return; /* only proceed if we are a master browser */
  
  for (token = 0; token < brse_num_domains(); token++)
  {
    /* each token represents a name of a workgroup. to save network
       traffic, do each workgroup separately. unfortunately, the
       workgroups are split up over several local subnets...
     */

    for (d = subnetlist; d; d = d->next)
    {
      struct work_record *work;
      for (work = d->workgrouplist; work; work = work->next)
      {
        struct server_record *s;
		struct nmb_name n;
        make_nmb_name(&n,work->work_group, 0x1b, work->scope);

        /* do each workgroup separately */
        if (work->token != token) continue;

        for (s = work->serverlist; s; s = s->next)
        {
          if (strequal(s->serv.name, brse_server_alias(work->token))) continue;
            
          /* all domain master browsers */
          if (s->serv.type & SV_TYPE_DOMAIN_MASTER)
          {
            /* check the existence of a domain master for this workgroup,
               and if one exists at the specified ip, sync with it and
               announce ourselves as a local master browser to it */
            
            /* exclude brse_domain_controller() from this check if it's
               in our browse lists, because it's dealt with separately
             */
            if (!*brse_domain_controller(work->token) ||
                !strequal(brse_domain_controller(work->token), s->serv.name))
            {
              if (*lp_wins_server())
              {
                /* query the WINS server (that may include samba itself)
                   to find the browse master
                 */
                struct nmb_query_status *nmb_data;

                nmb_data=(struct nmb_query_status*)(malloc(sizeof(*nmb_data)));

                if (nmb_data)
                {

				  nmb_data->d = d;
  
                  /* response_n_d_s_chk() deals with results of query */
                  netbios_name_query(time_now,ServerNMB,response_name_dom_srv_chk,
                             (void*)nmb_data, &n,
                             False, False, ipgrp);
                }
              }
              else
              {
                struct subnet_record *d2;
                for (d2 = subnetlist; d2; d2 = d2->next)
                {
                  /* query interfaces to find the browse master */
                  BOOL wins = ip_equal(d2->bcast_ip,ipgrp);

                  struct nmb_query_status *nmb_data;

                  nmb_data = (struct nmb_query_status *)
							(malloc(sizeof(*nmb_data)));

                  if (nmb_data)
                  {
				    nmb_data->d = d;
  
                    /* response_n_d_s_chk() deals with results of query */
                    netbios_name_query(time_now,ServerNMB,
                             response_name_dom_srv_chk,
                             (void*)nmb_data, &n,
                             !wins, False, d2->bcast_ip);
                  }
                }
              }
            }
          }
        }
        
        /* now do primary domain controller - the one that's not
           necessarily in our browse lists, although it ought to be
           this pdc is the one that we get TOLD about through smb.conf.
           basically, if it's on a subnet that we know about, it may end
           up in our browse lists (which is why it's explicitly excluded
           in the code above) */
        
        if (*brse_domain_controller(work->token))
        {
          struct in_addr ip;
          struct nmb_query_status *nmb_data;
            
          ip = *interpret_addr2(brse_domain_controller(work->token));
          
          /* if the ip is zero, then make the query to WINS server */
          /* XXXX later, if this also fails, we could also do a
             broadcast query on samba's local subnets
           */
            
          DEBUG(2, ("Searching for DOM %s at %s\n",
              brse_domain_controller(work->token), inet_ntoa(ip)));
            
          /* check the existence of a pdc for this workgroup, and if
             one exists at the specified ip, sync with it and announce
             ourselves as a master browser to it */

          nmb_data = (struct nmb_query_status *)(malloc(sizeof(*nmb_data)));

          if (nmb_data)
          {
            nmb_data->d = d;
  
            /* response_n_d_s_chk() deals with results of query */
            netbios_name_query(time_now,ServerNMB, response_name_dom_srv_chk,
                             (void*)nmb_data, &n,
                             False, False, ip);
          }
	    }
      }
    }
  }
}


/****************************************************************************
  results of a name status on a domain master browser
  **************************************************************************/
static void response_status_dom_srv(time_t timestamp, struct packet_struct *p,
                                    struct response_record *n)
{
	if (p)
	{
		/* response was received */
		DEBUG(4,("DOMAIN_MASTER_STATUS_CHK\n")); 
		response_status_check(timestamp, True, p, n);
	}
}


/****************************************************************************
  results of a name query for a domain master browser
  **************************************************************************/
static void response_name_dom_srv_chk(time_t timestamp, struct packet_struct *p,
                                      struct response_record *n)
{
	if (p)
	{
		/* response was received */
    	DEBUG(4,("DOMAIN_MASTER_CHECK\n")); 
		response_name_srv(timestamp,response_status_dom_srv, p, n);
	}
}


/****************************************************************************
  do all the "remote" announcements. These are used to put ourselves
  on a remote browse list. They are done blind, no checking is done to
  see if there is actually a browse master at the other end.
  **************************************************************************/
void announce_remote(time_t time_now)
{
  char *s,*ptr;
  static time_t last_time = 0;
  pstring s2;
  struct in_addr addr;
  char *comment,*workgroup;
  int token;
  int stype = DFLT_SERVER_TYPE;

  if (last_time && time_now < last_time + REMOTE_ANNOUNCE_INTERVAL)
    return;

  last_time = time_now;

  s = lp_remote_announce();
  if (!*s) return;

  comment = lp_serverstring(); /* default comment */
  workgroup = lp_workgroup();  /* default workgroup name */

  for (ptr=s; next_token(&ptr,s2,NULL); )
  {
    /* the entries are of the form a.b.c.d/WORKGROUP with 
       WORKGROUP being optional */
    struct nmb_name from, to;

    char *wgroup;
    char *my_name;

    wgroup = strchr(s2,'/');
    if (wgroup) *wgroup++ = 0;
    if (!wgroup || !*wgroup)
      wgroup = workgroup;

    addr = *interpret_addr2(s2);
    
    token = brse_domain_to_token(wgroup);
    my_name = brse_server_alias(token);
    my_name = my_name ? my_name : lp_server_alias();

    make_nmb_name(&from,my_name, 0x20, brse_domain_scope(token));
    make_nmb_name(&to  ,wgroup , 0x1e, brse_domain_scope(token));

    do_announce_host(ANN_HostAnnouncement,
             &from, *iface_ip(addr),
             &to  , addr,
             REMOTE_ANNOUNCE_INTERVAL,
             my_name,stype,
			HOST_MAJOR_VERSION, HOST_MINOR_VERSION,
			HOST_BROWSE_VERSION, HOST_BROWSE_SIGNATURE,
			comment);    
  }
}
