/*
 * Copyright 2018 The OpenSSL Project Authors. All Rights Reserved.
 *
 * Licensed under the Apache License 2.0 (the "License").  You may not use
 * this file except in compliance with the License.  You can obtain a copy
 * in the file LICENSE in the source distribution or at
 * https://www.openssl.org/source/license.html
 */

/*
 * Subset of DRBG test vectors from:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/
 * The index in the names given below (e.g- kat1680)- refers to the CAVS index.
 */

#include <openssl/obj_mac.h>
#include "internal/nelem.h"
#include "drbg_cavs_data.h"


static const unsigned char kat1_nor_entropyin[] = {
    0x89, 0x0e, 0xb0, 0x67, 0xac, 0xf7, 0x38, 0x2e, 0xff, 0x80, 0xb0, 0xc7,
    0x3b, 0xc8, 0x72, 0xc6,
};
static const unsigned char kat1_nor_nonce[] = {
    0xaa, 0xd4, 0x71, 0xef, 0x3e, 0xf1, 0xd2, 0x03,
};
static const unsigned char kat1_nor_persstr[] = {0};
static const unsigned char kat1_nor_addin0[] = {0};
static const unsigned char kat1_nor_addin1[] = {0};
static const unsigned char kat1_nor_retbytes[] = {
    0xa5, 0x51, 0x4e, 0xd7, 0x09, 0x5f, 0x64, 0xf3, 0xd0, 0xd3, 0xa5, 0x76,
    0x03, 0x94, 0xab, 0x42, 0x06, 0x2f, 0x37, 0x3a, 0x25, 0x07, 0x2a, 0x6e,
    0xa6, 0xbc, 0xfd, 0x84, 0x89, 0xe9, 0x4a, 0xf6, 0xcf, 0x18, 0x65, 0x9f,
    0xea, 0x22, 0xed, 0x1c, 0xa0, 0xa9, 0xe3, 0x3f, 0x71, 0x8b, 0x11, 0x5e,
    0xe5, 0x36, 0xb1, 0x28, 0x09, 0xc3, 0x1b, 0x72, 0xb0, 0x8d, 0xdd, 0x8b,
    0xe1, 0x91, 0x0f, 0xa3,
};
static const struct drbg_kat_no_reseed kat1_nor_t = {
    0, kat1_nor_entropyin, kat1_nor_nonce, kat1_nor_persstr,
    kat1_nor_addin0, kat1_nor_addin1, kat1_nor_retbytes
};
static const struct drbg_kat kat1_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat1_nor_t
};

static const unsigned char kat2_nor_entropyin[] = {
    0xc4, 0x7b, 0xe8, 0xe8, 0x21, 0x9a, 0x5a, 0x87, 0xc9, 0x40, 0x64, 0xa5,
    0x12, 0x08, 0x9f, 0x2b,
};
static const unsigned char kat2_nor_nonce[] = {
    0xf2, 0xa2, 0x3e, 0x63, 0x6a, 0xee, 0x75, 0xc6,
};
static const unsigned char kat2_nor_persstr[] = {0};
static const unsigned char kat2_nor_addin0[] = {0};
static const unsigned char kat2_nor_addin1[] = {0};
static const unsigned char kat2_nor_retbytes[] = {
    0x5a, 0x16, 0x50, 0xbb, 0x6d, 0x6a, 0x16, 0xf6, 0x04, 0x05, 0x91, 0xd5,
    0x6a, 0xbc, 0xd5, 0xdd, 0x3d, 0xb8, 0x77, 0x2a, 0x9c, 0x75, 0xc4, 0x4d,
    0x9f, 0xc6, 0x4d, 0x51, 0xb7, 0x33, 0xd4, 0xa6, 0x75, 0x9b, 0xd5, 0xa6,
    0x4e, 0xc4, 0x23, 0x1a, 0x24, 0xe6, 0x62, 0xfd, 0xd4, 0x7c, 0x82, 0xdb,
    0x63, 0xb2, 0x00, 0xda, 0xf8, 0xd0, 0x98, 0x56, 0x0e, 0xb5, 0xba, 0x7b,
    0xf3, 0xf9, 0xab, 0xf7,
};
static const struct drbg_kat_no_reseed kat2_nor_t = {
    1, kat2_nor_entropyin, kat2_nor_nonce, kat2_nor_persstr,
    kat2_nor_addin0, kat2_nor_addin1, kat2_nor_retbytes
};
static const struct drbg_kat kat2_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat2_nor_t
};

static const unsigned char kat3_nor_entropyin[] = {
    0x13, 0x0b, 0x8c, 0x3d, 0x2d, 0x7b, 0x6e, 0x02, 0xc4, 0x10, 0xb4, 0x16,
    0x8e, 0x12, 0x2c, 0x38,
};
static const unsigned char kat3_nor_nonce[] = {
    0x79, 0xa6, 0x74, 0xc5, 0xb2, 0xc5, 0x1a, 0xa9,
};
static const unsigned char kat3_nor_persstr[] = {0};
static const unsigned char kat3_nor_addin0[] = {0};
static const unsigned char kat3_nor_addin1[] = {0};
static const unsigned char kat3_nor_retbytes[] = {
    0x57, 0xe8, 0xa1, 0xe5, 0x78, 0xed, 0xe1, 0xc6, 0x68, 0x79, 0xc4, 0x30,
    0xdf, 0x72, 0x64, 0x35, 0xd5, 0x1a, 0x36, 0x9a, 0x0f, 0xe5, 0x9a, 0x03,
    0x58, 0xd1, 0xde, 0x35, 0x2d, 0x42, 0x80, 0xfd, 0x7b, 0x22, 0x5f, 0x5f,
    0x38, 0x6a, 0x4f, 0xcf, 0x12, 0xf7, 0x27, 0x94, 0xad, 0x0f, 0x37, 0x57,
    0xfb, 0x25, 0xde, 0xba, 0x3c, 0x75, 0x12, 0xce, 0x4d, 0x37, 0x33, 0xc7,
    0xee, 0x06, 0x70, 0x43,
};
static const struct drbg_kat_no_reseed kat3_nor_t = {
    2, kat3_nor_entropyin, kat3_nor_nonce, kat3_nor_persstr,
    kat3_nor_addin0, kat3_nor_addin1, kat3_nor_retbytes
};
static const struct drbg_kat kat3_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat3_nor_t
};

static const unsigned char kat4_nor_entropyin[] = {
    0x04, 0xd2, 0x41, 0x45, 0x28, 0x76, 0x64, 0xf6, 0x76, 0x2b, 0x5d, 0x2a,
    0x10, 0x2a, 0xc6, 0x03,
};
static const unsigned char kat4_nor_nonce[] = {
    0xec, 0xac, 0x63, 0xe1, 0x21, 0x7e, 0xe3, 0x35,
};
static const unsigned char kat4_nor_persstr[] = {0};
static const unsigned char kat4_nor_addin0[] = {0};
static const unsigned char kat4_nor_addin1[] = {0};
static const unsigned char kat4_nor_retbytes[] = {
    0xfa, 0x5e, 0x74, 0x6d, 0xec, 0xd6, 0x80, 0x1e, 0xb7, 0x08, 0x3b, 0x6f,
    0x0e, 0x72, 0x43, 0x2e, 0x1f, 0xd4, 0x24, 0x31, 0x04, 0xf7, 0x48, 0xd0,
    0xf1, 0x90, 0x83, 0x92, 0x3b, 0x95, 0x55, 0x68, 0x8f, 0x43, 0x14, 0x6d,
    0x5a, 0xce, 0xa9, 0x62, 0xda, 0x01, 0x23, 0x1d, 0x9e, 0x5f, 0xaf, 0xf0,
    0xe8, 0x1f, 0x3d, 0x39, 0x4a, 0xce, 0x3a, 0x34, 0x54, 0x53, 0x6d, 0x72,
    0x65, 0x75, 0x04, 0x1f,
};
static const struct drbg_kat_no_reseed kat4_nor_t = {
    3, kat4_nor_entropyin, kat4_nor_nonce, kat4_nor_persstr,
    kat4_nor_addin0, kat4_nor_addin1, kat4_nor_retbytes
};
static const struct drbg_kat kat4_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat4_nor_t
};

static const unsigned char kat5_nor_entropyin[] = {
    0xae, 0xde, 0x4e, 0x61, 0x35, 0x56, 0xb1, 0xd5, 0xa3, 0x0f, 0xce, 0x26,
    0x1f, 0xbb, 0x82, 0x0c,
};
static const unsigned char kat5_nor_nonce[] = {
    0x39, 0xac, 0xba, 0x03, 0xc5, 0xf1, 0x0a, 0xf4,
};
static const unsigned char kat5_nor_persstr[] = {0};
static const unsigned char kat5_nor_addin0[] = {0};
static const unsigned char kat5_nor_addin1[] = {0};
static const unsigned char kat5_nor_retbytes[] = {
    0x23, 0x2c, 0x44, 0xb8, 0x19, 0xb8, 0x8f, 0x1a, 0xeb, 0x83, 0xf2, 0x03,
    0x4f, 0x84, 0x2d, 0x5a, 0x00, 0xf0, 0x30, 0x15, 0x05, 0xd2, 0xaa, 0x69,
    0xaa, 0xec, 0xb3, 0xcb, 0x14, 0xbc, 0xb1, 0x58, 0x75, 0xe0, 0xfd, 0x60,
    0x07, 0x1a, 0x80, 0xf6, 0x26, 0x2d, 0xce, 0xbc, 0xf4, 0x1a, 0x0e, 0x14,
    0x76, 0xd9, 0x6f, 0x40, 0x97, 0x12, 0xd8, 0x28, 0xae, 0x31, 0x3a, 0x9d,
    0x28, 0xec, 0x2d, 0xee,
};
static const struct drbg_kat_no_reseed kat5_nor_t = {
    4, kat5_nor_entropyin, kat5_nor_nonce, kat5_nor_persstr,
    kat5_nor_addin0, kat5_nor_addin1, kat5_nor_retbytes
};
static const struct drbg_kat kat5_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat5_nor_t
};

static const unsigned char kat6_nor_entropyin[] = {
    0x9f, 0x90, 0x54, 0x1c, 0x10, 0xd4, 0xb7, 0xc0, 0x89, 0xfe, 0x68, 0x8e,
    0xa3, 0xef, 0x4f, 0xc6,
};
static const unsigned char kat6_nor_nonce[] = {
    0x1e, 0xac, 0x1c, 0x22, 0x03, 0x6e, 0x2b, 0x22,
};
static const unsigned char kat6_nor_persstr[] = {0};
static const unsigned char kat6_nor_addin0[] = {0};
static const unsigned char kat6_nor_addin1[] = {0};
static const unsigned char kat6_nor_retbytes[] = {
    0x71, 0xaf, 0x3f, 0xdf, 0x67, 0x34, 0x04, 0x16, 0x3b, 0x06, 0x73, 0x7e,
    0x0f, 0x39, 0x91, 0x5f, 0xae, 0xc2, 0x18, 0x21, 0x81, 0x6c, 0x31, 0x42,
    0xe8, 0x0a, 0x50, 0x3c, 0x70, 0xcb, 0x2e, 0xdd, 0x46, 0x8c, 0x3f, 0x03,
    0xcb, 0x1f, 0x8a, 0x2f, 0x92, 0x48, 0x63, 0x5b, 0x63, 0xd7, 0xb8, 0xf1,
    0x9e, 0x9e, 0x11, 0xca, 0xf0, 0xab, 0x0c, 0x3d, 0x2f, 0xf9, 0xc7, 0x13,
    0x21, 0xca, 0xc1, 0x54,
};
static const struct drbg_kat_no_reseed kat6_nor_t = {
    5, kat6_nor_entropyin, kat6_nor_nonce, kat6_nor_persstr,
    kat6_nor_addin0, kat6_nor_addin1, kat6_nor_retbytes
};
static const struct drbg_kat kat6_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat6_nor_t
};

static const unsigned char kat7_nor_entropyin[] = {
    0x3b, 0x8a, 0x18, 0x1c, 0xea, 0x83, 0x90, 0xd5, 0xd7, 0xa4, 0xe6, 0x51,
    0x5c, 0xf9, 0x2e, 0x3e,
};
static const unsigned char kat7_nor_nonce[] = {
    0x37, 0xf4, 0xd1, 0x74, 0x87, 0x14, 0x34, 0x5b,
};
static const unsigned char kat7_nor_persstr[] = {0};
static const unsigned char kat7_nor_addin0[] = {0};
static const unsigned char kat7_nor_addin1[] = {0};
static const unsigned char kat7_nor_retbytes[] = {
    0xd6, 0xfd, 0x0f, 0xfb, 0x10, 0x85, 0x0e, 0xb6, 0xeb, 0x70, 0x50, 0xe6,
    0x1e, 0xac, 0x00, 0xd4, 0x72, 0xf6, 0x5c, 0xd3, 0xd9, 0x35, 0x08, 0x14,
    0x77, 0xfe, 0x44, 0xaa, 0x85, 0x69, 0x4e, 0x12, 0x0a, 0xb5, 0xa1, 0xca,
    0x1f, 0xa1, 0x98, 0xaf, 0x76, 0xdf, 0xa8, 0xd0, 0xab, 0xdf, 0x53, 0xe8,
    0x5a, 0xa8, 0xc8, 0x7f, 0xed, 0x0a, 0x8c, 0x24, 0x16, 0x39, 0x43, 0xb9,
    0x6d, 0x80, 0xaa, 0xfb,
};
static const struct drbg_kat_no_reseed kat7_nor_t = {
    6, kat7_nor_entropyin, kat7_nor_nonce, kat7_nor_persstr,
    kat7_nor_addin0, kat7_nor_addin1, kat7_nor_retbytes
};
static const struct drbg_kat kat7_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat7_nor_t
};

static const unsigned char kat8_nor_entropyin[] = {
    0x85, 0x4c, 0xaa, 0x2a, 0x74, 0xf3, 0xf4, 0x3b, 0x6a, 0xbd, 0x80, 0x6d,
    0x67, 0x48, 0xed, 0x80,
};
static const unsigned char kat8_nor_nonce[] = {
    0x0c, 0xbd, 0x13, 0x72, 0xbe, 0xb6, 0x27, 0x36,
};
static const unsigned char kat8_nor_persstr[] = {0};
static const unsigned char kat8_nor_addin0[] = {0};
static const unsigned char kat8_nor_addin1[] = {0};
static const unsigned char kat8_nor_retbytes[] = {
    0x46, 0x53, 0x40, 0x28, 0x16, 0x5f, 0xfe, 0x28, 0x45, 0xa9, 0x72, 0x62,
    0x7e, 0x67, 0xe1, 0x53, 0x51, 0x0c, 0xa1, 0x88, 0x0e, 0x6a, 0x3a, 0xd3,
    0x1f, 0xde, 0xe7, 0x1f, 0xf2, 0x40, 0xd3, 0x27, 0x86, 0x24, 0xb3, 0x1f,
    0x2d, 0x38, 0x6d, 0x7b, 0x22, 0x28, 0xce, 0xd6, 0x24, 0xa4, 0x2e, 0x7a,
    0x3b, 0x07, 0x48, 0x0b, 0x23, 0x23, 0x16, 0x6c, 0x18, 0xd1, 0xac, 0x0f,
    0x60, 0x00, 0x2e, 0xc4,
};
static const struct drbg_kat_no_reseed kat8_nor_t = {
    7, kat8_nor_entropyin, kat8_nor_nonce, kat8_nor_persstr,
    kat8_nor_addin0, kat8_nor_addin1, kat8_nor_retbytes
};
static const struct drbg_kat kat8_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat8_nor_t
};

static const unsigned char kat9_nor_entropyin[] = {
    0x87, 0xe1, 0xc5, 0x32, 0x99, 0x7f, 0x57, 0xa3, 0x5c, 0x28, 0x6d, 0xe8,
    0x64, 0xbf, 0xf2, 0x64,
};
static const unsigned char kat9_nor_nonce[] = {
    0xa3, 0x9e, 0x98, 0xdb, 0x6c, 0x10, 0x78, 0x7f,
};
static const unsigned char kat9_nor_persstr[] = {0};
static const unsigned char kat9_nor_addin0[] = {0};
static const unsigned char kat9_nor_addin1[] = {0};
static const unsigned char kat9_nor_retbytes[] = {
    0x2c, 0x14, 0x7e, 0x24, 0x11, 0x9a, 0xd8, 0xd4, 0xb2, 0xed, 0x61, 0xc1,
    0x53, 0xd0, 0x50, 0xc9, 0x24, 0xff, 0x59, 0x75, 0x15, 0xf1, 0x17, 0x3a,
    0x3d, 0xf4, 0x4b, 0x2c, 0x84, 0x28, 0xef, 0x89, 0x0e, 0xb9, 0xde, 0xf3,
    0xe4, 0x78, 0x04, 0xb2, 0xfd, 0x9b, 0x35, 0x7f, 0xe1, 0x3f, 0x8a, 0x3e,
    0x10, 0xc8, 0x67, 0x0a, 0xf9, 0xdf, 0x2d, 0x6c, 0x96, 0xfb, 0xb2, 0xb8,
    0xcb, 0x2d, 0xd6, 0xb0,
};
static const struct drbg_kat_no_reseed kat9_nor_t = {
    8, kat9_nor_entropyin, kat9_nor_nonce, kat9_nor_persstr,
    kat9_nor_addin0, kat9_nor_addin1, kat9_nor_retbytes
};
static const struct drbg_kat kat9_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat9_nor_t
};

static const unsigned char kat10_nor_entropyin[] = {
    0x2f, 0xc6, 0x23, 0x42, 0x90, 0x29, 0xc9, 0x6e, 0xde, 0xf6, 0x16, 0x62,
    0x47, 0xb0, 0x8c, 0xb0,
};
static const unsigned char kat10_nor_nonce[] = {
    0x99, 0x82, 0x66, 0x33, 0x55, 0x58, 0x27, 0x88,
};
static const unsigned char kat10_nor_persstr[] = {0};
static const unsigned char kat10_nor_addin0[] = {0};
static const unsigned char kat10_nor_addin1[] = {0};
static const unsigned char kat10_nor_retbytes[] = {
    0x55, 0x96, 0xcb, 0x16, 0xf3, 0xbe, 0x85, 0x52, 0xc1, 0xe5, 0xc1, 0x64,
    0xd5, 0x40, 0xcb, 0x1f, 0xaf, 0x4b, 0xea, 0x87, 0x33, 0xb6, 0x0a, 0x8a,
    0xd0, 0xc4, 0x06, 0x26, 0x25, 0x65, 0x48, 0xc7, 0xaa, 0x96, 0xd1, 0xd2,
    0x72, 0x9d, 0x26, 0xf0, 0x08, 0x73, 0x1f, 0xc3, 0x93, 0x07, 0xbe, 0x5b,
    0xcd, 0x20, 0x81, 0xc6, 0x9e, 0x31, 0x4e, 0x0c, 0x73, 0xe3, 0xd0, 0xfd,
    0x1d, 0x90, 0x58, 0x28,
};
static const struct drbg_kat_no_reseed kat10_nor_t = {
    9, kat10_nor_entropyin, kat10_nor_nonce, kat10_nor_persstr,
    kat10_nor_addin0, kat10_nor_addin1, kat10_nor_retbytes
};
static const struct drbg_kat kat10_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat10_nor_t
};

static const unsigned char kat11_nor_entropyin[] = {
    0x98, 0xab, 0x8b, 0x4e, 0xaf, 0xab, 0x6e, 0x53, 0x6f, 0x78, 0x45, 0xab,
    0xec, 0x13, 0x78, 0x08,
};
static const unsigned char kat11_nor_nonce[] = {
    0xdb, 0xa9, 0x44, 0xc9, 0x8b, 0x31, 0x1d, 0x8e,
};
static const unsigned char kat11_nor_persstr[] = {0};
static const unsigned char kat11_nor_addin0[] = {0};
static const unsigned char kat11_nor_addin1[] = {0};
static const unsigned char kat11_nor_retbytes[] = {
    0x86, 0xee, 0xd3, 0xa9, 0xfa, 0x53, 0x45, 0x2f, 0xb1, 0x1d, 0xba, 0x9c,
    0xac, 0x8e, 0x44, 0x02, 0x52, 0x29, 0x28, 0xf2, 0x70, 0x5a, 0x5e, 0x58,
    0x2f, 0x4d, 0x00, 0xeb, 0x8f, 0xed, 0x81, 0x8e, 0x62, 0x9c, 0x72, 0xa6,
    0xa7, 0x79, 0xbe, 0xb4, 0xed, 0x9a, 0x23, 0x93, 0x68, 0x23, 0x3c, 0xbf,
    0xcf, 0x55, 0x68, 0x5d, 0xbf, 0x2d, 0xe3, 0x4a, 0xb5, 0x89, 0x20, 0xcf,
    0xac, 0xa4, 0xaa, 0xfe,
};
static const struct drbg_kat_no_reseed kat11_nor_t = {
    10, kat11_nor_entropyin, kat11_nor_nonce, kat11_nor_persstr,
    kat11_nor_addin0, kat11_nor_addin1, kat11_nor_retbytes
};
static const struct drbg_kat kat11_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat11_nor_t
};

static const unsigned char kat12_nor_entropyin[] = {
    0x7d, 0xcf, 0x4f, 0xa7, 0x31, 0x13, 0x9c, 0x5b, 0xb6, 0x44, 0x2f, 0xed,
    0x91, 0xe8, 0x9f, 0x68,
};
static const unsigned char kat12_nor_nonce[] = {
    0xff, 0xd9, 0x4f, 0xe2, 0x1a, 0x80, 0x8b, 0x15,
};
static const unsigned char kat12_nor_persstr[] = {0};
static const unsigned char kat12_nor_addin0[] = {0};
static const unsigned char kat12_nor_addin1[] = {0};
static const unsigned char kat12_nor_retbytes[] = {
    0x8e, 0xca, 0x20, 0xe3, 0x1c, 0x98, 0x39, 0xb7, 0x41, 0xaa, 0xa9, 0xbf,
    0x6c, 0xee, 0xe2, 0x24, 0xd3, 0x21, 0x26, 0xb3, 0x19, 0x6e, 0xef, 0x3e,
    0xcd, 0x34, 0x3d, 0x41, 0x4d, 0x32, 0x33, 0xb9, 0xfd, 0x0e, 0xa0, 0xed,
    0x1b, 0xc7, 0x70, 0x0c, 0x88, 0xcd, 0x7c, 0x88, 0xd3, 0xc0, 0x76, 0x13,
    0xc4, 0x2c, 0xd1, 0xf9, 0x09, 0xfe, 0xd8, 0xc6, 0xa7, 0x08, 0xd0, 0x5d,
    0x6b, 0x68, 0xfb, 0x2e,
};
static const struct drbg_kat_no_reseed kat12_nor_t = {
    11, kat12_nor_entropyin, kat12_nor_nonce, kat12_nor_persstr,
    kat12_nor_addin0, kat12_nor_addin1, kat12_nor_retbytes
};
static const struct drbg_kat kat12_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat12_nor_t
};

static const unsigned char kat13_nor_entropyin[] = {
    0x51, 0x9c, 0x6e, 0xfe, 0xde, 0xd2, 0xa1, 0x10, 0xac, 0x41, 0x83, 0x9a,
    0x8b, 0x8a, 0xbf, 0xce,
};
static const unsigned char kat13_nor_nonce[] = {
    0x81, 0xd9, 0x5e, 0xdc, 0x06, 0xdd, 0xe6, 0xb3,
};
static const unsigned char kat13_nor_persstr[] = {0};
static const unsigned char kat13_nor_addin0[] = {0};
static const unsigned char kat13_nor_addin1[] = {0};
static const unsigned char kat13_nor_retbytes[] = {
    0x39, 0xcd, 0xd1, 0x0e, 0x49, 0xe0, 0x35, 0x81, 0xe3, 0x81, 0x1d, 0xdd,
    0x07, 0xd9, 0xd0, 0xac, 0xc3, 0x40, 0x85, 0xa1, 0x2c, 0x1c, 0x6b, 0x87,
    0xa7, 0x63, 0x86, 0xe9, 0x70, 0x32, 0xdc, 0x01, 0xf5, 0x23, 0xf6, 0x32,
    0xec, 0x95, 0x0d, 0x04, 0x34, 0xc2, 0x59, 0x37, 0xe4, 0x32, 0xd9, 0x85,
    0x54, 0x59, 0x9a, 0x85, 0x5d, 0xb9, 0xad, 0xf5, 0x8b, 0x9e, 0x04, 0x59,
    0x7a, 0x21, 0xd0, 0x0d,
};
static const struct drbg_kat_no_reseed kat13_nor_t = {
    12, kat13_nor_entropyin, kat13_nor_nonce, kat13_nor_persstr,
    kat13_nor_addin0, kat13_nor_addin1, kat13_nor_retbytes
};
static const struct drbg_kat kat13_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat13_nor_t
};

static const unsigned char kat14_nor_entropyin[] = {
    0x80, 0xc1, 0x9e, 0xea, 0xad, 0x1a, 0x58, 0x37, 0x94, 0x66, 0xaf, 0xae,
    0x1d, 0x80, 0x87, 0x6d,
};
static const unsigned char kat14_nor_nonce[] = {
    0xd9, 0xa2, 0x09, 0x3f, 0x11, 0x89, 0x2b, 0x82,
};
static const unsigned char kat14_nor_persstr[] = {0};
static const unsigned char kat14_nor_addin0[] = {0};
static const unsigned char kat14_nor_addin1[] = {0};
static const unsigned char kat14_nor_retbytes[] = {
    0xce, 0xc5, 0x1b, 0x98, 0x5e, 0xc9, 0x7b, 0x18, 0xee, 0x8e, 0xad, 0x36,
    0x15, 0x7b, 0xea, 0xf9, 0x6f, 0x12, 0x9f, 0x28, 0x28, 0x08, 0x89, 0xec,
    0x87, 0x3c, 0x27, 0xb5, 0x62, 0x71, 0x98, 0xc5, 0x85, 0xa6, 0xad, 0x21,
    0xae, 0x23, 0xa9, 0x59, 0xc9, 0xfa, 0x49, 0xd9, 0x85, 0xaf, 0x0d, 0xf4,
    0x02, 0x8f, 0xdf, 0x1f, 0x51, 0xd8, 0x2e, 0x8f, 0x2b, 0x3f, 0x02, 0x88,
    0x53, 0xf1, 0x4e, 0x8f,
};
static const struct drbg_kat_no_reseed kat14_nor_t = {
    13, kat14_nor_entropyin, kat14_nor_nonce, kat14_nor_persstr,
    kat14_nor_addin0, kat14_nor_addin1, kat14_nor_retbytes
};
static const struct drbg_kat kat14_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat14_nor_t
};

static const unsigned char kat15_nor_entropyin[] = {
    0xb7, 0x7b, 0xd2, 0x2e, 0xfe, 0xb7, 0x71, 0x58, 0x6d, 0x51, 0x6f, 0x58,
    0x21, 0x92, 0xa3, 0x11,
};
static const unsigned char kat15_nor_nonce[] = {
    0xa6, 0x99, 0xf4, 0x2a, 0x49, 0x81, 0xfe, 0xfc,
};
static const unsigned char kat15_nor_persstr[] = {0};
static const unsigned char kat15_nor_addin0[] = {0};
static const unsigned char kat15_nor_addin1[] = {0};
static const unsigned char kat15_nor_retbytes[] = {
    0x7f, 0xf0, 0xab, 0xea, 0xff, 0xc6, 0xec, 0x92, 0x3c, 0xeb, 0xd9, 0x10,
    0xf5, 0x93, 0x7b, 0xf1, 0x4f, 0xc5, 0x2d, 0x2a, 0x74, 0x25, 0x83, 0x88,
    0xc7, 0x6c, 0x1b, 0xc9, 0xe6, 0x35, 0xad, 0xf1, 0x75, 0x85, 0x8f, 0x0a,
    0x55, 0x87, 0x7e, 0x7e, 0x9f, 0x5f, 0x86, 0x7d, 0x00, 0xb9, 0xb1, 0x36,
    0x3d, 0xde, 0x46, 0x28, 0x8a, 0x6f, 0xff, 0x53, 0x21, 0xf4, 0x69, 0x44,
    0x5a, 0xad, 0x41, 0x98,
};
static const struct drbg_kat_no_reseed kat15_nor_t = {
    14, kat15_nor_entropyin, kat15_nor_nonce, kat15_nor_persstr,
    kat15_nor_addin0, kat15_nor_addin1, kat15_nor_retbytes
};
static const struct drbg_kat kat15_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat15_nor_t
};

static const unsigned char kat16_nor_entropyin[] = {
    0xb4, 0x08, 0xce, 0xfb, 0x5b, 0xc7, 0x15, 0x7d, 0x3f, 0x26, 0xcb, 0x95,
    0xa8, 0xb1, 0xd7, 0xac,
};
static const unsigned char kat16_nor_nonce[] = {
    0x02, 0x6c, 0x76, 0x8f, 0xd5, 0x77, 0xb9, 0x2a,
};
static const unsigned char kat16_nor_persstr[] = {0};
static const unsigned char kat16_nor_addin0[] = {
    0x57, 0x37, 0xef, 0x81, 0xde, 0xe3, 0x65, 0xb6, 0xda, 0xdb, 0x3f, 0xee,
    0xbf, 0x5d, 0x10, 0x84,
};
static const unsigned char kat16_nor_addin1[] = {
    0x33, 0x68, 0xa5, 0x16, 0xb3, 0x43, 0x1a, 0x3d, 0xaa, 0xa6, 0x0d, 0xc8,
    0x74, 0x3c, 0x82, 0x97,
};
static const unsigned char kat16_nor_retbytes[] = {
    0x4e, 0x90, 0x9e, 0xbb, 0x24, 0x14, 0x7a, 0x00, 0x04, 0x06, 0x3a, 0x5e,
    0x47, 0xee, 0x04, 0x4f, 0xea, 0xd6, 0x10, 0xd6, 0x23, 0x24, 0xbd, 0x0f,
    0x96, 0x3f, 0x75, 0x6f, 0xb9, 0x13, 0x61, 0xe8, 0xb8, 0x7e, 0x3a, 0x76,
    0xa3, 0x98, 0x14, 0x3f, 0xe8, 0x81, 0x30, 0xfe, 0x1b, 0x54, 0x7b, 0x66,
    0x1a, 0x64, 0x80, 0xc7, 0x11, 0xb7, 0x39, 0xf1, 0x8a, 0x9d, 0xf3, 0xae,
    0x51, 0xd4, 0x1b, 0xc9,
};
static const struct drbg_kat_no_reseed kat16_nor_t = {
    0, kat16_nor_entropyin, kat16_nor_nonce, kat16_nor_persstr,
    kat16_nor_addin0, kat16_nor_addin1, kat16_nor_retbytes
};
static const struct drbg_kat kat16_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat16_nor_t
};

static const unsigned char kat17_nor_entropyin[] = {
    0x71, 0xbd, 0xce, 0x35, 0x42, 0x7d, 0x20, 0xbf, 0x58, 0xcf, 0x17, 0x74,
    0xce, 0x72, 0xd8, 0x33,
};
static const unsigned char kat17_nor_nonce[] = {
    0x34, 0x50, 0x2d, 0x8f, 0x5b, 0x14, 0xc4, 0xdd,
};
static const unsigned char kat17_nor_persstr[] = {0};
static const unsigned char kat17_nor_addin0[] = {
    0x66, 0xef, 0x42, 0xd6, 0x9a, 0x8c, 0x3d, 0x6d, 0x4a, 0x9e, 0x95, 0xa6,
    0x91, 0x4d, 0x81, 0x56,
};
static const unsigned char kat17_nor_addin1[] = {
    0xe3, 0x18, 0x83, 0xd9, 0x4b, 0x5e, 0xc4, 0xcc, 0xaa, 0x61, 0x2f, 0xbb,
    0x4a, 0x55, 0xd1, 0xc6,
};
static const unsigned char kat17_nor_retbytes[] = {
    0x97, 0x33, 0xe8, 0x20, 0x12, 0xe2, 0x7b, 0xa1, 0x46, 0x8f, 0xf2, 0x34,
    0xb3, 0xc9, 0xb6, 0x6b, 0x20, 0xb2, 0x4f, 0xee, 0x27, 0xd8, 0x0b, 0x21,
    0x8c, 0xff, 0x63, 0x73, 0x69, 0x29, 0xfb, 0xf3, 0x85, 0xcd, 0x88, 0x8e,
    0x43, 0x2c, 0x71, 0x8b, 0xa2, 0x55, 0xd2, 0x0f, 0x1d, 0x7f, 0xe3, 0xe1,
    0x2a, 0xa3, 0xe9, 0x2c, 0x25, 0x89, 0xc7, 0x14, 0x52, 0x99, 0x56, 0xcc,
    0xc3, 0xdf, 0xb3, 0x81,
};
static const struct drbg_kat_no_reseed kat17_nor_t = {
    1, kat17_nor_entropyin, kat17_nor_nonce, kat17_nor_persstr,
    kat17_nor_addin0, kat17_nor_addin1, kat17_nor_retbytes
};
static const struct drbg_kat kat17_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat17_nor_t
};

static const unsigned char kat18_nor_entropyin[] = {
    0x9c, 0x0e, 0x4a, 0xea, 0xfc, 0x35, 0x7f, 0xf8, 0xe3, 0xf5, 0x40, 0xa4,
    0x55, 0x67, 0x8d, 0x7e,
};
static const unsigned char kat18_nor_nonce[] = {
    0xef, 0xd8, 0x3d, 0xc5, 0xc5, 0x31, 0xd1, 0x67,
};
static const unsigned char kat18_nor_persstr[] = {0};
static const unsigned char kat18_nor_addin0[] = {
    0x58, 0xa1, 0x18, 0x14, 0x08, 0x1c, 0x1c, 0x35, 0xaf, 0x89, 0x88, 0xf1,
    0x31, 0x40, 0xf6, 0xcc,
};
static const unsigned char kat18_nor_addin1[] = {
    0x19, 0x19, 0x22, 0x70, 0xd6, 0x1a, 0x43, 0x26, 0x4a, 0x30, 0xba, 0x9e,
    0xbc, 0x72, 0x8a, 0x5b,
};
static const unsigned char kat18_nor_retbytes[] = {
    0x1e, 0xff, 0x1c, 0xf6, 0xb5, 0x75, 0x31, 0x55, 0x90, 0x70, 0x38, 0x82,
    0x35, 0x9b, 0x6a, 0xdb, 0x3c, 0x21, 0xeb, 0xcf, 0xda, 0x5d, 0x3b, 0x5d,
    0x6e, 0x88, 0xac, 0xab, 0x6e, 0x87, 0x9e, 0x0a, 0x6f, 0x75, 0x37, 0x5c,
    0x02, 0xd1, 0xd3, 0xfb, 0xae, 0x94, 0x3c, 0x34, 0x43, 0xe3, 0x9c, 0x29,
    0x9b, 0x80, 0xe2, 0x41, 0xcd, 0x36, 0x5d, 0x4a, 0xf8, 0xcd, 0x1a, 0xd5,
    0xd9, 0xd5, 0x4c, 0x58,
};
static const struct drbg_kat_no_reseed kat18_nor_t = {
    2, kat18_nor_entropyin, kat18_nor_nonce, kat18_nor_persstr,
    kat18_nor_addin0, kat18_nor_addin1, kat18_nor_retbytes
};
static const struct drbg_kat kat18_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat18_nor_t
};

static const unsigned char kat19_nor_entropyin[] = {
    0x95, 0x07, 0x01, 0x9b, 0x1f, 0xe8, 0x79, 0x84, 0xbe, 0xc1, 0xd0, 0x9c,
    0xec, 0xcf, 0x36, 0x35,
};
static const unsigned char kat19_nor_nonce[] = {
    0x47, 0x58, 0x62, 0x74, 0xfa, 0x75, 0x3e, 0x0d,
};
static const unsigned char kat19_nor_persstr[] = {0};
static const unsigned char kat19_nor_addin0[] = {
    0xf1, 0x7d, 0xe4, 0xce, 0xef, 0x9f, 0x2f, 0x23, 0x42, 0x20, 0xcb, 0x71,
    0xd8, 0xce, 0xb7, 0x0d,
};
static const unsigned char kat19_nor_addin1[] = {
    0x99, 0xd4, 0x8d, 0x9c, 0xe2, 0xfe, 0x76, 0xdc, 0xb8, 0xd0, 0x5f, 0xa4,
    0xec, 0x07, 0x1a, 0xec,
};
static const unsigned char kat19_nor_retbytes[] = {
    0x25, 0xa0, 0xd1, 0x67, 0x8e, 0x25, 0xb1, 0x94, 0x39, 0xd8, 0x33, 0x4c,
    0xa9, 0x02, 0xdb, 0x63, 0x47, 0xa9, 0xa4, 0x2b, 0x86, 0xf3, 0x54, 0xb9,
    0xe0, 0xb3, 0xc4, 0xf4, 0xe8, 0xba, 0x9d, 0x19, 0x2a, 0x5d, 0x76, 0x5a,
    0xae, 0x5f, 0x4c, 0x5a, 0x74, 0xa0, 0x17, 0x7d, 0x3e, 0xb6, 0x13, 0x70,
    0x62, 0xe6, 0xd9, 0x8b, 0x65, 0x18, 0x22, 0x26, 0x5b, 0xfe, 0xfc, 0x92,
    0x9a, 0x39, 0x69, 0xdd,
};
static const struct drbg_kat_no_reseed kat19_nor_t = {
    3, kat19_nor_entropyin, kat19_nor_nonce, kat19_nor_persstr,
    kat19_nor_addin0, kat19_nor_addin1, kat19_nor_retbytes
};
static const struct drbg_kat kat19_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat19_nor_t
};

static const unsigned char kat20_nor_entropyin[] = {
    0x12, 0xe8, 0xb3, 0xda, 0x3a, 0x3b, 0x3e, 0x77, 0x3a, 0xfc, 0x23, 0x05,
    0xbd, 0xa8, 0x6d, 0xb6,
};
static const unsigned char kat20_nor_nonce[] = {
    0xc4, 0x9f, 0x26, 0x1a, 0x9a, 0xb9, 0xd9, 0xf8,
};
static const unsigned char kat20_nor_persstr[] = {0};
static const unsigned char kat20_nor_addin0[] = {
    0x86, 0xb6, 0x38, 0x52, 0x41, 0x96, 0x15, 0x9d, 0xea, 0x01, 0x76, 0xe5,
    0x46, 0x2a, 0xe6, 0x5c,
};
static const unsigned char kat20_nor_addin1[] = {
    0x5e, 0xad, 0xb7, 0x67, 0x72, 0xbe, 0x3c, 0xe8, 0x47, 0xaf, 0x6b, 0x77,
    0x94, 0xe0, 0x02, 0xe3,
};
static const unsigned char kat20_nor_retbytes[] = {
    0x1b, 0xbf, 0x85, 0xe0, 0x2c, 0x1f, 0xc0, 0x4b, 0xea, 0x5f, 0xc4, 0x7f,
    0x23, 0x6a, 0x07, 0x1c, 0x13, 0xf8, 0xf5, 0x2e, 0xb1, 0x13, 0xcb, 0x83,
    0xc4, 0x15, 0x2a, 0xc8, 0x32, 0x62, 0x7a, 0xbb, 0xa6, 0xec, 0xde, 0xc1,
    0x7a, 0x9f, 0xf0, 0x30, 0x5b, 0x7a, 0x49, 0x81, 0x79, 0x83, 0xfa, 0x5b,
    0x66, 0x51, 0x45, 0xe7, 0xba, 0x9a, 0x5d, 0xf5, 0x72, 0x26, 0xec, 0xcf,
    0xc1, 0x64, 0xe3, 0xbb,
};
static const struct drbg_kat_no_reseed kat20_nor_t = {
    4, kat20_nor_entropyin, kat20_nor_nonce, kat20_nor_persstr,
    kat20_nor_addin0, kat20_nor_addin1, kat20_nor_retbytes
};
static const struct drbg_kat kat20_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat20_nor_t
};

static const unsigned char kat21_nor_entropyin[] = {
    0xb6, 0x08, 0xb1, 0x49, 0xcf, 0x44, 0xfe, 0x06, 0x06, 0x13, 0xb1, 0x66,
    0x1d, 0xa0, 0xdf, 0x63,
};
static const unsigned char kat21_nor_nonce[] = {
    0x95, 0x17, 0xc9, 0x93, 0xf1, 0x50, 0x34, 0x77,
};
static const unsigned char kat21_nor_persstr[] = {0};
static const unsigned char kat21_nor_addin0[] = {
    0x55, 0x9c, 0xb8, 0x12, 0xc3, 0x75, 0x8b, 0x2b, 0x31, 0x9d, 0xd9, 0xf5,
    0x76, 0x8e, 0x8d, 0x3d,
};
static const unsigned char kat21_nor_addin1[] = {
    0xc9, 0xaf, 0x98, 0x7e, 0xfc, 0xe4, 0x70, 0x0f, 0xa9, 0xc8, 0xeb, 0xb9,
    0xfe, 0xcf, 0x00, 0xda,
};
static const unsigned char kat21_nor_retbytes[] = {
    0x5b, 0x48, 0x13, 0x4d, 0xc7, 0xe6, 0x58, 0x9f, 0x1b, 0x40, 0x2c, 0xf5,
    0xc4, 0xf6, 0xe3, 0xa3, 0xeb, 0xfd, 0xa8, 0x26, 0x72, 0x50, 0xe8, 0xe8,
    0x11, 0xec, 0x05, 0x98, 0x6f, 0xf2, 0x1c, 0x78, 0xa1, 0x69, 0x30, 0xa8,
    0xe0, 0x81, 0x03, 0x07, 0x76, 0x08, 0xc0, 0x6f, 0xa5, 0xe2, 0xa0, 0x48,
    0xe8, 0x5d, 0xd6, 0xf3, 0x95, 0xce, 0xe2, 0x01, 0xd3, 0xcd, 0xe6, 0x35,
    0xe1, 0x7e, 0x20, 0x3e,
};
static const struct drbg_kat_no_reseed kat21_nor_t = {
    5, kat21_nor_entropyin, kat21_nor_nonce, kat21_nor_persstr,
    kat21_nor_addin0, kat21_nor_addin1, kat21_nor_retbytes
};
static const struct drbg_kat kat21_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat21_nor_t
};

static const unsigned char kat22_nor_entropyin[] = {
    0xe9, 0x2f, 0xd0, 0x7a, 0x09, 0xf2, 0xbc, 0xcb, 0x43, 0x42, 0x4c, 0x57,
    0x57, 0x73, 0xb0, 0xf0,
};
static const unsigned char kat22_nor_nonce[] = {
    0xb4, 0xe3, 0xca, 0x71, 0xae, 0x15, 0xa0, 0x36,
};
static const unsigned char kat22_nor_persstr[] = {0};
static const unsigned char kat22_nor_addin0[] = {
    0xe1, 0xb4, 0xb0, 0x40, 0x13, 0xdc, 0x12, 0x18, 0x8c, 0x62, 0x3d, 0x10,
    0x11, 0xd4, 0x08, 0xf1,
};
static const unsigned char kat22_nor_addin1[] = {
    0x79, 0x56, 0xbf, 0x5b, 0x16, 0x58, 0x0b, 0x3e, 0x72, 0x0a, 0x81, 0x33,
    0x3e, 0xae, 0x24, 0x83,
};
static const unsigned char kat22_nor_retbytes[] = {
    0x21, 0xa4, 0x67, 0x22, 0x42, 0x46, 0x56, 0x37, 0x0c, 0x47, 0x5b, 0x69,
    0xa1, 0xfb, 0xc7, 0x0c, 0x8c, 0xbf, 0x44, 0x2d, 0xc2, 0xbc, 0x28, 0xe0,
    0xab, 0x34, 0xbb, 0x6f, 0xfe, 0xd0, 0x6d, 0x10, 0xf4, 0x90, 0xdf, 0x7e,
    0x78, 0x7c, 0x12, 0xb6, 0x52, 0xd2, 0x19, 0xa5, 0x77, 0x30, 0x6f, 0x04,
    0xa9, 0xc8, 0x78, 0xb6, 0xcf, 0xdf, 0xf4, 0x3e, 0x13, 0x55, 0xc4, 0xd1,
    0x10, 0xfb, 0x0f, 0xb2,
};
static const struct drbg_kat_no_reseed kat22_nor_t = {
    6, kat22_nor_entropyin, kat22_nor_nonce, kat22_nor_persstr,
    kat22_nor_addin0, kat22_nor_addin1, kat22_nor_retbytes
};
static const struct drbg_kat kat22_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat22_nor_t
};

static const unsigned char kat23_nor_entropyin[] = {
    0xde, 0x0b, 0x4e, 0xa1, 0xa4, 0x40, 0xce, 0x68, 0x0e, 0xd0, 0x04, 0xcf,
    0x64, 0xaf, 0xbe, 0x0f,
};
static const unsigned char kat23_nor_nonce[] = {
    0xab, 0xb3, 0x1a, 0x3a, 0xb1, 0x5e, 0xd4, 0x49,
};
static const unsigned char kat23_nor_persstr[] = {0};
static const unsigned char kat23_nor_addin0[] = {
    0x45, 0xe7, 0x10, 0x10, 0x0c, 0xad, 0x00, 0xf3, 0xd2, 0x93, 0xdb, 0x75,
    0x5e, 0xe1, 0x99, 0x0d,
};
static const unsigned char kat23_nor_addin1[] = {
    0xb9, 0x11, 0xc9, 0xa4, 0x38, 0x9e, 0x71, 0x67, 0x98, 0x2a, 0x51, 0x24,
    0x8e, 0xe2, 0x93, 0x7e,
};
static const unsigned char kat23_nor_retbytes[] = {
    0x01, 0xc3, 0xaf, 0x83, 0x55, 0x7c, 0x63, 0x2d, 0xd8, 0x22, 0x83, 0x3f,
    0x18, 0xe5, 0x48, 0xfb, 0x60, 0x36, 0xe9, 0xcc, 0xac, 0xc3, 0xb4, 0x33,
    0xc5, 0xa7, 0x00, 0x79, 0x39, 0x93, 0x30, 0x89, 0x16, 0x22, 0x0e, 0x5a,
    0x74, 0x16, 0x39, 0x86, 0x1c, 0x2f, 0x17, 0x71, 0x9b, 0x32, 0x51, 0x22,
    0x06, 0xe3, 0x6e, 0x94, 0x7b, 0x28, 0x60, 0x07, 0x3b, 0x11, 0x3c, 0xcd,
    0xa3, 0xfe, 0x31, 0x68,
};
static const struct drbg_kat_no_reseed kat23_nor_t = {
    7, kat23_nor_entropyin, kat23_nor_nonce, kat23_nor_persstr,
    kat23_nor_addin0, kat23_nor_addin1, kat23_nor_retbytes
};
static const struct drbg_kat kat23_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat23_nor_t
};

static const unsigned char kat24_nor_entropyin[] = {
    0x72, 0x27, 0xe1, 0x1c, 0xfc, 0x1a, 0x6b, 0x84, 0xb4, 0x0e, 0xca, 0x24,
    0xe7, 0x7f, 0x8c, 0x64,
};
static const unsigned char kat24_nor_nonce[] = {
    0xab, 0x7e, 0x9f, 0xc4, 0x5e, 0x58, 0x63, 0xfb,
};
static const unsigned char kat24_nor_persstr[] = {0};
static const unsigned char kat24_nor_addin0[] = {
    0xca, 0xeb, 0xb3, 0xcd, 0x32, 0xc9, 0xd3, 0xe1, 0xf5, 0xd3, 0x8f, 0x8b,
    0xdd, 0x89, 0x58, 0x7d,
};
static const unsigned char kat24_nor_addin1[] = {
    0x31, 0x38, 0x09, 0x26, 0xb8, 0x60, 0x80, 0x06, 0xb7, 0xc7, 0x93, 0x61,
    0xf0, 0xf9, 0x1d, 0x5e,
};
static const unsigned char kat24_nor_retbytes[] = {
    0x3a, 0xd7, 0xb9, 0x32, 0x59, 0xc6, 0xd0, 0xfd, 0xb5, 0x14, 0x27, 0x7a,
    0xe9, 0x95, 0x01, 0x22, 0xfb, 0x97, 0x11, 0x33, 0x74, 0x2d, 0xe8, 0xe5,
    0x17, 0xdd, 0x22, 0x69, 0xdd, 0x5a, 0x3c, 0x25, 0x7e, 0x3d, 0x10, 0x6b,
    0x55, 0x56, 0x60, 0xcf, 0xdd, 0x6a, 0x92, 0xc2, 0xcd, 0xdf, 0x1e, 0x53,
    0xc4, 0x94, 0x5e, 0xd7, 0x72, 0xee, 0x30, 0x38, 0x04, 0xd0, 0x56, 0xf2,
    0x42, 0xbf, 0x53, 0x99,
};
static const struct drbg_kat_no_reseed kat24_nor_t = {
    8, kat24_nor_entropyin, kat24_nor_nonce, kat24_nor_persstr,
    kat24_nor_addin0, kat24_nor_addin1, kat24_nor_retbytes
};
static const struct drbg_kat kat24_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat24_nor_t
};

static const unsigned char kat25_nor_entropyin[] = {
    0x3b, 0x9e, 0xe2, 0x0c, 0x82, 0x42, 0xb3, 0xf3, 0x58, 0xdb, 0xe8, 0xea,
    0x58, 0x9e, 0x6e, 0x7c,
};
static const unsigned char kat25_nor_nonce[] = {
    0xb1, 0xdc, 0x3a, 0x47, 0x03, 0x68, 0x52, 0x15,
};
static const unsigned char kat25_nor_persstr[] = {0};
static const unsigned char kat25_nor_addin0[] = {
    0x47, 0x33, 0x34, 0x47, 0x86, 0x73, 0xe7, 0xb4, 0xa4, 0x3e, 0x9a, 0xe6,
    0xa8, 0xac, 0x98, 0xbf,
};
static const unsigned char kat25_nor_addin1[] = {
    0x06, 0x1f, 0xe4, 0x45, 0x5a, 0x17, 0x08, 0x74, 0x98, 0xfe, 0x20, 0x9c,
    0x2c, 0xba, 0x7a, 0x91,
};
static const unsigned char kat25_nor_retbytes[] = {
    0x8e, 0x50, 0x62, 0xb1, 0xfb, 0x29, 0x05, 0xc1, 0x57, 0xc5, 0x26, 0xfa,
    0x5f, 0x67, 0xcc, 0x99, 0xc6, 0xd7, 0x08, 0x0a, 0x3d, 0x31, 0x42, 0xfa,
    0x37, 0xc2, 0x09, 0xcf, 0x53, 0x4a, 0xf1, 0x6a, 0x9a, 0x2f, 0x28, 0x61,
    0xb8, 0x94, 0xea, 0x84, 0xa3, 0x73, 0xd4, 0xa8, 0x04, 0xc3, 0xdc, 0xd5,
    0x1f, 0xcc, 0x09, 0xdd, 0x0f, 0x5b, 0x39, 0x45, 0x69, 0x7a, 0xac, 0xd0,
    0xb8, 0xf3, 0xea, 0xfd,
};
static const struct drbg_kat_no_reseed kat25_nor_t = {
    9, kat25_nor_entropyin, kat25_nor_nonce, kat25_nor_persstr,
    kat25_nor_addin0, kat25_nor_addin1, kat25_nor_retbytes
};
static const struct drbg_kat kat25_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat25_nor_t
};

static const unsigned char kat26_nor_entropyin[] = {
    0x8f, 0xf7, 0x57, 0xb2, 0x11, 0x69, 0xb3, 0xfd, 0x6e, 0x88, 0x00, 0xaa,
    0xc0, 0xe3, 0xc4, 0x1f,
};
static const unsigned char kat26_nor_nonce[] = {
    0x49, 0x4d, 0x42, 0x51, 0xe8, 0x88, 0x15, 0xb4,
};
static const unsigned char kat26_nor_persstr[] = {0};
static const unsigned char kat26_nor_addin0[] = {
    0xe0, 0x19, 0x9f, 0xb9, 0xf0, 0xf0, 0xa1, 0xa8, 0xdd, 0x9d, 0x19, 0xd1,
    0x43, 0x68, 0xfa, 0x1e,
};
static const unsigned char kat26_nor_addin1[] = {
    0x21, 0x91, 0xab, 0xe7, 0x24, 0x41, 0x28, 0x56, 0x36, 0xc1, 0x1b, 0xde,
    0xe9, 0x83, 0xed, 0x8c,
};
static const unsigned char kat26_nor_retbytes[] = {
    0x06, 0x75, 0xb2, 0x25, 0x4b, 0x71, 0x31, 0x8e, 0xb3, 0xa3, 0x60, 0x4d,
    0x7b, 0x6d, 0x77, 0x80, 0x6c, 0xd7, 0xdf, 0x32, 0x5c, 0x7c, 0xe8, 0x9e,
    0x16, 0xd3, 0xbd, 0x56, 0x16, 0xb0, 0xfa, 0xec, 0x33, 0x0c, 0xfe, 0x6b,
    0x84, 0xb6, 0x87, 0x61, 0xad, 0xf7, 0x0c, 0xc1, 0x4e, 0x46, 0x25, 0x3e,
    0x11, 0x55, 0x72, 0xc1, 0xa4, 0xac, 0xbb, 0xae, 0xee, 0x1a, 0xd6, 0x81,
    0xb5, 0x38, 0xd6, 0xc9,
};
static const struct drbg_kat_no_reseed kat26_nor_t = {
    10, kat26_nor_entropyin, kat26_nor_nonce, kat26_nor_persstr,
    kat26_nor_addin0, kat26_nor_addin1, kat26_nor_retbytes
};
static const struct drbg_kat kat26_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat26_nor_t
};

static const unsigned char kat27_nor_entropyin[] = {
    0x46, 0x0c, 0x0a, 0x1e, 0x92, 0x9b, 0x75, 0x18, 0xdb, 0xcf, 0x44, 0xe4,
    0xe7, 0xee, 0x4d, 0x83,
};
static const unsigned char kat27_nor_nonce[] = {
    0x88, 0x65, 0xd5, 0x01, 0x53, 0x48, 0x25, 0x9e,
};
static const unsigned char kat27_nor_persstr[] = {0};
static const unsigned char kat27_nor_addin0[] = {
    0xf9, 0x32, 0x78, 0x91, 0x40, 0xa0, 0xa7, 0x6f, 0x07, 0xb2, 0x5b, 0x5f,
    0x8e, 0x1d, 0x00, 0xdc,
};
static const unsigned char kat27_nor_addin1[] = {
    0xe6, 0x44, 0x6a, 0xe2, 0xc6, 0x62, 0x66, 0x73, 0xe1, 0x8e, 0x37, 0x3d,
    0xeb, 0x12, 0xbd, 0x26,
};
static const unsigned char kat27_nor_retbytes[] = {
    0xc5, 0xde, 0x2e, 0x1a, 0x2a, 0xa7, 0x0a, 0x28, 0xac, 0xe6, 0x8e, 0x62,
    0x0c, 0x94, 0xb8, 0x88, 0xa0, 0x7b, 0x07, 0x80, 0xb3, 0x14, 0x38, 0xc5,
    0xe1, 0x1c, 0x5d, 0x76, 0xdf, 0xd6, 0x15, 0xf4, 0x1d, 0x4b, 0xa3, 0x24,
    0xd2, 0x2c, 0x12, 0xfe, 0xe3, 0x8f, 0x58, 0x93, 0x5d, 0xad, 0x35, 0xe0,
    0x0f, 0xe6, 0xa6, 0xe6, 0x11, 0x86, 0xc7, 0x88, 0x1c, 0x30, 0x6e, 0x96,
    0x31, 0xcc, 0x15, 0xec,
};
static const struct drbg_kat_no_reseed kat27_nor_t = {
    11, kat27_nor_entropyin, kat27_nor_nonce, kat27_nor_persstr,
    kat27_nor_addin0, kat27_nor_addin1, kat27_nor_retbytes
};
static const struct drbg_kat kat27_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat27_nor_t
};

static const unsigned char kat28_nor_entropyin[] = {
    0xb2, 0xda, 0x44, 0xbc, 0x3b, 0xe2, 0xd1, 0x7a, 0x9a, 0xc2, 0x08, 0x80,
    0x85, 0xe9, 0x26, 0xf8,
};
static const unsigned char kat28_nor_nonce[] = {
    0x04, 0xbe, 0x54, 0x49, 0x35, 0x26, 0xc5, 0x78,
};
static const unsigned char kat28_nor_persstr[] = {0};
static const unsigned char kat28_nor_addin0[] = {
    0x67, 0xbe, 0x89, 0x19, 0x59, 0x65, 0xcd, 0x7d, 0xc9, 0x84, 0x27, 0x1b,
    0x49, 0x7d, 0x61, 0x90,
};
static const unsigned char kat28_nor_addin1[] = {
    0xc9, 0x95, 0x6b, 0x00, 0x4b, 0x0c, 0x26, 0xb3, 0xf6, 0x57, 0xc6, 0xc1,
    0x55, 0xd3, 0x60, 0xcc,
};
static const unsigned char kat28_nor_retbytes[] = {
    0xc7, 0x74, 0x56, 0x62, 0x19, 0x48, 0xf6, 0x08, 0x38, 0xe3, 0x77, 0x9a,
    0x14, 0x60, 0x63, 0x22, 0x52, 0x96, 0xda, 0x38, 0x18, 0x0e, 0x2c, 0x4c,
    0x58, 0x35, 0xea, 0x27, 0xb2, 0x84, 0x24, 0xa8, 0x83, 0x1c, 0xee, 0x8c,
    0x8d, 0xf7, 0x39, 0xa2, 0xb1, 0x73, 0xb3, 0x7b, 0xe2, 0xc0, 0x2b, 0xff,
    0x38, 0xe0, 0xb3, 0x67, 0xd4, 0x82, 0xfb, 0x76, 0x54, 0x5f, 0xb3, 0xc9,
    0x8d, 0x82, 0xb6, 0xe7,
};
static const struct drbg_kat_no_reseed kat28_nor_t = {
    12, kat28_nor_entropyin, kat28_nor_nonce, kat28_nor_persstr,
    kat28_nor_addin0, kat28_nor_addin1, kat28_nor_retbytes
};
static const struct drbg_kat kat28_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat28_nor_t
};

static const unsigned char kat29_nor_entropyin[] = {
    0xf1, 0x77, 0xbc, 0x8a, 0xf7, 0x5c, 0x49, 0xe1, 0xd3, 0x79, 0xb5, 0x72,
    0x26, 0x92, 0xbb, 0x3f,
};
static const unsigned char kat29_nor_nonce[] = {
    0xd8, 0xb5, 0xfb, 0xdc, 0x5d, 0xa7, 0xea, 0x7b,
};
static const unsigned char kat29_nor_persstr[] = {0};
static const unsigned char kat29_nor_addin0[] = {
    0x37, 0x41, 0x94, 0x38, 0x35, 0x54, 0xa4, 0x21, 0x41, 0x0a, 0xf5, 0xde,
    0x2f, 0xc2, 0xec, 0x82,
};
static const unsigned char kat29_nor_addin1[] = {
    0x00, 0x0b, 0xe9, 0xac, 0x8e, 0x4b, 0x0c, 0xd1, 0x24, 0x5d, 0x7b, 0x7d,
    0xbf, 0x8d, 0x87, 0x4f,
};
static const unsigned char kat29_nor_retbytes[] = {
    0xb4, 0xa0, 0x0c, 0xd6, 0xe5, 0xf4, 0x7e, 0xa8, 0x7b, 0x12, 0x07, 0xc3,
    0xa2, 0x18, 0xd8, 0x98, 0x8d, 0x4c, 0xb7, 0xed, 0x52, 0xb1, 0x4b, 0x79,
    0xbb, 0xa5, 0x6a, 0xa7, 0xe5, 0x7e, 0x63, 0x64, 0x05, 0x70, 0x25, 0xab,
    0x8f, 0x60, 0xd9, 0x7b, 0x84, 0x71, 0x7b, 0xbf, 0x25, 0x58, 0xdd, 0x43,
    0x1f, 0x4b, 0xc3, 0x8c, 0xf1, 0xf6, 0x1e, 0xc0, 0xe7, 0xc2, 0xb4, 0xae,
    0xb5, 0x4b, 0x2f, 0xaa,
};
static const struct drbg_kat_no_reseed kat29_nor_t = {
    13, kat29_nor_entropyin, kat29_nor_nonce, kat29_nor_persstr,
    kat29_nor_addin0, kat29_nor_addin1, kat29_nor_retbytes
};
static const struct drbg_kat kat29_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat29_nor_t
};

static const unsigned char kat30_nor_entropyin[] = {
    0x52, 0x6b, 0xc8, 0xb7, 0x97, 0xad, 0x67, 0xeb, 0x9b, 0x11, 0xbb, 0x5d,
    0x17, 0x32, 0x00, 0x87,
};
static const unsigned char kat30_nor_nonce[] = {
    0x64, 0x7b, 0xd2, 0x32, 0xad, 0x4d, 0x13, 0x00,
};
static const unsigned char kat30_nor_persstr[] = {0};
static const unsigned char kat30_nor_addin0[] = {
    0x58, 0xc8, 0xbf, 0x95, 0x0f, 0xab, 0x02, 0x71, 0x19, 0xee, 0x4a, 0x59,
    0x27, 0x27, 0xaa, 0x02,
};
static const unsigned char kat30_nor_addin1[] = {
    0x25, 0x39, 0xe1, 0x23, 0x92, 0x6b, 0x74, 0xab, 0x23, 0x41, 0x4d, 0x63,
    0x50, 0xb8, 0x92, 0x87,
};
static const unsigned char kat30_nor_retbytes[] = {
    0x35, 0x9c, 0x56, 0x08, 0x34, 0xbb, 0xeb, 0xdb, 0x41, 0x34, 0xcc, 0x3e,
    0x66, 0xbc, 0x06, 0x95, 0xce, 0xfd, 0xb4, 0xfe, 0x56, 0x63, 0x75, 0x2b,
    0x6e, 0xba, 0x97, 0x4c, 0x4b, 0x85, 0x92, 0x4a, 0x01, 0xdc, 0xd4, 0xea,
    0x3e, 0xd7, 0x7f, 0x2a, 0xf3, 0x95, 0x7b, 0x87, 0x51, 0xf7, 0x37, 0x15,
    0x74, 0xbb, 0x8b, 0x09, 0x0b, 0xc0, 0xd5, 0x12, 0xc1, 0x57, 0xc6, 0x55,
    0x9c, 0x16, 0xd0, 0xfd,
};
static const struct drbg_kat_no_reseed kat30_nor_t = {
    14, kat30_nor_entropyin, kat30_nor_nonce, kat30_nor_persstr,
    kat30_nor_addin0, kat30_nor_addin1, kat30_nor_retbytes
};
static const struct drbg_kat kat30_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat30_nor_t
};

static const unsigned char kat31_nor_entropyin[] = {
    0xe1, 0x0b, 0xc2, 0x8a, 0x0b, 0xfd, 0xdf, 0xe9, 0x3e, 0x7f, 0x51, 0x86,
    0xe0, 0xca, 0x0b, 0x3b,
};
static const unsigned char kat31_nor_nonce[] = {
    0x9f, 0xf4, 0x77, 0xc1, 0x86, 0x73, 0x84, 0x0d,
};
static const unsigned char kat31_nor_persstr[] = {
    0xc9, 0x80, 0xde, 0xdf, 0x98, 0x82, 0xed, 0x44, 0x64, 0xa6, 0x74, 0x96,
    0x78, 0x68, 0xf1, 0x43,
};
static const unsigned char kat31_nor_addin0[] = {0};
static const unsigned char kat31_nor_addin1[] = {0};
static const unsigned char kat31_nor_retbytes[] = {
    0x35, 0xb0, 0x0d, 0xf6, 0x26, 0x9b, 0x66, 0x41, 0xfd, 0x4c, 0xcb, 0x35,
    0x4d, 0x56, 0xd8, 0x51, 0xde, 0x7a, 0x77, 0x52, 0x7e, 0x03, 0x4d, 0x60,
    0xc9, 0xe1, 0xa9, 0xe1, 0x52, 0x5a, 0x30, 0xed, 0x36, 0x1f, 0xde, 0xd8,
    0x9d, 0x3d, 0xcc, 0xb9, 0x78, 0xd4, 0xe7, 0xa9, 0xe1, 0x00, 0xeb, 0xf6,
    0x30, 0x62, 0x73, 0x5b, 0x52, 0x83, 0x1c, 0x6f, 0x0a, 0x1d, 0x3e, 0x1b,
    0xdc, 0x5e, 0xbc, 0x72,
};
static const struct drbg_kat_no_reseed kat31_nor_t = {
    0, kat31_nor_entropyin, kat31_nor_nonce, kat31_nor_persstr,
    kat31_nor_addin0, kat31_nor_addin1, kat31_nor_retbytes
};
static const struct drbg_kat kat31_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat31_nor_t
};

static const unsigned char kat32_nor_entropyin[] = {
    0xca, 0x4b, 0x1e, 0xfa, 0x75, 0xbd, 0x69, 0x36, 0x38, 0x73, 0xb8, 0xf9,
    0xdb, 0x4d, 0x35, 0x0e,
};
static const unsigned char kat32_nor_nonce[] = {
    0x47, 0xbf, 0x6c, 0x37, 0x72, 0xfd, 0xf7, 0xa9,
};
static const unsigned char kat32_nor_persstr[] = {
    0xeb, 0xaa, 0x60, 0x2c, 0x4d, 0xbe, 0x33, 0xff, 0x1b, 0xef, 0xbf, 0x0a,
    0x0b, 0xc6, 0x97, 0x54,
};
static const unsigned char kat32_nor_addin0[] = {0};
static const unsigned char kat32_nor_addin1[] = {0};
static const unsigned char kat32_nor_retbytes[] = {
    0x59, 0xc3, 0x19, 0x79, 0x1b, 0xb1, 0xf3, 0x0e, 0xe9, 0x34, 0xae, 0x6e,
    0x8b, 0x1f, 0xad, 0x1f, 0x74, 0xca, 0x25, 0x45, 0x68, 0xb8, 0x7f, 0x75,
    0x12, 0xf8, 0xf2, 0xab, 0x4c, 0x23, 0x01, 0x03, 0x05, 0xe1, 0x70, 0xee,
    0x75, 0xd8, 0xcb, 0xeb, 0x23, 0x4c, 0x7a, 0x23, 0x6e, 0x12, 0x27, 0xdb,
    0x6f, 0x7a, 0xac, 0x3c, 0x44, 0xb7, 0x87, 0x4b, 0x65, 0x56, 0x74, 0x45,
    0x34, 0x30, 0x0c, 0x3d,
};
static const struct drbg_kat_no_reseed kat32_nor_t = {
    1, kat32_nor_entropyin, kat32_nor_nonce, kat32_nor_persstr,
    kat32_nor_addin0, kat32_nor_addin1, kat32_nor_retbytes
};
static const struct drbg_kat kat32_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat32_nor_t
};

static const unsigned char kat33_nor_entropyin[] = {
    0x35, 0x19, 0x49, 0x15, 0x74, 0x71, 0x9d, 0xdb, 0x1f, 0x51, 0xb5, 0xf5,
    0x9e, 0x21, 0xad, 0x3d,
};
static const unsigned char kat33_nor_nonce[] = {
    0xef, 0x1d, 0xbe, 0xae, 0x79, 0xf0, 0x89, 0x8b,
};
static const unsigned char kat33_nor_persstr[] = {
    0x7f, 0xdc, 0x21, 0xe3, 0x53, 0x24, 0x9e, 0x93, 0xdf, 0x98, 0xf2, 0x91,
    0x02, 0xb9, 0xac, 0xec,
};
static const unsigned char kat33_nor_addin0[] = {0};
static const unsigned char kat33_nor_addin1[] = {0};
static const unsigned char kat33_nor_retbytes[] = {
    0x19, 0x2c, 0x71, 0x85, 0x74, 0x47, 0xaf, 0x82, 0x01, 0x37, 0x06, 0xb8,
    0xeb, 0x00, 0x8f, 0x8e, 0x2c, 0x2e, 0xb3, 0x99, 0x4a, 0x23, 0x8c, 0xf7,
    0x39, 0xe1, 0x74, 0x60, 0xfe, 0x84, 0xef, 0x71, 0xca, 0x43, 0x7e, 0x8a,
    0xcb, 0xb8, 0xd9, 0xd2, 0x88, 0x07, 0xc4, 0x7a, 0x5f, 0x03, 0x44, 0x42,
    0xaa, 0xf3, 0xc2, 0xf3, 0xe7, 0xde, 0xbe, 0x53, 0x1b, 0xc3, 0x59, 0x60,
    0x56, 0x68, 0x58, 0x48,
};
static const struct drbg_kat_no_reseed kat33_nor_t = {
    2, kat33_nor_entropyin, kat33_nor_nonce, kat33_nor_persstr,
    kat33_nor_addin0, kat33_nor_addin1, kat33_nor_retbytes
};
static const struct drbg_kat kat33_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat33_nor_t
};

static const unsigned char kat34_nor_entropyin[] = {
    0x68, 0xed, 0xc1, 0xc7, 0xf0, 0x4f, 0xa2, 0x2f, 0x75, 0x1b, 0x91, 0x8b,
    0x01, 0x6b, 0xbf, 0xb5,
};
static const unsigned char kat34_nor_nonce[] = {
    0xa9, 0xb6, 0x7a, 0x39, 0x58, 0xf2, 0x02, 0xf0,
};
static const unsigned char kat34_nor_persstr[] = {
    0x9d, 0x6f, 0xbb, 0x02, 0x0e, 0x3d, 0x99, 0xd1, 0x95, 0xff, 0x46, 0x9f,
    0x0a, 0xc0, 0xb8, 0xca,
};
static const unsigned char kat34_nor_addin0[] = {0};
static const unsigned char kat34_nor_addin1[] = {0};
static const unsigned char kat34_nor_retbytes[] = {
    0xd9, 0x02, 0x89, 0xb1, 0xb6, 0x76, 0x3a, 0x76, 0x90, 0xfc, 0xb5, 0xd6,
    0x7c, 0x81, 0xc2, 0x03, 0xa1, 0xcf, 0xb0, 0x88, 0x71, 0xb9, 0x4a, 0xf6,
    0x51, 0xc4, 0xcd, 0x7e, 0xc2, 0xfa, 0x4b, 0x9d, 0x47, 0xc8, 0xe5, 0xfc,
    0xc9, 0x9a, 0xff, 0xc2, 0x21, 0x43, 0xb6, 0x12, 0x70, 0x18, 0xe9, 0xe2,
    0xc0, 0x91, 0x4f, 0x9d, 0x47, 0xa5, 0xa3, 0x0b, 0x26, 0xe7, 0x4d, 0xdc,
    0x31, 0xa4, 0x48, 0x42,
};
static const struct drbg_kat_no_reseed kat34_nor_t = {
    3, kat34_nor_entropyin, kat34_nor_nonce, kat34_nor_persstr,
    kat34_nor_addin0, kat34_nor_addin1, kat34_nor_retbytes
};
static const struct drbg_kat kat34_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat34_nor_t
};

static const unsigned char kat35_nor_entropyin[] = {
    0x8e, 0x6b, 0xaf, 0x81, 0xa3, 0xf9, 0xe7, 0x32, 0xef, 0xb9, 0x65, 0xaf,
    0xa5, 0x9e, 0x71, 0xee,
};
static const unsigned char kat35_nor_nonce[] = {
    0x9a, 0x40, 0x36, 0xfe, 0x7f, 0xf2, 0x25, 0xb8,
};
static const unsigned char kat35_nor_persstr[] = {
    0xc5, 0x7e, 0x42, 0xad, 0x7a, 0xef, 0x53, 0x77, 0xfc, 0x4c, 0xd6, 0x20,
    0xd6, 0x31, 0xb0, 0x61,
};
static const unsigned char kat35_nor_addin0[] = {0};
static const unsigned char kat35_nor_addin1[] = {0};
static const unsigned char kat35_nor_retbytes[] = {
    0x71, 0x39, 0xa2, 0xd6, 0x74, 0xee, 0xfe, 0x54, 0x08, 0x7a, 0x2d, 0xd5,
    0xf4, 0x97, 0xf5, 0xa1, 0xcb, 0x44, 0x44, 0x72, 0xe3, 0x64, 0x04, 0x5c,
    0xb9, 0x20, 0x2b, 0xce, 0x24, 0x58, 0x1b, 0x4b, 0x1b, 0x80, 0x3f, 0xa5,
    0xf4, 0xa9, 0xac, 0x6d, 0x66, 0x37, 0xbd, 0xd9, 0x06, 0xaf, 0x3d, 0x8f,
    0x49, 0xb5, 0xb8, 0x0f, 0x44, 0xcd, 0x16, 0xee, 0x5b, 0x1a, 0xe3, 0xbf,
    0x51, 0xb5, 0x98, 0xba,
};
static const struct drbg_kat_no_reseed kat35_nor_t = {
    4, kat35_nor_entropyin, kat35_nor_nonce, kat35_nor_persstr,
    kat35_nor_addin0, kat35_nor_addin1, kat35_nor_retbytes
};
static const struct drbg_kat kat35_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat35_nor_t
};

static const unsigned char kat36_nor_entropyin[] = {
    0xf8, 0x99, 0x2f, 0x34, 0x0f, 0x90, 0xc8, 0xd8, 0x16, 0xb4, 0xbd, 0x8b,
    0x43, 0x5d, 0xa0, 0x35,
};
static const unsigned char kat36_nor_nonce[] = {
    0xfc, 0xf9, 0x44, 0x00, 0x6f, 0x29, 0x27, 0x98,
};
static const unsigned char kat36_nor_persstr[] = {
    0x56, 0x32, 0x23, 0x5e, 0x85, 0x2c, 0x07, 0x40, 0x09, 0xb4, 0x97, 0xed,
    0xa4, 0x38, 0xa1, 0x6f,
};
static const unsigned char kat36_nor_addin0[] = {0};
static const unsigned char kat36_nor_addin1[] = {0};
static const unsigned char kat36_nor_retbytes[] = {
    0x9e, 0xab, 0x55, 0xfa, 0x4b, 0x2c, 0x1c, 0xfa, 0xc6, 0x3d, 0x4d, 0xe4,
    0x7a, 0xa8, 0xf5, 0x9f, 0x78, 0x5e, 0xe4, 0x90, 0xda, 0x44, 0xce, 0xe3,
    0x0f, 0x91, 0x61, 0xe9, 0x3a, 0xfc, 0x46, 0x1d, 0xad, 0x0e, 0x7f, 0x4a,
    0x06, 0x4e, 0x92, 0xef, 0xd6, 0x00, 0x89, 0x16, 0xc2, 0x2f, 0x83, 0x64,
    0xc1, 0x90, 0x46, 0x7a, 0xde, 0x12, 0xab, 0x4e, 0x18, 0xb1, 0xf3, 0x9c,
    0xe1, 0x8b, 0x35, 0x8a,
};
static const struct drbg_kat_no_reseed kat36_nor_t = {
    5, kat36_nor_entropyin, kat36_nor_nonce, kat36_nor_persstr,
    kat36_nor_addin0, kat36_nor_addin1, kat36_nor_retbytes
};
static const struct drbg_kat kat36_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat36_nor_t
};

static const unsigned char kat37_nor_entropyin[] = {
    0x07, 0x43, 0x8f, 0x83, 0x53, 0xb7, 0xb0, 0x67, 0x42, 0x18, 0x31, 0x9e,
    0x29, 0x54, 0x0b, 0xca,
};
static const unsigned char kat37_nor_nonce[] = {
    0x41, 0x14, 0x77, 0x63, 0x6a, 0x50, 0x0c, 0x62,
};
static const unsigned char kat37_nor_persstr[] = {
    0xcf, 0x36, 0xf2, 0xaa, 0x72, 0xf2, 0xf3, 0x5e, 0x33, 0x5e, 0x65, 0xcf,
    0xd1, 0x7e, 0x2d, 0x3d,
};
static const unsigned char kat37_nor_addin0[] = {0};
static const unsigned char kat37_nor_addin1[] = {0};
static const unsigned char kat37_nor_retbytes[] = {
    0x6c, 0xfa, 0xcf, 0xe4, 0x7a, 0xcf, 0x8f, 0x61, 0xd8, 0xaf, 0x3e, 0xd4,
    0xfb, 0xac, 0xef, 0x5c, 0xd7, 0x41, 0xac, 0x2f, 0x16, 0x5e, 0x15, 0xbd,
    0xaa, 0xb1, 0xd0, 0x30, 0xc6, 0x85, 0x67, 0x83, 0x7e, 0xf3, 0xda, 0x1e,
    0xc8, 0x3d, 0xa0, 0x42, 0x3c, 0xa3, 0x42, 0x4c, 0xf2, 0x93, 0xf6, 0x1f,
    0x71, 0x75, 0xfb, 0xa8, 0xe9, 0x1e, 0x5f, 0xdc, 0x8b, 0x39, 0x55, 0x77,
    0x73, 0x61, 0xd6, 0x90,
};
static const struct drbg_kat_no_reseed kat37_nor_t = {
    6, kat37_nor_entropyin, kat37_nor_nonce, kat37_nor_persstr,
    kat37_nor_addin0, kat37_nor_addin1, kat37_nor_retbytes
};
static const struct drbg_kat kat37_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat37_nor_t
};

static const unsigned char kat38_nor_entropyin[] = {
    0x8e, 0x93, 0x73, 0x5c, 0x34, 0xae, 0x0c, 0x5e, 0xa5, 0x21, 0x3d, 0x83,
    0xa8, 0xda, 0x80, 0x45,
};
static const unsigned char kat38_nor_nonce[] = {
    0x2a, 0x0b, 0x27, 0x9a, 0x9d, 0x67, 0x78, 0x38,
};
static const unsigned char kat38_nor_persstr[] = {
    0x14, 0xf4, 0x52, 0x66, 0x52, 0x4a, 0x81, 0x52, 0x85, 0x7a, 0x83, 0x31,
    0x6a, 0x17, 0x75, 0x85,
};
static const unsigned char kat38_nor_addin0[] = {0};
static const unsigned char kat38_nor_addin1[] = {0};
static const unsigned char kat38_nor_retbytes[] = {
    0xe0, 0x0c, 0xb8, 0x63, 0x6f, 0x96, 0x02, 0xc3, 0x30, 0x21, 0xa8, 0x2a,
    0x7f, 0x9d, 0xc9, 0xc7, 0x4e, 0xd3, 0x43, 0x5b, 0x34, 0x89, 0x7d, 0x6e,
    0xfd, 0x8c, 0x05, 0xf1, 0x77, 0xe8, 0x55, 0x6a, 0x44, 0x3b, 0xd1, 0x8c,
    0x20, 0xa5, 0x57, 0xb9, 0xbd, 0xc4, 0x17, 0xe8, 0x99, 0x85, 0x06, 0x48,
    0x50, 0x91, 0xd6, 0xbc, 0x91, 0x8e, 0x9c, 0xbf, 0xfb, 0xb8, 0x5c, 0x62,
    0x0f, 0x74, 0xc6, 0xec,
};
static const struct drbg_kat_no_reseed kat38_nor_t = {
    7, kat38_nor_entropyin, kat38_nor_nonce, kat38_nor_persstr,
    kat38_nor_addin0, kat38_nor_addin1, kat38_nor_retbytes
};
static const struct drbg_kat kat38_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat38_nor_t
};

static const unsigned char kat39_nor_entropyin[] = {
    0xec, 0x6b, 0x87, 0xe9, 0xe0, 0x3f, 0x44, 0xf9, 0x19, 0xe4, 0x0e, 0x1d,
    0xac, 0x02, 0xf8, 0x73,
};
static const unsigned char kat39_nor_nonce[] = {
    0x5e, 0x9a, 0xc5, 0x09, 0x0a, 0x5e, 0x28, 0x97,
};
static const unsigned char kat39_nor_persstr[] = {
    0x0d, 0x9b, 0xa4, 0x62, 0xbf, 0x16, 0x6f, 0x3b, 0x36, 0x6a, 0x2a, 0xdf,
    0x55, 0xd2, 0x76, 0x45,
};
static const unsigned char kat39_nor_addin0[] = {0};
static const unsigned char kat39_nor_addin1[] = {0};
static const unsigned char kat39_nor_retbytes[] = {
    0xec, 0x37, 0xac, 0xbd, 0xdf, 0xb0, 0x41, 0xe4, 0x24, 0x9c, 0x1c, 0xb9,
    0x83, 0x3a, 0x31, 0xfc, 0x6f, 0xbd, 0xaa, 0x66, 0x58, 0xe4, 0xb7, 0x48,
    0xc0, 0xeb, 0xf6, 0xab, 0x54, 0xa3, 0xb9, 0xc0, 0xf6, 0x2d, 0x7c, 0x89,
    0xdb, 0xb2, 0x1d, 0x1d, 0xe1, 0x33, 0x15, 0xcb, 0xae, 0x2f, 0xf4, 0xf5,
    0x46, 0x24, 0x91, 0x32, 0x1f, 0xbe, 0x04, 0xb4, 0x14, 0xfb, 0xa2, 0x80,
    0x7e, 0xd9, 0x14, 0xee,
};
static const struct drbg_kat_no_reseed kat39_nor_t = {
    8, kat39_nor_entropyin, kat39_nor_nonce, kat39_nor_persstr,
    kat39_nor_addin0, kat39_nor_addin1, kat39_nor_retbytes
};
static const struct drbg_kat kat39_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat39_nor_t
};

static const unsigned char kat40_nor_entropyin[] = {
    0x9d, 0x28, 0x2f, 0x5a, 0x0c, 0x15, 0xa6, 0x53, 0x04, 0x73, 0x10, 0xb5,
    0x62, 0x7d, 0x8f, 0xa2,
};
static const unsigned char kat40_nor_nonce[] = {
    0x55, 0x22, 0xc4, 0x2f, 0x4b, 0xb0, 0x76, 0xdc,
};
static const unsigned char kat40_nor_persstr[] = {
    0x84, 0x6b, 0x2e, 0x17, 0xb0, 0x62, 0xc4, 0xc7, 0xc5, 0x61, 0x94, 0x3b,
    0x5d, 0x97, 0x56, 0x5d,
};
static const unsigned char kat40_nor_addin0[] = {0};
static const unsigned char kat40_nor_addin1[] = {0};
static const unsigned char kat40_nor_retbytes[] = {
    0x45, 0xf6, 0x0e, 0x1b, 0xa9, 0x81, 0xaf, 0x7f, 0x52, 0xef, 0x49, 0x39,
    0xc0, 0xbb, 0x0b, 0xdb, 0x6f, 0xe4, 0x6f, 0x37, 0x2c, 0xc6, 0x48, 0x06,
    0x4b, 0xbd, 0x3a, 0x0a, 0x2b, 0x85, 0xc0, 0x23, 0x91, 0xb2, 0x97, 0x92,
    0xcc, 0x88, 0x05, 0xbb, 0x5d, 0x45, 0x3e, 0xe2, 0x90, 0xcd, 0x1b, 0x9c,
    0x9f, 0x8f, 0x20, 0xb0, 0x91, 0x16, 0xaf, 0x1f, 0xd5, 0xec, 0xea, 0xe8,
    0x0d, 0x03, 0x58, 0xd8,
};
static const struct drbg_kat_no_reseed kat40_nor_t = {
    9, kat40_nor_entropyin, kat40_nor_nonce, kat40_nor_persstr,
    kat40_nor_addin0, kat40_nor_addin1, kat40_nor_retbytes
};
static const struct drbg_kat kat40_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat40_nor_t
};

static const unsigned char kat41_nor_entropyin[] = {
    0x83, 0x82, 0x7c, 0x08, 0x52, 0x4b, 0x54, 0xe8, 0x0f, 0x60, 0xe1, 0x86,
    0xcf, 0xce, 0x36, 0xdb,
};
static const unsigned char kat41_nor_nonce[] = {
    0x17, 0x7e, 0x5c, 0x00, 0xf7, 0x85, 0xa6, 0xf9,
};
static const unsigned char kat41_nor_persstr[] = {
    0x71, 0x75, 0xa2, 0xc2, 0x2e, 0xab, 0x4d, 0x0b, 0x1a, 0x65, 0xfd, 0xad,
    0xed, 0x35, 0x6a, 0xeb,
};
static const unsigned char kat41_nor_addin0[] = {0};
static const unsigned char kat41_nor_addin1[] = {0};
static const unsigned char kat41_nor_retbytes[] = {
    0x84, 0x24, 0xe7, 0x6c, 0x73, 0x6c, 0x03, 0x09, 0x42, 0x3c, 0xf4, 0x8d,
    0xc1, 0x05, 0xfe, 0xef, 0x22, 0x8d, 0xd4, 0x7b, 0xfd, 0x8d, 0x33, 0x80,
    0x19, 0x50, 0xd2, 0x10, 0x2d, 0x5d, 0xba, 0xfb, 0x51, 0xe1, 0xa8, 0x5b,
    0x4a, 0x97, 0x1e, 0x4f, 0x0b, 0x9d, 0x52, 0x65, 0x69, 0x73, 0xdb, 0x45,
    0x5b, 0x28, 0x6f, 0x58, 0x8c, 0xa6, 0x1d, 0x15, 0x99, 0xc4, 0xec, 0x60,
    0xdd, 0x80, 0xbe, 0x98,
};
static const struct drbg_kat_no_reseed kat41_nor_t = {
    10, kat41_nor_entropyin, kat41_nor_nonce, kat41_nor_persstr,
    kat41_nor_addin0, kat41_nor_addin1, kat41_nor_retbytes
};
static const struct drbg_kat kat41_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat41_nor_t
};

static const unsigned char kat42_nor_entropyin[] = {
    0x9f, 0x32, 0x13, 0xa2, 0xd6, 0xfe, 0x5f, 0x3d, 0xfa, 0xf0, 0xe4, 0xed,
    0x29, 0x2e, 0x0f, 0x2a,
};
static const unsigned char kat42_nor_nonce[] = {
    0xf9, 0xc0, 0x55, 0x3b, 0x18, 0xb2, 0x1c, 0xc3,
};
static const unsigned char kat42_nor_persstr[] = {
    0x21, 0x73, 0xd3, 0x9b, 0x37, 0x55, 0x04, 0x19, 0x11, 0x98, 0x9f, 0x89,
    0xf1, 0xe9, 0x5d, 0x3e,
};
static const unsigned char kat42_nor_addin0[] = {0};
static const unsigned char kat42_nor_addin1[] = {0};
static const unsigned char kat42_nor_retbytes[] = {
    0x10, 0x45, 0xf7, 0xb5, 0x36, 0xda, 0xb4, 0xde, 0x01, 0x54, 0x48, 0xac,
    0x47, 0x66, 0x18, 0x8a, 0x36, 0x7b, 0xb6, 0x93, 0x97, 0x91, 0x3d, 0xfd,
    0x3d, 0x5b, 0xbf, 0x83, 0xf1, 0xbb, 0xac, 0x73, 0x23, 0x2d, 0x47, 0x64,
    0x4e, 0x5a, 0x9c, 0x16, 0x99, 0x60, 0x58, 0x1e, 0x4a, 0xd5, 0xed, 0x26,
    0x95, 0x36, 0x0b, 0x94, 0xf4, 0x20, 0x9c, 0x77, 0xd7, 0x69, 0x40, 0x16,
    0xb4, 0xb3, 0x97, 0xc2,
};
static const struct drbg_kat_no_reseed kat42_nor_t = {
    11, kat42_nor_entropyin, kat42_nor_nonce, kat42_nor_persstr,
    kat42_nor_addin0, kat42_nor_addin1, kat42_nor_retbytes
};
static const struct drbg_kat kat42_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat42_nor_t
};

static const unsigned char kat43_nor_entropyin[] = {
    0xfb, 0xc8, 0x19, 0x90, 0x74, 0xa7, 0x6f, 0x23, 0x68, 0x86, 0x36, 0xae,
    0x31, 0x14, 0x28, 0x1e,
};
static const unsigned char kat43_nor_nonce[] = {
    0xeb, 0xc2, 0x2e, 0x95, 0xdd, 0xae, 0xb6, 0xfc,
};
static const unsigned char kat43_nor_persstr[] = {
    0x71, 0xfb, 0xd5, 0x9f, 0x88, 0xfc, 0x99, 0xa9, 0xb6, 0x07, 0x00, 0x27,
    0xe7, 0x35, 0x41, 0xc2,
};
static const unsigned char kat43_nor_addin0[] = {0};
static const unsigned char kat43_nor_addin1[] = {0};
static const unsigned char kat43_nor_retbytes[] = {
    0x87, 0x25, 0x66, 0xa7, 0x00, 0x47, 0x98, 0xe0, 0x06, 0x97, 0xbd, 0xb3,
    0x3a, 0xf1, 0xee, 0x7a, 0x07, 0x6e, 0x9f, 0xd2, 0x89, 0xaf, 0x83, 0xaf,
    0xcb, 0x10, 0xa0, 0x25, 0x2b, 0xb5, 0xc1, 0x06, 0xc8, 0xdd, 0x20, 0xd7,
    0x7c, 0x85, 0x9f, 0x14, 0x01, 0x11, 0x8c, 0x24, 0xf5, 0xa8, 0x96, 0x8f,
    0x11, 0xfa, 0x9f, 0x77, 0x78, 0xc2, 0x80, 0x31, 0x89, 0xc8, 0xda, 0x87,
    0xe6, 0x49, 0x45, 0xc8,
};
static const struct drbg_kat_no_reseed kat43_nor_t = {
    12, kat43_nor_entropyin, kat43_nor_nonce, kat43_nor_persstr,
    kat43_nor_addin0, kat43_nor_addin1, kat43_nor_retbytes
};
static const struct drbg_kat kat43_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat43_nor_t
};

static const unsigned char kat44_nor_entropyin[] = {
    0x43, 0xf4, 0xd0, 0x89, 0x06, 0x0c, 0x90, 0x7f, 0x38, 0x76, 0x05, 0x1f,
    0xd5, 0xe3, 0x6f, 0x74,
};
static const unsigned char kat44_nor_nonce[] = {
    0x45, 0x71, 0xe6, 0x61, 0x5b, 0xa3, 0xbb, 0xce,
};
static const unsigned char kat44_nor_persstr[] = {
    0xfc, 0x90, 0x57, 0x74, 0x9b, 0xa4, 0x37, 0xd8, 0xfb, 0xe9, 0xf8, 0x1d,
    0x29, 0xdf, 0x5e, 0xf1,
};
static const unsigned char kat44_nor_addin0[] = {0};
static const unsigned char kat44_nor_addin1[] = {0};
static const unsigned char kat44_nor_retbytes[] = {
    0x90, 0x78, 0xf1, 0xbc, 0x92, 0x91, 0x0e, 0xbc, 0xf8, 0xe3, 0x17, 0x57,
    0xb6, 0x24, 0xbd, 0x23, 0xdb, 0xcd, 0x74, 0xf9, 0xca, 0x70, 0xff, 0x1c,
    0x6f, 0x2b, 0x21, 0xd8, 0x59, 0xa4, 0xe8, 0xa4, 0xe9, 0x96, 0x3e, 0xf1,
    0x32, 0xa0, 0x25, 0xb3, 0xae, 0x28, 0x5b, 0x43, 0xa6, 0x96, 0x98, 0x90,
    0x0d, 0x4f, 0x8a, 0x30, 0xbb, 0x5e, 0x99, 0x29, 0x0e, 0x45, 0x04, 0x16,
    0x19, 0x97, 0xa8, 0x37,
};
static const struct drbg_kat_no_reseed kat44_nor_t = {
    13, kat44_nor_entropyin, kat44_nor_nonce, kat44_nor_persstr,
    kat44_nor_addin0, kat44_nor_addin1, kat44_nor_retbytes
};
static const struct drbg_kat kat44_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat44_nor_t
};

static const unsigned char kat45_nor_entropyin[] = {
    0x0e, 0xe3, 0x8e, 0xa2, 0x9d, 0x71, 0x95, 0x00, 0x3e, 0x2b, 0x94, 0x2a,
    0xbf, 0x13, 0x48, 0xfc,
};
static const unsigned char kat45_nor_nonce[] = {
    0x80, 0xa7, 0x5c, 0x77, 0xce, 0xff, 0x54, 0xe4,
};
static const unsigned char kat45_nor_persstr[] = {
    0x83, 0x8e, 0xae, 0xbc, 0x98, 0x2c, 0xaa, 0x15, 0x62, 0xc3, 0x06, 0x4c,
    0xce, 0x42, 0xe5, 0x1e,
};
static const unsigned char kat45_nor_addin0[] = {0};
static const unsigned char kat45_nor_addin1[] = {0};
static const unsigned char kat45_nor_retbytes[] = {
    0x66, 0x29, 0x1b, 0x86, 0x85, 0xe9, 0x7c, 0x76, 0xe2, 0x21, 0x6d, 0x70,
    0x8b, 0x40, 0x70, 0x22, 0x68, 0x4b, 0x28, 0x29, 0x1f, 0x3d, 0xc7, 0x1b,
    0x5d, 0x60, 0xde, 0x14, 0x3f, 0x5f, 0xdf, 0x0b, 0xb0, 0x7a, 0xe9, 0xd7,
    0x52, 0x44, 0x56, 0xc4, 0xdf, 0xd0, 0x89, 0xea, 0x88, 0xde, 0x86, 0xdd,
    0xfa, 0x1d, 0x48, 0xca, 0x54, 0x2f, 0x00, 0xfa, 0x58, 0x6d, 0xa7, 0xa6,
    0x02, 0x7d, 0x83, 0x7b,
};
static const struct drbg_kat_no_reseed kat45_nor_t = {
    14, kat45_nor_entropyin, kat45_nor_nonce, kat45_nor_persstr,
    kat45_nor_addin0, kat45_nor_addin1, kat45_nor_retbytes
};
static const struct drbg_kat kat45_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat45_nor_t
};

static const unsigned char kat46_nor_entropyin[] = {
    0xca, 0xe4, 0x8d, 0xd8, 0x0d, 0x29, 0x81, 0x03, 0xef, 0x1e, 0xc0, 0xbf,
    0x1b, 0xb9, 0x62, 0x70,
};
static const unsigned char kat46_nor_nonce[] = {
    0xd8, 0x27, 0xf9, 0x16, 0x13, 0xe0, 0xb4, 0x7f,
};
static const unsigned char kat46_nor_persstr[] = {
    0xcc, 0x92, 0x8f, 0x3d, 0x2d, 0xf3, 0x1a, 0x29, 0xf4, 0xe4, 0x44, 0xf3,
    0xdf, 0x08, 0xbe, 0x21,
};
static const unsigned char kat46_nor_addin0[] = {
    0x7e, 0xaa, 0x1b, 0xbe, 0xc7, 0x93, 0x93, 0xa7, 0xf4, 0xa8, 0x22, 0x7b,
    0x69, 0x1e, 0xcb, 0x68,
};
static const unsigned char kat46_nor_addin1[] = {
    0x68, 0x69, 0xc6, 0xc7, 0xb9, 0xe6, 0x65, 0x3b, 0x39, 0x77, 0xf0, 0x78,
    0x9e, 0x94, 0x47, 0x8a,
};
static const unsigned char kat46_nor_retbytes[] = {
    0x92, 0x01, 0x32, 0xcd, 0x28, 0x46, 0x95, 0xb8, 0x68, 0xb5, 0xbc, 0x4b,
    0x70, 0x3a, 0xfe, 0xa4, 0xd9, 0x96, 0x62, 0x4a, 0x8f, 0x57, 0xe9, 0xfb,
    0xf5, 0xe7, 0x93, 0xb5, 0x09, 0xcb, 0x15, 0xb4, 0xbe, 0xaf, 0x70, 0x2d,
    0xac, 0x28, 0x71, 0x2d, 0x24, 0x9a, 0xe7, 0x50, 0x90, 0xa9, 0x1f, 0xd3,
    0x57, 0x75, 0x29, 0x4b, 0xf2, 0x4d, 0xde, 0xbf, 0xd2, 0x4e, 0x45, 0xd1,
    0x3f, 0x4a, 0x17, 0x48,
};
static const struct drbg_kat_no_reseed kat46_nor_t = {
    0, kat46_nor_entropyin, kat46_nor_nonce, kat46_nor_persstr,
    kat46_nor_addin0, kat46_nor_addin1, kat46_nor_retbytes
};
static const struct drbg_kat kat46_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat46_nor_t
};

static const unsigned char kat47_nor_entropyin[] = {
    0xc0, 0x70, 0x1f, 0x92, 0x50, 0x75, 0x8f, 0xcd, 0xf2, 0xbe, 0x73, 0x98,
    0x80, 0xdb, 0x66, 0xeb,
};
static const unsigned char kat47_nor_nonce[] = {
    0x14, 0x68, 0xb4, 0xa5, 0x87, 0x9c, 0x2d, 0xa6,
};
static const unsigned char kat47_nor_persstr[] = {
    0x80, 0x08, 0xae, 0xe8, 0xe9, 0x69, 0x40, 0xc5, 0x08, 0x73, 0xc7, 0x9f,
    0x8e, 0xcf, 0xe0, 0x02,
};
static const unsigned char kat47_nor_addin0[] = {
    0xf9, 0x01, 0xf8, 0x16, 0x7a, 0x1d, 0xff, 0xde, 0x8e, 0x3c, 0x83, 0xe2,
    0x44, 0x85, 0xe7, 0xfe,
};
static const unsigned char kat47_nor_addin1[] = {
    0x17, 0x1c, 0x09, 0x38, 0xc2, 0x38, 0x9f, 0x97, 0x87, 0x60, 0x55, 0xb4,
    0x82, 0x16, 0x62, 0x7f,
};
static const unsigned char kat47_nor_retbytes[] = {
    0x97, 0xc0, 0xc0, 0xe5, 0xa0, 0xcc, 0xf2, 0x4f, 0x33, 0x63, 0x48, 0x8a,
    0xdb, 0x13, 0x0a, 0x35, 0x89, 0xbf, 0x80, 0x65, 0x62, 0xee, 0x13, 0x95,
    0x7c, 0x33, 0xd3, 0x7d, 0xf4, 0x07, 0x77, 0x7a, 0x2b, 0x65, 0x0b, 0x5f,
    0x45, 0x5c, 0x13, 0xf1, 0x90, 0x77, 0x7f, 0xc5, 0x04, 0x3f, 0xcc, 0x1a,
    0x38, 0xf8, 0xcd, 0x1b, 0xbb, 0xd5, 0x57, 0xd1, 0x4a, 0x4c, 0x2e, 0x8a,
    0x2b, 0x49, 0x1e, 0x5c,
};
static const struct drbg_kat_no_reseed kat47_nor_t = {
    1, kat47_nor_entropyin, kat47_nor_nonce, kat47_nor_persstr,
    kat47_nor_addin0, kat47_nor_addin1, kat47_nor_retbytes
};
static const struct drbg_kat kat47_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat47_nor_t
};

static const unsigned char kat48_nor_entropyin[] = {
    0x6b, 0x98, 0x53, 0x2d, 0x67, 0x8f, 0xd2, 0x79, 0xcf, 0x73, 0x7c, 0x58,
    0x15, 0x0c, 0xe5, 0x89,
};
static const unsigned char kat48_nor_nonce[] = {
    0xca, 0x55, 0x45, 0x07, 0x3d, 0x54, 0x9d, 0x06,
};
static const unsigned char kat48_nor_persstr[] = {
    0x80, 0x90, 0x8a, 0xc1, 0x34, 0xe0, 0xcd, 0x23, 0xef, 0x7e, 0x31, 0x85,
    0x48, 0xf9, 0x21, 0x91,
};
static const unsigned char kat48_nor_addin0[] = {
    0x53, 0x79, 0x0d, 0xa9, 0x09, 0xbf, 0xde, 0xed, 0xaa, 0xfd, 0x99, 0x61,
    0x9f, 0x3f, 0x46, 0x1b,
};
static const unsigned char kat48_nor_addin1[] = {
    0xc7, 0xa5, 0x87, 0x77, 0xd1, 0xfb, 0xd1, 0x55, 0x3e, 0x95, 0x46, 0xc8,
    0x3e, 0x40, 0x9f, 0xc2,
};
static const unsigned char kat48_nor_retbytes[] = {
    0xd9, 0x84, 0xa2, 0x40, 0x58, 0x87, 0x39, 0x02, 0x27, 0x94, 0x09, 0x59,
    0xe4, 0x9f, 0x94, 0x4e, 0x49, 0x04, 0xc4, 0xf0, 0x46, 0x7f, 0xf2, 0x4d,
    0x87, 0x64, 0xdb, 0x91, 0x53, 0x0a, 0x1e, 0x59, 0x65, 0x68, 0x84, 0x72,
    0xfa, 0xc1, 0x31, 0xbe, 0xa0, 0x55, 0xc7, 0x15, 0x12, 0x27, 0xe2, 0xfc,
    0xb0, 0xda, 0xd9, 0x65, 0x5e, 0xc3, 0xac, 0x22, 0xcd, 0x7d, 0xff, 0x72,
    0xf6, 0xc7, 0x4a, 0xa0,
};
static const struct drbg_kat_no_reseed kat48_nor_t = {
    2, kat48_nor_entropyin, kat48_nor_nonce, kat48_nor_persstr,
    kat48_nor_addin0, kat48_nor_addin1, kat48_nor_retbytes
};
static const struct drbg_kat kat48_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat48_nor_t
};

static const unsigned char kat49_nor_entropyin[] = {
    0x97, 0x51, 0x6e, 0x48, 0xd9, 0x10, 0x37, 0x2a, 0x3f, 0x1f, 0xb4, 0xaa,
    0x7a, 0xe4, 0x93, 0x0f,
};
static const unsigned char kat49_nor_nonce[] = {
    0xbb, 0xa3, 0x0b, 0x3d, 0xeb, 0x19, 0x33, 0x89,
};
static const unsigned char kat49_nor_persstr[] = {
    0x8c, 0x0e, 0x3e, 0x71, 0xc8, 0x20, 0x1a, 0x77, 0x10, 0x55, 0x19, 0xdf,
    0xfa, 0x89, 0xc1, 0xaa,
};
static const unsigned char kat49_nor_addin0[] = {
    0x0a, 0x18, 0x78, 0x7e, 0x46, 0x50, 0xfa, 0x02, 0x49, 0xf7, 0x79, 0x62,
    0xac, 0x15, 0x8c, 0x63,
};
static const unsigned char kat49_nor_addin1[] = {
    0x2f, 0xb1, 0x92, 0x31, 0x4c, 0x3e, 0x07, 0xf9, 0x63, 0xfb, 0xa3, 0x4d,
    0x8d, 0xfc, 0x8c, 0x9c,
};
static const unsigned char kat49_nor_retbytes[] = {
    0xd7, 0xb7, 0x3e, 0x44, 0x9a, 0xed, 0x8f, 0x0a, 0x06, 0x93, 0x84, 0x83,
    0xe8, 0x78, 0x11, 0x04, 0x9b, 0x4a, 0xcf, 0xe5, 0x1c, 0xa5, 0x55, 0xf1,
    0xee, 0x97, 0x86, 0xe3, 0x3f, 0xce, 0x90, 0x69, 0x7d, 0xb6, 0x8d, 0x2d,
    0xc6, 0x36, 0x1c, 0x94, 0x2c, 0xf6, 0x7d, 0x89, 0x26, 0x19, 0x49, 0x7e,
    0x96, 0x19, 0x43, 0x32, 0xae, 0x32, 0xa6, 0x35, 0xcf, 0x35, 0x5b, 0xbc,
    0x3c, 0x66, 0x79, 0x55,
};
static const struct drbg_kat_no_reseed kat49_nor_t = {
    3, kat49_nor_entropyin, kat49_nor_nonce, kat49_nor_persstr,
    kat49_nor_addin0, kat49_nor_addin1, kat49_nor_retbytes
};
static const struct drbg_kat kat49_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat49_nor_t
};

static const unsigned char kat50_nor_entropyin[] = {
    0x62, 0xce, 0xf3, 0x5a, 0x51, 0x8e, 0xf6, 0x52, 0x0b, 0x6e, 0xdb, 0x1a,
    0xa9, 0xa2, 0x14, 0x96,
};
static const unsigned char kat50_nor_nonce[] = {
    0x2b, 0x98, 0xe2, 0x69, 0xf5, 0x1f, 0xb3, 0xba,
};
static const unsigned char kat50_nor_persstr[] = {
    0x89, 0xa6, 0xc9, 0x11, 0x27, 0x13, 0xd0, 0xd1, 0x96, 0xc8, 0xe0, 0x26,
    0x60, 0xa2, 0x51, 0x31,
};
static const unsigned char kat50_nor_addin0[] = {
    0x35, 0xed, 0xb8, 0x77, 0x01, 0xcb, 0x50, 0x01, 0x43, 0x4d, 0xb1, 0x85,
    0x7b, 0xf9, 0x8e, 0xfb,
};
static const unsigned char kat50_nor_addin1[] = {
    0xa9, 0xfa, 0x53, 0xb5, 0xd3, 0x03, 0x2e, 0x4b, 0xe6, 0x0d, 0xcd, 0xb1,
    0xe9, 0x4b, 0x4f, 0x50,
};
static const unsigned char kat50_nor_retbytes[] = {
    0xde, 0x8b, 0x77, 0x0b, 0x3e, 0xd4, 0x33, 0x15, 0xab, 0x7e, 0x83, 0x31,
    0xe2, 0xdf, 0x12, 0xcb, 0x9a, 0xbe, 0xb3, 0xf6, 0xb1, 0x4c, 0x45, 0x7a,
    0x30, 0x4b, 0xcd, 0x1d, 0xf7, 0x30, 0xdb, 0x52, 0xd4, 0xb4, 0x02, 0xf7,
    0x8b, 0x80, 0x54, 0x59, 0x65, 0xfc, 0x5a, 0x83, 0xdd, 0xf9, 0x99, 0x63,
    0xaa, 0xb5, 0xe1, 0xb3, 0x79, 0x24, 0x57, 0xdf, 0x4b, 0xf9, 0x27, 0xb6,
    0x6a, 0x4d, 0x67, 0x45,
};
static const struct drbg_kat_no_reseed kat50_nor_t = {
    4, kat50_nor_entropyin, kat50_nor_nonce, kat50_nor_persstr,
    kat50_nor_addin0, kat50_nor_addin1, kat50_nor_retbytes
};
static const struct drbg_kat kat50_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat50_nor_t
};

static const unsigned char kat51_nor_entropyin[] = {
    0xc0, 0x2a, 0xf7, 0x05, 0xf6, 0x99, 0xc3, 0xa7, 0xf5, 0xe2, 0xbb, 0x9b,
    0x14, 0xc7, 0xce, 0xa7,
};
static const unsigned char kat51_nor_nonce[] = {
    0xeb, 0x24, 0x6b, 0x26, 0xa3, 0xa0, 0xa1, 0x0b,
};
static const unsigned char kat51_nor_persstr[] = {
    0x07, 0xdf, 0x3d, 0x3d, 0x48, 0x6b, 0xbe, 0x68, 0x19, 0x04, 0x40, 0x94,
    0xc9, 0x15, 0x89, 0x99,
};
static const unsigned char kat51_nor_addin0[] = {
    0xf4, 0x9d, 0x27, 0x9f, 0x04, 0x81, 0x5d, 0x98, 0x30, 0x92, 0xb6, 0xb5,
    0xfe, 0x4f, 0x10, 0xf7,
};
static const unsigned char kat51_nor_addin1[] = {
    0xd7, 0x66, 0x07, 0x00, 0x64, 0xdf, 0xe2, 0x03, 0xb2, 0xa3, 0x9d, 0x1b,
    0x24, 0x75, 0x84, 0x10,
};
static const unsigned char kat51_nor_retbytes[] = {
    0xdd, 0xbb, 0x48, 0x8c, 0x6b, 0x6f, 0x0f, 0x9c, 0xb7, 0x05, 0xa3, 0xc5,
    0xab, 0x13, 0xc3, 0x17, 0x6f, 0x86, 0x7b, 0x12, 0x0c, 0x29, 0xdb, 0x37,
    0xf7, 0x9c, 0x00, 0x23, 0x8a, 0x77, 0xa0, 0x41, 0x45, 0x04, 0xd3, 0x28,
    0x93, 0xa0, 0x8c, 0x7e, 0x99, 0xc2, 0xb1, 0xa8, 0x8d, 0x50, 0x92, 0x2e,
    0x52, 0xb0, 0xdf, 0x56, 0x0b, 0x75, 0x3a, 0xdd, 0x48, 0x60, 0x4c, 0xaa,
    0xbb, 0xb9, 0x75, 0x2f,
};
static const struct drbg_kat_no_reseed kat51_nor_t = {
    5, kat51_nor_entropyin, kat51_nor_nonce, kat51_nor_persstr,
    kat51_nor_addin0, kat51_nor_addin1, kat51_nor_retbytes
};
static const struct drbg_kat kat51_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat51_nor_t
};

static const unsigned char kat52_nor_entropyin[] = {
    0x56, 0x6f, 0xc3, 0x38, 0x98, 0x5a, 0x8f, 0xc8, 0x67, 0xda, 0xbe, 0x24,
    0xf7, 0xf9, 0x09, 0x7d,
};
static const unsigned char kat52_nor_nonce[] = {
    0xdb, 0xf7, 0x1d, 0x79, 0x17, 0xe0, 0xb1, 0x51,
};
static const unsigned char kat52_nor_persstr[] = {
    0x66, 0x56, 0x94, 0x18, 0x17, 0xa6, 0x15, 0xe2, 0x9f, 0x62, 0xbd, 0x4f,
    0x1d, 0x93, 0xcb, 0x11,
};
static const unsigned char kat52_nor_addin0[] = {
    0xa2, 0x74, 0xbe, 0x52, 0x97, 0x46, 0xbd, 0x0b, 0x6a, 0x58, 0x73, 0xdc,
    0x3c, 0x1d, 0xae, 0xbc,
};
static const unsigned char kat52_nor_addin1[] = {
    0x4e, 0x23, 0xfa, 0x29, 0xd2, 0x6b, 0x1f, 0xe9, 0x2b, 0xb4, 0x2b, 0x3e,
    0x9f, 0x2b, 0x4c, 0x08,
};
static const unsigned char kat52_nor_retbytes[] = {
    0xb5, 0xda, 0x60, 0xb8, 0x70, 0x22, 0x6a, 0x48, 0xb2, 0xb9, 0xb5, 0xd9,
    0xa9, 0x46, 0x49, 0x54, 0xa8, 0xdb, 0x0d, 0x3e, 0x5f, 0xb3, 0x06, 0x3c,
    0x80, 0xe6, 0x75, 0x19, 0x7d, 0xc5, 0x84, 0x07, 0xb4, 0xc0, 0x40, 0x9c,
    0xd0, 0x69, 0x75, 0xed, 0xa8, 0xaf, 0xe6, 0x66, 0x7a, 0xa0, 0xd7, 0x9e,
    0xcf, 0x55, 0x5e, 0x23, 0x8f, 0x83, 0x42, 0x4f, 0x13, 0x08, 0x19, 0x20,
    0x0c, 0x24, 0x4a, 0x77,
};
static const struct drbg_kat_no_reseed kat52_nor_t = {
    6, kat52_nor_entropyin, kat52_nor_nonce, kat52_nor_persstr,
    kat52_nor_addin0, kat52_nor_addin1, kat52_nor_retbytes
};
static const struct drbg_kat kat52_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat52_nor_t
};

static const unsigned char kat53_nor_entropyin[] = {
    0x67, 0x43, 0xbc, 0x12, 0x2a, 0xaa, 0x13, 0x4f, 0x82, 0xac, 0x8c, 0x78,
    0xcc, 0x0b, 0x64, 0xe7,
};
static const unsigned char kat53_nor_nonce[] = {
    0x2d, 0xc3, 0x2e, 0x4f, 0x3b, 0xba, 0x5a, 0x20,
};
static const unsigned char kat53_nor_persstr[] = {
    0xb8, 0x07, 0x7f, 0x23, 0xa9, 0x08, 0xc4, 0x25, 0x92, 0xc1, 0x3b, 0x63,
    0x89, 0x09, 0x98, 0xd0,
};
static const unsigned char kat53_nor_addin0[] = {
    0xd6, 0x83, 0x74, 0xf5, 0x8c, 0xe2, 0x5b, 0xd8, 0xe9, 0x5f, 0xe0, 0x6d,
    0xe8, 0xa9, 0x50, 0xa1,
};
static const unsigned char kat53_nor_addin1[] = {
    0x07, 0xb7, 0xab, 0x9c, 0x32, 0xa9, 0x78, 0x30, 0xa0, 0x0e, 0xd9, 0x89,
    0x5e, 0x3b, 0xbe, 0x19,
};
static const unsigned char kat53_nor_retbytes[] = {
    0xfe, 0x82, 0x7f, 0x53, 0x07, 0x6d, 0xf3, 0x08, 0x47, 0xd0, 0xfb, 0x0f,
    0x55, 0xa2, 0xcd, 0x63, 0x81, 0xca, 0x0e, 0x79, 0xe6, 0x81, 0x62, 0x1f,
    0x49, 0xf3, 0x1e, 0x48, 0x11, 0xe9, 0xa6, 0x4c, 0x57, 0xc0, 0x99, 0x4f,
    0x18, 0x0c, 0x31, 0x1a, 0x40, 0x44, 0x40, 0xf2, 0x77, 0xfa, 0x62, 0xe0,
    0xf9, 0x9c, 0x6f, 0x5c, 0x7b, 0x88, 0xbe, 0xe2, 0x07, 0xfc, 0xa9, 0x92,
    0x6b, 0xfc, 0x9e, 0x46,
};
static const struct drbg_kat_no_reseed kat53_nor_t = {
    7, kat53_nor_entropyin, kat53_nor_nonce, kat53_nor_persstr,
    kat53_nor_addin0, kat53_nor_addin1, kat53_nor_retbytes
};
static const struct drbg_kat kat53_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat53_nor_t
};

static const unsigned char kat54_nor_entropyin[] = {
    0xf2, 0xad, 0xdf, 0x01, 0x67, 0xa1, 0x6d, 0xa3, 0x8d, 0x2f, 0xd7, 0x92,
    0xa7, 0x18, 0x9c, 0x06,
};
static const unsigned char kat54_nor_nonce[] = {
    0x2b, 0x50, 0xd8, 0xcc, 0x7f, 0x71, 0x40, 0x1c,
};
static const unsigned char kat54_nor_persstr[] = {
    0xb9, 0x17, 0xa5, 0x4c, 0x13, 0xd5, 0x8c, 0xb6, 0xad, 0x47, 0xc8, 0xfa,
    0xf6, 0xf9, 0xa7, 0xf5,
};
static const unsigned char kat54_nor_addin0[] = {
    0xcf, 0x2e, 0x78, 0x5a, 0x06, 0xbc, 0x5e, 0xb7, 0x1e, 0x3b, 0x52, 0x4f,
    0x11, 0x72, 0xc7, 0x55,
};
static const unsigned char kat54_nor_addin1[] = {
    0xc9, 0xa7, 0x53, 0xa9, 0xce, 0xdd, 0xd7, 0x5d, 0x1a, 0x62, 0x6d, 0xa7,
    0x27, 0x95, 0xa3, 0xc6,
};
static const unsigned char kat54_nor_retbytes[] = {
    0xe2, 0xaa, 0xf1, 0x1c, 0x20, 0xde, 0x2b, 0xaf, 0x89, 0x23, 0x04, 0x1d,
    0x83, 0x8f, 0x4a, 0xcb, 0x38, 0xe2, 0x22, 0xd3, 0x06, 0xb9, 0x7c, 0x97,
    0xbf, 0xe5, 0x46, 0x61, 0x7b, 0x3d, 0x8d, 0x92, 0xe4, 0x0c, 0xaa, 0x2e,
    0xdc, 0xac, 0x7a, 0x2f, 0xb0, 0x64, 0x3e, 0xd5, 0x7d, 0x47, 0xa9, 0x0a,
    0x11, 0x64, 0x10, 0x44, 0xb6, 0xc6, 0xa7, 0xdf, 0x42, 0x4d, 0xbd, 0xa2,
    0xb4, 0x5f, 0x0d, 0xf9,
};
static const struct drbg_kat_no_reseed kat54_nor_t = {
    8, kat54_nor_entropyin, kat54_nor_nonce, kat54_nor_persstr,
    kat54_nor_addin0, kat54_nor_addin1, kat54_nor_retbytes
};
static const struct drbg_kat kat54_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat54_nor_t
};

static const unsigned char kat55_nor_entropyin[] = {
    0xca, 0x23, 0x34, 0x58, 0xf6, 0x6b, 0x26, 0xf2, 0xc1, 0x35, 0x7a, 0xef,
    0xce, 0xb6, 0x79, 0xef,
};
static const unsigned char kat55_nor_nonce[] = {
    0xfe, 0x13, 0x64, 0x04, 0xad, 0x7a, 0x49, 0xe7,
};
static const unsigned char kat55_nor_persstr[] = {
    0x26, 0x83, 0xdc, 0x31, 0x3b, 0xc0, 0xbb, 0x93, 0xeb, 0xb4, 0x6b, 0xbb,
    0x55, 0x3f, 0x58, 0x75,
};
static const unsigned char kat55_nor_addin0[] = {
    0xf7, 0x8a, 0xb1, 0xf4, 0x52, 0x6f, 0x0f, 0xa5, 0xcb, 0x75, 0x43, 0xd1,
    0x46, 0x43, 0xfe, 0xb5,
};
static const unsigned char kat55_nor_addin1[] = {
    0x40, 0xb9, 0x26, 0x32, 0x4c, 0xc2, 0x94, 0x3a, 0xca, 0x83, 0x7f, 0x69,
    0x0d, 0x0b, 0xcd, 0xa4,
};
static const unsigned char kat55_nor_retbytes[] = {
    0x37, 0x24, 0xd0, 0xe5, 0x80, 0xe9, 0x0d, 0x4e, 0xf3, 0x2e, 0x5f, 0x8a,
    0x5b, 0xb5, 0x62, 0x25, 0xa9, 0x13, 0xf2, 0x10, 0xb7, 0x6d, 0x5a, 0x78,
    0xfd, 0x8e, 0xf2, 0x5d, 0x6e, 0xec, 0x9f, 0xf1, 0x47, 0x11, 0xd6, 0xdb,
    0x80, 0xe1, 0xe8, 0x11, 0x42, 0xa5, 0x60, 0x75, 0x34, 0xa7, 0x25, 0xd2,
    0x78, 0x5c, 0x8e, 0x3e, 0x4a, 0x48, 0xa5, 0x94, 0xb0, 0xd9, 0x1b, 0xc9,
    0x32, 0x82, 0x7b, 0xd7,
};
static const struct drbg_kat_no_reseed kat55_nor_t = {
    9, kat55_nor_entropyin, kat55_nor_nonce, kat55_nor_persstr,
    kat55_nor_addin0, kat55_nor_addin1, kat55_nor_retbytes
};
static const struct drbg_kat kat55_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat55_nor_t
};

static const unsigned char kat56_nor_entropyin[] = {
    0x2b, 0x52, 0x1d, 0xc4, 0x03, 0x12, 0x9f, 0xd9, 0xed, 0x0b, 0xe9, 0xf5,
    0xfd, 0xe0, 0x66, 0x17,
};
static const unsigned char kat56_nor_nonce[] = {
    0xf0, 0xdf, 0xa0, 0x1e, 0x09, 0x9b, 0x95, 0xa1,
};
static const unsigned char kat56_nor_persstr[] = {
    0xfe, 0x85, 0x8d, 0x59, 0xd7, 0x34, 0xf6, 0x0f, 0x71, 0xa3, 0x87, 0x55,
    0x3c, 0xca, 0x46, 0xff,
};
static const unsigned char kat56_nor_addin0[] = {
    0x6f, 0xba, 0x8d, 0x06, 0xff, 0x43, 0x30, 0x07, 0xbd, 0x7d, 0x15, 0x56,
    0x9a, 0xf3, 0xd0, 0x69,
};
static const unsigned char kat56_nor_addin1[] = {
    0x80, 0xb0, 0xc1, 0x1f, 0x18, 0xb5, 0x92, 0x3a, 0x89, 0xf0, 0xfe, 0x23,
    0x77, 0x6d, 0x39, 0x05,
};
static const unsigned char kat56_nor_retbytes[] = {
    0x41, 0x73, 0x88, 0xde, 0xe9, 0x08, 0x4f, 0x58, 0x8d, 0x0a, 0xdb, 0x5e,
    0x5d, 0x1b, 0x54, 0x6b, 0x12, 0x55, 0xcd, 0xf5, 0x9d, 0x69, 0x95, 0x7a,
    0xe6, 0x87, 0xc7, 0x82, 0x2b, 0x93, 0x5c, 0xe0, 0x29, 0x30, 0x6c, 0x02,
    0x50, 0xd4, 0x69, 0x83, 0x9c, 0x7d, 0xf9, 0xd3, 0xd0, 0x8e, 0xc8, 0xc5,
    0x0a, 0x66, 0x7a, 0xf1, 0x0d, 0x8d, 0x5b, 0x00, 0x73, 0x02, 0xa6, 0xe8,
    0x7b, 0x4e, 0x7f, 0x2c,
};
static const struct drbg_kat_no_reseed kat56_nor_t = {
    10, kat56_nor_entropyin, kat56_nor_nonce, kat56_nor_persstr,
    kat56_nor_addin0, kat56_nor_addin1, kat56_nor_retbytes
};
static const struct drbg_kat kat56_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat56_nor_t
};

static const unsigned char kat57_nor_entropyin[] = {
    0xd0, 0x50, 0x2b, 0x78, 0xf7, 0xaa, 0x6f, 0xa1, 0xd8, 0x88, 0x42, 0x1d,
    0x70, 0x2c, 0xad, 0xd1,
};
static const unsigned char kat57_nor_nonce[] = {
    0x4b, 0x37, 0xb8, 0x96, 0x59, 0x69, 0xea, 0x8d,
};
static const unsigned char kat57_nor_persstr[] = {
    0x56, 0xbd, 0x43, 0x1f, 0x04, 0x88, 0xfe, 0x6d, 0x7b, 0xa3, 0x83, 0x6e,
    0x5c, 0x8c, 0x7a, 0xa9,
};
static const unsigned char kat57_nor_addin0[] = {
    0x26, 0x92, 0x5d, 0xbc, 0xbe, 0xd7, 0x33, 0xbe, 0x2a, 0x5e, 0x76, 0x49,
    0xc8, 0x76, 0xa0, 0xf8,
};
static const unsigned char kat57_nor_addin1[] = {
    0x9c, 0x70, 0x35, 0x2d, 0x4f, 0x9c, 0x88, 0x0e, 0x76, 0xd4, 0x68, 0x08,
    0x38, 0xdd, 0x0d, 0xea,
};
static const unsigned char kat57_nor_retbytes[] = {
    0x65, 0x00, 0x24, 0x2c, 0x35, 0x88, 0xb4, 0x52, 0x30, 0x7b, 0x2a, 0x06,
    0x34, 0x90, 0x0e, 0x75, 0x6a, 0x3d, 0x26, 0x19, 0xfc, 0x2a, 0xdf, 0x10,
    0x77, 0x69, 0x22, 0xb0, 0x06, 0x61, 0x7e, 0xe8, 0x81, 0x82, 0x83, 0x66,
    0xac, 0x52, 0x7f, 0xad, 0xa0, 0x01, 0x86, 0xbc, 0x8b, 0xfd, 0x6b, 0xee,
    0x66, 0xd6, 0x38, 0x03, 0x5a, 0xaf, 0xff, 0xbc, 0x83, 0x32, 0xa5, 0x4c,
    0xa1, 0x7d, 0xf1, 0x6f,
};
static const struct drbg_kat_no_reseed kat57_nor_t = {
    11, kat57_nor_entropyin, kat57_nor_nonce, kat57_nor_persstr,
    kat57_nor_addin0, kat57_nor_addin1, kat57_nor_retbytes
};
static const struct drbg_kat kat57_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat57_nor_t
};

static const unsigned char kat58_nor_entropyin[] = {
    0xa3, 0xb9, 0x5f, 0x32, 0x01, 0xfa, 0x01, 0x01, 0x07, 0x57, 0x30, 0x74,
    0xb1, 0x19, 0x9b, 0xc0,
};
static const unsigned char kat58_nor_nonce[] = {
    0x60, 0x30, 0xe5, 0xb6, 0x13, 0x47, 0xb0, 0xe2,
};
static const unsigned char kat58_nor_persstr[] = {
    0x9e, 0x77, 0x85, 0x46, 0xd4, 0x5f, 0x17, 0x93, 0xa1, 0x1d, 0x80, 0xa0,
    0x2e, 0x4c, 0xa3, 0xc3,
};
static const unsigned char kat58_nor_addin0[] = {
    0xe5, 0x07, 0x33, 0x61, 0x98, 0x25, 0x31, 0x37, 0xa5, 0x55, 0xe8, 0xc1,
    0x18, 0xa1, 0x2d, 0x2f,
};
static const unsigned char kat58_nor_addin1[] = {
    0xe5, 0xce, 0xea, 0xcb, 0xb5, 0x22, 0xf5, 0xde, 0x81, 0x78, 0x9c, 0x32,
    0xcb, 0xcd, 0xef, 0x30,
};
static const unsigned char kat58_nor_retbytes[] = {
    0x4a, 0xd1, 0x8e, 0xa0, 0x83, 0x7f, 0x06, 0x3e, 0x9f, 0xbf, 0x6c, 0x72,
    0xd1, 0xaf, 0x6f, 0x77, 0x96, 0x41, 0xf4, 0xe1, 0x1c, 0x40, 0x0d, 0x95,
    0xcb, 0x2b, 0xa4, 0x22, 0x99, 0xa9, 0x66, 0x48, 0xe0, 0x59, 0xe1, 0xa6,
    0x3f, 0x7c, 0x03, 0x79, 0x25, 0x44, 0x89, 0xc7, 0xcb, 0xff, 0xfb, 0x22,
    0x11, 0x84, 0x57, 0x18, 0xc5, 0x26, 0x00, 0xc9, 0xe8, 0x58, 0x49, 0xd6,
    0xe0, 0x95, 0x1f, 0x7b,
};
static const struct drbg_kat_no_reseed kat58_nor_t = {
    12, kat58_nor_entropyin, kat58_nor_nonce, kat58_nor_persstr,
    kat58_nor_addin0, kat58_nor_addin1, kat58_nor_retbytes
};
static const struct drbg_kat kat58_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat58_nor_t
};

static const unsigned char kat59_nor_entropyin[] = {
    0x88, 0x18, 0xbe, 0x7e, 0x8e, 0x81, 0x5b, 0xca, 0x0c, 0xb5, 0xa0, 0x14,
    0x4d, 0x44, 0xb6, 0x72,
};
static const unsigned char kat59_nor_nonce[] = {
    0x12, 0x2b, 0x14, 0x8c, 0xa1, 0xaa, 0xba, 0x60,
};
static const unsigned char kat59_nor_persstr[] = {
    0x90, 0x6d, 0xc3, 0x0b, 0x39, 0x54, 0x27, 0x38, 0x56, 0x73, 0x26, 0x13,
    0x66, 0xf5, 0xe9, 0xa8,
};
static const unsigned char kat59_nor_addin0[] = {
    0x56, 0xee, 0x67, 0xff, 0x14, 0x66, 0xce, 0xed, 0xd7, 0x28, 0x1d, 0x73,
    0xf3, 0x9c, 0x6c, 0x85,
};
static const unsigned char kat59_nor_addin1[] = {
    0x82, 0xe4, 0xa6, 0xed, 0xb0, 0x78, 0x00, 0x79, 0xfa, 0xee, 0x25, 0x8e,
    0xd1, 0x13, 0x7e, 0xcf,
};
static const unsigned char kat59_nor_retbytes[] = {
    0x88, 0x55, 0xad, 0x03, 0xe3, 0x87, 0x27, 0xd0, 0x49, 0xe8, 0x55, 0x8e,
    0xcc, 0x66, 0x3b, 0x8c, 0x89, 0xcb, 0x5b, 0xd1, 0x38, 0xac, 0x34, 0x9c,
    0x6e, 0x5f, 0x78, 0x41, 0x70, 0xc5, 0x07, 0x69, 0xc8, 0xc9, 0x7b, 0x09,
    0x4d, 0xfa, 0x07, 0x52, 0xd0, 0x65, 0x53, 0x1d, 0xa4, 0x00, 0x85, 0x81,
    0x7c, 0xcb, 0x08, 0xf2, 0xb9, 0x4c, 0xbb, 0xba, 0xe7, 0x59, 0x8c, 0x96,
    0x3e, 0x3a, 0xd6, 0xd1,
};
static const struct drbg_kat_no_reseed kat59_nor_t = {
    13, kat59_nor_entropyin, kat59_nor_nonce, kat59_nor_persstr,
    kat59_nor_addin0, kat59_nor_addin1, kat59_nor_retbytes
};
static const struct drbg_kat kat59_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat59_nor_t
};

static const unsigned char kat60_nor_entropyin[] = {
    0x00, 0x90, 0xdc, 0xb9, 0x1f, 0x02, 0x70, 0xf6, 0x58, 0x20, 0x7d, 0x49,
    0xa7, 0x47, 0x38, 0xe2,
};
static const unsigned char kat60_nor_nonce[] = {
    0x9d, 0x18, 0xae, 0xa2, 0x68, 0x4e, 0xe2, 0xb7,
};
static const unsigned char kat60_nor_persstr[] = {
    0x64, 0xa8, 0x18, 0x57, 0xce, 0x29, 0xf0, 0xd2, 0xa8, 0x51, 0x03, 0x00,
    0xb0, 0xa3, 0xfa, 0xc0,
};
static const unsigned char kat60_nor_addin0[] = {
    0xcd, 0x7f, 0x89, 0xf8, 0xd5, 0x46, 0x8e, 0xae, 0x8b, 0xca, 0x31, 0x4f,
    0xd9, 0xf2, 0x5b, 0x23,
};
static const unsigned char kat60_nor_addin1[] = {
    0x94, 0x9b, 0x94, 0x41, 0x7f, 0xb4, 0x2b, 0xbc, 0x7c, 0xdb, 0x3f, 0x53,
    0xa0, 0x55, 0x80, 0x19,
};
static const unsigned char kat60_nor_retbytes[] = {
    0x63, 0xd3, 0xe9, 0x19, 0x88, 0x1f, 0x87, 0xd6, 0x9c, 0xba, 0xdc, 0xb6,
    0x78, 0x9e, 0x0b, 0x8f, 0x2e, 0x49, 0xf4, 0xc9, 0xd0, 0xe3, 0xba, 0x1e,
    0x4a, 0x59, 0x16, 0x2a, 0xf7, 0x18, 0xbc, 0x40, 0xdd, 0x67, 0x3e, 0xae,
    0x8f, 0x48, 0xfd, 0x27, 0x82, 0xe0, 0x6c, 0xde, 0xc5, 0x63, 0x2a, 0x4b,
    0x2e, 0x60, 0x65, 0x5c, 0xba, 0x75, 0xaf, 0xaf, 0x4c, 0x12, 0xa6, 0x74,
    0xee, 0xe4, 0xd3, 0xfc,
};
static const struct drbg_kat_no_reseed kat60_nor_t = {
    14, kat60_nor_entropyin, kat60_nor_nonce, kat60_nor_persstr,
    kat60_nor_addin0, kat60_nor_addin1, kat60_nor_retbytes
};
static const struct drbg_kat kat60_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat60_nor_t
};

static const unsigned char kat61_nor_entropyin[] = {
    0x2d, 0x2a, 0xb5, 0x64, 0x20, 0x29, 0x18, 0xc4, 0xef, 0x5b, 0x10, 0x2d,
    0xda, 0x38, 0x5a, 0x18,
};
static const unsigned char kat61_nor_nonce[] = {
    0x25, 0x91, 0x95, 0x26, 0x9e, 0xc1, 0x1a, 0xf6,
};
static const unsigned char kat61_nor_persstr[] = {0};
static const unsigned char kat61_nor_addin0[] = {0};
static const unsigned char kat61_nor_addin1[] = {0};
static const unsigned char kat61_nor_retbytes[] = {
    0x2c, 0x5c, 0xd7, 0x9e, 0xd8, 0x76, 0x22, 0xa9, 0x1b, 0x86, 0x54, 0xc8,
    0x90, 0x3d, 0x85, 0x22, 0x42, 0xcd, 0x49, 0xcb, 0x5d, 0xf2, 0xd4, 0xb4,
    0x15, 0x05, 0x84, 0x30, 0x1c, 0x59, 0xf0, 0x1f, 0xd9, 0x5a, 0x70, 0x2a,
    0xc1, 0x57, 0xc8, 0x4c, 0xc1, 0x5f, 0x42, 0xc8, 0x21, 0x13, 0x35, 0x67,
    0x2d, 0x8c, 0xe1, 0x29, 0x1e, 0xf9, 0xb1, 0xde, 0xf7, 0x81, 0x49, 0xa0,
    0x4f, 0xa2, 0x69, 0x7c,
};
static const struct drbg_kat_no_reseed kat61_nor_t = {
    0, kat61_nor_entropyin, kat61_nor_nonce, kat61_nor_persstr,
    kat61_nor_addin0, kat61_nor_addin1, kat61_nor_retbytes
};
static const struct drbg_kat kat61_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat61_nor_t
};

static const unsigned char kat62_nor_entropyin[] = {
    0xa0, 0x16, 0x46, 0x3d, 0xbb, 0x49, 0x99, 0x90, 0xcb, 0xcd, 0xa4, 0x50,
    0x46, 0xd8, 0xf3, 0x37,
};
static const unsigned char kat62_nor_nonce[] = {
    0x24, 0x9d, 0x02, 0xde, 0x2d, 0xcf, 0x3e, 0x57,
};
static const unsigned char kat62_nor_persstr[] = {0};
static const unsigned char kat62_nor_addin0[] = {0};
static const unsigned char kat62_nor_addin1[] = {0};
static const unsigned char kat62_nor_retbytes[] = {
    0x35, 0xb8, 0x1f, 0xb9, 0x42, 0x41, 0xf4, 0xc9, 0x31, 0x9a, 0x7f, 0x16,
    0xb4, 0x42, 0x49, 0x52, 0x52, 0xdb, 0x4c, 0x98, 0x48, 0x60, 0xd0, 0x35,
    0xf6, 0xc4, 0x54, 0x03, 0xf9, 0x74, 0xf5, 0x34, 0xfa, 0x93, 0xb2, 0x1b,
    0x6b, 0x89, 0x44, 0x1b, 0xe0, 0x7c, 0x5c, 0x29, 0x31, 0x7f, 0x86, 0x8d,
    0xc9, 0xab, 0x5c, 0x18, 0x37, 0x74, 0x37, 0xfa, 0xdb, 0x4d, 0x85, 0x7e,
    0xe0, 0x92, 0xf9, 0x23,
};
static const struct drbg_kat_no_reseed kat62_nor_t = {
    1, kat62_nor_entropyin, kat62_nor_nonce, kat62_nor_persstr,
    kat62_nor_addin0, kat62_nor_addin1, kat62_nor_retbytes
};
static const struct drbg_kat kat62_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat62_nor_t
};

static const unsigned char kat63_nor_entropyin[] = {
    0x1e, 0x38, 0x20, 0xc4, 0x6e, 0x50, 0xb7, 0xcb, 0xc5, 0x8b, 0x05, 0xaa,
    0x72, 0x24, 0x27, 0xa8,
};
static const unsigned char kat63_nor_nonce[] = {
    0xfe, 0x0f, 0x40, 0x01, 0x4e, 0xb3, 0x27, 0x9f,
};
static const unsigned char kat63_nor_persstr[] = {0};
static const unsigned char kat63_nor_addin0[] = {0};
static const unsigned char kat63_nor_addin1[] = {0};
static const unsigned char kat63_nor_retbytes[] = {
    0x18, 0x53, 0x0b, 0x9d, 0xe8, 0x89, 0xd8, 0xad, 0x3f, 0x41, 0xd5, 0x39,
    0x79, 0x6c, 0x2a, 0x95, 0xd6, 0x82, 0xfb, 0x02, 0x6d, 0xf7, 0xca, 0x09,
    0x4a, 0xf4, 0xab, 0x23, 0x95, 0x29, 0x2e, 0x6f, 0xcd, 0xb1, 0x75, 0xba,
    0x25, 0x7d, 0x61, 0x27, 0x9d, 0xa4, 0xd0, 0xfc, 0x78, 0xfd, 0xf8, 0xa4,
    0xeb, 0x46, 0xa0, 0xe2, 0xd7, 0x54, 0xf5, 0xf8, 0xf0, 0x9e, 0xcb, 0x15,
    0x7c, 0x8a, 0x58, 0xa1,
};
static const struct drbg_kat_no_reseed kat63_nor_t = {
    2, kat63_nor_entropyin, kat63_nor_nonce, kat63_nor_persstr,
    kat63_nor_addin0, kat63_nor_addin1, kat63_nor_retbytes
};
static const struct drbg_kat kat63_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat63_nor_t
};

static const unsigned char kat64_nor_entropyin[] = {
    0x3b, 0x02, 0xc5, 0xa4, 0x74, 0x67, 0x96, 0x48, 0xe2, 0xfe, 0x09, 0x0a,
    0x13, 0xbe, 0x04, 0x1c,
};
static const unsigned char kat64_nor_nonce[] = {
    0x71, 0xed, 0x5a, 0xa0, 0x78, 0xb8, 0x3a, 0x7b,
};
static const unsigned char kat64_nor_persstr[] = {0};
static const unsigned char kat64_nor_addin0[] = {0};
static const unsigned char kat64_nor_addin1[] = {0};
static const unsigned char kat64_nor_retbytes[] = {
    0xe8, 0xe2, 0xe5, 0xd8, 0xbd, 0x99, 0x29, 0xe4, 0xe1, 0x67, 0x5c, 0x84,
    0x61, 0x45, 0x7c, 0xab, 0x06, 0x33, 0xd8, 0x3f, 0x9b, 0xd2, 0x43, 0xbc,
    0x58, 0x29, 0x37, 0xc2, 0xb9, 0x61, 0xb7, 0x1c, 0x11, 0x85, 0x6f, 0x83,
    0x6c, 0x37, 0x32, 0x9d, 0x3f, 0xa8, 0x63, 0x76, 0xee, 0xc9, 0x71, 0x39,
    0x10, 0x4d, 0xe5, 0x32, 0x60, 0xe1, 0x22, 0xcb, 0x2b, 0x6d, 0x82, 0x4e,
    0x21, 0xe7, 0x5c, 0xe7,
};
static const struct drbg_kat_no_reseed kat64_nor_t = {
    3, kat64_nor_entropyin, kat64_nor_nonce, kat64_nor_persstr,
    kat64_nor_addin0, kat64_nor_addin1, kat64_nor_retbytes
};
static const struct drbg_kat kat64_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat64_nor_t
};

static const unsigned char kat65_nor_entropyin[] = {
    0x05, 0x0c, 0x48, 0xab, 0x8a, 0x05, 0xff, 0x69, 0x0b, 0x62, 0xe3, 0x7d,
    0x9b, 0x21, 0xf2, 0xfa,
};
static const unsigned char kat65_nor_nonce[] = {
    0xf7, 0x96, 0x48, 0x75, 0xbe, 0x28, 0x83, 0x53,
};
static const unsigned char kat65_nor_persstr[] = {0};
static const unsigned char kat65_nor_addin0[] = {0};
static const unsigned char kat65_nor_addin1[] = {0};
static const unsigned char kat65_nor_retbytes[] = {
    0x24, 0xcb, 0x0e, 0xdc, 0xb8, 0x9e, 0xec, 0x61, 0x76, 0x13, 0xa2, 0x7f,
    0xe3, 0xe4, 0x47, 0x5a, 0xff, 0xab, 0x49, 0x8d, 0x57, 0xbc, 0x3c, 0x4e,
    0xf8, 0x44, 0xfe, 0x19, 0xb3, 0x84, 0x13, 0x47, 0x7c, 0x1d, 0xb0, 0x40,
    0xf1, 0x67, 0x57, 0xf1, 0x85, 0x1b, 0x79, 0x9c, 0x1f, 0x26, 0x69, 0xaa,
    0x9f, 0x9c, 0x50, 0x53, 0x2e, 0xaf, 0xbe, 0x1e, 0x29, 0x48, 0xe3, 0x38,
    0xd1, 0xca, 0x4b, 0xd7,
};
static const struct drbg_kat_no_reseed kat65_nor_t = {
    4, kat65_nor_entropyin, kat65_nor_nonce, kat65_nor_persstr,
    kat65_nor_addin0, kat65_nor_addin1, kat65_nor_retbytes
};
static const struct drbg_kat kat65_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat65_nor_t
};

static const unsigned char kat66_nor_entropyin[] = {
    0xec, 0xd9, 0x9c, 0x0b, 0x49, 0x1c, 0xf6, 0xa3, 0x23, 0xbc, 0x33, 0x36,
    0x46, 0xe5, 0xc5, 0x6a,
};
static const unsigned char kat66_nor_nonce[] = {
    0xb1, 0x0a, 0x17, 0x84, 0x99, 0x5c, 0xb3, 0x4c,
};
static const unsigned char kat66_nor_persstr[] = {0};
static const unsigned char kat66_nor_addin0[] = {0};
static const unsigned char kat66_nor_addin1[] = {0};
static const unsigned char kat66_nor_retbytes[] = {
    0x65, 0x35, 0xc4, 0x8d, 0x99, 0x9a, 0x30, 0x6a, 0xcb, 0x83, 0xf7, 0xb7,
    0x20, 0xb3, 0xc3, 0x97, 0x72, 0xa4, 0x96, 0xe7, 0x5a, 0x89, 0x0f, 0x6e,
    0x14, 0xd7, 0x0a, 0x82, 0x41, 0x42, 0xa1, 0x2d, 0x01, 0xde, 0xe5, 0x52,
    0x71, 0xb8, 0x2c, 0x7b, 0x02, 0x7b, 0x94, 0x68, 0xc3, 0x09, 0xfa, 0x2d,
    0x08, 0xe9, 0x1e, 0x5e, 0x37, 0x9e, 0xb9, 0x0c, 0xf8, 0x91, 0x5c, 0xf4,
    0xcc, 0x24, 0x1b, 0x50,
};
static const struct drbg_kat_no_reseed kat66_nor_t = {
    5, kat66_nor_entropyin, kat66_nor_nonce, kat66_nor_persstr,
    kat66_nor_addin0, kat66_nor_addin1, kat66_nor_retbytes
};
static const struct drbg_kat kat66_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat66_nor_t
};

static const unsigned char kat67_nor_entropyin[] = {
    0x34, 0xae, 0xe3, 0xef, 0x35, 0xb2, 0xa1, 0xe3, 0x1c, 0x42, 0x97, 0x25,
    0xfe, 0xaa, 0x6b, 0xfd,
};
static const unsigned char kat67_nor_nonce[] = {
    0xe4, 0x36, 0x9b, 0x6a, 0xda, 0x90, 0x91, 0xed,
};
static const unsigned char kat67_nor_persstr[] = {0};
static const unsigned char kat67_nor_addin0[] = {0};
static const unsigned char kat67_nor_addin1[] = {0};
static const unsigned char kat67_nor_retbytes[] = {
    0xaf, 0x9d, 0xb4, 0xdf, 0x46, 0xac, 0x80, 0xfe, 0xca, 0xd6, 0xc3, 0xd1,
    0xd9, 0x75, 0xd2, 0xd0, 0xa6, 0x65, 0xf3, 0x1b, 0x4e, 0x3b, 0x07, 0xd9,
    0x62, 0x63, 0x22, 0x96, 0xb7, 0x60, 0x39, 0xdd, 0xca, 0xb8, 0xed, 0xde,
    0x17, 0xba, 0xd6, 0x23, 0x01, 0x31, 0x16, 0x2e, 0x59, 0xd0, 0x7d, 0xbb,
    0xd0, 0xf0, 0xb5, 0x2e, 0x5c, 0xa9, 0x3c, 0xf3, 0x7f, 0x12, 0xc0, 0x30,
    0x42, 0xbf, 0x0c, 0x82,
};
static const struct drbg_kat_no_reseed kat67_nor_t = {
    6, kat67_nor_entropyin, kat67_nor_nonce, kat67_nor_persstr,
    kat67_nor_addin0, kat67_nor_addin1, kat67_nor_retbytes
};
static const struct drbg_kat kat67_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat67_nor_t
};

static const unsigned char kat68_nor_entropyin[] = {
    0x91, 0x15, 0x64, 0x69, 0x54, 0x0c, 0x78, 0x5d, 0xf5, 0x6a, 0x10, 0xe1,
    0x6b, 0x66, 0x10, 0x7c,
};
static const unsigned char kat68_nor_nonce[] = {
    0x7e, 0x2b, 0x2a, 0x7f, 0x58, 0xf3, 0xef, 0x20,
};
static const unsigned char kat68_nor_persstr[] = {0};
static const unsigned char kat68_nor_addin0[] = {0};
static const unsigned char kat68_nor_addin1[] = {0};
static const unsigned char kat68_nor_retbytes[] = {
    0x9a, 0x76, 0x3a, 0x4c, 0x05, 0x27, 0x25, 0xe7, 0xae, 0x9d, 0x21, 0x22,
    0xa4, 0x5c, 0xd1, 0x8e, 0xa2, 0x8c, 0x58, 0x17, 0x2b, 0x87, 0xc5, 0x7e,
    0xdb, 0x9d, 0xaa, 0x32, 0xf8, 0xa2, 0x6e, 0xb9, 0x8d, 0x6b, 0x88, 0x12,
    0x09, 0x5a, 0x54, 0xcf, 0x62, 0x82, 0xc5, 0xe0, 0xfc, 0x18, 0xcd, 0x30,
    0x12, 0x44, 0x15, 0xd2, 0x67, 0xc9, 0xfc, 0x2c, 0xda, 0xe9, 0xbe, 0xed,
    0xbb, 0x86, 0x58, 0x4e,
};
static const struct drbg_kat_no_reseed kat68_nor_t = {
    7, kat68_nor_entropyin, kat68_nor_nonce, kat68_nor_persstr,
    kat68_nor_addin0, kat68_nor_addin1, kat68_nor_retbytes
};
static const struct drbg_kat kat68_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat68_nor_t
};

static const unsigned char kat69_nor_entropyin[] = {
    0xcd, 0x5d, 0xbc, 0x35, 0xd1, 0xb4, 0x43, 0x5c, 0x4c, 0x94, 0x5b, 0xaf,
    0xe4, 0xb8, 0xfc, 0x7a,
};
static const unsigned char kat69_nor_nonce[] = {
    0xaf, 0x4a, 0xcd, 0xfc, 0x81, 0x19, 0x32, 0x2c,
};
static const unsigned char kat69_nor_persstr[] = {0};
static const unsigned char kat69_nor_addin0[] = {0};
static const unsigned char kat69_nor_addin1[] = {0};
static const unsigned char kat69_nor_retbytes[] = {
    0xfe, 0x67, 0x82, 0x99, 0xf1, 0xad, 0xe1, 0x96, 0x21, 0x88, 0x02, 0x35,
    0xfa, 0x6f, 0xe2, 0xb5, 0x01, 0x84, 0xf8, 0x12, 0x43, 0x83, 0x3e, 0xbe,
    0x68, 0xc6, 0x16, 0xa9, 0x03, 0x4e, 0x23, 0xe0, 0xcc, 0x87, 0x3e, 0x8a,
    0xc4, 0xd0, 0xe3, 0x65, 0xb5, 0x24, 0xa3, 0x78, 0x32, 0xb5, 0x2d, 0x97,
    0xa8, 0x38, 0x62, 0x93, 0x24, 0x91, 0xd2, 0x64, 0x3a, 0x73, 0xe4, 0x6a,
    0x80, 0xb5, 0x14, 0x9c,
};
static const struct drbg_kat_no_reseed kat69_nor_t = {
    8, kat69_nor_entropyin, kat69_nor_nonce, kat69_nor_persstr,
    kat69_nor_addin0, kat69_nor_addin1, kat69_nor_retbytes
};
static const struct drbg_kat kat69_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat69_nor_t
};

static const unsigned char kat70_nor_entropyin[] = {
    0xd5, 0x2d, 0x11, 0x59, 0x78, 0x7b, 0xaa, 0xb2, 0x13, 0xe8, 0x79, 0x21,
    0xfd, 0x95, 0x91, 0x10,
};
static const unsigned char kat70_nor_nonce[] = {
    0x7a, 0xb3, 0x75, 0x38, 0xee, 0x02, 0x45, 0xc8,
};
static const unsigned char kat70_nor_persstr[] = {0};
static const unsigned char kat70_nor_addin0[] = {0};
static const unsigned char kat70_nor_addin1[] = {0};
static const unsigned char kat70_nor_retbytes[] = {
    0xf4, 0xd9, 0xea, 0xd3, 0x59, 0x46, 0x06, 0xac, 0x51, 0x94, 0x22, 0x15,
    0xaa, 0x0b, 0xd2, 0x9b, 0xaa, 0x7c, 0x28, 0x1e, 0x62, 0xd2, 0xbc, 0xc1,
    0xa7, 0x7f, 0x57, 0xe3, 0x59, 0x66, 0x83, 0x19, 0x9d, 0x70, 0xc0, 0x09,
    0xc0, 0x84, 0xeb, 0x36, 0x54, 0x22, 0x4e, 0xd7, 0xb0, 0x7a, 0x09, 0xb4,
    0x94, 0x48, 0x0a, 0x11, 0x3b, 0x33, 0xe0, 0x36, 0x6c, 0xb2, 0x25, 0x3e,
    0xdb, 0xb0, 0xcb, 0x5e,
};
static const struct drbg_kat_no_reseed kat70_nor_t = {
    9, kat70_nor_entropyin, kat70_nor_nonce, kat70_nor_persstr,
    kat70_nor_addin0, kat70_nor_addin1, kat70_nor_retbytes
};
static const struct drbg_kat kat70_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat70_nor_t
};

static const unsigned char kat71_nor_entropyin[] = {
    0x51, 0x10, 0xdb, 0x9d, 0x5f, 0x4f, 0xbd, 0xc0, 0x37, 0x55, 0x0e, 0x83,
    0xc2, 0xbd, 0x96, 0xfe,
};
static const unsigned char kat71_nor_nonce[] = {
    0xde, 0xe6, 0x41, 0xf9, 0x17, 0x7c, 0x9d, 0xa4,
};
static const unsigned char kat71_nor_persstr[] = {0};
static const unsigned char kat71_nor_addin0[] = {0};
static const unsigned char kat71_nor_addin1[] = {0};
static const unsigned char kat71_nor_retbytes[] = {
    0x05, 0xf8, 0xca, 0xd2, 0xd0, 0x31, 0xce, 0x6b, 0x52, 0x7e, 0xe0, 0x43,
    0xb5, 0x05, 0x6d, 0xb2, 0x0e, 0xe5, 0xca, 0x66, 0x0a, 0x95, 0xc2, 0x9b,
    0x8f, 0x8b, 0x2e, 0x43, 0xef, 0xa5, 0xc7, 0xaa, 0x62, 0x35, 0x12, 0x33,
    0xf4, 0xf1, 0x60, 0x36, 0x8c, 0x3d, 0xad, 0x4a, 0x74, 0xf4, 0x71, 0xbb,
    0x02, 0x15, 0x3f, 0x86, 0xc8, 0x7a, 0x14, 0x0a, 0x36, 0xe0, 0xf1, 0x22,
    0xc6, 0x8b, 0xa7, 0x7d,
};
static const struct drbg_kat_no_reseed kat71_nor_t = {
    10, kat71_nor_entropyin, kat71_nor_nonce, kat71_nor_persstr,
    kat71_nor_addin0, kat71_nor_addin1, kat71_nor_retbytes
};
static const struct drbg_kat kat71_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat71_nor_t
};

static const unsigned char kat72_nor_entropyin[] = {
    0x97, 0x90, 0x65, 0xbc, 0x1a, 0x1c, 0x8c, 0x82, 0x90, 0xdd, 0xe2, 0x19,
    0xf0, 0x12, 0x1a, 0xbe,
};
static const unsigned char kat72_nor_nonce[] = {
    0x59, 0x3a, 0x0e, 0x1f, 0xa9, 0x85, 0x64, 0xa3,
};
static const unsigned char kat72_nor_persstr[] = {0};
static const unsigned char kat72_nor_addin0[] = {0};
static const unsigned char kat72_nor_addin1[] = {0};
static const unsigned char kat72_nor_retbytes[] = {
    0x19, 0x9e, 0x68, 0xe5, 0xb6, 0x37, 0xa2, 0x7b, 0xe7, 0x94, 0x53, 0xd9,
    0x95, 0x3a, 0x0e, 0x52, 0x95, 0x36, 0x37, 0x4e, 0x7f, 0x63, 0xe8, 0x59,
    0x5a, 0xfe, 0x4b, 0x5d, 0x4e, 0x3b, 0xb3, 0x79, 0xc6, 0xd4, 0x62, 0xf6,
    0x43, 0x12, 0x44, 0xaf, 0x43, 0x36, 0xb1, 0x7f, 0x9d, 0xfd, 0xfe, 0xaf,
    0xa6, 0xf4, 0xff, 0xf5, 0x00, 0x4a, 0xa6, 0x5f, 0x7d, 0x6b, 0xb3, 0xb8,
    0x8a, 0xf8, 0xe9, 0xe2,
};
static const struct drbg_kat_no_reseed kat72_nor_t = {
    11, kat72_nor_entropyin, kat72_nor_nonce, kat72_nor_persstr,
    kat72_nor_addin0, kat72_nor_addin1, kat72_nor_retbytes
};
static const struct drbg_kat kat72_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat72_nor_t
};

static const unsigned char kat73_nor_entropyin[] = {
    0x89, 0xa7, 0x87, 0xbe, 0x64, 0xba, 0x33, 0x58, 0x95, 0x2f, 0x9d, 0x60,
    0x2b, 0xd8, 0xfa, 0xb4,
};
static const unsigned char kat73_nor_nonce[] = {
    0xab, 0x96, 0x1b, 0x68, 0xc6, 0x58, 0x92, 0xd3,
};
static const unsigned char kat73_nor_persstr[] = {0};
static const unsigned char kat73_nor_addin0[] = {0};
static const unsigned char kat73_nor_addin1[] = {0};
static const unsigned char kat73_nor_retbytes[] = {
    0x6e, 0x5d, 0xd2, 0xa6, 0x51, 0xf2, 0x61, 0x4f, 0xca, 0x88, 0x1f, 0x5e,
    0xdb, 0x05, 0xd5, 0xee, 0xc6, 0x12, 0x75, 0xdd, 0x05, 0xb3, 0x1f, 0xd8,
    0xa3, 0x06, 0x2f, 0xac, 0x51, 0x59, 0x60, 0xde, 0xf7, 0xb0, 0xd4, 0xdb,
    0x30, 0x4f, 0x64, 0x2b, 0x91, 0xe6, 0x40, 0x1f, 0x2a, 0x00, 0xb0, 0xfb,
    0x69, 0x3e, 0xdd, 0x73, 0x13, 0x10, 0x2f, 0x58, 0xd7, 0x33, 0x02, 0xea,
    0x8b, 0x5a, 0x9d, 0x7a,
};
static const struct drbg_kat_no_reseed kat73_nor_t = {
    12, kat73_nor_entropyin, kat73_nor_nonce, kat73_nor_persstr,
    kat73_nor_addin0, kat73_nor_addin1, kat73_nor_retbytes
};
static const struct drbg_kat kat73_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat73_nor_t
};

static const unsigned char kat74_nor_entropyin[] = {
    0x21, 0xf0, 0x2f, 0xc7, 0x5a, 0x28, 0x2d, 0x7f, 0x87, 0xca, 0xb0, 0x57,
    0x67, 0x55, 0x8e, 0x08,
};
static const unsigned char kat74_nor_nonce[] = {
    0x3b, 0x43, 0xd7, 0x7b, 0x0c, 0x64, 0x71, 0xe9,
};
static const unsigned char kat74_nor_persstr[] = {0};
static const unsigned char kat74_nor_addin0[] = {0};
static const unsigned char kat74_nor_addin1[] = {0};
static const unsigned char kat74_nor_retbytes[] = {
    0x3b, 0x87, 0x09, 0xce, 0xfe, 0x27, 0x06, 0x27, 0xeb, 0x49, 0x47, 0xf3,
    0xc7, 0x69, 0x5e, 0xed, 0x2c, 0x85, 0xcc, 0x15, 0x25, 0xbe, 0x25, 0x3f,
    0x20, 0x0a, 0x44, 0xad, 0x55, 0xf7, 0x58, 0x3e, 0x10, 0x91, 0xb8, 0x80,
    0xee, 0x16, 0xa8, 0x08, 0xe9, 0x0d, 0xc1, 0xdd, 0x5c, 0x0e, 0xd3, 0xb8,
    0xd1, 0x96, 0x40, 0x76, 0x61, 0xde, 0x80, 0x61, 0xf6, 0xa2, 0xd7, 0x8c,
    0xdc, 0x93, 0x42, 0x9d,
};
static const struct drbg_kat_no_reseed kat74_nor_t = {
    13, kat74_nor_entropyin, kat74_nor_nonce, kat74_nor_persstr,
    kat74_nor_addin0, kat74_nor_addin1, kat74_nor_retbytes
};
static const struct drbg_kat kat74_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat74_nor_t
};

static const unsigned char kat75_nor_entropyin[] = {
    0xdf, 0x5b, 0x1c, 0xd5, 0x10, 0x45, 0xc8, 0x38, 0x01, 0x53, 0x24, 0x49,
    0xd6, 0xa8, 0x7a, 0xed,
};
static const unsigned char kat75_nor_nonce[] = {
    0x8d, 0xd1, 0x72, 0xb8, 0x12, 0x86, 0xa2, 0x38,
};
static const unsigned char kat75_nor_persstr[] = {0};
static const unsigned char kat75_nor_addin0[] = {0};
static const unsigned char kat75_nor_addin1[] = {0};
static const unsigned char kat75_nor_retbytes[] = {
    0xf0, 0x17, 0x2b, 0x3a, 0x5b, 0x42, 0xea, 0xd6, 0xe6, 0x54, 0xd3, 0xab,
    0xc7, 0x8d, 0xda, 0x08, 0x78, 0x0e, 0xa4, 0x32, 0x23, 0x62, 0x57, 0x07,
    0xf8, 0xab, 0x02, 0x1b, 0xef, 0x54, 0x12, 0xdb, 0x89, 0xb0, 0x89, 0xcf,
    0x75, 0x11, 0x78, 0xeb, 0x1b, 0x97, 0xbe, 0x13, 0xf5, 0xe8, 0x3b, 0x9f,
    0x10, 0x76, 0xd0, 0xfc, 0x88, 0xe6, 0xbb, 0xd6, 0xee, 0xf1, 0xef, 0x63,
    0x55, 0xe3, 0x67, 0x2b,
};
static const struct drbg_kat_no_reseed kat75_nor_t = {
    14, kat75_nor_entropyin, kat75_nor_nonce, kat75_nor_persstr,
    kat75_nor_addin0, kat75_nor_addin1, kat75_nor_retbytes
};
static const struct drbg_kat kat75_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat75_nor_t
};

static const unsigned char kat76_nor_entropyin[] = {
    0xad, 0xf5, 0x71, 0x1f, 0x93, 0xd8, 0xc8, 0x99, 0x73, 0x49, 0x42, 0x9c,
    0xca, 0xed, 0xae, 0x0a,
};
static const unsigned char kat76_nor_nonce[] = {
    0xb2, 0x57, 0x16, 0x93, 0x1b, 0x6e, 0x3c, 0xc1,
};
static const unsigned char kat76_nor_persstr[] = {0};
static const unsigned char kat76_nor_addin0[] = {
    0xab, 0xf8, 0xcd, 0x66, 0xdd, 0x39, 0x75, 0x8b, 0x01, 0xd7, 0xdb, 0xb9,
    0x9a, 0xb1, 0x7d, 0xc3,
};
static const unsigned char kat76_nor_addin1[] = {
    0x4b, 0xe0, 0xf6, 0xb2, 0x75, 0x53, 0x77, 0xc6, 0xe8, 0x81, 0xfb, 0xb2,
    0x61, 0xb5, 0x6b, 0xeb,
};
static const unsigned char kat76_nor_retbytes[] = {
    0xd4, 0x20, 0x60, 0x4d, 0xee, 0x64, 0x67, 0x49, 0x2d, 0xb5, 0x95, 0x7c,
    0x86, 0x20, 0x7a, 0x70, 0x8f, 0xd2, 0x42, 0xed, 0x67, 0x94, 0x2a, 0xed,
    0x29, 0x94, 0x25, 0x33, 0x5c, 0x83, 0xb4, 0x14, 0x37, 0x41, 0x85, 0x82,
    0xf4, 0x1b, 0xc7, 0xfc, 0x0e, 0xf0, 0xd6, 0x92, 0x7f, 0x34, 0xd8, 0x3a,
    0xcd, 0x67, 0xc7, 0x01, 0x33, 0x64, 0x4f, 0xd7, 0x11, 0xdd, 0x5a, 0x65,
    0x73, 0x1f, 0x9f, 0x02,
};
static const struct drbg_kat_no_reseed kat76_nor_t = {
    0, kat76_nor_entropyin, kat76_nor_nonce, kat76_nor_persstr,
    kat76_nor_addin0, kat76_nor_addin1, kat76_nor_retbytes
};
static const struct drbg_kat kat76_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat76_nor_t
};

static const unsigned char kat77_nor_entropyin[] = {
    0xe9, 0x21, 0x00, 0x02, 0x07, 0x34, 0x90, 0x9d, 0x61, 0x09, 0xd2, 0x9c,
    0xf2, 0xc8, 0x98, 0xb0,
};
static const unsigned char kat77_nor_nonce[] = {
    0x91, 0xfb, 0xd9, 0xa3, 0xa8, 0xed, 0xc3, 0xa4,
};
static const unsigned char kat77_nor_persstr[] = {0};
static const unsigned char kat77_nor_addin0[] = {
    0x09, 0x80, 0xc7, 0xdb, 0xcf, 0xd0, 0xe7, 0xf7, 0x7a, 0xf8, 0x35, 0xac,
    0xc7, 0x4a, 0x4b, 0xa3,
};
static const unsigned char kat77_nor_addin1[] = {
    0x3c, 0xbf, 0x47, 0xd4, 0x66, 0x55, 0xa0, 0xce, 0x75, 0xa5, 0xd5, 0x65,
    0x28, 0x12, 0x7b, 0xc0,
};
static const unsigned char kat77_nor_retbytes[] = {
    0xe7, 0x7c, 0xcc, 0x52, 0x63, 0x62, 0x54, 0x29, 0xce, 0x7e, 0x53, 0x56,
    0x22, 0x48, 0x1d, 0x0b, 0xef, 0x47, 0x61, 0x41, 0xd1, 0xa0, 0xab, 0xfa,
    0x7f, 0x69, 0x67, 0xc1, 0x1f, 0x47, 0xf7, 0xa7, 0x0f, 0xa9, 0xbc, 0x70,
    0x59, 0x8e, 0xce, 0x3f, 0xf2, 0x83, 0xe5, 0xae, 0x04, 0xf1, 0x05, 0x35,
    0xd3, 0x49, 0xc2, 0x31, 0x34, 0x9a, 0xf8, 0x07, 0x1a, 0x22, 0x32, 0x3a,
    0x6e, 0x95, 0x10, 0x39,
};
static const struct drbg_kat_no_reseed kat77_nor_t = {
    1, kat77_nor_entropyin, kat77_nor_nonce, kat77_nor_persstr,
    kat77_nor_addin0, kat77_nor_addin1, kat77_nor_retbytes
};
static const struct drbg_kat kat77_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat77_nor_t
};

static const unsigned char kat78_nor_entropyin[] = {
    0x09, 0x2f, 0xfe, 0x74, 0x57, 0xd6, 0x17, 0x3e, 0x67, 0x13, 0xb0, 0xaf,
    0x76, 0x52, 0x1d, 0x82,
};
static const unsigned char kat78_nor_nonce[] = {
    0xd0, 0xb8, 0xca, 0xe6, 0xab, 0x9e, 0x29, 0xac,
};
static const unsigned char kat78_nor_persstr[] = {0};
static const unsigned char kat78_nor_addin0[] = {
    0xe3, 0x1a, 0xf5, 0xe1, 0x81, 0xc4, 0x34, 0x6e, 0x42, 0x3e, 0xbc, 0x1f,
    0xf9, 0xf4, 0x6d, 0xdd,
};
static const unsigned char kat78_nor_addin1[] = {
    0x02, 0xb9, 0x49, 0xd1, 0x34, 0xfc, 0x89, 0x68, 0xba, 0x51, 0x8f, 0xda,
    0x3b, 0x42, 0xc9, 0x7f,
};
static const unsigned char kat78_nor_retbytes[] = {
    0x0b, 0x64, 0xd9, 0xc6, 0x1d, 0x1f, 0x23, 0x44, 0x0c, 0xe5, 0x7d, 0xbf,
    0x46, 0x5f, 0x8c, 0x62, 0x9a, 0x9f, 0xfe, 0xdd, 0x16, 0x43, 0x99, 0xab,
    0x84, 0xfc, 0x11, 0x84, 0x66, 0xa5, 0x67, 0x9b, 0xb2, 0xd9, 0xc8, 0x05,
    0xea, 0x9d, 0x29, 0x49, 0x72, 0x61, 0xea, 0x58, 0xbd, 0x8f, 0xa4, 0x5c,
    0xf2, 0x4e, 0x21, 0x2c, 0x07, 0x7b, 0xab, 0xa4, 0xd4, 0xe5, 0x73, 0x8b,
    0xb8, 0x83, 0xc8, 0x83,
};
static const struct drbg_kat_no_reseed kat78_nor_t = {
    2, kat78_nor_entropyin, kat78_nor_nonce, kat78_nor_persstr,
    kat78_nor_addin0, kat78_nor_addin1, kat78_nor_retbytes
};
static const struct drbg_kat kat78_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat78_nor_t
};

static const unsigned char kat79_nor_entropyin[] = {
    0xa7, 0xa4, 0x78, 0x41, 0xcf, 0x1a, 0x75, 0xee, 0x96, 0x93, 0x5f, 0x52,
    0xa3, 0x32, 0x2d, 0x9d,
};
static const unsigned char kat79_nor_nonce[] = {
    0x5b, 0x48, 0xc3, 0xc3, 0x47, 0x39, 0x46, 0x56,
};
static const unsigned char kat79_nor_persstr[] = {0};
static const unsigned char kat79_nor_addin0[] = {
    0xaa, 0xc2, 0x03, 0x1f, 0x6a, 0xa4, 0x0f, 0x08, 0x75, 0x49, 0x33, 0x70,
    0x92, 0x7c, 0x7c, 0xa8,
};
static const unsigned char kat79_nor_addin1[] = {
    0x1a, 0x67, 0x44, 0xd0, 0x3f, 0xc2, 0x15, 0x12, 0x94, 0x70, 0x7c, 0x3b,
    0xce, 0xa8, 0x84, 0x02,
};
static const unsigned char kat79_nor_retbytes[] = {
    0xb4, 0x67, 0x48, 0x67, 0xcb, 0x6c, 0xd4, 0xfa, 0xde, 0x78, 0x46, 0xf4,
    0x92, 0xbd, 0x9e, 0xca, 0x69, 0xc1, 0x33, 0xf1, 0x17, 0x1c, 0xc6, 0x0b,
    0x6c, 0xd7, 0x5f, 0x22, 0xe2, 0xba, 0x7e, 0x70, 0x20, 0x7c, 0xa3, 0x8b,
    0xdc, 0xef, 0x17, 0xac, 0x15, 0xe4, 0x88, 0x49, 0xba, 0x13, 0x0d, 0x5f,
    0xb8, 0x10, 0x84, 0x4c, 0x25, 0x39, 0xfe, 0xc5, 0x19, 0xc4, 0xfa, 0x97,
    0x7f, 0xfa, 0x07, 0x38,
};
static const struct drbg_kat_no_reseed kat79_nor_t = {
    3, kat79_nor_entropyin, kat79_nor_nonce, kat79_nor_persstr,
    kat79_nor_addin0, kat79_nor_addin1, kat79_nor_retbytes
};
static const struct drbg_kat kat79_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat79_nor_t
};

static const unsigned char kat80_nor_entropyin[] = {
    0x52, 0x89, 0xc3, 0x33, 0xce, 0x1c, 0xee, 0xf5, 0x6a, 0x19, 0xa1, 0x11,
    0xe9, 0x20, 0x1a, 0x38,
};
static const unsigned char kat80_nor_nonce[] = {
    0xdb, 0xd6, 0xb7, 0x6b, 0x1f, 0xdf, 0x5d, 0x1b,
};
static const unsigned char kat80_nor_persstr[] = {0};
static const unsigned char kat80_nor_addin0[] = {
    0xf1, 0xfa, 0x6e, 0xc8, 0xca, 0xf9, 0x4f, 0x65, 0xf1, 0x2e, 0x9c, 0x0d,
    0x8f, 0x57, 0x04, 0xf6,
};
static const unsigned char kat80_nor_addin1[] = {
    0x6e, 0xb6, 0xa5, 0x57, 0xf1, 0x70, 0x18, 0x83, 0x0b, 0x5d, 0x72, 0xb6,
    0x7d, 0x32, 0x22, 0x63,
};
static const unsigned char kat80_nor_retbytes[] = {
    0x39, 0xec, 0x96, 0xbc, 0x71, 0x6e, 0x1d, 0x21, 0x48, 0x80, 0xd1, 0xf4,
    0xb8, 0xa9, 0x84, 0xbb, 0x83, 0x39, 0x94, 0x53, 0xec, 0x1e, 0xb9, 0xf3,
    0x71, 0x28, 0x2d, 0x65, 0x8e, 0x32, 0xd7, 0x39, 0xb4, 0x89, 0xf4, 0xc5,
    0xee, 0xae, 0x62, 0xa8, 0xfe, 0xe8, 0xe0, 0x84, 0xee, 0xae, 0x6d, 0x01,
    0xcc, 0xa0, 0xad, 0x80, 0xda, 0x1c, 0x53, 0x23, 0x0f, 0x8a, 0x1f, 0xaf,
    0x7c, 0x33, 0x5e, 0xdf,
};
static const struct drbg_kat_no_reseed kat80_nor_t = {
    4, kat80_nor_entropyin, kat80_nor_nonce, kat80_nor_persstr,
    kat80_nor_addin0, kat80_nor_addin1, kat80_nor_retbytes
};
static const struct drbg_kat kat80_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat80_nor_t
};

static const unsigned char kat81_nor_entropyin[] = {
    0x4f, 0x05, 0xef, 0x26, 0xc6, 0x85, 0x44, 0x4d, 0x6e, 0x65, 0xd8, 0xf2,
    0x6e, 0x0f, 0x2e, 0xf2,
};
static const unsigned char kat81_nor_nonce[] = {
    0xd0, 0xab, 0x06, 0x6c, 0x7a, 0x49, 0x03, 0xb7,
};
static const unsigned char kat81_nor_persstr[] = {0};
static const unsigned char kat81_nor_addin0[] = {
    0xd2, 0x75, 0x42, 0x60, 0x43, 0xa8, 0x2d, 0xe2, 0x0f, 0x89, 0xf1, 0xcb,
    0x2c, 0xff, 0x5c, 0x9b,
};
static const unsigned char kat81_nor_addin1[] = {
    0x02, 0xa3, 0x27, 0x63, 0xed, 0x9a, 0xab, 0x16, 0x72, 0x82, 0xd0, 0xa0,
    0x97, 0xf3, 0x10, 0x3e,
};
static const unsigned char kat81_nor_retbytes[] = {
    0xaf, 0xa2, 0xd3, 0x7c, 0xc1, 0xc7, 0x9d, 0x20, 0xa6, 0x10, 0x4d, 0x67,
    0x29, 0xfc, 0x75, 0x44, 0x36, 0x1b, 0x95, 0xd2, 0x36, 0x33, 0x48, 0x7e,
    0x8b, 0xb2, 0x82, 0xb2, 0x77, 0xe2, 0x5f, 0x8a, 0xd5, 0x31, 0x36, 0x10,
    0x9b, 0x21, 0xc3, 0x80, 0xaa, 0x05, 0xe4, 0x1d, 0xd9, 0xef, 0xd8, 0xad,
    0xf4, 0x6b, 0x7d, 0x69, 0x5b, 0xba, 0xf5, 0xb0, 0x30, 0x72, 0x16, 0xa1,
    0x94, 0xfd, 0x91, 0x28,
};
static const struct drbg_kat_no_reseed kat81_nor_t = {
    5, kat81_nor_entropyin, kat81_nor_nonce, kat81_nor_persstr,
    kat81_nor_addin0, kat81_nor_addin1, kat81_nor_retbytes
};
static const struct drbg_kat kat81_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat81_nor_t
};

static const unsigned char kat82_nor_entropyin[] = {
    0x0c, 0x2c, 0xcf, 0xd6, 0x5c, 0x23, 0xbc, 0x06, 0x93, 0xc9, 0xd4, 0x66,
    0x7a, 0x94, 0x2b, 0x42,
};
static const unsigned char kat82_nor_nonce[] = {
    0xc0, 0x8d, 0x55, 0x22, 0x8f, 0xe8, 0x4b, 0xd7,
};
static const unsigned char kat82_nor_persstr[] = {0};
static const unsigned char kat82_nor_addin0[] = {
    0xcc, 0x01, 0x7b, 0xb7, 0x96, 0x2d, 0xa6, 0xdb, 0x0f, 0x51, 0x7b, 0xe9,
    0x18, 0x3f, 0xa6, 0xf4,
};
static const unsigned char kat82_nor_addin1[] = {
    0xc4, 0x24, 0xaa, 0x0c, 0xe8, 0x4a, 0x80, 0x7a, 0x78, 0xb7, 0xb7, 0x32,
    0x11, 0x3c, 0x22, 0xa1,
};
static const unsigned char kat82_nor_retbytes[] = {
    0xcb, 0x55, 0x6f, 0x6d, 0x22, 0x3a, 0x51, 0xa2, 0x00, 0xd7, 0x35, 0xf2,
    0xf4, 0xba, 0x21, 0x7a, 0x67, 0x2e, 0xf6, 0xbc, 0x14, 0x51, 0x97, 0x6d,
    0x1d, 0x05, 0xb4, 0x4a, 0x2c, 0x0d, 0x54, 0x27, 0x8a, 0xf7, 0x47, 0x83,
    0x79, 0x22, 0xcd, 0x0b, 0x28, 0xd7, 0xaa, 0xa2, 0xa8, 0x7e, 0x23, 0x05,
    0x5e, 0x18, 0xd6, 0x7d, 0x76, 0xfa, 0x90, 0x72, 0x82, 0x94, 0xab, 0xa4,
    0xac, 0x36, 0x96, 0x5e,
};
static const struct drbg_kat_no_reseed kat82_nor_t = {
    6, kat82_nor_entropyin, kat82_nor_nonce, kat82_nor_persstr,
    kat82_nor_addin0, kat82_nor_addin1, kat82_nor_retbytes
};
static const struct drbg_kat kat82_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat82_nor_t
};

static const unsigned char kat83_nor_entropyin[] = {
    0x3b, 0x74, 0x81, 0x3e, 0xeb, 0xf5, 0xf5, 0xf2, 0xa6, 0xa5, 0xa4, 0xf3,
    0xf9, 0x07, 0xdc, 0xb8,
};
static const unsigned char kat83_nor_nonce[] = {
    0x35, 0xda, 0xa2, 0xef, 0xda, 0xd8, 0x8f, 0xb8,
};
static const unsigned char kat83_nor_persstr[] = {0};
static const unsigned char kat83_nor_addin0[] = {
    0x5e, 0x0a, 0xe3, 0x69, 0xb6, 0xe8, 0x50, 0xed, 0x30, 0xf9, 0xae, 0x64,
    0xfe, 0x10, 0x5a, 0xf3,
};
static const unsigned char kat83_nor_addin1[] = {
    0x3e, 0x69, 0x7b, 0xb2, 0xbf, 0x25, 0x85, 0x26, 0x45, 0xf2, 0xd1, 0x6f,
    0x67, 0x6f, 0x90, 0x78,
};
static const unsigned char kat83_nor_retbytes[] = {
    0x79, 0x0b, 0x1f, 0x6c, 0xcd, 0xf3, 0x52, 0xde, 0xc0, 0xe5, 0x7f, 0xa0,
    0xc7, 0x72, 0xc6, 0x4d, 0x77, 0x77, 0xe4, 0xbd, 0xb9, 0xff, 0x8a, 0x91,
    0xda, 0x01, 0x89, 0x02, 0x20, 0x3d, 0xb5, 0x82, 0xf9, 0x01, 0xc2, 0x0d,
    0xd2, 0x53, 0x61, 0xdf, 0x22, 0xe7, 0x17, 0x12, 0xa4, 0x97, 0x57, 0x8d,
    0x22, 0xee, 0xa8, 0x52, 0x46, 0xce, 0xfd, 0x42, 0x5b, 0x08, 0xf1, 0x50,
    0x74, 0x34, 0x1f, 0x39,
};
static const struct drbg_kat_no_reseed kat83_nor_t = {
    7, kat83_nor_entropyin, kat83_nor_nonce, kat83_nor_persstr,
    kat83_nor_addin0, kat83_nor_addin1, kat83_nor_retbytes
};
static const struct drbg_kat kat83_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat83_nor_t
};

static const unsigned char kat84_nor_entropyin[] = {
    0x46, 0x26, 0x07, 0xb2, 0xb7, 0x7b, 0x90, 0x3f, 0xdb, 0xd4, 0x2b, 0xf1,
    0xd7, 0x7b, 0xc8, 0x1a,
};
static const unsigned char kat84_nor_nonce[] = {
    0xb5, 0x4d, 0x43, 0x00, 0x6a, 0xde, 0x2a, 0x7c,
};
static const unsigned char kat84_nor_persstr[] = {0};
static const unsigned char kat84_nor_addin0[] = {
    0x61, 0x05, 0xc5, 0xdd, 0xed, 0xcc, 0x74, 0x46, 0x65, 0x91, 0x5e, 0xbe,
    0x48, 0xad, 0x3b, 0x54,
};
static const unsigned char kat84_nor_addin1[] = {
    0xfd, 0xde, 0xe2, 0x35, 0xc3, 0x7e, 0xd4, 0xbc, 0x21, 0xea, 0xe0, 0x8e,
    0x61, 0x53, 0x33, 0x1f,
};
static const unsigned char kat84_nor_retbytes[] = {
    0x16, 0x33, 0x9e, 0x91, 0x0e, 0xaf, 0x9b, 0xad, 0x90, 0x4a, 0x7b, 0x3f,
    0x9b, 0x2e, 0x02, 0x41, 0xd2, 0xb4, 0x66, 0x5d, 0x1f, 0x9c, 0x76, 0x55,
    0xf9, 0x21, 0x7e, 0x34, 0x25, 0x3a, 0xcc, 0x33, 0x0d, 0xfc, 0x98, 0x81,
    0x7d, 0x97, 0x81, 0x86, 0xf8, 0xad, 0x77, 0xb1, 0x0f, 0x38, 0xdd, 0x72,
    0xf4, 0x71, 0xb4, 0x61, 0x5b, 0x28, 0xf3, 0xa3, 0x9c, 0x40, 0x66, 0x23,
    0xea, 0x5d, 0x63, 0x77,
};
static const struct drbg_kat_no_reseed kat84_nor_t = {
    8, kat84_nor_entropyin, kat84_nor_nonce, kat84_nor_persstr,
    kat84_nor_addin0, kat84_nor_addin1, kat84_nor_retbytes
};
static const struct drbg_kat kat84_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat84_nor_t
};

static const unsigned char kat85_nor_entropyin[] = {
    0xba, 0x2e, 0xa9, 0x9b, 0x79, 0x2b, 0xde, 0xe8, 0xbd, 0xba, 0xf9, 0xc0,
    0x5a, 0xb7, 0xb6, 0x33,
};
static const unsigned char kat85_nor_nonce[] = {
    0x44, 0x53, 0x34, 0x46, 0x44, 0xd4, 0x34, 0xa5,
};
static const unsigned char kat85_nor_persstr[] = {0};
static const unsigned char kat85_nor_addin0[] = {
    0x81, 0x72, 0x7d, 0x30, 0xd6, 0x4a, 0xe0, 0x7f, 0xf6, 0x15, 0x69, 0xfc,
    0x7a, 0x35, 0x0c, 0xcc,
};
static const unsigned char kat85_nor_addin1[] = {
    0x65, 0x56, 0xf6, 0x2f, 0x75, 0xd5, 0x26, 0x84, 0x93, 0xcc, 0xa5, 0x12,
    0x01, 0x14, 0x93, 0x4c,
};
static const unsigned char kat85_nor_retbytes[] = {
    0x2a, 0x3b, 0x39, 0x87, 0x02, 0xbc, 0x5f, 0xff, 0xd9, 0x14, 0xf9, 0x16,
    0x72, 0xaa, 0x05, 0xbe, 0xb6, 0x87, 0x82, 0xe0, 0xa6, 0x22, 0xf6, 0x7b,
    0x21, 0x82, 0x6c, 0x23, 0x32, 0x6c, 0x50, 0x78, 0xcc, 0xda, 0x6a, 0xa3,
    0xdb, 0x01, 0x4c, 0xec, 0x56, 0xa0, 0x65, 0x63, 0x6b, 0x80, 0x57, 0xe8,
    0x12, 0xa6, 0xf4, 0xb7, 0xc6, 0xc4, 0xde, 0x6e, 0x9f, 0x06, 0xaa, 0x53,
    0x77, 0xb2, 0x8b, 0xe6,
};
static const struct drbg_kat_no_reseed kat85_nor_t = {
    9, kat85_nor_entropyin, kat85_nor_nonce, kat85_nor_persstr,
    kat85_nor_addin0, kat85_nor_addin1, kat85_nor_retbytes
};
static const struct drbg_kat kat85_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat85_nor_t
};

static const unsigned char kat86_nor_entropyin[] = {
    0x61, 0xcf, 0x4e, 0x09, 0xd9, 0xa4, 0x6a, 0xf3, 0x18, 0xc7, 0x06, 0x42,
    0x5c, 0xd3, 0x50, 0xfb,
};
static const unsigned char kat86_nor_nonce[] = {
    0x28, 0xec, 0x9c, 0x4a, 0xe9, 0x8f, 0x05, 0x74,
};
static const unsigned char kat86_nor_persstr[] = {0};
static const unsigned char kat86_nor_addin0[] = {
    0x7f, 0xc3, 0xbf, 0x8a, 0xd3, 0x3e, 0x59, 0x61, 0x74, 0x5b, 0xb2, 0xae,
    0x7c, 0x39, 0x9b, 0xa0,
};
static const unsigned char kat86_nor_addin1[] = {
    0x31, 0x35, 0x71, 0xd3, 0x0d, 0xc5, 0x90, 0x9a, 0x03, 0x38, 0x8c, 0x69,
    0x3f, 0x4e, 0xf9, 0x4c,
};
static const unsigned char kat86_nor_retbytes[] = {
    0xd2, 0x25, 0xf0, 0xc6, 0x5e, 0x9a, 0xe8, 0xc3, 0x64, 0x93, 0xb0, 0x09,
    0x47, 0x6f, 0x5a, 0x56, 0x36, 0xd4, 0x52, 0x54, 0x4c, 0xe1, 0x6b, 0x99,
    0x1c, 0x73, 0xd6, 0xea, 0xe5, 0xef, 0x5e, 0xff, 0x3b, 0x74, 0xe0, 0x03,
    0x28, 0xf8, 0x7f, 0x20, 0x30, 0xf3, 0xdf, 0x53, 0xb2, 0xfc, 0x26, 0x47,
    0xd6, 0x3a, 0x28, 0xe5, 0xc7, 0xb5, 0x83, 0x44, 0xed, 0x99, 0xc5, 0xd0,
    0xbc, 0x62, 0x93, 0xb4,
};
static const struct drbg_kat_no_reseed kat86_nor_t = {
    10, kat86_nor_entropyin, kat86_nor_nonce, kat86_nor_persstr,
    kat86_nor_addin0, kat86_nor_addin1, kat86_nor_retbytes
};
static const struct drbg_kat kat86_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat86_nor_t
};

static const unsigned char kat87_nor_entropyin[] = {
    0xdd, 0x6c, 0x7d, 0xba, 0xc4, 0xfc, 0x4c, 0x8d, 0x9c, 0x69, 0xce, 0x4d,
    0xcf, 0x43, 0xb6, 0x67,
};
static const unsigned char kat87_nor_nonce[] = {
    0x83, 0x26, 0x16, 0x1c, 0x2a, 0xac, 0x48, 0x06,
};
static const unsigned char kat87_nor_persstr[] = {0};
static const unsigned char kat87_nor_addin0[] = {
    0xcf, 0x95, 0x9f, 0x9e, 0x93, 0x89, 0x09, 0xd9, 0x5c, 0x47, 0x25, 0xdc,
    0x39, 0xd7, 0xef, 0xd7,
};
static const unsigned char kat87_nor_addin1[] = {
    0xe4, 0x37, 0xd2, 0x26, 0xc4, 0x02, 0x68, 0xcb, 0x9f, 0x63, 0x94, 0x5c,
    0xbc, 0xe5, 0x4b, 0x2d,
};
static const unsigned char kat87_nor_retbytes[] = {
    0x90, 0xdd, 0xc5, 0x3c, 0x48, 0x04, 0x1f, 0x0c, 0x45, 0x01, 0xff, 0xee,
    0x61, 0x7b, 0x3a, 0x35, 0x83, 0xcb, 0xf5, 0x1a, 0x4c, 0x45, 0xad, 0x7b,
    0xa5, 0xb3, 0xdd, 0x34, 0x65, 0xff, 0x13, 0x92, 0xbd, 0x2e, 0xb4, 0x65,
    0x1a, 0xc2, 0x2a, 0x34, 0xa0, 0xf5, 0x37, 0x72, 0x01, 0x27, 0x12, 0xf2,
    0x39, 0x31, 0xba, 0x8f, 0x0a, 0x3b, 0x74, 0xfa, 0xe0, 0xa1, 0x6b, 0xd4,
    0xa1, 0x88, 0x35, 0x13,
};
static const struct drbg_kat_no_reseed kat87_nor_t = {
    11, kat87_nor_entropyin, kat87_nor_nonce, kat87_nor_persstr,
    kat87_nor_addin0, kat87_nor_addin1, kat87_nor_retbytes
};
static const struct drbg_kat kat87_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat87_nor_t
};

static const unsigned char kat88_nor_entropyin[] = {
    0xdf, 0x4f, 0xef, 0xc1, 0xc3, 0xd4, 0x03, 0xfb, 0x17, 0x9b, 0x9b, 0x95,
    0x77, 0x53, 0x81, 0xe3,
};
static const unsigned char kat88_nor_nonce[] = {
    0xd6, 0x84, 0x83, 0xae, 0xe4, 0x2c, 0xfc, 0x00,
};
static const unsigned char kat88_nor_persstr[] = {0};
static const unsigned char kat88_nor_addin0[] = {
    0xe8, 0x77, 0xeb, 0x37, 0x51, 0xe2, 0xba, 0xfe, 0xa0, 0x33, 0xac, 0x8c,
    0xc5, 0xba, 0x66, 0xf6,
};
static const unsigned char kat88_nor_addin1[] = {
    0x79, 0x56, 0x87, 0x40, 0x22, 0xb5, 0x0a, 0xa0, 0x32, 0xba, 0xfa, 0x29,
    0xe7, 0x21, 0xf3, 0x56,
};
static const unsigned char kat88_nor_retbytes[] = {
    0x6a, 0xda, 0x06, 0x20, 0xd4, 0x29, 0xcd, 0x38, 0x75, 0x4e, 0x55, 0x9a,
    0x2e, 0xb8, 0x8c, 0x08, 0x26, 0x6c, 0xd9, 0xcb, 0xac, 0x24, 0x48, 0xb2,
    0xb8, 0xa0, 0x5d, 0x92, 0x7d, 0xa1, 0x72, 0x59, 0x19, 0x9d, 0xbc, 0xc5,
    0xeb, 0x0b, 0x7a, 0xde, 0xe6, 0x1c, 0x60, 0x74, 0xae, 0x2e, 0x7a, 0xf6,
    0xb2, 0xc9, 0xb1, 0xfd, 0x52, 0x89, 0xfb, 0xc3, 0xd9, 0xbe, 0x30, 0x85,
    0x84, 0xd0, 0xae, 0x4a,
};
static const struct drbg_kat_no_reseed kat88_nor_t = {
    12, kat88_nor_entropyin, kat88_nor_nonce, kat88_nor_persstr,
    kat88_nor_addin0, kat88_nor_addin1, kat88_nor_retbytes
};
static const struct drbg_kat kat88_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat88_nor_t
};

static const unsigned char kat89_nor_entropyin[] = {
    0x62, 0x0e, 0xa3, 0x17, 0x34, 0x4d, 0x86, 0x85, 0x8d, 0x64, 0xf6, 0xbb,
    0x05, 0xd4, 0x47, 0x97,
};
static const unsigned char kat89_nor_nonce[] = {
    0xf6, 0xdd, 0xc5, 0x84, 0x4f, 0xbd, 0x73, 0x9e,
};
static const unsigned char kat89_nor_persstr[] = {0};
static const unsigned char kat89_nor_addin0[] = {
    0x3c, 0xcf, 0xb0, 0x69, 0xef, 0x27, 0xf8, 0x64, 0xe3, 0xb7, 0x31, 0xd9,
    0x4c, 0x3f, 0x2c, 0x6c,
};
static const unsigned char kat89_nor_addin1[] = {
    0x80, 0x85, 0x7d, 0x3f, 0x59, 0x8b, 0x1b, 0x16, 0x01, 0x4e, 0x6a, 0x35,
    0x49, 0xa2, 0xd2, 0x2c,
};
static const unsigned char kat89_nor_retbytes[] = {
    0xd7, 0xa7, 0x87, 0xbe, 0xeb, 0x7e, 0xfc, 0x9c, 0x58, 0x4f, 0x93, 0x34,
    0x55, 0x33, 0xa5, 0x99, 0xed, 0x61, 0x5e, 0xb2, 0x13, 0x6b, 0x8f, 0xb8,
    0x74, 0xa7, 0xf5, 0xa7, 0x1a, 0xea, 0x98, 0xf8, 0x0c, 0x56, 0x92, 0x30,
    0x98, 0x56, 0x7a, 0xa3, 0xb4, 0xc3, 0x16, 0x96, 0x82, 0x6f, 0x08, 0x16,
    0x76, 0xa3, 0xce, 0xed, 0x1d, 0x91, 0x4c, 0x94, 0x9d, 0x54, 0x23, 0x90,
    0xb6, 0x99, 0xd3, 0xfe,
};
static const struct drbg_kat_no_reseed kat89_nor_t = {
    13, kat89_nor_entropyin, kat89_nor_nonce, kat89_nor_persstr,
    kat89_nor_addin0, kat89_nor_addin1, kat89_nor_retbytes
};
static const struct drbg_kat kat89_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat89_nor_t
};

static const unsigned char kat90_nor_entropyin[] = {
    0xd8, 0xba, 0x9d, 0x16, 0xa0, 0x07, 0x0f, 0xbd, 0xf5, 0x10, 0x9d, 0x3c,
    0x1e, 0xd8, 0x59, 0x16,
};
static const unsigned char kat90_nor_nonce[] = {
    0x29, 0xce, 0x07, 0xdd, 0xa6, 0x1b, 0x50, 0x14,
};
static const unsigned char kat90_nor_persstr[] = {0};
static const unsigned char kat90_nor_addin0[] = {
    0x16, 0x0a, 0xd7, 0x76, 0x75, 0xc6, 0x4c, 0x74, 0x87, 0xee, 0xdb, 0x34,
    0x34, 0x37, 0x00, 0xc0,
};
static const unsigned char kat90_nor_addin1[] = {
    0x16, 0x32, 0x54, 0x60, 0x58, 0xf1, 0x9e, 0xe1, 0xb8, 0x49, 0xe5, 0xd2,
    0x7e, 0x2d, 0x75, 0x5a,
};
static const unsigned char kat90_nor_retbytes[] = {
    0xe0, 0x5c, 0x62, 0xcb, 0xef, 0xaf, 0x03, 0x80, 0x43, 0xd4, 0x9c, 0x97,
    0x6b, 0xa8, 0x55, 0x21, 0xc2, 0x06, 0x71, 0xd9, 0xf8, 0x1c, 0xeb, 0xf5,
    0xa8, 0x75, 0x81, 0x65, 0x9e, 0x3d, 0x18, 0x19, 0x90, 0x8e, 0x84, 0xec,
    0x01, 0x93, 0xa5, 0x30, 0xf3, 0x10, 0xcc, 0x35, 0x44, 0xba, 0x90, 0xad,
    0x92, 0xa6, 0xc9, 0x9d, 0x3a, 0x1a, 0xf5, 0x5b, 0x5a, 0xed, 0x72, 0x87,
    0x1d, 0xb8, 0x90, 0x02,
};
static const struct drbg_kat_no_reseed kat90_nor_t = {
    14, kat90_nor_entropyin, kat90_nor_nonce, kat90_nor_persstr,
    kat90_nor_addin0, kat90_nor_addin1, kat90_nor_retbytes
};
static const struct drbg_kat kat90_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat90_nor_t
};

static const unsigned char kat91_nor_entropyin[] = {
    0x0a, 0xf1, 0x3f, 0x64, 0x59, 0x02, 0xaf, 0x49, 0xe9, 0xa7, 0xce, 0xd6,
    0xe3, 0x6a, 0x21, 0x0d,
};
static const unsigned char kat91_nor_nonce[] = {
    0xc3, 0xbf, 0xf2, 0x91, 0xa1, 0x1a, 0xc4, 0x97,
};
static const unsigned char kat91_nor_persstr[] = {
    0xe8, 0xf1, 0xd1, 0xb4, 0x73, 0x1c, 0x4d, 0x57, 0xd7, 0xea, 0xd9, 0xc2,
    0xf6, 0x00, 0xfd, 0xc6,
};
static const unsigned char kat91_nor_addin0[] = {0};
static const unsigned char kat91_nor_addin1[] = {0};
static const unsigned char kat91_nor_retbytes[] = {
    0xac, 0x6f, 0x94, 0x5a, 0x4b, 0x9f, 0xd3, 0xb4, 0x7c, 0x74, 0x37, 0x9e,
    0xb1, 0xf2, 0xa7, 0xbb, 0xed, 0xf8, 0xee, 0xc6, 0x8e, 0xfd, 0x3c, 0x7a,
    0x6c, 0xf6, 0x8c, 0x54, 0xae, 0x7a, 0x3f, 0x7b, 0xe7, 0x28, 0x0f, 0x45,
    0x9c, 0x2e, 0x0b, 0x72, 0xaf, 0xa4, 0x5c, 0xeb, 0xbe, 0xbb, 0xa1, 0x7c,
    0x86, 0x7e, 0x96, 0x11, 0xc8, 0x96, 0xa5, 0x7d, 0x51, 0x5b, 0xeb, 0x06,
    0xa7, 0xb9, 0x1f, 0x4c,
};
static const struct drbg_kat_no_reseed kat91_nor_t = {
    0, kat91_nor_entropyin, kat91_nor_nonce, kat91_nor_persstr,
    kat91_nor_addin0, kat91_nor_addin1, kat91_nor_retbytes
};
static const struct drbg_kat kat91_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat91_nor_t
};

static const unsigned char kat92_nor_entropyin[] = {
    0x0e, 0x11, 0x3f, 0x47, 0xf2, 0xfc, 0x76, 0xe8, 0x3e, 0x2d, 0x13, 0xd5,
    0x72, 0x24, 0x56, 0x08,
};
static const unsigned char kat92_nor_nonce[] = {
    0x5f, 0xf8, 0x5c, 0xc6, 0xa5, 0x34, 0xf1, 0x5a,
};
static const unsigned char kat92_nor_persstr[] = {
    0x50, 0x25, 0x06, 0x68, 0xe5, 0x9d, 0xe3, 0x5f, 0xde, 0x91, 0xe0, 0x8f,
    0xe1, 0x84, 0x84, 0xab,
};
static const unsigned char kat92_nor_addin0[] = {0};
static const unsigned char kat92_nor_addin1[] = {0};
static const unsigned char kat92_nor_retbytes[] = {
    0xdc, 0xc6, 0x4a, 0x96, 0x6a, 0x52, 0xd6, 0x00, 0x8d, 0xbe, 0x07, 0xa2,
    0x48, 0x4b, 0xca, 0xad, 0x67, 0xb2, 0x54, 0xd6, 0xf2, 0x46, 0xe4, 0x50,
    0x1d, 0x98, 0x64, 0xb6, 0x4a, 0xd8, 0xb7, 0xed, 0xf1, 0x0f, 0xdb, 0xc6,
    0xdd, 0xc4, 0x14, 0xa9, 0xb4, 0x31, 0xb0, 0x58, 0xa7, 0xee, 0x5c, 0xed,
    0x23, 0xf7, 0xa6, 0xac, 0x7e, 0xea, 0x0f, 0xe6, 0x13, 0x1c, 0x9e, 0xb7,
    0x41, 0x2e, 0x68, 0xdf,
};
static const struct drbg_kat_no_reseed kat92_nor_t = {
    1, kat92_nor_entropyin, kat92_nor_nonce, kat92_nor_persstr,
    kat92_nor_addin0, kat92_nor_addin1, kat92_nor_retbytes
};
static const struct drbg_kat kat92_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat92_nor_t
};

static const unsigned char kat93_nor_entropyin[] = {
    0x16, 0xd2, 0xa2, 0x33, 0xf0, 0x49, 0x7c, 0x75, 0x1d, 0xdd, 0x85, 0xef,
    0x7c, 0xd8, 0x62, 0xe6,
};
static const unsigned char kat93_nor_nonce[] = {
    0x7e, 0x7e, 0x1a, 0xab, 0x1b, 0x82, 0x67, 0x5d,
};
static const unsigned char kat93_nor_persstr[] = {
    0x15, 0xf8, 0x1e, 0x40, 0xa2, 0x49, 0x3b, 0x85, 0x9c, 0xce, 0xd3, 0x3c,
    0xdf, 0x7b, 0x75, 0x90,
};
static const unsigned char kat93_nor_addin0[] = {0};
static const unsigned char kat93_nor_addin1[] = {0};
static const unsigned char kat93_nor_retbytes[] = {
    0xc5, 0x20, 0x97, 0xd2, 0xd0, 0x09, 0xdc, 0xe7, 0xcb, 0xd4, 0x6d, 0x74,
    0x0d, 0xe5, 0x4d, 0x70, 0xc0, 0x73, 0x2f, 0x96, 0xfa, 0xc0, 0xbd, 0x16,
    0x9f, 0x38, 0x56, 0xe5, 0x0f, 0x9e, 0x46, 0x01, 0x67, 0x25, 0x38, 0xb2,
    0x3f, 0x37, 0x1f, 0x78, 0x0e, 0x61, 0xe8, 0x2f, 0x2d, 0xb9, 0x9e, 0xf3,
    0x83, 0x40, 0x08, 0xaa, 0xdc, 0x26, 0xc2, 0x78, 0x55, 0xa6, 0x86, 0x96,
    0xee, 0x81, 0x2c, 0x9c,
};
static const struct drbg_kat_no_reseed kat93_nor_t = {
    2, kat93_nor_entropyin, kat93_nor_nonce, kat93_nor_persstr,
    kat93_nor_addin0, kat93_nor_addin1, kat93_nor_retbytes
};
static const struct drbg_kat kat93_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat93_nor_t
};

static const unsigned char kat94_nor_entropyin[] = {
    0xea, 0x36, 0x89, 0x1e, 0x0d, 0x64, 0x26, 0x94, 0x5a, 0x6e, 0x1f, 0x33,
    0x8c, 0x86, 0xca, 0x4b,
};
static const unsigned char kat94_nor_nonce[] = {
    0x89, 0x69, 0x66, 0x60, 0x0e, 0x85, 0xd3, 0xda,
};
static const unsigned char kat94_nor_persstr[] = {
    0xb8, 0xc7, 0x01, 0xa1, 0x1e, 0x7c, 0x00, 0x8e, 0x37, 0xc2, 0x32, 0x62,
    0x7c, 0x24, 0xb6, 0xd1,
};
static const unsigned char kat94_nor_addin0[] = {0};
static const unsigned char kat94_nor_addin1[] = {0};
static const unsigned char kat94_nor_retbytes[] = {
    0x44, 0x63, 0x60, 0xf1, 0x6b, 0x1e, 0x59, 0xa8, 0x25, 0x96, 0xe6, 0x1d,
    0xf2, 0x3a, 0xf1, 0x8f, 0x5a, 0x95, 0xb7, 0xd8, 0xb7, 0xdd, 0x61, 0x72,
    0xf3, 0x57, 0x5d, 0x45, 0x25, 0x13, 0x86, 0xb0, 0x4b, 0x69, 0x78, 0x5b,
    0xa4, 0x4a, 0x75, 0xeb, 0x2a, 0x54, 0x15, 0x71, 0x3a, 0x84, 0x36, 0x5c,
    0xac, 0x81, 0xb3, 0x0d, 0xf0, 0xb4, 0x8a, 0x61, 0x09, 0x77, 0xfe, 0x69,
    0x61, 0xe7, 0xee, 0x8a,
};
static const struct drbg_kat_no_reseed kat94_nor_t = {
    3, kat94_nor_entropyin, kat94_nor_nonce, kat94_nor_persstr,
    kat94_nor_addin0, kat94_nor_addin1, kat94_nor_retbytes
};
static const struct drbg_kat kat94_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat94_nor_t
};

static const unsigned char kat95_nor_entropyin[] = {
    0x3f, 0x41, 0x38, 0xd8, 0x4d, 0x7a, 0xbd, 0x13, 0xbb, 0xe1, 0x5e, 0x18,
    0xb2, 0x92, 0xf1, 0x5f,
};
static const unsigned char kat95_nor_nonce[] = {
    0x34, 0x62, 0xe5, 0x2e, 0x82, 0xf2, 0x86, 0x53,
};
static const unsigned char kat95_nor_persstr[] = {
    0x24, 0xc5, 0x17, 0x70, 0x84, 0x95, 0x1c, 0xf5, 0x32, 0xf2, 0x43, 0x2a,
    0x18, 0x82, 0x59, 0x6f,
};
static const unsigned char kat95_nor_addin0[] = {0};
static const unsigned char kat95_nor_addin1[] = {0};
static const unsigned char kat95_nor_retbytes[] = {
    0xba, 0x2c, 0xa1, 0xe4, 0x43, 0xc7, 0x5d, 0x55, 0xc4, 0x20, 0x65, 0xba,
    0x91, 0xf7, 0x7d, 0xb5, 0xb8, 0xde, 0x6c, 0x34, 0x2b, 0x65, 0xa2, 0xc1,
    0x49, 0x44, 0x59, 0x79, 0xff, 0x26, 0x3e, 0x9d, 0x01, 0x8f, 0x56, 0x44,
    0x30, 0xfe, 0x1e, 0xdb, 0x48, 0xb4, 0x03, 0xce, 0xf2, 0x3f, 0x86, 0x0e,
    0xa2, 0x71, 0x46, 0xd8, 0x51, 0x18, 0x43, 0xe0, 0xa3, 0x9f, 0xf3, 0x37,
    0xba, 0x74, 0xd2, 0x21,
};
static const struct drbg_kat_no_reseed kat95_nor_t = {
    4, kat95_nor_entropyin, kat95_nor_nonce, kat95_nor_persstr,
    kat95_nor_addin0, kat95_nor_addin1, kat95_nor_retbytes
};
static const struct drbg_kat kat95_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat95_nor_t
};

static const unsigned char kat96_nor_entropyin[] = {
    0xa6, 0xae, 0xf8, 0x0f, 0xab, 0x61, 0xe2, 0x7e, 0xf6, 0x11, 0xb8, 0x6f,
    0x92, 0x4e, 0x97, 0x6c,
};
static const unsigned char kat96_nor_nonce[] = {
    0x74, 0x27, 0x8f, 0x50, 0x23, 0xc2, 0x55, 0x2c,
};
static const unsigned char kat96_nor_persstr[] = {
    0xa3, 0x85, 0x16, 0x79, 0xe1, 0x12, 0x94, 0x82, 0x01, 0x39, 0x6d, 0x98,
    0xd6, 0xbe, 0x5c, 0x13,
};
static const unsigned char kat96_nor_addin0[] = {0};
static const unsigned char kat96_nor_addin1[] = {0};
static const unsigned char kat96_nor_retbytes[] = {
    0x68, 0x57, 0x26, 0x82, 0x7f, 0x3d, 0x38, 0x42, 0x7d, 0x07, 0xdf, 0xcf,
    0xb6, 0x4a, 0x7f, 0x95, 0xb6, 0xf0, 0x6e, 0xf0, 0x1c, 0xed, 0x28, 0x1d,
    0xce, 0x7f, 0x41, 0x30, 0x34, 0x12, 0x36, 0x11, 0x24, 0x36, 0x2b, 0xa5,
    0x1c, 0xe8, 0xd2, 0x55, 0x2a, 0xf0, 0x54, 0xa2, 0x0f, 0xda, 0xc9, 0x4e,
    0x51, 0x69, 0x16, 0x5b, 0x71, 0x52, 0x67, 0x63, 0x8c, 0x1e, 0x0b, 0x9b,
    0x34, 0x52, 0xb5, 0xb7,
};
static const struct drbg_kat_no_reseed kat96_nor_t = {
    5, kat96_nor_entropyin, kat96_nor_nonce, kat96_nor_persstr,
    kat96_nor_addin0, kat96_nor_addin1, kat96_nor_retbytes
};
static const struct drbg_kat kat96_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat96_nor_t
};

static const unsigned char kat97_nor_entropyin[] = {
    0x01, 0x64, 0xae, 0x53, 0xe5, 0x33, 0xf2, 0xf1, 0x22, 0x2d, 0x4a, 0xa2,
    0x72, 0x76, 0x81, 0x4c,
};
static const unsigned char kat97_nor_nonce[] = {
    0x26, 0x05, 0xf5, 0x89, 0xcb, 0x1b, 0xea, 0x9c,
};
static const unsigned char kat97_nor_persstr[] = {
    0x9d, 0xe7, 0x03, 0xa5, 0xe2, 0xbb, 0xf2, 0xe5, 0x3d, 0x0f, 0xe3, 0xb5,
    0x73, 0xf5, 0x2a, 0xcf,
};
static const unsigned char kat97_nor_addin0[] = {0};
static const unsigned char kat97_nor_addin1[] = {0};
static const unsigned char kat97_nor_retbytes[] = {
    0xfd, 0x7f, 0x64, 0x62, 0x2e, 0x87, 0xde, 0x3b, 0x66, 0xe1, 0x99, 0x30,
    0xcf, 0x4c, 0xcd, 0xc3, 0xe6, 0x72, 0xf5, 0xa4, 0xde, 0xf0, 0xda, 0xdd,
    0xb3, 0x71, 0x07, 0x43, 0x0e, 0xc7, 0xf6, 0x91, 0xc6, 0x32, 0x15, 0x45,
    0x10, 0x3c, 0x2a, 0x61, 0x4d, 0x7f, 0x0f, 0x33, 0x17, 0x3b, 0xde, 0xc4,
    0x20, 0x58, 0x58, 0x22, 0xcd, 0xa8, 0xc2, 0x72, 0x20, 0x19, 0x22, 0xb6,
    0x02, 0xf5, 0xd8, 0xa7,
};
static const struct drbg_kat_no_reseed kat97_nor_t = {
    6, kat97_nor_entropyin, kat97_nor_nonce, kat97_nor_persstr,
    kat97_nor_addin0, kat97_nor_addin1, kat97_nor_retbytes
};
static const struct drbg_kat kat97_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat97_nor_t
};

static const unsigned char kat98_nor_entropyin[] = {
    0x29, 0xdc, 0x8b, 0xad, 0x50, 0x0c, 0xce, 0x29, 0x5b, 0x9e, 0x86, 0xb2,
    0x15, 0x9c, 0x6d, 0xcd,
};
static const unsigned char kat98_nor_nonce[] = {
    0xb6, 0x9a, 0x5a, 0x77, 0x69, 0x4f, 0xc4, 0xa0,
};
static const unsigned char kat98_nor_persstr[] = {
    0xc5, 0x3b, 0x13, 0xaf, 0xbf, 0x21, 0xf1, 0x7e, 0xe9, 0xf4, 0xdd, 0x7c,
    0x09, 0x93, 0xe0, 0xf9,
};
static const unsigned char kat98_nor_addin0[] = {0};
static const unsigned char kat98_nor_addin1[] = {0};
static const unsigned char kat98_nor_retbytes[] = {
    0x08, 0xc9, 0x7d, 0x63, 0x8b, 0x5d, 0x82, 0x59, 0x93, 0x10, 0x32, 0x5d,
    0x3b, 0x1e, 0x63, 0x27, 0x74, 0x1c, 0xd9, 0x18, 0x87, 0xd6, 0x58, 0xb5,
    0xd8, 0x15, 0xee, 0xed, 0x7a, 0x72, 0xab, 0x86, 0xe2, 0xfa, 0x17, 0xe9,
    0xd0, 0x13, 0xe1, 0x7a, 0x92, 0x14, 0xd6, 0x39, 0x6a, 0xc1, 0x48, 0x1f,
    0x76, 0x88, 0x30, 0xa4, 0xd8, 0xc4, 0xfc, 0x39, 0x20, 0x12, 0xe9, 0x07,
    0x64, 0x34, 0x93, 0x9f,
};
static const struct drbg_kat_no_reseed kat98_nor_t = {
    7, kat98_nor_entropyin, kat98_nor_nonce, kat98_nor_persstr,
    kat98_nor_addin0, kat98_nor_addin1, kat98_nor_retbytes
};
static const struct drbg_kat kat98_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat98_nor_t
};

static const unsigned char kat99_nor_entropyin[] = {
    0xb3, 0xae, 0xe2, 0xcf, 0x80, 0xe3, 0xd0, 0x5a, 0xed, 0xa0, 0xe0, 0x1c,
    0xf6, 0x5e, 0xa9, 0x65,
};
static const unsigned char kat99_nor_nonce[] = {
    0x24, 0xb5, 0x8b, 0x7c, 0x6a, 0x99, 0xeb, 0x2a,
};
static const unsigned char kat99_nor_persstr[] = {
    0x73, 0x44, 0x04, 0x3c, 0x84, 0x91, 0x5b, 0xcd, 0xd6, 0xd8, 0x1f, 0x3f,
    0xe2, 0x3b, 0xa2, 0x73,
};
static const unsigned char kat99_nor_addin0[] = {0};
static const unsigned char kat99_nor_addin1[] = {0};
static const unsigned char kat99_nor_retbytes[] = {
    0xbe, 0x35, 0x3c, 0xf8, 0x32, 0xe8, 0x46, 0x44, 0x4a, 0xf0, 0x01, 0x5e,
    0xd0, 0x5d, 0x54, 0xb7, 0x2c, 0x9a, 0x1a, 0x02, 0x31, 0xf2, 0xa4, 0xc4,
    0x75, 0x61, 0x17, 0x46, 0xef, 0x86, 0x10, 0x29, 0xe1, 0x88, 0x42, 0xc1,
    0xb7, 0x61, 0x48, 0xd2, 0xa3, 0xeb, 0x01, 0x74, 0xa3, 0x0c, 0xad, 0x81,
    0xb5, 0x58, 0xd0, 0xd1, 0xe0, 0x1a, 0x8c, 0x7d, 0xd5, 0x10, 0xf0, 0xb5,
    0x6f, 0x1f, 0xdd, 0xc3,
};
static const struct drbg_kat_no_reseed kat99_nor_t = {
    8, kat99_nor_entropyin, kat99_nor_nonce, kat99_nor_persstr,
    kat99_nor_addin0, kat99_nor_addin1, kat99_nor_retbytes
};
static const struct drbg_kat kat99_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat99_nor_t
};

static const unsigned char kat100_nor_entropyin[] = {
    0xa5, 0x25, 0xf6, 0x91, 0x65, 0xc4, 0x4f, 0x3a, 0x8a, 0xf8, 0x08, 0x57,
    0xa5, 0xe8, 0x74, 0xb0,
};
static const unsigned char kat100_nor_nonce[] = {
    0x0c, 0x80, 0x6a, 0x40, 0xc1, 0xdc, 0xb7, 0xcc,
};
static const unsigned char kat100_nor_persstr[] = {
    0x43, 0x12, 0xc2, 0x30, 0x9a, 0x1d, 0xfe, 0x2d, 0x9a, 0x66, 0xf5, 0xc1,
    0x8f, 0xc4, 0x0b, 0x87,
};
static const unsigned char kat100_nor_addin0[] = {0};
static const unsigned char kat100_nor_addin1[] = {0};
static const unsigned char kat100_nor_retbytes[] = {
    0xd0, 0xb4, 0x6f, 0x7e, 0x18, 0x77, 0xdb, 0x24, 0x4f, 0xf4, 0xe0, 0xae,
    0xb6, 0x4c, 0x4e, 0xfa, 0x3e, 0x8c, 0x2a, 0x86, 0x62, 0x41, 0x5c, 0xf5,
    0xba, 0x67, 0xf2, 0xdc, 0x91, 0xe8, 0x23, 0x10, 0xa6, 0xa9, 0x33, 0xc4,
    0xdf, 0x6a, 0x40, 0x2f, 0x8f, 0xae, 0x27, 0x0b, 0x22, 0x9b, 0xf2, 0x54,
    0xcd, 0x35, 0x94, 0x3a, 0x13, 0x34, 0x09, 0x79, 0x37, 0x66, 0x69, 0x11,
    0x9e, 0x0e, 0x5c, 0xf8,
};
static const struct drbg_kat_no_reseed kat100_nor_t = {
    9, kat100_nor_entropyin, kat100_nor_nonce, kat100_nor_persstr,
    kat100_nor_addin0, kat100_nor_addin1, kat100_nor_retbytes
};
static const struct drbg_kat kat100_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat100_nor_t
};

static const unsigned char kat101_nor_entropyin[] = {
    0x06, 0x08, 0xdc, 0x6c, 0x8f, 0x9f, 0x19, 0x56, 0xfb, 0xe7, 0xd5, 0xa7,
    0xc8, 0x2f, 0xd3, 0xb3,
};
static const unsigned char kat101_nor_nonce[] = {
    0x61, 0x5b, 0x62, 0xa0, 0x0d, 0x6d, 0x8d, 0x85,
};
static const unsigned char kat101_nor_persstr[] = {
    0x70, 0xdd, 0x43, 0xc1, 0x76, 0x43, 0x18, 0x17, 0x4a, 0x28, 0x57, 0xc4,
    0x20, 0xdd, 0xf4, 0x07,
};
static const unsigned char kat101_nor_addin0[] = {0};
static const unsigned char kat101_nor_addin1[] = {0};
static const unsigned char kat101_nor_retbytes[] = {
    0x71, 0xf4, 0xb5, 0xdf, 0x5b, 0xfa, 0x5f, 0xd5, 0x2c, 0xdc, 0x98, 0x51,
    0xa6, 0x33, 0xf7, 0x7e, 0xd9, 0x04, 0x58, 0xb5, 0xa2, 0x90, 0xb0, 0x4d,
    0x2f, 0x35, 0x13, 0x0d, 0x67, 0xa4, 0x2c, 0x16, 0x98, 0xac, 0x5f, 0x9b,
    0x13, 0x9a, 0x9e, 0xcf, 0x35, 0x90, 0x75, 0x5a, 0x20, 0x41, 0x60, 0xa3,
    0xa8, 0xf1, 0x7b, 0x77, 0x72, 0x66, 0x52, 0xc6, 0xdc, 0x6e, 0x9f, 0x00,
    0x96, 0x64, 0x54, 0xfd,
};
static const struct drbg_kat_no_reseed kat101_nor_t = {
    10, kat101_nor_entropyin, kat101_nor_nonce, kat101_nor_persstr,
    kat101_nor_addin0, kat101_nor_addin1, kat101_nor_retbytes
};
static const struct drbg_kat kat101_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat101_nor_t
};

static const unsigned char kat102_nor_entropyin[] = {
    0x7b, 0x39, 0x1f, 0x35, 0xdb, 0x0e, 0x53, 0xbf, 0x48, 0xee, 0x55, 0x76,
    0xb0, 0x23, 0xff, 0x90,
};
static const unsigned char kat102_nor_nonce[] = {
    0x76, 0x21, 0xaa, 0xd5, 0x32, 0x98, 0x2b, 0x23,
};
static const unsigned char kat102_nor_persstr[] = {
    0xa6, 0xe7, 0x57, 0x78, 0xb7, 0x77, 0xd0, 0x1f, 0x00, 0x62, 0x27, 0xc4,
    0x00, 0xd8, 0xff, 0x86,
};
static const unsigned char kat102_nor_addin0[] = {0};
static const unsigned char kat102_nor_addin1[] = {0};
static const unsigned char kat102_nor_retbytes[] = {
    0xd6, 0x17, 0x80, 0xb5, 0xc7, 0x81, 0x41, 0x2b, 0xf3, 0x08, 0x5e, 0xec,
    0xce, 0xe4, 0x9b, 0x99, 0x35, 0x8a, 0x18, 0x32, 0x23, 0x82, 0x8c, 0x1d,
    0x80, 0x13, 0xac, 0xe6, 0x13, 0xd8, 0x9a, 0x45, 0x04, 0xd7, 0x5b, 0xa3,
    0x09, 0xe5, 0x10, 0x58, 0x9a, 0x53, 0xb4, 0x72, 0xbd, 0x5f, 0xa1, 0xee,
    0x2a, 0x22, 0x39, 0x2b, 0x82, 0x65, 0x70, 0x7c, 0x15, 0xe3, 0x29, 0x35,
    0xbc, 0x8e, 0xfb, 0x4e,
};
static const struct drbg_kat_no_reseed kat102_nor_t = {
    11, kat102_nor_entropyin, kat102_nor_nonce, kat102_nor_persstr,
    kat102_nor_addin0, kat102_nor_addin1, kat102_nor_retbytes
};
static const struct drbg_kat kat102_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat102_nor_t
};

static const unsigned char kat103_nor_entropyin[] = {
    0x09, 0x9f, 0x1f, 0x59, 0x17, 0x07, 0xd1, 0xf9, 0xab, 0xae, 0x7e, 0x2b,
    0x65, 0xad, 0xab, 0xc3,
};
static const unsigned char kat103_nor_nonce[] = {
    0xf9, 0x0d, 0x35, 0x7a, 0x90, 0x1f, 0x11, 0x35,
};
static const unsigned char kat103_nor_persstr[] = {
    0x74, 0xb8, 0x26, 0x0c, 0xed, 0x4b, 0xbb, 0x80, 0x41, 0x77, 0x92, 0xf1,
    0x93, 0xc6, 0xf9, 0xac,
};
static const unsigned char kat103_nor_addin0[] = {0};
static const unsigned char kat103_nor_addin1[] = {0};
static const unsigned char kat103_nor_retbytes[] = {
    0x86, 0xf5, 0x06, 0x25, 0x6d, 0x29, 0x45, 0x0f, 0x16, 0x07, 0x31, 0x27,
    0x39, 0xe5, 0xa0, 0x89, 0xb6, 0xb7, 0x86, 0xfd, 0xf5, 0x21, 0x45, 0x76,
    0x9c, 0xb7, 0x0e, 0xf4, 0xee, 0xd3, 0x21, 0x09, 0xcf, 0x62, 0x36, 0x9d,
    0x7a, 0x76, 0x59, 0x55, 0x78, 0x1e, 0xae, 0x63, 0x52, 0x02, 0x75, 0x16,
    0x93, 0x68, 0x25, 0x7c, 0x6d, 0x34, 0x8e, 0xcc, 0x90, 0x0a, 0x38, 0x98,
    0x77, 0x8d, 0x9a, 0x1a,
};
static const struct drbg_kat_no_reseed kat103_nor_t = {
    12, kat103_nor_entropyin, kat103_nor_nonce, kat103_nor_persstr,
    kat103_nor_addin0, kat103_nor_addin1, kat103_nor_retbytes
};
static const struct drbg_kat kat103_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat103_nor_t
};

static const unsigned char kat104_nor_entropyin[] = {
    0xc9, 0x05, 0x9d, 0x7c, 0x01, 0x5c, 0x64, 0xf6, 0xe4, 0x1a, 0xa4, 0x92,
    0x09, 0x9f, 0x4e, 0xf0,
};
static const unsigned char kat104_nor_nonce[] = {
    0x3f, 0x6e, 0x07, 0x69, 0xce, 0xe1, 0x77, 0xc6,
};
static const unsigned char kat104_nor_persstr[] = {
    0x76, 0xc6, 0xc6, 0x61, 0x32, 0x98, 0xd9, 0x25, 0xa6, 0xc5, 0x07, 0xae,
    0x65, 0x4d, 0x6d, 0x9a,
};
static const unsigned char kat104_nor_addin0[] = {0};
static const unsigned char kat104_nor_addin1[] = {0};
static const unsigned char kat104_nor_retbytes[] = {
    0x23, 0x81, 0x11, 0xc0, 0x06, 0xe5, 0x14, 0xc0, 0x5f, 0x2a, 0xe9, 0x35,
    0xa3, 0x2d, 0xba, 0xa0, 0xb1, 0x28, 0xda, 0xa0, 0x73, 0x6e, 0x83, 0xd9,
    0x34, 0x03, 0x77, 0x6c, 0x91, 0xe4, 0x77, 0xd0, 0xd0, 0x2c, 0x4a, 0x24,
    0x87, 0x6a, 0x23, 0x29, 0xa0, 0xf1, 0x06, 0x03, 0x8d, 0x70, 0x1f, 0xed,
    0xdf, 0x02, 0x47, 0xcc, 0x75, 0x84, 0x6d, 0xd3, 0x01, 0x08, 0x29, 0x9d,
    0x84, 0x0e, 0xc4, 0xa8,
};
static const struct drbg_kat_no_reseed kat104_nor_t = {
    13, kat104_nor_entropyin, kat104_nor_nonce, kat104_nor_persstr,
    kat104_nor_addin0, kat104_nor_addin1, kat104_nor_retbytes
};
static const struct drbg_kat kat104_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat104_nor_t
};

static const unsigned char kat105_nor_entropyin[] = {
    0x98, 0xd1, 0x3c, 0x30, 0xa6, 0xda, 0xe2, 0xca, 0x76, 0xd7, 0xd6, 0x67,
    0x9f, 0x1f, 0xfb, 0xc4,
};
static const unsigned char kat105_nor_nonce[] = {
    0x49, 0x0b, 0x93, 0x9e, 0xb5, 0x56, 0xff, 0x1b,
};
static const unsigned char kat105_nor_persstr[] = {
    0x6c, 0x10, 0xa7, 0xc3, 0x0f, 0xb5, 0x34, 0xb5, 0x93, 0x4b, 0x1c, 0x29,
    0x80, 0x84, 0x94, 0xb9,
};
static const unsigned char kat105_nor_addin0[] = {0};
static const unsigned char kat105_nor_addin1[] = {0};
static const unsigned char kat105_nor_retbytes[] = {
    0xcb, 0x7d, 0xee, 0xea, 0x97, 0xbd, 0xb0, 0x3f, 0x72, 0xa0, 0x3b, 0x2d,
    0x9b, 0x5f, 0x1b, 0x33, 0x03, 0x44, 0x48, 0x6e, 0x53, 0xe0, 0x49, 0x27,
    0xc5, 0x8f, 0xb6, 0x3c, 0x97, 0x1a, 0xc9, 0x87, 0x33, 0xb0, 0x24, 0x9a,
    0x1e, 0xfa, 0x85, 0xf8, 0x8b, 0xec, 0x48, 0x18, 0xe1, 0x31, 0x10, 0x22,
    0x07, 0x76, 0xe4, 0xfd, 0xe4, 0x29, 0xe6, 0x39, 0x0a, 0xf4, 0x66, 0x71,
    0x29, 0x01, 0x18, 0x9b,
};
static const struct drbg_kat_no_reseed kat105_nor_t = {
    14, kat105_nor_entropyin, kat105_nor_nonce, kat105_nor_persstr,
    kat105_nor_addin0, kat105_nor_addin1, kat105_nor_retbytes
};
static const struct drbg_kat kat105_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat105_nor_t
};

static const unsigned char kat106_nor_entropyin[] = {
    0x25, 0xad, 0x6e, 0x73, 0x05, 0x5e, 0xcb, 0xc9, 0x49, 0xf2, 0x91, 0xf0,
    0xf7, 0x97, 0xf1, 0x7a,
};
static const unsigned char kat106_nor_nonce[] = {
    0x1f, 0x85, 0xa9, 0x2a, 0x46, 0x0a, 0x2e, 0xea,
};
static const unsigned char kat106_nor_persstr[] = {
    0x83, 0xd0, 0x4b, 0x4b, 0x15, 0x91, 0xc7, 0xba, 0xc9, 0xbc, 0xa9, 0x07,
    0xbb, 0xe9, 0xdd, 0xd4,
};
static const unsigned char kat106_nor_addin0[] = {
    0x52, 0x3c, 0x6c, 0xb0, 0xbc, 0x27, 0xd1, 0x9f, 0x8d, 0x2e, 0xbe, 0xf5,
    0x79, 0x26, 0xdd, 0xe4,
};
static const unsigned char kat106_nor_addin1[] = {
    0xd3, 0x3c, 0xd1, 0x4b, 0x5b, 0x79, 0x37, 0x38, 0x8e, 0x89, 0xec, 0xc0,
    0x80, 0x63, 0x03, 0xf0,
};
static const unsigned char kat106_nor_retbytes[] = {
    0x23, 0xd9, 0x19, 0x5e, 0x40, 0x26, 0xed, 0xf0, 0x70, 0x88, 0xb8, 0x37,
    0x62, 0x75, 0x47, 0xa0, 0xea, 0x91, 0x37, 0xed, 0x03, 0x36, 0xd7, 0x69,
    0x60, 0x15, 0xdd, 0x6f, 0x29, 0x84, 0xde, 0xdb, 0x2a, 0x20, 0x7f, 0x2e,
    0xb8, 0xa2, 0x56, 0x60, 0xa5, 0xee, 0x78, 0x11, 0x78, 0x57, 0x9a, 0x0f,
    0x23, 0x3f, 0xb6, 0xf7, 0x26, 0x03, 0x58, 0xdb, 0xfd, 0x53, 0x25, 0xc3,
    0xf8, 0xc8, 0xfe, 0x33,
};
static const struct drbg_kat_no_reseed kat106_nor_t = {
    0, kat106_nor_entropyin, kat106_nor_nonce, kat106_nor_persstr,
    kat106_nor_addin0, kat106_nor_addin1, kat106_nor_retbytes
};
static const struct drbg_kat kat106_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat106_nor_t
};

static const unsigned char kat107_nor_entropyin[] = {
    0xae, 0x33, 0x2f, 0x73, 0x39, 0x0e, 0x27, 0x75, 0x6b, 0xc9, 0x3d, 0x2b,
    0x95, 0x1b, 0x8b, 0x44,
};
static const unsigned char kat107_nor_nonce[] = {
    0xe6, 0x9f, 0xc8, 0xa1, 0x64, 0x50, 0xfb, 0x6d,
};
static const unsigned char kat107_nor_persstr[] = {
    0x5f, 0x76, 0x86, 0x3d, 0x3a, 0x20, 0x87, 0xe9, 0xcb, 0x90, 0xe1, 0x12,
    0xc1, 0x68, 0x06, 0xe2,
};
static const unsigned char kat107_nor_addin0[] = {
    0xbf, 0x6a, 0x8e, 0x05, 0x6e, 0x6c, 0x3c, 0xc1, 0xa6, 0xcb, 0xdb, 0x6b,
    0x59, 0xeb, 0xea, 0xe2,
};
static const unsigned char kat107_nor_addin1[] = {
    0xd8, 0x7a, 0xb9, 0x22, 0x4a, 0xa9, 0xcd, 0x9b, 0x5b, 0x84, 0x78, 0x35,
    0xcb, 0x0d, 0xaa, 0xc4,
};
static const unsigned char kat107_nor_retbytes[] = {
    0xe0, 0xa2, 0xdd, 0x56, 0x06, 0xa0, 0xa2, 0x61, 0x57, 0xf9, 0x21, 0x05,
    0x11, 0xbb, 0xde, 0x50, 0xc9, 0x2f, 0x34, 0xad, 0x92, 0x36, 0x3c, 0xb9,
    0x2a, 0x05, 0x20, 0x8b, 0x60, 0xb3, 0x3d, 0x21, 0xc3, 0xd6, 0xc0, 0xc6,
    0xef, 0x05, 0x4e, 0xc4, 0x09, 0xb4, 0x63, 0x0d, 0xea, 0xc4, 0xc6, 0xd1,
    0xad, 0x1b, 0x6b, 0xe7, 0x5f, 0xff, 0x27, 0xaa, 0x74, 0x94, 0x13, 0xc8,
    0xb6, 0x4c, 0xf3, 0xe1,
};
static const struct drbg_kat_no_reseed kat107_nor_t = {
    1, kat107_nor_entropyin, kat107_nor_nonce, kat107_nor_persstr,
    kat107_nor_addin0, kat107_nor_addin1, kat107_nor_retbytes
};
static const struct drbg_kat kat107_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat107_nor_t
};

static const unsigned char kat108_nor_entropyin[] = {
    0xe1, 0xa5, 0x89, 0xaf, 0xab, 0x83, 0x91, 0x69, 0x2d, 0xba, 0x52, 0x31,
    0x8a, 0xa8, 0xdc, 0x0d,
};
static const unsigned char kat108_nor_nonce[] = {
    0x22, 0xeb, 0xcf, 0x2e, 0x24, 0xa9, 0x55, 0x28,
};
static const unsigned char kat108_nor_persstr[] = {
    0x82, 0xe8, 0xc3, 0x55, 0xdb, 0xc9, 0xcd, 0x4c, 0x8d, 0x52, 0xfa, 0x45,
    0x21, 0x18, 0xf9, 0xf8,
};
static const unsigned char kat108_nor_addin0[] = {
    0xf7, 0x74, 0x57, 0xec, 0x32, 0xde, 0x66, 0x6f, 0x37, 0x0d, 0xa9, 0xe6,
    0xc2, 0x20, 0xc5, 0x80,
};
static const unsigned char kat108_nor_addin1[] = {
    0xba, 0x25, 0xaa, 0x3a, 0xb3, 0xd5, 0x0a, 0x72, 0xb6, 0xc1, 0xb4, 0x3f,
    0x30, 0xdb, 0x44, 0x55,
};
static const unsigned char kat108_nor_retbytes[] = {
    0x88, 0x59, 0x55, 0xd1, 0x27, 0xfa, 0xf4, 0xc9, 0x21, 0xc4, 0xed, 0x3f,
    0x38, 0xbc, 0x15, 0x42, 0xb9, 0x5a, 0x87, 0x83, 0x4d, 0x4d, 0x73, 0xd1,
    0x56, 0x24, 0x00, 0xa6, 0xd3, 0xaa, 0x8e, 0x6c, 0x69, 0x7b, 0x71, 0x1d,
    0x91, 0x0e, 0xec, 0xeb, 0xc9, 0xca, 0x4e, 0xc2, 0xdc, 0x17, 0xe7, 0x98,
    0x93, 0xe7, 0xf7, 0x34, 0xc4, 0x56, 0x75, 0xde, 0x47, 0x7f, 0x55, 0x45,
    0xbd, 0x94, 0xcd, 0xee,
};
static const struct drbg_kat_no_reseed kat108_nor_t = {
    2, kat108_nor_entropyin, kat108_nor_nonce, kat108_nor_persstr,
    kat108_nor_addin0, kat108_nor_addin1, kat108_nor_retbytes
};
static const struct drbg_kat kat108_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat108_nor_t
};

static const unsigned char kat109_nor_entropyin[] = {
    0x6f, 0xb7, 0x6f, 0xcd, 0x3d, 0x76, 0x2f, 0xa4, 0xc5, 0x0d, 0xe1, 0xf2,
    0x2e, 0x92, 0x76, 0x83,
};
static const unsigned char kat109_nor_nonce[] = {
    0xa5, 0xee, 0xf1, 0x2f, 0x3b, 0x3c, 0x25, 0xa3,
};
static const unsigned char kat109_nor_persstr[] = {
    0x54, 0x86, 0x7d, 0x7d, 0xc8, 0xce, 0xf5, 0x23, 0x42, 0xbb, 0xc0, 0x09,
    0x95, 0x4e, 0xfb, 0xf8,
};
static const unsigned char kat109_nor_addin0[] = {
    0xfa, 0xc5, 0x52, 0xaa, 0x87, 0x96, 0xe8, 0x85, 0x8d, 0x55, 0xae, 0x37,
    0x1a, 0x0f, 0x54, 0x8f,
};
static const unsigned char kat109_nor_addin1[] = {
    0xfe, 0xfb, 0xe9, 0xc8, 0x3c, 0x15, 0x9f, 0xcf, 0xa9, 0x2e, 0xe8, 0x18,
    0x76, 0x1d, 0x49, 0xdc,
};
static const unsigned char kat109_nor_retbytes[] = {
    0x09, 0x4f, 0x86, 0x3b, 0x88, 0xe2, 0xb5, 0x5e, 0xf4, 0x15, 0xd2, 0x68,
    0x82, 0x94, 0xc7, 0xd8, 0xd2, 0xd7, 0x76, 0xc1, 0x69, 0x16, 0x73, 0x91,
    0xdf, 0x36, 0xf8, 0x0a, 0xd8, 0xe9, 0x83, 0x3e, 0x97, 0x61, 0x6e, 0xf6,
    0xe0, 0xa4, 0x5f, 0x94, 0x0d, 0x66, 0x38, 0x30, 0x18, 0xae, 0xe8, 0x6a,
    0x48, 0x0a, 0x26, 0x51, 0xd6, 0xe2, 0xef, 0xad, 0x61, 0x5a, 0x05, 0x1f,
    0x0d, 0xca, 0x64, 0xae,
};
static const struct drbg_kat_no_reseed kat109_nor_t = {
    3, kat109_nor_entropyin, kat109_nor_nonce, kat109_nor_persstr,
    kat109_nor_addin0, kat109_nor_addin1, kat109_nor_retbytes
};
static const struct drbg_kat kat109_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat109_nor_t
};

static const unsigned char kat110_nor_entropyin[] = {
    0x5b, 0x32, 0xa3, 0x92, 0xdc, 0x66, 0x5a, 0xb5, 0x8f, 0x25, 0x98, 0xea,
    0xb0, 0x7d, 0x8e, 0x53,
};
static const unsigned char kat110_nor_nonce[] = {
    0xc6, 0xf6, 0x13, 0x10, 0x73, 0x88, 0x60, 0xf5,
};
static const unsigned char kat110_nor_persstr[] = {
    0x9f, 0x30, 0x9a, 0x68, 0x53, 0x29, 0x82, 0x16, 0x92, 0xa9, 0x73, 0x00,
    0x62, 0x0c, 0x01, 0x56,
};
static const unsigned char kat110_nor_addin0[] = {
    0x87, 0x5a, 0x4a, 0x43, 0xbd, 0xdf, 0x1a, 0x73, 0x05, 0x52, 0x6a, 0x48,
    0x8a, 0x1e, 0x0e, 0xb4,
};
static const unsigned char kat110_nor_addin1[] = {
    0x21, 0x14, 0xa7, 0xcb, 0x15, 0x49, 0x0c, 0x06, 0x3b, 0xb4, 0x66, 0x8f,
    0x36, 0x9a, 0x3d, 0x8e,
};
static const unsigned char kat110_nor_retbytes[] = {
    0x91, 0xc1, 0xf1, 0x99, 0xf6, 0x71, 0xff, 0x56, 0x8e, 0x0e, 0xce, 0x46,
    0x08, 0x87, 0x65, 0x16, 0x23, 0x22, 0xa8, 0x21, 0x60, 0x66, 0xf9, 0xa4,
    0x15, 0xd7, 0xcc, 0xd9, 0xdf, 0xe7, 0x23, 0xc5, 0x21, 0xf5, 0x7d, 0xbd,
    0x84, 0x69, 0xe3, 0x9b, 0xac, 0xda, 0xc3, 0x1e, 0x74, 0x2a, 0x2d, 0x2f,
    0xf2, 0xa3, 0x1d, 0x5a, 0x15, 0xa0, 0x1b, 0xfd, 0x32, 0xbe, 0x50, 0x28,
    0x6c, 0x73, 0x0e, 0xad,
};
static const struct drbg_kat_no_reseed kat110_nor_t = {
    4, kat110_nor_entropyin, kat110_nor_nonce, kat110_nor_persstr,
    kat110_nor_addin0, kat110_nor_addin1, kat110_nor_retbytes
};
static const struct drbg_kat kat110_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat110_nor_t
};

static const unsigned char kat111_nor_entropyin[] = {
    0x79, 0x95, 0x7c, 0x33, 0xe4, 0x5b, 0xba, 0x1c, 0x8a, 0xe6, 0x90, 0x29,
    0x07, 0x7a, 0xe3, 0x6b,
};
static const unsigned char kat111_nor_nonce[] = {
    0x5b, 0x49, 0xd8, 0xe7, 0xc1, 0x80, 0xbc, 0xb6,
};
static const unsigned char kat111_nor_persstr[] = {
    0x2a, 0xa8, 0xe7, 0x7b, 0x94, 0xfe, 0x0d, 0xdf, 0x0e, 0x57, 0x30, 0x2c,
    0x8e, 0xbd, 0xc2, 0x2a,
};
static const unsigned char kat111_nor_addin0[] = {
    0x14, 0x84, 0x4d, 0x85, 0xe5, 0x0c, 0xb1, 0xfc, 0xb4, 0xe6, 0x88, 0x7b,
    0x3f, 0xc6, 0x7f, 0x54,
};
static const unsigned char kat111_nor_addin1[] = {
    0x4c, 0x7c, 0xb0, 0x69, 0x29, 0x7c, 0x18, 0x65, 0x33, 0x94, 0xb8, 0x2b,
    0xe6, 0xa9, 0x30, 0x99,
};
static const unsigned char kat111_nor_retbytes[] = {
    0x06, 0x61, 0xe2, 0x41, 0x6e, 0x71, 0xfe, 0x49, 0x11, 0x9f, 0x96, 0x76,
    0x42, 0xfd, 0x37, 0xdd, 0xb5, 0x82, 0xa1, 0x89, 0xa8, 0x88, 0xe7, 0xd4,
    0xa2, 0xaa, 0x56, 0x27, 0x51, 0x94, 0xc8, 0xca, 0x69, 0x8f, 0xb2, 0xd6,
    0x96, 0x83, 0x4b, 0x88, 0xb5, 0x72, 0xf0, 0xfd, 0xf5, 0xba, 0xdb, 0x6b,
    0xf5, 0x8d, 0x16, 0x12, 0xd2, 0x5f, 0x5e, 0xe3, 0xf6, 0x2d, 0x33, 0xdb,
    0x9a, 0x86, 0xb0, 0x87,
};
static const struct drbg_kat_no_reseed kat111_nor_t = {
    5, kat111_nor_entropyin, kat111_nor_nonce, kat111_nor_persstr,
    kat111_nor_addin0, kat111_nor_addin1, kat111_nor_retbytes
};
static const struct drbg_kat kat111_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat111_nor_t
};

static const unsigned char kat112_nor_entropyin[] = {
    0x70, 0x50, 0x89, 0x80, 0x36, 0x2e, 0x96, 0x2a, 0xe7, 0x78, 0x45, 0x80,
    0x81, 0xcf, 0xbb, 0xe6,
};
static const unsigned char kat112_nor_nonce[] = {
    0x29, 0xcc, 0xf6, 0xed, 0xba, 0x46, 0x2d, 0xc2,
};
static const unsigned char kat112_nor_persstr[] = {
    0x48, 0xa3, 0xd8, 0xdb, 0xdc, 0x85, 0x18, 0x8a, 0xbb, 0x35, 0xb5, 0x6c,
    0x19, 0x4d, 0x60, 0x20,
};
static const unsigned char kat112_nor_addin0[] = {
    0xa8, 0x14, 0x72, 0x42, 0xe2, 0x47, 0x37, 0xfc, 0xf3, 0xb0, 0x07, 0xd2,
    0x84, 0xd6, 0x61, 0xc4,
};
static const unsigned char kat112_nor_addin1[] = {
    0x96, 0x4a, 0x9d, 0x06, 0x86, 0xd9, 0x61, 0x93, 0xfb, 0x2e, 0x4a, 0x35,
    0x14, 0x5e, 0xdc, 0xc8,
};
static const unsigned char kat112_nor_retbytes[] = {
    0xd8, 0x1b, 0xd5, 0xda, 0x60, 0xd9, 0x9d, 0x6d, 0xaa, 0xec, 0xeb, 0xc0,
    0x60, 0x65, 0x26, 0x33, 0xec, 0x14, 0xdc, 0x3b, 0x84, 0x69, 0x3b, 0xdf,
    0xf7, 0x67, 0x77, 0x1f, 0x03, 0x2b, 0xfa, 0x2e, 0x4d, 0xfa, 0xd5, 0x3e,
    0xf6, 0xd1, 0x99, 0x10, 0xbd, 0x92, 0x5d, 0xf1, 0x20, 0x9f, 0x33, 0xea,
    0x11, 0xe3, 0xf1, 0xe8, 0x37, 0xb0, 0x3e, 0x7a, 0x70, 0x03, 0x25, 0x7f,
    0xf4, 0x20, 0xa6, 0x92,
};
static const struct drbg_kat_no_reseed kat112_nor_t = {
    6, kat112_nor_entropyin, kat112_nor_nonce, kat112_nor_persstr,
    kat112_nor_addin0, kat112_nor_addin1, kat112_nor_retbytes
};
static const struct drbg_kat kat112_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat112_nor_t
};

static const unsigned char kat113_nor_entropyin[] = {
    0xb7, 0x83, 0x71, 0xba, 0xf5, 0x86, 0xc0, 0xdd, 0x05, 0x4b, 0xfd, 0x29,
    0x3f, 0xc7, 0xe8, 0xdc,
};
static const unsigned char kat113_nor_nonce[] = {
    0x11, 0x9b, 0x23, 0x2c, 0xe4, 0xca, 0xda, 0x97,
};
static const unsigned char kat113_nor_persstr[] = {
    0x3a, 0x58, 0x75, 0xb4, 0x43, 0x07, 0x3c, 0x9b, 0xf6, 0x4a, 0x0a, 0xaa,
    0xf2, 0x6b, 0xa4, 0x4f,
};
static const unsigned char kat113_nor_addin0[] = {
    0x4d, 0x2d, 0x4f, 0x16, 0xb7, 0xb7, 0xd2, 0xd2, 0xbb, 0x4d, 0x83, 0xa4,
    0x92, 0x99, 0x96, 0xb2,
};
static const unsigned char kat113_nor_addin1[] = {
    0x7d, 0x31, 0x73, 0xe0, 0x8a, 0xea, 0x4c, 0x8b, 0x6e, 0xf9, 0xe3, 0xcd,
    0x69, 0x8f, 0x9b, 0x09,
};
static const unsigned char kat113_nor_retbytes[] = {
    0xbb, 0x04, 0x86, 0x41, 0xf0, 0x36, 0xdc, 0x11, 0xb8, 0xdf, 0x8a, 0xd3,
    0xe4, 0xd5, 0xbd, 0xfc, 0x30, 0x9b, 0x2c, 0x3f, 0x22, 0xb3, 0xd2, 0xb8,
    0xac, 0xf1, 0x96, 0xd5, 0x0c, 0xd5, 0x42, 0xde, 0x13, 0xcd, 0x99, 0x97,
    0x8a, 0x92, 0x0a, 0xcb, 0xa6, 0x1d, 0xc0, 0x7c, 0x64, 0x6b, 0x48, 0x09,
    0x44, 0x79, 0xbb, 0x44, 0x17, 0xd8, 0x40, 0x9a, 0xba, 0xe3, 0xd6, 0x9e,
    0xd5, 0x2a, 0x2a, 0x3e,
};
static const struct drbg_kat_no_reseed kat113_nor_t = {
    7, kat113_nor_entropyin, kat113_nor_nonce, kat113_nor_persstr,
    kat113_nor_addin0, kat113_nor_addin1, kat113_nor_retbytes
};
static const struct drbg_kat kat113_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat113_nor_t
};

static const unsigned char kat114_nor_entropyin[] = {
    0x96, 0xbf, 0xc3, 0x47, 0xb3, 0x06, 0x80, 0x5a, 0xf8, 0x8d, 0xa0, 0x07,
    0x67, 0x5d, 0xb4, 0x7c,
};
static const unsigned char kat114_nor_nonce[] = {
    0x12, 0x95, 0x37, 0x06, 0xa0, 0x6e, 0xb5, 0x30,
};
static const unsigned char kat114_nor_persstr[] = {
    0x46, 0x73, 0x3d, 0xdd, 0xaf, 0x70, 0xe3, 0xf2, 0x07, 0x5c, 0xb8, 0x70,
    0xf4, 0x7e, 0x72, 0xdf,
};
static const unsigned char kat114_nor_addin0[] = {
    0x64, 0xf7, 0x9a, 0xd1, 0xdf, 0xa3, 0x93, 0xd3, 0x6b, 0xf5, 0xbe, 0x83,
    0x32, 0x90, 0xb6, 0x94,
};
static const unsigned char kat114_nor_addin1[] = {
    0x5d, 0x21, 0x0f, 0x12, 0x85, 0xf0, 0xfb, 0x03, 0x8a, 0x75, 0x34, 0x76,
    0xef, 0x24, 0x91, 0x79,
};
static const unsigned char kat114_nor_retbytes[] = {
    0x67, 0xb6, 0x9b, 0xa4, 0x34, 0x90, 0x3e, 0x2b, 0xbf, 0x46, 0xd3, 0xf5,
    0xe3, 0x00, 0xb9, 0xe5, 0xcf, 0x33, 0x01, 0xbe, 0xaa, 0xb7, 0xd3, 0x0b,
    0x46, 0x52, 0xed, 0x3b, 0xf6, 0x62, 0x65, 0x79, 0x50, 0x3a, 0x54, 0x4e,
    0x58, 0x23, 0x7c, 0x5e, 0x55, 0x1a, 0xd5, 0xee, 0x17, 0xb7, 0x36, 0x12,
    0x0c, 0x37, 0xa3, 0x02, 0x9b, 0xd7, 0xa1, 0x03, 0x4d, 0x4c, 0x49, 0xf6,
    0x1e, 0x04, 0x0a, 0xca,
};
static const struct drbg_kat_no_reseed kat114_nor_t = {
    8, kat114_nor_entropyin, kat114_nor_nonce, kat114_nor_persstr,
    kat114_nor_addin0, kat114_nor_addin1, kat114_nor_retbytes
};
static const struct drbg_kat kat114_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat114_nor_t
};

static const unsigned char kat115_nor_entropyin[] = {
    0x9a, 0x4c, 0x8b, 0x46, 0xa9, 0x74, 0xfb, 0xc8, 0x5e, 0x25, 0x6d, 0x1f,
    0x07, 0xcc, 0x59, 0xaa,
};
static const unsigned char kat115_nor_nonce[] = {
    0xeb, 0x8d, 0x66, 0x21, 0x02, 0x09, 0x01, 0x14,
};
static const unsigned char kat115_nor_persstr[] = {
    0xb9, 0x4b, 0x8d, 0x2a, 0x64, 0x92, 0x66, 0x64, 0x5a, 0x71, 0x5a, 0xfc,
    0xa0, 0x99, 0x5d, 0x9d,
};
static const unsigned char kat115_nor_addin0[] = {
    0xf3, 0x36, 0xe1, 0xd6, 0x2a, 0xc5, 0x68, 0xed, 0x13, 0x2e, 0xa0, 0x18,
    0x15, 0xda, 0xd5, 0x08,
};
static const unsigned char kat115_nor_addin1[] = {
    0x66, 0x76, 0x26, 0xa2, 0x3d, 0x23, 0xb7, 0x16, 0x5a, 0x6a, 0x88, 0x5d,
    0x41, 0xfa, 0x90, 0x1c,
};
static const unsigned char kat115_nor_retbytes[] = {
    0x2d, 0xf3, 0x3f, 0x7a, 0xd2, 0xc2, 0xd8, 0x6e, 0x61, 0x63, 0x3c, 0x25,
    0xa8, 0xac, 0x57, 0xb5, 0xde, 0xad, 0x88, 0x02, 0x1c, 0x93, 0xad, 0x69,
    0xba, 0x6c, 0x52, 0x27, 0x99, 0xc4, 0xbe, 0xbf, 0xda, 0x78, 0x27, 0x72,
    0xc8, 0x1e, 0x7e, 0x02, 0xd8, 0xb8, 0x82, 0x54, 0x6b, 0x24, 0x2b, 0xb2,
    0x2f, 0x31, 0x05, 0x65, 0xba, 0xc3, 0x16, 0x26, 0x9f, 0xfc, 0x80, 0x04,
    0xd5, 0x44, 0x06, 0x31,
};
static const struct drbg_kat_no_reseed kat115_nor_t = {
    9, kat115_nor_entropyin, kat115_nor_nonce, kat115_nor_persstr,
    kat115_nor_addin0, kat115_nor_addin1, kat115_nor_retbytes
};
static const struct drbg_kat kat115_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat115_nor_t
};

static const unsigned char kat116_nor_entropyin[] = {
    0xc7, 0xd6, 0x55, 0x75, 0x65, 0xa1, 0xfd, 0xe4, 0x80, 0x84, 0x1f, 0x1f,
    0x1a, 0x10, 0xf6, 0x7d,
};
static const unsigned char kat116_nor_nonce[] = {
    0x02, 0xf8, 0x47, 0x98, 0x66, 0x26, 0xbb, 0x4b,
};
static const unsigned char kat116_nor_persstr[] = {
    0x61, 0x8c, 0xac, 0x99, 0x53, 0x17, 0x91, 0x67, 0x36, 0x03, 0xa1, 0x6b,
    0x99, 0xab, 0x0e, 0x80,
};
static const unsigned char kat116_nor_addin0[] = {
    0x81, 0x0c, 0x8b, 0xe4, 0xb1, 0x80, 0x1f, 0xec, 0x75, 0x40, 0x63, 0xc2,
    0x61, 0x15, 0x66, 0x60,
};
static const unsigned char kat116_nor_addin1[] = {
    0xe2, 0x59, 0x4d, 0x34, 0xce, 0x1a, 0xd6, 0x53, 0x51, 0x28, 0x60, 0x69,
    0xeb, 0x1a, 0x63, 0x08,
};
static const unsigned char kat116_nor_retbytes[] = {
    0xa3, 0xc4, 0xd0, 0x2d, 0xa0, 0x16, 0x6f, 0xef, 0x32, 0xce, 0x69, 0x97,
    0x5d, 0xb9, 0x4d, 0x19, 0x60, 0x44, 0xe8, 0x1f, 0x59, 0x04, 0x4d, 0x22,
    0xdf, 0x84, 0x05, 0xb4, 0x68, 0xeb, 0xd5, 0xa9, 0x29, 0x97, 0xfe, 0xeb,
    0xa8, 0x2d, 0x7a, 0xdf, 0x97, 0xcc, 0xbb, 0x1a, 0x59, 0x35, 0x56, 0x26,
    0xae, 0xfa, 0xe0, 0xd4, 0x35, 0xb4, 0x03, 0xb7, 0xde, 0xb9, 0x00, 0x5b,
    0x8f, 0x5e, 0xa9, 0xa5,
};
static const struct drbg_kat_no_reseed kat116_nor_t = {
    10, kat116_nor_entropyin, kat116_nor_nonce, kat116_nor_persstr,
    kat116_nor_addin0, kat116_nor_addin1, kat116_nor_retbytes
};
static const struct drbg_kat kat116_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat116_nor_t
};

static const unsigned char kat117_nor_entropyin[] = {
    0x14, 0x0d, 0xbf, 0xf6, 0x3e, 0x70, 0x83, 0xc3, 0x08, 0x41, 0x9f, 0x86,
    0x4c, 0x09, 0x3e, 0x99,
};
static const unsigned char kat117_nor_nonce[] = {
    0xeb, 0x1c, 0x66, 0x92, 0xdb, 0x94, 0x9d, 0x5a,
};
static const unsigned char kat117_nor_persstr[] = {
    0x23, 0xd3, 0x27, 0x21, 0xf1, 0xc0, 0x3a, 0x90, 0xf3, 0x9e, 0xf0, 0x21,
    0xac, 0x88, 0x84, 0xc7,
};
static const unsigned char kat117_nor_addin0[] = {
    0xc6, 0xe3, 0x80, 0x8d, 0xb9, 0xf9, 0xda, 0x14, 0x23, 0x9a, 0xd7, 0xd7,
    0x18, 0x28, 0x6a, 0x96,
};
static const unsigned char kat117_nor_addin1[] = {
    0xa8, 0x4a, 0xaa, 0xdb, 0x7d, 0x57, 0x2a, 0x9a, 0xfc, 0xa3, 0xf7, 0x6c,
    0xed, 0xf0, 0xa4, 0xb9,
};
static const unsigned char kat117_nor_retbytes[] = {
    0x74, 0x3d, 0xf9, 0x7c, 0x2e, 0x99, 0xd9, 0x74, 0x07, 0x77, 0x72, 0xa6,
    0xfe, 0xa4, 0x03, 0x12, 0x8f, 0x41, 0xff, 0x32, 0xa9, 0x5f, 0xb4, 0xf8,
    0xb0, 0xec, 0xca, 0xa0, 0x57, 0x1c, 0xd2, 0x57, 0x8e, 0x48, 0x2c, 0xe7,
    0x09, 0x84, 0x99, 0xc4, 0xd3, 0x01, 0x5f, 0x91, 0x92, 0x35, 0x52, 0x38,
    0xc7, 0x0f, 0x8b, 0x76, 0xa5, 0x04, 0x27, 0x2a, 0x87, 0x09, 0xcd, 0x0e,
    0x2d, 0xc8, 0x2c, 0x9c,
};
static const struct drbg_kat_no_reseed kat117_nor_t = {
    11, kat117_nor_entropyin, kat117_nor_nonce, kat117_nor_persstr,
    kat117_nor_addin0, kat117_nor_addin1, kat117_nor_retbytes
};
static const struct drbg_kat kat117_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat117_nor_t
};

static const unsigned char kat118_nor_entropyin[] = {
    0x6a, 0xf3, 0x96, 0x2f, 0xd6, 0xae, 0x84, 0xde, 0xc3, 0xf8, 0x94, 0xbf,
    0x39, 0xc6, 0x46, 0x42,
};
static const unsigned char kat118_nor_nonce[] = {
    0xc0, 0x20, 0x49, 0x64, 0x61, 0xa3, 0x91, 0x61,
};
static const unsigned char kat118_nor_persstr[] = {
    0x75, 0xca, 0xef, 0xe7, 0x75, 0x20, 0x45, 0xfc, 0xb1, 0xe5, 0xfc, 0x22,
    0x7f, 0xfb, 0x94, 0x0d,
};
static const unsigned char kat118_nor_addin0[] = {
    0x16, 0x64, 0xf7, 0xcc, 0x88, 0xd0, 0x1d, 0x7f, 0x27, 0x86, 0x15, 0xc5,
    0x63, 0x09, 0x2f, 0xd6,
};
static const unsigned char kat118_nor_addin1[] = {
    0x60, 0xb7, 0xa8, 0x6b, 0xc0, 0xc8, 0xdd, 0x2b, 0xd0, 0x7a, 0xda, 0xa4,
    0xd5, 0xc2, 0x4d, 0x68,
};
static const unsigned char kat118_nor_retbytes[] = {
    0xb4, 0xed, 0xc9, 0x65, 0x0d, 0x81, 0x47, 0x17, 0x0c, 0x81, 0x52, 0x9f,
    0x2f, 0xd7, 0x33, 0x11, 0x53, 0xa6, 0x1d, 0x0d, 0x58, 0xb7, 0x6b, 0x50,
    0xbb, 0x52, 0xda, 0x0a, 0x79, 0x51, 0x64, 0xbe, 0xeb, 0xc6, 0x09, 0x32,
    0x05, 0x20, 0x32, 0x8f, 0x3d, 0x4e, 0x83, 0xcd, 0x80, 0xd0, 0xb1, 0xcb,
    0xbb, 0xfd, 0x0a, 0x84, 0xa9, 0x65, 0x10, 0x88, 0x11, 0x01, 0xfb, 0xc7,
    0x40, 0xb2, 0x81, 0x7f,
};
static const struct drbg_kat_no_reseed kat118_nor_t = {
    12, kat118_nor_entropyin, kat118_nor_nonce, kat118_nor_persstr,
    kat118_nor_addin0, kat118_nor_addin1, kat118_nor_retbytes
};
static const struct drbg_kat kat118_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat118_nor_t
};

static const unsigned char kat119_nor_entropyin[] = {
    0x36, 0xe2, 0x7a, 0xa6, 0xb3, 0x66, 0x99, 0xe9, 0x46, 0x45, 0x95, 0xa9,
    0x32, 0xcc, 0x92, 0x8e,
};
static const unsigned char kat119_nor_nonce[] = {
    0x65, 0x16, 0xbb, 0x91, 0x85, 0x66, 0xbf, 0x89,
};
static const unsigned char kat119_nor_persstr[] = {
    0x9b, 0xc6, 0x83, 0xa1, 0xf4, 0x65, 0xba, 0x0f, 0x2c, 0xe1, 0x8c, 0x8b,
    0x9c, 0x61, 0x42, 0x44,
};
static const unsigned char kat119_nor_addin0[] = {
    0x0d, 0x17, 0x5b, 0x34, 0xd8, 0xb9, 0xe2, 0x70, 0x74, 0x5a, 0x71, 0xf3,
    0x5d, 0x41, 0x6f, 0x10,
};
static const unsigned char kat119_nor_addin1[] = {
    0xa7, 0x90, 0xc8, 0x6f, 0xdf, 0x0c, 0xa8, 0x0e, 0xb3, 0xd6, 0xf7, 0x02,
    0x91, 0x6e, 0xa9, 0x9f,
};
static const unsigned char kat119_nor_retbytes[] = {
    0x34, 0xea, 0x96, 0x29, 0x62, 0x7f, 0x66, 0x84, 0xb9, 0xab, 0x85, 0xf8,
    0x16, 0x46, 0x86, 0x4d, 0xd8, 0xef, 0x3a, 0x15, 0xed, 0xc8, 0x01, 0xdc,
    0x39, 0xa7, 0x31, 0xdc, 0x9e, 0x80, 0xe6, 0x26, 0xfc, 0x8c, 0xe2, 0x1e,
    0x9b, 0x62, 0xe9, 0x48, 0x88, 0x64, 0x89, 0xd5, 0xe4, 0x43, 0x6e, 0x65,
    0x9f, 0x26, 0xbf, 0xb3, 0x43, 0x51, 0x04, 0xad, 0x9b, 0xfa, 0x0a, 0xaf,
    0xc5, 0x50, 0x69, 0x17,
};
static const struct drbg_kat_no_reseed kat119_nor_t = {
    13, kat119_nor_entropyin, kat119_nor_nonce, kat119_nor_persstr,
    kat119_nor_addin0, kat119_nor_addin1, kat119_nor_retbytes
};
static const struct drbg_kat kat119_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat119_nor_t
};

static const unsigned char kat120_nor_entropyin[] = {
    0xac, 0xa3, 0x59, 0xf1, 0x07, 0xc3, 0x24, 0x64, 0x8a, 0xd4, 0x45, 0x56,
    0x4b, 0xfa, 0xe5, 0x6d,
};
static const unsigned char kat120_nor_nonce[] = {
    0xf9, 0x5d, 0x1c, 0x8f, 0xd2, 0x66, 0xdf, 0x64,
};
static const unsigned char kat120_nor_persstr[] = {
    0xa5, 0xcf, 0xdc, 0x76, 0xc4, 0xe8, 0x49, 0xe4, 0x5d, 0x8b, 0xcc, 0x68,
    0xee, 0xb0, 0x12, 0x84,
};
static const unsigned char kat120_nor_addin0[] = {
    0x57, 0x80, 0x2c, 0xc0, 0x6e, 0xf0, 0x34, 0x4b, 0xd1, 0x4a, 0x48, 0x57,
    0xc6, 0x9f, 0x68, 0xd4,
};
static const unsigned char kat120_nor_addin1[] = {
    0x53, 0x14, 0xfc, 0xcb, 0xe4, 0x52, 0x01, 0x8f, 0xc1, 0x6e, 0x5f, 0xfb,
    0x5d, 0x90, 0xe8, 0x88,
};
static const unsigned char kat120_nor_retbytes[] = {
    0x21, 0x1d, 0x90, 0xec, 0xd7, 0xd3, 0xdd, 0xd0, 0x3f, 0x91, 0xe8, 0xa6,
    0x74, 0x40, 0x0f, 0xb1, 0x8d, 0x3f, 0xb3, 0x98, 0x86, 0xa2, 0x26, 0x2f,
    0x0e, 0x09, 0xff, 0x07, 0x5f, 0x99, 0x8d, 0x21, 0x74, 0x19, 0xf2, 0xe9,
    0x35, 0x3d, 0x45, 0xb9, 0xdf, 0x10, 0x72, 0x3e, 0xba, 0xeb, 0xcf, 0xf1,
    0xaa, 0x80, 0x24, 0xd9, 0xb0, 0x96, 0x06, 0x4d, 0x6f, 0x31, 0x83, 0x5b,
    0x75, 0xf8, 0xea, 0xa6,
};
static const struct drbg_kat_no_reseed kat120_nor_t = {
    14, kat120_nor_entropyin, kat120_nor_nonce, kat120_nor_persstr,
    kat120_nor_addin0, kat120_nor_addin1, kat120_nor_retbytes
};
static const struct drbg_kat kat120_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat120_nor_t
};

static const unsigned char kat121_nor_entropyin[] = {
    0x2e, 0x17, 0x24, 0xdb, 0x48, 0x22, 0x32, 0xa3, 0xe6, 0x1f, 0x92, 0xc1,
    0xc2, 0x66, 0xfa, 0xf8,
};
static const unsigned char kat121_nor_nonce[] = {
    0x38, 0xaa, 0x55, 0x90, 0xf6, 0xbf, 0xaa, 0x4b,
};
static const unsigned char kat121_nor_persstr[] = {0};
static const unsigned char kat121_nor_addin0[] = {0};
static const unsigned char kat121_nor_addin1[] = {0};
static const unsigned char kat121_nor_retbytes[] = {
    0x44, 0x38, 0xb4, 0x8a, 0x45, 0xfb, 0x01, 0x41, 0xe3, 0x1f, 0x0a, 0x96,
    0x24, 0xdf, 0xe6, 0xfc, 0xc2, 0xf9, 0xed, 0xc0, 0x75, 0xc0, 0xa5, 0x2b,
    0xc5, 0xfc, 0x46, 0xd8, 0x5a, 0x96, 0x6c, 0x85, 0x3f, 0xee, 0xe6, 0xaf,
    0x91, 0x32, 0x34, 0xb3, 0xf9, 0xa6, 0x79, 0xf6, 0x67, 0x89, 0x8d, 0xc1,
    0x5a, 0x24, 0xaa, 0xed, 0x89, 0xf0, 0x35, 0xbf, 0xa5, 0xda, 0x51, 0x6e,
    0x43, 0x5b, 0xba, 0xd1,
};
static const struct drbg_kat_no_reseed kat121_nor_t = {
    0, kat121_nor_entropyin, kat121_nor_nonce, kat121_nor_persstr,
    kat121_nor_addin0, kat121_nor_addin1, kat121_nor_retbytes
};
static const struct drbg_kat kat121_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat121_nor_t
};

static const unsigned char kat122_nor_entropyin[] = {
    0x22, 0x56, 0x4f, 0x77, 0xc4, 0x5b, 0x05, 0x3c, 0xdf, 0x61, 0x43, 0x3e,
    0xb9, 0x6b, 0x1d, 0x7c,
};
static const unsigned char kat122_nor_nonce[] = {
    0xcf, 0x73, 0xe6, 0x20, 0xf8, 0x51, 0x52, 0x03,
};
static const unsigned char kat122_nor_persstr[] = {0};
static const unsigned char kat122_nor_addin0[] = {0};
static const unsigned char kat122_nor_addin1[] = {0};
static const unsigned char kat122_nor_retbytes[] = {
    0xc7, 0x90, 0x8e, 0x71, 0x2c, 0x71, 0x6d, 0x1f, 0x5e, 0xd5, 0x33, 0xe1,
    0x42, 0xe7, 0x21, 0x87, 0xea, 0x77, 0xfb, 0x4f, 0x51, 0x6d, 0xc3, 0x1a,
    0xa1, 0x0a, 0x1e, 0x54, 0x9d, 0x85, 0xea, 0xdb, 0x7a, 0x46, 0x46, 0x17,
    0x04, 0x64, 0xc1, 0xf7, 0xa7, 0x52, 0xc0, 0x1a, 0x94, 0x06, 0xbe, 0x66,
    0x43, 0xee, 0x96, 0x7d, 0x04, 0x64, 0xb8, 0x4b, 0x6a, 0x08, 0xb2, 0xed,
    0x0a, 0x7a, 0xcb, 0x07,
};
static const struct drbg_kat_no_reseed kat122_nor_t = {
    1, kat122_nor_entropyin, kat122_nor_nonce, kat122_nor_persstr,
    kat122_nor_addin0, kat122_nor_addin1, kat122_nor_retbytes
};
static const struct drbg_kat kat122_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat122_nor_t
};

static const unsigned char kat123_nor_entropyin[] = {
    0x2a, 0x5b, 0x01, 0x2b, 0x09, 0x79, 0x26, 0xe8, 0xf8, 0x57, 0x0f, 0xf8,
    0x69, 0x2c, 0xc5, 0xd1,
};
static const unsigned char kat123_nor_nonce[] = {
    0xde, 0x8e, 0x07, 0x2d, 0x15, 0x81, 0xaf, 0xe6,
};
static const unsigned char kat123_nor_persstr[] = {0};
static const unsigned char kat123_nor_addin0[] = {0};
static const unsigned char kat123_nor_addin1[] = {0};
static const unsigned char kat123_nor_retbytes[] = {
    0x81, 0xf2, 0xe2, 0xc8, 0x58, 0x5e, 0x8d, 0xc4, 0x65, 0xd7, 0x8e, 0x7b,
    0x75, 0xb9, 0xf6, 0xc2, 0xbb, 0xdc, 0xde, 0x94, 0x75, 0xd4, 0x25, 0x0f,
    0xb4, 0x9e, 0x04, 0xc5, 0x6e, 0x30, 0x48, 0x9e, 0x24, 0xdf, 0x48, 0x58,
    0xf7, 0x4e, 0xd0, 0x85, 0xcb, 0xa9, 0xf9, 0x92, 0xeb, 0x7d, 0x13, 0xe4,
    0xe0, 0x64, 0xa7, 0x45, 0xf4, 0x51, 0xcb, 0x6e, 0xdf, 0xc3, 0x7c, 0x57,
    0xf3, 0x5e, 0x8d, 0x57,
};
static const struct drbg_kat_no_reseed kat123_nor_t = {
    2, kat123_nor_entropyin, kat123_nor_nonce, kat123_nor_persstr,
    kat123_nor_addin0, kat123_nor_addin1, kat123_nor_retbytes
};
static const struct drbg_kat kat123_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat123_nor_t
};

static const unsigned char kat124_nor_entropyin[] = {
    0x2c, 0x6b, 0x05, 0xd1, 0xc8, 0x6a, 0xae, 0x86, 0xa8, 0x9e, 0x81, 0x64,
    0x82, 0x98, 0x32, 0x36,
};
static const unsigned char kat124_nor_nonce[] = {
    0x33, 0x85, 0xd9, 0x29, 0xad, 0xe9, 0x96, 0xe8,
};
static const unsigned char kat124_nor_persstr[] = {0};
static const unsigned char kat124_nor_addin0[] = {0};
static const unsigned char kat124_nor_addin1[] = {0};
static const unsigned char kat124_nor_retbytes[] = {
    0x52, 0x56, 0xa6, 0x44, 0x29, 0xa5, 0x56, 0xd1, 0xa9, 0x1d, 0x58, 0x99,
    0x9c, 0x75, 0xb3, 0x6d, 0xe7, 0xcc, 0x01, 0xf7, 0x46, 0x3c, 0x4e, 0x24,
    0xaf, 0xd1, 0x5d, 0xe0, 0xa3, 0x5d, 0xcb, 0x5a, 0xda, 0x26, 0x79, 0x13,
    0x4f, 0x15, 0xf4, 0xc5, 0x1d, 0xc0, 0x6b, 0x34, 0x45, 0x4d, 0x6d, 0xca,
    0xa1, 0xd2, 0x51, 0x1c, 0x1d, 0x22, 0x6f, 0x23, 0x2f, 0x44, 0x52, 0x76,
    0x25, 0x57, 0x51, 0xe6,
};
static const struct drbg_kat_no_reseed kat124_nor_t = {
    3, kat124_nor_entropyin, kat124_nor_nonce, kat124_nor_persstr,
    kat124_nor_addin0, kat124_nor_addin1, kat124_nor_retbytes
};
static const struct drbg_kat kat124_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat124_nor_t
};

static const unsigned char kat125_nor_entropyin[] = {
    0x48, 0xba, 0xb2, 0xf8, 0x2d, 0x80, 0x59, 0x7f, 0x93, 0xad, 0xdb, 0x7e,
    0x1f, 0x0f, 0x2e, 0x72,
};
static const unsigned char kat125_nor_nonce[] = {
    0xf5, 0xf4, 0x90, 0x18, 0xe8, 0x55, 0x9b, 0x0a,
};
static const unsigned char kat125_nor_persstr[] = {0};
static const unsigned char kat125_nor_addin0[] = {0};
static const unsigned char kat125_nor_addin1[] = {0};
static const unsigned char kat125_nor_retbytes[] = {
    0xfa, 0x32, 0x3d, 0x2d, 0xae, 0x96, 0x74, 0xd8, 0x0b, 0xd2, 0xf1, 0x42,
    0x7c, 0x1c, 0x79, 0x53, 0x2b, 0x23, 0x74, 0xed, 0x1f, 0xb3, 0xa1, 0x3c,
    0x62, 0x06, 0x05, 0x04, 0x8a, 0xc5, 0x78, 0xb3, 0x07, 0x0c, 0x67, 0x48,
    0x31, 0x4e, 0x5e, 0xd1, 0xdb, 0xd3, 0xea, 0xaa, 0x64, 0x1e, 0x50, 0x5c,
    0x3d, 0x3f, 0x59, 0xfa, 0xc2, 0x5d, 0x89, 0x7b, 0xf3, 0x94, 0xdc, 0xad,
    0xb6, 0x3b, 0x7f, 0xf9,
};
static const struct drbg_kat_no_reseed kat125_nor_t = {
    4, kat125_nor_entropyin, kat125_nor_nonce, kat125_nor_persstr,
    kat125_nor_addin0, kat125_nor_addin1, kat125_nor_retbytes
};
static const struct drbg_kat kat125_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat125_nor_t
};

static const unsigned char kat126_nor_entropyin[] = {
    0xad, 0x7a, 0xff, 0x42, 0x24, 0xe9, 0x3f, 0x32, 0x35, 0x45, 0x41, 0x6a,
    0x1e, 0x56, 0x97, 0xce,
};
static const unsigned char kat126_nor_nonce[] = {
    0x9c, 0xc1, 0x2c, 0xe2, 0xed, 0x5e, 0x8d, 0x1c,
};
static const unsigned char kat126_nor_persstr[] = {0};
static const unsigned char kat126_nor_addin0[] = {0};
static const unsigned char kat126_nor_addin1[] = {0};
static const unsigned char kat126_nor_retbytes[] = {
    0x36, 0xc7, 0x22, 0x23, 0xb9, 0x24, 0x4c, 0xdb, 0x2c, 0x2c, 0x0d, 0xd5,
    0xa5, 0x97, 0x06, 0x55, 0x8e, 0x2e, 0x5a, 0x11, 0x84, 0x5c, 0xab, 0xf9,
    0x54, 0x5c, 0xd4, 0xad, 0x08, 0x15, 0x4a, 0x46, 0x70, 0x3a, 0xe7, 0x50,
    0xb7, 0xf0, 0xc4, 0xf5, 0xbb, 0x33, 0xac, 0xd3, 0xc3, 0x81, 0xe5, 0xee,
    0x4c, 0xe0, 0x99, 0x16, 0x43, 0x11, 0x44, 0xe8, 0x51, 0x5f, 0xed, 0x91,
    0x4d, 0x5c, 0x5b, 0x5e,
};
static const struct drbg_kat_no_reseed kat126_nor_t = {
    5, kat126_nor_entropyin, kat126_nor_nonce, kat126_nor_persstr,
    kat126_nor_addin0, kat126_nor_addin1, kat126_nor_retbytes
};
static const struct drbg_kat kat126_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat126_nor_t
};

static const unsigned char kat127_nor_entropyin[] = {
    0x29, 0x9c, 0x01, 0xd3, 0xa2, 0xf1, 0x32, 0x3d, 0xf7, 0x53, 0xcf, 0x14,
    0x84, 0x5e, 0x0d, 0xe5,
};
static const unsigned char kat127_nor_nonce[] = {
    0x51, 0x1e, 0x36, 0x23, 0x2a, 0x11, 0x29, 0x1d,
};
static const unsigned char kat127_nor_persstr[] = {0};
static const unsigned char kat127_nor_addin0[] = {0};
static const unsigned char kat127_nor_addin1[] = {0};
static const unsigned char kat127_nor_retbytes[] = {
    0x77, 0x09, 0xfd, 0xc6, 0x27, 0x8d, 0xb4, 0x4b, 0x21, 0xd3, 0x9a, 0x19,
    0x4b, 0x80, 0x6e, 0x48, 0xe7, 0xfe, 0x3e, 0x9a, 0xe1, 0x16, 0xe2, 0x38,
    0xc2, 0x05, 0xc2, 0xc3, 0x45, 0x98, 0x1c, 0xe8, 0x1f, 0x25, 0x57, 0x13,
    0x59, 0x7c, 0xec, 0x2b, 0x3a, 0xd3, 0x91, 0x0f, 0x2b, 0x67, 0x42, 0xae,
    0xa6, 0x64, 0x04, 0x71, 0x2d, 0xf8, 0x32, 0x8d, 0x2f, 0x2d, 0x19, 0x48,
    0x41, 0x30, 0x97, 0xdb,
};
static const struct drbg_kat_no_reseed kat127_nor_t = {
    6, kat127_nor_entropyin, kat127_nor_nonce, kat127_nor_persstr,
    kat127_nor_addin0, kat127_nor_addin1, kat127_nor_retbytes
};
static const struct drbg_kat kat127_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat127_nor_t
};

static const unsigned char kat128_nor_entropyin[] = {
    0xb6, 0xee, 0x77, 0x79, 0x99, 0x4b, 0xa8, 0xcd, 0x49, 0x0a, 0x03, 0xcf,
    0x68, 0x99, 0xb1, 0x35,
};
static const unsigned char kat128_nor_nonce[] = {
    0x66, 0xed, 0xa9, 0xb5, 0xa5, 0x4d, 0x7d, 0xed,
};
static const unsigned char kat128_nor_persstr[] = {0};
static const unsigned char kat128_nor_addin0[] = {0};
static const unsigned char kat128_nor_addin1[] = {0};
static const unsigned char kat128_nor_retbytes[] = {
    0x4e, 0x21, 0xb4, 0x8f, 0xde, 0x08, 0x22, 0x63, 0xd7, 0x6a, 0x10, 0x34,
    0xe8, 0x7a, 0x56, 0x6e, 0x1a, 0x1c, 0x9d, 0x2e, 0x1b, 0xd5, 0xc7, 0x48,
    0xe3, 0x0e, 0x1d, 0x87, 0x50, 0xf2, 0xff, 0x03, 0x93, 0x1c, 0x4b, 0xfe,
    0x19, 0x4d, 0x2d, 0xa4, 0xed, 0x1c, 0xf1, 0x53, 0x03, 0x01, 0xe5, 0xb1,
    0xab, 0xc4, 0xbd, 0x2b, 0xda, 0x7b, 0xe8, 0x92, 0x84, 0xf8, 0xc2, 0x19,
    0x36, 0x88, 0xc9, 0x82,
};
static const struct drbg_kat_no_reseed kat128_nor_t = {
    7, kat128_nor_entropyin, kat128_nor_nonce, kat128_nor_persstr,
    kat128_nor_addin0, kat128_nor_addin1, kat128_nor_retbytes
};
static const struct drbg_kat kat128_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat128_nor_t
};

static const unsigned char kat129_nor_entropyin[] = {
    0xe3, 0xcc, 0xb9, 0x91, 0xc3, 0xe1, 0xb3, 0xed, 0xa0, 0xb4, 0x0b, 0x51,
    0x42, 0xac, 0x84, 0xd3,
};
static const unsigned char kat129_nor_nonce[] = {
    0x99, 0x87, 0x57, 0xe0, 0x0d, 0xa2, 0xb9, 0xef,
};
static const unsigned char kat129_nor_persstr[] = {0};
static const unsigned char kat129_nor_addin0[] = {0};
static const unsigned char kat129_nor_addin1[] = {0};
static const unsigned char kat129_nor_retbytes[] = {
    0xd3, 0x2b, 0xc1, 0x90, 0x99, 0x8f, 0x18, 0xe9, 0xd5, 0x50, 0x9f, 0x46,
    0x02, 0x29, 0x05, 0xd0, 0x11, 0xbb, 0xec, 0x77, 0x4f, 0x05, 0x83, 0x69,
    0x1d, 0x48, 0x12, 0x43, 0x79, 0xe8, 0x1d, 0x99, 0xf0, 0xcd, 0xd4, 0x61,
    0x38, 0xbc, 0xcc, 0x47, 0xeb, 0x77, 0x3f, 0x25, 0x7a, 0x66, 0x2b, 0x79,
    0x8f, 0xab, 0x27, 0x58, 0x86, 0x94, 0x89, 0x36, 0xbc, 0xe7, 0x2d, 0xbd,
    0x2c, 0x61, 0x88, 0xac,
};
static const struct drbg_kat_no_reseed kat129_nor_t = {
    8, kat129_nor_entropyin, kat129_nor_nonce, kat129_nor_persstr,
    kat129_nor_addin0, kat129_nor_addin1, kat129_nor_retbytes
};
static const struct drbg_kat kat129_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat129_nor_t
};

static const unsigned char kat130_nor_entropyin[] = {
    0xc3, 0x4e, 0x39, 0x80, 0x41, 0xcc, 0xee, 0x23, 0x89, 0x7f, 0x7c, 0xa3,
    0x43, 0xf3, 0x56, 0x40,
};
static const unsigned char kat130_nor_nonce[] = {
    0xa2, 0x4f, 0x8e, 0xa0, 0x88, 0x6b, 0xf6, 0xbf,
};
static const unsigned char kat130_nor_persstr[] = {0};
static const unsigned char kat130_nor_addin0[] = {0};
static const unsigned char kat130_nor_addin1[] = {0};
static const unsigned char kat130_nor_retbytes[] = {
    0x5f, 0xf0, 0xda, 0xef, 0x30, 0x04, 0xc5, 0x03, 0xb8, 0x09, 0x8e, 0x3b,
    0x96, 0x8a, 0x8e, 0x32, 0x33, 0x19, 0xbe, 0x78, 0x6c, 0x7b, 0x74, 0x2a,
    0xac, 0xee, 0x35, 0x5f, 0x1a, 0x3c, 0x9d, 0xe7, 0x50, 0x61, 0x10, 0x8e,
    0x79, 0x18, 0x13, 0x61, 0xf2, 0xe3, 0x30, 0x6a, 0xf0, 0x7b, 0xcd, 0xed,
    0x10, 0xe3, 0x2d, 0xef, 0x1b, 0x7b, 0xb3, 0xe4, 0xeb, 0xc1, 0x70, 0x96,
    0xc6, 0x93, 0x30, 0x58,
};
static const struct drbg_kat_no_reseed kat130_nor_t = {
    9, kat130_nor_entropyin, kat130_nor_nonce, kat130_nor_persstr,
    kat130_nor_addin0, kat130_nor_addin1, kat130_nor_retbytes
};
static const struct drbg_kat kat130_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat130_nor_t
};

static const unsigned char kat131_nor_entropyin[] = {
    0xfc, 0x1f, 0xd2, 0x5c, 0xed, 0x5b, 0xd3, 0x0a, 0x62, 0x1c, 0xd4, 0xae,
    0x77, 0x96, 0x80, 0xad,
};
static const unsigned char kat131_nor_nonce[] = {
    0xaf, 0x17, 0xa9, 0xc9, 0x74, 0xb3, 0x6e, 0x6c,
};
static const unsigned char kat131_nor_persstr[] = {0};
static const unsigned char kat131_nor_addin0[] = {0};
static const unsigned char kat131_nor_addin1[] = {0};
static const unsigned char kat131_nor_retbytes[] = {
    0x59, 0xf9, 0x9d, 0x08, 0x57, 0x49, 0x36, 0x74, 0x78, 0x68, 0x4a, 0x5d,
    0xdc, 0x8f, 0xe1, 0x81, 0xb9, 0x7a, 0x4e, 0x67, 0xfd, 0xe5, 0xc1, 0x51,
    0xc4, 0x69, 0x6d, 0x52, 0x3d, 0x7c, 0x14, 0xb7, 0x26, 0x89, 0xa9, 0x5a,
    0x5b, 0x60, 0x92, 0xe9, 0x49, 0xdd, 0x16, 0x3b, 0xd8, 0xf9, 0xe4, 0x57,
    0x27, 0xd2, 0xb8, 0xa3, 0x1e, 0xd2, 0x88, 0xc8, 0xc6, 0x22, 0x9e, 0x8b,
    0xe6, 0x80, 0x8e, 0xc8,
};
static const struct drbg_kat_no_reseed kat131_nor_t = {
    10, kat131_nor_entropyin, kat131_nor_nonce, kat131_nor_persstr,
    kat131_nor_addin0, kat131_nor_addin1, kat131_nor_retbytes
};
static const struct drbg_kat kat131_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat131_nor_t
};

static const unsigned char kat132_nor_entropyin[] = {
    0x72, 0xc0, 0xe2, 0x3d, 0x92, 0x07, 0x0a, 0x8b, 0xab, 0x70, 0x7f, 0x65,
    0xd5, 0x95, 0x18, 0x6d,
};
static const unsigned char kat132_nor_nonce[] = {
    0x1a, 0x63, 0xdc, 0xfc, 0x52, 0xe5, 0x5c, 0x58,
};
static const unsigned char kat132_nor_persstr[] = {0};
static const unsigned char kat132_nor_addin0[] = {0};
static const unsigned char kat132_nor_addin1[] = {0};
static const unsigned char kat132_nor_retbytes[] = {
    0x49, 0xfe, 0xc2, 0x67, 0x58, 0x85, 0xd5, 0x4a, 0x4c, 0x6b, 0x10, 0x1f,
    0x29, 0x19, 0x45, 0xc7, 0x35, 0xad, 0x9c, 0x2d, 0xb5, 0x1a, 0x63, 0xb9,
    0x41, 0xbc, 0x18, 0x2e, 0xe5, 0x1f, 0xd8, 0xfd, 0x84, 0xb8, 0xc6, 0x33,
    0x7b, 0x0f, 0x77, 0xf3, 0x10, 0xca, 0x50, 0x69, 0x3b, 0x91, 0xb5, 0x90,
    0xe3, 0xef, 0x65, 0x5b, 0xe7, 0xad, 0x76, 0x21, 0xed, 0x21, 0xff, 0x39,
    0xd3, 0x29, 0x2e, 0xfd,
};
static const struct drbg_kat_no_reseed kat132_nor_t = {
    11, kat132_nor_entropyin, kat132_nor_nonce, kat132_nor_persstr,
    kat132_nor_addin0, kat132_nor_addin1, kat132_nor_retbytes
};
static const struct drbg_kat kat132_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat132_nor_t
};

static const unsigned char kat133_nor_entropyin[] = {
    0xa0, 0x90, 0x91, 0xca, 0x32, 0x80, 0xf7, 0xf5, 0x83, 0x76, 0xbf, 0x68,
    0x0d, 0xe1, 0x11, 0x92,
};
static const unsigned char kat133_nor_nonce[] = {
    0x1d, 0x08, 0xdc, 0x0d, 0x06, 0x00, 0x95, 0xb3,
};
static const unsigned char kat133_nor_persstr[] = {0};
static const unsigned char kat133_nor_addin0[] = {0};
static const unsigned char kat133_nor_addin1[] = {0};
static const unsigned char kat133_nor_retbytes[] = {
    0x0d, 0x59, 0xac, 0xa0, 0x23, 0x64, 0x7f, 0x0b, 0xf2, 0x68, 0x81, 0xd9,
    0x12, 0x6b, 0xab, 0x8c, 0x7f, 0xea, 0x92, 0x2d, 0x2b, 0x4c, 0x24, 0xf1,
    0xdc, 0xc0, 0xbf, 0xf8, 0x7a, 0x3d, 0x0d, 0x1b, 0x1d, 0xa0, 0xe8, 0x75,
    0x62, 0x6a, 0x56, 0x42, 0x61, 0x86, 0xef, 0xd0, 0x07, 0x1f, 0x5a, 0x78,
    0x9f, 0xbf, 0x35, 0xfa, 0x8b, 0xfc, 0x85, 0xaf, 0xaf, 0xd3, 0xaf, 0x6c,
    0x9c, 0x7c, 0xd0, 0x7a,
};
static const struct drbg_kat_no_reseed kat133_nor_t = {
    12, kat133_nor_entropyin, kat133_nor_nonce, kat133_nor_persstr,
    kat133_nor_addin0, kat133_nor_addin1, kat133_nor_retbytes
};
static const struct drbg_kat kat133_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat133_nor_t
};

static const unsigned char kat134_nor_entropyin[] = {
    0x8b, 0x41, 0x75, 0xd0, 0xa1, 0x95, 0x39, 0xef, 0x3d, 0x5d, 0x40, 0x84,
    0x6c, 0xb4, 0x0a, 0xb3,
};
static const unsigned char kat134_nor_nonce[] = {
    0x8c, 0x31, 0x0d, 0x43, 0x1c, 0xf0, 0x0b, 0x3e,
};
static const unsigned char kat134_nor_persstr[] = {0};
static const unsigned char kat134_nor_addin0[] = {0};
static const unsigned char kat134_nor_addin1[] = {0};
static const unsigned char kat134_nor_retbytes[] = {
    0x92, 0xb3, 0xb0, 0xe5, 0x7a, 0xe4, 0xce, 0xfc, 0x40, 0xd6, 0xe5, 0xfa,
    0x0d, 0x9f, 0xa8, 0x5c, 0x97, 0x0c, 0x2d, 0xd0, 0xcd, 0x4e, 0x04, 0xd7,
    0x27, 0x27, 0x56, 0x63, 0x4b, 0x84, 0xcc, 0x20, 0xc5, 0x36, 0x8f, 0x3a,
    0x7b, 0x3e, 0x12, 0x11, 0xc5, 0xfa, 0x2e, 0x63, 0x35, 0x43, 0x6b, 0x88,
    0x58, 0x2d, 0x04, 0x8b, 0xc7, 0x6a, 0x7c, 0x19, 0xbb, 0xfe, 0xc1, 0x35,
    0xa1, 0x05, 0x5c, 0xbd,
};
static const struct drbg_kat_no_reseed kat134_nor_t = {
    13, kat134_nor_entropyin, kat134_nor_nonce, kat134_nor_persstr,
    kat134_nor_addin0, kat134_nor_addin1, kat134_nor_retbytes
};
static const struct drbg_kat kat134_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat134_nor_t
};

static const unsigned char kat135_nor_entropyin[] = {
    0x68, 0xc8, 0x53, 0xb9, 0x12, 0x9c, 0xb2, 0x0a, 0x3d, 0xdf, 0x11, 0xe7,
    0x8a, 0x58, 0x75, 0xba,
};
static const unsigned char kat135_nor_nonce[] = {
    0xf5, 0xab, 0x54, 0x86, 0x05, 0xa5, 0x11, 0x03,
};
static const unsigned char kat135_nor_persstr[] = {0};
static const unsigned char kat135_nor_addin0[] = {0};
static const unsigned char kat135_nor_addin1[] = {0};
static const unsigned char kat135_nor_retbytes[] = {
    0x43, 0xcf, 0xb0, 0x3a, 0x51, 0xd7, 0xda, 0x40, 0xb3, 0x94, 0x68, 0x36,
    0x1c, 0x2e, 0xe0, 0x78, 0x81, 0x90, 0x75, 0x87, 0x2f, 0x4f, 0x7c, 0x5d,
    0x2b, 0x09, 0xef, 0x39, 0x91, 0x46, 0x02, 0xa7, 0x2a, 0x62, 0xc6, 0x3e,
    0x29, 0x38, 0x3f, 0xbb, 0x9e, 0x45, 0x0f, 0xb2, 0xae, 0xf3, 0x2e, 0xb9,
    0xf3, 0x70, 0xcb, 0xbc, 0x1a, 0xb4, 0x70, 0x8a, 0x5d, 0x28, 0x98, 0xdf,
    0x8a, 0xe4, 0xf6, 0x26,
};
static const struct drbg_kat_no_reseed kat135_nor_t = {
    14, kat135_nor_entropyin, kat135_nor_nonce, kat135_nor_persstr,
    kat135_nor_addin0, kat135_nor_addin1, kat135_nor_retbytes
};
static const struct drbg_kat kat135_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat135_nor_t
};

static const unsigned char kat136_nor_entropyin[] = {
    0x9b, 0xfa, 0xef, 0xb6, 0x98, 0xb1, 0xb5, 0xfc, 0xc6, 0x2d, 0xb2, 0xc1,
    0x64, 0x98, 0xc3, 0x3a,
};
static const unsigned char kat136_nor_nonce[] = {
    0x11, 0x1d, 0x86, 0x12, 0xa0, 0xf0, 0x4e, 0x2a,
};
static const unsigned char kat136_nor_persstr[] = {0};
static const unsigned char kat136_nor_addin0[] = {
    0xae, 0xdb, 0xe0, 0x28, 0x47, 0xb1, 0xb0, 0x8b, 0x6a, 0x67, 0x3b, 0xdf,
    0x25, 0xb0, 0x22, 0x4c,
};
static const unsigned char kat136_nor_addin1[] = {
    0x99, 0x01, 0xea, 0xd6, 0x2c, 0xe5, 0x65, 0x73, 0xb0, 0xf7, 0x1c, 0xd0,
    0x20, 0xfe, 0x34, 0x69,
};
static const unsigned char kat136_nor_retbytes[] = {
    0xdf, 0xf8, 0xbf, 0x2a, 0xec, 0x53, 0x1f, 0x85, 0x32, 0x60, 0x7e, 0x73,
    0x8b, 0xd7, 0x9f, 0x91, 0xd6, 0x08, 0x5c, 0xb1, 0x95, 0x68, 0xb7, 0xb0,
    0x24, 0x0c, 0xe6, 0xa6, 0xb3, 0x71, 0xa2, 0x82, 0xba, 0xfc, 0xdb, 0xa0,
    0x21, 0x37, 0xdf, 0x99, 0x05, 0x35, 0xd9, 0xeb, 0xf0, 0xba, 0x77, 0x11,
    0x77, 0x51, 0x62, 0x6b, 0x26, 0x78, 0xac, 0xa7, 0xbe, 0x4d, 0xec, 0xfd,
    0x6b, 0x9d, 0x4b, 0x38,
};
static const struct drbg_kat_no_reseed kat136_nor_t = {
    0, kat136_nor_entropyin, kat136_nor_nonce, kat136_nor_persstr,
    kat136_nor_addin0, kat136_nor_addin1, kat136_nor_retbytes
};
static const struct drbg_kat kat136_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat136_nor_t
};

static const unsigned char kat137_nor_entropyin[] = {
    0xdf, 0x99, 0x3f, 0xed, 0xd5, 0x96, 0x74, 0xa8, 0x7a, 0x15, 0x14, 0x7b,
    0x80, 0xbe, 0x37, 0xae,
};
static const unsigned char kat137_nor_nonce[] = {
    0x22, 0xb3, 0x31, 0x5a, 0xcc, 0xf1, 0xad, 0x13,
};
static const unsigned char kat137_nor_persstr[] = {0};
static const unsigned char kat137_nor_addin0[] = {
    0x9f, 0x9b, 0x94, 0x09, 0x04, 0x8a, 0x71, 0x17, 0x45, 0xc7, 0xef, 0x7a,
    0x6d, 0xdd, 0xc1, 0x7d,
};
static const unsigned char kat137_nor_addin1[] = {
    0xc5, 0x60, 0x68, 0x5b, 0xd4, 0x9c, 0x05, 0x9f, 0x04, 0x38, 0xe9, 0xdf,
    0xf6, 0x2d, 0x82, 0xd7,
};
static const unsigned char kat137_nor_retbytes[] = {
    0x04, 0xd7, 0x4f, 0xa1, 0xb6, 0x9d, 0xe6, 0x89, 0x3a, 0x47, 0xbf, 0xb0,
    0xb6, 0xae, 0x58, 0xa7, 0x98, 0x4b, 0xbb, 0x08, 0x8f, 0xce, 0x62, 0x0b,
    0x9d, 0x8e, 0xbc, 0x0b, 0x54, 0xcd, 0xdb, 0xca, 0x00, 0x45, 0xd7, 0x5d,
    0x5b, 0x04, 0x6f, 0xcd, 0x88, 0x95, 0xc1, 0x6b, 0x05, 0x13, 0xaa, 0x52,
    0x1b, 0x8d, 0x4a, 0xf2, 0x76, 0x78, 0x3d, 0x9d, 0x25, 0x77, 0xac, 0xb3,
    0x2c, 0xea, 0xdb, 0x89,
};
static const struct drbg_kat_no_reseed kat137_nor_t = {
    1, kat137_nor_entropyin, kat137_nor_nonce, kat137_nor_persstr,
    kat137_nor_addin0, kat137_nor_addin1, kat137_nor_retbytes
};
static const struct drbg_kat kat137_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat137_nor_t
};

static const unsigned char kat138_nor_entropyin[] = {
    0xdb, 0x4f, 0xae, 0x0c, 0x43, 0x02, 0xeb, 0xe8, 0x4f, 0x2a, 0x28, 0xad,
    0x98, 0x4d, 0xae, 0xc1,
};
static const unsigned char kat138_nor_nonce[] = {
    0x07, 0x9e, 0xbf, 0x0f, 0x00, 0x93, 0xcd, 0xb2,
};
static const unsigned char kat138_nor_persstr[] = {0};
static const unsigned char kat138_nor_addin0[] = {
    0x2f, 0xfb, 0x64, 0x85, 0xac, 0xe2, 0xad, 0x77, 0xd5, 0xf8, 0xad, 0xbd,
    0x09, 0xb3, 0xf3, 0x72,
};
static const unsigned char kat138_nor_addin1[] = {
    0xf4, 0xb8, 0x44, 0x74, 0x3d, 0x00, 0x65, 0xa0, 0x38, 0x98, 0x80, 0x49,
    0xf6, 0xaa, 0x53, 0xb5,
};
static const unsigned char kat138_nor_retbytes[] = {
    0xcb, 0x8e, 0x71, 0x44, 0x8f, 0xf7, 0x91, 0x1e, 0x5c, 0xae, 0xa7, 0xd5,
    0x4a, 0x12, 0x94, 0x5e, 0xd3, 0x4b, 0xea, 0x42, 0xdb, 0xb6, 0x57, 0x3b,
    0xf8, 0x42, 0x0d, 0xbb, 0x6b, 0xae, 0x6a, 0x11, 0xe0, 0x24, 0x82, 0x92,
    0xc9, 0xd9, 0x3d, 0xa3, 0x09, 0x68, 0x56, 0xe0, 0xf2, 0x94, 0x18, 0xf1,
    0xc8, 0x0d, 0x13, 0x8b, 0x34, 0x15, 0x66, 0x7b, 0xfb, 0x45, 0x6b, 0x08,
    0x9f, 0x26, 0x62, 0x1a,
};
static const struct drbg_kat_no_reseed kat138_nor_t = {
    2, kat138_nor_entropyin, kat138_nor_nonce, kat138_nor_persstr,
    kat138_nor_addin0, kat138_nor_addin1, kat138_nor_retbytes
};
static const struct drbg_kat kat138_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat138_nor_t
};

static const unsigned char kat139_nor_entropyin[] = {
    0xa4, 0x23, 0x48, 0xf1, 0xbe, 0xc0, 0x6f, 0x58, 0x36, 0xfc, 0xa0, 0x60,
    0x66, 0x14, 0x34, 0xc0,
};
static const unsigned char kat139_nor_nonce[] = {
    0x2f, 0x70, 0xf6, 0xa2, 0xe7, 0xd0, 0xb4, 0x36,
};
static const unsigned char kat139_nor_persstr[] = {0};
static const unsigned char kat139_nor_addin0[] = {
    0x54, 0x70, 0x91, 0xef, 0x85, 0xb0, 0x13, 0xf3, 0xf5, 0xdc, 0x82, 0x2a,
    0x5b, 0x3f, 0x27, 0xf9,
};
static const unsigned char kat139_nor_addin1[] = {
    0x2c, 0xa9, 0x9e, 0xe7, 0x97, 0xf4, 0xa9, 0xb1, 0x67, 0x88, 0xd2, 0x98,
    0xbb, 0xa2, 0xd1, 0x83,
};
static const unsigned char kat139_nor_retbytes[] = {
    0x4b, 0x41, 0xed, 0x62, 0x81, 0xa8, 0x1d, 0x44, 0x2a, 0x8c, 0xb0, 0x3a,
    0x81, 0x45, 0xfe, 0x78, 0x86, 0x3d, 0x25, 0xd7, 0xee, 0x70, 0xe7, 0x2d,
    0x4d, 0x3a, 0xfd, 0x51, 0x68, 0x16, 0x48, 0x76, 0xa6, 0x6b, 0x5a, 0xca,
    0x31, 0x8c, 0x9c, 0x91, 0x17, 0x23, 0x05, 0xe3, 0x88, 0xe1, 0xda, 0xc3,
    0x86, 0xcb, 0xab, 0x6d, 0x15, 0x0a, 0x39, 0x12, 0x36, 0x9c, 0x93, 0x90,
    0x08, 0x6b, 0xd7, 0x44,
};
static const struct drbg_kat_no_reseed kat139_nor_t = {
    3, kat139_nor_entropyin, kat139_nor_nonce, kat139_nor_persstr,
    kat139_nor_addin0, kat139_nor_addin1, kat139_nor_retbytes
};
static const struct drbg_kat kat139_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat139_nor_t
};

static const unsigned char kat140_nor_entropyin[] = {
    0x14, 0x83, 0xb9, 0x8e, 0x08, 0xc0, 0x40, 0x12, 0xb0, 0x3f, 0x1f, 0x61,
    0x02, 0xa2, 0x83, 0x06,
};
static const unsigned char kat140_nor_nonce[] = {
    0x89, 0xe6, 0x7c, 0xa1, 0xda, 0x05, 0xdd, 0x4f,
};
static const unsigned char kat140_nor_persstr[] = {0};
static const unsigned char kat140_nor_addin0[] = {
    0x30, 0x85, 0x9e, 0xcc, 0x30, 0x25, 0x04, 0x79, 0x30, 0xb3, 0x85, 0x3a,
    0xe6, 0x8a, 0xbd, 0xa2,
};
static const unsigned char kat140_nor_addin1[] = {
    0xf5, 0x05, 0x57, 0xf0, 0x40, 0x29, 0x28, 0x6c, 0xaf, 0xb2, 0xfc, 0xbf,
    0x14, 0x64, 0xe4, 0xe5,
};
static const unsigned char kat140_nor_retbytes[] = {
    0x45, 0xac, 0xf5, 0x2f, 0xf5, 0x69, 0xcc, 0x48, 0x15, 0x95, 0x88, 0x01,
    0x62, 0x89, 0x3b, 0x57, 0x9b, 0xe4, 0x05, 0x43, 0xc5, 0x6a, 0x94, 0xb7,
    0x18, 0x4b, 0xc9, 0x92, 0xd1, 0xdf, 0x37, 0x85, 0x18, 0xf9, 0x3f, 0xb4,
    0xd9, 0x75, 0x5d, 0x86, 0x58, 0xe9, 0x72, 0x2b, 0x34, 0x79, 0x53, 0x6b,
    0xa4, 0x83, 0x9a, 0x8e, 0xee, 0x7a, 0x65, 0x5f, 0xed, 0x88, 0x4a, 0xaf,
    0x5a, 0x55, 0x6e, 0xb7,
};
static const struct drbg_kat_no_reseed kat140_nor_t = {
    4, kat140_nor_entropyin, kat140_nor_nonce, kat140_nor_persstr,
    kat140_nor_addin0, kat140_nor_addin1, kat140_nor_retbytes
};
static const struct drbg_kat kat140_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat140_nor_t
};

static const unsigned char kat141_nor_entropyin[] = {
    0xbe, 0x9f, 0xcd, 0xd7, 0x11, 0xce, 0x1b, 0x0a, 0x3c, 0x97, 0xb1, 0xac,
    0x19, 0x79, 0x20, 0x7d,
};
static const unsigned char kat141_nor_nonce[] = {
    0xa8, 0x02, 0x94, 0xb5, 0xd6, 0x59, 0xff, 0xb4,
};
static const unsigned char kat141_nor_persstr[] = {0};
static const unsigned char kat141_nor_addin0[] = {
    0xb2, 0x0c, 0x04, 0x54, 0x6a, 0x0c, 0x6a, 0x5e, 0x6b, 0x95, 0x63, 0x7c,
    0x96, 0x0b, 0xca, 0x63,
};
static const unsigned char kat141_nor_addin1[] = {
    0x67, 0x07, 0xcc, 0x21, 0x7f, 0xb1, 0x98, 0xaf, 0x85, 0x2e, 0x06, 0xdb,
    0x14, 0x61, 0x58, 0x05,
};
static const unsigned char kat141_nor_retbytes[] = {
    0x6b, 0x62, 0x0c, 0x76, 0xf0, 0xb1, 0xa4, 0xa3, 0xd7, 0xf0, 0xf6, 0x0a,
    0x76, 0x45, 0x81, 0x1b, 0xb7, 0x9b, 0xf1, 0x84, 0x77, 0xf8, 0x5b, 0x94,
    0xf0, 0xee, 0x09, 0xe0, 0xc6, 0x8b, 0x0f, 0xdb, 0x3e, 0x11, 0xd8, 0xe5,
    0x8a, 0x34, 0xd2, 0x4f, 0xe3, 0x6e, 0x8b, 0x7e, 0xa7, 0x2a, 0x26, 0xb6,
    0xcb, 0x92, 0xd9, 0x87, 0xc5, 0x81, 0xab, 0x48, 0xe5, 0xfb, 0x8f, 0xb7,
    0x0f, 0x76, 0x5f, 0x5f,
};
static const struct drbg_kat_no_reseed kat141_nor_t = {
    5, kat141_nor_entropyin, kat141_nor_nonce, kat141_nor_persstr,
    kat141_nor_addin0, kat141_nor_addin1, kat141_nor_retbytes
};
static const struct drbg_kat kat141_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat141_nor_t
};

static const unsigned char kat142_nor_entropyin[] = {
    0xa5, 0x8e, 0xb5, 0x4d, 0x10, 0x9b, 0x51, 0x4d, 0xb3, 0x38, 0xfc, 0xce,
    0xa2, 0x8e, 0xf2, 0xea,
};
static const unsigned char kat142_nor_nonce[] = {
    0x91, 0x2d, 0x24, 0xa1, 0xd8, 0x1d, 0xe7, 0xdc,
};
static const unsigned char kat142_nor_persstr[] = {0};
static const unsigned char kat142_nor_addin0[] = {
    0x59, 0xd4, 0x59, 0xb0, 0x73, 0x52, 0x60, 0x21, 0xec, 0x45, 0x51, 0x12,
    0x56, 0xcf, 0xb3, 0x58,
};
static const unsigned char kat142_nor_addin1[] = {
    0xf7, 0x9f, 0x7e, 0x5a, 0x2a, 0x79, 0x1a, 0x39, 0xbb, 0x32, 0xb6, 0xb1,
    0x5f, 0xe0, 0x14, 0x61,
};
static const unsigned char kat142_nor_retbytes[] = {
    0xc1, 0x26, 0xd4, 0x09, 0xc2, 0x0d, 0x5d, 0x3d, 0x4c, 0x79, 0x46, 0x12,
    0x56, 0xf5, 0x40, 0x3d, 0x1c, 0xc5, 0x9b, 0xcf, 0x11, 0xa4, 0xa6, 0x16,
    0xab, 0xf0, 0x65, 0x30, 0x32, 0xe0, 0x45, 0x0a, 0x11, 0xf3, 0x2f, 0x38,
    0x16, 0xc3, 0x51, 0x91, 0x2f, 0xe8, 0xe3, 0x00, 0x84, 0xed, 0x34, 0xdb,
    0xcd, 0x0a, 0x9c, 0x95, 0xe1, 0xc0, 0x13, 0x62, 0xef, 0x61, 0x6d, 0xd2,
    0x2a, 0x13, 0x7f, 0x72,
};
static const struct drbg_kat_no_reseed kat142_nor_t = {
    6, kat142_nor_entropyin, kat142_nor_nonce, kat142_nor_persstr,
    kat142_nor_addin0, kat142_nor_addin1, kat142_nor_retbytes
};
static const struct drbg_kat kat142_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat142_nor_t
};

static const unsigned char kat143_nor_entropyin[] = {
    0xe0, 0xa5, 0xd9, 0x4d, 0x0c, 0xa9, 0x5a, 0x7b, 0x38, 0x0b, 0x99, 0x9c,
    0x69, 0xd9, 0x01, 0x1c,
};
static const unsigned char kat143_nor_nonce[] = {
    0x47, 0xcd, 0xc5, 0x5a, 0x19, 0x56, 0xaf, 0x8d,
};
static const unsigned char kat143_nor_persstr[] = {0};
static const unsigned char kat143_nor_addin0[] = {
    0xa9, 0x16, 0x84, 0xd0, 0x12, 0xdc, 0x48, 0x83, 0x0b, 0xe4, 0xcf, 0xae,
    0xe2, 0xfc, 0x88, 0x54,
};
static const unsigned char kat143_nor_addin1[] = {
    0xa0, 0x61, 0xc2, 0xb1, 0x3b, 0xf1, 0x00, 0x6d, 0x3b, 0xa2, 0xf2, 0x29,
    0x7f, 0x95, 0x4f, 0xb5,
};
static const unsigned char kat143_nor_retbytes[] = {
    0xe1, 0x36, 0xea, 0x57, 0x3d, 0x1d, 0x81, 0xec, 0xb7, 0x84, 0x2e, 0xc4,
    0x3a, 0xf0, 0xb4, 0xb9, 0x78, 0x3f, 0x3f, 0xe4, 0xb1, 0xab, 0x90, 0x24,
    0xc6, 0x2e, 0xae, 0xf0, 0x86, 0x0e, 0x81, 0x3d, 0x5a, 0x24, 0xf1, 0xa5,
    0xfa, 0xb7, 0x4b, 0x8f, 0x1f, 0x66, 0x1b, 0x50, 0x39, 0x29, 0x02, 0x56,
    0xd0, 0xe5, 0xaa, 0xa9, 0xe0, 0xfd, 0xab, 0x3a, 0x18, 0x9b, 0x2d, 0x66,
    0x95, 0x89, 0x39, 0x99,
};
static const struct drbg_kat_no_reseed kat143_nor_t = {
    7, kat143_nor_entropyin, kat143_nor_nonce, kat143_nor_persstr,
    kat143_nor_addin0, kat143_nor_addin1, kat143_nor_retbytes
};
static const struct drbg_kat kat143_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat143_nor_t
};

static const unsigned char kat144_nor_entropyin[] = {
    0x91, 0x8b, 0xda, 0x9e, 0x89, 0x72, 0x7b, 0xbd, 0x97, 0x04, 0x25, 0xc7,
    0x48, 0x3e, 0xf1, 0xaf,
};
static const unsigned char kat144_nor_nonce[] = {
    0xfc, 0x3b, 0xee, 0xec, 0x38, 0x0b, 0x02, 0xc2,
};
static const unsigned char kat144_nor_persstr[] = {0};
static const unsigned char kat144_nor_addin0[] = {
    0x6f, 0x48, 0x36, 0x73, 0xaf, 0x64, 0xbc, 0xd7, 0x4c, 0xfe, 0xff, 0x3f,
    0x98, 0xb6, 0xcd, 0x5a,
};
static const unsigned char kat144_nor_addin1[] = {
    0x15, 0xbd, 0x5e, 0xab, 0xba, 0x23, 0x2d, 0xf5, 0xa1, 0xb6, 0x03, 0xa2,
    0xfc, 0x16, 0x39, 0xc0,
};
static const unsigned char kat144_nor_retbytes[] = {
    0x43, 0x85, 0x55, 0x1a, 0x3a, 0x7d, 0xd9, 0x0d, 0x77, 0xa7, 0x65, 0xf7,
    0xcb, 0x85, 0x9d, 0x80, 0xab, 0x6a, 0xc4, 0x84, 0x8e, 0x3b, 0xd9, 0x11,
    0x01, 0xb4, 0x52, 0x32, 0x34, 0xca, 0x0f, 0x16, 0xc0, 0x6a, 0x45, 0x79,
    0xa9, 0x7e, 0xa5, 0x1d, 0xb0, 0x37, 0xfd, 0x8b, 0x8c, 0x40, 0x83, 0xcd,
    0xb6, 0xf4, 0xed, 0x97, 0x56, 0xf5, 0xa4, 0x48, 0x8b, 0xb5, 0xdf, 0xcf,
    0x7a, 0x8f, 0x8a, 0x9a,
};
static const struct drbg_kat_no_reseed kat144_nor_t = {
    8, kat144_nor_entropyin, kat144_nor_nonce, kat144_nor_persstr,
    kat144_nor_addin0, kat144_nor_addin1, kat144_nor_retbytes
};
static const struct drbg_kat kat144_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat144_nor_t
};

static const unsigned char kat145_nor_entropyin[] = {
    0xd8, 0xad, 0xd3, 0x82, 0xd4, 0xdc, 0x7a, 0x1d, 0x10, 0xd4, 0x63, 0x03,
    0x17, 0x39, 0x18, 0x01,
};
static const unsigned char kat145_nor_nonce[] = {
    0x6f, 0xc0, 0xb1, 0xa3, 0xaa, 0x2b, 0x3a, 0xdb,
};
static const unsigned char kat145_nor_persstr[] = {0};
static const unsigned char kat145_nor_addin0[] = {
    0x10, 0x1f, 0xcf, 0xd6, 0x60, 0x80, 0x75, 0xdc, 0xbb, 0xf2, 0xeb, 0xe8,
    0x32, 0xd9, 0x6b, 0x13,
};
static const unsigned char kat145_nor_addin1[] = {
    0x63, 0x2f, 0x7a, 0x11, 0xfb, 0xa9, 0x79, 0xd2, 0x97, 0x9e, 0x41, 0x7e,
    0x2d, 0xed, 0x1d, 0x30,
};
static const unsigned char kat145_nor_retbytes[] = {
    0xa6, 0xa1, 0x55, 0x95, 0xbb, 0xd8, 0x57, 0xbb, 0x62, 0x15, 0x04, 0xc8,
    0x5f, 0x03, 0x03, 0x3c, 0xe4, 0x7b, 0x5d, 0xf8, 0x62, 0x96, 0xb8, 0x0d,
    0x40, 0xd0, 0x09, 0xfe, 0x6c, 0x5e, 0x1f, 0xff, 0xf0, 0x84, 0x04, 0x56,
    0x99, 0x0a, 0x14, 0xf3, 0xa1, 0xc4, 0x9c, 0x36, 0x73, 0x7b, 0x70, 0xb6,
    0x2f, 0x40, 0x6f, 0xa3, 0xc5, 0x32, 0x95, 0x2f, 0xb2, 0x2e, 0xfc, 0x76,
    0x00, 0x9a, 0x1b, 0x1a,
};
static const struct drbg_kat_no_reseed kat145_nor_t = {
    9, kat145_nor_entropyin, kat145_nor_nonce, kat145_nor_persstr,
    kat145_nor_addin0, kat145_nor_addin1, kat145_nor_retbytes
};
static const struct drbg_kat kat145_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat145_nor_t
};

static const unsigned char kat146_nor_entropyin[] = {
    0xd1, 0x9f, 0xa3, 0x38, 0xc7, 0xbd, 0x52, 0x40, 0x74, 0x7b, 0xac, 0xab,
    0x70, 0x32, 0x8f, 0xb3,
};
static const unsigned char kat146_nor_nonce[] = {
    0x5b, 0x42, 0x82, 0x35, 0x41, 0x42, 0x83, 0xb4,
};
static const unsigned char kat146_nor_persstr[] = {0};
static const unsigned char kat146_nor_addin0[] = {
    0xf3, 0x7c, 0x9f, 0xeb, 0x90, 0xc9, 0x6a, 0x26, 0x3f, 0x41, 0xb1, 0xc4,
    0x98, 0xd5, 0xe0, 0x75,
};
static const unsigned char kat146_nor_addin1[] = {
    0xa3, 0xd8, 0x86, 0x96, 0xfc, 0x1a, 0x44, 0x70, 0x28, 0x42, 0x31, 0x71,
    0xba, 0xd4, 0x65, 0x24,
};
static const unsigned char kat146_nor_retbytes[] = {
    0x11, 0xee, 0x72, 0xb4, 0x81, 0xd5, 0x54, 0xf0, 0xfe, 0x49, 0xdc, 0x27,
    0x37, 0x46, 0x6d, 0x5f, 0x5a, 0x64, 0x76, 0xa2, 0xb5, 0xb2, 0xf9, 0x3c,
    0xd6, 0x0e, 0xe8, 0xab, 0x1b, 0xf7, 0x56, 0x3d, 0x3e, 0xbc, 0x60, 0x5e,
    0x44, 0xc3, 0x65, 0xe7, 0x86, 0x5b, 0xff, 0x31, 0xc0, 0x77, 0xd1, 0x76,
    0xd3, 0x61, 0xa4, 0x24, 0x06, 0x27, 0xde, 0xb2, 0x8a, 0xd5, 0x68, 0x50,
    0x46, 0x91, 0xf9, 0x47,
};
static const struct drbg_kat_no_reseed kat146_nor_t = {
    10, kat146_nor_entropyin, kat146_nor_nonce, kat146_nor_persstr,
    kat146_nor_addin0, kat146_nor_addin1, kat146_nor_retbytes
};
static const struct drbg_kat kat146_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat146_nor_t
};

static const unsigned char kat147_nor_entropyin[] = {
    0xc3, 0xd8, 0x2c, 0x63, 0x01, 0x41, 0x77, 0xe6, 0x2b, 0xda, 0x82, 0xdd,
    0xe9, 0x11, 0x54, 0x50,
};
static const unsigned char kat147_nor_nonce[] = {
    0x5e, 0x6f, 0x3b, 0x1c, 0x75, 0x70, 0x6f, 0x5f,
};
static const unsigned char kat147_nor_persstr[] = {0};
static const unsigned char kat147_nor_addin0[] = {
    0x72, 0xa7, 0x17, 0xc6, 0x6f, 0xb2, 0x69, 0x06, 0xd6, 0x99, 0x5d, 0x3e,
    0xec, 0xc5, 0x79, 0xd9,
};
static const unsigned char kat147_nor_addin1[] = {
    0xb1, 0xb7, 0xa9, 0xad, 0x3c, 0x64, 0x35, 0x5d, 0x5d, 0x70, 0xf0, 0x3c,
    0x7b, 0x83, 0x29, 0xb0,
};
static const unsigned char kat147_nor_retbytes[] = {
    0x34, 0xc3, 0x09, 0xf2, 0x35, 0x27, 0x09, 0xa9, 0x1f, 0x1f, 0x1a, 0x6a,
    0xfb, 0x60, 0x14, 0x66, 0xf8, 0x6d, 0xc8, 0x02, 0x2b, 0xcb, 0xfd, 0xee,
    0x09, 0x51, 0x90, 0xa8, 0x5f, 0xe1, 0xf0, 0x33, 0x2b, 0x8b, 0xab, 0x32,
    0xf4, 0x42, 0x49, 0x43, 0x00, 0x41, 0xcb, 0x0c, 0xf2, 0xc4, 0x0d, 0x9c,
    0xdd, 0x0c, 0x0d, 0xf4, 0x23, 0xe3, 0x4f, 0x2d, 0xa8, 0xd7, 0xf7, 0xb8,
    0x32, 0x94, 0x56, 0x19,
};
static const struct drbg_kat_no_reseed kat147_nor_t = {
    11, kat147_nor_entropyin, kat147_nor_nonce, kat147_nor_persstr,
    kat147_nor_addin0, kat147_nor_addin1, kat147_nor_retbytes
};
static const struct drbg_kat kat147_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat147_nor_t
};

static const unsigned char kat148_nor_entropyin[] = {
    0x1c, 0x34, 0x80, 0x46, 0x79, 0x40, 0x8c, 0x0d, 0xd1, 0xc0, 0xb0, 0xb2,
    0xcb, 0x4c, 0x0c, 0x8d,
};
static const unsigned char kat148_nor_nonce[] = {
    0x23, 0xe6, 0xd5, 0xa1, 0x31, 0x74, 0x0e, 0xc9,
};
static const unsigned char kat148_nor_persstr[] = {0};
static const unsigned char kat148_nor_addin0[] = {
    0x8b, 0x99, 0xe4, 0x48, 0x55, 0x10, 0xe7, 0xc4, 0xef, 0x31, 0x86, 0xc7,
    0x5d, 0x0f, 0x42, 0x1d,
};
static const unsigned char kat148_nor_addin1[] = {
    0xcc, 0x1e, 0xcf, 0x02, 0x3c, 0x9b, 0xea, 0xfb, 0x63, 0x48, 0x69, 0x57,
    0x32, 0x7c, 0x2b, 0xde,
};
static const unsigned char kat148_nor_retbytes[] = {
    0x7f, 0xbf, 0x33, 0xf0, 0x51, 0x08, 0x78, 0x6f, 0xe0, 0x19, 0x97, 0x14,
    0x6a, 0x27, 0xe5, 0x94, 0x68, 0x76, 0x49, 0x9a, 0x8b, 0xa5, 0x2a, 0x71,
    0x47, 0x16, 0xd9, 0x82, 0xce, 0xa2, 0x3f, 0x39, 0x2d, 0x40, 0x23, 0x4d,
    0x3e, 0x00, 0x33, 0x8d, 0x1a, 0xc8, 0x80, 0x9d, 0x43, 0xc7, 0x79, 0x42,
    0x40, 0x30, 0x19, 0x3b, 0xc6, 0x12, 0x3b, 0x70, 0x67, 0xb6, 0xa8, 0xc2,
    0xed, 0x17, 0x9a, 0x25,
};
static const struct drbg_kat_no_reseed kat148_nor_t = {
    12, kat148_nor_entropyin, kat148_nor_nonce, kat148_nor_persstr,
    kat148_nor_addin0, kat148_nor_addin1, kat148_nor_retbytes
};
static const struct drbg_kat kat148_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat148_nor_t
};

static const unsigned char kat149_nor_entropyin[] = {
    0xaa, 0x70, 0xaa, 0x9d, 0x3c, 0x0a, 0x77, 0xf8, 0x68, 0x38, 0x80, 0x5e,
    0xef, 0x34, 0x82, 0xba,
};
static const unsigned char kat149_nor_nonce[] = {
    0xcf, 0x7a, 0x0f, 0x57, 0xc7, 0xde, 0xdf, 0x50,
};
static const unsigned char kat149_nor_persstr[] = {0};
static const unsigned char kat149_nor_addin0[] = {
    0xf6, 0xd4, 0x76, 0xea, 0xe4, 0x2f, 0x02, 0xc9, 0xec, 0x21, 0x98, 0x12,
    0x29, 0xd0, 0xf9, 0x76,
};
static const unsigned char kat149_nor_addin1[] = {
    0x1a, 0xa4, 0x1e, 0xfd, 0xf1, 0x06, 0xe5, 0xa3, 0xe5, 0xd9, 0x76, 0x77,
    0x8b, 0x8f, 0x0c, 0x32,
};
static const unsigned char kat149_nor_retbytes[] = {
    0x84, 0xc1, 0xc0, 0x61, 0x9f, 0x69, 0x79, 0xe2, 0xf4, 0xd2, 0xab, 0xc6,
    0x33, 0x35, 0x42, 0xbf, 0x42, 0xfd, 0x32, 0x79, 0x14, 0x2f, 0x07, 0x1d,
    0xad, 0xb2, 0x64, 0x45, 0xfb, 0x2d, 0xe5, 0x1c, 0x4f, 0xad, 0x68, 0x1d,
    0xce, 0x89, 0x18, 0xaf, 0x3a, 0xe1, 0x64, 0x19, 0xfa, 0xfa, 0x3c, 0x55,
    0x26, 0xd8, 0xc4, 0x78, 0x59, 0x9e, 0x85, 0xee, 0x61, 0xa2, 0x73, 0x18,
    0x36, 0x05, 0x45, 0x0e,
};
static const struct drbg_kat_no_reseed kat149_nor_t = {
    13, kat149_nor_entropyin, kat149_nor_nonce, kat149_nor_persstr,
    kat149_nor_addin0, kat149_nor_addin1, kat149_nor_retbytes
};
static const struct drbg_kat kat149_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat149_nor_t
};

static const unsigned char kat150_nor_entropyin[] = {
    0x94, 0xc6, 0x97, 0x57, 0xc9, 0x75, 0xe5, 0x3c, 0xa9, 0x44, 0xcb, 0xe8,
    0xf7, 0x68, 0x6f, 0x85,
};
static const unsigned char kat150_nor_nonce[] = {
    0x6e, 0x14, 0x5c, 0xc7, 0xd7, 0x2d, 0xec, 0x57,
};
static const unsigned char kat150_nor_persstr[] = {0};
static const unsigned char kat150_nor_addin0[] = {
    0xce, 0x08, 0x94, 0x6b, 0x04, 0xfd, 0x83, 0x13, 0xdc, 0xda, 0x8c, 0xde,
    0x3b, 0x0f, 0xf9, 0xdf,
};
static const unsigned char kat150_nor_addin1[] = {
    0x9f, 0xfc, 0x93, 0x9d, 0x30, 0x5a, 0xe5, 0x86, 0xd8, 0x6a, 0x1a, 0x14,
    0x70, 0x32, 0x68, 0x0b,
};
static const unsigned char kat150_nor_retbytes[] = {
    0xb1, 0xd7, 0xb3, 0x87, 0xa9, 0xf6, 0x7b, 0x91, 0x50, 0x55, 0xe6, 0x82,
    0x98, 0xe0, 0x37, 0x73, 0xa0, 0x19, 0x75, 0x56, 0xf8, 0xd4, 0xb7, 0xe0,
    0x29, 0x52, 0x03, 0x35, 0xef, 0xdc, 0xe2, 0xac, 0xb0, 0xd4, 0xd4, 0xcd,
    0xae, 0x8a, 0xf8, 0xc0, 0x9f, 0xb5, 0x7a, 0x21, 0x5f, 0x20, 0x87, 0xab,
    0xb8, 0x4e, 0xc8, 0x5b, 0x12, 0xdc, 0x3a, 0x85, 0x3d, 0x73, 0xa4, 0xdd,
    0x11, 0x43, 0x00, 0xc5,
};
static const struct drbg_kat_no_reseed kat150_nor_t = {
    14, kat150_nor_entropyin, kat150_nor_nonce, kat150_nor_persstr,
    kat150_nor_addin0, kat150_nor_addin1, kat150_nor_retbytes
};
static const struct drbg_kat kat150_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat150_nor_t
};

static const unsigned char kat151_nor_entropyin[] = {
    0xab, 0xa3, 0x6f, 0xf7, 0xa5, 0x35, 0x37, 0x45, 0x4b, 0x5c, 0xb2, 0x68,
    0x39, 0x17, 0x15, 0x40,
};
static const unsigned char kat151_nor_nonce[] = {
    0x68, 0x71, 0xc0, 0xf5, 0x26, 0xfb, 0xcd, 0xc7,
};
static const unsigned char kat151_nor_persstr[] = {
    0xdb, 0xc4, 0x4a, 0xf4, 0x98, 0x16, 0x1f, 0x1f, 0x2a, 0xf6, 0xfc, 0xe6,
    0x6c, 0xcc, 0x30, 0xa8,
};
static const unsigned char kat151_nor_addin0[] = {0};
static const unsigned char kat151_nor_addin1[] = {0};
static const unsigned char kat151_nor_retbytes[] = {
    0xaf, 0x68, 0x6e, 0x9a, 0xaf, 0x10, 0xaa, 0xbc, 0xbb, 0x44, 0xb3, 0x74,
    0x89, 0x53, 0xad, 0x18, 0x5d, 0xbf, 0x12, 0x89, 0x8e, 0x52, 0x4d, 0x04,
    0x08, 0x61, 0x02, 0xe4, 0x5f, 0x38, 0x41, 0xc6, 0x50, 0xf6, 0x23, 0xf4,
    0x8f, 0x54, 0x2c, 0xaa, 0x14, 0x79, 0x3e, 0x4f, 0xcb, 0xbc, 0xf2, 0xe4,
    0x61, 0xbe, 0x1c, 0x01, 0xed, 0x8f, 0x1f, 0x48, 0xb9, 0x70, 0x4d, 0x79,
    0xa8, 0xeb, 0xf7, 0x9d,
};
static const struct drbg_kat_no_reseed kat151_nor_t = {
    0, kat151_nor_entropyin, kat151_nor_nonce, kat151_nor_persstr,
    kat151_nor_addin0, kat151_nor_addin1, kat151_nor_retbytes
};
static const struct drbg_kat kat151_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat151_nor_t
};

static const unsigned char kat152_nor_entropyin[] = {
    0x47, 0xaf, 0xb8, 0x3e, 0x3a, 0xda, 0x22, 0x0f, 0x7d, 0x7e, 0x63, 0x82,
    0xa8, 0xb3, 0x8c, 0xbb,
};
static const unsigned char kat152_nor_nonce[] = {
    0x7a, 0xb8, 0xae, 0x9b, 0xac, 0x8b, 0x15, 0xa5,
};
static const unsigned char kat152_nor_persstr[] = {
    0x8b, 0xec, 0x1c, 0xb2, 0xd1, 0x80, 0xb3, 0x67, 0x7c, 0xd1, 0xa8, 0x60,
    0x4b, 0x61, 0x4d, 0xbe,
};
static const unsigned char kat152_nor_addin0[] = {0};
static const unsigned char kat152_nor_addin1[] = {0};
static const unsigned char kat152_nor_retbytes[] = {
    0x90, 0xcf, 0x1d, 0x9e, 0x65, 0xd9, 0x76, 0xca, 0xce, 0x2f, 0x20, 0xe7,
    0x81, 0x47, 0xd5, 0x04, 0x0d, 0x02, 0x23, 0x7e, 0x04, 0x17, 0x3f, 0x1f,
    0x37, 0x10, 0xe5, 0x22, 0x7d, 0xcb, 0x85, 0x64, 0x68, 0x4f, 0x2e, 0xba,
    0x38, 0xe1, 0xde, 0xf7, 0x2b, 0x93, 0xbe, 0xdb, 0x44, 0x85, 0xf2, 0xb8,
    0x17, 0xee, 0x66, 0xc1, 0x89, 0x02, 0x4b, 0x2a, 0x12, 0x73, 0x65, 0xbc,
    0x83, 0x50, 0x08, 0x71,
};
static const struct drbg_kat_no_reseed kat152_nor_t = {
    1, kat152_nor_entropyin, kat152_nor_nonce, kat152_nor_persstr,
    kat152_nor_addin0, kat152_nor_addin1, kat152_nor_retbytes
};
static const struct drbg_kat kat152_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat152_nor_t
};

static const unsigned char kat153_nor_entropyin[] = {
    0xbb, 0x69, 0x93, 0xaf, 0xf4, 0x80, 0x4a, 0x87, 0xdd, 0x42, 0x59, 0x73,
    0xfd, 0xe3, 0x53, 0x11,
};
static const unsigned char kat153_nor_nonce[] = {
    0xfb, 0x14, 0x70, 0x0b, 0x33, 0x5a, 0x01, 0x46,
};
static const unsigned char kat153_nor_persstr[] = {
    0x1e, 0xb0, 0x65, 0x44, 0xca, 0xa8, 0x6e, 0x2a, 0xc4, 0x58, 0x8a, 0xa8,
    0x51, 0xe8, 0xfc, 0x0e,
};
static const unsigned char kat153_nor_addin0[] = {0};
static const unsigned char kat153_nor_addin1[] = {0};
static const unsigned char kat153_nor_retbytes[] = {
    0x4e, 0xe7, 0x27, 0x07, 0x3a, 0xba, 0xa3, 0xd7, 0x41, 0x8d, 0x6b, 0x3e,
    0x3b, 0xd4, 0x67, 0xc9, 0x28, 0x48, 0x54, 0x42, 0x3b, 0xa6, 0xd4, 0xef,
    0x02, 0xb2, 0xda, 0xbf, 0x14, 0xb9, 0xb8, 0x24, 0xb2, 0x7a, 0xda, 0x2b,
    0x4a, 0x42, 0xf7, 0xdd, 0x1c, 0xd3, 0x9d, 0xc4, 0x42, 0x0e, 0xe6, 0xe8,
    0x43, 0xfa, 0x7f, 0x2e, 0xee, 0x06, 0xbb, 0x05, 0xc6, 0x47, 0xdc, 0xc0,
    0xd6, 0x97, 0xc0, 0x09,
};
static const struct drbg_kat_no_reseed kat153_nor_t = {
    2, kat153_nor_entropyin, kat153_nor_nonce, kat153_nor_persstr,
    kat153_nor_addin0, kat153_nor_addin1, kat153_nor_retbytes
};
static const struct drbg_kat kat153_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat153_nor_t
};

static const unsigned char kat154_nor_entropyin[] = {
    0xa0, 0x5d, 0x9d, 0x84, 0xa5, 0x0d, 0xe7, 0x97, 0x01, 0xc7, 0x59, 0x59,
    0xaf, 0x26, 0xa8, 0xfa,
};
static const unsigned char kat154_nor_nonce[] = {
    0x7e, 0xde, 0x7f, 0x16, 0x30, 0xdc, 0x01, 0xf7,
};
static const unsigned char kat154_nor_persstr[] = {
    0x54, 0xf5, 0xd6, 0xe5, 0xd9, 0xb1, 0xfe, 0xb0, 0xa1, 0xc3, 0xd9, 0xa7,
    0xec, 0x81, 0xdd, 0x98,
};
static const unsigned char kat154_nor_addin0[] = {0};
static const unsigned char kat154_nor_addin1[] = {0};
static const unsigned char kat154_nor_retbytes[] = {
    0xf4, 0x7a, 0xab, 0x57, 0xdf, 0xff, 0x93, 0x1d, 0x4b, 0xfc, 0x75, 0xa0,
    0x13, 0x11, 0x86, 0xf0, 0xf5, 0xc1, 0x50, 0x5e, 0x9c, 0x6c, 0x7e, 0xb9,
    0x35, 0xe3, 0x1b, 0x49, 0xf1, 0x34, 0xef, 0xc0, 0x0e, 0x45, 0xfc, 0x96,
    0x73, 0x58, 0xee, 0xc4, 0xa9, 0x21, 0xfd, 0xa0, 0xd0, 0x53, 0x7d, 0x9e,
    0x4f, 0xb3, 0x3b, 0x26, 0x3d, 0xb8, 0xe0, 0x8e, 0x73, 0xf2, 0x1f, 0xe1,
    0x75, 0x05, 0x19, 0x6f,
};
static const struct drbg_kat_no_reseed kat154_nor_t = {
    3, kat154_nor_entropyin, kat154_nor_nonce, kat154_nor_persstr,
    kat154_nor_addin0, kat154_nor_addin1, kat154_nor_retbytes
};
static const struct drbg_kat kat154_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat154_nor_t
};

static const unsigned char kat155_nor_entropyin[] = {
    0x80, 0x7d, 0x1b, 0x10, 0x96, 0xe3, 0xd7, 0xf7, 0x89, 0x03, 0x6d, 0x64,
    0x4d, 0xfb, 0x9e, 0x8b,
};
static const unsigned char kat155_nor_nonce[] = {
    0x8c, 0xd7, 0x7f, 0x71, 0xcc, 0xd8, 0xa3, 0x22,
};
static const unsigned char kat155_nor_persstr[] = {
    0x4f, 0x72, 0x74, 0x5e, 0x2f, 0x2a, 0xa5, 0x43, 0x61, 0x89, 0xdb, 0x92,
    0x27, 0x82, 0x0e, 0x46,
};
static const unsigned char kat155_nor_addin0[] = {0};
static const unsigned char kat155_nor_addin1[] = {0};
static const unsigned char kat155_nor_retbytes[] = {
    0xff, 0x4a, 0xe0, 0x06, 0x4c, 0x25, 0xfc, 0xef, 0x07, 0x4a, 0xb3, 0x65,
    0x0e, 0xb6, 0xd3, 0x04, 0x4f, 0x86, 0x68, 0x7e, 0x6d, 0xb2, 0x26, 0x29,
    0x99, 0x2b, 0x08, 0xed, 0x07, 0x8c, 0x65, 0xd0, 0x3d, 0xaf, 0x62, 0x41,
    0xa3, 0x10, 0xe5, 0x76, 0x3c, 0x29, 0x86, 0x63, 0x84, 0x8e, 0x32, 0xd0,
    0x61, 0x4e, 0x98, 0x98, 0x9a, 0x16, 0xd3, 0x7d, 0xc1, 0x72, 0x91, 0x35,
    0xfc, 0xa5, 0xe6, 0x2e,
};
static const struct drbg_kat_no_reseed kat155_nor_t = {
    4, kat155_nor_entropyin, kat155_nor_nonce, kat155_nor_persstr,
    kat155_nor_addin0, kat155_nor_addin1, kat155_nor_retbytes
};
static const struct drbg_kat kat155_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat155_nor_t
};

static const unsigned char kat156_nor_entropyin[] = {
    0x1a, 0xbc, 0xe2, 0x1f, 0x12, 0x33, 0x68, 0xb2, 0x6c, 0x66, 0x56, 0xb0,
    0xce, 0xbc, 0x03, 0xe8,
};
static const unsigned char kat156_nor_nonce[] = {
    0x1f, 0x7a, 0x6d, 0x5b, 0xc2, 0xcb, 0x97, 0x59,
};
static const unsigned char kat156_nor_persstr[] = {
    0x00, 0xcd, 0x25, 0x9b, 0xb8, 0x7e, 0x4e, 0x6e, 0x21, 0x3e, 0xd4, 0x4e,
    0xec, 0x19, 0xd9, 0x9d,
};
static const unsigned char kat156_nor_addin0[] = {0};
static const unsigned char kat156_nor_addin1[] = {0};
static const unsigned char kat156_nor_retbytes[] = {
    0x86, 0xc0, 0x0d, 0x31, 0x81, 0xc1, 0xb6, 0x06, 0xc5, 0x1f, 0x90, 0x98,
    0x93, 0x83, 0xb4, 0xb8, 0x92, 0x2e, 0x4a, 0x19, 0x0b, 0x94, 0x01, 0x65,
    0x8f, 0x8d, 0x45, 0x13, 0x69, 0x7c, 0xa7, 0xd5, 0x84, 0xf5, 0xfc, 0xce,
    0xb3, 0x32, 0x4f, 0x62, 0x4c, 0x47, 0x81, 0xdf, 0xe5, 0x5c, 0x0d, 0xba,
    0x2d, 0x66, 0xbd, 0x85, 0x8f, 0xb6, 0x43, 0xf2, 0x3c, 0xe5, 0xd6, 0x7e,
    0x57, 0x25, 0x00, 0x07,
};
static const struct drbg_kat_no_reseed kat156_nor_t = {
    5, kat156_nor_entropyin, kat156_nor_nonce, kat156_nor_persstr,
    kat156_nor_addin0, kat156_nor_addin1, kat156_nor_retbytes
};
static const struct drbg_kat kat156_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat156_nor_t
};

static const unsigned char kat157_nor_entropyin[] = {
    0x06, 0xd5, 0xfb, 0x81, 0x4c, 0xa4, 0xb2, 0xba, 0xc9, 0xe1, 0xcf, 0xba,
    0x0f, 0x13, 0x69, 0x9d,
};
static const unsigned char kat157_nor_nonce[] = {
    0x0e, 0xa8, 0x7b, 0x9f, 0x3b, 0xb1, 0xe6, 0x29,
};
static const unsigned char kat157_nor_persstr[] = {
    0xf6, 0x36, 0x79, 0xdf, 0xa3, 0x53, 0x70, 0x3f, 0x12, 0xe7, 0x23, 0x61,
    0x73, 0xc7, 0xd3, 0x20,
};
static const unsigned char kat157_nor_addin0[] = {0};
static const unsigned char kat157_nor_addin1[] = {0};
static const unsigned char kat157_nor_retbytes[] = {
    0xe0, 0xc8, 0xff, 0x1f, 0x1d, 0x2c, 0x69, 0xaf, 0xca, 0xe0, 0xb7, 0x3b,
    0xe8, 0xb3, 0xc4, 0xc7, 0x41, 0x3f, 0x7f, 0xbc, 0xbe, 0xfc, 0x3b, 0xcf,
    0x1e, 0x68, 0x8d, 0x2a, 0x7d, 0x08, 0x49, 0xfd, 0xee, 0x60, 0xbd, 0xe9,
    0x1a, 0x0f, 0xb1, 0xa5, 0xef, 0x4b, 0xf3, 0xdf, 0xb3, 0x36, 0xb7, 0x3e,
    0xd0, 0x47, 0xdf, 0xa7, 0x74, 0x77, 0xa5, 0x1c, 0x6d, 0xee, 0x81, 0x72,
    0x63, 0xa2, 0x0c, 0x37,
};
static const struct drbg_kat_no_reseed kat157_nor_t = {
    6, kat157_nor_entropyin, kat157_nor_nonce, kat157_nor_persstr,
    kat157_nor_addin0, kat157_nor_addin1, kat157_nor_retbytes
};
static const struct drbg_kat kat157_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat157_nor_t
};

static const unsigned char kat158_nor_entropyin[] = {
    0xdd, 0x3b, 0x7c, 0xfc, 0x51, 0x8c, 0x18, 0x0c, 0xf2, 0x89, 0xf1, 0x45,
    0x25, 0x15, 0x0c, 0xa5,
};
static const unsigned char kat158_nor_nonce[] = {
    0x36, 0x94, 0x04, 0x3c, 0x71, 0xc2, 0xdd, 0xc5,
};
static const unsigned char kat158_nor_persstr[] = {
    0x3b, 0x1c, 0x08, 0xfa, 0xb6, 0x36, 0x18, 0x51, 0xd7, 0xa5, 0x18, 0xae,
    0x35, 0x5b, 0x8c, 0x9b,
};
static const unsigned char kat158_nor_addin0[] = {0};
static const unsigned char kat158_nor_addin1[] = {0};
static const unsigned char kat158_nor_retbytes[] = {
    0x59, 0x16, 0x4b, 0xac, 0x7a, 0x71, 0x4f, 0xcd, 0x5a, 0x45, 0x80, 0xfb,
    0x54, 0xed, 0x4d, 0xdf, 0x99, 0xc3, 0x9c, 0xf5, 0x9c, 0x23, 0xf8, 0x5f,
    0x6c, 0x52, 0x16, 0xf4, 0xe8, 0x9c, 0xf2, 0x8d, 0xa1, 0x59, 0x9f, 0x82,
    0x57, 0xa6, 0xaf, 0xc3, 0x02, 0xed, 0x3a, 0x1d, 0xec, 0x00, 0x3f, 0xf4,
    0x50, 0x91, 0x2c, 0x2b, 0xcd, 0x68, 0x2c, 0xd3, 0x40, 0x79, 0xcf, 0xb3,
    0xcc, 0xf2, 0x59, 0x3a,
};
static const struct drbg_kat_no_reseed kat158_nor_t = {
    7, kat158_nor_entropyin, kat158_nor_nonce, kat158_nor_persstr,
    kat158_nor_addin0, kat158_nor_addin1, kat158_nor_retbytes
};
static const struct drbg_kat kat158_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat158_nor_t
};

static const unsigned char kat159_nor_entropyin[] = {
    0xb7, 0x31, 0x4b, 0xac, 0xd0, 0xb3, 0xe9, 0xe2, 0x21, 0x4e, 0x11, 0xa4,
    0x9c, 0x4f, 0xab, 0x54,
};
static const unsigned char kat159_nor_nonce[] = {
    0x8c, 0xd5, 0x9a, 0x9c, 0x39, 0x50, 0x1c, 0x0b,
};
static const unsigned char kat159_nor_persstr[] = {
    0xb6, 0xe3, 0xa4, 0x64, 0x4f, 0xb9, 0x13, 0xa5, 0x4c, 0x89, 0x21, 0xcb,
    0xc1, 0x73, 0x72, 0x38,
};
static const unsigned char kat159_nor_addin0[] = {0};
static const unsigned char kat159_nor_addin1[] = {0};
static const unsigned char kat159_nor_retbytes[] = {
    0xe9, 0x2c, 0x97, 0xcc, 0xbb, 0xd6, 0x01, 0x31, 0x78, 0xee, 0x06, 0xd0,
    0x1a, 0xd2, 0xc9, 0xeb, 0x54, 0x64, 0xa7, 0xe3, 0x04, 0x32, 0xb9, 0x43,
    0xe0, 0xb3, 0x71, 0xf1, 0x36, 0xa9, 0x94, 0xb9, 0xf5, 0x44, 0xf3, 0x7b,
    0x60, 0x56, 0x1e, 0x10, 0x25, 0xb1, 0x2b, 0x5a, 0x15, 0xcb, 0x66, 0x1b,
    0x30, 0x1b, 0x5d, 0xd4, 0x38, 0x4b, 0x8b, 0xc0, 0x0d, 0x1d, 0x72, 0xb3,
    0xc6, 0x18, 0xf8, 0x75,
};
static const struct drbg_kat_no_reseed kat159_nor_t = {
    8, kat159_nor_entropyin, kat159_nor_nonce, kat159_nor_persstr,
    kat159_nor_addin0, kat159_nor_addin1, kat159_nor_retbytes
};
static const struct drbg_kat kat159_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat159_nor_t
};

static const unsigned char kat160_nor_entropyin[] = {
    0xd6, 0x2b, 0x1f, 0xc2, 0xb0, 0x45, 0x71, 0x8a, 0x3f, 0xe5, 0x9d, 0xfa,
    0xc4, 0xa1, 0xc0, 0x3b,
};
static const unsigned char kat160_nor_nonce[] = {
    0x41, 0x99, 0x17, 0xd5, 0x8a, 0x67, 0x56, 0xa5,
};
static const unsigned char kat160_nor_persstr[] = {
    0xf4, 0x79, 0x4b, 0xdc, 0x10, 0x9f, 0x13, 0x00, 0x4b, 0x89, 0x3b, 0x32,
    0x83, 0xc0, 0x97, 0x7e,
};
static const unsigned char kat160_nor_addin0[] = {0};
static const unsigned char kat160_nor_addin1[] = {0};
static const unsigned char kat160_nor_retbytes[] = {
    0x05, 0x42, 0xdf, 0xda, 0xb8, 0x8e, 0x34, 0x91, 0x63, 0xed, 0x9b, 0xa6,
    0x34, 0xee, 0x76, 0x25, 0x7d, 0xc9, 0x27, 0x66, 0x61, 0xcd, 0x5d, 0xd2,
    0xfa, 0xa9, 0x31, 0xbc, 0x3a, 0x2e, 0x9c, 0x2d, 0x17, 0xe5, 0x70, 0xff,
    0xa1, 0xa5, 0xf1, 0x44, 0x96, 0xf0, 0xea, 0xc3, 0x33, 0x9e, 0xfd, 0xe4,
    0x6a, 0xa4, 0x0e, 0x87, 0xf2, 0x1a, 0x98, 0x54, 0x95, 0xfd, 0xa3, 0x94,
    0xf2, 0x06, 0x6e, 0xbb,
};
static const struct drbg_kat_no_reseed kat160_nor_t = {
    9, kat160_nor_entropyin, kat160_nor_nonce, kat160_nor_persstr,
    kat160_nor_addin0, kat160_nor_addin1, kat160_nor_retbytes
};
static const struct drbg_kat kat160_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat160_nor_t
};

static const unsigned char kat161_nor_entropyin[] = {
    0x91, 0xb4, 0xaa, 0x7d, 0x56, 0x58, 0x32, 0xe9, 0x6a, 0x21, 0xd6, 0x80,
    0xa0, 0xec, 0xdb, 0x4c,
};
static const unsigned char kat161_nor_nonce[] = {
    0x06, 0x09, 0x09, 0xe2, 0xce, 0x8b, 0x2d, 0xc8,
};
static const unsigned char kat161_nor_persstr[] = {
    0x5c, 0xba, 0xce, 0xb8, 0x9c, 0x15, 0x0d, 0x22, 0x9b, 0x51, 0x6c, 0x34,
    0x93, 0x60, 0xf2, 0x7f,
};
static const unsigned char kat161_nor_addin0[] = {0};
static const unsigned char kat161_nor_addin1[] = {0};
static const unsigned char kat161_nor_retbytes[] = {
    0xc0, 0x79, 0xc2, 0x91, 0x7e, 0x8f, 0x6b, 0x84, 0xc5, 0x8e, 0x02, 0x26,
    0xad, 0x0b, 0x8a, 0x60, 0xa8, 0x7b, 0x88, 0x22, 0xa9, 0x90, 0x45, 0x9d,
    0xe2, 0x4c, 0xc6, 0x55, 0x4c, 0x7f, 0x24, 0x1a, 0xff, 0x30, 0xcd, 0xea,
    0x61, 0xa7, 0x48, 0x47, 0x0a, 0x58, 0xc9, 0x4a, 0x15, 0x0e, 0xbd, 0xdc,
    0x35, 0x5c, 0x64, 0x4d, 0xd4, 0x78, 0x6e, 0x36, 0xf1, 0x7e, 0xcf, 0xce,
    0xa7, 0x33, 0x9d, 0x42,
};
static const struct drbg_kat_no_reseed kat161_nor_t = {
    10, kat161_nor_entropyin, kat161_nor_nonce, kat161_nor_persstr,
    kat161_nor_addin0, kat161_nor_addin1, kat161_nor_retbytes
};
static const struct drbg_kat kat161_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat161_nor_t
};

static const unsigned char kat162_nor_entropyin[] = {
    0x32, 0xe1, 0x66, 0xc0, 0x93, 0x8d, 0x6d, 0xa7, 0x21, 0xcf, 0xce, 0x23,
    0x5a, 0x66, 0xa1, 0x80,
};
static const unsigned char kat162_nor_nonce[] = {
    0xeb, 0x68, 0x79, 0x96, 0xa8, 0xff, 0x63, 0x94,
};
static const unsigned char kat162_nor_persstr[] = {
    0x05, 0xb9, 0x9f, 0x8d, 0x9f, 0x10, 0x2c, 0x5f, 0xf6, 0xd0, 0xd4, 0x5c,
    0x4d, 0x68, 0x5f, 0xf8,
};
static const unsigned char kat162_nor_addin0[] = {0};
static const unsigned char kat162_nor_addin1[] = {0};
static const unsigned char kat162_nor_retbytes[] = {
    0xec, 0x5a, 0xe3, 0x97, 0x25, 0x72, 0x20, 0x45, 0x2e, 0x61, 0x05, 0xa5,
    0xa2, 0x9f, 0x56, 0xd9, 0x33, 0x21, 0x32, 0xf5, 0x33, 0x8d, 0x19, 0xe8,
    0x1a, 0xf6, 0xd3, 0x10, 0x22, 0xbb, 0x91, 0xf8, 0x7f, 0x04, 0xe4, 0x52,
    0x0d, 0x50, 0x76, 0x61, 0x43, 0xd3, 0x53, 0x3e, 0x7b, 0x9d, 0x57, 0x0a,
    0x80, 0x68, 0x84, 0xa3, 0xdf, 0xe2, 0x81, 0xe0, 0x00, 0xf7, 0x7e, 0x7b,
    0x85, 0x59, 0x44, 0x34,
};
static const struct drbg_kat_no_reseed kat162_nor_t = {
    11, kat162_nor_entropyin, kat162_nor_nonce, kat162_nor_persstr,
    kat162_nor_addin0, kat162_nor_addin1, kat162_nor_retbytes
};
static const struct drbg_kat kat162_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat162_nor_t
};

static const unsigned char kat163_nor_entropyin[] = {
    0x3a, 0x3d, 0x6a, 0xa9, 0xdb, 0x0f, 0x54, 0x0a, 0x37, 0xeb, 0x81, 0x50,
    0x9d, 0xdb, 0x65, 0x5d,
};
static const unsigned char kat163_nor_nonce[] = {
    0x1f, 0xe8, 0xc2, 0x5b, 0x27, 0xee, 0x3a, 0xbe,
};
static const unsigned char kat163_nor_persstr[] = {
    0x8e, 0x15, 0xa5, 0xad, 0x7a, 0x5f, 0x29, 0x38, 0xeb, 0xdb, 0x5c, 0x08,
    0x8b, 0xdc, 0xf3, 0x07,
};
static const unsigned char kat163_nor_addin0[] = {0};
static const unsigned char kat163_nor_addin1[] = {0};
static const unsigned char kat163_nor_retbytes[] = {
    0x8d, 0x7e, 0xae, 0xe5, 0x0e, 0x87, 0xf0, 0xbc, 0x34, 0xa4, 0x9b, 0xe9,
    0x4e, 0x7a, 0x4a, 0x64, 0xfd, 0x1e, 0xc9, 0x86, 0x1e, 0xf3, 0x34, 0x22,
    0x2f, 0xfb, 0x15, 0xaf, 0x89, 0x29, 0xf9, 0xed, 0x42, 0xfb, 0x3f, 0xa6,
    0x29, 0xd6, 0xba, 0x39, 0x70, 0x6b, 0x91, 0x93, 0xe1, 0x2b, 0xa2, 0x40,
    0x44, 0x76, 0x17, 0x7c, 0x4f, 0x69, 0xbf, 0xd1, 0x8c, 0xfe, 0x59, 0x55,
    0x5d, 0xca, 0x7c, 0x04,
};
static const struct drbg_kat_no_reseed kat163_nor_t = {
    12, kat163_nor_entropyin, kat163_nor_nonce, kat163_nor_persstr,
    kat163_nor_addin0, kat163_nor_addin1, kat163_nor_retbytes
};
static const struct drbg_kat kat163_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat163_nor_t
};

static const unsigned char kat164_nor_entropyin[] = {
    0xc0, 0x84, 0x69, 0xf8, 0x36, 0xfc, 0x12, 0x48, 0xde, 0xd4, 0xed, 0xd8,
    0xaf, 0x4d, 0x9d, 0xe8,
};
static const unsigned char kat164_nor_nonce[] = {
    0x74, 0x6e, 0xa2, 0xd7, 0x4c, 0xcc, 0xc4, 0xcd,
};
static const unsigned char kat164_nor_persstr[] = {
    0xa5, 0x18, 0x18, 0xae, 0x54, 0x37, 0x56, 0x25, 0x52, 0x65, 0x11, 0x80,
    0xf3, 0xde, 0x5d, 0xae,
};
static const unsigned char kat164_nor_addin0[] = {0};
static const unsigned char kat164_nor_addin1[] = {0};
static const unsigned char kat164_nor_retbytes[] = {
    0xa0, 0x8a, 0xee, 0xde, 0x46, 0x3f, 0xc9, 0xac, 0x48, 0x90, 0x7a, 0xb4,
    0xb0, 0xb3, 0x9d, 0x6f, 0x86, 0xa8, 0x79, 0xf4, 0x21, 0x81, 0x61, 0x41,
    0xa0, 0x52, 0xeb, 0x48, 0xd8, 0x7f, 0xf5, 0xd9, 0xb5, 0xb6, 0xcb, 0xfe,
    0xef, 0xdd, 0x8c, 0xfc, 0x17, 0x72, 0xeb, 0x62, 0x67, 0x12, 0x45, 0x3f,
    0x88, 0xec, 0x74, 0x7f, 0x6a, 0x05, 0xaf, 0x91, 0x7c, 0x9f, 0xa1, 0x61,
    0xed, 0xe1, 0x1d, 0x7f,
};
static const struct drbg_kat_no_reseed kat164_nor_t = {
    13, kat164_nor_entropyin, kat164_nor_nonce, kat164_nor_persstr,
    kat164_nor_addin0, kat164_nor_addin1, kat164_nor_retbytes
};
static const struct drbg_kat kat164_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat164_nor_t
};

static const unsigned char kat165_nor_entropyin[] = {
    0x0c, 0xe4, 0xc1, 0x82, 0x8f, 0x9c, 0x9f, 0x0a, 0xea, 0xb4, 0x95, 0x36,
    0x72, 0x23, 0x59, 0x23,
};
static const unsigned char kat165_nor_nonce[] = {
    0x46, 0x4b, 0x42, 0xc9, 0xfc, 0x5b, 0x38, 0x31,
};
static const unsigned char kat165_nor_persstr[] = {
    0x48, 0xde, 0x34, 0xd8, 0x48, 0x29, 0x8d, 0xcf, 0x3f, 0x58, 0xc5, 0x2d,
    0x96, 0xc7, 0x7b, 0xf5,
};
static const unsigned char kat165_nor_addin0[] = {0};
static const unsigned char kat165_nor_addin1[] = {0};
static const unsigned char kat165_nor_retbytes[] = {
    0xa7, 0xd3, 0x79, 0x02, 0x26, 0xf5, 0xab, 0x5b, 0x38, 0x33, 0xdc, 0xc7,
    0x63, 0xc2, 0xe9, 0xe7, 0xc5, 0xb7, 0x7d, 0x57, 0xc2, 0xfa, 0x26, 0x15,
    0x47, 0xae, 0x0e, 0x39, 0xe9, 0x78, 0x4d, 0xf2, 0x69, 0xd0, 0x8b, 0xbd,
    0x40, 0x36, 0x46, 0x2f, 0x3a, 0xcc, 0xc7, 0x1b, 0x37, 0x8b, 0x09, 0x41,
    0xe9, 0x9c, 0x32, 0x7c, 0x4a, 0x50, 0x34, 0x39, 0xa9, 0x3b, 0x4e, 0xc7,
    0xa0, 0x39, 0xa2, 0xa8,
};
static const struct drbg_kat_no_reseed kat165_nor_t = {
    14, kat165_nor_entropyin, kat165_nor_nonce, kat165_nor_persstr,
    kat165_nor_addin0, kat165_nor_addin1, kat165_nor_retbytes
};
static const struct drbg_kat kat165_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat165_nor_t
};

static const unsigned char kat166_nor_entropyin[] = {
    0x70, 0x7a, 0x43, 0x07, 0x2e, 0xac, 0xb8, 0xce, 0xa9, 0x5f, 0xb8, 0x52,
    0x1a, 0xe5, 0x16, 0x0b,
};
static const unsigned char kat166_nor_nonce[] = {
    0xfd, 0x62, 0x12, 0x13, 0x26, 0x16, 0x30, 0xad,
};
static const unsigned char kat166_nor_persstr[] = {
    0x6b, 0xf9, 0xa7, 0x2e, 0xaf, 0xe3, 0x5d, 0x58, 0x3c, 0x91, 0x5c, 0x95,
    0x25, 0x74, 0x7b, 0xa0,
};
static const unsigned char kat166_nor_addin0[] = {
    0xe3, 0xdf, 0xd6, 0x92, 0x7c, 0x4a, 0xe1, 0x03, 0x43, 0x2e, 0xb6, 0x19,
    0x63, 0x67, 0xec, 0xee,
};
static const unsigned char kat166_nor_addin1[] = {
    0xe1, 0x8c, 0xd5, 0x04, 0xe9, 0x40, 0x27, 0x53, 0x3c, 0xf3, 0x30, 0x71,
    0xca, 0x93, 0x1b, 0x60,
};
static const unsigned char kat166_nor_retbytes[] = {
    0xee, 0x46, 0x3c, 0xdc, 0x78, 0xdd, 0x25, 0x3c, 0x44, 0x66, 0xdd, 0xc2,
    0xe3, 0x5c, 0xc4, 0xa9, 0x1a, 0xf2, 0x0a, 0xd3, 0xb3, 0x39, 0x66, 0x69,
    0xce, 0xf5, 0x22, 0x1b, 0x0a, 0x9c, 0xcc, 0x5d, 0xfc, 0x72, 0x3b, 0x2f,
    0xe1, 0xe5, 0x7f, 0xa2, 0x6d, 0xd0, 0x30, 0xad, 0x0b, 0x6e, 0xa1, 0x4e,
    0xcb, 0x89, 0xf9, 0x1c, 0x4b, 0xc6, 0x93, 0x82, 0xa0, 0x1d, 0x8d, 0xfa,
    0x6f, 0x1d, 0xff, 0x8a,
};
static const struct drbg_kat_no_reseed kat166_nor_t = {
    0, kat166_nor_entropyin, kat166_nor_nonce, kat166_nor_persstr,
    kat166_nor_addin0, kat166_nor_addin1, kat166_nor_retbytes
};
static const struct drbg_kat kat166_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat166_nor_t
};

static const unsigned char kat167_nor_entropyin[] = {
    0x35, 0x67, 0xac, 0x54, 0xb9, 0x03, 0xfd, 0x5d, 0xdc, 0x57, 0x23, 0x4c,
    0x31, 0x9b, 0x14, 0x15,
};
static const unsigned char kat167_nor_nonce[] = {
    0x8f, 0x75, 0xc5, 0x6a, 0x85, 0x77, 0x21, 0x92,
};
static const unsigned char kat167_nor_persstr[] = {
    0xd1, 0xe3, 0xff, 0xed, 0x85, 0x21, 0x1c, 0x40, 0x8e, 0x31, 0xdb, 0x12,
    0xe8, 0x5b, 0x75, 0xdb,
};
static const unsigned char kat167_nor_addin0[] = {
    0xcc, 0x38, 0x84, 0x33, 0x5e, 0x19, 0x42, 0x7c, 0xaf, 0x09, 0xf0, 0x1c,
    0x63, 0x4b, 0x30, 0x8a,
};
static const unsigned char kat167_nor_addin1[] = {
    0xf7, 0x73, 0x3d, 0x02, 0x51, 0xff, 0x02, 0xaf, 0xc0, 0xe9, 0x61, 0xc1,
    0x12, 0x28, 0x84, 0xfc,
};
static const unsigned char kat167_nor_retbytes[] = {
    0x28, 0xc6, 0x9b, 0xb4, 0x98, 0x76, 0x42, 0xe5, 0xa4, 0x67, 0x93, 0x8f,
    0x52, 0xa7, 0x37, 0x14, 0xd3, 0x9a, 0xd0, 0x89, 0xee, 0x09, 0xda, 0xbe,
    0x70, 0xd6, 0x90, 0x36, 0x17, 0x8f, 0x59, 0x98, 0x18, 0x6c, 0xc6, 0x45,
    0x10, 0x0b, 0x56, 0xf9, 0x8e, 0xf4, 0x5c, 0xe3, 0x5b, 0x06, 0xf0, 0x9c,
    0x26, 0x20, 0xba, 0x68, 0x0a, 0x5a, 0x03, 0xd3, 0xd9, 0x59, 0x43, 0xed,
    0x17, 0x16, 0x0f, 0xe4,
};
static const struct drbg_kat_no_reseed kat167_nor_t = {
    1, kat167_nor_entropyin, kat167_nor_nonce, kat167_nor_persstr,
    kat167_nor_addin0, kat167_nor_addin1, kat167_nor_retbytes
};
static const struct drbg_kat kat167_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat167_nor_t
};

static const unsigned char kat168_nor_entropyin[] = {
    0x41, 0x76, 0x72, 0x58, 0xe0, 0x6e, 0x50, 0x11, 0x24, 0xf3, 0xdb, 0x8b,
    0xe4, 0x58, 0xd3, 0xbf,
};
static const unsigned char kat168_nor_nonce[] = {
    0xde, 0x8f, 0xa0, 0xa7, 0x9f, 0x1a, 0x7f, 0x0c,
};
static const unsigned char kat168_nor_persstr[] = {
    0x9d, 0xab, 0x2f, 0x46, 0xd2, 0x2d, 0x37, 0x86, 0x08, 0xe2, 0x1f, 0xc2,
    0x37, 0xe9, 0x98, 0x71,
};
static const unsigned char kat168_nor_addin0[] = {
    0xea, 0x41, 0xd1, 0x33, 0xb4, 0xf8, 0x6a, 0x35, 0x22, 0x7b, 0x76, 0x16,
    0x7f, 0x51, 0xbb, 0x5f,
};
static const unsigned char kat168_nor_addin1[] = {
    0x71, 0x61, 0xe1, 0x2c, 0x6f, 0x65, 0xc1, 0x42, 0xba, 0xf9, 0x4f, 0x1a,
    0x09, 0xe9, 0x4e, 0x22,
};
static const unsigned char kat168_nor_retbytes[] = {
    0xb3, 0x90, 0xc9, 0x52, 0x0e, 0xad, 0x28, 0x82, 0x69, 0x2a, 0x29, 0x59,
    0x3d, 0x16, 0x5c, 0x58, 0xbc, 0x66, 0x14, 0x62, 0x11, 0x2a, 0xf6, 0xd2,
    0x5a, 0x57, 0x14, 0x73, 0xe6, 0x1f, 0x1c, 0xbc, 0xd0, 0x75, 0xe3, 0xad,
    0x1c, 0xae, 0xeb, 0x51, 0xcb, 0x99, 0x28, 0x40, 0x51, 0xce, 0xa4, 0x5a,
    0x17, 0x06, 0xe9, 0x84, 0xe2, 0x7c, 0x32, 0x84, 0xfc, 0x1e, 0x87, 0xab,
    0x68, 0xa6, 0xf1, 0x46,
};
static const struct drbg_kat_no_reseed kat168_nor_t = {
    2, kat168_nor_entropyin, kat168_nor_nonce, kat168_nor_persstr,
    kat168_nor_addin0, kat168_nor_addin1, kat168_nor_retbytes
};
static const struct drbg_kat kat168_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat168_nor_t
};

static const unsigned char kat169_nor_entropyin[] = {
    0x19, 0xdb, 0x2d, 0xcd, 0xc5, 0x7d, 0xf2, 0x1e, 0x55, 0xb6, 0x56, 0x4e,
    0xae, 0x47, 0x41, 0x0b,
};
static const unsigned char kat169_nor_nonce[] = {
    0xeb, 0xec, 0x8e, 0xce, 0x62, 0xd3, 0x29, 0xb6,
};
static const unsigned char kat169_nor_persstr[] = {
    0x80, 0xba, 0xec, 0xa8, 0x2b, 0x3b, 0xa2, 0x07, 0xb9, 0xf3, 0x75, 0x08,
    0x38, 0xbd, 0xe4, 0x98,
};
static const unsigned char kat169_nor_addin0[] = {
    0x5d, 0x74, 0xcd, 0xd7, 0xbc, 0xbe, 0x7b, 0xc9, 0x48, 0xb8, 0xf1, 0x3e,
    0xfa, 0x83, 0x5a, 0xd4,
};
static const unsigned char kat169_nor_addin1[] = {
    0x17, 0xbb, 0x7c, 0xb9, 0x1b, 0x48, 0x11, 0xa5, 0xe7, 0xdb, 0x26, 0x78,
    0x4c, 0xbb, 0xc2, 0xe5,
};
static const unsigned char kat169_nor_retbytes[] = {
    0x26, 0x54, 0xd9, 0x86, 0x67, 0xf7, 0x56, 0x15, 0xfb, 0x6c, 0x74, 0xd5,
    0xb9, 0x0b, 0x62, 0x3c, 0xcc, 0xb9, 0x62, 0x49, 0xe1, 0x4e, 0x08, 0xe8,
    0x8d, 0x1c, 0xb5, 0xe0, 0x90, 0x93, 0x26, 0xb6, 0x77, 0x67, 0xa5, 0x73,
    0xe0, 0xa4, 0xd1, 0xdf, 0x05, 0x7f, 0x2d, 0xe7, 0xa4, 0x95, 0xde, 0xae,
    0xcb, 0xeb, 0x7e, 0x25, 0xe3, 0xaa, 0x82, 0xde, 0x55, 0x59, 0x08, 0xbf,
    0xe3, 0xcc, 0x6a, 0x85,
};
static const struct drbg_kat_no_reseed kat169_nor_t = {
    3, kat169_nor_entropyin, kat169_nor_nonce, kat169_nor_persstr,
    kat169_nor_addin0, kat169_nor_addin1, kat169_nor_retbytes
};
static const struct drbg_kat kat169_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat169_nor_t
};

static const unsigned char kat170_nor_entropyin[] = {
    0x64, 0xd7, 0xb4, 0x2c, 0x52, 0x64, 0x33, 0x76, 0xd8, 0xa1, 0x1b, 0x0d,
    0xa5, 0xcc, 0xa5, 0xe6,
};
static const unsigned char kat170_nor_nonce[] = {
    0x6f, 0xeb, 0x03, 0x5f, 0x39, 0x14, 0x25, 0x14,
};
static const unsigned char kat170_nor_persstr[] = {
    0x3f, 0x12, 0xa6, 0xf2, 0x40, 0x01, 0xb9, 0x8c, 0x1f, 0x24, 0xab, 0xcf,
    0xab, 0x68, 0x95, 0x08,
};
static const unsigned char kat170_nor_addin0[] = {
    0xfe, 0xe4, 0x82, 0xb1, 0xfe, 0xa0, 0x53, 0x65, 0xab, 0xa0, 0x7b, 0xc7,
    0xc3, 0x35, 0x1c, 0x6d,
};
static const unsigned char kat170_nor_addin1[] = {
    0xaf, 0x13, 0x59, 0x83, 0xab, 0x81, 0x13, 0x91, 0x45, 0x35, 0x13, 0x49,
    0x18, 0x78, 0xb5, 0xb5,
};
static const unsigned char kat170_nor_retbytes[] = {
    0x0c, 0x21, 0xf6, 0x97, 0xf1, 0xb0, 0x0d, 0x22, 0x83, 0xae, 0xe4, 0xe7,
    0xb7, 0xe1, 0x01, 0x33, 0x38, 0x6a, 0xb0, 0xab, 0xfa, 0xdb, 0x5e, 0x14,
    0xbf, 0xe5, 0xb4, 0x52, 0xa6, 0xd9, 0x17, 0x46, 0x2a, 0x7b, 0xd0, 0xed,
    0xa0, 0x6f, 0xa1, 0x41, 0xb1, 0xbc, 0x3e, 0xab, 0x4f, 0x47, 0x73, 0xa2,
    0x40, 0xbe, 0x73, 0x51, 0xe3, 0x2a, 0x1d, 0xab, 0xdd, 0xc5, 0x7b, 0xae,
    0x02, 0xd2, 0x6c, 0xd2,
};
static const struct drbg_kat_no_reseed kat170_nor_t = {
    4, kat170_nor_entropyin, kat170_nor_nonce, kat170_nor_persstr,
    kat170_nor_addin0, kat170_nor_addin1, kat170_nor_retbytes
};
static const struct drbg_kat kat170_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat170_nor_t
};

static const unsigned char kat171_nor_entropyin[] = {
    0xa5, 0x5a, 0x5a, 0x07, 0x29, 0xc3, 0x9e, 0xfb, 0xc8, 0x6c, 0xe6, 0x4d,
    0x91, 0x51, 0x03, 0x4e,
};
static const unsigned char kat171_nor_nonce[] = {
    0x10, 0xf1, 0x15, 0x57, 0x73, 0xfb, 0xcd, 0x3b,
};
static const unsigned char kat171_nor_persstr[] = {
    0x6c, 0x6f, 0x3f, 0x77, 0x80, 0x15, 0xb2, 0xd3, 0x34, 0x92, 0x2e, 0x7a,
    0x13, 0xea, 0xd6, 0xe6,
};
static const unsigned char kat171_nor_addin0[] = {
    0x84, 0x10, 0x15, 0x5a, 0x9f, 0x4d, 0x40, 0xa3, 0x29, 0x8d, 0x5b, 0x0f,
    0x42, 0xc0, 0xef, 0x46,
};
static const unsigned char kat171_nor_addin1[] = {
    0xfd, 0x91, 0x1c, 0x8b, 0xb6, 0xff, 0x14, 0xe7, 0xd7, 0x6a, 0x61, 0x1a,
    0xdd, 0xda, 0x72, 0x11,
};
static const unsigned char kat171_nor_retbytes[] = {
    0x16, 0x5f, 0x07, 0xd1, 0x4b, 0xb1, 0xb3, 0x7b, 0xd9, 0xe0, 0x94, 0x9e,
    0x7b, 0x0c, 0x0d, 0xfa, 0xdd, 0x06, 0x7a, 0xdf, 0x15, 0xb6, 0x6c, 0xb3,
    0x96, 0xe6, 0x44, 0x4b, 0xcd, 0xf1, 0x1b, 0xc2, 0x34, 0xb0, 0xd5, 0x9b,
    0x23, 0x17, 0x21, 0x4a, 0x9c, 0xde, 0x72, 0xe5, 0x32, 0xf3, 0x00, 0xc9,
    0xd0, 0x90, 0x7f, 0xf5, 0x9e, 0x62, 0x12, 0x82, 0x6a, 0x92, 0x0f, 0xd9,
    0x2b, 0x53, 0xad, 0xf7,
};
static const struct drbg_kat_no_reseed kat171_nor_t = {
    5, kat171_nor_entropyin, kat171_nor_nonce, kat171_nor_persstr,
    kat171_nor_addin0, kat171_nor_addin1, kat171_nor_retbytes
};
static const struct drbg_kat kat171_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat171_nor_t
};

static const unsigned char kat172_nor_entropyin[] = {
    0x53, 0xea, 0x1b, 0xef, 0x71, 0x12, 0xfd, 0xcb, 0x15, 0x51, 0x36, 0x16,
    0xae, 0x7e, 0x07, 0x03,
};
static const unsigned char kat172_nor_nonce[] = {
    0xca, 0xac, 0x5b, 0x2f, 0xbf, 0x5b, 0xd1, 0xb8,
};
static const unsigned char kat172_nor_persstr[] = {
    0x79, 0x04, 0x26, 0x8d, 0x84, 0x6f, 0x81, 0x41, 0xcd, 0x78, 0x3e, 0xcb,
    0xa6, 0xef, 0xc9, 0x01,
};
static const unsigned char kat172_nor_addin0[] = {
    0x03, 0x64, 0xae, 0xea, 0xf7, 0x42, 0x4c, 0x92, 0xc2, 0x61, 0x1f, 0x33,
    0x86, 0xfb, 0xe3, 0x8a,
};
static const unsigned char kat172_nor_addin1[] = {
    0x9c, 0x77, 0xd6, 0x80, 0xc5, 0xf2, 0x70, 0xae, 0xfa, 0x8a, 0x45, 0xb3,
    0xe9, 0x35, 0x53, 0x30,
};
static const unsigned char kat172_nor_retbytes[] = {
    0xe5, 0x6f, 0xa9, 0x81, 0x29, 0x23, 0x89, 0x0b, 0x94, 0x81, 0xee, 0x07,
    0x80, 0xea, 0x23, 0xb2, 0x05, 0x94, 0x7e, 0x80, 0xf3, 0xe3, 0x8f, 0x55,
    0xc2, 0x2d, 0xef, 0xce, 0x6c, 0xae, 0xa8, 0x4f, 0xd9, 0x85, 0x0f, 0x2c,
    0x50, 0x9c, 0x51, 0x48, 0xe0, 0x1c, 0xd9, 0x77, 0x34, 0x8c, 0xc8, 0xb3,
    0xcc, 0xab, 0xd3, 0xd7, 0x51, 0x5c, 0xc2, 0xca, 0x95, 0xd0, 0xbc, 0x13,
    0x48, 0xba, 0xc5, 0xc2,
};
static const struct drbg_kat_no_reseed kat172_nor_t = {
    6, kat172_nor_entropyin, kat172_nor_nonce, kat172_nor_persstr,
    kat172_nor_addin0, kat172_nor_addin1, kat172_nor_retbytes
};
static const struct drbg_kat kat172_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat172_nor_t
};

static const unsigned char kat173_nor_entropyin[] = {
    0x1f, 0xae, 0xf5, 0x93, 0x5c, 0xb8, 0x2d, 0x27, 0x08, 0x13, 0x25, 0x2c,
    0x77, 0xef, 0x3d, 0x6c,
};
static const unsigned char kat173_nor_nonce[] = {
    0x75, 0xf6, 0x8a, 0x1b, 0x65, 0x8c, 0x72, 0x77,
};
static const unsigned char kat173_nor_persstr[] = {
    0x08, 0x15, 0xfa, 0x38, 0x8d, 0xb0, 0x3a, 0x89, 0x2d, 0xf4, 0x47, 0x4f,
    0x19, 0xde, 0x31, 0xbc,
};
static const unsigned char kat173_nor_addin0[] = {
    0x9f, 0x92, 0x23, 0xae, 0x37, 0x86, 0xd3, 0x8c, 0x43, 0xe0, 0xe9, 0x2a,
    0xa1, 0xbf, 0x5e, 0x3f,
};
static const unsigned char kat173_nor_addin1[] = {
    0x4d, 0x9c, 0x28, 0xcd, 0x69, 0x4e, 0x01, 0x6f, 0xaf, 0xbd, 0xe3, 0xcb,
    0x30, 0x85, 0xb8, 0xd9,
};
static const unsigned char kat173_nor_retbytes[] = {
    0xcd, 0x69, 0x3a, 0xab, 0x02, 0xec, 0x80, 0xa0, 0x4b, 0xeb, 0xcf, 0xef,
    0x66, 0x04, 0x38, 0x4f, 0x48, 0xc8, 0xeb, 0x7d, 0x44, 0xd6, 0x91, 0xac,
    0x35, 0xfb, 0x11, 0xe1, 0x81, 0x83, 0x88, 0x4b, 0x9e, 0x35, 0xd5, 0x8e,
    0x62, 0xfc, 0x56, 0x7b, 0x9a, 0x3d, 0xa4, 0x0e, 0x3a, 0xee, 0xa7, 0x83,
    0x11, 0x5e, 0x80, 0x19, 0x22, 0x09, 0x03, 0xab, 0x55, 0x9b, 0xe9, 0x83,
    0x4c, 0x37, 0x77, 0x58,
};
static const struct drbg_kat_no_reseed kat173_nor_t = {
    7, kat173_nor_entropyin, kat173_nor_nonce, kat173_nor_persstr,
    kat173_nor_addin0, kat173_nor_addin1, kat173_nor_retbytes
};
static const struct drbg_kat kat173_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat173_nor_t
};

static const unsigned char kat174_nor_entropyin[] = {
    0x0b, 0x8c, 0x52, 0x08, 0xd9, 0x11, 0x3c, 0xbe, 0x8a, 0x31, 0xcf, 0x3d,
    0xe1, 0x43, 0x38, 0x1a,
};
static const unsigned char kat174_nor_nonce[] = {
    0x44, 0x9e, 0xbf, 0x57, 0xb8, 0x2a, 0xa5, 0xcb,
};
static const unsigned char kat174_nor_persstr[] = {
    0x7f, 0x60, 0x01, 0xd6, 0x0c, 0x31, 0xcf, 0x8a, 0x90, 0x4c, 0xb3, 0x18,
    0xcb, 0x70, 0x04, 0x7b,
};
static const unsigned char kat174_nor_addin0[] = {
    0x4f, 0xe0, 0x45, 0xed, 0xbb, 0x98, 0x16, 0x0a, 0x42, 0x63, 0x5b, 0x77,
    0x79, 0x6e, 0x8a, 0x97,
};
static const unsigned char kat174_nor_addin1[] = {
    0xd8, 0x96, 0x4e, 0x5f, 0xa8, 0x16, 0xea, 0x83, 0xd1, 0xa5, 0xab, 0xbd,
    0x15, 0x68, 0xad, 0xaa,
};
static const unsigned char kat174_nor_retbytes[] = {
    0xab, 0x1f, 0xba, 0xf1, 0x54, 0xa9, 0x07, 0x2b, 0x62, 0xc3, 0xd6, 0xbb,
    0x9d, 0x01, 0x48, 0xac, 0xe4, 0xb4, 0xcd, 0xe0, 0x8d, 0x9a, 0x71, 0x2a,
    0xa2, 0x94, 0x10, 0xab, 0x51, 0x4d, 0x03, 0xe3, 0x5d, 0xb3, 0xf4, 0x91,
    0x80, 0x66, 0x73, 0x04, 0x18, 0x33, 0xa2, 0xeb, 0xeb, 0xe8, 0x70, 0x55,
    0xf1, 0xf5, 0xfb, 0x64, 0xdd, 0x8e, 0x40, 0x8e, 0x3b, 0x99, 0x57, 0xd2,
    0x1e, 0x66, 0xca, 0xa2,
};
static const struct drbg_kat_no_reseed kat174_nor_t = {
    8, kat174_nor_entropyin, kat174_nor_nonce, kat174_nor_persstr,
    kat174_nor_addin0, kat174_nor_addin1, kat174_nor_retbytes
};
static const struct drbg_kat kat174_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat174_nor_t
};

static const unsigned char kat175_nor_entropyin[] = {
    0xae, 0x90, 0x6d, 0x89, 0x04, 0x15, 0x45, 0xc7, 0xef, 0x70, 0x71, 0xf5,
    0x3e, 0xce, 0xea, 0x52,
};
static const unsigned char kat175_nor_nonce[] = {
    0x1f, 0x4b, 0x28, 0x4c, 0x36, 0xaa, 0xe3, 0xba,
};
static const unsigned char kat175_nor_persstr[] = {
    0xb9, 0x19, 0x98, 0xc3, 0x21, 0x83, 0x00, 0x6f, 0x7d, 0xf1, 0xc5, 0xd7,
    0x1f, 0xba, 0xc5, 0x7b,
};
static const unsigned char kat175_nor_addin0[] = {
    0x86, 0x47, 0x5c, 0xa2, 0xb4, 0x84, 0xb8, 0xc6, 0xf3, 0x3b, 0x11, 0x22,
    0x8b, 0xc6, 0x65, 0x05,
};
static const unsigned char kat175_nor_addin1[] = {
    0x0c, 0x34, 0xe0, 0xea, 0xe8, 0xfa, 0x94, 0xc2, 0x6c, 0xb5, 0x11, 0xc7,
    0x9a, 0x49, 0x83, 0x3e,
};
static const unsigned char kat175_nor_retbytes[] = {
    0x89, 0xeb, 0xf1, 0x81, 0x5e, 0x2c, 0x33, 0x32, 0x59, 0xd6, 0xaa, 0xa0,
    0x07, 0xd1, 0x9c, 0xa5, 0xe0, 0x1e, 0x59, 0x87, 0x76, 0xac, 0x2a, 0xf5,
    0x5a, 0xbf, 0x78, 0xc1, 0x1b, 0x4e, 0xd7, 0x44, 0xb6, 0xde, 0x87, 0x5a,
    0x36, 0xf4, 0x52, 0xc2, 0x7a, 0xc4, 0xa0, 0xe6, 0x77, 0x93, 0x8b, 0x16,
    0x1b, 0x2c, 0x00, 0xeb, 0x06, 0x59, 0xbd, 0xf2, 0xdc, 0x42, 0x5a, 0x89,
    0xa2, 0x4a, 0xff, 0xd0,
};
static const struct drbg_kat_no_reseed kat175_nor_t = {
    9, kat175_nor_entropyin, kat175_nor_nonce, kat175_nor_persstr,
    kat175_nor_addin0, kat175_nor_addin1, kat175_nor_retbytes
};
static const struct drbg_kat kat175_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat175_nor_t
};

static const unsigned char kat176_nor_entropyin[] = {
    0x1d, 0x0e, 0x60, 0xe6, 0x80, 0x17, 0x40, 0x42, 0x1d, 0x29, 0xf6, 0x5a,
    0x60, 0x60, 0x6c, 0x1b,
};
static const unsigned char kat176_nor_nonce[] = {
    0x37, 0x03, 0x36, 0xdb, 0x08, 0x35, 0xe9, 0x1a,
};
static const unsigned char kat176_nor_persstr[] = {
    0x7a, 0x27, 0x3b, 0x4d, 0x99, 0x75, 0x35, 0x11, 0xfc, 0xf3, 0x4f, 0x3b,
    0x1b, 0xb2, 0x9e, 0x16,
};
static const unsigned char kat176_nor_addin0[] = {
    0xd6, 0x44, 0xcb, 0x46, 0xe0, 0x1c, 0x07, 0x12, 0xef, 0x9d, 0xa4, 0xed,
    0xe5, 0xca, 0x40, 0x7c,
};
static const unsigned char kat176_nor_addin1[] = {
    0x5c, 0x99, 0x05, 0x45, 0xee, 0xaf, 0x59, 0xde, 0x4d, 0x34, 0x9a, 0xe6,
    0x66, 0x1f, 0x70, 0x41,
};
static const unsigned char kat176_nor_retbytes[] = {
    0xda, 0xd5, 0x03, 0xe7, 0x20, 0x96, 0x02, 0xa8, 0x6f, 0x18, 0xd0, 0x1a,
    0x2b, 0x05, 0x15, 0xb4, 0xb3, 0xd4, 0xa4, 0xc0, 0x37, 0xaf, 0x71, 0x86,
    0xec, 0xba, 0x25, 0xdc, 0xb7, 0x81, 0xd9, 0x73, 0x11, 0x81, 0x0f, 0x98,
    0xc1, 0x7c, 0x34, 0x17, 0x30, 0x61, 0xab, 0x78, 0xec, 0xcb, 0xd4, 0x91,
    0x85, 0xcf, 0x5d, 0x4f, 0xb9, 0x1d, 0xff, 0xf4, 0xb0, 0xb6, 0x53, 0xdc,
    0x15, 0x89, 0x92, 0x29,
};
static const struct drbg_kat_no_reseed kat176_nor_t = {
    10, kat176_nor_entropyin, kat176_nor_nonce, kat176_nor_persstr,
    kat176_nor_addin0, kat176_nor_addin1, kat176_nor_retbytes
};
static const struct drbg_kat kat176_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat176_nor_t
};

static const unsigned char kat177_nor_entropyin[] = {
    0x67, 0x58, 0x85, 0x38, 0x3c, 0x6b, 0x3b, 0x00, 0x8e, 0x88, 0x49, 0x37,
    0xed, 0xa7, 0x7b, 0xe8,
};
static const unsigned char kat177_nor_nonce[] = {
    0xb9, 0x1c, 0x7c, 0x06, 0x87, 0x85, 0x32, 0xf2,
};
static const unsigned char kat177_nor_persstr[] = {
    0x12, 0xce, 0x97, 0xb4, 0x0a, 0x1b, 0xcd, 0xe5, 0x03, 0x90, 0xe3, 0x15,
    0xcd, 0x91, 0x1d, 0x53,
};
static const unsigned char kat177_nor_addin0[] = {
    0x10, 0x58, 0x43, 0x4c, 0xf9, 0xb3, 0x8f, 0xd7, 0x2a, 0x14, 0x0c, 0xf0,
    0x94, 0x24, 0x36, 0xe3,
};
static const unsigned char kat177_nor_addin1[] = {
    0x59, 0x8b, 0xf8, 0x01, 0x43, 0x39, 0x82, 0x09, 0x66, 0xd9, 0x4b, 0xe1,
    0xe2, 0xe9, 0xc8, 0x55,
};
static const unsigned char kat177_nor_retbytes[] = {
    0x45, 0x48, 0x8c, 0xa0, 0xe2, 0xf2, 0x83, 0xf1, 0x74, 0x1b, 0xcb, 0xf4,
    0x9a, 0x79, 0x36, 0x54, 0x59, 0x07, 0x52, 0x49, 0xe1, 0xc4, 0x17, 0x77,
    0xf3, 0xe0, 0x08, 0xd8, 0x02, 0x0e, 0xe1, 0xf5, 0x98, 0xac, 0xa0, 0xad,
    0x26, 0xed, 0xcd, 0x92, 0x0c, 0x85, 0x59, 0xd0, 0xea, 0xf7, 0xcc, 0x5d,
    0x5a, 0x71, 0x44, 0x37, 0xc0, 0x45, 0x81, 0xa6, 0x4c, 0x74, 0x99, 0xe5,
    0xf0, 0xbe, 0x08, 0x9c,
};
static const struct drbg_kat_no_reseed kat177_nor_t = {
    11, kat177_nor_entropyin, kat177_nor_nonce, kat177_nor_persstr,
    kat177_nor_addin0, kat177_nor_addin1, kat177_nor_retbytes
};
static const struct drbg_kat kat177_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat177_nor_t
};

static const unsigned char kat178_nor_entropyin[] = {
    0x71, 0x23, 0x93, 0xa9, 0xf4, 0xa1, 0xb0, 0xbb, 0xab, 0xf4, 0xf4, 0x96,
    0xe8, 0x17, 0x0a, 0xdc,
};
static const unsigned char kat178_nor_nonce[] = {
    0x0f, 0x59, 0x0c, 0x32, 0x39, 0xc6, 0xcf, 0x47,
};
static const unsigned char kat178_nor_persstr[] = {
    0xb7, 0x2b, 0x40, 0x97, 0xa9, 0xfb, 0x3d, 0x45, 0xbb, 0x06, 0x24, 0x15,
    0xb6, 0xf2, 0xdf, 0xb1,
};
static const unsigned char kat178_nor_addin0[] = {
    0xe2, 0x7a, 0xde, 0x7c, 0x11, 0x86, 0x56, 0x55, 0x3f, 0x06, 0xec, 0x20,
    0x19, 0x9b, 0x53, 0x70,
};
static const unsigned char kat178_nor_addin1[] = {
    0x4e, 0xb4, 0xfd, 0xa3, 0xff, 0xda, 0xd4, 0xf9, 0xdd, 0xaf, 0xd0, 0xd0,
    0xbf, 0xec, 0xd4, 0x44,
};
static const unsigned char kat178_nor_retbytes[] = {
    0x04, 0x49, 0xa8, 0x2a, 0x31, 0x7e, 0x22, 0xe5, 0xa7, 0xa6, 0x68, 0x4d,
    0x08, 0xb8, 0xf6, 0x3e, 0x02, 0xd8, 0xb9, 0x08, 0x5d, 0x61, 0x9a, 0xb5,
    0x6c, 0xde, 0x52, 0x2c, 0x86, 0x67, 0xd5, 0xf8, 0x8b, 0x2e, 0xa8, 0x83,
    0x03, 0x29, 0x49, 0x1a, 0x5e, 0xa8, 0x28, 0xe0, 0xa3, 0x66, 0xae, 0x7e,
    0x84, 0xf2, 0xcd, 0xee, 0x69, 0x8a, 0x92, 0x41, 0x18, 0x88, 0x83, 0xbe,
    0x00, 0xc5, 0x33, 0xe8,
};
static const struct drbg_kat_no_reseed kat178_nor_t = {
    12, kat178_nor_entropyin, kat178_nor_nonce, kat178_nor_persstr,
    kat178_nor_addin0, kat178_nor_addin1, kat178_nor_retbytes
};
static const struct drbg_kat kat178_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat178_nor_t
};

static const unsigned char kat179_nor_entropyin[] = {
    0x57, 0x66, 0x40, 0xf9, 0x76, 0xa6, 0x28, 0xb0, 0x9e, 0x98, 0xaf, 0x48,
    0x3b, 0x64, 0x5e, 0x56,
};
static const unsigned char kat179_nor_nonce[] = {
    0x82, 0x2e, 0x8a, 0x10, 0xdf, 0x06, 0xd0, 0xfe,
};
static const unsigned char kat179_nor_persstr[] = {
    0x6d, 0x6a, 0x6f, 0x68, 0xbb, 0x38, 0x1c, 0x29, 0x1f, 0x3f, 0xa6, 0x21,
    0xd6, 0xab, 0x64, 0xb2,
};
static const unsigned char kat179_nor_addin0[] = {
    0xb0, 0x5f, 0x14, 0xe4, 0x1f, 0x54, 0x33, 0x8c, 0x44, 0x15, 0xa5, 0xb5,
    0xc8, 0xc5, 0x61, 0x99,
};
static const unsigned char kat179_nor_addin1[] = {
    0xc6, 0x84, 0x7c, 0x9e, 0xd7, 0x4c, 0x86, 0x9a, 0xc4, 0xa5, 0x6f, 0xf2,
    0xd0, 0x95, 0x6d, 0x6b,
};
static const unsigned char kat179_nor_retbytes[] = {
    0x5c, 0x6a, 0x7e, 0x58, 0xea, 0xcf, 0xa5, 0x8d, 0xca, 0x93, 0x99, 0xa3,
    0xf9, 0xb6, 0xcb, 0x41, 0xd2, 0xbc, 0x24, 0xf6, 0x72, 0x58, 0x7d, 0xb2,
    0x29, 0xbf, 0x22, 0x20, 0xc2, 0x85, 0x13, 0x47, 0xd4, 0x7d, 0xa1, 0xca,
    0x0a, 0x6a, 0x13, 0x77, 0x96, 0x05, 0xdb, 0x90, 0xd9, 0x1a, 0x61, 0x48,
    0x88, 0x18, 0xd3, 0x3b, 0x2a, 0x4a, 0xd6, 0x74, 0x09, 0x00, 0x68, 0x5f,
    0x8e, 0xc6, 0x13, 0x6f,
};
static const struct drbg_kat_no_reseed kat179_nor_t = {
    13, kat179_nor_entropyin, kat179_nor_nonce, kat179_nor_persstr,
    kat179_nor_addin0, kat179_nor_addin1, kat179_nor_retbytes
};
static const struct drbg_kat kat179_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat179_nor_t
};

static const unsigned char kat180_nor_entropyin[] = {
    0x69, 0x65, 0x03, 0xc8, 0x48, 0x60, 0x51, 0x87, 0x84, 0xdd, 0x4c, 0x6c,
    0x31, 0x71, 0x0f, 0x43,
};
static const unsigned char kat180_nor_nonce[] = {
    0xee, 0x52, 0xc0, 0xa4, 0xcd, 0x90, 0xdf, 0x1e,
};
static const unsigned char kat180_nor_persstr[] = {
    0x9a, 0x08, 0xd6, 0xb4, 0x80, 0xef, 0x3a, 0xc2, 0x00, 0xa1, 0x30, 0x87,
    0x8d, 0x80, 0xd0, 0x79,
};
static const unsigned char kat180_nor_addin0[] = {
    0x5c, 0x51, 0x31, 0x59, 0x7b, 0xd3, 0xab, 0x02, 0x5c, 0xa0, 0x05, 0x79,
    0xc8, 0x6f, 0xfd, 0xed,
};
static const unsigned char kat180_nor_addin1[] = {
    0xa8, 0xcc, 0x33, 0xc8, 0x51, 0xee, 0x11, 0xe4, 0x70, 0x1a, 0x32, 0x19,
    0x7b, 0x61, 0x76, 0x23,
};
static const unsigned char kat180_nor_retbytes[] = {
    0x9a, 0xfd, 0xc4, 0x54, 0xf5, 0xc2, 0xce, 0x29, 0x95, 0xfc, 0x7c, 0xcf,
    0xb4, 0xe8, 0x67, 0x1a, 0x4b, 0x27, 0xdf, 0xcb, 0x1c, 0x02, 0xfe, 0xc6,
    0x07, 0xaa, 0xfc, 0x64, 0x22, 0x65, 0x70, 0x96, 0xce, 0x18, 0x14, 0x10,
    0x1b, 0x8a, 0x74, 0x3e, 0x2d, 0x4a, 0x2c, 0x4e, 0x2e, 0x8d, 0xa2, 0x57,
    0x57, 0x8d, 0xe7, 0x10, 0x54, 0x12, 0xf3, 0x52, 0xa4, 0xcd, 0x14, 0xaf,
    0x7d, 0xb0, 0xeb, 0x03,
};
static const struct drbg_kat_no_reseed kat180_nor_t = {
    14, kat180_nor_entropyin, kat180_nor_nonce, kat180_nor_persstr,
    kat180_nor_addin0, kat180_nor_addin1, kat180_nor_retbytes
};
static const struct drbg_kat kat180_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat180_nor_t
};

static const unsigned char kat181_nor_entropyin[] = {
    0x6b, 0xdf, 0x53, 0x32, 0xbd, 0xce, 0x46, 0x55, 0xd4, 0x5c, 0x2c, 0xfe,
    0xa8, 0x97, 0xb0, 0x00,
};
static const unsigned char kat181_nor_nonce[] = {
    0xe7, 0x8c, 0x55, 0x71, 0xc5, 0xf9, 0x26, 0xf9,
};
static const unsigned char kat181_nor_persstr[] = {0};
static const unsigned char kat181_nor_addin0[] = {0};
static const unsigned char kat181_nor_addin1[] = {0};
static const unsigned char kat181_nor_retbytes[] = {
    0xe0, 0x71, 0x56, 0x88, 0x76, 0x5a, 0x32, 0x85, 0xe7, 0xb7, 0xdb, 0x55,
    0x5f, 0x27, 0x79, 0x24, 0xe7, 0x17, 0x1f, 0x75, 0x41, 0xbf, 0x26, 0x12,
    0x2b, 0x13, 0xdb, 0xaa, 0xa3, 0x9f, 0x9e, 0x2b, 0x03, 0x45, 0xc6, 0x59,
    0x58, 0x3f, 0xf8, 0xc9, 0xcf, 0xd8, 0x88, 0xf1, 0xab, 0xd2, 0xf3, 0xb3,
    0x6a, 0x7c, 0x9d, 0x47, 0xc6, 0x87, 0xb0, 0x1c, 0x81, 0x9a, 0x9f, 0x98,
    0x88, 0x54, 0x2e, 0x0f,
};
static const struct drbg_kat_no_reseed kat181_nor_t = {
    0, kat181_nor_entropyin, kat181_nor_nonce, kat181_nor_persstr,
    kat181_nor_addin0, kat181_nor_addin1, kat181_nor_retbytes
};
static const struct drbg_kat kat181_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat181_nor_t
};

static const unsigned char kat182_nor_entropyin[] = {
    0xa4, 0x73, 0x7d, 0x48, 0xa8, 0x93, 0x25, 0x07, 0x85, 0x79, 0xe6, 0x49,
    0xe2, 0xfa, 0x65, 0xeb,
};
static const unsigned char kat182_nor_nonce[] = {
    0x6a, 0x79, 0x9a, 0x7a, 0x2f, 0x13, 0xe8, 0x13,
};
static const unsigned char kat182_nor_persstr[] = {0};
static const unsigned char kat182_nor_addin0[] = {0};
static const unsigned char kat182_nor_addin1[] = {0};
static const unsigned char kat182_nor_retbytes[] = {
    0x75, 0x2a, 0x91, 0x6d, 0x98, 0x05, 0x18, 0xd9, 0xe7, 0xe4, 0x75, 0x99,
    0x06, 0x6b, 0x45, 0x71, 0x46, 0x61, 0xf3, 0x41, 0x59, 0xf0, 0xc8, 0xce,
    0xa8, 0xda, 0xbd, 0x59, 0x6a, 0x06, 0x6a, 0xff, 0x7a, 0xe6, 0xc2, 0x1e,
    0x69, 0xa3, 0x56, 0xdd, 0x2e, 0xe0, 0xda, 0x55, 0x42, 0x9c, 0x67, 0x5a,
    0xa6, 0xfa, 0x09, 0x00, 0x17, 0x3f, 0x54, 0x77, 0xcd, 0x7f, 0xd6, 0x49,
    0xea, 0xe0, 0xc9, 0x9a,
};
static const struct drbg_kat_no_reseed kat182_nor_t = {
    1, kat182_nor_entropyin, kat182_nor_nonce, kat182_nor_persstr,
    kat182_nor_addin0, kat182_nor_addin1, kat182_nor_retbytes
};
static const struct drbg_kat kat182_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat182_nor_t
};

static const unsigned char kat183_nor_entropyin[] = {
    0x99, 0xb4, 0xe9, 0xe5, 0xba, 0xaa, 0x99, 0xbf, 0x8b, 0x8a, 0x0c, 0x2c,
    0xfd, 0x9a, 0x89, 0xce,
};
static const unsigned char kat183_nor_nonce[] = {
    0xf4, 0x8b, 0x31, 0x29, 0x41, 0xd3, 0x55, 0x4d,
};
static const unsigned char kat183_nor_persstr[] = {0};
static const unsigned char kat183_nor_addin0[] = {0};
static const unsigned char kat183_nor_addin1[] = {0};
static const unsigned char kat183_nor_retbytes[] = {
    0x85, 0xcf, 0x14, 0x8c, 0x65, 0xec, 0xe7, 0x52, 0x5d, 0xea, 0x96, 0x34,
    0x4d, 0xa6, 0x45, 0x11, 0x99, 0x49, 0x21, 0x85, 0xc8, 0xb8, 0xdf, 0xff,
    0x50, 0x0d, 0xdd, 0x68, 0xca, 0xab, 0x50, 0xbd, 0x74, 0x18, 0x66, 0x93,
    0x7a, 0x50, 0x1a, 0xf8, 0x76, 0xae, 0x84, 0x9b, 0x5b, 0x1b, 0x53, 0x30,
    0xde, 0x65, 0xeb, 0xf3, 0x8e, 0x9d, 0x55, 0x96, 0x93, 0xfa, 0xe8, 0x05,
    0xcc, 0xc9, 0xaa, 0xed,
};
static const struct drbg_kat_no_reseed kat183_nor_t = {
    2, kat183_nor_entropyin, kat183_nor_nonce, kat183_nor_persstr,
    kat183_nor_addin0, kat183_nor_addin1, kat183_nor_retbytes
};
static const struct drbg_kat kat183_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat183_nor_t
};

static const unsigned char kat184_nor_entropyin[] = {
    0x7c, 0x1e, 0xf8, 0x46, 0xe7, 0xf3, 0xa7, 0xcf, 0xe5, 0xc7, 0xbb, 0x47,
    0xe4, 0x6a, 0xa0, 0xe8,
};
static const unsigned char kat184_nor_nonce[] = {
    0x78, 0xeb, 0x00, 0x29, 0x55, 0xd2, 0x70, 0xd3,
};
static const unsigned char kat184_nor_persstr[] = {0};
static const unsigned char kat184_nor_addin0[] = {0};
static const unsigned char kat184_nor_addin1[] = {0};
static const unsigned char kat184_nor_retbytes[] = {
    0xb7, 0x39, 0xf9, 0x53, 0x39, 0xe8, 0x3f, 0xe8, 0xa3, 0x39, 0x16, 0x2f,
    0x1e, 0x96, 0xcc, 0x82, 0xb6, 0x7a, 0xf4, 0x17, 0x59, 0xf4, 0x83, 0x06,
    0x4e, 0x61, 0xa0, 0x39, 0x13, 0x7a, 0xf0, 0x7d, 0x93, 0x4a, 0x0e, 0xb7,
    0xca, 0x72, 0x84, 0xc1, 0x46, 0x86, 0xca, 0x04, 0x88, 0x95, 0x3e, 0xe4,
    0x40, 0xf6, 0xcd, 0x11, 0x45, 0xd0, 0x76, 0x6f, 0x4f, 0xf1, 0xc7, 0x7b,
    0x0d, 0x81, 0xd3, 0x61,
};
static const struct drbg_kat_no_reseed kat184_nor_t = {
    3, kat184_nor_entropyin, kat184_nor_nonce, kat184_nor_persstr,
    kat184_nor_addin0, kat184_nor_addin1, kat184_nor_retbytes
};
static const struct drbg_kat kat184_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat184_nor_t
};

static const unsigned char kat185_nor_entropyin[] = {
    0x6f, 0x75, 0x0f, 0xb7, 0x3f, 0xbc, 0xb6, 0xdb, 0x57, 0x50, 0x6c, 0xf2,
    0xde, 0xfa, 0xf6, 0xd4,
};
static const unsigned char kat185_nor_nonce[] = {
    0xa5, 0x2c, 0x92, 0x97, 0xe0, 0x2f, 0x42, 0x55,
};
static const unsigned char kat185_nor_persstr[] = {0};
static const unsigned char kat185_nor_addin0[] = {0};
static const unsigned char kat185_nor_addin1[] = {0};
static const unsigned char kat185_nor_retbytes[] = {
    0x0c, 0xa7, 0x48, 0xfc, 0xb3, 0xba, 0x23, 0xbd, 0x04, 0x3f, 0x48, 0x50,
    0x75, 0x85, 0x42, 0x52, 0x34, 0xec, 0x4a, 0x0f, 0x35, 0x0e, 0xfd, 0xcc,
    0x87, 0xf9, 0x06, 0x2c, 0x8a, 0xe0, 0xcf, 0x1e, 0x03, 0x3b, 0x7d, 0xf8,
    0x35, 0x7f, 0x5b, 0x0e, 0x4f, 0x7c, 0x21, 0xd4, 0xfb, 0xdc, 0xf8, 0x9c,
    0xe0, 0x19, 0x9c, 0x25, 0x79, 0x02, 0x70, 0xfe, 0x67, 0xff, 0xee, 0xc2,
    0x6c, 0xfc, 0x4d, 0x18,
};
static const struct drbg_kat_no_reseed kat185_nor_t = {
    4, kat185_nor_entropyin, kat185_nor_nonce, kat185_nor_persstr,
    kat185_nor_addin0, kat185_nor_addin1, kat185_nor_retbytes
};
static const struct drbg_kat kat185_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat185_nor_t
};

static const unsigned char kat186_nor_entropyin[] = {
    0x44, 0xb6, 0xc3, 0x9a, 0x3a, 0xf6, 0xa4, 0x73, 0x14, 0x8e, 0x32, 0xd9,
    0x79, 0x2b, 0x9e, 0xee,
};
static const unsigned char kat186_nor_nonce[] = {
    0xde, 0x44, 0xdd, 0x36, 0x00, 0x3e, 0x48, 0x22,
};
static const unsigned char kat186_nor_persstr[] = {0};
static const unsigned char kat186_nor_addin0[] = {0};
static const unsigned char kat186_nor_addin1[] = {0};
static const unsigned char kat186_nor_retbytes[] = {
    0x10, 0x12, 0x20, 0x08, 0x7c, 0xa7, 0x70, 0xe4, 0x57, 0x4f, 0xd0, 0x5b,
    0x2e, 0x88, 0x85, 0x1b, 0x48, 0x09, 0xcf, 0x21, 0xe6, 0x5e, 0xb0, 0xe3,
    0xd1, 0xec, 0xda, 0x29, 0xaa, 0xcf, 0x2d, 0x93, 0xe7, 0x5d, 0x79, 0x3b,
    0x99, 0x24, 0xae, 0xdd, 0x9b, 0x9a, 0x38, 0x84, 0x4d, 0xb4, 0x30, 0x47,
    0x0d, 0x01, 0x5d, 0xa6, 0x04, 0x18, 0x75, 0x3f, 0x2e, 0x3c, 0x6a, 0x15,
    0xf5, 0x58, 0xf4, 0xa9,
};
static const struct drbg_kat_no_reseed kat186_nor_t = {
    5, kat186_nor_entropyin, kat186_nor_nonce, kat186_nor_persstr,
    kat186_nor_addin0, kat186_nor_addin1, kat186_nor_retbytes
};
static const struct drbg_kat kat186_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat186_nor_t
};

static const unsigned char kat187_nor_entropyin[] = {
    0xe4, 0xd1, 0xff, 0x1c, 0x3c, 0xd2, 0x78, 0x49, 0xa2, 0x97, 0xe5, 0xa9,
    0x8e, 0xde, 0x08, 0x6a,
};
static const unsigned char kat187_nor_nonce[] = {
    0xf8, 0x9d, 0x58, 0xb0, 0x61, 0x7e, 0x5d, 0x10,
};
static const unsigned char kat187_nor_persstr[] = {0};
static const unsigned char kat187_nor_addin0[] = {0};
static const unsigned char kat187_nor_addin1[] = {0};
static const unsigned char kat187_nor_retbytes[] = {
    0x54, 0xc4, 0x67, 0x97, 0xa3, 0x8e, 0x11, 0xc5, 0x43, 0x47, 0x78, 0x09,
    0xd9, 0xd6, 0xed, 0x0a, 0xe4, 0x02, 0x02, 0x85, 0xed, 0xf9, 0x9e, 0x7e,
    0xc9, 0x45, 0xcf, 0xcf, 0x21, 0xdd, 0xaa, 0x6d, 0x9a, 0x71, 0x85, 0x6b,
    0xdb, 0xd6, 0x61, 0x3b, 0xf8, 0xe9, 0x7e, 0x0a, 0xf8, 0xd3, 0xc4, 0xce,
    0xdd, 0x0d, 0xfc, 0xfb, 0x47, 0x42, 0xa2, 0xef, 0x0a, 0x44, 0x3e, 0xf8,
    0x78, 0x96, 0x0e, 0x6f,
};
static const struct drbg_kat_no_reseed kat187_nor_t = {
    6, kat187_nor_entropyin, kat187_nor_nonce, kat187_nor_persstr,
    kat187_nor_addin0, kat187_nor_addin1, kat187_nor_retbytes
};
static const struct drbg_kat kat187_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat187_nor_t
};

static const unsigned char kat188_nor_entropyin[] = {
    0x15, 0x14, 0xbe, 0x70, 0x6e, 0x4c, 0x16, 0x72, 0x90, 0x55, 0x06, 0xb0,
    0xc5, 0x53, 0x47, 0xf2,
};
static const unsigned char kat188_nor_nonce[] = {
    0xad, 0x8d, 0x88, 0x9f, 0x2b, 0x91, 0xa6, 0xdd,
};
static const unsigned char kat188_nor_persstr[] = {0};
static const unsigned char kat188_nor_addin0[] = {0};
static const unsigned char kat188_nor_addin1[] = {0};
static const unsigned char kat188_nor_retbytes[] = {
    0xb3, 0x64, 0x4a, 0x6d, 0x16, 0x34, 0x0e, 0xf8, 0xb2, 0x8a, 0xd0, 0x6f,
    0x4c, 0xbf, 0xd5, 0xe7, 0x99, 0xf0, 0x80, 0xd4, 0xb2, 0xf8, 0x31, 0xfd,
    0x90, 0xb3, 0x13, 0xc8, 0x62, 0xd5, 0x76, 0x3d, 0x22, 0xb1, 0x21, 0x7f,
    0xe3, 0xd4, 0xf0, 0x7c, 0x00, 0x6a, 0x39, 0x89, 0x5d, 0x46, 0xd5, 0x24,
    0x70, 0xb3, 0xf8, 0xb7, 0x75, 0x21, 0xe1, 0x41, 0xf6, 0x8b, 0x06, 0xb5,
    0x4c, 0x38, 0x66, 0x6f,
};
static const struct drbg_kat_no_reseed kat188_nor_t = {
    7, kat188_nor_entropyin, kat188_nor_nonce, kat188_nor_persstr,
    kat188_nor_addin0, kat188_nor_addin1, kat188_nor_retbytes
};
static const struct drbg_kat kat188_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat188_nor_t
};

static const unsigned char kat189_nor_entropyin[] = {
    0x83, 0x35, 0xa4, 0x33, 0xa8, 0x0d, 0x5a, 0x3e, 0x8a, 0xb1, 0xf7, 0x38,
    0x55, 0xf9, 0x7a, 0x9b,
};
static const unsigned char kat189_nor_nonce[] = {
    0x02, 0x62, 0xb4, 0x7c, 0x2a, 0xb5, 0xfc, 0xba,
};
static const unsigned char kat189_nor_persstr[] = {0};
static const unsigned char kat189_nor_addin0[] = {0};
static const unsigned char kat189_nor_addin1[] = {0};
static const unsigned char kat189_nor_retbytes[] = {
    0x66, 0xa0, 0x3a, 0x0b, 0x3b, 0x2e, 0xf8, 0xd0, 0x4e, 0x01, 0xda, 0x6e,
    0x31, 0xbc, 0x90, 0x02, 0xeb, 0xac, 0xa9, 0x8d, 0x63, 0xfd, 0xde, 0xb0,
    0x7f, 0xa3, 0xe0, 0x65, 0x0a, 0x32, 0xe1, 0x42, 0xfc, 0xab, 0x99, 0x63,
    0xf0, 0x6b, 0xa8, 0x8c, 0xd5, 0x8b, 0xe7, 0x5a, 0x99, 0xdf, 0xa1, 0x3b,
    0x4f, 0x8f, 0xea, 0xe6, 0xf8, 0x4c, 0xe2, 0xde, 0x4d, 0x20, 0x19, 0x92,
    0x23, 0x1f, 0xf3, 0xb8,
};
static const struct drbg_kat_no_reseed kat189_nor_t = {
    8, kat189_nor_entropyin, kat189_nor_nonce, kat189_nor_persstr,
    kat189_nor_addin0, kat189_nor_addin1, kat189_nor_retbytes
};
static const struct drbg_kat kat189_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat189_nor_t
};

static const unsigned char kat190_nor_entropyin[] = {
    0x37, 0xfa, 0xeb, 0x0b, 0x3f, 0x85, 0x37, 0x2c, 0xcb, 0x44, 0x22, 0xe3,
    0x66, 0x90, 0xe9, 0x6c,
};
static const unsigned char kat190_nor_nonce[] = {
    0xda, 0x16, 0x5c, 0x92, 0xd5, 0xfa, 0xaa, 0x3a,
};
static const unsigned char kat190_nor_persstr[] = {0};
static const unsigned char kat190_nor_addin0[] = {0};
static const unsigned char kat190_nor_addin1[] = {0};
static const unsigned char kat190_nor_retbytes[] = {
    0x47, 0x75, 0x15, 0x90, 0x25, 0xd5, 0xb8, 0x82, 0xf3, 0x7c, 0x7e, 0xdb,
    0x8d, 0x89, 0xb3, 0x8a, 0xcb, 0x14, 0xa0, 0x28, 0x39, 0x80, 0x2c, 0x37,
    0x7a, 0xc4, 0x82, 0xf8, 0xc6, 0xb5, 0x9a, 0xd6, 0x9c, 0x7d, 0x67, 0x55,
    0x6b, 0xc4, 0x1c, 0xc3, 0x22, 0xd9, 0xcd, 0x75, 0x03, 0x2a, 0xf5, 0x27,
    0x15, 0xb3, 0x60, 0x2f, 0x24, 0xc2, 0xb0, 0x99, 0x84, 0x0e, 0xab, 0x00,
    0xb7, 0x42, 0x93, 0xad,
};
static const struct drbg_kat_no_reseed kat190_nor_t = {
    9, kat190_nor_entropyin, kat190_nor_nonce, kat190_nor_persstr,
    kat190_nor_addin0, kat190_nor_addin1, kat190_nor_retbytes
};
static const struct drbg_kat kat190_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat190_nor_t
};

static const unsigned char kat191_nor_entropyin[] = {
    0xb6, 0x83, 0x0f, 0x64, 0xe4, 0x73, 0x88, 0xa9, 0xc3, 0xc6, 0x4a, 0xa0,
    0xbe, 0x70, 0x84, 0x51,
};
static const unsigned char kat191_nor_nonce[] = {
    0x5c, 0xd4, 0x30, 0x34, 0xc9, 0x89, 0xda, 0x56,
};
static const unsigned char kat191_nor_persstr[] = {0};
static const unsigned char kat191_nor_addin0[] = {0};
static const unsigned char kat191_nor_addin1[] = {0};
static const unsigned char kat191_nor_retbytes[] = {
    0xe1, 0x2f, 0x1e, 0x9f, 0xd9, 0x0e, 0x4f, 0x16, 0x6a, 0x13, 0xa4, 0xd3,
    0x5f, 0x90, 0x91, 0xa9, 0x74, 0x44, 0x32, 0x91, 0xff, 0xae, 0x44, 0x02,
    0xcb, 0xcd, 0x9a, 0x24, 0x89, 0x3b, 0x77, 0x09, 0x6b, 0x0a, 0x00, 0xdb,
    0x20, 0xe4, 0x32, 0x88, 0x62, 0x61, 0x39, 0xf7, 0xc1, 0x76, 0x7f, 0x6e,
    0x7f, 0xfe, 0xe9, 0x02, 0x2d, 0xde, 0x41, 0xb4, 0x78, 0x48, 0x5e, 0xe9,
    0x12, 0x5f, 0x3e, 0xb8,
};
static const struct drbg_kat_no_reseed kat191_nor_t = {
    10, kat191_nor_entropyin, kat191_nor_nonce, kat191_nor_persstr,
    kat191_nor_addin0, kat191_nor_addin1, kat191_nor_retbytes
};
static const struct drbg_kat kat191_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat191_nor_t
};

static const unsigned char kat192_nor_entropyin[] = {
    0xf7, 0x1a, 0x01, 0xef, 0x00, 0xb3, 0xd6, 0x13, 0x46, 0x07, 0xdf, 0xc5,
    0x73, 0x25, 0xf6, 0x87,
};
static const unsigned char kat192_nor_nonce[] = {
    0x49, 0x76, 0x0a, 0xeb, 0xbc, 0x89, 0x61, 0xfe,
};
static const unsigned char kat192_nor_persstr[] = {0};
static const unsigned char kat192_nor_addin0[] = {0};
static const unsigned char kat192_nor_addin1[] = {0};
static const unsigned char kat192_nor_retbytes[] = {
    0x0f, 0x00, 0x8e, 0x60, 0xab, 0x48, 0x14, 0xcc, 0xd8, 0xd2, 0x89, 0x77,
    0x53, 0xf5, 0x8c, 0xa3, 0x58, 0xba, 0xcd, 0x3f, 0x2e, 0xfc, 0x22, 0xc0,
    0xec, 0x89, 0xa6, 0x5b, 0x0a, 0x91, 0x82, 0xf5, 0x55, 0xcc, 0x49, 0x7a,
    0xa5, 0x9a, 0xf9, 0x14, 0xbc, 0x9c, 0x65, 0xbe, 0x7c, 0x09, 0x21, 0x46,
    0xcb, 0x78, 0xfc, 0xe2, 0x40, 0xfc, 0x8f, 0xe1, 0x36, 0x72, 0x9e, 0xa7,
    0x77, 0x16, 0x44, 0x7f,
};
static const struct drbg_kat_no_reseed kat192_nor_t = {
    11, kat192_nor_entropyin, kat192_nor_nonce, kat192_nor_persstr,
    kat192_nor_addin0, kat192_nor_addin1, kat192_nor_retbytes
};
static const struct drbg_kat kat192_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat192_nor_t
};

static const unsigned char kat193_nor_entropyin[] = {
    0x79, 0x0e, 0xf8, 0xf0, 0xee, 0x6d, 0xf4, 0x80, 0xb2, 0xd4, 0x4d, 0xac,
    0x1c, 0x37, 0x61, 0x4c,
};
static const unsigned char kat193_nor_nonce[] = {
    0xd5, 0x52, 0x44, 0x4d, 0x7d, 0xc3, 0x5e, 0x5c,
};
static const unsigned char kat193_nor_persstr[] = {0};
static const unsigned char kat193_nor_addin0[] = {0};
static const unsigned char kat193_nor_addin1[] = {0};
static const unsigned char kat193_nor_retbytes[] = {
    0x2a, 0xf8, 0xe3, 0x09, 0x77, 0x55, 0x6f, 0xae, 0x7c, 0x18, 0xae, 0x12,
    0xba, 0x22, 0x01, 0xb4, 0x0d, 0xe8, 0xc0, 0x21, 0x66, 0xed, 0x94, 0xc1,
    0x41, 0x27, 0x20, 0x50, 0xdb, 0xea, 0x72, 0xf3, 0xb8, 0xf9, 0x91, 0x54,
    0x7f, 0xc0, 0xf5, 0x58, 0x36, 0x17, 0x12, 0x67, 0xd4, 0x2a, 0x53, 0xdc,
    0xfb, 0x37, 0xc1, 0x5a, 0x22, 0x76, 0x76, 0xbf, 0x21, 0x8a, 0x49, 0xd0,
    0xb7, 0x23, 0x68, 0x9c,
};
static const struct drbg_kat_no_reseed kat193_nor_t = {
    12, kat193_nor_entropyin, kat193_nor_nonce, kat193_nor_persstr,
    kat193_nor_addin0, kat193_nor_addin1, kat193_nor_retbytes
};
static const struct drbg_kat kat193_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat193_nor_t
};

static const unsigned char kat194_nor_entropyin[] = {
    0xbe, 0x19, 0x6c, 0x9c, 0xe3, 0x83, 0xc8, 0xb1, 0x01, 0xcd, 0x27, 0xa1,
    0x12, 0x91, 0x9f, 0xbc,
};
static const unsigned char kat194_nor_nonce[] = {
    0x33, 0x2e, 0xbe, 0x3d, 0x47, 0x3a, 0x3a, 0x35,
};
static const unsigned char kat194_nor_persstr[] = {0};
static const unsigned char kat194_nor_addin0[] = {0};
static const unsigned char kat194_nor_addin1[] = {0};
static const unsigned char kat194_nor_retbytes[] = {
    0x01, 0x6b, 0x05, 0xd5, 0x57, 0x77, 0x7b, 0x36, 0xa1, 0x92, 0x04, 0x0c,
    0x58, 0x0a, 0x5c, 0x51, 0x6d, 0xda, 0x11, 0x55, 0x93, 0x4a, 0xfb, 0xd9,
    0x6c, 0xc8, 0x5b, 0x6e, 0x8b, 0x00, 0x83, 0xbb, 0xda, 0x27, 0x3e, 0xa6,
    0xdb, 0x52, 0xa0, 0x58, 0x9d, 0x9d, 0xe2, 0xe5, 0x56, 0x8f, 0xfe, 0xf9,
    0xdb, 0x39, 0x50, 0x09, 0x3c, 0x4d, 0xb2, 0xe8, 0xce, 0x9c, 0x25, 0x13,
    0x08, 0x46, 0x93, 0xe4,
};
static const struct drbg_kat_no_reseed kat194_nor_t = {
    13, kat194_nor_entropyin, kat194_nor_nonce, kat194_nor_persstr,
    kat194_nor_addin0, kat194_nor_addin1, kat194_nor_retbytes
};
static const struct drbg_kat kat194_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat194_nor_t
};

static const unsigned char kat195_nor_entropyin[] = {
    0x76, 0x39, 0x61, 0x6e, 0xe5, 0xcb, 0x26, 0x9b, 0x5f, 0x54, 0x0d, 0xe1,
    0x50, 0xb4, 0x65, 0x0e,
};
static const unsigned char kat195_nor_nonce[] = {
    0x3c, 0x93, 0x3a, 0xbe, 0xd3, 0x07, 0xf6, 0xe2,
};
static const unsigned char kat195_nor_persstr[] = {0};
static const unsigned char kat195_nor_addin0[] = {0};
static const unsigned char kat195_nor_addin1[] = {0};
static const unsigned char kat195_nor_retbytes[] = {
    0x06, 0x33, 0xc9, 0xaf, 0x4d, 0x40, 0xf3, 0xc2, 0x1a, 0xd4, 0xe0, 0x78,
    0xa0, 0xc8, 0x4e, 0xe0, 0x3a, 0xd7, 0xad, 0x41, 0x0a, 0x0f, 0x7c, 0xf3,
    0xdf, 0xdf, 0xf5, 0x92, 0x0d, 0xcd, 0x39, 0x87, 0xfc, 0xec, 0xed, 0x11,
    0xa2, 0xb3, 0x8b, 0x15, 0x53, 0x5e, 0x44, 0xb5, 0x53, 0x77, 0xa8, 0xf2,
    0x0f, 0x4f, 0xe0, 0x51, 0x87, 0xf9, 0x76, 0xa2, 0x76, 0xe6, 0x4b, 0x81,
    0x3e, 0x55, 0xa8, 0xb9,
};
static const struct drbg_kat_no_reseed kat195_nor_t = {
    14, kat195_nor_entropyin, kat195_nor_nonce, kat195_nor_persstr,
    kat195_nor_addin0, kat195_nor_addin1, kat195_nor_retbytes
};
static const struct drbg_kat kat195_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat195_nor_t
};

static const unsigned char kat196_nor_entropyin[] = {
    0x8b, 0x80, 0x93, 0x6e, 0x69, 0xc6, 0x7e, 0xdb, 0x77, 0x1c, 0x28, 0xf9,
    0xb9, 0x45, 0x21, 0x24,
};
static const unsigned char kat196_nor_nonce[] = {
    0x7e, 0xe2, 0x61, 0x4e, 0xad, 0x3c, 0x12, 0x8e,
};
static const unsigned char kat196_nor_persstr[] = {0};
static const unsigned char kat196_nor_addin0[] = {
    0xfc, 0x35, 0xcb, 0xa9, 0x7a, 0x1e, 0x21, 0x1b, 0xc4, 0x20, 0xe8, 0xaf,
    0x53, 0xf8, 0xe1, 0x3c,
};
static const unsigned char kat196_nor_addin1[] = {
    0xfb, 0xa4, 0x38, 0xaa, 0xa7, 0x5a, 0x3c, 0xd4, 0xcd, 0x0c, 0xce, 0x39,
    0x9b, 0xfe, 0xc7, 0x4a,
};
static const unsigned char kat196_nor_retbytes[] = {
    0x67, 0x21, 0xcc, 0x1a, 0xda, 0x5e, 0xbc, 0x17, 0x13, 0xf7, 0x4c, 0x75,
    0x90, 0x00, 0x76, 0x56, 0x52, 0xee, 0xb5, 0xf3, 0xf9, 0xc2, 0x4f, 0xb9,
    0x34, 0x1b, 0x36, 0xa3, 0x69, 0xce, 0xc1, 0xd2, 0x7e, 0xa8, 0x0d, 0x6b,
    0x73, 0xb5, 0x60, 0x47, 0xaf, 0x07, 0x13, 0x8c, 0x5a, 0x43, 0xc9, 0x9a,
    0x87, 0x75, 0x31, 0x15, 0xc4, 0x71, 0xb8, 0x58, 0x7e, 0xa6, 0x5f, 0xa2,
    0x06, 0x5e, 0x3c, 0xe0,
};
static const struct drbg_kat_no_reseed kat196_nor_t = {
    0, kat196_nor_entropyin, kat196_nor_nonce, kat196_nor_persstr,
    kat196_nor_addin0, kat196_nor_addin1, kat196_nor_retbytes
};
static const struct drbg_kat kat196_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat196_nor_t
};

static const unsigned char kat197_nor_entropyin[] = {
    0x89, 0xce, 0x6c, 0x76, 0xbc, 0x41, 0xaa, 0x32, 0xa9, 0xc8, 0xe3, 0xe3,
    0x7d, 0x62, 0x02, 0xef,
};
static const unsigned char kat197_nor_nonce[] = {
    0xa3, 0x47, 0xf6, 0xcb, 0x64, 0xed, 0x19, 0xdd,
};
static const unsigned char kat197_nor_persstr[] = {0};
static const unsigned char kat197_nor_addin0[] = {
    0xc5, 0x03, 0xb1, 0x22, 0x95, 0x7d, 0xce, 0xc8, 0xd9, 0xeb, 0x9c, 0xc9,
    0x94, 0xb8, 0xd1, 0x22,
};
static const unsigned char kat197_nor_addin1[] = {
    0x3b, 0xb2, 0xf1, 0x19, 0x7a, 0x99, 0x58, 0x8d, 0xf3, 0x57, 0xc0, 0xd1,
    0x98, 0x6a, 0xc6, 0xed,
};
static const unsigned char kat197_nor_retbytes[] = {
    0xf0, 0xf4, 0x25, 0xb1, 0x9e, 0xb7, 0x5a, 0xea, 0x68, 0x99, 0xfa, 0xb7,
    0x61, 0x2c, 0x98, 0xa0, 0xb5, 0xc5, 0xc5, 0xa3, 0xc8, 0x61, 0x07, 0xa8,
    0xc2, 0x01, 0x62, 0x3f, 0x75, 0x99, 0x31, 0x90, 0x9a, 0xfd, 0x63, 0x41,
    0x9d, 0x1c, 0xce, 0xb8, 0x6b, 0xd4, 0xf1, 0x6e, 0x94, 0x8a, 0xef, 0x08,
    0x47, 0x61, 0x70, 0x75, 0x7b, 0xca, 0xf7, 0x98, 0x84, 0xf9, 0xc3, 0x6b,
    0xc7, 0x7e, 0x9f, 0xf9,
};
static const struct drbg_kat_no_reseed kat197_nor_t = {
    1, kat197_nor_entropyin, kat197_nor_nonce, kat197_nor_persstr,
    kat197_nor_addin0, kat197_nor_addin1, kat197_nor_retbytes
};
static const struct drbg_kat kat197_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat197_nor_t
};

static const unsigned char kat198_nor_entropyin[] = {
    0xbc, 0x51, 0x67, 0x8d, 0x1f, 0x31, 0x28, 0x65, 0x7b, 0x38, 0x84, 0x9a,
    0xe9, 0xc4, 0xbb, 0x1e,
};
static const unsigned char kat198_nor_nonce[] = {
    0xdd, 0x29, 0xd9, 0xd8, 0x3f, 0xdb, 0x5e, 0x09,
};
static const unsigned char kat198_nor_persstr[] = {0};
static const unsigned char kat198_nor_addin0[] = {
    0xd4, 0xcd, 0x00, 0x67, 0x5c, 0xfe, 0x22, 0x7d, 0xe0, 0xcb, 0xab, 0x65,
    0x11, 0x35, 0x22, 0x89,
};
static const unsigned char kat198_nor_addin1[] = {
    0x55, 0xbb, 0x93, 0x62, 0x80, 0xae, 0x46, 0xb2, 0x6d, 0x66, 0xb2, 0xf3,
    0xfe, 0xb2, 0x68, 0xd8,
};
static const unsigned char kat198_nor_retbytes[] = {
    0xe5, 0x81, 0x38, 0xb6, 0xb0, 0x23, 0x2e, 0x4d, 0x6d, 0x9e, 0xa8, 0xb6,
    0x5b, 0xd7, 0x69, 0x65, 0xcd, 0xbc, 0x58, 0x4c, 0xca, 0x2e, 0x83, 0xd4,
    0xea, 0xd9, 0x36, 0xbf, 0xb9, 0xc7, 0x9d, 0x77, 0xde, 0xc9, 0x42, 0x4f,
    0x2c, 0xf8, 0x95, 0x01, 0x1e, 0xf1, 0x03, 0x3c, 0x61, 0x9e, 0x28, 0xd1,
    0xdb, 0xa5, 0xdd, 0x3c, 0xed, 0x04, 0x42, 0x9c, 0x1b, 0xd3, 0xc0, 0x3e,
    0x5d, 0x13, 0xe2, 0xb8,
};
static const struct drbg_kat_no_reseed kat198_nor_t = {
    2, kat198_nor_entropyin, kat198_nor_nonce, kat198_nor_persstr,
    kat198_nor_addin0, kat198_nor_addin1, kat198_nor_retbytes
};
static const struct drbg_kat kat198_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat198_nor_t
};

static const unsigned char kat199_nor_entropyin[] = {
    0x06, 0x5a, 0x5d, 0x4f, 0xdb, 0x34, 0xb0, 0x8f, 0xd8, 0xef, 0xc8, 0x1f,
    0x99, 0x62, 0x69, 0xab,
};
static const unsigned char kat199_nor_nonce[] = {
    0xd5, 0xa6, 0x74, 0x7c, 0x7f, 0x8e, 0xef, 0x46,
};
static const unsigned char kat199_nor_persstr[] = {0};
static const unsigned char kat199_nor_addin0[] = {
    0x72, 0xeb, 0xd3, 0x51, 0xed, 0xc0, 0x00, 0xd8, 0x36, 0x32, 0x58, 0xa6,
    0x9f, 0x2b, 0x98, 0xcc,
};
static const unsigned char kat199_nor_addin1[] = {
    0xe5, 0xc8, 0x61, 0x21, 0x32, 0x96, 0xea, 0x2a, 0xd5, 0x09, 0x8f, 0x9b,
    0x4c, 0x50, 0x64, 0x7d,
};
static const unsigned char kat199_nor_retbytes[] = {
    0xa8, 0xea, 0x47, 0x0f, 0x17, 0xbe, 0xe5, 0x02, 0x1d, 0x49, 0xc8, 0x4b,
    0x72, 0x1e, 0x0b, 0x67, 0x47, 0x0d, 0x3b, 0xc4, 0xf9, 0x8f, 0xac, 0x68,
    0xc6, 0x74, 0x42, 0x97, 0x9d, 0x4f, 0xbc, 0xcb, 0xbc, 0xd0, 0x47, 0x7c,
    0x1e, 0x74, 0x93, 0x12, 0x6c, 0xbe, 0x16, 0x9b, 0x84, 0x44, 0x6f, 0x5f,
    0xcf, 0x66, 0x67, 0x86, 0x77, 0x5d, 0x19, 0xf3, 0x5c, 0x61, 0x52, 0x52,
    0x9a, 0x41, 0xde, 0x6d,
};
static const struct drbg_kat_no_reseed kat199_nor_t = {
    3, kat199_nor_entropyin, kat199_nor_nonce, kat199_nor_persstr,
    kat199_nor_addin0, kat199_nor_addin1, kat199_nor_retbytes
};
static const struct drbg_kat kat199_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat199_nor_t
};

static const unsigned char kat200_nor_entropyin[] = {
    0x12, 0x48, 0xed, 0x4b, 0xf6, 0x06, 0xcc, 0x65, 0x8b, 0x64, 0x1b, 0x79,
    0x58, 0x59, 0x7f, 0x5a,
};
static const unsigned char kat200_nor_nonce[] = {
    0x16, 0xc8, 0x8d, 0xea, 0x4e, 0x75, 0x3d, 0x50,
};
static const unsigned char kat200_nor_persstr[] = {0};
static const unsigned char kat200_nor_addin0[] = {
    0xfc, 0x67, 0xe7, 0x63, 0x41, 0x32, 0x02, 0xe0, 0x4c, 0x55, 0x69, 0x7e,
    0xb7, 0x54, 0x8c, 0x56,
};
static const unsigned char kat200_nor_addin1[] = {
    0x21, 0xe0, 0xe3, 0x86, 0x65, 0xec, 0xf6, 0x0f, 0xa9, 0x73, 0x0b, 0xdf,
    0x7e, 0x22, 0xcc, 0x7f,
};
static const unsigned char kat200_nor_retbytes[] = {
    0x63, 0xd9, 0xb5, 0x0d, 0xd6, 0x9e, 0xa9, 0xda, 0xc7, 0x5f, 0x48, 0xae,
    0x01, 0x2f, 0xce, 0xc2, 0x7c, 0x2e, 0x1d, 0xcf, 0xc2, 0xac, 0x2e, 0x59,
    0x24, 0x4a, 0xf4, 0xa6, 0xce, 0x9a, 0x73, 0xc4, 0xd8, 0xbd, 0xe6, 0x57,
    0x0d, 0x85, 0xee, 0x15, 0xb0, 0x8a, 0x48, 0xef, 0x47, 0x3d, 0x12, 0xdb,
    0x87, 0x1f, 0x6d, 0xf1, 0xd8, 0x14, 0x41, 0x61, 0x3d, 0xcc, 0x1e, 0xfb,
    0x20, 0x18, 0xf0, 0xc3,
};
static const struct drbg_kat_no_reseed kat200_nor_t = {
    4, kat200_nor_entropyin, kat200_nor_nonce, kat200_nor_persstr,
    kat200_nor_addin0, kat200_nor_addin1, kat200_nor_retbytes
};
static const struct drbg_kat kat200_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat200_nor_t
};

static const unsigned char kat201_nor_entropyin[] = {
    0x6f, 0xa6, 0x29, 0xd0, 0x3c, 0xb4, 0xeb, 0x53, 0x4e, 0xfa, 0x03, 0x33,
    0x4d, 0x3b, 0x1d, 0x02,
};
static const unsigned char kat201_nor_nonce[] = {
    0xd9, 0x52, 0x15, 0x3e, 0x79, 0xdf, 0x53, 0x8f,
};
static const unsigned char kat201_nor_persstr[] = {0};
static const unsigned char kat201_nor_addin0[] = {
    0x2e, 0x5c, 0x55, 0x45, 0x78, 0xa0, 0x69, 0xf5, 0xe4, 0x95, 0x9d, 0xcb,
    0x35, 0x1a, 0x29, 0x4d,
};
static const unsigned char kat201_nor_addin1[] = {
    0x2d, 0x26, 0xae, 0xaa, 0xd9, 0xcb, 0xf2, 0x53, 0xad, 0xd8, 0x68, 0x4d,
    0x29, 0xb1, 0xa6, 0x33,
};
static const unsigned char kat201_nor_retbytes[] = {
    0xf5, 0x82, 0x46, 0x31, 0x32, 0x84, 0x99, 0x02, 0x98, 0x86, 0x4a, 0x89,
    0xbf, 0xe7, 0xbe, 0x19, 0x70, 0xc0, 0x33, 0xdc, 0x16, 0x64, 0xad, 0x7f,
    0x5c, 0xd9, 0xb8, 0x12, 0xf5, 0xb7, 0xe9, 0x0f, 0x69, 0xa4, 0x9c, 0xdf,
    0xbe, 0xe0, 0xe2, 0x79, 0xf7, 0xdf, 0x1a, 0x8e, 0xd7, 0x9e, 0xca, 0x6e,
    0x68, 0x0d, 0x74, 0x0c, 0x0f, 0x2a, 0xe1, 0x2f, 0x87, 0x79, 0x8a, 0xcc,
    0x6d, 0x73, 0xd4, 0x2f,
};
static const struct drbg_kat_no_reseed kat201_nor_t = {
    5, kat201_nor_entropyin, kat201_nor_nonce, kat201_nor_persstr,
    kat201_nor_addin0, kat201_nor_addin1, kat201_nor_retbytes
};
static const struct drbg_kat kat201_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat201_nor_t
};

static const unsigned char kat202_nor_entropyin[] = {
    0xc5, 0x8e, 0xeb, 0x2e, 0x57, 0xf2, 0x95, 0xd7, 0xaf, 0x1a, 0x37, 0x67,
    0x93, 0x5b, 0x85, 0xeb,
};
static const unsigned char kat202_nor_nonce[] = {
    0x98, 0x61, 0x2b, 0x6c, 0xa0, 0x2f, 0x60, 0xe4,
};
static const unsigned char kat202_nor_persstr[] = {0};
static const unsigned char kat202_nor_addin0[] = {
    0x10, 0xe8, 0x9c, 0x27, 0x1e, 0x2b, 0x28, 0x3d, 0x69, 0xb0, 0x4a, 0xbf,
    0x6c, 0x54, 0xb1, 0xe1,
};
static const unsigned char kat202_nor_addin1[] = {
    0x20, 0x79, 0x50, 0xc2, 0x8b, 0x26, 0xec, 0x16, 0xc4, 0xf2, 0x81, 0x21,
    0x60, 0x7f, 0x4a, 0x5a,
};
static const unsigned char kat202_nor_retbytes[] = {
    0x28, 0xef, 0xf6, 0x03, 0xff, 0x0d, 0xc8, 0x39, 0xf8, 0x8f, 0x84, 0x47,
    0x33, 0xb2, 0x75, 0xbc, 0x6f, 0xda, 0x6d, 0x45, 0x46, 0xee, 0x26, 0x03,
    0x4f, 0x51, 0xf8, 0x4c, 0xe4, 0x30, 0xb0, 0x73, 0xc3, 0xda, 0x8a, 0xbf,
    0xe8, 0x2b, 0x53, 0x13, 0xba, 0x2e, 0x0f, 0x60, 0x04, 0x5b, 0x80, 0x96,
    0xb9, 0xcf, 0x31, 0x42, 0x48, 0x63, 0xe0, 0x6f, 0x72, 0xef, 0xf1, 0xba,
    0xa4, 0xb2, 0x52, 0x70,
};
static const struct drbg_kat_no_reseed kat202_nor_t = {
    6, kat202_nor_entropyin, kat202_nor_nonce, kat202_nor_persstr,
    kat202_nor_addin0, kat202_nor_addin1, kat202_nor_retbytes
};
static const struct drbg_kat kat202_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat202_nor_t
};

static const unsigned char kat203_nor_entropyin[] = {
    0x8a, 0x34, 0x5b, 0x5a, 0x5e, 0xb7, 0xd7, 0xaa, 0x70, 0x0b, 0xff, 0x26,
    0x61, 0x25, 0xed, 0xe1,
};
static const unsigned char kat203_nor_nonce[] = {
    0x00, 0xb2, 0x96, 0x37, 0x27, 0x8d, 0xa4, 0x57,
};
static const unsigned char kat203_nor_persstr[] = {0};
static const unsigned char kat203_nor_addin0[] = {
    0x44, 0x53, 0x4b, 0x92, 0xaf, 0x03, 0x81, 0x71, 0xb6, 0x7b, 0x70, 0xf4,
    0xe1, 0xd5, 0xe7, 0xf5,
};
static const unsigned char kat203_nor_addin1[] = {
    0x53, 0x14, 0x76, 0x60, 0xc0, 0xfd, 0xe5, 0xf9, 0x89, 0x34, 0x74, 0xf0,
    0x38, 0x84, 0xab, 0x14,
};
static const unsigned char kat203_nor_retbytes[] = {
    0xa6, 0xa7, 0xa5, 0x6f, 0xc3, 0x2c, 0x3d, 0x3c, 0x33, 0x06, 0xdd, 0x65,
    0x10, 0x9f, 0xf8, 0xd6, 0x80, 0x31, 0x19, 0x3b, 0xbf, 0x5b, 0x38, 0x38,
    0x0e, 0x38, 0x25, 0xdb, 0x7b, 0xef, 0x72, 0x94, 0x05, 0x14, 0x16, 0x26,
    0x37, 0x95, 0xf3, 0x34, 0xd1, 0xf8, 0x70, 0x54, 0xe9, 0x7d, 0xbb, 0x52,
    0xdd, 0x24, 0x4d, 0x52, 0x7a, 0x6f, 0xfc, 0xe0, 0x86, 0xd1, 0xad, 0x17,
    0x7b, 0xa8, 0xfb, 0x81,
};
static const struct drbg_kat_no_reseed kat203_nor_t = {
    7, kat203_nor_entropyin, kat203_nor_nonce, kat203_nor_persstr,
    kat203_nor_addin0, kat203_nor_addin1, kat203_nor_retbytes
};
static const struct drbg_kat kat203_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat203_nor_t
};

static const unsigned char kat204_nor_entropyin[] = {
    0xfe, 0x50, 0x4e, 0x06, 0x89, 0x5d, 0x34, 0x08, 0x90, 0x50, 0x9f, 0x39,
    0x8f, 0xe7, 0x08, 0x32,
};
static const unsigned char kat204_nor_nonce[] = {
    0x3b, 0x3d, 0x94, 0xb6, 0x94, 0x8f, 0x74, 0x01,
};
static const unsigned char kat204_nor_persstr[] = {0};
static const unsigned char kat204_nor_addin0[] = {
    0x64, 0x20, 0x03, 0x18, 0x1f, 0x79, 0x1c, 0xb5, 0xba, 0xce, 0xf7, 0x23,
    0xc9, 0x6c, 0xe2, 0x62,
};
static const unsigned char kat204_nor_addin1[] = {
    0x34, 0x5e, 0x8d, 0x29, 0xd5, 0xef, 0xdd, 0xc8, 0xab, 0x37, 0x83, 0x8b,
    0x18, 0x91, 0xbc, 0xd0,
};
static const unsigned char kat204_nor_retbytes[] = {
    0x45, 0x30, 0x40, 0x99, 0x14, 0x5b, 0x8e, 0x3d, 0xa8, 0x00, 0xa9, 0x2e,
    0x0f, 0x59, 0x15, 0xc8, 0x98, 0x1f, 0xb7, 0x70, 0x6a, 0x1d, 0xf0, 0xa6,
    0xc0, 0xb2, 0xad, 0x54, 0x35, 0xda, 0x19, 0xce, 0xd3, 0xac, 0xdf, 0xd5,
    0x83, 0x27, 0x95, 0x01, 0x3a, 0xfa, 0x5f, 0xf2, 0x1f, 0xf3, 0x32, 0x6a,
    0xda, 0x7d, 0x57, 0x78, 0x17, 0xf6, 0xb9, 0x4e, 0xb3, 0x5f, 0x33, 0x62,
    0x3d, 0xb4, 0x71, 0x76,
};
static const struct drbg_kat_no_reseed kat204_nor_t = {
    8, kat204_nor_entropyin, kat204_nor_nonce, kat204_nor_persstr,
    kat204_nor_addin0, kat204_nor_addin1, kat204_nor_retbytes
};
static const struct drbg_kat kat204_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat204_nor_t
};

static const unsigned char kat205_nor_entropyin[] = {
    0xe0, 0x22, 0x7d, 0xe1, 0x4d, 0x5b, 0xc6, 0xad, 0xe2, 0xa0, 0x14, 0xee,
    0x43, 0x01, 0x51, 0xb4,
};
static const unsigned char kat205_nor_nonce[] = {
    0x15, 0x37, 0xf8, 0x56, 0x16, 0x75, 0x87, 0x53,
};
static const unsigned char kat205_nor_persstr[] = {0};
static const unsigned char kat205_nor_addin0[] = {
    0x87, 0xee, 0xfe, 0xf0, 0xe0, 0xcf, 0xd9, 0x0d, 0xd2, 0x6d, 0x48, 0x33,
    0x79, 0x78, 0x7c, 0x7f,
};
static const unsigned char kat205_nor_addin1[] = {
    0x74, 0xd2, 0xff, 0xf6, 0xf2, 0x9f, 0x89, 0x2e, 0x66, 0xa9, 0xf8, 0x0b,
    0x2a, 0x5d, 0x9d, 0x20,
};
static const unsigned char kat205_nor_retbytes[] = {
    0xdd, 0xe0, 0x86, 0xce, 0xc0, 0xad, 0xd3, 0x1f, 0xb9, 0x55, 0x3d, 0x56,
    0xcc, 0x2c, 0x6b, 0x7c, 0x02, 0x2d, 0x8c, 0x87, 0xff, 0x54, 0x99, 0xfa,
    0x78, 0xa6, 0x8e, 0xeb, 0x05, 0x1c, 0xdb, 0xf8, 0x99, 0x67, 0x3b, 0x4e,
    0xd7, 0x6f, 0xe8, 0x67, 0xb7, 0x9e, 0xcf, 0x6f, 0xc1, 0xd7, 0xc6, 0xf8,
    0xf5, 0x97, 0x01, 0x31, 0x39, 0xd5, 0x28, 0xd3, 0x48, 0xca, 0x3b, 0xee,
    0x96, 0x05, 0xdd, 0x80,
};
static const struct drbg_kat_no_reseed kat205_nor_t = {
    9, kat205_nor_entropyin, kat205_nor_nonce, kat205_nor_persstr,
    kat205_nor_addin0, kat205_nor_addin1, kat205_nor_retbytes
};
static const struct drbg_kat kat205_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat205_nor_t
};

static const unsigned char kat206_nor_entropyin[] = {
    0x44, 0xb7, 0x28, 0x2f, 0x95, 0x1c, 0x59, 0x1a, 0x7a, 0x3d, 0x9f, 0xf6,
    0x96, 0x0f, 0x20, 0xb9,
};
static const unsigned char kat206_nor_nonce[] = {
    0xb7, 0x7e, 0xbe, 0x26, 0x99, 0x0c, 0xbe, 0x8c,
};
static const unsigned char kat206_nor_persstr[] = {0};
static const unsigned char kat206_nor_addin0[] = {
    0x90, 0x00, 0x58, 0x7b, 0xb1, 0x8d, 0x20, 0x1e, 0xcf, 0xd5, 0x6f, 0x30,
    0xdb, 0xa4, 0x83, 0xd2,
};
static const unsigned char kat206_nor_addin1[] = {
    0x24, 0x55, 0x23, 0x06, 0x0b, 0x0a, 0xf3, 0xe5, 0x75, 0xb1, 0x48, 0x0a,
    0xa6, 0xd8, 0xa3, 0x3b,
};
static const unsigned char kat206_nor_retbytes[] = {
    0x9f, 0x25, 0x67, 0x4a, 0xc0, 0x85, 0xa7, 0xa4, 0x87, 0xe7, 0xd7, 0x20,
    0x84, 0xd5, 0xd3, 0xc0, 0xfb, 0xd7, 0xd4, 0x18, 0x70, 0xf0, 0xf7, 0x66,
    0xd6, 0x13, 0x01, 0x59, 0x96, 0x05, 0x2e, 0xbf, 0xc6, 0xf6, 0x2e, 0x4c,
    0xf3, 0x89, 0xaf, 0x85, 0xcf, 0x12, 0x5d, 0x9d, 0x99, 0xc6, 0x49, 0xd4,
    0xe8, 0x75, 0x50, 0x79, 0xa6, 0x81, 0x7a, 0x9e, 0x81, 0x44, 0x5d, 0x1b,
    0x99, 0x4d, 0x29, 0x61,
};
static const struct drbg_kat_no_reseed kat206_nor_t = {
    10, kat206_nor_entropyin, kat206_nor_nonce, kat206_nor_persstr,
    kat206_nor_addin0, kat206_nor_addin1, kat206_nor_retbytes
};
static const struct drbg_kat kat206_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat206_nor_t
};

static const unsigned char kat207_nor_entropyin[] = {
    0x74, 0x5c, 0xa1, 0x6e, 0xbc, 0x9e, 0x1a, 0x58, 0xda, 0x81, 0x02, 0x78,
    0x88, 0x5d, 0xff, 0x33,
};
static const unsigned char kat207_nor_nonce[] = {
    0x7e, 0x11, 0xf9, 0x49, 0x86, 0x2c, 0xbc, 0xc9,
};
static const unsigned char kat207_nor_persstr[] = {0};
static const unsigned char kat207_nor_addin0[] = {
    0x4b, 0x89, 0x8e, 0xbd, 0xd3, 0x21, 0x42, 0x62, 0x78, 0x15, 0xbe, 0x10,
    0x35, 0x43, 0x56, 0x96,
};
static const unsigned char kat207_nor_addin1[] = {
    0x33, 0x0b, 0x0c, 0xe2, 0x87, 0x84, 0x2d, 0x3d, 0x71, 0x9a, 0x0c, 0xe4,
    0x15, 0x36, 0x3b, 0x77,
};
static const unsigned char kat207_nor_retbytes[] = {
    0x9c, 0x6d, 0xd7, 0x63, 0x88, 0x80, 0x65, 0x54, 0x1b, 0x1a, 0x38, 0x0c,
    0xf4, 0xf4, 0x59, 0x83, 0x9f, 0xb4, 0xf1, 0xef, 0x4f, 0x78, 0x17, 0x28,
    0x6b, 0xe4, 0x50, 0x33, 0x95, 0x7f, 0x79, 0xd1, 0x42, 0x9c, 0xfe, 0xfe,
    0xd1, 0xab, 0xd6, 0x22, 0x8b, 0x26, 0x74, 0x7e, 0x69, 0xec, 0x9b, 0x1a,
    0x02, 0x7b, 0x3a, 0x0b, 0x9a, 0x28, 0x48, 0x7e, 0x95, 0x4d, 0xda, 0x06,
    0xb7, 0xee, 0xb4, 0x54,
};
static const struct drbg_kat_no_reseed kat207_nor_t = {
    11, kat207_nor_entropyin, kat207_nor_nonce, kat207_nor_persstr,
    kat207_nor_addin0, kat207_nor_addin1, kat207_nor_retbytes
};
static const struct drbg_kat kat207_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat207_nor_t
};

static const unsigned char kat208_nor_entropyin[] = {
    0x1e, 0x54, 0x6f, 0xee, 0xba, 0xf0, 0xfe, 0xba, 0x3a, 0x07, 0xc6, 0x35,
    0xe6, 0x23, 0x5b, 0xfb,
};
static const unsigned char kat208_nor_nonce[] = {
    0x17, 0x88, 0xd9, 0xee, 0xa4, 0x44, 0xad, 0x7b,
};
static const unsigned char kat208_nor_persstr[] = {0};
static const unsigned char kat208_nor_addin0[] = {
    0x51, 0x99, 0xd4, 0xaa, 0x59, 0x30, 0x22, 0x0c, 0x27, 0x34, 0x2e, 0x1c,
    0x0b, 0x99, 0x46, 0x4f,
};
static const unsigned char kat208_nor_addin1[] = {
    0x16, 0xdd, 0x52, 0x0f, 0x00, 0x93, 0x29, 0xed, 0x7b, 0xe1, 0x4d, 0x25,
    0xe5, 0xc7, 0x91, 0x12,
};
static const unsigned char kat208_nor_retbytes[] = {
    0x34, 0xfa, 0x78, 0x30, 0x6c, 0x9e, 0xfb, 0x91, 0x2e, 0xe7, 0x8f, 0xb9,
    0x8a, 0x03, 0x54, 0x6d, 0x16, 0xc1, 0xcc, 0x9b, 0xc3, 0xb6, 0x7a, 0x04,
    0xf2, 0xa8, 0xe6, 0xfb, 0x26, 0x24, 0x14, 0xbc, 0xf3, 0xcc, 0x51, 0xd3,
    0xcf, 0x3a, 0x1a, 0x30, 0x48, 0x09, 0xf1, 0x86, 0xbd, 0x74, 0xf2, 0x09,
    0xb2, 0x42, 0x86, 0xf7, 0x71, 0x76, 0x67, 0xcc, 0x56, 0x60, 0xd3, 0xec,
    0x73, 0x2f, 0xb0, 0xd3,
};
static const struct drbg_kat_no_reseed kat208_nor_t = {
    12, kat208_nor_entropyin, kat208_nor_nonce, kat208_nor_persstr,
    kat208_nor_addin0, kat208_nor_addin1, kat208_nor_retbytes
};
static const struct drbg_kat kat208_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat208_nor_t
};

static const unsigned char kat209_nor_entropyin[] = {
    0xde, 0x1f, 0xca, 0xee, 0x63, 0x29, 0xe9, 0xac, 0x06, 0xa6, 0xdc, 0x18,
    0xa1, 0x8d, 0x14, 0x7d,
};
static const unsigned char kat209_nor_nonce[] = {
    0xe1, 0x0e, 0xa5, 0x48, 0xcc, 0x37, 0x73, 0xab,
};
static const unsigned char kat209_nor_persstr[] = {0};
static const unsigned char kat209_nor_addin0[] = {
    0x5c, 0xdd, 0xd4, 0xfd, 0x07, 0xb6, 0xac, 0x3e, 0x96, 0xb1, 0x33, 0x39,
    0xa7, 0x5f, 0x56, 0xf9,
};
static const unsigned char kat209_nor_addin1[] = {
    0x45, 0x47, 0x14, 0xbe, 0xbf, 0x96, 0x03, 0xe1, 0xcf, 0xef, 0x7b, 0x80,
    0x38, 0x37, 0x48, 0x99,
};
static const unsigned char kat209_nor_retbytes[] = {
    0x5e, 0x34, 0x83, 0x12, 0x58, 0x35, 0x24, 0x30, 0x48, 0xa5, 0x10, 0x18,
    0x4f, 0x08, 0x64, 0x87, 0xe9, 0xb0, 0x2f, 0xc0, 0xca, 0x60, 0xfb, 0x46,
    0x41, 0x6f, 0xb0, 0xcc, 0xc1, 0x3e, 0x45, 0x08, 0x1d, 0xa5, 0x96, 0x91,
    0xe3, 0x26, 0x78, 0xfc, 0xb6, 0xaa, 0xbe, 0xf5, 0x85, 0xfb, 0x49, 0x2b,
    0x19, 0x4e, 0x06, 0x17, 0x1a, 0x8d, 0x17, 0xaf, 0x85, 0x56, 0xa3, 0x6b,
    0xf4, 0x93, 0x1f, 0xf5,
};
static const struct drbg_kat_no_reseed kat209_nor_t = {
    13, kat209_nor_entropyin, kat209_nor_nonce, kat209_nor_persstr,
    kat209_nor_addin0, kat209_nor_addin1, kat209_nor_retbytes
};
static const struct drbg_kat kat209_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat209_nor_t
};

static const unsigned char kat210_nor_entropyin[] = {
    0x8b, 0x52, 0x95, 0x07, 0xbc, 0x2e, 0x8f, 0x71, 0x19, 0x34, 0xad, 0xc8,
    0x8c, 0x65, 0x30, 0xa3,
};
static const unsigned char kat210_nor_nonce[] = {
    0x72, 0x15, 0x95, 0x6b, 0x7c, 0x57, 0x4e, 0x71,
};
static const unsigned char kat210_nor_persstr[] = {0};
static const unsigned char kat210_nor_addin0[] = {
    0x1a, 0xdb, 0x1e, 0xf6, 0x33, 0x47, 0xb2, 0x1e, 0x33, 0x51, 0x8b, 0x96,
    0x2f, 0x6d, 0xb8, 0x98,
};
static const unsigned char kat210_nor_addin1[] = {
    0xd5, 0xc6, 0x9f, 0x42, 0xf3, 0x6e, 0xa2, 0x24, 0x98, 0x42, 0xb6, 0xfc,
    0x26, 0xac, 0x54, 0xfe,
};
static const unsigned char kat210_nor_retbytes[] = {
    0xc3, 0x74, 0xbd, 0xd6, 0x2c, 0xb1, 0xe4, 0x2e, 0x64, 0x84, 0x03, 0x84,
    0x3b, 0x8c, 0x06, 0xc0, 0xb3, 0x05, 0xf9, 0x45, 0x68, 0x5f, 0x72, 0xd1,
    0xbc, 0x2e, 0x42, 0x8c, 0x19, 0xdf, 0x45, 0x20, 0x18, 0xdd, 0xea, 0x81,
    0xeb, 0xe5, 0xb1, 0x5f, 0xad, 0x3b, 0xe4, 0xeb, 0x17, 0xfa, 0x2d, 0x2c,
    0xf5, 0x78, 0x36, 0xdd, 0x08, 0x09, 0x70, 0xf7, 0x16, 0xc0, 0xb2, 0x2f,
    0x72, 0x63, 0x41, 0x4a,
};
static const struct drbg_kat_no_reseed kat210_nor_t = {
    14, kat210_nor_entropyin, kat210_nor_nonce, kat210_nor_persstr,
    kat210_nor_addin0, kat210_nor_addin1, kat210_nor_retbytes
};
static const struct drbg_kat kat210_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 0, 16, 64, &kat210_nor_t
};

static const unsigned char kat211_nor_entropyin[] = {
    0xf4, 0x1f, 0x46, 0x6b, 0x32, 0x19, 0xbe, 0x21, 0x59, 0x77, 0x63, 0xfa,
    0x7b, 0x76, 0xfb, 0x40,
};
static const unsigned char kat211_nor_nonce[] = {
    0xcd, 0x93, 0xfe, 0xb9, 0x96, 0x2e, 0x81, 0xac,
};
static const unsigned char kat211_nor_persstr[] = {
    0xb5, 0x8f, 0x86, 0x9a, 0xd0, 0xaa, 0x98, 0x08, 0xf6, 0x64, 0x61, 0x37,
    0x43, 0x1d, 0x43, 0x0c,
};
static const unsigned char kat211_nor_addin0[] = {0};
static const unsigned char kat211_nor_addin1[] = {0};
static const unsigned char kat211_nor_retbytes[] = {
    0x2f, 0xb6, 0xd7, 0xec, 0xa3, 0x92, 0x67, 0x4f, 0xc7, 0x22, 0xa6, 0x19,
    0x20, 0x2e, 0x81, 0x9d, 0x0d, 0xa9, 0xd1, 0x1b, 0xc6, 0x7d, 0xb1, 0x0b,
    0xe4, 0xc1, 0x3c, 0xb9, 0x64, 0xe3, 0x0a, 0xda, 0x96, 0xdc, 0xcf, 0x0c,
    0x92, 0x2b, 0x71, 0x0a, 0xc0, 0x0d, 0xed, 0x54, 0x57, 0xfa, 0x97, 0x1b,
    0xb1, 0xc6, 0x61, 0xa0, 0x9a, 0xfa, 0x72, 0x0a, 0x58, 0x64, 0x34, 0x4b,
    0xf7, 0x7a, 0x36, 0xae,
};
static const struct drbg_kat_no_reseed kat211_nor_t = {
    0, kat211_nor_entropyin, kat211_nor_nonce, kat211_nor_persstr,
    kat211_nor_addin0, kat211_nor_addin1, kat211_nor_retbytes
};
static const struct drbg_kat kat211_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat211_nor_t
};

static const unsigned char kat212_nor_entropyin[] = {
    0x9d, 0x1b, 0x88, 0x34, 0x83, 0x2f, 0xfa, 0x13, 0x83, 0x2e, 0xb0, 0x86,
    0x04, 0x7b, 0xf3, 0xb1,
};
static const unsigned char kat212_nor_nonce[] = {
    0xd0, 0xf1, 0x5e, 0xfe, 0x86, 0x47, 0x7f, 0x75,
};
static const unsigned char kat212_nor_persstr[] = {
    0x73, 0xc9, 0x37, 0x34, 0xf6, 0xea, 0x39, 0xae, 0x04, 0xe6, 0xa4, 0xb4,
    0x97, 0x66, 0xb8, 0x20,
};
static const unsigned char kat212_nor_addin0[] = {0};
static const unsigned char kat212_nor_addin1[] = {0};
static const unsigned char kat212_nor_retbytes[] = {
    0x9f, 0xb6, 0x7d, 0x35, 0x37, 0x89, 0x40, 0xa5, 0xd7, 0x6b, 0x96, 0x3a,
    0xce, 0x4f, 0x81, 0x58, 0xe9, 0x3f, 0xe0, 0xca, 0x06, 0x4f, 0x96, 0x56,
    0xd4, 0x6d, 0xf1, 0xc1, 0x0d, 0x02, 0x5f, 0x48, 0xb3, 0x35, 0x69, 0xda,
    0x07, 0xc7, 0x7e, 0xc5, 0x12, 0x23, 0x6d, 0x08, 0xd2, 0x69, 0x97, 0xd6,
    0xb9, 0xbb, 0x69, 0x15, 0xdf, 0x63, 0x9e, 0xa8, 0x9d, 0xa9, 0x57, 0xe6,
    0x6f, 0xc2, 0x90, 0x03,
};
static const struct drbg_kat_no_reseed kat212_nor_t = {
    1, kat212_nor_entropyin, kat212_nor_nonce, kat212_nor_persstr,
    kat212_nor_addin0, kat212_nor_addin1, kat212_nor_retbytes
};
static const struct drbg_kat kat212_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat212_nor_t
};

static const unsigned char kat213_nor_entropyin[] = {
    0x29, 0x89, 0x86, 0x2a, 0x79, 0xe2, 0x55, 0x19, 0x5a, 0x24, 0x82, 0x8f,
    0xd3, 0x00, 0xeb, 0x34,
};
static const unsigned char kat213_nor_nonce[] = {
    0x30, 0xff, 0xda, 0xcb, 0x3a, 0xc7, 0xb2, 0x7b,
};
static const unsigned char kat213_nor_persstr[] = {
    0x71, 0x9b, 0x89, 0x9c, 0x9e, 0x4a, 0x5d, 0xb9, 0xe7, 0x1d, 0xfd, 0xa4,
    0x8f, 0xa6, 0x58, 0xcd,
};
static const unsigned char kat213_nor_addin0[] = {0};
static const unsigned char kat213_nor_addin1[] = {0};
static const unsigned char kat213_nor_retbytes[] = {
    0xb4, 0xf2, 0x00, 0x60, 0xea, 0x30, 0x01, 0xef, 0xdb, 0xd5, 0xcc, 0x89,
    0x83, 0x8e, 0x0a, 0x08, 0xc0, 0x9f, 0x7a, 0x6f, 0xe5, 0xbc, 0x02, 0x3c,
    0x33, 0xd1, 0x15, 0xfe, 0xdd, 0x6a, 0xe1, 0x51, 0x30, 0x74, 0x22, 0xf9,
    0x97, 0xd3, 0x2b, 0x3c, 0xea, 0xb8, 0x79, 0x95, 0x86, 0x23, 0x68, 0xc4,
    0xc3, 0xaf, 0x7a, 0xc4, 0x81, 0x58, 0x74, 0xc0, 0x08, 0x4e, 0xa1, 0xdc,
    0xec, 0x50, 0x58, 0xba,
};
static const struct drbg_kat_no_reseed kat213_nor_t = {
    2, kat213_nor_entropyin, kat213_nor_nonce, kat213_nor_persstr,
    kat213_nor_addin0, kat213_nor_addin1, kat213_nor_retbytes
};
static const struct drbg_kat kat213_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat213_nor_t
};

static const unsigned char kat214_nor_entropyin[] = {
    0x74, 0x82, 0xb2, 0xb0, 0x2a, 0x74, 0x46, 0xde, 0x58, 0x9d, 0x7e, 0x60,
    0xcf, 0x01, 0x29, 0x69,
};
static const unsigned char kat214_nor_nonce[] = {
    0x1c, 0x4b, 0xda, 0x6a, 0x57, 0xf4, 0x1f, 0xfb,
};
static const unsigned char kat214_nor_persstr[] = {
    0x52, 0x9e, 0x4e, 0x80, 0xf5, 0x01, 0xe7, 0x3e, 0xc8, 0x6e, 0xaa, 0x25,
    0x96, 0x28, 0xa1, 0x97,
};
static const unsigned char kat214_nor_addin0[] = {0};
static const unsigned char kat214_nor_addin1[] = {0};
static const unsigned char kat214_nor_retbytes[] = {
    0x74, 0x98, 0xf9, 0x17, 0x2a, 0xf7, 0xf5, 0xf2, 0x6d, 0x84, 0x77, 0x97,
    0x76, 0x8e, 0x45, 0x91, 0x70, 0xdd, 0x9e, 0xc7, 0xf4, 0x2a, 0x1f, 0xe9,
    0x79, 0xa2, 0xe4, 0xfa, 0x32, 0xa5, 0xe1, 0x24, 0xc5, 0xcb, 0x1a, 0xd4,
    0xc3, 0x94, 0xa2, 0xc2, 0x09, 0x9e, 0x8f, 0x94, 0x2e, 0xfb, 0xe5, 0x9a,
    0xf0, 0x97, 0x5b, 0x56, 0xa9, 0xaf, 0xa7, 0x74, 0x33, 0x16, 0x12, 0xad,
    0x88, 0x7b, 0x3f, 0x55,
};
static const struct drbg_kat_no_reseed kat214_nor_t = {
    3, kat214_nor_entropyin, kat214_nor_nonce, kat214_nor_persstr,
    kat214_nor_addin0, kat214_nor_addin1, kat214_nor_retbytes
};
static const struct drbg_kat kat214_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat214_nor_t
};

static const unsigned char kat215_nor_entropyin[] = {
    0x0a, 0x84, 0x05, 0x99, 0x1a, 0xeb, 0x64, 0xf3, 0xa8, 0x2d, 0x8b, 0xef,
    0x2b, 0x6c, 0x94, 0x22,
};
static const unsigned char kat215_nor_nonce[] = {
    0xa7, 0xa5, 0x8d, 0xa9, 0xb2, 0x16, 0xf7, 0xfb,
};
static const unsigned char kat215_nor_persstr[] = {
    0x68, 0xdf, 0x62, 0xfc, 0x01, 0xd3, 0xdb, 0xb0, 0x18, 0xc1, 0x63, 0xbe,
    0x34, 0x29, 0xf2, 0xaa,
};
static const unsigned char kat215_nor_addin0[] = {0};
static const unsigned char kat215_nor_addin1[] = {0};
static const unsigned char kat215_nor_retbytes[] = {
    0x7a, 0x9b, 0xa1, 0xe8, 0x25, 0x13, 0x3f, 0xf4, 0xc1, 0xd6, 0x46, 0xce,
    0x55, 0x77, 0xf3, 0x5a, 0x17, 0x84, 0xee, 0xc2, 0xc1, 0x97, 0x70, 0x90,
    0xb4, 0x8e, 0x30, 0xbd, 0x3b, 0x75, 0x06, 0xf4, 0x47, 0xee, 0x62, 0xd0,
    0x21, 0xca, 0xe1, 0x2a, 0xd2, 0x87, 0xb4, 0x17, 0xed, 0xdb, 0x9e, 0xc6,
    0x46, 0x0e, 0x3e, 0x28, 0x4a, 0xfa, 0x73, 0xb7, 0x39, 0x56, 0x4e, 0x40,
    0x73, 0xd0, 0x0e, 0x3c,
};
static const struct drbg_kat_no_reseed kat215_nor_t = {
    4, kat215_nor_entropyin, kat215_nor_nonce, kat215_nor_persstr,
    kat215_nor_addin0, kat215_nor_addin1, kat215_nor_retbytes
};
static const struct drbg_kat kat215_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat215_nor_t
};

static const unsigned char kat216_nor_entropyin[] = {
    0xc7, 0xe9, 0xf6, 0x88, 0x33, 0xb1, 0x8e, 0x03, 0x6a, 0xa1, 0xda, 0x02,
    0x5a, 0x35, 0x9e, 0xd7,
};
static const unsigned char kat216_nor_nonce[] = {
    0x99, 0xf0, 0xe4, 0x9c, 0xe8, 0x11, 0xee, 0x7e,
};
static const unsigned char kat216_nor_persstr[] = {
    0x8d, 0xe7, 0xc8, 0x6b, 0x8e, 0xc6, 0x1d, 0x6a, 0xbb, 0x52, 0xa7, 0x91,
    0x66, 0x71, 0xad, 0xb9,
};
static const unsigned char kat216_nor_addin0[] = {0};
static const unsigned char kat216_nor_addin1[] = {0};
static const unsigned char kat216_nor_retbytes[] = {
    0x14, 0x07, 0xb6, 0x81, 0x51, 0xfc, 0xb0, 0xf0, 0x8e, 0xba, 0xbc, 0x21,
    0xc6, 0xc1, 0x81, 0xac, 0x1d, 0xbf, 0x9c, 0x6f, 0xb1, 0xb2, 0xc1, 0x6e,
    0xaf, 0x1f, 0x8c, 0x49, 0x0d, 0x6f, 0x7d, 0x52, 0xd0, 0xf4, 0x21, 0x11,
    0x6a, 0x59, 0x98, 0x33, 0x0d, 0x81, 0x05, 0xf5, 0x02, 0x76, 0x17, 0xdc,
    0x94, 0xb1, 0x4c, 0x08, 0x3f, 0x49, 0xd1, 0x1c, 0x34, 0xf4, 0xf2, 0x63,
    0x02, 0x31, 0x66, 0x24,
};
static const struct drbg_kat_no_reseed kat216_nor_t = {
    5, kat216_nor_entropyin, kat216_nor_nonce, kat216_nor_persstr,
    kat216_nor_addin0, kat216_nor_addin1, kat216_nor_retbytes
};
static const struct drbg_kat kat216_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat216_nor_t
};

static const unsigned char kat217_nor_entropyin[] = {
    0xd6, 0x7e, 0xeb, 0x00, 0xa4, 0x29, 0x1a, 0x18, 0x47, 0x11, 0x84, 0xb4,
    0x31, 0x59, 0xd2, 0xbd,
};
static const unsigned char kat217_nor_nonce[] = {
    0xe4, 0xa3, 0x61, 0x49, 0x7e, 0xe1, 0x43, 0x8a,
};
static const unsigned char kat217_nor_persstr[] = {
    0xf0, 0x54, 0xad, 0xf2, 0xad, 0x38, 0x49, 0xda, 0x72, 0x72, 0xb3, 0xb3,
    0x2a, 0xe0, 0xfc, 0xab,
};
static const unsigned char kat217_nor_addin0[] = {0};
static const unsigned char kat217_nor_addin1[] = {0};
static const unsigned char kat217_nor_retbytes[] = {
    0x52, 0x45, 0xc3, 0x0a, 0x65, 0x1d, 0x98, 0x61, 0xb6, 0x36, 0xc8, 0xe8,
    0xcc, 0x8b, 0x84, 0x52, 0x46, 0xda, 0x10, 0xc1, 0x04, 0xd7, 0x81, 0x34,
    0xa8, 0x4e, 0x41, 0xfe, 0xa8, 0x0e, 0x0e, 0x73, 0x69, 0x2f, 0x84, 0x81,
    0xcd, 0x4d, 0x75, 0x0c, 0x79, 0xe3, 0x87, 0x6b, 0x9a, 0xbf, 0xa9, 0xd6,
    0x14, 0xd8, 0x68, 0x24, 0x96, 0x05, 0x61, 0x9d, 0xef, 0xef, 0x29, 0x68,
    0xfd, 0x33, 0x14, 0x1e,
};
static const struct drbg_kat_no_reseed kat217_nor_t = {
    6, kat217_nor_entropyin, kat217_nor_nonce, kat217_nor_persstr,
    kat217_nor_addin0, kat217_nor_addin1, kat217_nor_retbytes
};
static const struct drbg_kat kat217_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat217_nor_t
};

static const unsigned char kat218_nor_entropyin[] = {
    0xce, 0x08, 0xf0, 0x2b, 0xcd, 0xe4, 0x7f, 0x60, 0x30, 0x82, 0x02, 0x43,
    0xe1, 0x01, 0x38, 0x87,
};
static const unsigned char kat218_nor_nonce[] = {
    0xd1, 0xb1, 0x51, 0x30, 0xcd, 0x03, 0x8d, 0x6d,
};
static const unsigned char kat218_nor_persstr[] = {
    0xad, 0x0b, 0xcb, 0x82, 0x01, 0x16, 0x0d, 0x82, 0xf1, 0x79, 0x66, 0xd4,
    0xc7, 0xb6, 0xa4, 0xec,
};
static const unsigned char kat218_nor_addin0[] = {0};
static const unsigned char kat218_nor_addin1[] = {0};
static const unsigned char kat218_nor_retbytes[] = {
    0xf4, 0x8e, 0xd0, 0x3b, 0xdb, 0xa9, 0x31, 0x0d, 0x7f, 0xe0, 0xa5, 0xdd,
    0xdf, 0x9d, 0xf4, 0x9c, 0x0d, 0xbe, 0x07, 0xb9, 0x5b, 0xde, 0x25, 0xa0,
    0xb6, 0x6e, 0xd0, 0x1a, 0x9f, 0x7a, 0x07, 0x82, 0x0f, 0x2d, 0x7e, 0xaa,
    0x98, 0x63, 0x70, 0xa0, 0xce, 0x00, 0x01, 0x3b, 0x43, 0x31, 0xe4, 0x4b,
    0xeb, 0x30, 0x10, 0x57, 0x5a, 0xf7, 0xd6, 0x25, 0xbe, 0xd5, 0x5a, 0x59,
    0x2d, 0x97, 0x38, 0x28,
};
static const struct drbg_kat_no_reseed kat218_nor_t = {
    7, kat218_nor_entropyin, kat218_nor_nonce, kat218_nor_persstr,
    kat218_nor_addin0, kat218_nor_addin1, kat218_nor_retbytes
};
static const struct drbg_kat kat218_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat218_nor_t
};

static const unsigned char kat219_nor_entropyin[] = {
    0x8d, 0xbc, 0xda, 0xa7, 0x8a, 0xdd, 0xb2, 0x85, 0xdf, 0xdb, 0x5e, 0x41,
    0xee, 0xf6, 0x7e, 0x6f,
};
static const unsigned char kat219_nor_nonce[] = {
    0x49, 0x06, 0x2c, 0x09, 0x8b, 0x64, 0xe7, 0xf8,
};
static const unsigned char kat219_nor_persstr[] = {
    0x7d, 0xc1, 0x4e, 0x73, 0x96, 0xf6, 0x24, 0x50, 0xbe, 0xbd, 0xf2, 0xeb,
    0xf0, 0x17, 0xaa, 0xd5,
};
static const unsigned char kat219_nor_addin0[] = {0};
static const unsigned char kat219_nor_addin1[] = {0};
static const unsigned char kat219_nor_retbytes[] = {
    0x49, 0xd0, 0x3f, 0xe3, 0xb7, 0x2d, 0x44, 0xa8, 0xa1, 0xe4, 0x69, 0xd2,
    0x5a, 0x14, 0x5b, 0xa6, 0x4d, 0x61, 0x69, 0xdd, 0x94, 0x7f, 0x87, 0x93,
    0xd5, 0xef, 0x43, 0xde, 0x7b, 0x23, 0x94, 0x13, 0x70, 0x83, 0xe6, 0xe7,
    0x69, 0xbb, 0xfd, 0xe9, 0x60, 0x0c, 0x36, 0xb0, 0x32, 0xb7, 0x78, 0x65,
    0x22, 0x07, 0x0b, 0x5a, 0x65, 0xc7, 0x93, 0x92, 0x68, 0x92, 0xb9, 0xfb,
    0x0d, 0x1c, 0x1d, 0x54,
};
static const struct drbg_kat_no_reseed kat219_nor_t = {
    8, kat219_nor_entropyin, kat219_nor_nonce, kat219_nor_persstr,
    kat219_nor_addin0, kat219_nor_addin1, kat219_nor_retbytes
};
static const struct drbg_kat kat219_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat219_nor_t
};

static const unsigned char kat220_nor_entropyin[] = {
    0x0d, 0x98, 0x37, 0x0c, 0xff, 0xa0, 0xa8, 0x79, 0xfe, 0x85, 0xb5, 0xde,
    0x4e, 0x69, 0xcf, 0x7d,
};
static const unsigned char kat220_nor_nonce[] = {
    0x93, 0x50, 0xa8, 0x6b, 0x7a, 0x1b, 0xc6, 0x8a,
};
static const unsigned char kat220_nor_persstr[] = {
    0x01, 0x61, 0xb2, 0x4d, 0x1c, 0x3c, 0xa5, 0x90, 0x11, 0x7e, 0xc2, 0x9a,
    0xcc, 0x39, 0x44, 0x6f,
};
static const unsigned char kat220_nor_addin0[] = {0};
static const unsigned char kat220_nor_addin1[] = {0};
static const unsigned char kat220_nor_retbytes[] = {
    0x7c, 0xd6, 0x46, 0x84, 0x28, 0x9b, 0x43, 0xb1, 0xe5, 0x93, 0xd9, 0x4a,
    0xe9, 0x79, 0xf7, 0xfa, 0x5d, 0xe8, 0x95, 0x77, 0xf3, 0x7c, 0x7a, 0xea,
    0x0b, 0x58, 0x4d, 0x18, 0x60, 0x2c, 0x25, 0x1b, 0x1e, 0xc2, 0x5f, 0xf6,
    0x12, 0xd9, 0x7f, 0xda, 0x37, 0x84, 0xe9, 0x65, 0x59, 0x73, 0xe3, 0x1f,
    0xd4, 0xb5, 0xd1, 0xea, 0xde, 0xc6, 0x68, 0x61, 0xae, 0x2e, 0x97, 0x19,
    0xad, 0x34, 0x47, 0x30,
};
static const struct drbg_kat_no_reseed kat220_nor_t = {
    9, kat220_nor_entropyin, kat220_nor_nonce, kat220_nor_persstr,
    kat220_nor_addin0, kat220_nor_addin1, kat220_nor_retbytes
};
static const struct drbg_kat kat220_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat220_nor_t
};

static const unsigned char kat221_nor_entropyin[] = {
    0xd9, 0xb6, 0xfb, 0x62, 0xda, 0x0a, 0x02, 0x32, 0x35, 0xe7, 0xcd, 0x78,
    0x16, 0x37, 0x7c, 0xf5,
};
static const unsigned char kat221_nor_nonce[] = {
    0x1c, 0xa8, 0xc5, 0xfa, 0x5b, 0x9f, 0x8c, 0x90,
};
static const unsigned char kat221_nor_persstr[] = {
    0x17, 0xbe, 0x48, 0x85, 0xb5, 0xf9, 0x64, 0x1b, 0xf5, 0xf6, 0x89, 0xbc,
    0x97, 0x97, 0x88, 0x58,
};
static const unsigned char kat221_nor_addin0[] = {0};
static const unsigned char kat221_nor_addin1[] = {0};
static const unsigned char kat221_nor_retbytes[] = {
    0xe1, 0x1e, 0x3c, 0xa6, 0xc8, 0x32, 0xf4, 0x6f, 0xf1, 0xf9, 0x71, 0xc0,
    0x7e, 0x7b, 0x66, 0xc7, 0xd5, 0xb1, 0xb2, 0xe6, 0xec, 0x8c, 0x5f, 0xfc,
    0x77, 0x10, 0x3f, 0x0a, 0xd3, 0x08, 0x80, 0x0b, 0xb9, 0x89, 0xb9, 0xab,
    0x70, 0x10, 0x68, 0x33, 0x78, 0xa3, 0xf7, 0x81, 0x9a, 0x29, 0x7a, 0x37,
    0x65, 0x25, 0x6a, 0xc4, 0x0f, 0xb0, 0xa9, 0xcb, 0x22, 0x46, 0xae, 0xb8,
    0x5d, 0x73, 0x60, 0x1b,
};
static const struct drbg_kat_no_reseed kat221_nor_t = {
    10, kat221_nor_entropyin, kat221_nor_nonce, kat221_nor_persstr,
    kat221_nor_addin0, kat221_nor_addin1, kat221_nor_retbytes
};
static const struct drbg_kat kat221_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat221_nor_t
};

static const unsigned char kat222_nor_entropyin[] = {
    0xa7, 0x11, 0x39, 0x44, 0xfc, 0x4d, 0x4b, 0x15, 0xa9, 0x49, 0x67, 0xc8,
    0xf5, 0x65, 0x2a, 0x36,
};
static const unsigned char kat222_nor_nonce[] = {
    0xa5, 0x06, 0xb7, 0x9a, 0xfe, 0x6a, 0xf8, 0x22,
};
static const unsigned char kat222_nor_persstr[] = {
    0x91, 0x8f, 0xe4, 0xf3, 0x65, 0x25, 0x9c, 0x18, 0xeb, 0x68, 0x50, 0xfb,
    0xee, 0x40, 0x3f, 0x5e,
};
static const unsigned char kat222_nor_addin0[] = {0};
static const unsigned char kat222_nor_addin1[] = {0};
static const unsigned char kat222_nor_retbytes[] = {
    0x60, 0x76, 0xb2, 0xc0, 0xf7, 0xde, 0x1d, 0xd3, 0xe4, 0x6a, 0xdb, 0x11,
    0x61, 0xb7, 0x2a, 0x7f, 0x83, 0xb8, 0x77, 0x3f, 0xab, 0x0d, 0xcb, 0x1c,
    0x1f, 0xde, 0x25, 0xdb, 0x0d, 0x09, 0x86, 0x2a, 0xcd, 0xd3, 0x8f, 0x8d,
    0x21, 0x64, 0x90, 0x3a, 0x8e, 0x88, 0x58, 0xf8, 0xa9, 0xb6, 0x1b, 0xd7,
    0xda, 0xe3, 0xf6, 0x06, 0x68, 0xe6, 0xee, 0x26, 0x4b, 0x9f, 0xfa, 0xf5,
    0x78, 0xb5, 0x56, 0x46,
};
static const struct drbg_kat_no_reseed kat222_nor_t = {
    11, kat222_nor_entropyin, kat222_nor_nonce, kat222_nor_persstr,
    kat222_nor_addin0, kat222_nor_addin1, kat222_nor_retbytes
};
static const struct drbg_kat kat222_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat222_nor_t
};

static const unsigned char kat223_nor_entropyin[] = {
    0xac, 0xf7, 0x62, 0x38, 0x39, 0xb9, 0x4b, 0xd7, 0xa8, 0x93, 0xc2, 0x87,
    0x61, 0x6c, 0xdb, 0x6a,
};
static const unsigned char kat223_nor_nonce[] = {
    0xa1, 0xf6, 0x03, 0x9f, 0x91, 0xc1, 0x76, 0x23,
};
static const unsigned char kat223_nor_persstr[] = {
    0x5f, 0x9c, 0x69, 0xde, 0x2c, 0x32, 0x87, 0x3f, 0x67, 0x9d, 0x17, 0x68,
    0xef, 0x3c, 0x70, 0xfb,
};
static const unsigned char kat223_nor_addin0[] = {0};
static const unsigned char kat223_nor_addin1[] = {0};
static const unsigned char kat223_nor_retbytes[] = {
    0xf6, 0xab, 0xbf, 0xfd, 0x79, 0x65, 0x64, 0x94, 0x36, 0xb4, 0x8b, 0x09,
    0x20, 0x86, 0xcd, 0xf0, 0x50, 0x2f, 0x52, 0xc8, 0x7c, 0x8f, 0xdc, 0x8f,
    0x6d, 0x49, 0xf0, 0x84, 0x33, 0xa2, 0x02, 0xd8, 0xbe, 0x44, 0xf6, 0x56,
    0x2a, 0xce, 0x58, 0x0e, 0x80, 0x75, 0xf0, 0xbc, 0x67, 0x0b, 0x2e, 0x9d,
    0x95, 0x1f, 0x15, 0xf8, 0x4d, 0x82, 0xaf, 0xe6, 0xb8, 0x32, 0x76, 0x5e,
    0xca, 0xb0, 0x37, 0xee,
};
static const struct drbg_kat_no_reseed kat223_nor_t = {
    12, kat223_nor_entropyin, kat223_nor_nonce, kat223_nor_persstr,
    kat223_nor_addin0, kat223_nor_addin1, kat223_nor_retbytes
};
static const struct drbg_kat kat223_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat223_nor_t
};

static const unsigned char kat224_nor_entropyin[] = {
    0xd9, 0x0d, 0x78, 0x6e, 0x9d, 0x19, 0x62, 0x84, 0x3b, 0x02, 0x7a, 0xa0,
    0xe8, 0x59, 0x8f, 0xd4,
};
static const unsigned char kat224_nor_nonce[] = {
    0x04, 0x17, 0xf1, 0x4f, 0x96, 0xf2, 0x02, 0x9c,
};
static const unsigned char kat224_nor_persstr[] = {
    0xec, 0x5c, 0x55, 0x14, 0x5a, 0x4f, 0x95, 0x52, 0xf2, 0x51, 0xf7, 0xdf,
    0xb0, 0x17, 0xc1, 0xd4,
};
static const unsigned char kat224_nor_addin0[] = {0};
static const unsigned char kat224_nor_addin1[] = {0};
static const unsigned char kat224_nor_retbytes[] = {
    0xfe, 0x2c, 0xa4, 0x44, 0x55, 0xa5, 0xe7, 0xde, 0x70, 0x8a, 0x71, 0x0a,
    0xb6, 0x46, 0xcf, 0xcf, 0xbc, 0x87, 0xd6, 0x78, 0xb3, 0xe9, 0x41, 0x06,
    0x3f, 0xa7, 0x01, 0xda, 0xc7, 0xcb, 0x4f, 0x09, 0x53, 0xaf, 0x8f, 0xce,
    0x45, 0x8c, 0xee, 0x54, 0x9f, 0x53, 0x7f, 0xa7, 0x71, 0xb8, 0x9b, 0x04,
    0x94, 0xc0, 0xbb, 0x5c, 0xfd, 0x35, 0xbf, 0x67, 0xf4, 0xb2, 0x70, 0xd6,
    0x20, 0x5f, 0x47, 0x03,
};
static const struct drbg_kat_no_reseed kat224_nor_t = {
    13, kat224_nor_entropyin, kat224_nor_nonce, kat224_nor_persstr,
    kat224_nor_addin0, kat224_nor_addin1, kat224_nor_retbytes
};
static const struct drbg_kat kat224_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat224_nor_t
};

static const unsigned char kat225_nor_entropyin[] = {
    0x1c, 0xdb, 0x74, 0x45, 0x59, 0xa8, 0x2c, 0x62, 0xfb, 0x3b, 0x15, 0xa2,
    0xa3, 0xaa, 0xd1, 0x89,
};
static const unsigned char kat225_nor_nonce[] = {
    0x43, 0x43, 0x2c, 0xe3, 0x0a, 0xe7, 0xf9, 0x6b,
};
static const unsigned char kat225_nor_persstr[] = {
    0xf5, 0x9a, 0xd9, 0x93, 0x9f, 0xcd, 0x6c, 0x0f, 0x47, 0x8c, 0xc5, 0x08,
    0x39, 0xf8, 0xff, 0xce,
};
static const unsigned char kat225_nor_addin0[] = {0};
static const unsigned char kat225_nor_addin1[] = {0};
static const unsigned char kat225_nor_retbytes[] = {
    0x91, 0xb9, 0xeb, 0xeb, 0x92, 0x9d, 0xc6, 0x99, 0x94, 0x91, 0x22, 0x97,
    0x0b, 0x21, 0x77, 0xd5, 0xd5, 0xb7, 0x05, 0x04, 0x2d, 0x3a, 0x0d, 0x60,
    0xd7, 0xd3, 0xbb, 0x21, 0x8b, 0x7a, 0x69, 0xe1, 0xcd, 0x68, 0x64, 0xb8,
    0xa2, 0x68, 0xca, 0x78, 0xc8, 0x34, 0x23, 0x2d, 0x0c, 0xb8, 0x8f, 0x93,
    0x77, 0x30, 0x08, 0x4b, 0x1e, 0xc7, 0xd2, 0x86, 0x7f, 0xbc, 0x85, 0x0e,
    0x04, 0x05, 0x03, 0x24,
};
static const struct drbg_kat_no_reseed kat225_nor_t = {
    14, kat225_nor_entropyin, kat225_nor_nonce, kat225_nor_persstr,
    kat225_nor_addin0, kat225_nor_addin1, kat225_nor_retbytes
};
static const struct drbg_kat kat225_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 0, 64, &kat225_nor_t
};

static const unsigned char kat226_nor_entropyin[] = {
    0x71, 0xba, 0x02, 0x9a, 0x7a, 0x92, 0x39, 0x1b, 0x3f, 0x29, 0x4f, 0x2f,
    0xbf, 0x57, 0x27, 0xab,
};
static const unsigned char kat226_nor_nonce[] = {
    0xf0, 0xe9, 0x12, 0x78, 0x8f, 0x98, 0x27, 0xff,
};
static const unsigned char kat226_nor_persstr[] = {
    0x23, 0xf3, 0x69, 0x80, 0xda, 0x40, 0x16, 0x64, 0x2c, 0x81, 0x0d, 0xa2,
    0x99, 0x0a, 0xa2, 0x5e,
};
static const unsigned char kat226_nor_addin0[] = {
    0x59, 0xab, 0x41, 0xb2, 0x4e, 0xe8, 0xe2, 0x71, 0xe2, 0x53, 0xc6, 0xcc,
    0x40, 0x48, 0x7c, 0xb5,
};
static const unsigned char kat226_nor_addin1[] = {
    0x91, 0x64, 0xf0, 0x28, 0x60, 0x77, 0x3e, 0x3b, 0x96, 0xd8, 0x5b, 0x37,
    0x38, 0x38, 0x50, 0x66,
};
static const unsigned char kat226_nor_retbytes[] = {
    0xde, 0x1b, 0x8a, 0x25, 0x95, 0x89, 0x23, 0x54, 0xda, 0x47, 0xb4, 0xea,
    0xaf, 0x9d, 0xdc, 0xec, 0x64, 0xa9, 0x61, 0x01, 0x17, 0xb0, 0x5e, 0x40,
    0xd0, 0x76, 0x60, 0xa8, 0x0b, 0xcf, 0x82, 0x5e, 0xef, 0xdb, 0xd2, 0x8e,
    0x07, 0xd5, 0x96, 0x81, 0xf9, 0xe0, 0x03, 0x7b, 0xdb, 0x72, 0x5f, 0xe6,
    0xce, 0x84, 0x6d, 0x82, 0x4b, 0x3b, 0x34, 0xc2, 0xc2, 0x1a, 0x48, 0xf8,
    0x89, 0x5f, 0x9f, 0x5e,
};
static const struct drbg_kat_no_reseed kat226_nor_t = {
    0, kat226_nor_entropyin, kat226_nor_nonce, kat226_nor_persstr,
    kat226_nor_addin0, kat226_nor_addin1, kat226_nor_retbytes
};
static const struct drbg_kat kat226_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat226_nor_t
};

static const unsigned char kat227_nor_entropyin[] = {
    0x67, 0x73, 0x94, 0xf0, 0x3e, 0xac, 0xb5, 0xa1, 0x37, 0x51, 0x5f, 0xcd,
    0x3e, 0xa2, 0xbb, 0xc7,
};
static const unsigned char kat227_nor_nonce[] = {
    0x9a, 0x5c, 0x1c, 0xea, 0x26, 0xef, 0xbf, 0x76,
};
static const unsigned char kat227_nor_persstr[] = {
    0xdd, 0x73, 0xb1, 0x8e, 0xee, 0x62, 0x94, 0x34, 0x9a, 0xa6, 0x45, 0x6d,
    0xaa, 0x77, 0xd2, 0x4f,
};
static const unsigned char kat227_nor_addin0[] = {
    0xeb, 0x70, 0x19, 0x82, 0x6d, 0xb3, 0x27, 0x17, 0xa5, 0x73, 0x5e, 0xb4,
    0x07, 0x73, 0xea, 0x56,
};
static const unsigned char kat227_nor_addin1[] = {
    0xb6, 0x2d, 0xe2, 0xb4, 0xaa, 0x8d, 0x97, 0xf5, 0x26, 0xdd, 0xba, 0x44,
    0x09, 0xf5, 0xaa, 0x26,
};
static const unsigned char kat227_nor_retbytes[] = {
    0x22, 0x22, 0xa6, 0xbb, 0x0f, 0xfe, 0x14, 0xd2, 0xf7, 0x89, 0xe6, 0x4d,
    0xcd, 0xf4, 0x78, 0x51, 0xa6, 0xc3, 0xa6, 0xe1, 0xd0, 0x83, 0x7d, 0x8b,
    0x51, 0x1a, 0xa2, 0xf5, 0x6a, 0x6d, 0x08, 0x53, 0x4c, 0x97, 0xf4, 0xb1,
    0x2a, 0x77, 0x44, 0x7d, 0xb2, 0x04, 0x09, 0xd3, 0x27, 0xfc, 0x08, 0x81,
    0x62, 0xc0, 0xf2, 0xc5, 0x9f, 0xe4, 0x7e, 0x8c, 0x92, 0xae, 0x5d, 0xcd,
    0xd7, 0x38, 0xc7, 0x68,
};
static const struct drbg_kat_no_reseed kat227_nor_t = {
    1, kat227_nor_entropyin, kat227_nor_nonce, kat227_nor_persstr,
    kat227_nor_addin0, kat227_nor_addin1, kat227_nor_retbytes
};
static const struct drbg_kat kat227_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat227_nor_t
};

static const unsigned char kat228_nor_entropyin[] = {
    0x3f, 0x51, 0xf6, 0xcc, 0xf2, 0x24, 0x79, 0xed, 0x89, 0x1b, 0x89, 0x1f,
    0xc4, 0xc6, 0xb5, 0x51,
};
static const unsigned char kat228_nor_nonce[] = {
    0x19, 0xe0, 0x75, 0x4f, 0x7b, 0xfa, 0x0c, 0xef,
};
static const unsigned char kat228_nor_persstr[] = {
    0x40, 0x2a, 0x1d, 0xab, 0x45, 0x0e, 0xa9, 0x04, 0x97, 0x38, 0x78, 0x2a,
    0x93, 0x34, 0x7a, 0xff,
};
static const unsigned char kat228_nor_addin0[] = {
    0x52, 0xc9, 0x0f, 0x57, 0x1c, 0x8c, 0x63, 0xf0, 0xdc, 0xdb, 0xd4, 0xe6,
    0xc9, 0x65, 0x34, 0x78,
};
static const unsigned char kat228_nor_addin1[] = {
    0x8b, 0xe6, 0x3b, 0x5e, 0xf2, 0x12, 0x69, 0x4e, 0xdb, 0xc2, 0xc8, 0xde,
    0xc9, 0xa0, 0x58, 0x7b,
};
static const unsigned char kat228_nor_retbytes[] = {
    0x1a, 0xbf, 0xe2, 0xd7, 0x21, 0x93, 0xb6, 0xe3, 0xd9, 0xbe, 0x85, 0x10,
    0x7e, 0xce, 0x8f, 0xea, 0x5d, 0x22, 0x95, 0x68, 0x56, 0x8c, 0x93, 0xeb,
    0x5e, 0xae, 0xd4, 0x63, 0xbf, 0x2b, 0x79, 0xcf, 0xdb, 0xd0, 0x47, 0x40,
    0x2f, 0xf4, 0xe4, 0x22, 0x34, 0xc4, 0xe3, 0xe1, 0x50, 0xc1, 0xf0, 0xe8,
    0x24, 0x00, 0xf6, 0xa5, 0x9e, 0xeb, 0xd0, 0x92, 0xee, 0x73, 0xad, 0xf1,
    0xff, 0xca, 0x64, 0x94,
};
static const struct drbg_kat_no_reseed kat228_nor_t = {
    2, kat228_nor_entropyin, kat228_nor_nonce, kat228_nor_persstr,
    kat228_nor_addin0, kat228_nor_addin1, kat228_nor_retbytes
};
static const struct drbg_kat kat228_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat228_nor_t
};

static const unsigned char kat229_nor_entropyin[] = {
    0x5a, 0x01, 0xb6, 0x38, 0x31, 0x1a, 0x29, 0x6b, 0xfe, 0x1f, 0x6e, 0x18,
    0x0f, 0x24, 0x3d, 0x5b,
};
static const unsigned char kat229_nor_nonce[] = {
    0x16, 0xbb, 0x4e, 0x36, 0xfd, 0x04, 0xb0, 0x5b,
};
static const unsigned char kat229_nor_persstr[] = {
    0xd5, 0x44, 0x0a, 0x6f, 0xa4, 0xd3, 0x45, 0xcb, 0x84, 0xd5, 0x6d, 0xec,
    0xaa, 0x3d, 0xbd, 0x18,
};
static const unsigned char kat229_nor_addin0[] = {
    0xbd, 0x3e, 0x11, 0xdb, 0xb9, 0xf4, 0x0a, 0x13, 0x51, 0xc8, 0xbf, 0x7b,
    0xd8, 0x94, 0xad, 0x20,
};
static const unsigned char kat229_nor_addin1[] = {
    0xe0, 0x4a, 0x83, 0xa5, 0x18, 0x75, 0x15, 0x10, 0x4a, 0xa1, 0xe7, 0xf4,
    0x34, 0xff, 0x43, 0x36,
};
static const unsigned char kat229_nor_retbytes[] = {
    0x27, 0x28, 0xfc, 0xc1, 0xfe, 0xa7, 0xfe, 0x63, 0x5d, 0xdf, 0xed, 0x8b,
    0xb3, 0xf7, 0xca, 0x29, 0xe1, 0xcb, 0xd8, 0x51, 0x6a, 0x2b, 0x5e, 0xab,
    0xb7, 0xd0, 0x91, 0xe6, 0xe7, 0xf2, 0x04, 0x6a, 0x80, 0x77, 0xef, 0x6d,
    0x1a, 0x90, 0x16, 0x5f, 0x18, 0x9d, 0x07, 0xa8, 0x97, 0xcd, 0xfc, 0x06,
    0x88, 0x2a, 0xdb, 0x78, 0x1b, 0xe5, 0x89, 0xe0, 0xe5, 0xeb, 0x2d, 0xc9,
    0x7b, 0xe6, 0x54, 0x6b,
};
static const struct drbg_kat_no_reseed kat229_nor_t = {
    3, kat229_nor_entropyin, kat229_nor_nonce, kat229_nor_persstr,
    kat229_nor_addin0, kat229_nor_addin1, kat229_nor_retbytes
};
static const struct drbg_kat kat229_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat229_nor_t
};

static const unsigned char kat230_nor_entropyin[] = {
    0x97, 0xe4, 0x15, 0xb3, 0xca, 0x11, 0x0a, 0x80, 0x9e, 0xa6, 0x00, 0xa1,
    0x3e, 0x47, 0xaa, 0x04,
};
static const unsigned char kat230_nor_nonce[] = {
    0x12, 0x6e, 0x16, 0xdc, 0x57, 0x22, 0xa4, 0x49,
};
static const unsigned char kat230_nor_persstr[] = {
    0xa9, 0xbe, 0x38, 0xe3, 0x01, 0x19, 0x86, 0xb9, 0x04, 0x71, 0x71, 0x93,
    0xec, 0x4a, 0x7d, 0x4c,
};
static const unsigned char kat230_nor_addin0[] = {
    0x19, 0x68, 0x69, 0x92, 0x21, 0xeb, 0x8b, 0x5e, 0xd0, 0x23, 0xa5, 0xd2,
    0xd2, 0xd8, 0x12, 0x56,
};
static const unsigned char kat230_nor_addin1[] = {
    0x4b, 0x5e, 0x1b, 0xde, 0x38, 0xb2, 0xa9, 0x79, 0x3b, 0x16, 0x6d, 0x86,
    0x81, 0x6e, 0xd2, 0x31,
};
static const unsigned char kat230_nor_retbytes[] = {
    0x4b, 0xe9, 0x89, 0x89, 0xdf, 0x2f, 0xe6, 0x68, 0x7e, 0x50, 0x88, 0xe5,
    0x06, 0x98, 0x50, 0x7d, 0x91, 0x06, 0x1a, 0x8d, 0x6e, 0xb9, 0x5d, 0x66,
    0x27, 0xd2, 0xb8, 0x57, 0x25, 0x3f, 0x05, 0x3c, 0x99, 0xee, 0x79, 0x84,
    0xa4, 0x98, 0xde, 0x4d, 0xf3, 0x12, 0x44, 0x42, 0x85, 0x6f, 0x18, 0xe9,
    0x8e, 0x66, 0x41, 0xf1, 0x52, 0xcd, 0x12, 0x9b, 0xd0, 0x4e, 0x99, 0x71,
    0x09, 0x30, 0xc3, 0xe1,
};
static const struct drbg_kat_no_reseed kat230_nor_t = {
    4, kat230_nor_entropyin, kat230_nor_nonce, kat230_nor_persstr,
    kat230_nor_addin0, kat230_nor_addin1, kat230_nor_retbytes
};
static const struct drbg_kat kat230_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat230_nor_t
};

static const unsigned char kat231_nor_entropyin[] = {
    0xb6, 0xc4, 0xcd, 0x96, 0xf7, 0xfd, 0xe8, 0x3e, 0xb5, 0x71, 0x22, 0x9b,
    0x53, 0x7a, 0xcf, 0x5a,
};
static const unsigned char kat231_nor_nonce[] = {
    0x4c, 0x4e, 0x0a, 0xac, 0xb5, 0xc5, 0x28, 0xe7,
};
static const unsigned char kat231_nor_persstr[] = {
    0x5c, 0xe1, 0x09, 0x4d, 0x0d, 0x30, 0x7b, 0x41, 0xdb, 0x9b, 0xf2, 0x6d,
    0xa4, 0x1d, 0xb1, 0x00,
};
static const unsigned char kat231_nor_addin0[] = {
    0x98, 0x04, 0x7e, 0xe9, 0x87, 0x63, 0x7e, 0x45, 0x84, 0xfe, 0x7f, 0xe2,
    0x86, 0x9c, 0x84, 0x5d,
};
static const unsigned char kat231_nor_addin1[] = {
    0x17, 0x7e, 0x3c, 0x29, 0xd4, 0xfa, 0xf0, 0x43, 0x07, 0x7f, 0xbe, 0xfe,
    0x0b, 0x99, 0x4b, 0x91,
};
static const unsigned char kat231_nor_retbytes[] = {
    0x5e, 0xaf, 0x01, 0x3f, 0xd3, 0x77, 0x5a, 0x2d, 0x3e, 0x97, 0x16, 0x6d,
    0x23, 0x15, 0x3a, 0x00, 0xf1, 0x86, 0x44, 0x96, 0x3e, 0x69, 0xcb, 0x96,
    0x2a, 0x7b, 0xd2, 0xc3, 0x0a, 0xf9, 0xbd, 0x6a, 0x29, 0xd4, 0xbc, 0x70,
    0x93, 0x0a, 0x92, 0x9b, 0x92, 0x52, 0xd8, 0x08, 0xdb, 0x41, 0x0c, 0x66,
    0x41, 0x23, 0xbd, 0x69, 0xc1, 0xd0, 0xc1, 0xd4, 0xba, 0x3c, 0x7d, 0x8f,
    0xf7, 0xae, 0x00, 0xf0,
};
static const struct drbg_kat_no_reseed kat231_nor_t = {
    5, kat231_nor_entropyin, kat231_nor_nonce, kat231_nor_persstr,
    kat231_nor_addin0, kat231_nor_addin1, kat231_nor_retbytes
};
static const struct drbg_kat kat231_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat231_nor_t
};

static const unsigned char kat232_nor_entropyin[] = {
    0x2d, 0x1b, 0x80, 0x61, 0xd7, 0xd5, 0x83, 0xf8, 0x3b, 0xe5, 0x53, 0xc4,
    0x83, 0x72, 0x12, 0xa0,
};
static const unsigned char kat232_nor_nonce[] = {
    0x62, 0xa1, 0xfa, 0x74, 0x53, 0x62, 0xba, 0xce,
};
static const unsigned char kat232_nor_persstr[] = {
    0x4d, 0x97, 0x6a, 0xdd, 0x18, 0xd5, 0xe2, 0x48, 0x4f, 0x67, 0xf7, 0xc8,
    0x4c, 0xc6, 0x8b, 0x52,
};
static const unsigned char kat232_nor_addin0[] = {
    0x57, 0x64, 0xde, 0x5d, 0xb9, 0x7e, 0x22, 0x3c, 0x04, 0x4a, 0x83, 0x3f,
    0xa4, 0x2c, 0xf6, 0x29,
};
static const unsigned char kat232_nor_addin1[] = {
    0xde, 0xef, 0x8d, 0x58, 0x1f, 0x6e, 0x4f, 0x80, 0x89, 0x92, 0xfd, 0x20,
    0xc0, 0x6e, 0x10, 0x2d,
};
static const unsigned char kat232_nor_retbytes[] = {
    0xac, 0xb7, 0x8f, 0x27, 0x33, 0x50, 0xd4, 0x59, 0xc7, 0x98, 0x09, 0xd8,
    0x9b, 0x40, 0x28, 0x25, 0xbb, 0x00, 0xe3, 0x8a, 0xbd, 0x5e, 0xc1, 0x2f,
    0xea, 0xdc, 0xde, 0xf6, 0x11, 0x46, 0x84, 0xb6, 0xfc, 0xb0, 0xe1, 0x76,
    0xcb, 0xe4, 0x65, 0x91, 0x0f, 0xc2, 0x84, 0x8f, 0x0a, 0x70, 0x88, 0x44,
    0x79, 0x4c, 0x5e, 0xea, 0xfe, 0xad, 0xa7, 0x56, 0x3c, 0x38, 0xc3, 0xee,
    0x63, 0xc9, 0x7e, 0xf9,
};
static const struct drbg_kat_no_reseed kat232_nor_t = {
    6, kat232_nor_entropyin, kat232_nor_nonce, kat232_nor_persstr,
    kat232_nor_addin0, kat232_nor_addin1, kat232_nor_retbytes
};
static const struct drbg_kat kat232_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat232_nor_t
};

static const unsigned char kat233_nor_entropyin[] = {
    0xef, 0x30, 0xfd, 0x99, 0x5f, 0xf0, 0xef, 0x44, 0xcc, 0xe6, 0x07, 0x7b,
    0xb2, 0x2d, 0xe6, 0x99,
};
static const unsigned char kat233_nor_nonce[] = {
    0x85, 0xc7, 0xd6, 0x05, 0x4a, 0x57, 0x76, 0x22,
};
static const unsigned char kat233_nor_persstr[] = {
    0x5d, 0x59, 0xb3, 0xf0, 0x25, 0x00, 0x60, 0xe2, 0x20, 0x02, 0xad, 0xf7,
    0x06, 0x48, 0x6d, 0x85,
};
static const unsigned char kat233_nor_addin0[] = {
    0x79, 0x46, 0x81, 0xf7, 0x5d, 0xf9, 0xc9, 0x88, 0x30, 0x3f, 0x75, 0x1f,
    0xb7, 0xe7, 0xfb, 0xa9,
};
static const unsigned char kat233_nor_addin1[] = {
    0x7b, 0x13, 0x74, 0xc8, 0x47, 0x25, 0x37, 0xfa, 0x31, 0x11, 0xeb, 0x44,
    0xe5, 0x2a, 0xfa, 0x23,
};
static const unsigned char kat233_nor_retbytes[] = {
    0xb0, 0x49, 0x50, 0xf1, 0xa3, 0x57, 0x52, 0xc0, 0x67, 0xbc, 0xe8, 0x35,
    0x76, 0x35, 0xf4, 0x70, 0xd1, 0x0b, 0x3b, 0xae, 0x47, 0x46, 0x6f, 0xd1,
    0x16, 0x6f, 0x9f, 0xfa, 0xef, 0xbd, 0x32, 0x4b, 0x26, 0x53, 0xc7, 0xf3,
    0x0e, 0xe4, 0x84, 0x86, 0x73, 0xd7, 0x04, 0x18, 0x65, 0xd9, 0x95, 0x45,
    0x12, 0x58, 0x94, 0x92, 0x98, 0x9a, 0x00, 0xea, 0x4d, 0xa1, 0xdd, 0x28,
    0x9e, 0x87, 0x17, 0xc3,
};
static const struct drbg_kat_no_reseed kat233_nor_t = {
    7, kat233_nor_entropyin, kat233_nor_nonce, kat233_nor_persstr,
    kat233_nor_addin0, kat233_nor_addin1, kat233_nor_retbytes
};
static const struct drbg_kat kat233_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat233_nor_t
};

static const unsigned char kat234_nor_entropyin[] = {
    0xee, 0x4b, 0xde, 0x4f, 0x71, 0x2d, 0xc9, 0xfc, 0x1f, 0x2c, 0x01, 0xf1,
    0xc6, 0x08, 0x6f, 0x35,
};
static const unsigned char kat234_nor_nonce[] = {
    0xf6, 0x83, 0x7c, 0xfe, 0x53, 0xec, 0x48, 0x74,
};
static const unsigned char kat234_nor_persstr[] = {
    0xfd, 0xb8, 0xf6, 0x4d, 0x99, 0xbd, 0x71, 0x7e, 0xee, 0x3d, 0x89, 0x35,
    0x3b, 0x73, 0xc9, 0xc1,
};
static const unsigned char kat234_nor_addin0[] = {
    0x09, 0xe5, 0xba, 0x3b, 0x23, 0x89, 0x5c, 0x5d, 0xff, 0x89, 0x38, 0x2d,
    0x5e, 0x91, 0x70, 0x0d,
};
static const unsigned char kat234_nor_addin1[] = {
    0xeb, 0x1a, 0x98, 0xde, 0x89, 0x62, 0xbb, 0xc4, 0xcb, 0x75, 0xcf, 0x0b,
    0xf0, 0xf8, 0xdf, 0xa4,
};
static const unsigned char kat234_nor_retbytes[] = {
    0xaa, 0xcd, 0xee, 0x0f, 0x3d, 0x1f, 0x95, 0x5f, 0x89, 0x6d, 0x5a, 0x5a,
    0x25, 0x30, 0xfd, 0x80, 0x9a, 0x20, 0x2c, 0x92, 0x1d, 0x90, 0xa9, 0xcf,
    0x59, 0x3e, 0x03, 0x93, 0x4e, 0x07, 0xc3, 0x92, 0xcc, 0x55, 0x54, 0xf5,
    0x99, 0x21, 0xb4, 0x0c, 0xc0, 0x03, 0xfc, 0x26, 0x82, 0xca, 0xef, 0x9e,
    0x6e, 0x61, 0xee, 0x4a, 0xbf, 0x45, 0x14, 0x8e, 0x5f, 0xf3, 0xff, 0x65,
    0x87, 0x1b, 0x0d, 0x81,
};
static const struct drbg_kat_no_reseed kat234_nor_t = {
    8, kat234_nor_entropyin, kat234_nor_nonce, kat234_nor_persstr,
    kat234_nor_addin0, kat234_nor_addin1, kat234_nor_retbytes
};
static const struct drbg_kat kat234_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat234_nor_t
};

static const unsigned char kat235_nor_entropyin[] = {
    0xa6, 0xaa, 0x94, 0x5e, 0xbe, 0x19, 0x29, 0x98, 0xbe, 0x8e, 0x24, 0x7d,
    0xb9, 0x9c, 0xe7, 0xd0,
};
static const unsigned char kat235_nor_nonce[] = {
    0x7d, 0xff, 0x26, 0xe9, 0x7a, 0x0a, 0x8d, 0xba,
};
static const unsigned char kat235_nor_persstr[] = {
    0x4b, 0x6c, 0xcf, 0xac, 0x6c, 0x47, 0xfd, 0x65, 0x64, 0xbc, 0xfd, 0x94,
    0xfe, 0x9e, 0xb2, 0x8d,
};
static const unsigned char kat235_nor_addin0[] = {
    0x07, 0x83, 0x70, 0x53, 0xd2, 0x4c, 0xd0, 0xf4, 0xf4, 0x32, 0x7e, 0x97,
    0xd2, 0x9d, 0x67, 0x95,
};
static const unsigned char kat235_nor_addin1[] = {
    0x3b, 0x17, 0x2d, 0xdc, 0x8d, 0x38, 0x3b, 0x98, 0xd4, 0x08, 0x39, 0x4c,
    0x0b, 0xa1, 0x52, 0xc1,
};
static const unsigned char kat235_nor_retbytes[] = {
    0xd9, 0x0a, 0xa4, 0x22, 0xa8, 0xd2, 0xb8, 0x6e, 0x1f, 0x09, 0xde, 0x29,
    0xdd, 0x16, 0xd4, 0x60, 0x50, 0x98, 0xa4, 0x5a, 0xea, 0xbe, 0x69, 0xc6,
    0xbc, 0x8a, 0xe0, 0xc8, 0xf0, 0x1b, 0x7c, 0xc7, 0x3f, 0x88, 0x50, 0xec,
    0x0c, 0x47, 0xd6, 0x16, 0xcb, 0x6c, 0xca, 0xb7, 0x7f, 0x22, 0x07, 0x99,
    0x38, 0xed, 0x11, 0xb4, 0x8f, 0xcf, 0x3c, 0xf4, 0x6a, 0x3f, 0x64, 0x2f,
    0x05, 0xae, 0xb8, 0xb7,
};
static const struct drbg_kat_no_reseed kat235_nor_t = {
    9, kat235_nor_entropyin, kat235_nor_nonce, kat235_nor_persstr,
    kat235_nor_addin0, kat235_nor_addin1, kat235_nor_retbytes
};
static const struct drbg_kat kat235_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat235_nor_t
};

static const unsigned char kat236_nor_entropyin[] = {
    0x7e, 0xb5, 0x0f, 0x53, 0x2b, 0x00, 0x76, 0x52, 0x8e, 0x2d, 0x1f, 0x26,
    0x6b, 0x38, 0x14, 0x06,
};
static const unsigned char kat236_nor_nonce[] = {
    0x2a, 0x5e, 0x92, 0xe8, 0xfb, 0xf0, 0xe9, 0xee,
};
static const unsigned char kat236_nor_persstr[] = {
    0x7e, 0xcb, 0xfd, 0x22, 0x98, 0x87, 0x2a, 0x79, 0x34, 0xe4, 0xed, 0x61,
    0xa9, 0xf0, 0x04, 0xcd,
};
static const unsigned char kat236_nor_addin0[] = {
    0x17, 0x9f, 0xa5, 0xd2, 0xeb, 0x90, 0xd4, 0x15, 0x48, 0x19, 0x2c, 0xc7,
    0x44, 0x98, 0xbd, 0x42,
};
static const unsigned char kat236_nor_addin1[] = {
    0x92, 0xdc, 0x9d, 0x60, 0xd2, 0xe3, 0x6d, 0x4b, 0xfb, 0x97, 0x90, 0x78,
    0x3d, 0x1c, 0x58, 0x19,
};
static const unsigned char kat236_nor_retbytes[] = {
    0x9a, 0x17, 0x1b, 0x50, 0x48, 0xba, 0xf1, 0x37, 0x5c, 0x14, 0x07, 0xa8,
    0xdc, 0x1c, 0x8e, 0xda, 0x53, 0x2d, 0xf5, 0x52, 0x96, 0x77, 0x06, 0x88,
    0x4e, 0x5f, 0xe3, 0xd4, 0x67, 0x86, 0x07, 0x74, 0x48, 0xf2, 0x8f, 0x89,
    0xb3, 0x8e, 0xb7, 0x64, 0xdf, 0x7e, 0x0e, 0x3b, 0x40, 0x1a, 0xed, 0x3a,
    0x23, 0x06, 0xfa, 0x2b, 0xed, 0xe3, 0x82, 0xaa, 0xe1, 0x08, 0xe1, 0x6a,
    0xb4, 0x85, 0x69, 0x1e,
};
static const struct drbg_kat_no_reseed kat236_nor_t = {
    10, kat236_nor_entropyin, kat236_nor_nonce, kat236_nor_persstr,
    kat236_nor_addin0, kat236_nor_addin1, kat236_nor_retbytes
};
static const struct drbg_kat kat236_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat236_nor_t
};

static const unsigned char kat237_nor_entropyin[] = {
    0x7d, 0x33, 0x40, 0xa8, 0xb1, 0xe1, 0x51, 0x71, 0xee, 0x34, 0x03, 0xcc,
    0x19, 0xd3, 0xfe, 0xa9,
};
static const unsigned char kat237_nor_nonce[] = {
    0xfd, 0xe6, 0x97, 0xac, 0xc7, 0xe7, 0x2b, 0x75,
};
static const unsigned char kat237_nor_persstr[] = {
    0xa1, 0x28, 0x79, 0x71, 0x79, 0xc5, 0xca, 0xd6, 0xa9, 0xb4, 0x76, 0xff,
    0x99, 0xf9, 0x82, 0x8f,
};
static const unsigned char kat237_nor_addin0[] = {
    0x2d, 0xfa, 0x30, 0x85, 0xfa, 0x02, 0x3b, 0x53, 0x61, 0xdb, 0xa7, 0x9d,
    0x40, 0x77, 0x38, 0xbc,
};
static const unsigned char kat237_nor_addin1[] = {
    0xae, 0xf5, 0x3a, 0xf3, 0xbe, 0x89, 0x41, 0xe7, 0xd2, 0x71, 0x3c, 0x71,
    0x28, 0x41, 0xdc, 0x68,
};
static const unsigned char kat237_nor_retbytes[] = {
    0xc1, 0xf7, 0x25, 0xb2, 0x90, 0xed, 0x13, 0x10, 0xdd, 0x3f, 0x39, 0xe9,
    0x9c, 0x7a, 0x65, 0xb8, 0x01, 0xb4, 0x74, 0x2f, 0x50, 0x66, 0xb3, 0xc7,
    0x1b, 0xb4, 0x6f, 0x3d, 0xe7, 0x4d, 0xd2, 0xca, 0xae, 0xa4, 0x08, 0x21,
    0x17, 0x4f, 0x67, 0x67, 0x22, 0xdb, 0x38, 0xfe, 0xda, 0x5d, 0x10, 0x96,
    0xc7, 0x9a, 0xd1, 0xea, 0xbe, 0xff, 0x78, 0xde, 0xf0, 0x84, 0x7a, 0x24,
    0x23, 0xbb, 0x46, 0x02,
};
static const struct drbg_kat_no_reseed kat237_nor_t = {
    11, kat237_nor_entropyin, kat237_nor_nonce, kat237_nor_persstr,
    kat237_nor_addin0, kat237_nor_addin1, kat237_nor_retbytes
};
static const struct drbg_kat kat237_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat237_nor_t
};

static const unsigned char kat238_nor_entropyin[] = {
    0x05, 0x71, 0xd4, 0x1f, 0x98, 0xb7, 0x1e, 0xad, 0x41, 0x5f, 0xcb, 0x97,
    0xfd, 0x05, 0xa1, 0xf1,
};
static const unsigned char kat238_nor_nonce[] = {
    0x42, 0x16, 0x18, 0x2c, 0x09, 0x4c, 0x4b, 0x14,
};
static const unsigned char kat238_nor_persstr[] = {
    0x6b, 0xed, 0x5c, 0xc7, 0x26, 0x60, 0xf7, 0xfd, 0x4c, 0x32, 0xab, 0xe5,
    0xcb, 0x1a, 0xef, 0xed,
};
static const unsigned char kat238_nor_addin0[] = {
    0xb1, 0x98, 0x97, 0x3f, 0x68, 0x9c, 0x37, 0xeb, 0x63, 0x76, 0xa0, 0x88,
    0xa7, 0xc7, 0x3e, 0x97,
};
static const unsigned char kat238_nor_addin1[] = {
    0xdf, 0xf3, 0xa3, 0xb0, 0x72, 0x43, 0x02, 0xf8, 0x3b, 0x85, 0x4f, 0x2f,
    0xa8, 0xec, 0x34, 0x96,
};
static const unsigned char kat238_nor_retbytes[] = {
    0x66, 0x7d, 0xdb, 0x87, 0x07, 0x9b, 0x39, 0x49, 0x03, 0x0f, 0x41, 0xff,
    0x91, 0xb2, 0xec, 0x0a, 0x59, 0x8a, 0xb2, 0xd5, 0x5c, 0x99, 0xd0, 0x17,
    0xe4, 0xfc, 0x79, 0xe9, 0x4f, 0x15, 0xb4, 0x10, 0xcc, 0x29, 0x70, 0xff,
    0x01, 0x74, 0x82, 0xb2, 0x9e, 0xef, 0x2c, 0x25, 0x0a, 0x09, 0x12, 0x8a,
    0x18, 0x60, 0x5e, 0x0f, 0xa1, 0x4a, 0xcc, 0xe3, 0x7b, 0x89, 0xc4, 0x38,
    0xc8, 0x21, 0x31, 0x94,
};
static const struct drbg_kat_no_reseed kat238_nor_t = {
    12, kat238_nor_entropyin, kat238_nor_nonce, kat238_nor_persstr,
    kat238_nor_addin0, kat238_nor_addin1, kat238_nor_retbytes
};
static const struct drbg_kat kat238_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat238_nor_t
};

static const unsigned char kat239_nor_entropyin[] = {
    0x8e, 0xe3, 0xed, 0xc1, 0xc6, 0x4e, 0xa7, 0xd6, 0xf0, 0xba, 0xb9, 0x73,
    0x1e, 0x5d, 0x17, 0x62,
};
static const unsigned char kat239_nor_nonce[] = {
    0x92, 0xa1, 0x42, 0x26, 0x08, 0x7a, 0xfa, 0xfb,
};
static const unsigned char kat239_nor_persstr[] = {
    0x2a, 0x24, 0x04, 0xbe, 0xf6, 0xf7, 0x11, 0x34, 0x17, 0xdf, 0x0b, 0x18,
    0x54, 0xd8, 0x47, 0x3d,
};
static const unsigned char kat239_nor_addin0[] = {
    0xa2, 0x78, 0xe8, 0xa5, 0x0b, 0xcc, 0xd6, 0x12, 0x97, 0x4d, 0x02, 0x6e,
    0xde, 0xf7, 0x50, 0x1c,
};
static const unsigned char kat239_nor_addin1[] = {
    0xe7, 0x48, 0xf5, 0xe2, 0xf9, 0x8b, 0x48, 0x8e, 0x74, 0xb0, 0xbb, 0xd5,
    0x59, 0xb5, 0xba, 0xbc,
};
static const unsigned char kat239_nor_retbytes[] = {
    0x81, 0xec, 0xf8, 0xb7, 0xb0, 0x7c, 0x64, 0xce, 0xd1, 0xa4, 0x9b, 0x21,
    0x33, 0x54, 0x99, 0x5a, 0x7b, 0xf1, 0x68, 0xd3, 0x8a, 0x83, 0x0b, 0x15,
    0x70, 0x41, 0x4f, 0x5a, 0x20, 0x15, 0x49, 0x9e, 0x09, 0xaf, 0xf3, 0xbf,
    0x58, 0x37, 0x19, 0x88, 0x6f, 0xe9, 0x9d, 0x00, 0xe6, 0xd9, 0xeb, 0x9a,
    0x4f, 0xde, 0x29, 0xd8, 0xd9, 0x78, 0x8d, 0xd6, 0xff, 0xe2, 0x09, 0x83,
    0x9a, 0x73, 0x9c, 0xd2,
};
static const struct drbg_kat_no_reseed kat239_nor_t = {
    13, kat239_nor_entropyin, kat239_nor_nonce, kat239_nor_persstr,
    kat239_nor_addin0, kat239_nor_addin1, kat239_nor_retbytes
};
static const struct drbg_kat kat239_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat239_nor_t
};

static const unsigned char kat240_nor_entropyin[] = {
    0x06, 0x18, 0xc2, 0x88, 0x49, 0x14, 0x71, 0x22, 0x26, 0xad, 0x4e, 0x95,
    0x80, 0xc0, 0x95, 0x23,
};
static const unsigned char kat240_nor_nonce[] = {
    0x6e, 0xbc, 0x4d, 0x5d, 0xb4, 0x76, 0x7f, 0x5f,
};
static const unsigned char kat240_nor_persstr[] = {
    0x7a, 0x82, 0x50, 0xa3, 0xbc, 0xbd, 0x81, 0x54, 0xc5, 0x05, 0x8b, 0x55,
    0x1c, 0xf2, 0x20, 0x36,
};
static const unsigned char kat240_nor_addin0[] = {
    0x30, 0x26, 0x05, 0x32, 0xb5, 0x6c, 0x0f, 0x8e, 0xde, 0x9b, 0x72, 0x5c,
    0x32, 0x20, 0x8f, 0x31,
};
static const unsigned char kat240_nor_addin1[] = {
    0x1e, 0x90, 0xdc, 0x4f, 0x67, 0xc1, 0xcb, 0x2d, 0xa1, 0x2d, 0x10, 0xa0,
    0xb0, 0xfb, 0xb6, 0x26,
};
static const unsigned char kat240_nor_retbytes[] = {
    0x49, 0x93, 0x57, 0x58, 0x80, 0xd0, 0x01, 0x45, 0xc1, 0xf9, 0x67, 0xdb,
    0x25, 0xcc, 0xc8, 0xf3, 0x4c, 0xf0, 0xc6, 0x2a, 0xcc, 0xcb, 0xcf, 0xa0,
    0xe7, 0x2c, 0x89, 0xea, 0xcf, 0x9f, 0xbe, 0x56, 0x14, 0x50, 0x74, 0xe4,
    0xd4, 0x7f, 0x2b, 0x76, 0x74, 0xf4, 0xb0, 0xe0, 0x7e, 0xe3, 0xb5, 0x10,
    0x4b, 0xf7, 0xcc, 0xfe, 0xd8, 0x00, 0x1d, 0x67, 0x11, 0xc6, 0x6f, 0xda,
    0x94, 0x44, 0xfb, 0x8a,
};
static const struct drbg_kat_no_reseed kat240_nor_t = {
    14, kat240_nor_entropyin, kat240_nor_nonce, kat240_nor_persstr,
    kat240_nor_addin0, kat240_nor_addin1, kat240_nor_retbytes
};
static const struct drbg_kat kat240_nor = {
    NO_RESEED, USE_DF, NID_aes_128_ctr, 16, 8, 16, 16, 64, &kat240_nor_t
};

static const unsigned char kat480_nor_entropyin[] = {
    0x3d, 0x17, 0x4d, 0x98, 0xec, 0xa8, 0x0a, 0xe3, 0x9c, 0x75, 0xd2, 0x7f,
    0x97, 0xb9, 0x9e, 0x9b, 0x0f, 0x74, 0x2b, 0x32, 0xe3, 0x7b, 0x94, 0xe8,
};
static const unsigned char kat480_nor_nonce[] = {
    0xbc, 0xfc, 0x46, 0x13, 0x17, 0x5b, 0x6e, 0x35, 0x34, 0x08, 0xe2, 0x3b,
    0x85, 0x81, 0xe7, 0x05,
};
static const unsigned char kat480_nor_persstr[] = {
    0x3d, 0x0e, 0x9f, 0xa5, 0x61, 0x6f, 0xa2, 0x5a, 0x0d, 0x58, 0xdd, 0xbd,
    0xc4, 0xa6, 0xa8, 0xe5, 0x5b, 0xef, 0x05, 0x1d, 0xb6, 0xfe, 0xdc, 0x8b,
    0xe8, 0xd4, 0xc7, 0x52, 0xf1, 0xd2, 0xc9, 0x4a,
};
static const unsigned char kat480_nor_addin0[] = {
    0x87, 0x18, 0xb1, 0x5a, 0x55, 0x0f, 0x97, 0x89, 0x39, 0xf5, 0x82, 0x79,
    0xec, 0xda, 0xaf, 0xe7, 0x98, 0x0a, 0x1b, 0xbf, 0xe6, 0xb9, 0xde, 0x85,
    0xb9, 0xfe, 0x10, 0x7c, 0x72, 0xa8, 0xfe, 0x20,
};
static const unsigned char kat480_nor_addin1[] = {
    0x07, 0x88, 0x99, 0xf5, 0xa1, 0xed, 0xd1, 0xba, 0x8e, 0x10, 0xeb, 0x8e,
    0xfa, 0x2c, 0x98, 0xba, 0x17, 0x4c, 0x1b, 0xd3, 0x5a, 0x74, 0x41, 0x4c,
    0xff, 0x86, 0x23, 0x63, 0x3a, 0xc1, 0xc2, 0x5b,
};
static const unsigned char kat480_nor_retbytes[] = {
    0x74, 0xe1, 0x55, 0xa5, 0x70, 0x7e, 0xb1, 0x9b, 0x79, 0xcf, 0x7e, 0xfc,
    0xe1, 0xc4, 0x23, 0x7b, 0xac, 0x69, 0xe3, 0xf9, 0x31, 0x14, 0x11, 0xc9,
    0x2d, 0xac, 0xfd, 0x8a, 0xc9, 0x2e, 0xa1, 0x1a, 0x87, 0x87, 0xa4, 0x7b,
    0xc4, 0x4f, 0x1c, 0x29, 0x44, 0x44, 0x5b, 0x2b, 0x05, 0x27, 0xe1, 0x6a,
    0xec, 0x6c, 0x01, 0x16, 0x54, 0x82, 0xd7, 0xe3, 0xb6, 0x44, 0x37, 0x09,
    0x3b, 0x87, 0xbf, 0xbc,
};
static const struct drbg_kat_no_reseed kat480_nor_t = {
    14, kat480_nor_entropyin, kat480_nor_nonce, kat480_nor_persstr,
    kat480_nor_addin0, kat480_nor_addin1, kat480_nor_retbytes
};
static const struct drbg_kat kat480_nor = {
    NO_RESEED, USE_DF, NID_aes_192_ctr, 24, 16, 32, 32, 64, &kat480_nor_t
};


static const unsigned char kat720_nor_entropyin[] = {
    0x5f, 0xc1, 0xa4, 0x6b, 0x9b, 0x53, 0xe8, 0x3a, 0xdd, 0xd6, 0x16, 0x51,
    0x79, 0x32, 0xad, 0x8a, 0x84, 0xe5, 0x50, 0xa5, 0x83, 0x27, 0xee, 0x24,
    0x5a, 0xef, 0x20, 0x82, 0x64, 0xc5, 0xb9, 0x1c,
};
static const unsigned char kat720_nor_nonce[] = {
    0x9c, 0xe7, 0x4e, 0x2b, 0x3c, 0x43, 0xba, 0x1d, 0x98, 0xf7, 0xd4, 0xe8,
    0xeb, 0x5c, 0x9b, 0x3e,
};
static const unsigned char kat720_nor_persstr[] = {
    0x5f, 0xe0, 0x1c, 0x84, 0xf8, 0x24, 0xc1, 0x7e, 0x4d, 0xfb, 0xfd, 0xb7,
    0x5c, 0x32, 0x12, 0xcb, 0x2b, 0xfd, 0xf2, 0xda, 0x7c, 0x49, 0x7f, 0xeb,
    0xc0, 0x96, 0x74, 0x75, 0xeb, 0xe2, 0x91, 0xf7,
};
static const unsigned char kat720_nor_addin0[] = {
    0x84, 0x92, 0x2c, 0x03, 0x35, 0xa0, 0xea, 0xd6, 0x09, 0xe5, 0xa9, 0x2c,
    0xfc, 0x4a, 0x22, 0x5b, 0xd3, 0xc7, 0xc0, 0x1a, 0xb4, 0x58, 0x0b, 0x78,
    0x63, 0x38, 0xe1, 0xca, 0xa3, 0x62, 0x2f, 0x44,
};
static const unsigned char kat720_nor_addin1[] = {
    0x34, 0xb0, 0x34, 0xca, 0x64, 0x3b, 0xbf, 0xd2, 0xfc, 0xc5, 0x7c, 0x9b,
    0x53, 0xe0, 0xf9, 0xb3, 0xfd, 0x6a, 0x73, 0x45, 0x4f, 0x18, 0x23, 0xdf,
    0xe7, 0xb7, 0x07, 0x6e, 0xc7, 0x3f, 0xd9, 0x56,
};
static const unsigned char kat720_nor_retbytes[] = {
    0x6c, 0xaa, 0x44, 0x75, 0x18, 0x9e, 0xe0, 0x0d, 0xdc, 0x54, 0x91, 0x0f,
    0x87, 0x23, 0xb0, 0xe5, 0xb8, 0xd3, 0xd0, 0xc3, 0x21, 0xce, 0x7f, 0x2a,
    0xc7, 0x19, 0x4d, 0x13, 0x4a, 0x0a, 0x31, 0xd9, 0x6b, 0x10, 0x2d, 0xd5,
    0x8e, 0x09, 0x2e, 0x08, 0xf8, 0xa0, 0x08, 0xf0, 0x5c, 0x4f, 0x2a, 0xfb,
    0x29, 0x01, 0xc6, 0xe2, 0x95, 0x49, 0xd3, 0xa7, 0x20, 0xaa, 0x2b, 0x1d,
    0x1f, 0x46, 0x1b, 0xb9,
};
static const struct drbg_kat_no_reseed kat720_nor_t = {
    14, kat720_nor_entropyin, kat720_nor_nonce, kat720_nor_persstr,
    kat720_nor_addin0, kat720_nor_addin1, kat720_nor_retbytes
};
static const struct drbg_kat kat720_nor = {
    NO_RESEED, USE_DF, NID_aes_256_ctr, 32, 16, 32, 32, 64, &kat720_nor_t
};

static const unsigned char kat960_nor_entropyin[] = {
    0x22, 0x54, 0x60, 0x44, 0xdf, 0x54, 0xe2, 0xbf, 0xe7, 0x46, 0x93, 0x18,
    0xc5, 0x7c, 0x15, 0x9d, 0x0f, 0x25, 0xa5, 0x8b, 0xcf, 0x4f, 0xbe, 0x70,
    0xae, 0x21, 0x4b, 0x66, 0x9b, 0x2e, 0x63, 0x14,
};
static const unsigned char kat960_nor_nonce[] = {0};
static const unsigned char kat960_nor_persstr[] = {
    0x28, 0xcf, 0x3c, 0xa7, 0xdf, 0x24, 0xcd, 0x72, 0x70, 0x4c, 0xc7, 0x3b,
    0x4a, 0x61, 0x8c, 0xad, 0xb6, 0xac, 0x93, 0xf2, 0x16, 0x8c, 0x47, 0x9c,
    0x35, 0x8d, 0xe1, 0x90, 0x7c, 0x0a, 0x0c, 0x82,
};
static const unsigned char kat960_nor_addin0[] = {
    0x36, 0x55, 0x1e, 0xf0, 0x92, 0x32, 0xd2, 0x19, 0x95, 0x47, 0xaa, 0xef,
    0xec, 0xaa, 0xad, 0x21, 0x74, 0x43, 0xd6, 0x16, 0x43, 0x3d, 0x9d, 0x16,
    0x9b, 0xd8, 0xcd, 0x3e, 0xae, 0x7f, 0x91, 0xe2,
};
static const unsigned char kat960_nor_addin1[] = {
    0x9e, 0x57, 0xa4, 0xe9, 0x6a, 0xce, 0x48, 0x3d, 0xbc, 0x3c, 0x22, 0x6d,
    0x27, 0x23, 0xc9, 0x25, 0x80, 0x63, 0x27, 0x81, 0x40, 0xd2, 0x20, 0xc4,
    0xfd, 0x02, 0x3c, 0x77, 0xfb, 0x20, 0xb8, 0x4c,
};
static const unsigned char kat960_nor_retbytes[] = {
    0xca, 0x43, 0xdc, 0xec, 0xda, 0xb6, 0x89, 0x54, 0x9b, 0xc4, 0x49, 0x3a,
    0x38, 0xa6, 0xa0, 0x17, 0xf6, 0x04, 0x82, 0x70, 0xe5, 0xd7, 0x0d, 0x0d,
    0x75, 0x76, 0xa2, 0xab, 0x76, 0x4d, 0x92, 0x2b, 0xc3, 0x46, 0xe3, 0x0f,
    0x42, 0xdc, 0x5a, 0x73, 0xea, 0x6a, 0x21, 0x0d, 0x2e, 0xba, 0x9b, 0xee,
    0x5b, 0x5a, 0xfe, 0x3c, 0x66, 0x86, 0x7a, 0x68, 0xab, 0x3f, 0xe0, 0xd0,
    0x8f, 0x51, 0x11, 0x01,
};
static const struct drbg_kat_no_reseed kat960_nor_t = {
    14, kat960_nor_entropyin, kat960_nor_nonce, kat960_nor_persstr,
    kat960_nor_addin0, kat960_nor_addin1, kat960_nor_retbytes
};
static const struct drbg_kat kat960_nor = {
    NO_RESEED, NO_DF, NID_aes_128_ctr, 32, 0, 32, 32, 64, &kat960_nor_t
};

static const unsigned char kat1200_nor_entropyin[] = {
    0x52, 0x40, 0x12, 0xbf, 0xc4, 0x7b, 0xed, 0x3d, 0xa7, 0xa3, 0x54, 0x88,
    0xe8, 0x26, 0xc4, 0x2a, 0x76, 0x60, 0x7c, 0x0d, 0x98, 0x0c, 0xa0, 0x21,
    0xc9, 0xea, 0x9c, 0xe0, 0xcf, 0x6d, 0xce, 0xd5, 0xc8, 0xea, 0x0c, 0x61,
    0x77, 0x2e, 0x4d, 0x64,
};
static const unsigned char kat1200_nor_nonce[] = {0};
static const unsigned char kat1200_nor_persstr[] = {
    0x3a, 0x27, 0xc2, 0x5e, 0xb5, 0x94, 0x30, 0x28, 0xc0, 0x19, 0x96, 0xb5,
    0x76, 0x6a, 0xf3, 0x39, 0x06, 0x1b, 0xee, 0xd2, 0xc8, 0xd8, 0xe7, 0x73,
    0xfe, 0x08, 0xba, 0xde, 0x72, 0x80, 0x25, 0xd4, 0x73, 0xe2, 0x73, 0x39,
    0xb6, 0x8d, 0xe5, 0xe6,
};
static const unsigned char kat1200_nor_addin0[] = {
    0x8b, 0x98, 0x1e, 0xc9, 0xca, 0x88, 0xb1, 0x49, 0x3e, 0x7f, 0xf3, 0xb9,
    0x0c, 0x02, 0xda, 0x6f, 0x47, 0x8b, 0xfa, 0x57, 0x3f, 0x5a, 0x03, 0x54,
    0x94, 0x1d, 0xfe, 0xdb, 0x86, 0x3c, 0xa9, 0xd0, 0x5e, 0xfe, 0xa9, 0xd8,
    0x83, 0x54, 0x59, 0xad,
};
static const unsigned char kat1200_nor_addin1[] = {
    0x82, 0x9e, 0x75, 0xa5, 0x8e, 0xdd, 0x00, 0xd8, 0x62, 0x69, 0xef, 0x33,
    0x2e, 0x67, 0x44, 0x72, 0x3b, 0x28, 0x9f, 0x7d, 0xf8, 0xf1, 0xc0, 0xbb,
    0xf7, 0x02, 0x22, 0xb5, 0x42, 0xb9, 0x01, 0x4e, 0x2d, 0x0c, 0xdd, 0x6a,
    0xae, 0xc8, 0xc1, 0x94,
};
static const unsigned char kat1200_nor_retbytes[] = {
    0x8c, 0x4a, 0xa7, 0x94, 0xaf, 0x3d, 0x7d, 0x4d, 0x68, 0x40, 0x06, 0x80,
    0x8c, 0x98, 0xc1, 0x1d, 0x81, 0x46, 0xb1, 0x1f, 0xd0, 0x62, 0xc6, 0x9c,
    0xac, 0x01, 0x9f, 0x19, 0x13, 0xc4, 0x57, 0xb4, 0x9d, 0x42, 0x3b, 0x5e,
    0xc6, 0x83, 0xf1, 0x91, 0x43, 0xea, 0xb3, 0x72, 0x07, 0x9a, 0x6d, 0xf5,
    0x51, 0xfc, 0x68, 0x6d, 0x9d, 0x6f, 0x9c, 0xe5, 0xf6, 0x4e, 0xf6, 0x19,
    0x18, 0x6f, 0x81, 0x6b,
};
static const struct drbg_kat_no_reseed kat1200_nor_t = {
    14, kat1200_nor_entropyin, kat1200_nor_nonce, kat1200_nor_persstr,
    kat1200_nor_addin0, kat1200_nor_addin1, kat1200_nor_retbytes
};
static const struct drbg_kat kat1200_nor = {
    NO_RESEED, NO_DF, NID_aes_192_ctr, 40, 0, 40, 40, 64, &kat1200_nor_t
};

static const unsigned char kat1440_nor_entropyin[] = {
    0x65, 0x62, 0xbb, 0xb3, 0x98, 0xc0, 0x2c, 0x7d, 0xdf, 0x0b, 0xdc, 0xbf,
    0xea, 0xcd, 0x5a, 0xfe, 0x31, 0xc7, 0x75, 0xbd, 0xa5, 0x3b, 0x4c, 0xb2,
    0xa3, 0x89, 0x6a, 0x89, 0x81, 0x49, 0xbb, 0x19, 0x5f, 0xa3, 0xdd, 0x69,
    0x23, 0x7b, 0xfa, 0xbe, 0x3a, 0x4c, 0x2d, 0xae, 0xbf, 0x30, 0xdc, 0x17,
};
static const unsigned char kat1440_nor_nonce[] = {0};
static const unsigned char kat1440_nor_persstr[] = {
    0x96, 0x3e, 0x80, 0xb8, 0x4c, 0xeb, 0x20, 0x45, 0xf9, 0xb5, 0x21, 0x92,
    0x39, 0xb4, 0x21, 0x39, 0xaa, 0xc6, 0x75, 0xca, 0x0b, 0x59, 0x1b, 0x25,
    0x53, 0xe8, 0x39, 0x64, 0x7a, 0xba, 0x20, 0x84, 0x99, 0x3b, 0x07, 0x65,
    0x45, 0x6a, 0xe9, 0x07, 0xaa, 0x5d, 0x6d, 0xa3, 0x44, 0x87, 0x43, 0x2a,
};
static const unsigned char kat1440_nor_addin0[] = {
    0x59, 0xcc, 0xb9, 0x0f, 0xb7, 0xe9, 0xae, 0x3a, 0x53, 0x2f, 0x95, 0x6b,
    0xc0, 0x03, 0x8c, 0xe0, 0xae, 0x32, 0xfa, 0x71, 0x4d, 0x88, 0x36, 0xc4,
    0x78, 0x91, 0x98, 0x3a, 0xfb, 0x68, 0x2e, 0xd2, 0x33, 0x56, 0x00, 0x63,
    0x9d, 0xa1, 0xd2, 0xa9, 0xe8, 0x9a, 0x95, 0x7a, 0xbc, 0x5b, 0x3d, 0x50,
};
static const unsigned char kat1440_nor_addin1[] = {
    0x91, 0x16, 0x5e, 0x46, 0x01, 0x6f, 0x2d, 0x56, 0x19, 0xcd, 0x45, 0x49,
    0xf5, 0xac, 0x20, 0x56, 0x39, 0x6b, 0xee, 0x73, 0x16, 0xa3, 0x66, 0xec,
    0x60, 0x51, 0xf3, 0xbd, 0x26, 0x77, 0x0b, 0x3f, 0xd6, 0x3c, 0x16, 0x34,
    0x94, 0xfb, 0x7e, 0x9d, 0x32, 0x6e, 0xca, 0xd9, 0xbf, 0xc1, 0x71, 0xa8,
};
static const unsigned char kat1440_nor_retbytes[] = {
    0x87, 0xed, 0x4a, 0xef, 0x26, 0x4f, 0x2a, 0xc5, 0x45, 0x10, 0xb8, 0x3b,
    0x98, 0xf7, 0x58, 0x5a, 0xef, 0x7f, 0x38, 0x29, 0x47, 0xcb, 0x49, 0x80,
    0x37, 0x84, 0xa2, 0x7c, 0xfb, 0xbc, 0x58, 0x74, 0x56, 0x2b, 0x6d, 0x6c,
    0xb5, 0x7b, 0x75, 0x48, 0xa1, 0xbe, 0x46, 0x4a, 0xbb, 0x20, 0x28, 0xc3,
    0x44, 0x07, 0x45, 0x27, 0xe8, 0xec, 0x56, 0xad, 0x6e, 0x2b, 0x3b, 0x8f,
    0xc5, 0x6f, 0x17, 0x7c,
};
static const struct drbg_kat_no_reseed kat1440_nor_t = {
    14, kat1440_nor_entropyin, kat1440_nor_nonce, kat1440_nor_persstr,
    kat1440_nor_addin0, kat1440_nor_addin1, kat1440_nor_retbytes
};
static const struct drbg_kat kat1440_nor = {
    NO_RESEED, NO_DF, NID_aes_256_ctr, 48, 0, 48, 48, 64, &kat1440_nor_t
};


static const unsigned char kat1_prt_entropyin[] = {
    0x5d, 0x40, 0x41, 0x94, 0x2b, 0xcf, 0x68, 0x86, 0x4a, 0x49, 0x97, 0xd8,
    0x17, 0x1f, 0x1f, 0x9f,
};
static const unsigned char kat1_prt_nonce[] = {
    0xd4, 0xf1, 0xf4, 0xae, 0x08, 0xbc, 0xb3, 0xe1,
};
static const unsigned char kat1_prt_persstr[] = {0};
static const unsigned char kat1_prt_entropyinpr0[] = {
    0xef, 0x55, 0xa7, 0x69, 0xb7, 0xea, 0xf0, 0x3f, 0xe0, 0x82, 0x02, 0x9b,
    0xb3, 0x2a, 0x2b, 0x9d,
};
static const unsigned char kat1_prt_entropyinpr1[] = {
    0x82, 0x39, 0xe8, 0x65, 0xc0, 0xa4, 0x2e, 0x14, 0xb9, 0x64, 0xb9, 0xc0,
    0x9d, 0xe8, 0x5a, 0x20,
};
static const unsigned char kat1_prt_addin0[] = {0};
static const unsigned char kat1_prt_addin1[] = {0};
static const unsigned char kat1_prt_retbytes[] = {
    0x41, 0x55, 0x32, 0x02, 0x87, 0xee, 0xdc, 0xf7, 0xd4, 0x84, 0xc2, 0xc2,
    0xa1, 0xe2, 0xeb, 0x64, 0xb9, 0xc9, 0xce, 0x77, 0xc8, 0x72, 0x02, 0xa1,
    0xae, 0x16, 0x16, 0xc7, 0xa5, 0xcf, 0xd1, 0xc6, 0x87, 0xc7, 0xa0, 0xbf,
    0xcc, 0x85, 0xbd, 0xa4, 0x8f, 0xdd, 0x46, 0x29, 0xfd, 0x33, 0x0c, 0x22,
    0xd0, 0xa7, 0x60, 0x76, 0xf8, 0x8f, 0xc7, 0xcd, 0x04, 0x03, 0x7e, 0xe0,
    0x6b, 0x7a, 0xf6, 0x02,
};
static const struct drbg_kat_pr_true kat1_prt_t = {
    0, kat1_prt_entropyin, kat1_prt_nonce, kat1_prt_persstr,
    kat1_prt_entropyinpr0, kat1_prt_addin0, kat1_prt_entropyinpr1,
    kat1_prt_addin1, kat1_prt_retbytes
};
static const struct drbg_kat kat1_prt = {
    PR_TRUE, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat1_prt_t
};

static const unsigned char kat1440_prt_entropyin[] = {
    0x5a, 0x6e, 0x33, 0xd1, 0x56, 0x67, 0x27, 0x28, 0x14, 0xb1, 0xb8, 0x3e,
    0x37, 0x24, 0xf9, 0x95, 0xaf, 0x0f, 0xb0, 0x5c, 0x24, 0x72, 0x03, 0xc3,
    0x4a, 0x07, 0xa6, 0x74, 0x36, 0xe7, 0xde, 0xdc, 0xd7, 0x17, 0xa2, 0x4c,
    0xa4, 0xa9, 0xd9, 0xf9, 0xd4, 0x7a, 0x6b, 0x6b, 0x84, 0xb6, 0x3b, 0x88,
};
static const unsigned char kat1440_prt_nonce[] = {0};
static const unsigned char kat1440_prt_persstr[] = {
    0xf3, 0xfe, 0x7a, 0xd7, 0xd0, 0xd6, 0xb8, 0x60, 0x93, 0x12, 0xf0, 0x49,
    0xf7, 0x5a, 0x2e, 0x9c, 0xe6, 0x43, 0xc9, 0x35, 0xd8, 0x2a, 0xce, 0x41,
    0xdb, 0xa5, 0xb0, 0xc1, 0xc9, 0x85, 0x68, 0x43, 0x28, 0xfe, 0x31, 0xc4,
    0xe7, 0x7d, 0xb6, 0x0f, 0x25, 0x14, 0x38, 0xed, 0x74, 0x83, 0xef, 0x19,
};
static const unsigned char kat1440_prt_entropyinpr0[] = {
    0xb4, 0xe1, 0x56, 0xfe, 0x5a, 0x63, 0x56, 0xcf, 0xcb, 0x2c, 0xbb, 0x79,
    0xa1, 0x81, 0xe4, 0x0a, 0x1d, 0x8f, 0xd0, 0xe1, 0x0f, 0xc4, 0x77, 0x58,
    0x93, 0xb8, 0x51, 0x32, 0xed, 0xe0, 0xca, 0x22, 0x26, 0x8e, 0xad, 0x5a,
    0x40, 0xb6, 0xde, 0x69, 0x84, 0xce, 0xe2, 0xfc, 0xa6, 0x2f, 0xdb, 0xee,
};
static const unsigned char kat1440_prt_entropyinpr1[] = {
    0xd0, 0x04, 0xfa, 0x7a, 0xe5, 0x59, 0x65, 0x6f, 0x6a, 0x8f, 0x85, 0x25,
    0xc6, 0xe5, 0x5d, 0x24, 0x5c, 0x8d, 0x7f, 0x58, 0x0c, 0xfe, 0x63, 0xba,
    0x84, 0x57, 0x5c, 0x5b, 0x6e, 0xf8, 0xbb, 0xf6, 0x13, 0x0f, 0xda, 0x8f,
    0x01, 0xfd, 0x87, 0x30, 0x79, 0x23, 0xe0, 0x63, 0x9a, 0x23, 0xf4, 0xe8,
};
static const unsigned char kat1440_prt_addin0[] = {
    0x56, 0x1d, 0x2a, 0x23, 0x52, 0xab, 0x0e, 0x73, 0xf3, 0xec, 0x11, 0x35,
    0xe4, 0xf2, 0xdd, 0x22, 0x6d, 0x5f, 0x3e, 0x4b, 0xc3, 0xc4, 0xd8, 0xc4,
    0x6d, 0xe8, 0x66, 0x44, 0x65, 0x99, 0x44, 0x6d, 0x0f, 0x15, 0x8b, 0xda,
    0x30, 0xc3, 0xee, 0xc2, 0x03, 0x3d, 0xbe, 0x09, 0x92, 0x4c, 0x94, 0xb1,
};
static const unsigned char kat1440_prt_addin1[] = {
    0x1f, 0xc6, 0x76, 0x04, 0x93, 0x0d, 0xec, 0xe4, 0x93, 0x7f, 0x2b, 0x01,
    0xaa, 0x4c, 0x44, 0xeb, 0x63, 0x70, 0xa5, 0xfb, 0xe1, 0xe1, 0x68, 0xa8,
    0x4c, 0x26, 0xfa, 0x3c, 0x6d, 0xd0, 0xfd, 0xa4, 0xe2, 0x07, 0x16, 0x96,
    0xe3, 0x24, 0x6d, 0x34, 0x85, 0xf4, 0xd5, 0xf1, 0x9c, 0x6b, 0x18, 0x53,
};
static const unsigned char kat1440_prt_retbytes[] = {
    0x65, 0xfa, 0x1b, 0x5c, 0xd3, 0xbc, 0x6f, 0x0a, 0x6a, 0x89, 0x90, 0x14,
    0xf8, 0xdd, 0x94, 0xc6, 0x24, 0xcb, 0xd6, 0xde, 0xef, 0xa5, 0x0b, 0x3f,
    0x65, 0xa1, 0x06, 0x7a, 0x1a, 0xfb, 0x95, 0x2f, 0x13, 0x4f, 0xf7, 0x33,
    0x35, 0x81, 0x84, 0x33, 0x52, 0x60, 0xd2, 0x85, 0xc1, 0x76, 0x71, 0x8f,
    0xcc, 0x58, 0xf0, 0xd2, 0x82, 0xa2, 0x48, 0x11, 0xb8, 0x60, 0x8f, 0x3d,
    0x75, 0x3d, 0xe0, 0x68,
};
static const struct drbg_kat_pr_true kat1440_prt_t = {
    14, kat1440_prt_entropyin, kat1440_prt_nonce, kat1440_prt_persstr,
    kat1440_prt_entropyinpr0, kat1440_prt_addin0, kat1440_prt_entropyinpr1,
    kat1440_prt_addin1, kat1440_prt_retbytes
};
static const struct drbg_kat kat1440_prt = {
    PR_TRUE, NO_DF, NID_aes_256_ctr, 48, 0, 48, 48, 64, &kat1440_prt_t
};


static const unsigned char kat1_prf_entropyin[] = {
    0x0f, 0x65, 0xda, 0x13, 0xdc, 0xa4, 0x07, 0x99, 0x9d, 0x47, 0x73, 0xc2,
    0xb4, 0xa1, 0x1d, 0x85,
};
static const unsigned char kat1_prf_nonce[] = {
    0x52, 0x09, 0xe5, 0xb4, 0xed, 0x82, 0xa2, 0x34,
};
static const unsigned char kat1_prf_persstr[] = {0};
static const unsigned char kat1_prf_entropyin_reseed[] = {
    0x1d, 0xea, 0x0a, 0x12, 0xc5, 0x2b, 0xf6, 0x43, 0x39, 0xdd, 0x29, 0x1c,
    0x80, 0xd8, 0xca, 0x89,
};
static const unsigned char kat1_prf_addin_reseed[] = {0};
static const unsigned char kat1_prf_addin0[] = {0};
static const unsigned char kat1_prf_addin1[] = {0};
static const unsigned char kat1_prf_retbytes[] = {
    0x28, 0x59, 0xcc, 0x46, 0x8a, 0x76, 0xb0, 0x86, 0x61, 0xff, 0xd2, 0x3b,
    0x28, 0x54, 0x7f, 0xfd, 0x09, 0x97, 0xad, 0x52, 0x6a, 0x0f, 0x51, 0x26,
    0x1b, 0x99, 0xed, 0x3a, 0x37, 0xbd, 0x40, 0x7b, 0xf4, 0x18, 0xdb, 0xe6,
    0xc6, 0xc3, 0xe2, 0x6e, 0xd0, 0xdd, 0xef, 0xcb, 0x74, 0x74, 0xd8, 0x99,
    0xbd, 0x99, 0xf3, 0x65, 0x54, 0x27, 0x51, 0x9f, 0xc5, 0xb4, 0x05, 0x7b,
    0xca, 0xf3, 0x06, 0xd4,
};
static const struct drbg_kat_pr_false kat1_prf_t = {
    0, kat1_prf_entropyin, kat1_prf_nonce, kat1_prf_persstr,
    kat1_prf_entropyin_reseed, kat1_prf_addin_reseed,
    kat1_prf_addin0, kat1_prf_addin1, kat1_prf_retbytes
};
static const struct drbg_kat kat1_prf = {
    PR_FALSE, USE_DF, NID_aes_128_ctr, 16, 8, 0, 0, 64, &kat1_prf_t
};

static const unsigned char kat1440_prf_entropyin[] = {
    0x88, 0x2e, 0xd0, 0x54, 0x87, 0xdc, 0xe4, 0xb4, 0xf9, 0xe5, 0x8e, 0xc4,
    0xf2, 0xda, 0x1f, 0xa5, 0xd8, 0xeb, 0xb4, 0xef, 0x9f, 0xcb, 0xf7, 0xb0,
    0xa0, 0xe1, 0x5c, 0x4d, 0xcc, 0xb8, 0xe1, 0x97, 0x88, 0xf8, 0x6d, 0xcf,
    0x28, 0x85, 0xe7, 0x1a, 0x20, 0xcb, 0xd9, 0xac, 0x10, 0xa6, 0x64, 0x8e,
};
static const unsigned char kat1440_prf_nonce[] = {0};
static const unsigned char kat1440_prf_persstr[] = {
    0x05, 0xf5, 0xbc, 0x41, 0x68, 0x7e, 0xa1, 0xe4, 0xc3, 0x4a, 0x69, 0x94,
    0x4f, 0xba, 0xe2, 0x83, 0xcf, 0xee, 0x4c, 0x42, 0xb1, 0xbb, 0xd7, 0x8f,
    0xdd, 0xb0, 0x97, 0x3d, 0x0f, 0xca, 0x94, 0x85, 0x39, 0xb6, 0x84, 0x36,
    0x58, 0xb6, 0x7c, 0x30, 0xb7, 0x31, 0x91, 0xb9, 0xa0, 0xbf, 0x29, 0x21,
};
static const unsigned char kat1440_prf_entropyin_reseed[] = {
    0xca, 0x16, 0x03, 0xd4, 0xc8, 0x71, 0x14, 0x04, 0xc7, 0xbd, 0xc1, 0x2c,
    0x7c, 0x75, 0xb2, 0x94, 0x3a, 0x4b, 0x04, 0x2e, 0xa1, 0xd2, 0xeb, 0x54,
    0x50, 0x6a, 0x68, 0x76, 0x95, 0x21, 0x57, 0xca, 0xf3, 0xb1, 0x52, 0xdc,
    0x75, 0xf2, 0x7f, 0x22, 0x13, 0x64, 0x5a, 0x14, 0x15, 0x77, 0xe8, 0xba,
};
static const unsigned char kat1440_prf_addin_reseed[] = {
    0x83, 0xcd, 0xa5, 0x33, 0x80, 0x88, 0x8d, 0x53, 0x51, 0x5e, 0x58, 0x15,
    0x4f, 0x89, 0xd5, 0x52, 0x8a, 0xb6, 0x9f, 0x31, 0xfb, 0xcf, 0xca, 0x34,
    0x98, 0x8c, 0xf0, 0x3c, 0x4c, 0xae, 0x5f, 0x60, 0xaa, 0x62, 0x91, 0xf3,
    0x2d, 0x99, 0xab, 0x2a, 0x72, 0x6b, 0x6e, 0x08, 0xd2, 0x50, 0x2c, 0xf5,
};
static const unsigned char kat1440_prf_addin0[] = {
    0x5b, 0xf5, 0xca, 0x9f, 0x96, 0x4e, 0xdd, 0x91, 0xe8, 0xef, 0x49, 0x1f,
    0xd3, 0xcd, 0x32, 0xfa, 0xf9, 0xcb, 0x9d, 0x19, 0x93, 0xd8, 0x22, 0x19,
    0x14, 0xd1, 0x75, 0x1f, 0xb0, 0xd4, 0x25, 0x2a, 0x5c, 0xa9, 0x50, 0xe2,
    0x13, 0xf0, 0x88, 0x05, 0x09, 0x00, 0xb2, 0xbd, 0x74, 0xf5, 0xe3, 0x36,
};
static const unsigned char kat1440_prf_addin1[] = {
    0xdb, 0xa2, 0x8d, 0xc1, 0xd8, 0xd6, 0x15, 0x65, 0x15, 0x47, 0x86, 0x7d,
    0x4e, 0xf4, 0x25, 0x19, 0x04, 0x5e, 0xe1, 0x63, 0x78, 0x14, 0x36, 0x85,
    0x10, 0x1d, 0xa4, 0x7a, 0x27, 0xb5, 0x54, 0x98, 0x07, 0x8e, 0x8a, 0x8f,
    0x48, 0x54, 0x05, 0x2f, 0x7c, 0xc6, 0xf5, 0xb0, 0x2e, 0x57, 0x1a, 0xe8,
};
static const unsigned char kat1440_prf_retbytes[] = {
    0x01, 0xf1, 0x19, 0x71, 0x83, 0x58, 0x19, 0xc1, 0x14, 0x8a, 0xa0, 0x79,
    0xee, 0xa0, 0x9f, 0xd5, 0xb1, 0xaa, 0x3a, 0xc6, 0xba, 0x55, 0x7a, 0xe3,
    0x31, 0x7b, 0x1a, 0x33, 0xf4, 0x50, 0x51, 0x74, 0xcf, 0x9d, 0x7e, 0x94,
    0x08, 0x21, 0xc9, 0xb0, 0xe5, 0x52, 0x7a, 0x1d, 0x3e, 0x18, 0x6a, 0x7a,
    0x83, 0xf1, 0x87, 0xc6, 0x2d, 0x32, 0x23, 0xcf, 0x59, 0x64, 0xff, 0x95,
    0x26, 0xd8, 0x48, 0x4c,
};
static const struct drbg_kat_pr_false kat1440_prf_t = {
    14, kat1440_prf_entropyin, kat1440_prf_nonce, kat1440_prf_persstr,
    kat1440_prf_entropyin_reseed, kat1440_prf_addin_reseed,
    kat1440_prf_addin0, kat1440_prf_addin1, kat1440_prf_retbytes
};
static const struct drbg_kat kat1440_prf = {
    PR_FALSE, NO_DF, NID_aes_256_ctr, 48, 0, 48, 48, 64, &kat1440_prf_t
};

const struct drbg_kat *drbg_ctr_test[] = {
    &kat1_nor, &kat2_nor, &kat3_nor, &kat4_nor, &kat5_nor,
    &kat6_nor, &kat7_nor, &kat8_nor, &kat9_nor, &kat10_nor,
    &kat11_nor, &kat12_nor, &kat13_nor, &kat14_nor, &kat15_nor,
    &kat16_nor, &kat17_nor, &kat18_nor, &kat19_nor, &kat20_nor,
    &kat21_nor, &kat22_nor, &kat23_nor, &kat24_nor, &kat25_nor,
    &kat26_nor, &kat27_nor, &kat28_nor, &kat29_nor, &kat30_nor,
    &kat31_nor, &kat32_nor, &kat33_nor, &kat34_nor, &kat35_nor,
    &kat36_nor, &kat37_nor, &kat38_nor, &kat39_nor, &kat40_nor,
    &kat41_nor, &kat42_nor, &kat43_nor, &kat44_nor, &kat45_nor,
    &kat46_nor, &kat47_nor, &kat48_nor, &kat49_nor, &kat50_nor,
    &kat51_nor, &kat52_nor, &kat53_nor, &kat54_nor, &kat55_nor,
    &kat56_nor, &kat57_nor, &kat58_nor, &kat59_nor, &kat60_nor,
    &kat61_nor, &kat62_nor, &kat63_nor, &kat64_nor, &kat65_nor,
    &kat66_nor, &kat67_nor, &kat68_nor, &kat69_nor, &kat70_nor,
    &kat71_nor, &kat72_nor, &kat73_nor, &kat74_nor, &kat75_nor,
    &kat76_nor, &kat77_nor, &kat78_nor, &kat79_nor, &kat80_nor,
    &kat81_nor, &kat82_nor, &kat83_nor, &kat84_nor, &kat85_nor,
    &kat86_nor, &kat87_nor, &kat88_nor, &kat89_nor, &kat90_nor,
    &kat91_nor, &kat92_nor, &kat93_nor, &kat94_nor, &kat95_nor,
    &kat96_nor, &kat97_nor, &kat98_nor, &kat99_nor, &kat100_nor,
    &kat101_nor, &kat102_nor, &kat103_nor, &kat104_nor, &kat105_nor,
    &kat106_nor, &kat107_nor, &kat108_nor, &kat109_nor, &kat110_nor,
    &kat111_nor, &kat112_nor, &kat113_nor, &kat114_nor, &kat115_nor,
    &kat116_nor, &kat117_nor, &kat118_nor, &kat119_nor, &kat120_nor,
    &kat121_nor, &kat122_nor, &kat123_nor, &kat124_nor, &kat125_nor,
    &kat126_nor, &kat127_nor, &kat128_nor, &kat129_nor, &kat130_nor,
    &kat131_nor, &kat132_nor, &kat133_nor, &kat134_nor, &kat135_nor,
    &kat136_nor, &kat137_nor, &kat138_nor, &kat139_nor, &kat140_nor,
    &kat141_nor, &kat142_nor, &kat143_nor, &kat144_nor, &kat145_nor,
    &kat146_nor, &kat147_nor, &kat148_nor, &kat149_nor, &kat150_nor,
    &kat151_nor, &kat152_nor, &kat153_nor, &kat154_nor, &kat155_nor,
    &kat156_nor, &kat157_nor, &kat158_nor, &kat159_nor, &kat160_nor,
    &kat161_nor, &kat162_nor, &kat163_nor, &kat164_nor, &kat165_nor,
    &kat166_nor, &kat167_nor, &kat168_nor, &kat169_nor, &kat170_nor,
    &kat171_nor, &kat172_nor, &kat173_nor, &kat174_nor, &kat175_nor,
    &kat176_nor, &kat177_nor, &kat178_nor, &kat179_nor, &kat180_nor,
    &kat181_nor, &kat182_nor, &kat183_nor, &kat184_nor, &kat185_nor,
    &kat186_nor, &kat187_nor, &kat188_nor, &kat189_nor, &kat190_nor,
    &kat191_nor, &kat192_nor, &kat193_nor, &kat194_nor, &kat195_nor,
    &kat196_nor, &kat197_nor, &kat198_nor, &kat199_nor, &kat200_nor,
    &kat201_nor, &kat202_nor, &kat203_nor, &kat204_nor, &kat205_nor,
    &kat206_nor, &kat207_nor, &kat208_nor, &kat209_nor, &kat210_nor,
    &kat211_nor, &kat212_nor, &kat213_nor, &kat214_nor, &kat215_nor,
    &kat216_nor, &kat217_nor, &kat218_nor, &kat219_nor, &kat220_nor,
    &kat221_nor, &kat222_nor, &kat223_nor, &kat224_nor, &kat225_nor,
    &kat226_nor, &kat227_nor, &kat228_nor, &kat229_nor, &kat230_nor,
    &kat231_nor, &kat232_nor, &kat233_nor, &kat234_nor, &kat235_nor,
    &kat236_nor, &kat237_nor, &kat238_nor, &kat239_nor, &kat240_nor,
    &kat480_nor, &kat720_nor, &kat960_nor, &kat1200_nor, &kat1440_nor,
    &kat1_prt,  &kat1440_prt, &kat1_prf,   &kat1440_prf
};
const size_t drbg_ctr_nelem = OSSL_NELEM(drbg_ctr_test);
