import {
  breakParent,
  group,
  hardline,
  indent,
  line,
} from "../document/index.js";
import htmlWhitespace from "../utilities/html-whitespace.js";
import printAngularControlFlowBlockParameters from "./embed/angular-control-flow-block-parameters.js";
import printAttribute from "./embed/attribute.js";
import { formatAttributeValue } from "./embed/utilities.js";
import getNodeContent from "./get-node-content.js";
import {
  needsToBorrowPrevClosingTagEndMarker,
  printClosingTag,
  printClosingTagSuffix,
  printOpeningTag,
  printOpeningTagPrefix,
} from "./print/tag.js";
import {
  htmlTrimPreserveIndentation,
  inferElementParser,
  isScriptLikeTag,
  isVueNonHtmlBlock,
} from "./utilities/index.js";
import isVueSfcWithTypescriptScript from "./utilities/is-vue-sfc-with-typescript-script.js";

const embeddedAngularControlFlowBlocks = new Set([
  "if",
  "else if",
  "for",
  "switch",
  "case",
]);

function embed(path, options) {
  const { node } = path;

  switch (node.kind) {
    case "element":
      if (isScriptLikeTag(node, options) || node.kind === "interpolation") {
        // Fall through to "text"
        return;
      }

      if (!node.isSelfClosing && isVueNonHtmlBlock(node, options)) {
        const parser = inferElementParser(node, options);
        if (!parser) {
          return;
        }

        return async (textToDoc, print) => {
          const content = getNodeContent(node, options);
          let isEmpty = /^\s*$/u.test(content);
          let doc = "";
          if (!isEmpty) {
            doc = await textToDoc(htmlTrimPreserveIndentation(content), {
              parser,
              __embeddedInHtml: true,
            });
            isEmpty = doc === "";
          }

          return [
            printOpeningTagPrefix(node, options),
            group(printOpeningTag(path, options, print)),
            isEmpty ? "" : hardline,
            doc,
            isEmpty ? "" : hardline,
            printClosingTag(node, options),
            printClosingTagSuffix(node, options),
          ];
        };
      }
      break;

    case "text":
      if (isScriptLikeTag(node.parent, options)) {
        const parser = inferElementParser(node.parent, options);
        if (parser) {
          return async (textToDoc) => {
            const value =
              parser === "markdown"
                ? htmlWhitespace.dedentString(
                    node.value.replace(/^[^\S\n]*\n/u, ""),
                  )
                : node.value;
            const textToDocOptions = { parser, __embeddedInHtml: true };
            if (options.parser === "html" && parser === "babel") {
              let sourceType = "script";
              const { attrMap } = node.parent;
              if (
                attrMap &&
                (attrMap.type === "module" ||
                  ((attrMap.type === "text/babel" ||
                    attrMap.type === "text/jsx") &&
                    attrMap["data-type"] === "module"))
              ) {
                sourceType = "module";
              }
              textToDocOptions.__babelSourceType = sourceType;
            }

            return [
              breakParent,
              printOpeningTagPrefix(node, options),
              await textToDoc(value, textToDocOptions),
              printClosingTagSuffix(node, options),
            ];
          };
        }
      } else if (node.parent.kind === "interpolation") {
        return async (textToDoc) => {
          const textToDocOptions = {
            __isInHtmlInterpolation: true, // to avoid unexpected `}}`
            __embeddedInHtml: true,
          };
          if (options.parser === "angular") {
            textToDocOptions.parser = "__ng_interpolation";
          } else if (options.parser === "vue") {
            textToDocOptions.parser = isVueSfcWithTypescriptScript(
              path,
              options,
            )
              ? "__vue_ts_expression"
              : "__vue_expression";
          } else {
            textToDocOptions.parser = "__js_expression";
          }

          return [
            indent([line, await textToDoc(node.value, textToDocOptions)]),
            node.parent.next &&
            needsToBorrowPrevClosingTagEndMarker(node.parent.next)
              ? " "
              : line,
          ];
        };
      }
      break;

    case "attribute":
      return printAttribute(path, options);

    case "angularControlFlowBlockParameters":
      if (!embeddedAngularControlFlowBlocks.has(path.parent.name)) {
        return;
      }

      return printAngularControlFlowBlockParameters;

    case "angularLetDeclarationInitializer":
      return (textToDoc) =>
        formatAttributeValue(node.value, textToDoc, {
          parser: "__ng_binding",
          __isInHtmlAttribute: false,
        });
  }
}

export default embed;
