//
// nono
// Copyright (C) 2023 nono project
// Licensed under nono-license.txt
//

//
// Nereid Ethernet/メモリ/USB 複合ボード (のうち Ethernet/バンクメモリ)
//

#pragma once

#include "device.h"
#include <array>

class BankRAMDevice;
class NereidBoardDevice;
class RTL8019ASDevice;

// Nereid 空間担当
class NereidDevice : public IODevice
{
	using inherited = IODevice;

 public:
	NereidDevice();
	~NereidDevice() override;

	bool Create() override;

	busdata Read(busaddr addr) override;
	busdata Write(busaddr addr, uint32 data) override;
	busdata Peek1(uint32 addr) override;

	// Nereid が1枚でもささっていれば true。
	bool IsInstalled() const;

	// Board デバイス取得。
	NereidBoardDevice *GetBoard(int id) const { return boards[id].get(); }

 private:
	inline uint32 Decoder(uint32 addr) const;
	DECLARE_MONITOR_SCREEN(MonitorScreen);

	// ボードの有無と配置場所 (有りなら $3 か $b、無しなら -1)
	std::array<int, 2> loc {};

	std::array<std::unique_ptr<NereidBoardDevice>, 2> boards {};

	Monitor *monitor {};
};

// Nereid ボード1枚分
class NereidBoardDevice : public IODevice
{
	using inherited = IODevice;

 public:
	explicit NereidBoardDevice(int n_);
	~NereidBoardDevice() override;

	bool Create() override;

	busdata Read(busaddr addr) override;
	busdata Write(busaddr addr, uint32 data) override;
	busdata Peek1(uint32 addr) override;

	// ネットワークが有効なら true を返す
	bool IsNetEnable() const { return (bool)net; }
	// バンクメモリの容量 [MB] を返す (負数の場合もあり)
	int GetBankRAMSize() const { return ramsize_MB; }

	uint32 GetPage() const;

 private:
	inline uint32 Decoder(uint32 addr) const;

	void SetPage(uint32 data);

	uint32 GetCtrl() const;
	void SetCtrl(uint32 data);

	// 1枚目 or 2枚目
	uint id {};

	// バンクメモリ容量 (負数あり)
	int ramsize_MB {};

	// デバイス
	std::unique_ptr<BankRAMDevice> bank /*{}*/;
	std::unique_ptr<RTL8019ASDevice> net /*{}*/;
};

inline NereidDevice *GetNereidDevice() {
	return Object::GetObject<NereidDevice>(OBJ_NEREID);
}
