//
// nono
// Copyright (C) 2024 nono project
// Licensed under nono-license.txt
//

#include "iodevstream.h"
#include "bus.h"
#include "mainbus.h"
#include "mainram.h"

// デストラクタ
IODeviceStream::~IODeviceStream()
{
}

// 現在位置から size バイト分読み込んでポインタを進める。
// size は busaddr::Size* する。
busdata
IODeviceStream::ReadN(busaddr size)
{
	busaddr ba = busaddr(addr) | size | BusAddr::SRead;
	busdata data;

	do {
		busdata bd = dev->Read(ba);
		if (__predict_false(bd.IsBusErr())) {
			return bd;
		}
		data |= MainbusDevice::DYNAMIC_BUS_SIZING_R(ba, bd);
	} while (ba.GetSize() != 0);

	addr += size.GetSize();
	data |= busdata(size.Get());
	return data;
}

// 現在位置から data を書き出してポインタを進める。
// size は busaddr::Size* を指定する。
busdata
IODeviceStream::WriteN(busaddr size, uint32 data)
{
	busaddr ba = busaddr(addr) | size | BusAddr::SWrite;
	do {
		busdata r = dev->Write(ba, data);
		if (__predict_false(r.IsBusErr())) {
			return r;
		}
		MainbusDevice::DYNAMIC_BUS_SIZING_W(ba, data, r);
	} while (ba.GetSize() != 0);

	addr += size.GetSize();
	return 0;
}

// 現在位置以降にホストの src から len バイトを書き出してポインタを進める。
void
IODeviceStream::WriteMem(const void *src, uint len)
{
	auto mainram = dynamic_cast<MainRAMDevice *>(dev);
	if (mainram) {
		mainram->WriteMem(addr, src, len);
		addr += len;
	} else {
		const uint8 *s = (const uint8 *)src;
		for (int i = 0; i < len; i++) {
			Write1(*s++);
		}
	}
}
