#!/usr/bin/env python3
"""Test the configuration checks generated by generate_config_checks.py.
"""

## Copyright The Mbed TLS Contributors
## SPDX-License-Identifier: Apache-2.0 OR GPL-2.0-or-later

import unittest

import scripts_path # pylint: disable=unused-import
from mbedtls_framework import unittest_config_checks


class MbedtlsTestConfigChecks(unittest_config_checks.TestConfigChecks):
    """Mbed TLS unit tests for checks generated by config_checks_generator."""

    #pylint: disable=invalid-name # uppercase letters make sense here

    PROJECT_CONFIG_C = 'library/mbedtls_config.c'
    PROJECT_SPECIFIC_INCLUDE_DIRECTORIES = [
        'tf-psa-crypto/include',
        'tf-psa-crypto/drivers/builtin/include',
    ]

    ## Method naming convention:
    ## * test_crypto_xxx when testing a tweak of crypto_config.h
    ## * test_mbedtls_xxx when testing a tweak of mbedtls_config.h

    def test_crypto_config_read(self) -> None:
        """Check that crypto_config.h is read in mbedtls."""
        self.bad_case('#error witness',
                      None,
                      error='witness')

    def test_mbedtls_config_read(self) -> None:
        """Check that mbedtls_config.h is read in mbedtls."""
        self.bad_case(''
                      '#error witness',
                      error='witness')

    @unittest.skip("At this time, mbedtls does not go through crypto's check_config.h.")
    def test_crypto_undef_MBEDTLS_FS_IO(self) -> None:
        """A sample error expected from crypto's check_config.h."""
        self.bad_case('#undef MBEDTLS_FS_IO',
                      error='MBEDTLS_PSA_ITS_FILE_C')

    def test_mbedtls_no_session_tickets_for_early_data(self) -> None:
        """An error expected from mbedtls_check_config.h based on the TLS configuration."""
        self.bad_case(None,
                      '''
                      #define MBEDTLS_SSL_EARLY_DATA
                      #undef MBEDTLS_SSL_SESSION_TICKETS
                      ''',
                      error='MBEDTLS_SSL_EARLY_DATA')

    def test_crypto_mbedtls_no_ecdsa(self) -> None:
        """An error expected from mbedtls_check_config.h based on crypto+TLS configuration."""
        self.bad_case('''
                      #undef PSA_WANT_ALG_ECDSA
                      #undef PSA_WANT_ALG_DETERMINISTIC_ECDSA
                      ''',
                      '''
                      #if defined(PSA_WANT_ALG_ECDSA)
                      #error PSA_WANT_ALG_ECDSA unexpected
                      #endif
                      #if defined(PSA_WANT_ALG_DETERMINSTIC_ECDSA)
                      #error PSA_WANT_ALG_DETERMINSTIC_ECDSA unexpected
                      #endif
                      ''',
                      error='MBEDTLS_KEY_EXCHANGE_ECDHE_ECDSA_ENABLED')

    def test_crypto_define_MBEDTLS_KEY_EXCHANGE_RSA_ENABLED(self) -> None:
        """Error when setting a removed option via crypto_config.h."""
        self.bad_case('#define MBEDTLS_KEY_EXCHANGE_RSA_ENABLED',
                      error='MBEDTLS_KEY_EXCHANGE_RSA_ENABLED was removed')

    def test_mbedtls_define_MBEDTLS_KEY_EXCHANGE_RSA_ENABLED(self) -> None:
        """Error when setting a removed option via mbedtls_config.h."""
        self.bad_case(None,
                      '#define MBEDTLS_KEY_EXCHANGE_RSA_ENABLED',
                      error='MBEDTLS_KEY_EXCHANGE_RSA_ENABLED was removed')

    def test_crypto_exempt_define_MBEDTLS_KEY_EXCHANGE_RSA_ENABLED(self) -> None:
        """Bypassed error when setting a removed option via crypto_config.h."""
        self.good_case('#define MBEDTLS_KEY_EXCHANGE_RSA_ENABLED',
                       extra_options=['-DMBEDTLS_CONFIG_CHECK_BYPASS'])

    def test_mbedtls_exempt_define_MBEDTLS_KEY_EXCHANGE_RSA_ENABLED(self) -> None:
        """Bypassed error when setting a removed option via mbedtls_config.h."""
        self.good_case(None,
                       '#define MBEDTLS_KEY_EXCHANGE_RSA_ENABLED',
                       extra_options=['-DMBEDTLS_CONFIG_CHECK_BYPASS'])

    def test_mbedtls_define_MBEDTLS_MD5_C_redundant(self) -> None:
        """Error when redundantly setting a subproject internal option."""
        self.bad_case('#define PSA_WANT_ALG_MD5 1',
                      '#define MBEDTLS_MD5_C',
                      error=r'MBEDTLS_MD5_C is an internal macro')

    def test_mbedtls_define_MBEDTLS_MD5_C_added(self) -> None:
        """Error when setting a subproject internal option that was disabled."""
        self.bad_case('''
                      #undef PSA_WANT_ALG_MD5
                      #undef MBEDTLS_MD5_C
                      ''',
                      '#define MBEDTLS_MD5_C',
                      error=r'MBEDTLS_MD5_C is an internal macro')

    def test_mbedtls_define_MBEDTLS_BASE64_C_redundant(self) -> None:
        """Ok to redundantly set a subproject option."""
        self.good_case(None,
                       '#define MBEDTLS_BASE64_C')

    def test_mbedtls_define_MBEDTLS_BASE64_C_added(self) -> None:
        """Error when setting a subproject option that was disabled."""
        self.bad_case('''
                      #undef MBEDTLS_BASE64_C
                      #undef MBEDTLS_PEM_PARSE_C
                      #undef MBEDTLS_PEM_WRITE_C
                      ''',
                      '#define MBEDTLS_BASE64_C',
                      error=r'MBEDTLS_BASE64_C .*psa/crypto_config\.h')

    @unittest.skip("Checks for #undef are not implemented yet.")
    def test_mbedtls_define_MBEDTLS_BASE64_C_unset(self) -> None:
        """Error when unsetting a subproject option that was enabled."""
        self.bad_case(None,
                      '#undef MBEDTLS_BASE64_C',
                      error=r'MBEDTLS_BASE64_C .*psa/crypto_config\.h')

    def test_crypto_define_MBEDTLS_USE_PSA_CRYPTO(self) -> None:
        """It's ok to set MBEDTLS_USE_PSA_CRYPTO (now effectively always on)."""
        self.good_case('#define MBEDTLS_USE_PSA_CRYPTO')

    def test_mbedtls_define_MBEDTLS_USE_PSA_CRYPTO(self) -> None:
        """It's ok to set MBEDTLS_USE_PSA_CRYPTO (now effectively always on)."""
        self.good_case(None,
                       '#define MBEDTLS_USE_PSA_CRYPTO')


if __name__ == '__main__':
    unittest.main()
