use miette::{Context, IntoDiagnostic};

use crate::actions::GlobalArgs;
use crate::types::config::BergConfig;
use crate::types::token::Token;
use crate::{client::BergClient, render::table::TableWrapper};

use super::git::{Git, OwnerRepo};

pub struct BergContext<T> {
    pub client: BergClient,
    pub config: BergConfig,
    pub git: Git,
    pub args: T,
    pub global_args: GlobalArgs,
}

impl<T> BergContext<T> {
    pub async fn new(args: T, global_args: GlobalArgs) -> miette::Result<Self> {
        let mut config = BergConfig::new()?;

        if let Some(width) = global_args.max_width {
            config.max_width = width;
        }

        let token = Token::read_from_env()
            .wrap_err(miette::miette!(
                "Could't find login data in environment variables"
            ))
            .or_else(|_| {
                Token::read_from_data_dir(config.base_url.as_str()).wrap_err(miette::miette!(
                    help = "Try authenticate via `berg auth login` first!",
                    "Couldn't find login data in token file!"
                ))
            })?;
        tracing::debug!("{token:?}");
        let base_url = config.url()?;
        tracing::debug!("{base_url:#?}");
        let client = BergClient::new(&token, base_url)?;
        let git = Git::new();

        let version = client.get_version().await.into_diagnostic()?;
        tracing::debug!("API VERSION: {version:?}");

        Ok(Self {
            client,
            config,
            git,
            args,
            global_args,
        })
    }

    pub fn owner_repo(&self) -> miette::Result<OwnerRepo> {
        self.global_args
            .owner_repo
            .clone()
            .map(Ok)
            .unwrap_or_else(|| self.git.owner_repo())
            .context("Could not determine a valid OWNER/REPO tuple")
            .context("Specify via CLI global args `--owner-repo`!")
            .context("Alternatively you can just cd into a valid repo!")
    }

    pub fn make_table(&self) -> TableWrapper {
        self.config.make_table()
    }

    pub fn editor_for(&self, prompt_for: &str, predefined_text: &str) -> miette::Result<String> {
        inquire::Editor::new(format!("Open editor to write {prompt_for}").as_str())
            .with_help_message(predefined_text)
            .with_editor_command(std::ffi::OsStr::new(&self.config.editor))
            .prompt()
            .into_diagnostic()
    }
}
