
Those few recipes illustrate the logic of using bacon, and may serve as inspiration for your own practices.

Tell me about your recipes so that I may share them there.

# Open your lib's doc

Let's first assume your `bacon.toml` file has been generated by a recent version of bacon.

Then you can launch bacon on its default task, then when you want to check the doc, hit the <kbd>d</kbd> key: as soon as the doc compiles, it's opened in your browser and bacon switches back to the previous job.

If you want to complete an old `bacon.toml` file, or just understand how it works, here's the relevant configuration:

The job:

```TOML
[jobs.doc-open]
command = ["cargo", "doc", "--no-deps", "--open"]
need_stdout = false
on_success = "back" # so that we don't open the browser at each change
```

And the key binding:

```TOML
[keybindings]
d = "job:doc-open"
```

# Configure Clippy lints

Jobs in the `bacon.toml` file are specific to your projects, there's no reason not to adapt them for its specificities.

You may for example want to tune the clippy rules:

```TOML
[jobs.clippy]
command = [
	"cargo", "clippy",
	"--",
	"-A", "clippy::collapsible_else_if",
	"-A", "clippy::collapsible_if",
	"-A", "clippy::field_reassign_with_default",
	"-A", "clippy::match_like_matches_macro",
]
need_stdout = false
```

You may also add some modifiers on spot sessions, eg

```bash
bacon clippy -- -W clippy::pedantic
```

Note that bacon doesn't need to be killed and relaunched when you change the job config.

# Deal with pedantic

The *pedantic* and *nursery* modes of clippy produce many warnings, and you may not want to permanently exclude the ones that you don't want to immediately handle.

Bacon has "dismiss" commands to this end, that you'd better call from your IDE.

Here's a dedicated article: [https://dystroy.org/blog/practical-pedantism/](https://dystroy.org/blog/practical-pedantism/)

# Check for other platforms

You may define specific jobs for specific targets:

```toml
[jobs.check-win]
command = ["cargo", "check", "--target", "x86_64-pc-windows-gnu"]
```

An habit I suggest: use <kbd>alt</kbd> keybindings for *alternative* platforms:

```toml
[keybindings]
alt-w = "job:check-win"
```

# Run binaries and examples

If you configure a `cargo run` job, you'll get the usual warnings and errors until there's none, at which point you'll have the output of your binary (assuming its terminal output is interesting).

```toml
[jobs.exs]
command = ["cargo", "run", "--example", "simple"]
allow_warnings = true
need_stdout = true
```

You may add `on_success = "back"` if you don't want the executable to run again on changes.

The `allow_warnings = true` line tells bacon to run the executable even when there are warnings. The executable's output would come below warnings.

Some libraries and programs test whether they run in a TTY and remove style in such case.
Most usually, those applications provide a way to bypass this test with a launch argument.
Depending on the desired output, you would have to add a setting to the run job, for example
([more on this](https://github.com/Canop/bacon/issues/89#issuecomment-1257752297)):

```toml
command = ["cargo", "run", "--", "--color", "yes"]
```

# Long running programs

If your program never stops (e.g. a server), you may set `background = false` to have the output of `cargo run` immediately displayed instead of waiting for the program's end.

If you want your program to restart at every change, use `on_change_strategy = "kill_then_restart"`.

You may also want to change the way your program is killed if it should release resources.
In this case, you can replace the standard interrupution by specifying your own `kill` command.

Combining all those changes would give you something like this:

```toml
[jobs.webserver]
command = ["cargo", "run", "--", "--serve"]
need_stdout = true
background = false
on_change_strategy = "kill_then_restart"
kill = ["kill", "-s", "INT"]
```

Of course you don't have to take them all, depending on your precise case.

# Variable arguments

Launch arguments after the `--` aren't interpreted by bacon but sent unchanged to the job commands.

This may be useful to add an argument only for one run without changing the `bacon.toml` file.

For example

```bash
bacon -- --target x86_64-pc-windows-gnu
```

Be careful that some programs already require `--` so you may have to double it.
For example, to run `cargo test` with a single thread, you'll need

```bash
bacon test -- -- --test-threads=1
```

Another use case, a job which needs a complementary argument:


```toml
[jobs.ex]
command = ["cargo", "run", "--example"]
need_stdout = true
```

You would call it with

```bash
bacon ex -- example4578
```

# Remote control

Setting `listen = true` in the configuration makes bacon listen for commands (see [config/listen](../config#listen), unix systems only).

It's possible to send commands with `bacon --send` launched in the same project.

This can be used to add shortcuts controlling bacon in your favourite code editor.

For example to launch clippy from vim on hitting <kbd>space</kbd><kbd>b</kbd><kbd>c</kbd>:

```vim
nnoremap <Leader>bc :execute ":%!bacon --send 'job:clippy'"<CR>`
```

# Specific Rust toolchain

Bacon calls `cargo` under the hood to analyze the workspace, before even trying to run a job.

So bacon itself must sometimes be called with the toolchain already set in order to correctly understand the `Cargo.toml` files.

A simple solution is to set it up with the `RUSTUP_TOOLCHAIN` env var.

For example:

```bash
RUSTUP_TOOLCHAIN=beta bacon test
```

# Bacon CLI snippets

You may share a command-line snippet without requiring a bacon.toml file, using the `--config-toml` argument:

```bash
bacon -j cli-test --config-toml '
[jobs.cli-test]
command = [
  "sh",
  "-c",
  "echo \"hello $(date +%H-%M-%S)\"; cargo run",
]
need_stdout = true
allow_warnings = true
background = false
on_change_strategy = "kill_then_restart"
kill = ["pkill", "-TERM", "-P"]'
```

Notes:

* wrap the inline configuration item in single quotes so that you may use double-quotes inside
* this configuration is simply added to the other ones and it may refer to them
* if you add a job this way and want it executed, either define it as `default_job` in the same inline TOML or use the `--job`/`-j` argument as in the example above

# Run headless

Sometimes, you may want to run bacon without a TUI, for example in a container, or when you run bacon to run an application on change and you want to keep all outputs.

Try the headless mode: `bacon --headless`

# Personal settings & jobs

When your personal settings or jobs are specific to a project, the best place to store them is the `.config/bacon.toml` file of the project.
Bacon here follows [https://dot-config.github.io/](https://dot-config.github.io/).

When your personal settings or jobs are global, store them in your global `prefs.toml` file.
