#include "keystatewatcher.h"

#include <QTimer>
#include "../keyboard.h"
#include "../joystick.h"
#include "../p6vxcommon.h"

KeyStateWatcher::KeyStateWatcher(KEY6 *key, JOY6 *joy, QObject *parent)
	: QObject(parent)
	, Key(key)
	, Joy(joy)
	, ON_SHIFT(false)
	, ON_GRAPH(false)
	, ON_KANA(false)
	, ON_KKANA(false)
	, ON_CAPS(false)
	, ON_ROMAJI(false)
	, Timer(new QTimer(this))
{
	Timer->setInterval(1000 / 60);
	connect(Timer, SIGNAL(timeout()), this, SLOT(poll()));
	start();
}

void KeyStateWatcher::start()
{
	Timer->start();
}

void KeyStateWatcher::stop()
{
	Timer->stop();
}

int KeyStateWatcher::GetModifierStatus()
{
	QMutexLocker lock(&mutex);
	int ret = 0;
	if (ON_SHIFT)	ret |= SHIFT;
	if (ON_GRAPH)	ret |= GRAPH;
	if (ON_KANA)	ret |= KANA;
	if (ON_KKANA)	ret |= KKANA;
	if (ON_CAPS)	ret |= CAPS;
	if (ON_ROMAJI)	ret |= ROMAJI;

	return ret;
}

void KeyStateWatcher::poll()
{
	QMutexLocker lock(&mutex);
	bool changed = false;
	auto keyStatus = Key->GetKeyIndicator();
	if(bool(keyStatus & KI_SHIFT) != this->ON_SHIFT)	changed = true;
	this->ON_SHIFT = bool(keyStatus & KI_SHIFT);

	if(bool(keyStatus & KI_CTRL) != this->ON_CTRL)		changed = true;
	this->ON_CTRL = bool(keyStatus & KI_CTRL);

	if(bool(keyStatus & KI_GRAPH) != this->ON_GRAPH)	changed = true;
	this->ON_GRAPH = bool(keyStatus & KI_GRAPH);

	if(bool(keyStatus & KI_KANA) != this->ON_KANA)		changed = true;
	this->ON_KANA = bool(keyStatus & KI_KANA);

	if(bool(keyStatus & KI_KKANA) != this->ON_KKANA)	changed = true;
	this->ON_KKANA = bool(keyStatus & KI_KKANA);

	if(bool(keyStatus & KI_CAPS) != this->ON_CAPS)		changed = true;
	this->ON_CAPS = bool(keyStatus & KI_CAPS);

	if(bool(keyStatus & KI_ROMAJI) != this->ON_ROMAJI)	changed = true;
	this->ON_ROMAJI = bool(keyStatus & KI_ROMAJI);

	// TILT判定(キーボード)
	auto joyKeyStatus = Key->GetKeyJoy();
	if (joyKeyStatus & 0b00100000){
		TiltScreen(TiltDirection::LEFT);
	} else if (joyKeyStatus & 0b00010000){
		TiltScreen(TiltDirection::RIGHT);
	} else {
		TiltScreen(TiltDirection::NEWTRAL);
	}

	// TILT判定(ジョイスティック)
	for (int jno=1; jno>=0; jno--){ // ジョイスティック1を優先
		auto joystickStatus = Joy->GetJoyState(jno);
		if (~joystickStatus & 0b00000100){
			TiltScreen(TiltDirection::LEFT);
		} else if (~joystickStatus & 0b00001000){
			TiltScreen(TiltDirection::RIGHT);
		}
	}

	UpdateTilt();

	if (changed){
		emit stateChanged(
					ON_SHIFT,
					ON_CTRL,
					ON_GRAPH,
					ON_KANA,
					ON_KKANA,
					ON_CAPS,
					ON_ROMAJI);
	}
}
