/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VrREALFIRFILTER_H_
#define _VrREALFIRFILTER_H_

#include <VrDecimatingSigProc.h>
#include <fstream>

/*  FIR filter definition:
    
    VrRealFIRfilter( cutoff freq., Num of taps, decimation factor, center frequency)
    
    cutoff (Hz) = 0.0  => LPF using Hamming window, o/w LPF transformed to have higher cutoff freq
    decimation factor => set to one unless some integer >1 specified
    center_freq (Hz) => used to specify a composite frequency-shifting filter (i.e. channel filter)
    */


// ********************************************************


template<class iType,class oType> 
class VrRealFIRfilter : public VrDecimatingSigProc<iType,oType> {
protected:
  int 		numTaps;
  float* 	taps;
  float 	cutoff, gain;
  int 		FileDefined;
  void buildFilter_real();
public: 
  virtual const char *name() { return "VrRealFIRfilter"; }
  virtual int work(VrSampleRange output, void *o[],
		   VrSampleRange inputs[], void *i[]);
  virtual void initialize();

  VrRealFIRfilter(float c,int t, float g);
  VrRealFIRfilter(int d,float c,int t, float g);
  VrRealFIRfilter(int d,char* file);
  ~VrRealFIRfilter();
};

template<class iType,class oType> int
VrRealFIRfilter<iType,oType>::work(VrSampleRange output, void *ao[],
				   VrSampleRange inputs[], void *ai[]) {
  iType **i = (iType **)ai;
  oType **o = (oType **)ao;
  float result;
  unsigned int size=output.size;
  for (;size>0;size--,i[0]+=decimation) {
    result = 0;
    iType* inputArray=i[0];
    for (int j=0; j < numTaps; j++)
      result += taps[j] * inputArray[j];
    *o[0]++=(oType)result;
  }
  return output.size;
}

template<class iType,class oType> void
VrRealFIRfilter<iType,oType>::buildFilter_real(){
  double inSampFreq;
  float index, arg;
  float N = (float)numTaps;
  float M = N-1; /* filter Order */

  inSampFreq = getInputSamplingFrequencyN(0); 
  
  // Build Real Filter
  if (cutoff == 0.0){
    
    // cut-off == 0.0 => Hamming window
    for (index=0;index < numTaps; index++) {
      taps[(int)index] =  gain*(0.54-0.46*cos(2*M_PI*index/(M)));
      //      printf("index: %f   tap: %f\n",index,taps[(int)index]);
    }
  } else {
    
    // cut-off != 0.0 => Hamming window * transform to move cutoff to correct place
    arg = 2*M_PI*cutoff/inSampFreq;
    for (index=0;index < numTaps;index++) {
      if (index-(M/2) != 0){     
	taps[(int)index] =  gain*(sin(arg*(index-(M/2)))/M_PI/(index-(M/2))*(0.54-0.46*cos(2*M_PI*index/M)));
      }
      //   printf("index: %f   tap: %f\n",index,taps[(int)index]);
    }
    if ( (((int)M)/2)*2 == (int)M ){ 
      taps[(int)M/2] =  gain*arg/M_PI;
    }
  }
}


template<class iType,class oType> 
VrRealFIRfilter<iType,oType>::VrRealFIRfilter(float c,int t, float g)
  :VrDecimatingSigProc<iType, oType>(1,1),numTaps(t),cutoff(c),gain(g),FileDefined(0)
{
  
}

template<class iType,class oType> 
VrRealFIRfilter<iType,oType>::VrRealFIRfilter(int dec, float c,int t, float g)
  :VrDecimatingSigProc<iType, oType>(1,dec),numTaps(t),cutoff(c),gain(g),FileDefined(0)
{
}

template<class iType,class oType> 
VrRealFIRfilter<iType,oType>::VrRealFIRfilter(int dec,char* filename)
  :VrDecimatingSigProc<iType, oType>(1,dec),cutoff(0.0),gain(1.0),
   FileDefined(1)
{
  // First parse file to determine the numebr of taps
  std::ifstream file(filename);
  if (!file) {
    fprintf(stderr, "Failed to open file\n");
    exit(0);
  }
  numTaps = 0;
  char foo;
  while (file.get(foo)) {
    if (foo == '\n')
      numTaps++;
  }

  if (numTaps < 1) {
    fprintf(stderr, "No taps defined in file\n");
    exit(1);
  }
  taps = new float[numTaps];
  file.close();
  // Re-open and read in taps
  std::ifstream file2(filename);
  if (!file2) {
    fprintf(stderr, "Failed to open file\n");
    exit(0);
  }

  char* asciiTap = new char[100];
  int i = 0;
  int j = 0;
  while (file2.get(asciiTap[i])) {
    if (asciiTap[i] == '\n') {
      taps[j] = atof(asciiTap);
      //      cout << "Tap # " << j << " = " << taps[j] << "\n";
      i = 0;j++;
    } else {
      i++;
    }
  }
  file2.close();
}

template<class iType,class oType> 
void VrRealFIRfilter<iType,oType>::initialize()
{
  history=numTaps;
  if (!FileDefined) 
    taps = new float[numTaps];
  buildFilter_real();
  //inputArray = new iType[numTaps];
}

template<class iType,class oType> 
VrRealFIRfilter<iType,oType>::~VrRealFIRfilter()
{
  delete taps;
}

#endif
