/*
Copyright 1991-1997 by the University of Edinburgh, Department of
Computer Science

Permission to use, copy, modify, distribute, and sell this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation, and that the name of the University of Edinburgh not be used
in advertising or publicity pertaining to distribution of the software
without specific, written prior permission.  The University of Edinburgh
makes no representations about the suitability of this software for any
purpose.  It is provided "as is" without express or implied warranty.

	Author:	George Ross
		Department of Computer Science
		University of Edinburgh
		gdmr@dcs.ed.ac.uk
*/

/* Mail watcher.  Main window lists message senders, status and subjects.
 * Active icon indicates arrival of (new) mail.
 */

/* $Id: xmailwatcher.c,v 1.6 1999/08/09 08:14:11 gdmr Exp $ */

#include "box.h"

#include <sys/types.h>
#include <sys/stat.h>

#ifdef sparc
#include <fcntl.h>
#endif

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#if XtSpecificationRelease <= 5
#include <X11/Xmu/Converters.h>
#endif

#define POLL_DEFAULT 67
#define BUFFER_DEFAULT 32000

#include <stdio.h>

typedef struct _Resources {
	int	pollInterval;
	char	*mailbox;
	Pixel	iconForeground;
	Pixel	iconBackground;
	Pixel	iconMailForeground;
	Pixel	iconMailBackground;
	Pixmap	mailBitmap;
	Pixmap	nomailBitmap;
	Boolean	silent;
	int	bellVolume;
	char	*soundFile;
	int	bufferSize;
	Boolean	newMail;
	Boolean	fullNames;
	Boolean	debug;
	Boolean setTitle;
	Boolean doMime;
	Boolean fakeIcon;
	Boolean keepIcon;
} Resources;

static Resources theResources;

static XtResource resourceSpec[] = {
	{ "pollInterval", "PollInterval", XtRInt, sizeof(int),
	  XtOffsetOf(Resources, pollInterval),
	  XtRImmediate, (XtPointer) POLL_DEFAULT },
	{ "mailbox", "Mailbox", XtRString, sizeof(char *),
	  XtOffsetOf(Resources, mailbox), XtRString, ".mail" },
	{ "iconForeground", XtCForeground, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(Resources, iconForeground),
	  XtRString, XtDefaultForeground },
	{ "iconBackground", XtCBackground, XtRPixel, sizeof(Pixel),
	  XtOffsetOf(Resources, iconBackground),
	  XtRString, XtDefaultBackground },
	{ "iconMailForeground", "IconMailForeground", XtRPixel, sizeof(Pixel),
	  XtOffsetOf(Resources, iconMailForeground),
	  XtRPixel, (XtPointer) &theResources.iconBackground },
	{ "iconMailBackground", "IconMailBackground", XtRPixel, sizeof(Pixel),
	  XtOffsetOf(Resources, iconMailBackground),
	  XtRPixel, (XtPointer) &theResources.iconForeground },
	{ "mailBitmap", "MailBitmap", XtRBitmap, sizeof(Pixmap),
	  XtOffsetOf(Resources, mailBitmap), XtRString, "letters" },
	{ "nomailBitmap", "NomailBitmap", XtRBitmap, sizeof(Pixmap),
	  XtOffsetOf(Resources, nomailBitmap), XtRString, "noletters" },
	{ "silent", "Silent", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, silent), XtRImmediate, (XtPointer) False },
	{ "bellVolume", "BellVolume", XtRInt, sizeof(int),
	  XtOffsetOf(Resources, bellVolume), XtRImmediate, (XtPointer) 0 },
	{ "soundFile", "SoundFile", XtRString, sizeof(char *),
	  XtOffsetOf(Resources, soundFile), XtRImmediate, (XtPointer) NULL },
	{ "bufferSize", "BufferSize", XtRInt, sizeof(int),
	  XtOffsetOf(Resources, bufferSize),
	  XtRImmediate, (XtPointer) BUFFER_DEFAULT },
	{ "newMail", "NewMail", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, newMail), XtRImmediate, (XtPointer) True },
	{ "fullNames", "FullNames", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, fullNames), XtRImmediate, (XtPointer) False },
	{ "debug", "Debug", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, debug), XtRImmediate, (XtPointer) False },
	{ "setTitle", "SetTitle", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, setTitle), XtRImmediate, (XtPointer) True },
	{ "doMime", "DoMime", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, doMime), XtRImmediate, (XtPointer) True },
	{ "fakeIcon", "FakeIcon", XtRBoolean, sizeof (Boolean),
	  XtOffsetOf(Resources, fakeIcon), XtRImmediate, (XtPointer) False },
	{ "keepIcon", "KeepIcon", XtRBoolean, sizeof(Boolean),
	  XtOffsetOf(Resources, keepIcon), XtRImmediate, (XtPointer) False },
};

static XrmOptionDescRec options[] = {
	{ "-poll", ".pollInterval", XrmoptionSepArg, NULL },
	{ "-mailBox", ".mailbox", XrmoptionSepArg, NULL },
	{ "-mailbox", ".mailbox", XrmoptionSepArg, NULL },
	{ "-iconGeometry", ".mailWatchIcon.geometry", XrmoptionSepArg, NULL },
	{ "-icongeometry", ".mailWatchIcon.geometry", XrmoptionSepArg, NULL },
	{ "-newMail", ".newMail", XrmoptionNoArg, "True" },
	{ "-newmail", ".newMail", XrmoptionNoArg, "True" },
	{ "-allMail", ".newMail", XrmoptionNoArg, "False" },
	{ "-allmail", ".newMail", XrmoptionNoArg, "False" },
	{ "-silent", ".silent", XrmoptionNoArg, "True" },
	{ "-soundfile", ".soundFile", XrmoptionSepArg, NULL },
	{ "-soundFile", ".soundFile", XrmoptionSepArg, NULL },
	{ "-debug", ".debug", XrmoptionNoArg, "True" },
	{ "-doMime", ".doMime", XrmoptionNoArg, "True" },
	{ "-noMime", ".doMime", XrmoptionNoArg, "False" },
	{ "-fakeIcon", ".fakeIcon", XrmoptionNoArg, "True" },
	{ "-realIcon", ".fakeIcon", XrmoptionNoArg, "False" },
	{ "-keepIcon", ".keepIcon", XrmoptionNoArg, "True" },
};

#if XtSpecificationRelease <= 5
static XtConvertArgRec screenConvertArg[] = {
    { XtBaseOffset, (XtPointer) XtOffset(Widget, core.screen), sizeof(Screen *)}
};
#endif

void timedRescan();
void listEvent();
void iconEvent();
void iconCallback();

static int mapped;

static char *buffer;
static int flags; 

static XtAppContext appContext;
static Widget shell;
static Widget viewport;
static Widget messages;
static Widget mailWatchList;
static Widget mailWatchIcon;
static Widget iconLabel;

main(argc, argv)
int argc;
char **argv;
{	Arg args[10];
	int n;
	Position iconX, iconY;
	XFontStruct *labelFont;
	Atom registryAtom, encodingAtom;
	unsigned long registryProp, encodingProp;
	char *registryName, *encodingName;
	Boolean startIconic = 0;
	String geometry;

#if XtSpecificationRelease <= 5
	shell = XtAppInitialize(&appContext, "MailWatcher", 
			options, XtNumber(options),
			&argc, argv, NULL, NULL, 0);

	XtAddConverter(XtRString, XtRBitmap, XmuCvtStringToBitmap,
		screenConvertArg, XtNumber(screenConvertArg));
#else
	shell = XtOpenApplication(&appContext, "MailWatcher",
			options, XtNumber(options),
			&argc, argv, NULL,
			sessionShellWidgetClass,
			NULL, 0);
#endif

	XtGetApplicationResources(shell, &theResources,
			resourceSpec, XtNumber(resourceSpec), NULL, 0);

	if (theResources.debug) {
		(void) printf("mailbox %s, pollInterval %d\n",
			theResources.mailbox, theResources.pollInterval);
		(void) printf("newMail %d, fullNames %d\n",
			theResources.newMail, theResources.fullNames);
		(void) printf("bellVolume %d\n", theResources.bellVolume);
		(void) printf("soundFile <%s>\n",
			theResources.soundFile ? \
				theResources.soundFile : "*none*");
		(void) printf("fakeIcon: %s\n",
			theResources.fakeIcon ? "yes" : "no");
	}

	if ((theResources.bellVolume < -100) ||
			(theResources.bellVolume > 100)) {
		(void) fprintf(stderr, "%s: bogus value %d for bellVolume\n",
			*argv, theResources.bellVolume);
		theResources.bellVolume = 0;
	}

	if (theResources.newMail)	flags  = BOX_NEWONLY;
	if (theResources.fullNames)	flags |= BOX_FULLNAMES;
	if (theResources.debug) 	flags |= BOX_DEBUG;
	if (theResources.doMime)	flags |= BOX_DOMIME;

	buffer = (char *) malloc(theResources.bufferSize);

	/* If we're -fakeIcon then create a separate popup for the
	 * list, (perhaps eventually) inheriting the geometry from 
	 * the application shell.  If we're not then just use the
	 * application shell directly
	 */
	if (theResources.fakeIcon) {
		n = 0;
		XtSetArg(args[n], XtNiconic, &startIconic);	n++;
		XtGetValues(shell, args, n);

		n = 0;
		XtSetArg(args[n], XtNgeometry, &geometry);	n++;
		XtGetValues(shell, args, n);
		if (theResources.debug)
			(void) printf("List shell geometry <%s>\n",
					geometry);

		n = 0;
		XtSetArg(args[n], XtNgeometry, geometry);	n++;
		mailWatchList = XtCreatePopupShell("mailWatchList",
					topLevelShellWidgetClass,
					shell, args, n);
	} else {
		mailWatchList = shell;
	}

	n = 0;
	XtSetArg(args[n], XtNallowVert, True);	n++;
	viewport = XtCreateManagedWidget("viewport", viewportWidgetClass,
			mailWatchList, args, n);

	n = 0;
	XtSetArg(args[n], XtNjustify, XtJustifyLeft);	n++;
	XtSetArg(args[n], XtNlabel, "");		n++;
	messages = XtCreateManagedWidget("messages", labelWidgetClass,
			viewport, args, n);

	mailWatchIcon = XtCreatePopupShell("mailWatchIcon", \
			topLevelShellWidgetClass,
			shell, NULL, 0);
	
	n = 0;
	XtSetArg(args[n], XtNbitmap, theResources.nomailBitmap);	n++;
	XtSetArg(args[n], XtNforeground, theResources.iconForeground);	n++;
	XtSetArg(args[n], XtNbackground, theResources.iconBackground);	n++;
	if (theResources.fakeIcon) {
		iconLabel = XtCreateManagedWidget("icon", commandWidgetClass,
				mailWatchIcon, args, n);
		XtAddCallback(iconLabel, XtNcallback, iconCallback, NULL);
	} else {
		iconLabel = XtCreateManagedWidget("icon", labelWidgetClass,
				mailWatchIcon, args, n);
	}

	XtRealizeWidget(mailWatchIcon);
	if (theResources.fakeIcon) {
		XtAddEventHandler(mailWatchIcon, StructureNotifyMask,
			False, iconEvent, NULL);
		if (startIconic || theResources.keepIcon) {
			XtPopup(mailWatchIcon, XtGrabNone);
		}
	}

	n = 0;
	XtSetArg(args[n], XtNx, &iconX);	n++;
	XtSetArg(args[n], XtNy, &iconY);	n++;
	XtGetValues(mailWatchIcon, args, n);
	n = 0;
	XtSetArg(args[n], XtNiconWindow, XtWindow(mailWatchIcon));	n++;
	if (iconX || iconY) {
		XtSetArg(args[n], XtNiconX, iconX);		n++;
		XtSetArg(args[n], XtNiconY, iconY);		n++;
	}
	XtSetValues(shell, args, n);

	XtRealizeWidget(mailWatchList);
	XtAddEventHandler(mailWatchList, StructureNotifyMask,
		False, listEvent, NULL);

	if (theResources.fakeIcon && !startIconic) {
		XtPopup(mailWatchList, XtGrabNone);
	}

	n = 0;
	XtSetArg(args[n], XtNfont, &labelFont);	n++;
	XtGetValues(messages, args, n);
	registryAtom = XInternAtom(XtDisplay(messages),
				"CHARSET_REGISTRY", False);
	encodingAtom = XInternAtom(XtDisplay(messages),
				"CHARSET_ENCODING", False);
	if (XGetFontProperty(labelFont, registryAtom, &registryProp)) {
		registryName = XGetAtomName(XtDisplay(messages), registryProp);
	} else {
		(void) fprintf(stderr,
			"%s: Couldn't get registry for message font\n", *argv);
		registryName = (char *) 0;
	}
	if (XGetFontProperty(labelFont, encodingAtom, &encodingProp)) {
		encodingName = XGetAtomName(XtDisplay(messages), encodingProp);
	} else {
		(void) fprintf(stderr,
			"%s: Couldn't get encoding for message font\n", *argv);
		encodingName = (char *) 0;
	}
	setCharset(registryName, encodingName);

	timedRescan(NULL, NULL);

	XtAppMainLoop(appContext);
}

/* This is the callback for the -fakeIcon command widget.  If we have
 * a "normal" icon then we won't be set up to be called here.
 * If we're not -keepIcon then pop ourselves down here.
 */
static void iconCallback(w, client_data, call_data)
Widget w;
XtPointer client_data, call_data;
{	if (theResources.debug)
		(void) printf("iconCallback: %d\n", mapped);
	if (mapped) {
		XtPopdown(mailWatchList);
		mapped = 0;
	} else {
		XtPopup(mailWatchList, XtGrabNone);
		mapped = 1;
	}
	/* Don't need to rescan in here, because the mailWatchList
	 * will get a Map/Unmap event and the handler will do the
	 * the rescan there.
	 */
	if (!theResources.keepIcon) {
		/* If we're not in -keepIcon mode then we pop ourselves
		 * down now.  Our event handler will (redundantly) pop
		 * up the list window, but that shouldn't matter. */
		XtPopdown(mailWatchIcon);
	}
}

static void setTitle(time, have)
long time, have;
{	char titleBuffer[80];
	Arg args[4];
	int n;
	static int lastTime;

	if (time == lastTime) return;

	if (time) {
		(void) sprintf(titleBuffer,
			have ? "Mail at %s" : "No mail since %s",
			ctime(&time));
		titleBuffer[strlen(titleBuffer) - 1] = '\0';
		if (theResources.debug)
			(void) printf("Title -> <%s>\n",
					titleBuffer);
		n = 0;
		XtSetArg(args[n], XtNtitle, titleBuffer);	n++;
	}
	else {
		n = 0;
		XtSetArg(args[n], XtNtitle, "No mail");	 	n++;
	}
	XtSetValues(mailWatchList, args, n);
	lastTime = time;
}

static void beep()
{
#ifdef sparc
	int sound;
	int audio;
	int n;
	char buffer[512];

	if (theResources.soundFile) {
		if ((audio = open("/dev/audio", O_WRONLY)) > 0 &&
		    (sound = open(theResources.soundFile, O_RDONLY)) > 0) {
			while ((n = read(sound, buffer, 512)) > 0)
				(void) write(audio, buffer, n);
		}
		if (sound) (void) close(sound);
		if (audio) (void) close(audio);
	}
	else
#endif
		XBell(XtDisplay(shell), theResources.bellVolume);
}

static void rescanMailbox(timedScan)
int timedScan;
{	Arg args[10];
	int n;
	static int haveMail;
	static int hadMail;
	static int pendingClear = 1;
	static int forceRescan = 0;
	struct stat sb;
	static time_t lastTime;
	Widget scrollbar;
	Dimension scrollWidth, scrollHeight;
	XButtonEvent be;
	static char *cont = "Continuous";

	if (theResources.debug)
		(void) printf("rescanMailbox %d\n", timedScan);

	if (stat(theResources.mailbox, &sb) < 0) {
		/* No mail file, so no mail */
		if (theResources.debug)
			(void) printf("No mail file, %d %d\n",
					timedScan, hadMail);
		if (timedScan && !hadMail) return;  /* No change */
		*buffer = '\0';
		haveMail = 0;
		lastTime = 0;
		pendingClear = 1;
	}
	else if (sb.st_mtime == lastTime && !(forceRescan && mapped)) {
		/* No change */
		if (theResources.debug)
			(void) printf("No change, %d %s",
					timedScan, ctime(&lastTime));
		if (timedScan) return;
	}
	else if (sb.st_size == 0) {
		/* Empty mail file */
		if (theResources.debug)
			(void) printf("Empty, %d %s",
					timedScan, ctime(&sb.st_mtime));
		*buffer = '\0';
		haveMail = 0;
		lastTime = sb.st_mtime;
		pendingClear = 1;
	}
	else if (mapped || theResources.newMail || S_ISDIR(sb.st_mode)) {
		/* Something in the mail file.  We only need to go to the
		 * trouble of reading it if we're mapped or we're only
		 * wanting to report on new mail */
		haveMail = parseMailbox(theResources.mailbox,
			buffer, theResources.bufferSize, flags);
		forceRescan = 0;
		lastTime = sb.st_mtime;
		if (theResources.debug)
			(void) printf("Scanned, %d %d %s",
					timedScan, haveMail, ctime(&lastTime));
	}
	else { /* Changed, not empty, not mapped */
		if (theResources.debug)
			(void) printf("Changed but not scanned, %d %x %x %d %s",
					timedScan, lastTime, sb.st_mtime,
					mapped, ctime(&sb.st_mtime));
		haveMail = 1;
		lastTime = sb.st_mtime;
		forceRescan = 1;
	}

	if (timedScan && haveMail && !theResources.silent) beep();

	if ((haveMail || hadMail || pendingClear) && mapped) {
		pendingClear = 0;
		n = 0;
		XtSetArg(args[n], XtNlabel, buffer);	n++;
		XtSetValues(messages, args, n);

		/* This is revolting, but there doesn't seem to be any
		 * other way to get the thing to scroll right to the bottom */
		scrollbar = XtNameToWidget(viewport, "vertical");
		if (scrollbar) {
			n = 0;
			XtSetArg(args[n], XtNwidth, &scrollWidth);	n++;
			XtSetArg(args[n], XtNheight, &scrollHeight);	n++;
			XtGetValues(scrollbar, args, n);
			/* Fake up a button event -- enough to pacify the
			 * Scrollbar widget, at any rate... */
			be.type = ButtonPress;
			be.x = scrollWidth / 2;
			be.y = scrollHeight - 1;
			XtCallActionProc(scrollbar, "StartScroll", \
						(XEvent *) &be, &cont, 1);
			XtCallActionProc(scrollbar, "MoveThumb", \
						(XEvent *) &be, NULL, 0);
			XtCallActionProc(scrollbar, "NotifyThumb", \
						(XEvent *) &be, NULL, 0);
			XtCallActionProc(scrollbar, "EndScroll", \
						(XEvent *) &be, NULL, 0);
		}
		/* YUK! */
	}

	if (theResources.setTitle) setTitle(lastTime, haveMail);

	if (haveMail > 0) {
		if (hadMail > 0) return;	/* No change */
		n = 0;
		XtSetArg(args[n], XtNforeground,
			theResources.iconMailForeground);		n++;
		XtSetArg(args[n], XtNbackground,
			theResources.iconMailBackground);		n++;
		XtSetArg(args[n], XtNbitmap, theResources.mailBitmap);	n++;
	}
	else {
		if (hadMail <= 0) return;	/* No change */
		n = 0;
		XtSetArg(args[n], XtNforeground, theResources.iconForeground);
		n++;
		XtSetArg(args[n], XtNbackground, theResources.iconBackground);
		n++;
		XtSetArg(args[n], XtNbitmap, theResources.nomailBitmap);
		n++;
	}
	XtSetValues(iconLabel, args, n);
	hadMail = haveMail;
}

void timedRescan(d, t)
XtPointer d;
XtIntervalId *t;
{	rescanMailbox(1);
	(void) XtAppAddTimeOut(appContext, 1000 * theResources.pollInterval,
		timedRescan, NULL);
}

/* This is the event handler for the list shell. */
void listEvent(w, cd, e, ctd)
Widget w;
XtPointer cd;
XEvent *e;
Boolean *ctd;
{	if (e->type == MapNotify) {
		if (theResources.debug) (void) printf("MapNotify\n");
		mapped = 1;
		rescanMailbox(0);
	}
	else if (e->type == UnmapNotify) {
		if (theResources.debug) (void) printf("UnmapNotify\n");
		if (theResources.fakeIcon) {
			/* Withdraw, but make sure the icon shell
			 * is on the screen */
			XtPopdown(w);
			XtPopup(mailWatchIcon, XtGrabNone);
		}
		mapped = 0;
		rescanMailbox(0);
	}
}

/* This is the event handler for the icon shell.  It's set up in
 * -fakeIcon mode so that one or other of the icon or list shells is
 * always mapped on the screen */
void iconEvent(w, cd, e, ctd)
Widget w;
XtPointer cd;
XEvent *e;
Boolean *ctd;
{	if (e->type == UnmapNotify) {
		/* Withdraw, but map the list shell instead */
		XtPopdown(w);
		if (!mapped) XtPopup(mailWatchList, XtGrabNone);
		/* Let its event handler deal with rescanning */
	}
}
/*
 * $Log: xmailwatcher.c,v $
 * Revision 1.6  1999/08/09 08:14:11  gdmr
 * Bigger buffer
 *
 * Revision 1.5  1999/08/03 13:33:16  gdmr
 * Always set the title
 *
 * Revision 1.4  1999/08/03 11:48:01  gdmr
 * Initial pendingClear to set title
 * Always scan directories
 *
 * Revision 1.3  1999/08/02 14:11:17  gdmr
 * parse can return < 0
 *
 * Revision 1.2  1997/09/09 13:13:53  gdmr
 * fakeIcon, keepIcon
 *
 *
 */
