/*
 *	FM-7 EMULATOR "XM7"
 *
 *	Copyright (C) 1999,2000 ohD(ytanaka@ipc-tokai.or.jp)
 *	[ L[{[h ]
 */

#include <assert.h>
#include <string.h>
#include "xm7.h"
#include "mainetc.h"
#include "gui.h"
#include "device.h"

/*
 *	O[o [N
 */
BOOL caps_flag;				/* CAPS tO */
BOOL kana_flag;				/* Ji tO */
BOOL ins_flag;				/* INS  tO */

BOOL shift_flag;			/* SHIFTL[tO */
BOOL ctrl_flag;				/* CTRLL[tO */
BOOL graph_flag;			/* GRAPHL[tO */
BOOL break_flag;			/* BreakL[tO */

BYTE key_scan;				/* L[R[h(R[hAMake/Breakp) */
WORD key_fm7;				/* L[R[h(FM-7݊R[h) */

BOOL key_repeat_flag;		/* L[s[gtO */
WORD key_repeat_cnt;		/* L[s[gJE^ */
BYTE key_repeat_code;		/* L[s[gR[h */

/*-[ L[e[u ]---------------------------------------------------------*/

/*
 *	CTRL[h
 *	R[hASHIFTȂASHIFT̏
 */
WORD ctrl_key_table[] = {
	0x0c, 0x1e, 0x1e,
	0x0d, 0x1c, 0x1c,
	0x11, 0x11, 0x11,
	0x12, 0x17, 0x17,
	0x13, 0x05, 0x05,
	0x14, 0x12, 0x12,
	0x15, 0x14, 0x14,
	0x16, 0x19, 0x19,
	0x17, 0x15, 0x15,
	0x18, 0x09, 0x09,
	0x19, 0x0f, 0x09,
	0x1a, 0x10, 0x10,
	0x1b, 0x00, 0x00,
	0x1c, 0x1b, 0x1b,
	0x1e, 0x01, 0x01,
	0x1f, 0x13, 0x13,
	0x20, 0x04, 0x04,
	0x21, 0x06, 0x06,
	0x22, 0x07, 0x07,
	0x23, 0x08, 0x08,
	0x24, 0x0a, 0x0a,
	0x25, 0x0b, 0x0b,
	0x26, 0x0c, 0x0c,
	0x29, 0x1d, 0x1d,
	0x2a, 0x1a, 0x1a,
	0x2b, 0x18, 0x18,
	0x2c, 0x03, 0x03,
	0x2d, 0x16, 0x16,
	0x2e, 0x02, 0x02,
	0x2f, 0x0e, 0x0e,
	0x30, 0x0d, 0x0d,
	0x34, 0x1f, 0x1f
};

/*
 *	GRAPH[h
 *	R[hASHIFTȂASHIFT̏
 */
WORD graph_key_table[] = {
	0x5d, 0x0101, 0,
	0x5e, 0x0102, 0,
	0x5f, 0x0103, 0,
	0x60, 0x0104, 0,
	0x61, 0x0105, 0,
	0x62, 0x0106, 0,
	0x63, 0x0107, 0,
	0x64, 0x0108, 0,
	0x65, 0x0109, 0,
	0x66, 0x010a, 0,
	
	0x01, 0x1b, 0x1b,
	0x02, 0xf9, 0xf9,
	0x03, 0xfa, 0xfa,
	0x04, 0xfb, 0xfb,
	0x05, 0xfc, 0xfc,
	0x06, 0xf2, 0xf2,
	0x07, 0xf3, 0xf3,
	0x08, 0xf4, 0xf4,
	0x09, 0xf5, 0xf5,
	0x0a, 0xf6, 0xf6,
	0x0b, 0xf7, 0xf7,
	0x0c, 0x8c, 0x8c,
	0x0d, 0x8b, 0x8b,
	0x0e, 0xf1, 0xf1,
	0x0f, 0x08, 0x08,

	0x10, 0x09, 0x09,
	0x11, 0xfd, 0xfd,
	0x12, 0xf8, 0xf8,
	0x13, 0xe4, 0xe4,
	0x14, 0xe5, 0xe5,
	0x15, 0x9c, 0x9c,
	0x16, 0x9d, 0x9d,
	0x17, 0xf0, 0xf0,
	0x18, 0xe8, 0xe8,
	0x19, 0xe9, 0xe9,
	0x1a, 0x8d, 0x8d,
	0x1b, 0x8a, 0x8a,
	0x1c, 0xed, 0xed,
	0x1d, 0x0d, 0x0d,

	0x1e, 0x95, 0x95,
	0x1f, 0x96, 0x96,
	0x20, 0xe6, 0xe6,
	0x21, 0xe7, 0xe7,
	0x22, 0x9e, 0x9e,
	0x23, 0x9f, 0x9f,
	0x24, 0xea, 0xea,
	0x25, 0xeb, 0xeb,
	0x26, 0x8e, 0x8e,
	0x27, 0x99, 0x99,
	0x28, 0x94, 0x94,
	0x29, 0xec, 0xec,

	0x2a, 0x80, 0x80,
	0x2b, 0x81, 0x81,
	0x2c, 0x82, 0x82,
	0x2d, 0x83, 0x83,
	0x2e, 0x84, 0x84,
	0x2f, 0x85, 0x85,
	0x30, 0x86, 0x86,
	0x31, 0x87, 0x87,
	0x32, 0x88, 0x88,
	0x33, 0x97, 0x97,
	0x34, 0xe0, 0xe0,

	0x48, 0x12, 0x12,
	0x4d, 0x1e, 0x19,
	0x4b, 0x7f, 0x7f,
	0x4f, 0x1d, 0x02,
	0x50, 0x1f, 0x1a,
	0x51, 0x1c, 0x06,

	0x49, 0x05, 0x05,
	0x4a, 0x0c, 0x0c,
	0x4c, 0x11, 0x11,
	0x4e, 0x0b, 0x0b,

	0x36, 0x98, 0x98,
	0x37, 0x91, 0x91,
	0x38, 0x99, 0x99,
	0x39, 0xee, 0xee,
	0x3a, 0xe1, 0xe1,
	0x3b, 0xe2, 0xe2,
	0x3c, 0xe3, 0xe3,
	0x3d, 0xef, 0xef,
	0x3e, 0x93, 0x93,
	0x3f, 0x8f, 0x8f,
	0x40, 0x92, 0x92,
	0x41, 0, 0,
	0x42, 0x9a, 0x9a,
	0x43, 0x90, 0x90,
	0x44, 0x9b, 0x9b,
	0x45, 0x0d, 0x0d,
	0x46, 0, 0,
	0x47, 0, 0,

	0x57, 0x20, 0x20,
	0x58, 0x20, 0x20,
	0x35, 0x20, 0x20
};

/*
 *	ȃ[h
 *	R[hASHIFTȂASHIFT̏
 */
WORD kana_key_table[] = {
	0x5d, 0x0101, 0,
	0x5e, 0x0102, 0,
	0x5f, 0x0103, 0,
	0x60, 0x0104, 0,
	0x61, 0x0105, 0,
	0x62, 0x0106, 0,
	0x63, 0x0107, 0,
	0x64, 0x0108, 0,
	0x65, 0x0109, 0,
	0x66, 0x010a, 0,
	
	0x01, 0x1b, 0x1b,
	0x02, 0xc7, 0,
	0x03, 0xcc, 0,
	0x04, 0xb1, 0xa7,
	0x05, 0xb3, 0xa9,
	0x06, 0xb4, 0xaa,
	0x07, 0xb5, 0xab,
	0x08, 0xd4, 0xac,
	0x09, 0xd5, 0xad,
	0x0a, 0xd6, 0xae,
	0x0b, 0xdc, 0xa6,
	0x0c, 0xce, 0,
	0x0d, 0xcd, 0,
	0x0e, 0xb0, 0,
	0x0f, 0x08, 0x08,

	0x10, 0x09, 0x09,
	0x11, 0xc0, 0,
	0x12, 0xc3, 0,
	0x13, 0xb2, 0xa8,
	0x14, 0xbd, 0,
	0x15, 0xb6, 0,
	0x16, 0xdd, 0,
	0x17, 0xc5, 0,
	0x18, 0xc6, 0,
	0x19, 0xd7, 0,
	0x1a, 0xbe, 0,
	0x1b, 0xde, 0,
	0x1c, 0xdf, 0xa2,
	0x1d, 0x0d, 0x0d,

	0x1e, 0xc1, 0,
	0x1f, 0xc4, 0,
	0x20, 0xbc, 0,
	0x21, 0xca, 0,
	0x22, 0xb7, 0,
	0x23, 0xb8, 0,
	0x24, 0xcf, 0,
	0x25, 0xc9, 0,
	0x26, 0xd8, 0,
	0x27, 0xda, 0,
	0x28, 0xb9, 0,
	0x29, 0xd1, 0xa3,

	0x2a, 0xc2, 0xaf,
	0x2b, 0xbb, 0,
	0x2c, 0xbf, 0,
	0x2d, 0xcb, 0,
	0x2e, 0xba, 0,
	0x2f, 0xd0, 0,
	0x30, 0xd3, 0,
	0x31, 0xc8, 0xa4,
	0x32, 0xd9, 0xa1,
	0x33, 0xd2, 0xa5,
	0x34, 0xdb, 0,

	0x48, 0x12, 0x12,
	0x4d, 0x1e, 0x19,
	0x4b, 0x7f, 0x7f,
	0x4f, 0x1d, 0x02,
	0x50, 0x1f, 0x1a,
	0x51, 0x1c, 0x06,

	0x49, 0x05, 0x05,
	0x4a, 0x0c, 0x0c,
	0x4c, 0x11, 0x11,
	0x4e, 0x0b, 0x0b,

	0x36, 0x2a, 0x2a,
	0x37, 0x2f, 0x2f,
	0x38, 0x2b, 0x2b,
	0x39, 0x2d, 0x2d,
	0x3a, 0x37, 0x37,
	0x3b, 0x38, 0x38,
	0x3c, 0x39, 0x39,
	0x3d, 0x3d, 0x3d,
	0x3e, 0x34, 0x34,
	0x3f, 0x35, 0x35,
	0x40, 0x36, 0x36,
	0x41, 0x2c, 0x2c,
	0x42, 0x31, 0x31,
	0x43, 0x32, 0x32,
	0x44, 0x33, 0x33,
	0x45, 0x0d, 0x0d,
	0x46, 0x30, 0x30,
	0x47, 0x2e, 0x2e,

	0x57, 0x20, 0x20,
	0x58, 0x20, 0x20,
	0x35, 0x20, 0x20
};

/*
 *	CAP[h
 *	R[hASHIFTȂASHIFT̏
 */
WORD caps_key_table[] = {
	0x5d, 0x0101, 0,
	0x5e, 0x0102, 0,
	0x5f, 0x0103, 0,
	0x60, 0x0104, 0,
	0x61, 0x0105, 0,
	0x62, 0x0106, 0,
	0x63, 0x0107, 0,
	0x64, 0x0108, 0,
	0x65, 0x0109, 0,
	0x66, 0x010a, 0,
	
	0x01, 0x1b, 0x1b,
	0x02, 0x31, 0x21,
	0x03, 0x32, 0x22,
	0x04, 0x33, 0x23,
	0x05, 0x34, 0x24,
	0x06, 0x35, 0x25,
	0x07, 0x36, 0x26,
	0x08, 0x37, 0x27,
	0x09, 0x38, 0x28,
	0x0a, 0x39, 0x29,
	0x0b, 0x30, 0,
	0x0c, 0x2d, 0x3d,
	0x0d, 0x5e, 0x7e,
	0x0e, 0x5c, 0x7c,
	0x0f, 0x08, 0x08,

	0x10, 0x09, 0x09,
	0x11, 0x51, 0x71,
	0x12, 0x57, 0x77,
	0x13, 0x45, 0x65,
	0x14, 0x52, 0x72,
	0x15, 0x54, 0x74,
	0x16, 0x59, 0x79,
	0x17, 0x55, 0x75,
	0x18, 0x49, 0x69,
	0x19, 0x4f, 0x6f,
	0x1a, 0x50, 0x70,
	0x1b, 0x40, 0x60,
	0x1c, 0x5b, 0x7b,
	0x1d, 0x0d, 0x0d,

	0x1e, 0x41, 0x61,
	0x1f, 0x53, 0x73,
	0x20, 0x44, 0x64,
	0x21, 0x46, 0x66,
	0x22, 0x47, 0x67,
	0x23, 0x48, 0x68,
	0x24, 0x4a, 0x6a,
	0x25, 0x4b, 0x6b,
	0x26, 0x4c, 0x6c,
	0x27, 0x3b, 0x2b,
	0x28, 0x3a, 0x2a,
	0x29, 0x5d, 0x7d,

	0x2a, 0x5a, 0x7a,
	0x2b, 0x58, 0x78,
	0x2c, 0x43, 0x63,
	0x2d, 0x56, 0x76,
	0x2e, 0x42, 0x62,
	0x2f, 0x4e, 0x6e,
	0x30, 0x4d, 0x6d,
	0x31, 0x2c, 0x3c,
	0x32, 0x2e, 0x3e,
	0x33, 0x2f, 0x3f,
	0x34, 0x22, 0x5f,

	0x48, 0x12, 0x12,
	0x4d, 0x1e, 0x19,
	0x4b, 0x7f, 0x7f,
	0x4f, 0x1d, 0x02,
	0x50, 0x1f, 0x1a,
	0x51, 0x1c, 0x06,

	0x49, 0x05, 0x05,
	0x4a, 0x0c, 0x0c,
	0x4c, 0x11, 0x11,
	0x4e, 0x0b, 0x0b,

	0x36, 0x2a, 0x2a,
	0x37, 0x2f, 0x2f,
	0x38, 0x2b, 0x2b,
	0x39, 0x2d, 0x2d,
	0x3a, 0x37, 0x37,
	0x3b, 0x38, 0x38,
	0x3c, 0x39, 0x39,
	0x3d, 0x3d, 0x3d,
	0x3e, 0x34, 0x34,
	0x3f, 0x35, 0x35,
	0x40, 0x36, 0x36,
	0x41, 0x2c, 0x2c,
	0x42, 0x31, 0x31,
	0x43, 0x32, 0x32,
	0x44, 0x33, 0x33,
	0x45, 0x0d, 0x0d,
	0x46, 0x30, 0x30,
	0x47, 0x2e, 0x2e,

	0x57, 0x20, 0x20,
	0x58, 0x20, 0x20,
	0x35, 0x20, 0x20
};

/*
 *	NORMAL[h
 *	R[hASHIFTȂASHIFT̏
 */
WORD norm_key_table[] = {
	0x5d, 0x0101, 0,
	0x5e, 0x0102, 0,
	0x5f, 0x0103, 0,
	0x60, 0x0104, 0,
	0x61, 0x0105, 0,
	0x62, 0x0106, 0,
	0x63, 0x0107, 0,
	0x64, 0x0108, 0,
	0x65, 0x0109, 0,
	0x66, 0x010a, 0,
	
	0x01, 0x1b, 0x1b,
	0x02, 0x31, 0x21,
	0x03, 0x32, 0x22,
	0x04, 0x33, 0x23,
	0x05, 0x34, 0x24,
	0x06, 0x35, 0x25,
	0x07, 0x36, 0x26,
	0x08, 0x37, 0x27,
	0x09, 0x38, 0x28,
	0x0a, 0x39, 0x29,
	0x0b, 0x30, 0,
	0x0c, 0x2d, 0x3d,
	0x0d, 0x5e, 0x7e,
	0x0e, 0x5c, 0x7c,
	0x0f, 0x08, 0x08,

	0x10, 0x09, 0x09,
	0x11, 0x71, 0x51,
	0x12, 0x77, 0x57,
	0x13, 0x65, 0x45,
	0x14, 0x72, 0x52,
	0x15, 0x74, 0x54,
	0x16, 0x79, 0x59,
	0x17, 0x75, 0x55,
	0x18, 0x69, 0x49,
	0x19, 0x6f, 0x4f,
	0x1a, 0x70, 0x50,
	0x1b, 0x40, 0x60,
	0x1c, 0x5b, 0x7b,
	0x1d, 0x0d, 0x0d,

	0x1e, 0x61, 0x41,
	0x1f, 0x73, 0x53,
	0x20, 0x64, 0x44,
	0x21, 0x66, 0x46,
	0x22, 0x67, 0x47,
	0x23, 0x68, 0x48,
	0x24, 0x6a, 0x4a,
	0x25, 0x6b, 0x4b,
	0x26, 0x6c, 0x4c,
	0x27, 0x3b, 0x2b,
	0x28, 0x3a, 0x2a,
	0x29, 0x5d, 0x7d,

	0x2a, 0x7a, 0x5a,
	0x2b, 0x78, 0x58,
	0x2c, 0x63, 0x43,
	0x2d, 0x76, 0x56,
	0x2e, 0x62, 0x42,
	0x2f, 0x6e, 0x4e,
	0x30, 0x6d, 0x4d,
	0x31, 0x2c, 0x3c,
	0x32, 0x2e, 0x3e,
	0x33, 0x2f, 0x3f,
	0x34, 0x22, 0x5f,

	0x48, 0x12, 0x12,
	0x4d, 0x1e, 0x19,
	0x4b, 0x7f, 0x7f,
	0x4f, 0x1d, 0x02,
	0x50, 0x1f, 0x1a,
	0x51, 0x1c, 0x06,

	0x49, 0x05, 0x05,
	0x4a, 0x0c, 0x0c,
	0x4c, 0x11, 0x11,
	0x4e, 0x0b, 0x0b,

	0x36, 0x2a, 0x2a,
	0x37, 0x2f, 0x2f,
	0x38, 0x2b, 0x2b,
	0x39, 0x2d, 0x2d,
	0x3a, 0x37, 0x37,
	0x3b, 0x38, 0x38,
	0x3c, 0x39, 0x39,
	0x3d, 0x3d, 0x3d,
	0x3e, 0x34, 0x34,
	0x3f, 0x35, 0x35,
	0x40, 0x36, 0x36,
	0x41, 0x2c, 0x2c,
	0x42, 0x31, 0x31,
	0x43, 0x32, 0x32,
	0x44, 0x33, 0x33,
	0x45, 0x0d, 0x0d,
	0x46, 0x30, 0x30,
	0x47, 0x2e, 0x2e,

	0x57, 0x20, 0x20,
	0x58, 0x20, 0x20,
	0x35, 0x20, 0x20
};

/*-[ L[{[h ]-----------------------------------------------------------*/

/*
 *	L[{[h
 *	
 */
BOOL keyboard_init(void)
{
	return TRUE;
}

/*
 *	L[{[h
 *	N[Abv
 */
void keyboard_cleanup(void)
{
}

/*
 *	L[{[h
 *	Zbg
 */
void keyboard_reset(void)
{
	caps_flag = FALSE;
	kana_flag = FALSE;
	ins_flag = FALSE;

	shift_flag = FALSE;
	ctrl_flag = FALSE;
	graph_flag = FALSE;
	break_flag = FALSE;

	key_scan = 0;
	key_fm7 = 0xffff;

	key_repeat_flag = TRUE;
	key_repeat_code = 0;
	key_repeat_cnt = 0;
}

/*
 *	L[{[h
 *	PoCgǂݍ
 */
BOOL keyboard_readb(WORD addr, BYTE *dat)
{
	switch (addr) {
		/* L[R[h */
		case 0xd400:
			if (key_fm7 & 0x0100) {
				*dat = 0xff;
			}
			else {
				*dat = 0x7f;
			}
			return TRUE;

		/* L[R[h */
		case 0xd401:
			*dat = (BYTE)(key_fm7 & 0xff);
			/* 荞off */
			key_irq_flag = FALSE;
			maincpu_irq();
			subcpu_firq();
			return TRUE;

		/* INS LED ON */
		case 0xd40d:
			ins_flag = TRUE;
			keyboard_setled();
			return TRUE;
	}

	return FALSE;
}

/*
 *	L[{[h
 *	PoCg
 */
BOOL keyboard_writeb(WORD addr, BYTE dat)
{
	switch (addr) {
		/* L[R[h */
		case 0xd400:
			return TRUE;

			/* L[R[h */
		case 0xd401:
			return TRUE;

			/* INS LED OFF */
		case 0xd40d:
			ins_flag = FALSE;
			keyboard_setled();
			return TRUE;
	}

	return FALSE;
}

/*
 *	L[{[h
 *	Make/BreakR[hFM-7R[hϊ
 */
WORD keyboard_to_fm7(BYTE dat)
{
	int i;

	switch (dat) {
		/* SHIFT  */
		case 0x53:
			shift_flag = TRUE;
			return 0;
		/* SHIFT  */
		case 0x53 + 0x80:
			shift_flag = FALSE;
			return 0;

		/* ESHIFT  */
		case 0x54:
			shift_flag = TRUE;
			return 0;
		/* ESHIFT  */
		case 0x54 + 0x80:
			shift_flag = FALSE;
			return 0;

		/* CTRL  */
		case 0x52:
			ctrl_flag = TRUE;
			return 0;
		/* CTRL  */
		case 0x52 + 0x80:
			ctrl_flag = FALSE;
			return 0;

		/* GRAPH  */
		case 0x56:
			graph_flag = TRUE;
			return 0;
		/* GRAPH  */
		case 0x56 + 0x80:
			graph_flag = FALSE;
			return 0;

		/* CAP  */
		case 0x55:
			caps_flag = !(caps_flag);
			keyboard_setled();
			return 0;

		/* Ji  */
		case 0x5a:
			kana_flag = !(kana_flag);
			keyboard_setled();
			return 0;

		/* BREAK  */
		case 0x5c:
			break_flag = TRUE;
			maincpu_firq();
			return 0;

		/* BREAK  */
		case 0x5c + 0x80:
			break_flag = FALSE;
			maincpu_firq();
			return 0;
	}

	/* MakeR[h */
	dat &= 0x7f;

	/* CTRL[h */
	if (ctrl_flag) {
		for (i=0; i<(sizeof(ctrl_key_table)/3); i++) {
			if (dat == ctrl_key_table[i * 3]) {
				if (!shift_flag) {
					return ctrl_key_table[i * 3 + 1];
				}
				else {
					return ctrl_key_table[i * 3 + 2];
				}
			}
		}
		return 0;
	}

	/* GRAPH[h */
	if (graph_flag) {
		for (i=0; i<(sizeof(graph_key_table)/3); i++) {
			if (dat == graph_key_table[i * 3]) {
				if (!shift_flag) {
					return graph_key_table[i * 3 + 1];
				}
				else {
					return graph_key_table[i * 3 + 2];
				}
			}
		}
		return 0;
	}

	/* ȃ[h */
	if (kana_flag) {
		for (i=0; i<(sizeof(kana_key_table)/3); i++) {
			if (dat == kana_key_table[i * 3]) {
				if (!shift_flag) {
					return kana_key_table[i * 3 + 1];
				}
				else {
					return kana_key_table[i * 3 + 2];
				}
			}
		}
		return 0;
	}

	/* CAP[h */
	if (caps_flag) {
		for (i=0; i<(sizeof(caps_key_table)/3); i++) {
			if (dat == caps_key_table[i * 3]) {
				if (!shift_flag) {
					return caps_key_table[i * 3 + 1];
				}
				else {
					return caps_key_table[i * 3 + 2];
				}
			}
		}
		return 0;
	}

	/* m[}[h */
	for (i=0; i<(sizeof(norm_key_table)/3); i++) {
		if (dat == norm_key_table[i * 3]) {
			if (!shift_flag) {
				return norm_key_table[i * 3 + 1];
			}
			else {
				return norm_key_table[i * 3 + 2];
			}
		}
	}

	return 0;
}

/*
 *	L[{[h
 *	Make
 */
void keyboard_make(BYTE dat)
{
	WORD code;

	/* R[hL */
	key_scan = dat;

	/* f[^ϊ */
	code = keyboard_to_fm7(dat);

	/* s[gon,off */
	if (shift_flag && ctrl_flag) {
		if (dat == 0x0b) {
			key_repeat_flag = FALSE;
			key_repeat_cnt = 0;
			return;
		}
		if (dat == 0x02) {
			key_repeat_flag = TRUE;
			key_repeat_cnt = 0;
			return;
		}
	}

	/* R[hLȂALĊ荞 */
	if (code) {
		key_fm7 = code;

		if (gui_flag) {
			gui_key_irq();
		}
		else {
			/* 荞 */
			key_irq_flag = TRUE;
			maincpu_irq();
			subcpu_firq();
		}

		/* PFL[ȊOȂAs[gJE^ݒ */
		if (code < 0x100) {
			key_repeat_cnt = 70;
			key_repeat_code = dat;
		}
		else {
			key_repeat_cnt = 0;
		}
	}
	else {
		key_repeat_cnt = 0;
	}
}

/*
 *	L[{[h
 *	Break
 */
void keyboard_break(BYTE dat)
{
	/* [NɋL */
	key_scan = dat | 0x80;

	/* f[^ϊ̂(shift,ctrl,graph,breaktO̍XV) */
	keyboard_to_fm7((BYTE)(dat | 0x80));

	/* s[gȂ */
	if (dat == key_repeat_code) {
		key_repeat_cnt = 0;
	}
}

/*
 *	L[{[h
 *	^C}[
 */
void keyboard_timer(void)
{
	if (key_repeat_flag) {
		if (key_repeat_cnt != 0) {
			key_repeat_cnt--;
			if (key_repeat_cnt == 0) {
				if (gui_flag) {
					/* GUI֗ */
					gui_key_irq();
				}
				else {
					/* CPU֊荞 */
					key_irq_flag = TRUE;
					maincpu_irq();
					subcpu_firq();
				}
				key_repeat_cnt = 7;
			}
		}
	}
}

/*
 *	L[{[h
 *	Z[u
 */
BOOL keyboard_save(int fileh)
{
	if (!file_bool_write(fileh, caps_flag)) {
		return FALSE;
	}
	if (!file_bool_write(fileh, kana_flag)) {
		return FALSE;
	}
	if (!file_bool_write(fileh, ins_flag)) {
		return FALSE;
	}

	if (!file_bool_write(fileh, shift_flag)) {
		return FALSE;
	}
	if (!file_bool_write(fileh, ctrl_flag)) {
		return FALSE;
	}
	if (!file_bool_write(fileh, graph_flag)) {
		return FALSE;
	}
	if (!file_bool_write(fileh, break_flag)) {
		return FALSE;
	}

	if (!file_byte_write(fileh, key_scan)) {
		return FALSE;
	}
	if (!file_word_write(fileh, key_fm7)) {
		return FALSE;
	}

	if (!file_bool_write(fileh, key_repeat_flag)) {
		return FALSE;
	}
	if (!file_word_write(fileh, key_repeat_cnt)) {
		return FALSE;
	}
	if (!file_byte_write(fileh, key_repeat_code)) {
		return FALSE;
	}

	return TRUE;
}

/*
 *	L[{[h
 *	[h
 */
BOOL keyboard_load(int fileh, int ver)
{
	/* o[W`FbN */
	if (ver > 1) {
		return FALSE;
	}

	if (!file_bool_read(fileh, &caps_flag)) {
		return FALSE;
	}
	if (!file_bool_read(fileh, &kana_flag)) {
		return FALSE;
	}
	if (!file_bool_read(fileh, &ins_flag)) {
		return FALSE;
	}

	if (!file_bool_read(fileh, &shift_flag)) {
		return FALSE;
	}
	if (!file_bool_read(fileh, &ctrl_flag)) {
		return FALSE;
	}
	if (!file_bool_read(fileh, &graph_flag)) {
		return FALSE;
	}
	if (!file_bool_read(fileh, &break_flag)) {
		return FALSE;
	}

	if (!file_byte_read(fileh, &key_scan)) {
		return FALSE;
	}
	if (!file_word_read(fileh, &key_fm7)) {
		return FALSE;
	}

	if (!file_bool_read(fileh, &key_repeat_flag)) {
		return FALSE;
	}
	if (!file_word_read(fileh, &key_repeat_cnt)) {
		return FALSE;
	}
	if (!file_byte_read(fileh, &key_repeat_code)) {
		return FALSE;
	}

	return TRUE;
}
