/*
 *	FM-7 EMULATOR "XM7"
 *
 *	Copyright (C) 1999,2000 ohD(ytanaka@ipc-tokai.or.jp)
 *	[ tAZu ]
 */

#include <string.h>
#include <assert.h>
#include "xm7.h"

/*
 *	X^eBbN [N
 */
static int cputype;					/* CPU */
static BYTE opc;					/* IyR[h */
static WORD pc;						/* sOPC */
static WORD addpc;					/* PCZl(ߒ) */
static char linebuf[32];			/* tAZuo̓obt@ */

/*
 *	OnP(0x00)e[u
 */
static char *except1_tbl[] = {
	"NEG   ",
	"NEG   ",
	"NGC   ",
	"COM   ",
	"LSR   ",
	"LSR   ",
	"ROR   ",
	"ASR   ",
	"LSL   ",
	"ROL   ",
	"DEC   ",
	"DCC   ",
	"INC   ",
	"TST   ",
	"JMP   ",
	"CLR   "
};

/*
 *	OnQ(0x10)e[u
 */
static char *except2_tbl[] = {
	NULL,
	NULL,
	"NOP   ",
	"SYNC  ",
	"HALT  ",
	"HALT  ",
	"LBRA  ",
	"LBSR  ",
	"ASLCC ",
	"DAA   ",
	"ORCC  ",
	"NOP   ",
	"ANDCC ",
	"SEX   ",
	"EXG   ",
	"TFR   "
};

/*
 *	u`n(0x20)e[u
 */
static char *branch_tbl[] = {
	"BRA   ",
	"BRN   ",
	"BHI   ",
	"BLS   ",
	"BCC   ",
	"BCS   ",
	"BNE   ",
	"BEQ   ",
	"BVC   ",
	"BVS   ",
	"BPL   ",
	"BMI   ",
	"BGE   ",
	"BLT   ",
	"BGT   ",
	"BLE   "
};

/*
 *	Ou`n(0x10 0x20)e[u
 */
static char *lbranch_tbl[] = {
	NULL,
	"LBRN  ",
	"LBHI  ",
	"LBLS  ",
	"LBCC  ",
	"LBCS  ",
	"LBNE  ",
	"LBEQ  ",
	"LBVC  ",
	"LBVS  ",
	"LBPL  ",
	"LBMI  ",
	"LBGE  ",
	"LBLT  ",
	"LBGT  ",
	"LBLE  "
};

/*
 *	LEAAX^bNn(0x30)e[u
 */
static char *leastack_tbl[] = {
	"LEAX  ",
	"LEAY  ",
	"LEAS  ",
	"LEAU  ",
	"PSHS  ",
	"PULS  ",
	"PSHU  ",
	"PULU  ",
	"ANDCC ",
	"RTS   ",
	"ABX   ",
	"RTI   ",
	"CWAI  ",
	"MUL   ",
	"RST   ",
	"SWI   "
};

/*
 *	Cwg`(0x40)e[u
 */
static char *inha_tbl[] = {
	"NEGA  ",
	"NEGA  ",
	"NGCA  ",
	"COMA  ",
	"LSRA  ",
	"LSRA  ",
	"RORA  ",
	"ASRA  ",
	"LSLA  ",
	"ROLA  ",
	"DECA  ",
	"DCCA  ",
	"INCA  ",
	"TSTA  ",
	"CLCA  ",
	"CLRA  "
};

/*
 *	Cwga(0x50)e[u
 */
static char *inhb_tbl[] = {
	"NEGB  ",
	"NEGB  ",
	"NGCB  ",
	"COMB  ",
	"LSRB  ",
	"LSRB  ",
	"RORB  ",
	"ASRB  ",
	"LSLB  ",
	"ROLB  ",
	"DECB  ",
	"DCCB  ",
	"INCB  ",
	"TSTB  ",
	"CLCB  ",
	"CLRB  "
};

/*
 *	Cwgl(0x60, 0x70)e[u
 */
static char *inhm_tbl[] = {
	"NEG   ",
	NULL,
	NULL,
	"COM   ",
	"LSR   ",
	NULL,
	"ROR   ",
	"ASR   ",
	"LSL   ",
	"ROL   ",
	"DEC   ",
	"DCC   ",
	"INC   ",
	"TST   ",
	NULL,
	"CLR   "
};

/*
 *	`WX^AwWX^n(0x80)e[u
 */
static char *regax_tbl[] = {
	"SUBA  ",
	"CMPA  ",
	"SBCA  ",
	"SUBD  ",
	"ANDA  ",
	"BITA  ",
	"LDA   ",
	"STA   ",
	"EORA  ",
	"ADCA  ",
	"ORA   ",
	"ADDA  ",
	"CMPX  ",
	"JSR   ",
	"LDX   ",
	"STX   "
};

/*
 *	aWX^AcWX^AtWX^n(0xc0)e[u
 */
static char *regbdu_tbl[] = {
	"SUBB  ",
	"CMPB  ",
	"SBCB  ",
	"ADDD  ",
	"ANDB  ",
	"BITB  ",
	"LDB   ",
	"STB   ",
	"EORB  ",
	"ADCB  ",
	"ORB   ",
	"ADDB  ",
	"LDD   ",
	"STD   ",
	"LDU   ",
	"STU   "
};

/*
 *	TFR/EXGe[u
 */
static char *tfrexg_tbl[] = {
	"D",
	"X",
	"Y",
	"U",
	"S",
	"PC",
	NULL,
	NULL,
	"A",
	"B",
	"CC",
	"DP",
	NULL,
	NULL
};

/*
 *	PSH/PULe[u
 */
static char *pshpul_tbl[] = {
	"CC",
	"A",
	"B",
	"DP",
	"X",
	"Y",
	"S/U",
	"PC"
};

/*
 *	CfbNXe[u
 */
static char *idx_tbl[] = {
	"X",
	"Y",
	"U",
	"S"
};

/*-[ ėpTu ]-------------------------------------------------------------*/

/*
 *	f[^tFb`
 */
static BYTE fetch(void)
{
	BYTE dat;

	switch (cputype) {
		case MAINCPU:
			dat = mainmem_readbnio((WORD)(pc + addpc));
			addpc++;
			break;
		case SUBCPU:
			dat = submem_readbnio((WORD)(pc + addpc));
			addpc++;
			break;
		default:
			ASSERT(FALSE);
			break;
	}

	return dat;
}

/*
 *	f[^ǂݏo
 */
static BYTE read_byte(WORD addr)
{
	BYTE dat;

	switch (cputype) {
		case MAINCPU:
			dat = mainmem_readbnio(addr);
			break;
		case SUBCPU:
			dat = submem_readbnio(addr);
			break;
		default:
			ASSERT(FALSE);
			break;
	}

	return dat;
}

/*
 *	PUiPZbg Tu
 */
static void sub1hex(BYTE dat, char *buffer)
{
	char buf[2];

	/* assert */
	ASSERT(buffer);

	buf[0] = dat + 0x30;
	if (dat > 9) {
		buf[0] += 7;
	}

	buf[1] = '\0';
	strcat(buffer, buf);
}

/*
 *	PUiQZbg Tu
 */
static void sub2hex(BYTE dat, char *buffer)
{
	sub1hex((BYTE)(dat >> 4), buffer);
	sub1hex((BYTE)(dat & 0x0f), buffer );
}

/*
 *	PUiQZbg
 */
static void set2hex(BYTE dat)
{
	strcat(linebuf, "$");

	sub2hex(dat, linebuf);
}

/*
 *	PUiSZbg Tu
 */
static void sub4hex(WORD dat, char *buffer)
{
	sub2hex((BYTE)(dat >> 8), buffer);
	sub2hex((BYTE)(dat & 0xff), buffer);
}

/*
 *	PUiSZbg
 */
static void set4hex(WORD dat)
{
	strcat(linebuf, "$");

	sub2hex((BYTE)(dat >> 8), linebuf);
	sub2hex((BYTE)(dat & 0xff), linebuf);
}

/*
 *	POiQZbg
 */
static void set2dec(BYTE dat)
{
	char buf[2];

	buf[1] = '\0';

	/* ʌ */
	buf[0] = dat / 10;
	if (buf[0] > 0) {
		buf[0] += 0x30;
		strcat(linebuf, buf);
	}

	/* ʌ */
	buf[0] = dat % 10;
	buf[0] += 0x30;
	strcat(linebuf, buf);
}

/*
 *	`
 */
static void undef(void)
{
	strcat(linebuf, "?");
}

/*-[ AhbVO[h ]---------------------------------------------*/

/*
 *	eBu[h(PoCg)
 */
static void rel1(void)
{
	BYTE opr;

	/* Iyh擾 */
	opr = fetch();

	/* Zbg */
	if (opr <= 0x7f) {
		set4hex((WORD)(pc + addpc + (BYTE)opr));
	}
	else {
		set4hex((WORD)(pc + addpc - (BYTE)(~opr + 1)));
	}
}

/*
 *	eBu[h(QoCg)
 */
static void rel2(void)
{
	WORD dat;

	/* Iyh擾 */
	dat = fetch() << 8;
	dat |= fetch();

	/* Zbg */
	if (dat <= 0x7fff) {
		set4hex((WORD)(pc + addpc + dat));
	}
	else {
		set4hex((WORD)(pc + addpc - (~dat + 1)));
	}
}

/*
 *	_CNg[h
 */
static void direct(void)
{
	BYTE opr;
	
	/* Iyh擾 */
	opr = fetch();

	/* Zbg */
	strcat(linebuf, "<");
	set2hex(opr);
}

/*
 *	GNXeh[h
 */
static void extend(void)
{
	WORD dat;
	
	/* Iyh擾 */
	dat = fetch() << 8;
	dat |= fetch();

	/* Zbg */
	set4hex(dat);
}

/*
 *	C~fBGCg[h(PoCg)
 */
static void imm1(void)
{
	BYTE opr;

	/* Iyh擾 */
	opr = fetch();

	/* Zbg */
	strcat(linebuf, "#");
	set2hex(opr);
}

/*
 *	C~fBGCg[h(QoCg)
 */
static void imm2(void)
{
	WORD dat;

	/* Iyh擾 */
	dat = fetch() << 8;
	dat |= fetch();

	/* Zbg */
	strcat(linebuf, "#");
	set4hex(dat);
}

/*
 *	CfbNX[h
 */
static void idx(void)
{
	BYTE opr;
	BYTE high, low;
	BYTE offset;
	WORD woffset;

	/* Iyh擾 */
	opr = fetch();
	high = opr & 0xf0;
	low = opr & 0x0f;

	/* 0x00`0x7f5bitItZbg */
	if (opr < 0x80) {
		if (opr & 0x10) {
			/* }CiX */
			offset = ~((opr & 0x0f) | 0xf0) + 1;
			strcat(linebuf, "-");
			set2dec(offset);
		}
		else {
			/* vX */
			offset = low;
			set2dec(offset);
		}

		strcat(linebuf, ",");
		
		/* X, Y, U, S */
		offset = (opr & 0x60) >> 5;
		ASSERT((offset >= 0) && (offset <= 3));
		strcat(linebuf, idx_tbl[offset]);
		return;
	}

	/* 0x80ȏŁAʂ0,1,2,3̓I[gCNgorfNg */
	if (low < 4) {
		if (high & 0x10) {
			if ((low == 0) || (low == 2)) {
				undef();
				return;
			}
			strcat(linebuf, "[");
		}
		strcat(linebuf, ",");

		/* I[gfNg */
		if (low >= 2) {
			strcat(linebuf, "-");
		}
		if (low == 3) {
			strcat(linebuf, "-");
		}

		/* X, Y, U, S */
		offset = (opr & 0x60) >> 5;
		ASSERT((offset >= 0) && (offset <= 3));
		strcat(linebuf, idx_tbl[offset]);

		/* I[gCNg */
		if (low < 2) {
			strcat(linebuf, "+");
		}
		if (low == 1) {
			strcat(linebuf, "+");
		}

		if (high & 0x10) {
			strcat(linebuf, "]");
		}
		return;
	}

	/* 4,5,6,B̓WX^ItZbg */
	if ((low == 4) || (low == 5) || (low == 6) || (low == 11)) {
		if (high & 0x10) {
			strcat(linebuf, "[");
		}

		switch (low) {
			case 4:
				strcat(linebuf, ",");
				break;
			case 5:
				strcat(linebuf, "B,");
				break;
			case 6:
				strcat(linebuf, "A,");
				break;
			case 11:
				strcat(linebuf, "D,");
				break;
			default:
				ASSERT(FALSE);
				break;
		}

		/* X, Y, U, S */
		offset = (opr & 0x60) >> 5;
		ASSERT((offset >= 0) && (offset <= 3));
		strcat(linebuf, idx_tbl[offset]);

		if (high & 0x10) {
			strcat(linebuf, "]");
		}
		return;
	}

	/* 8,C8bitItZbg */
	if ((low == 8) || (low == 12)) {
		if (high & 0x10) {
			strcat(linebuf, "[");
		}

		offset = fetch();

		/* X, Y, U, S, PCR */
		if (low == 12) {
			if (offset >= 0x80) {
				woffset = 0xff00 + offset;
			}
			else {
				woffset = offset;
			}
			set4hex((WORD)(pc + addpc + woffset));
			strcat(linebuf, ",");
			strcat(linebuf, "PCR");
		}
		else {
			if (offset >= 0x80) {
				offset = (~offset + 1);
				strcat(linebuf, "-");
			}
			set2hex(offset);
			strcat(linebuf, ",");
			offset = (opr & 0x60) >> 5;
			ASSERT((offset >= 0) && (offset <= 3));
			strcat(linebuf, idx_tbl[offset]);
		}

		if (high & 0x10) {
			strcat(linebuf, "]");
		}
		return;
	}

	/* 9,D16bitItZbg */
	if ((low == 9) || (low == 13)) {
		if (high & 0x10) {
			strcat(linebuf, "[");
		}

		woffset = (WORD)fetch();
		woffset = (woffset << 8) + (WORD)fetch();

		/* X, Y, U, S, PCR */
		if (low == 13) {
			set4hex((WORD)(woffset + pc + addpc));
			strcat(linebuf, ",");
			strcat(linebuf, "PCR");
		}
		else {
			if (woffset >= 0x8000) {
				woffset = (~woffset + 1);
				strcat(linebuf, "-");
			}
			set4hex(woffset);
			strcat(linebuf, ",");
			offset = (opr & 0x60) >> 5;
			ASSERT((offset >= 0) && (offset <= 3));
			strcat(linebuf, idx_tbl[offset]);
		}

		if (high & 0x10) {
			strcat(linebuf, "]");
		}
		return;
	}

	/* 0x9f,0xbf,0xdf,0xff͗O[addr] */
	if ((opr == 0x9f) || (opr == 0xbf) || (opr == 0xdf) || (opr == 0xff)) {
		strcat(linebuf, "[");
		woffset = (WORD)fetch();
		woffset = (woffset << 8) + (WORD)fetch();
		set4hex(woffset);
		strcat(linebuf, "]");
		return;
	}
	
	/* ȊO͖` */
	undef();
}

/*
 *	TFR,EXG
 */
static void tfrexg(void)
{
	BYTE opr;

	/* Iyh擾 */
	opr = fetch();

	/* 쐬 */
	if (tfrexg_tbl[(opr & 0xf0) >> 4] == NULL) {
		linebuf[0] = '\0';
		undef();
		return;
	}
	strcat(linebuf, tfrexg_tbl[(opr & 0xf0) >> 4]);
	strcat(linebuf, ",");
	if (tfrexg_tbl[opr & 0x0f] == NULL) {
		linebuf[0] = '\0';
		undef();
		return;
	}
 	strcat(linebuf, tfrexg_tbl[opr & 0x0f]);
}

/*
 *	PSH,PUL
 */
static void pshpul(void)
{
	BYTE opr;
	char sreg[2];
	int i;
	int flag;

	/* Iyh擾 */
	opr = fetch();

	/* S,U肷 */
	if (linebuf[3] == 'S') {
		sreg[0] = 'U';
	}
	else {
		sreg[0] = 'S';
	}
	sreg[1] = '\0';

	/* 8] */
	flag = FALSE;
	for (i=0; i<8; i++) {
		if (opr & 0x01) {
			if (flag == TRUE) {
				strcat(linebuf, ",");
			}
			if (i == 6) {
				/* S,U */
				strcat(linebuf, sreg);
			}
			else {
				/* ȊO */
				strcat(linebuf, pshpul_tbl[i]);
			}
			flag = TRUE;
		}
		opr >>= 1;
	}
}

/*-[ IyR[h ]-------------------------------------------------------*/

/*
 *	y[WQ(0x10)
 */
static void page2(void)
{
	BYTE high, low;
	
	/* IyR[hēx擾 */
	opc = fetch();
	high = opc & 0xf0;
	low = opc & 0x0f;

	/* 0x3fSWI2 */
	if (opc == 0x3f) {
		strcat(linebuf, "SWI2  ");
		return;
	}

	/* 0x20̓Ou` */
	if (high == 0x20) {
		/* ``FbN */
		if (lbranch_tbl[low] == NULL) {
			undef();
			return;
		}
		strcat(linebuf, lbranch_tbl[low]);
		rel2();
		return;
	}

	/* 0xc0ȏSWX^ */
	if (opc >= 0xc0) {
		if (low <= 0x0d) {
			undef();
			return;
		}
		if (opc == 0xcf) {
			undef();
			return;
		}
		/* 0x?eLDSA0x?fSTS */
		if (low == 0x0e) {
			strcat(linebuf, "LDS   ");
		}
		if (low == 0x0f) {
			strcat(linebuf, "STS   ");
		}
		/* LDS,STS AhbVO[h */
		switch (high) {
			case 0xc0:
				imm2();
				break;
			case 0xd0:
				direct();
				break;
			case 0xe0:
				idx();
				break;
			case 0xf0:
				extend();
				break;
			default:
				ASSERT(FALSE);
				break;
		}
		return;
	}

	/* 0x80ȏCMPD, CMPY, LDY, STY */
	if (opc >= 0x80) {
		switch (low) {
			case 0x03:
				strcat(linebuf, "CMPD  ");
				break;
			case 0x0c:
				strcat(linebuf, "CMPY  ");
				break;
			case 0x0e:
				strcat(linebuf, "LDY   ");
				break;
			case 0x0f:
				if (high == 0x80) {
					undef();
					return;
				}
				else {
					strcat(linebuf, "STY   ");
				}
				break;
			default:
				undef();
				return;
		}

		/* AhbVO[h */
		switch (high) {
			case 0x80:
				imm2();
				break;
			case 0x90:
				direct();
				break;
			case 0xa0:
				idx();
				break;
			case 0xb0:
				extend();
				break;
			default:
				ASSERT(FALSE);
				break;
		}

		return;
	}

	/* ȊO͖` */
	undef();
}

/*
 *	y[WR(0x11)
 */
static void page3(void)
{
	BYTE high, low;
	
	/* IyR[hēx擾 */
	opc = fetch();
	high = opc & 0xf0;
	low = opc & 0x0f;

	/* 0x3fSWI3 */
	if (opc == 0x3f) {
		strcat(linebuf, "SWI3  ");
		return;
	}

	/* ʂ8,9,A,B */
	if ((high >= 0x80) && (high <= 0xb0)) {
		/* ʃ`FbN */
		switch (low) {
			case 3:
				strcat(linebuf, "CMPU  ");
				break;
			case 12:
				strcat(linebuf, "CMPS  ");
				break;
			default:
				undef();
				return;
		}

		/* AhbVO[h */
		switch (high) {
			case 0x80:
				imm2();
				break;
			case 0x90:
				direct();
				break;
			case 0xa0:
				idx();
				break;
			case 0xb0:
				extend();
				break;
			default:
				ASSERT(FALSE);
				break;
		}

		return;
	}

	/* ȊO͖` */
	undef();
}

/*
 *	OnP(0x00)
 */
static void except1(void)
{
	/* ``FbN */
	if (except1_tbl[opc & 0x0f] == NULL) {
		undef();
		return;
	}
	strcat(linebuf, except1_tbl[opc & 0x0f]);

	/* ׂă_CNg */
	direct();
}

/*
 *	OnQ(0x10)
 */
static void except2(void)
{
	/* 0x10, 0x11Ŏn܂y[W */
	if (opc == 0x10) {
		page2();
		return;
	}
	if (opc == 0x11) {
		page3();
		return;
	}

	/* ``FbN */
	if (except2_tbl[opc & 0x0f] == NULL) {
		undef();
		return;
	}
	strcat(linebuf, except2_tbl[opc & 0x0f]);

	/* 0x16, 0x17̓Ou` */
	if ((opc == 0x16) || (opc == 0x17)) {
		rel2();
		return;
	}

	/* 0x1a, 0x1c̓C~fBGCg */
	if ((opc == 0x1a) || (opc == 0x1c)) {
		imm1();
		return;
	}

	/* 0x1e, 0x1fTFR/EXG */
	if ((opc == 0x1e) || (opc == 0x1f)) {
		tfrexg();
		return;
	}
}

/*
 *	u`n(0x20)
 */
static void branch(void)
{
	strcat(linebuf, branch_tbl[opc - 0x20]);
	rel1();
}

/*
 *	LEA,X^bNn(0x30)
 */
static void leastack(void)
{
	/* IyR[h */
	if (leastack_tbl[opc & 0x0f] == NULL) {
		undef();
		return;
	}
	strcat(linebuf, leastack_tbl[opc & 0x0f]);

	/* LEA̓CfbNX̂ */
	if (opc < 0x34) {
		idx();
		return;
	}

	/* PSH,PUL͐p */
	if (opc < 0x38) {
		pshpul();
		return;
	}

	/* BANDCC̓C~fBGCg */
	if (opc == 0x38) {
		imm1();
	}
}

/*
 *	CwgA,B,Mn(0x40,0x50,0x60,0x70)
 */
static void inhabm(void)
{
	/* 0x6e, 0x7e͓ */
	if (opc == 0x6e) {
		strcat(linebuf, "JMP   ");
		idx();
		return;
	}
	if (opc == 0x7e) {
		strcat(linebuf, "JMP   ");
		extend();
		return;
	}

	switch (opc >> 4) {
		/* `WX^ */
		case 0x4:
			if (inha_tbl[opc & 0x0f] == NULL) {
				undef();
				return;
			}
			strcat(linebuf, inha_tbl[opc & 0x0f]);
			break;
		/* aWX^ */
		case 0x5:
			if (inhb_tbl[opc & 0x0f] == NULL) {
				undef();
				return;
			}
			strcat(linebuf, inhb_tbl[opc & 0x0f]);
			break;
		/* ACfbNX */
		case 0x6:
			if (inhm_tbl[opc & 0x0f] == NULL) {
				undef();
				return;
			}
			strcat(linebuf, inhm_tbl[opc & 0x0f]);
			idx();
			break;
		/* AGNXeh */
		case 0x7:
			if (inhm_tbl[opc & 0x0f] == NULL) {
				undef();
				return;
			}
			strcat(linebuf, inhm_tbl[opc & 0x0f]);
			extend();
			break;
		default:
			ASSERT(FALSE);
			break;
	}
}

/*
 *	`WX^AwWX^n(0x80, 0x90, 0xa0, 0xb0)
 */
static void regax(void)
{
	/* 0x87, 0x8f͉B */
	if (opc == 0x87) {
		strcat(linebuf, "FLAG  ");
		imm1();
		return;
	}
	if (opc == 0x8f) {
		strcat(linebuf, "FLAG  ");
		imm2();
		return;
	}

	/* 0x8dBSR */
	if (opc == 0x8d) {
		strcat(linebuf, "BSR   ");
		rel1();
		return;
	}

	/* ȊO */
	strcat(linebuf, regax_tbl[opc & 0x0f]);

	/* AhbVO[h */
	switch (opc >> 4) {
		case 0x8:
			if ((opc == 0x83) || (opc == 0x8c) || (opc == 0x8e)) {
				imm2();
			}
			else {
				imm1();
			}
			break;
		case 0x9:
			direct();
			break;
		case 0xa:
			idx();
			break;
		case 0xb:
			extend();
			break;
		default:
			ASSERT(FALSE);
			break;
	}
}

/*
 *	aWX^AcWX^AtWX^n(0xc0, 0xd0, 0xe0, 0xf0)
 */
static void regbdu(void)
{
	/* 0xc7, 0xcd, 0xcf͉B */
	if (opc == 0xc7) {
		strcat(linebuf, "FLAG  ");
		imm1();
		return;
	}
	if (opc == 0xcd) {
		strcat(linebuf, "HALT  ");
		return;
	}
	if (opc == 0xcf) {
		strcat(linebuf, "FLAG  ");
		imm2();
		return;
	}

	/* ȊO */
	strcat(linebuf, regbdu_tbl[opc & 0x0f]);

	/* AhbVO[h */
	switch (opc >> 4) {
		case 0xc:
			if ((opc == 0xc3) || (opc == 0xcc) || (opc == 0xce)) {
				imm2();
			}
			else {
				imm1();
			}
			break;
		case 0xd:
			direct();
			break;
		case 0xe:
			idx();
			break;
		case 0xf:
			extend();
			break;
		default:
			ASSERT(FALSE);
			break;
	}
}

/*-[ C ]---------------------------------------------------------------*/

/*
 *	tAZu{
 */
int disline(int cpu, WORD pcreg, char *buffer)
{
	int i;
	int j;

	/* assert */
	ASSERT((cpu == MAINCPU) || (cpu == SUBCPU));
	ASSERT(buffer);

	/* ݒ */
	cputype = cpu;
	pc = pcreg;
	addpc = 0;
	linebuf[0] = '\0';

	/* 擪̃oCgǂݏo */
	opc = fetch();

	/* O[v */
	switch ((int)(opc >> 4)) {
		case 0x0:
			except1();	
			break;
		case 0x1:
			except2();
			break;
		case 0x2:
			branch();
			break;
		case 0x3:
			leastack();
			break;
		case 0x4:
		case 0x5:
		case 0x6:
		case 0x7:
			inhabm();
			break;
		case 0x8:
		case 0x9:
		case 0xa:
		case 0xb:
			regax();
			break;
		case 0xc:
		case 0xd:
		case 0xe:
		case 0xf:
			regbdu();
			break;
		default:
			ASSERT(FALSE);
	}

	/* ߃f[^Zbg */
	buffer[0] = '\0';

	sub4hex(pcreg, buffer);
	strcat(buffer, " ");
	for (i=0; i<addpc; i++) {
		sub2hex(read_byte((WORD)(pcreg + i)), buffer);
		strcat(buffer, " ");
	}

	/* buffer20oCg+'\0'ɂȂ悤 */
	j = strlen(buffer);
	if (j < 20) {
		for (i=0; i<20 - j; i++) {
			buffer[i + j] = ' ';
		}
		buffer[i + j] = '\0';
	}

	strcat(buffer, linebuf);
 	return (int)addpc;
}
