/* @(#)wc.c	1.6 */

#include <config.h>

#include <stdio.h>
#include <string.h>

#include <utf.h>
#include <utftools.h>

static char sccsid[] = "@(#)wc.c	1.6";

static char *usage = "wc [-v] [-lwrbc] [--] [file ...]";

static int countlines			= 0;
static int countwords			= 0;
static int countrunes			= 0;
static int countbadsequences	= 0;
static int countchars			= 0;

static unsigned long totallines			= 0;
static unsigned long totalwords			= 0;
static unsigned long totalrunes			= 0;
static unsigned long totalbadsequences	= 0;
static unsigned long totalchars			= 0;

static void wcfile(char *);
static void report(unsigned long, unsigned long, unsigned long, unsigned long, unsigned long, char *);

int
main(int argc, char **argv)
{
	int c;
	int files;

	errorname = "wc";

	opterr = 0;
	while ((c = getopt(argc, argv, "vlwrbc")) != -1) {
		switch (c) {
		case 'v':
			fprintf(stderr, "%s\n", sccsid);
			exit(0);
		case 'l':
			countlines = 1;
			break;
		case 'w':
			countwords = 1;
			break;
		case 'r':
			countrunes = 1;
			break;
		case 'b':
			countbadsequences = 1;
			break;
		case 'c':
			countchars = 1;
			break;
		default:
			fprintf(stderr, "usage: %s\n", usage);
			exit(1);
		}
	}

	if (countlines == 0 &&
		countwords == 0 &&
		countrunes == 0 &&
		countbadsequences == 0 &&
		countchars == 0)
		countlines = countwords = countchars = 1;

	files = 0;
	
	if (argv[optind] == 0) {
		wcfile("-");
		++files;
	} else {
		while (argv[optind] != 0) {
			wcfile(argv[optind]);
			++optind;
			++files;
		}
	}

	if (files != 1)
		report(totallines, totalwords, totalrunes, totalbadsequences, totalchars, "total");
	
	exit(0);
}

static void
wcfile(char *fname)
{
	FILE *fp;
	unsigned long lines, words, runes, badsequences, chars;
	int c, i;
	char s[UTFmax];
	Rune r;
	int space;

	if (strcmp(fname, "-") == 0)
		fp = stdin;
	else {
		fp = fopen(fname, "r");
		if (fp == 0)
			fatal("unable to open %s", fname);
	}

	lines = words = runes = badsequences = chars = 0;
	
	i = 0;
	c = getc(fp);
	space = 1;

	while (c != EOF) {
		++chars;
		s[i] = c;
		++i;
		if (fullrune(s, i)) {
			chartorune(&r, s);
			++runes;
			if (r == '\n') {
				++lines;
				space = 1;
			} else if (r == ' ' || r == '\t') {
				space = 1;
			} else {
				words += space;
				space = 0;
			}
			i = 0;
		} else if (i == UTFmax) {
			++badsequences;
			++runes;
			i = 0;
		}
		c = getc(fp);
	}
	if (i != 0) {
		++badsequences;
		++runes;
	}

	report(lines, words, runes, badsequences, chars, fname);
	
	totallines        += lines;
	totalwords        += words;
	totalrunes        += runes;
	totalbadsequences += badsequences;
	totalchars        += chars;

	if (strcmp(fname, "-") != 0)
		fclose(fp);
		
}

static void
report(unsigned long lines, unsigned long words, unsigned long runes, 
	unsigned long badsequences, unsigned long chars, char *fname)
{
	if (countlines)
		printf(" %9lu", lines);
	if (countwords)
		printf(" %9lu", words);
	if (countrunes)
		printf(" %9lu", runes);
	if (countbadsequences)
		printf(" %9lu", badsequences);
	if (countchars)
		printf(" %9lu", chars);
	if (strcmp(fname, "-") != 0)
		printf(" %s", fname);
	printf("\n");
}
