#include <config.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <utf.h>
#include <utftools.h>

static char sccsid[] = "@(#)fmt.c	1.6";

static char usage[] = "fmt [-v] [-l length] [--] [file ...]";

static int tabs;
static int maxlinelen	= 70;

static Rune *word         = 0;
static Rune *leadingspace = 0;
static int outputlinelen  = 0;

static void	fmtfile(char *);
static int	fmtline(FILE *);
static void	emit(long r);

static size_t rlen(Rune *);
static Rune * rappend(Rune *, Rune);

int
main(int argc, char **argv)
{
	int c;
	
	errorname = "fmt";
	
	tabs = 8;
	
	opterr = 0;
	while ((c = getopt(argc, argv, "vl:t:")) != -1) {
		switch (c) {
		case 'v':
			fprintf(stderr, "%s\n", sccsid);
			exit(0);
		case 'l':
			errno = 0;
			maxlinelen = strtoix(optarg, 10);
			if (errno == ERANGE || errno == EDOM || tabs < 0)
				fatal("-l option must specify a decimal integer greater than zero");
			break;
		case 't':
			errno = 0;
			tabs = strtoix(optarg, 10);
			if (errno == ERANGE || errno == EDOM || tabs < 0)
				fatal("-t option must specify a decimal integer greater than zero");
			break;
		default:
			fprintf(stderr, "usage: %s\n", usage);
			exit(1);
		}
	}

	if (argv[optind] == 0) {
		fmtfile("-");
	} else {
		while (argv[optind] != 0) {
			fmtfile(argv[optind]);
			++optind;
		}
	}

	exit(0);
}

static void
fmtfile(char *fname)
{
	FILE *fp;

	if (strcmp(fname, "-") == 0)
		fp = stdin;
	else {
		fp = fopen(fname, "r");
		if (fp == 0)
			fatal("unable to open %s", fname);
	}

	while (fmtline(fp) != EOF)
		;

	if (strcmp(fname, "-") != 0)
		fclose(fp);
}


static int
fmtline(FILE *fp)
{
	int r;
	int i;

	/* gather leading horizontal space */
	if (leadingspace != 0)
		free(leadingspace);
	leadingspace = 0;
	r = fgetr(fp);
	while (r == ' ' || r == '\t') {
		leadingspace = rappend(leadingspace, r);
		r = fgetr(fp);
	}
			
	/* deal with EOF and no words */
	if (r == EOF) {
		if (outputlinelen != 0)
			emit('\n');
		if (leadingspace != 0)
			emit('\n');
		return EOF;
	}

	/* deal with \n and no words */
	if (r == '\n') {
		if (outputlinelen != 0)
			emit('\n');
		emit('\n');
		return r;
	}

	/* deal with words */

	while (r != '\n' && r != EOF) {

		/* gather word */
		word = 0;
		while (r != EOF && r != ' ' && r != '\t' && r != '\n') {
			word = rappend(word, r);
			r = fgetr(fp);
		}

		/* emit appropriate space */
		if (outputlinelen != 0) {
			if (outputlinelen + 1 + rlen(word) > maxlinelen)
				emit('\n');
			else
				emit(' ');
		}

		/* emit word */
		for (i = 0; word[i] != 0; ++i)
			emit(word[i]);
			
		free(word);

		/* skip horizontal space */
		while (r == ' ' || r == '\t')
			r = fgetr(fp);

	}

	return r;
}

static void
emit(long r)
{
	int i;

	if (outputlinelen == 0) {
		if (leadingspace != 0) {
			for (i = 0; leadingspace[i] != 0; ++i) {
				fputr(leadingspace[i], stdout);
				if (leadingspace[i] == ' ')
					++outputlinelen;
				else
					outputlinelen = (outputlinelen / tabs + 1) * tabs;
			}
		}
	}

	if (r != EOF)
		fputr(r, stdout);

	if (r == '\n')
		outputlinelen = 0;
	else
		++outputlinelen;

	return;
}

static size_t
rlen(Rune *r)
{
	size_t len = 0;
	while (r[len] != 0)
		++len;
	return len;
}

static Rune *
rappend(Rune *old, Rune r)
{
	Rune *new;
	size_t len;

	if (old == 0)
		len = 0;
	else
		len = rlen(old);
	
	new = realloc(old, (len + 2) * sizeof (Rune));
	if (new == 0)
		fatal("realloc() failed");
	
	new[len] = r;
	new[len + 1] = 0;
	
	return new;
}
