/**
 * Uptime Client v4.2
 *
 *   This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 *   This is the brand new version of the Uptime Client, using the new
 * uptimes protocol v4. Do with it whatever you like and have fun using
 * it!
 *
 * alex@uptimes.net
 */

/* Common system includes */
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>

/* Include if not compiling on WinNT */
#if !defined(PLATFORM_WINNT)
  #include <unistd.h>
  #include <netdb.h>
  #include <sys/time.h>
  #include <netinet/in.h>
  #include <sys/socket.h>
  #include <sys/utsname.h>
  #include <syslog.h>
#endif

/* Include if not compiling on WinNT or BeOS */
#if !defined(PLATFORM_WINNT) && !defined(PLATFORM_BEOS)
  #include <sysexits.h>
  #include <arpa/inet.h>
#endif

#ifdef PLATFORM_WINNT
  /* Special Windows includes */
  #include <windows.h>
#endif

#ifdef PLATFORM_BEOS
  /* Special BeOS includes */
  #include <OS.h>
  #include <netdb.h>
#endif

#ifdef PLATFORM_ULTRIX
  /**
   * Special Ultrix includes. Are they all needed
   * here in upclient.c? or only in stats-sol.c?
   */
  #include <fcntl.h>
  #include <nlist.h>
  #include <sys/fixpoint.h>
  #include <sys/cpudata.h>
#endif

#ifdef PLATFORM_AIX
  /**
   * Special AIX includes. Are they all needed
   * in in upclient.c? or only in stats-aix.c?
   */
  #include <utmp.h>
  #include <fcntl.h>
  #include <sys/nlist.h>
  #include <sys/select.h>
  #include <sys/sysinfo.h>
  #include <sys/systemcfg.h>
#endif

#ifdef PLATFORM_SOLARIS
  /**
   * Special Solaris includes. Are they all needed
   * here in upclient.c? or only in stats-sol.c?
   */
  #include <utmpx.h>
  #include <fcntl.h>
  #include <sys/stat.h>
  #include <sys/types.h>
  #include <sys/systeminfo.h>
#endif

/* Satisfy our own needs */
#include "upclient.h"
#include "version.h"
#include "network.h"
#include "base64.h"
#include "options.h"
#include "stats.h"

#ifdef PLATFORM_UNIXWARE
#define NUM_IDLE_ELEMENTS ((24 * 60 * 60) / INTERVAL) /* Calculate idle time as an average over the whole day */
int past_idle_times[NUM_IDLE_ELEMENTS] = { -1 };
#endif


/**
 * @desc	Actual function that makes the connection to the
 *		Uptimes Project server after it composed the packet
 *		to send upstream.
 */
char *www_post(const char *hostname, const char *url, const char *proxyusername, const char *proxypassword, const char *data) {
  int sock;
  struct sockaddr_in lsin;
  struct hostent *he;
  char auth[128];
  char header[512];
  int n;
  char rec[4096];
  char *s;
  char *s2;

  /* Create socket */
  sock = socket(AF_INET, SOCK_STREAM, 0);
  if(sock < 0) {
    return "001 ERR: Can't create socket";
  }

  memset(&lsin, 0, sizeof(lsin));
  lsin.sin_family = AF_INET;

  /* Direct connect or through a proxy server? */
  if(have_proxyserver) {
    lsin.sin_addr.s_addr = inet_addr(cfg_proxyserver);
    lsin.sin_port = htons(cfg_proxyport);
  } else {
    lsin.sin_addr.s_addr = inet_addr(hostname);
    lsin.sin_port = htons(80);
  }

  if((unsigned int)lsin.sin_addr.s_addr == -1) {
    /* Resolve host */
    if(have_proxyserver)
      he = gethostbyname(cfg_proxyserver);
    else
      he = gethostbyname(hostname);

    if(!he) {
      close(sock);
      if(have_proxyserver)
        return "002 ERR: Can't resolve hostname of proxyserver";
      else
        return "002 ERR: Can't resolve hostname of uptime server";
    }
    memcpy(&lsin.sin_addr, he->h_addr, he->h_length);
  }

  /* Connect to Uptime Server */
  if(connect(sock, (struct sockaddr *)&lsin, sizeof(lsin)) < 0) {
    close(sock);
    if(have_proxyserver)
      return "003 ERR: Can't connect to Proxy Server";
    else
      return "003 ERR: Can't connect to Uptime Server";
  }

  /* ... and proxy username and password */
  if(have_proxyuser) {
    sprintf(auth, "%s:%s", cfg_proxyuser, cfg_proxypass);
    s2 = base64_encode(auth);
  }

  /* Send POST */
  if(have_proxyserver) {
    if(have_proxyuser) {
      sprintf(header, "POST http://%s%s HTTP/1.1\r\nHost: %s\r\nProxy-Connection: Close\r\nUser-Agent: upclient/%s/uptime-client-%s\r\nProxy-Authorization: Basic %s\r\nContent-type: application/x-www-form-urlencoded\r\nContent-length: %d\r\n\r\n", hostname, url, hostname, PROTOCOL, VERSION, s2, strlen(data) - 1);
    } else {
      sprintf(header, "POST http://%s%s HTTP/1.1\r\nHost: %s\r\nProxy-Connection: Close\r\nUser-Agent: upclient/%s/uptime-client-%s\r\nContent-type: application/x-www-form-urlencoded\r\nContent-length: %d\r\n\r\n", hostname, url, hostname, PROTOCOL, VERSION, strlen(data) - 1);
    }
  } else {
    sprintf(header, "POST %s HTTP/1.1\r\nHost: %s\r\nConnection: Close\r\nUser-Agent: upclient/%s/uptime-client-%s\r\nContent-type: application/x-www-form-urlencoded\r\nContent-length: %d\r\n\r\n", url, hostname, PROTOCOL, VERSION, strlen(data) - 1);
  }

#if defined(DEBUG)
  /* Print the header that will be send */
  printf("HTTP header:\n%s", header);
#endif

  if(have_proxyuser) {
    free(s2);
  }

  if(socket_write(sock, header, strlen(header)) < 0) {
    close(sock);
    return "004 ERR: Can't send header";
  }

  if(socket_write(sock, data, strlen(data)) < 0) {
    close(sock);
    return "005 ERR: Can't send data";
  }

  /* Wait for reply */
  n = socket_read(sock, rec, sizeof(rec)-1);
  if(n < 0) {
    close(sock);
    return "006 ERR: Can't receive data";
  }
  rec[n] = '\0';
  s = strstr(rec, "\r\n\r\n");
  if(s) {
    close(sock);
    return s + 4;
  }

  /* Always cleanup your own mess */
  close(sock);

  return "007 ERR: No result received";
}

/**
 * @desc	Send our little update to the Uptimes Project(tm) server
 */
void send_update(void) {
#if !defined(PARANOID)
  char *ret;
#endif
  unsigned long uptime = 0;
  double load = -1.0;
  int idle = -1;
  char os[OS_SIZE] = "";
  char oslevel[OSLEVEL_SIZE] = "";
  char cpu[CPU_SIZE] = "";
  char data[256];

  /* Get the machine's uptime and other stats */
  getstats(&uptime, &load, &idle, os, oslevel, cpu);

#if defined(DEBUG)
  printf("Stats: uptime %ld, load %.1f, idle %d, OS %s, level %s, CPU %s.\n", uptime, load, idle, os, oslevel, cpu);
#endif

  /* Create packet */
#ifdef PLATFORM_WINNT
  /* the sprintf(data, "%sfoo", data); doesn't work for bcb5. This is a temp solution */
  sprintf(data, "auth=%s&uptime=%ld&os=%s&oslevel=%s&cpu=%s\n", cfg_authkey, uptime, os, oslevel, cpu);
#else
  sprintf(data, "auth=%s&uptime=%ld", cfg_authkey, uptime);
  
  if(cfg_sendload)
    sprintf(data, "%s&load=%.2f", data, load);

  if(cfg_sendidle)
    sprintf(data, "%s&idle=%d", data, idle);

  if(cfg_sendos)
    sprintf(data, "%s&os=%s", data, os);

  if(cfg_sendoslevel)
    sprintf(data, "%s&oslevel=%s", data, oslevel);

  if(cfg_sendcpu)
    sprintf(data, "%s&cpu=%s", data, cpu);

  sprintf(data, "%s\n", data);
#endif

  /* Send packet */
#if !defined(PARANOID)
  if(have_proxyuser)
    ret = www_post(cfg_upserver, "/server.html", cfg_proxyuser, cfg_proxypass, data);
  else
    ret = www_post(cfg_upserver, "/server.html", NULL, NULL, data);
#endif

#ifdef DEBUG
  #ifdef PARANOID
    printf("Paranoid mode enabled, no packet sent\n");
  #else
    printf("The uptime server returned: %s\n", ret);
  #endif
#endif
}

/**
 * @desc	For most Unices, this makes Upclient ticks
 */
void timertick(int sig) {
  send_update();
}

/**
 * @desc	For Unices, write a nice process ID file.
 */
void write_pidfile(int pid) {
  FILE *fp;
  fp = fopen(cfg_pidfile, "w");
  if(!fp) {
#if !defined(PLATFORM_WINNT) && !defined(PLATFORM_BEOS)
    syslog(LOG_ERR, "Warning: Can't write pidfile: %s\n", strerror(errno));
#else
    fprintf(stderr, "Warning: Can't write pidfile: %s\n", strerror(errno));
#endif
    return;
  }
  fprintf(fp, "%d\n", pid);
  fclose(fp);
}

/**
 * @desc	main(), if you don't know what this does, please leave Right Now
 */
int main(int argc, char **argv) {
#if !defined(PLATFORM_UNIXWARE) && !defined(PLATFORM_WINNT) && !defined(PLATFORM_BEOS) /* Warning Eater(tm) */
  struct sigaction sa;
  struct itimerval timer;
#endif
#if !defined(PLATFORM_WINNT) && !defined(DEBUG)             /* Yet Another Warning Nullifyer (yawn) */
  int pid;
#endif

  /* Read configuration file */
  read_config();

#if !defined(PLATFORM_WINNT) && !defined(DEBUG)
  /* Fork into background */
  switch(pid = fork()) {
    case -1:
      fprintf(stderr, "ERROR: Couldn't fork into background (%s)\n", strerror(errno));
      exit(1);
    case 0:
      close(0); close(1); close(2);
      break;
    default:
#if !defined(PLATFORM_BEOS)
      syslog(LOG_INFO, "Uptime Client started (pid %d)", pid);
#endif
      write_pidfile(pid);
      exit(0);
  }
#endif

#if !defined(PLATFORM_UNIXWARE) && !defined(PLATFORM_WINNT) && !defined(PLATFORM_BEOS)
  /* Setup the timer */
  sigemptyset(&sa.sa_mask);
  sa.sa_flags = 0;
  sa.sa_handler = timertick;
  if(sigaction(SIGALRM, &sa, NULL) == -1) {
    perror("sigaction");
    exit(1);
  }

  timer.it_value.tv_sec = 1;
  timer.it_value.tv_usec = 0;
  timer.it_interval.tv_sec = cfg_interval;
  timer.it_interval.tv_usec = 0;
  if(setitimer(ITIMER_REAL, &timer, NULL) == -1) {
    perror("setitimer");
    exit(2);
  }

  /* Now go idle(tm) */
  for(;;) {
    pause();
  }
#endif

#if defined(PLATFORM_UNIXWARE)
  for(;;) {
    send_update();
  }
#elif defined(PLATFORM_BEOS)
  for(;;) {
    send_update();
    sleep(cfg_interval);
  }
#elif defined(PLATFORM_WINNT)
  startWinsock();
  printf("Uptime Client started\n");
  for(;;) {
    send_update();
    Sleep(cfg_interval * 1000); /* sleep interval in seconds */
  }
#endif

#if !defined(PLATFORM_UNIXWARE) /* warning eater */
  return 1;
#endif
}
