/**
 * @filename	stats-aix.c
 *
 * @desc	Retrieve stats for the AIX platform
 */

/* My includes */
#include "upclient.h"
#include "options.h"
#include "stats.h"

/* System includes */
#include <stdio.h>
#include <errno.h>
#include <syslog.h>
#include <utmp.h>
#include <fcntl.h>
#include <sys/nlist.h>
#include <sys/select.h>
#include <sys/sysinfo.h>
#include <sys/systemcfg.h>

/**
 * @desc	Get statistics
 */
void getstats(unsigned long *puptime, double *pload, int *pidle, char *os, char *oslevel, char *cpu) {
  struct utmp ut;
  struct sysinfo sys;
  struct utsname uts;
  struct nlist kernelnames[2];
  int fd, bytes, avenrun[3];
  time_t prtime;

  /* get uptime from utmp: */
  if((fd = open(UTMP_FILE, O_RDONLY)) < 0) {
    perror("open");
    exit(-1);
  }
  while((bytes = read(fd, &ut, sizeof(ut)))) {
    if(bytes == -1) {
      perror("read");
      exit(-2);
    }
    if(ut.ut_type == BOOT_TIME) {
      time(&prtime);
      *puptime = (prtime - ut.ut_time) / 60;
      break;
    }
  }
  /* open /dev/kmem for reading load average and idle: */
  if((fd = open("/dev/kmem", O_RDONLY)) < 0) {
    perror("open");
    exit(-3);
  }

  if(cfg_sendload) {
    kernelnames[1].n_name = NULL;
    kernelnames[0].n_name = "avenrun";

    if(knlist(kernelnames,1,sizeof(struct nlist)) == -1) {
      perror("knlist");
      exit(-4);
    }
     if(lseek(fd, kernelnames->n_value, SEEK_SET) == -1) {
      perror("lseek");
      exit(-5);
    }
    if(read(fd, avenrun, sizeof(avenrun)) < 0) {
      perror("read");
      exit(-6);
    }
    *pload = avenrun[2]/65536.0;
  }

  if(cfg_sendidle) {
    kernelnames[1].n_name = NULL;
    kernelnames[0].n_name = "sysinfo";

    if(knlist(kernelnames,1,sizeof(struct nlist)) == -1) {
      perror("knlist");
      exit(-7);
    }
     if(lseek(fd, kernelnames->n_value, SEEK_SET) == -1) {
      perror("lseek");
      exit(-8);
    }
    if(read(fd, &sys, sizeof(sys)) < 0) {
      perror("read");
      exit(-9);
    }
    *pidle = (int)(100.0 * (1.0 - ((double) sys.runocc / (prtime - ut.ut_time))));
  }

  if(cfg_sendos) {
    uname(&uts);
  }

  if(cfg_sendos) {
    strncpy(os, uts.sysname, OS_SIZE - 1);
    if(cfg_sendoslevel) {
      strncpy(oslevel, uts.version, OSLEVEL_SIZE - 1);
    }
  }

  if(cfg_sendcpu) {
    /* rs1, rs2 and rsc architectures: */
    if(__power_rs1())
      strncpy(cpu, "power_rs1", CPU_SIZE -1);
    else if(__power_rsc())
      strncpy(cpu, "power_rsc", CPU_SIZE -1);
    else if(__power_rs2())
      strncpy(cpu, "power_rs2", CPU_SIZE -1);
    /* powerpc: */
    else if(__power_601())
      strncpy(cpu, "power_601", CPU_SIZE -1);
    else if(__power_603())
      strncpy(cpu, "power_603", CPU_SIZE -1);
    else if(__power_604())
      strncpy(cpu, "power_604", CPU_SIZE -1);
    else if(__power_620())
      strncpy(cpu, "power_620", CPU_SIZE -1);
    /* any rs or ppc not listed above: */
    else if(__power_rs())
      strncpy(cpu, "power_rs", CPU_SIZE -1);
    else strncpy(cpu, "power_pc", CPU_SIZE -1);
  }

  /* close /dev/kmem: */
  close(fd);
}
