/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include "ip.h"  	// for hdr_ip def
#include "flags.h"  // for hdr_flags def

#include "nsProtoAgent.h" 

static class NsProtoAgentClass : public TclClass
{
	public:
		NsProtoAgentClass() : TclClass("Agent/Protean") {}
	 	TclObject *create(int argc, const char*const* argv) 
			{return (new NsProtoAgent());}
} class_protean_agent;	


NsProtoAgent::NsProtoAgent()
	: Agent(PT_UDP), 
      udp_socket(NULL), 
      mcast_ttl(32), 
      mcast_loopback(false),
      example(this)
{
}  // end NsProtoAgent::NsProtoAgent()


int NsProtoAgent::command(int argc, const char*const* argv) 
{
    // Intercept commands as needed here
    if (2 == argc)
    {
        if (!strcmp(argv[1], "stop"))
        {
            example.Stop();
            return TCL_OK  ; 
        }
        else if (!strcmp(argv[1], "addr"))
		{
			Tcl& tcl = Tcl::instance();
			tcl.resultf("%lu", GetAgentId());
			return TCL_OK;
		}   
    }
    else if (4 == argc)
    {
        if (!strcmp(argv[1], "start"))
        {
            nsaddr_t dstAddr;
            unsigned short dstPort;
            if ((1 == sscanf(argv[2], "%i", &dstAddr)) && 
                (1 == sscanf(argv[3], "%hu", &dstPort)))
            {
                example.Start(dstAddr, dstPort);
                return TCL_OK;
            }
            else
            {
                fprintf(stderr, "NsProtoAgent: Bad destination address: %s:%s\n", 
                                argv[2], argv[3]);
                return TCL_ERROR;    
            }             
        }   
    }
    return Agent::command(argc, argv);
}  // end NsProtoAgent::command()


// Right now we just use whatever port the agent got when it
// was attached to a node.  In the future, our socket class
// could actually create a new Agent/UDP agent class for each
// socket opened (and we could extend to other socket types 
// (e.g. TCP)
bool NsProtoAgent::OpenSocket(UdpSocket* theSocket, unsigned short* thePort)
{
    // Ns agent ports are assigned as nodes are attached
    if (*thePort && (here_.port_ != *thePort))
    {
        fprintf(stderr, "NsProtoAgent::Bind to arbitrary port not yet allowed!\n");
        return false;
    }
    else
    {
        if (here_.port_ < 0)
        {
            fprintf(stderr, "NsProtoAgent::OpenSocket() Agent not attached to node?\n");
            return false;
        }
        else
        {
            udp_socket = theSocket;
            *thePort = here_.port_;
            return true;
        }
    }
}  // end NsProtoAgent::OpenSocket()

// (TBD) Different "sockets" could join different groups but
// we would need to manage state on a per socket basis then
bool NsProtoAgent::JoinGroup(UdpSocket* /*theSocket*/, SIMADDR theGroup)
{
	Tcl& tcl = Tcl::instance();	
	tcl.evalf("%s set node_", this->name());
	const char *result = tcl.result();
    Tcl::instance().evalf("%s join-group %s 0x%x", result, 
                          this->name(), theGroup);	
	return true;
}  // end NsProtoAgent::JoinGroup() 

void NsProtoAgent::LeaveGroup(UdpSocket* /*theSocket*/, SIMADDR theGroup)
{
	Tcl& tcl = Tcl::instance();	
	tcl.evalf("%s info vars node_", this->name());
	const char *result = tcl.result();
	Tcl::instance().evalf("%s leave-group %s 0x%x", result, 
                          this->name(), theGroup);	
}  // end NsProtoAgent::LeaveGroup() 


bool NsProtoAgent::SendTo(UdpSocket* /*theSocket*/, 
                          SIMADDR theAddr, unsigned short thePort,
                          char* buffer, unsigned int len)
{	
    // Allocate _and_ init packet
	Packet* p = allocpkt(len);
    
	// Set packet destination addr/port
	hdr_ip* iph = hdr_ip::access(p);
	hdr_flags::access(p)->ect() = 1;   
	iph->daddr() = theAddr;
	iph->dport() = thePort;
    
    if (theAddr & 0x80000000)
        iph->ttl() = mcast_ttl;
	else
        iph->ttl() = 255;
	
	hdr_cmn* ch = hdr_cmn::access(p);  
    // Other packet types my be desired
    ch->ptype() = PT_UDP;
	// Copy data to be sent into packet
	char* data = (char*) p->accessdata();
	memcpy(data, buffer, len);
	// Set packet length field to UDP payload size for now
    // (TBD) Should we add UDP/IP header overhead?
	ch->size() = len;
	send(p, 0);
    return true;
}  // end NsProtoAgent::SendTo()

bool NsProtoAgent::RecvFrom(UdpSocket* /*theSocket*/,
                            char* buffer, unsigned int* buflen,
                            SIMADDR* srcAddr, unsigned short* srcPort)
{
    if (recv_data && (*buflen >= recv_len))
    {
        memcpy(buffer, recv_data, recv_len);
        *buflen = recv_len;
        *srcAddr = recv_addr;
        *srcPort = recv_port;
        return true;
    }
    else
    {
        return false;   
    }
}  // end NsProtoAgent::RecvFrom()

// If multiple sockets/agent are to be supported, this function
// will need to be changed to demux recv packets to the 
// correct "socket(s)" based on packet destination address/port

void NsProtoAgent::recv(Packet *p, Handler */*h*/)
{   
    // Give the packet to the socket given mcast/loopback/packet source
	nsaddr_t srcAddr = hdr_ip::access(p)->src().addr_;
    unsigned short srcPort = hdr_ip::access(p)->src().port_;
	nsaddr_t dstAddr = hdr_ip::access(p)->dst().addr_;    
    bool isUnicast = (0 == (dstAddr & 0x80000000));  
    
    if ((srcAddr != here_.addr_) ||
        (isUnicast) ||
        (mcast_loopback))
	{		
		// Note: In ns, we have to be careful that our "udp_socket"
        // was not closed before the packet was received    
        if (udp_socket)
        {
            recv_data = (char*)p->accessdata();
            recv_len = ((PacketData*)(p->userdata()))->size();
            recv_addr = srcAddr;
            recv_port = srcPort;
            udp_socket->OnReceive();
            recv_data = NULL;
            recv_len = 0;            
        }
    }  
	Packet::free(p);
}  // end NsProtoAgent::recv()


// Our example protocol instantiation

ProtoExample::ProtoExample(class NsProtoAgent* agent)
{
    // Init timer manager
    timer_mgr.SetInstaller(ProtoSimAgent::TimerInstaller, 
                           dynamic_cast<ProtoSimAgent*>(agent)); 
    
    // Init tx_timer for 1.0 second interval, infinite repeats
    tx_timer.Init(1.0, -1, 
                 (ProtocolTimerOwner*)this, 
                 (ProtocolTimeoutFunc)&ProtoExample::OnTxTimeout);  
    
    // Init socket, specifying async recv owner/handler, async installer
    socket.Init((UdpSocketOwner*)this, 
                (UdpSocketRecvHandler)&ProtoExample::OnSocketRecv, 
                ProtoSimAgent::SocketInstaller,
                dynamic_cast<ProtoSimAgent*>(agent)); 
}  // end ProtoExample::ProtoExample()


void ProtoExample::Start(SIMADDR dstAddr, unsigned short dstPort)
{
    dst_addr = dstAddr;
    dst_port = dstPort;
    socket.Open();  
    timer_mgr.InstallTimer(&tx_timer);   
}

void ProtoExample::Stop()
{
    if (tx_timer.IsActive()) tx_timer.Deactivate();  
}

bool ProtoExample::OnTxTimeout()
{
    NetworkAddress addr;   
    addr.SimSetAddress(dst_addr);
    addr.SetPort(dst_port);
    char* buffer = "Hello, Proteus";
    unsigned int len = strlen(buffer) + 1;
    socket.SendTo(&addr, buffer, len);
    return true;
}  // end ProtoApp::OnTxTimeout()

bool ProtoExample::OnSocketRecv(UdpSocket* /*theSocket*/)
{
    char buffer[512];
    unsigned int len = 512;
    NetworkAddress addr;
    socket.RecvFrom(buffer, &len, &addr);
    fprintf(stderr, "protoApp:: Received \"%s\" from \"%s\"\n",
            buffer, addr.HostAddressString());
    return true;
}  // end ProtoApp::OnSocketRecv()


