/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
/*************************************************************
Multicast Dissemination Protocol version 2 (MDPv2)

This source code is part of the prototype NRL MDPv2 release
and was written and developed by Brian Adamson and Joe Macker.
This is presently experimental code and therefore use it at your 
own risk.  Please include this notice and provide credit to the 
authors and NRL if this program or parts of it are used for any 
purpose.

We would appreciate receiving any contributed enhancements 
or modifications to this code release. Please contact the developers 
with comments/questions regarding this code release.  Feedback on
use of this code can help continue support for further development.

Joe Macker  				Brian Adamson
Email: <macker@itd.nrl.navy.mil>	<adamson@newlink.net>
Telephone: +1-202-767-2001		+1-202-404-1194
Naval Research Laboratory		Newlink Global Engineering Corp.
Information Technology Division 	6506 Loisdale Road Suite 209
4555 Overlook Avenue			Springfield VA 22150
Washington DC 20375                     <http://www.ngec.com>
**************************************************************/

// This is the WIN32 implementation of the MdpFile class 
#include <string.h>  // for strerror()
#include <errno.h>
#include <direct.h>

#include <share.h>
#include "mdpFile.h"
        
MdpFile::MdpFile()
    : handle(-1)
{    
}

MdpFile::~MdpFile()
{
    if (IsOpen()) Close();
}  // end MdpFile::~MdpFile()

// This should be called with a full path only!
bool MdpFile::Open(char *path, int theFlags)
{
    ASSERT(!IsOpen());    
	// See if we need to build a directory
	if (theFlags & O_CREAT)
	{
		char *ptr = path;
		if (DIR_DELIMITER == *ptr) ptr += 1;
		// Descend path, making and cd to dirs as needed 
		// Save current directory
		char dirBuf[PATH_MAX];
		char* cwd = _getcwd(dirBuf, PATH_MAX);
		while((ptr = strchr(ptr, DIR_DELIMITER)))
		{
			*ptr = '\0';
			if (_chdir(path))
			{
				DMSG(0, "Making directory: %s\n", path);
				if (_mkdir(path))
				{
					DMSG(0, "Error opening file \"%s\": %s\n", path,
						  strerror(errno));
					return false;     
				}
			}
			*ptr++ = DIR_DELIMITER;
		}
		if (cwd) _chdir(cwd);
	}

	// Make sure we're in binary mode
	theFlags |= _O_BINARY;	

	// Allow sharing of tx files but not of receive files
	if (theFlags & _O_RDONLY)
		handle = _sopen(path, theFlags, _SH_DENYNO);
    else
		handle = _open(path, theFlags, 0640);

    if(handle >= 0)
    {
        offset = 0;
		flags = theFlags;
        return true;  // no error
    }
    else
    {       
        DMSG(0, "Error opening file \"%s\": %s\n", path,
                strerror(errno));
		flags = 0;
        return false;
    }
}  // end MdpFile::Open()

bool MdpFileIsLocked(char *path)
{
	// If file doesn't exist, it's not locked
	if (!MdpFileExists(path)) return false;  

	MdpFile testFile;
    if(testFile.Open(path, O_WRONLY))
	{
		testFile.Close();
		return false;	
	}
    else
	{
		return true;  // Couldn't open, must be in use
	} 
}  // end MdpFileIsLocked()

bool MdpFile::Rename(char *old_name, char *new_name)
{
    // Make sure the new file name isn't an existing "busy" file
    if (MdpFileIsLocked(new_name)) return false;

	// In Win32, the new file can't already exist
	if (MdpFileExists(new_name)) 
		_unlink(new_name);

	// Close the current file
	int old_flags = 0;
	if (IsOpen())
	{
		old_flags = flags;
		old_flags &= ~(O_CREAT | O_TRUNC);  // unset these
		Close();
	}
    if (rename(old_name, new_name))
	{
		DMSG(0, "File rename() error: %s\n", strerror(errno));
		if (old_flags) Open(old_name, old_flags);
		return false;
	}
    else
	{
		if (old_flags) Open(new_name, old_flags);
        return true;
	}
}  // end MdpFile::Rename()

bool MdpFile::Unlink(char *path)
{
	if (MdpFileIsLocked(path))
	{
		return false;
	}
	else if (_unlink(path))
	{
		DMSG(0, "Error unlinking file(%s): %s\n", 
			     path, strerror(errno));       
        return false;
	}
    else
	{
        return true;
	}
}  // end MdpFile::Unlink()

void MdpFile::Close()
{
    ASSERT(IsOpen());  
    _close(handle);
	flags = 0;
    handle = -1;
}  // end MdpFile::Close()



bool MdpFile::Seek(unsigned long theOffset)
{
    ASSERT(IsOpen());
    off_t result = _lseek(handle, theOffset, SEEK_SET);
    if (result < 0)
    {
        DMSG(0, "Error positioning file pointer: %s\n", 
			strerror(errno));
        return false;
    }
    else
    {
        offset = result;
        return true; // no error
    }
}  // end MdpFile::Seek()

unsigned long MdpFile::Size()
{
    ASSERT(IsOpen());
    struct _stat info;
    int result = _fstat(handle, &info);
    if (result)
    {
        DMSG(0, "Error getting file size: %s\n", strerror(errno));
        return 0;   
    }
    else
    {
        return info.st_size;
    }
}  // end MdpFile::Size()

/******************************************
 * The MdpDirectory and MdpDirectoryIterator classes
 * are used to walk directory trees for file transmission
 */

MdpDirectory::MdpDirectory(const char *thePath, MdpDirectory *theParent)
    : hSearch((HANDLE)-1), parent(theParent)
{
	strncpy(path, thePath, PATH_MAX);
	int len = min(PATH_MAX, (int) strlen(path));
	if ((len < PATH_MAX) && (DIR_DELIMITER != path[len-1]))
	{
		path[len++] = DIR_DELIMITER;
		if (len < PATH_MAX) path[len] = '\0';
	}
}

// Make sure it's a valid directory and set dir name
bool MdpDirectory::Open()
{
	if (hSearch != (HANDLE) -1)
	{
		FindClose(hSearch);
		hSearch = (HANDLE) -1;
	}

	char full_name[PATH_MAX];
	GetFullName(full_name);
	// Get rid of trailing DIR_DELIMITER
	int len = min(PATH_MAX, (int) strlen(full_name));
	if (DIR_DELIMITER == full_name[len-1])
		full_name[len-1] = '\0';
    DWORD attr = GetFileAttributes(full_name);
	if (0xFFFFFFFF == attr)
		return false;
	else if (attr & FILE_ATTRIBUTE_DIRECTORY)
		return true;
	else
		return false;
} // end MdpDirectory::Open()

void MdpDirectory::Close()
{
    if (hSearch != (HANDLE) -1) 
	{
		FindClose(hSearch);
		hSearch = (HANDLE) -1;
	}
}  // end MdpDirectory::Close()


void MdpDirectory::GetFullName(char *ptr)
{
    ptr[0] = '\0';
    RecursiveCatName(ptr);
}  // end GetFullName()

void MdpDirectory::RecursiveCatName(char *ptr)
{
	if (parent) parent->RecursiveCatName(ptr);
	int len = min(PATH_MAX, (int)strlen(ptr));
	strncat(ptr, path, PATH_MAX-len);
}  // end RecursiveCatName()



MdpDirectoryIterator::MdpDirectoryIterator()
    : current(NULL)
{

}

MdpDirectoryIterator::~MdpDirectoryIterator()
{
    Destroy();
}


bool MdpDirectoryIterator::Init(const char *thePath)
{
    if (current) Destroy();
	// Make sure it's a valid directory	
	current = new MdpDirectory(thePath, NULL);
	if (current && current->Open())
	{
		path_len = min(PATH_MAX, (int)strlen(current->Path()));
		return true;
	}
	else
	{
		if (current) delete current;
		current = NULL;
		return false;
	}
}  // end MdpDirectoryIterator::Init()

void MdpDirectoryIterator::Destroy()
{
    MdpDirectory *ptr;
    while ((ptr = current))
    {
        current = ptr->parent;
        ptr->Close();
        delete ptr;
    }
}  // end MdpDirectoryIterator::Destroy()

bool MdpDirectoryIterator::GetNextFile(char *file_name)
{
	if (!current) return false;

	bool success = true;

	while(success)
	{
		WIN32_FIND_DATA find_data;
		if (current->hSearch == (HANDLE) -1)
		{
			// Construct search string
			current->GetFullName(file_name);
			strcat(file_name, "\\*");
			if ((HANDLE) -1 == 
				  (current->hSearch = FindFirstFile(file_name, &find_data)) )
				success = false;
			else
				success = true;
		}
		else
		{
			success = (0 != FindNextFile(current->hSearch, &find_data));
		}

		// Do we have a candidate file?
		if (success)
		{
			char *ptr = strrchr(find_data.cFileName, DIR_DELIMITER);
			if (ptr)
				ptr += 1;
			else
				ptr = find_data.cFileName;

			// Skip "." and ".." directories
			if (ptr[0] == '.')
			{
				if ((1 == strlen(ptr)) ||
					((ptr[1] == '.') && (2 == strlen(ptr))))
				{
					continue;
				}
			}

			current->GetFullName(file_name);
			strcat(file_name, ptr);
			int type = MdpFileGetType(file_name);
			if (MDP_FILE_NORMAL == type)
			{
				int name_len = min(PATH_MAX, (int) strlen(file_name)) -
					path_len;
				memmove(file_name, file_name+path_len, name_len);
				if (name_len < PATH_MAX) file_name[name_len] = '\0';
				return true;
			}
			else if (MDP_FILE_DIRECTORY == type)
			{

				MdpDirectory *dir = new MdpDirectory(ptr, current);
				if (dir && dir->Open())
				{
					// Push sub-directory onto stack and search it
					current = dir;
					return GetNextFile(file_name);
				}
				else
				{
					// Couldn't open try next one
					if (dir) delete dir;
				}
			}
			else
			{
				// Invalid file, try next one
			}
		}  // end if(success)
	}  // end while(success)

	// if parent, popup a level and continue search
	if (current->parent)
	{
		current->Close();
		MdpDirectory *dir = current;
		current = current->parent;
		delete dir;
		return GetNextFile(file_name);
	}
	else
	{
		current->Close();
		delete current;
		current = NULL;
		return false;
	}	
}  // end MdpDirectoryIterator::GetNextFile()

unsigned long MdpFileGetSize(const char* path)
{
	struct _stat info;
    int result = _stat(path, &info);
    if (result)
    {
        DMSG(0, "Error getting file size: %s\n", strerror(errno));
        return 0;   
    }
    else
    {
        return info.st_size;
    }
}  // end MdpFileGetSize

// Is the named item a valid directory or file (or neither)??
MdpFileType MdpFileGetType(const char *path)
{
	DWORD attr = GetFileAttributes(path);
	if (0xFFFFFFFF == attr)
		return MDP_FILE_INVALID;  // error
	else if (attr & FILE_ATTRIBUTE_DIRECTORY)
		return MDP_FILE_DIRECTORY;
	else
		return MDP_FILE_NORMAL;
}  // end GetMdpFileType


time_t MdpFileGetUpdateTime(const char *path)
{
	struct _stat info;
	if (_stat(path, &info))
	{
		return (time_t) 0;
	}
	else
	{
		// Hack because Win2K and Win98 seem to work differently
		time_t updateTime = MAX(info.st_ctime, info.st_atime);
		updateTime = MAX(updateTime, info.st_mtime);
		return updateTime;
	}
}  // end GetMdpFileUpdateTime()

