/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 
#include "eventDispatcher.h"
#include "mdpApi.h"
#include "debug.h"
#include "sysdefs.h"

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
    
#ifdef UNIX
#include <sys/time.h>
#include <unistd.h>
#include <ctype.h>
#endif  // UNIX
		
void Usage()
{
	fprintf(stdout, "Usage: mdpChat -A <addr/port> [-t <ttl>][-N <localUserName>] "
					"[-d <debugLevel>]\n");
}

const int MDP_CLIENT = 0x01;
const int MDP_SERVER = 0x02;

class MdpChatApp
#ifdef USE_INHERITANCE
    : public ProtocolTimerOwner
#endif // USE_INHERITANCE
{
	public:
		MdpChatApp();
		~MdpChatApp();
		bool Init(int argc, char *argv[]);
		void Run() {dispatcher.Run();}
		
		
			
	private:
        MdpInstanceHandle   instance;
		char* 			    current_input_buffer;
		int 			    input_count;
		EventDispatcher	    dispatcher;
        ProtocolTimer       mdp_timer;
        MdpTimerHandle      mdp_timer_handle;
		MdpSessionHandle 	session_handle;
		char*			    session_name;
		char*			    session_address;
		unsigned short	    session_port;
		unsigned char		session_ttl;
		int 				session_rate;
		int 				session_flags;
		char*			    user_name;
		
		void OnUserInput();
		bool OnNotify(MdpNotifyCode     notifyCode,
                      MdpSessionHandle  sessionHandle,
                      MdpNodeHandle     nodeHandle,
                      MdpObjectHandle   objectHandle,
                      MdpError          errorCode);
        
        static bool MdpTimerInstaller(MdpTimerInstallCmd    cmd, 
                                      double                delay,
                                      MdpTimerHandle        timerHandle, 
                                      MdpInstanceHandle     instanceHandle);
        
        static bool MdpSocketInstaller(MdpSocketInstallCmd  cmd,
                                       MdpSocketHandle      socketHandle,
                                       MdpInstanceHandle    instanceHandle);
        
        static void MdpSocketHandler(EventDispatcher::Descriptor descriptor, 
                                     EventDispatcher::EventType  eventType, 
                                     const void*                 userData);
        
        bool OnMdpTimeout();
        
        static void DoUserInput(EventDispatcher::Descriptor descriptor, 
                                EventDispatcher::EventType  eventType, 
                                const void*                 userData);
		static bool NotifyFunc(MdpNotifyCode        notifyCode,
                               MdpInstanceHandle    instanceHandle,
                        	   MdpSessionHandle     sessionHandle,
                               MdpNodeHandle        nodeId,
                        	   MdpObjectHandle      objectHandle,
                        	   MdpError             errorCode);
};

MdpChatApp::MdpChatApp()
	: current_input_buffer(NULL), input_count(0), 
      mdp_timer_handle(NULL), session_handle(MDP_NULL_SESSION),
	  session_name(NULL), session_address(NULL), session_port(0),
	  session_ttl(32), session_rate(9600), session_flags(MDP_CLIENT | MDP_SERVER),
	  user_name(NULL)
	  
{
    mdp_timer.Init(0.0, 0, (ProtocolTimerOwner *)this, 
                    (ProtocolTimeoutFunc)&MdpChatApp::OnMdpTimeout);
}


MdpChatApp::~MdpChatApp()
{
	MdpSessionCloseClient(session_handle);
	MdpSessionCloseServer(session_handle);
	if (current_input_buffer) delete []current_input_buffer;
}


bool MdpChatApp::MdpSocketInstaller(MdpSocketInstallCmd  cmd,
                                  MdpSocketHandle       socketHandle,
                                  MdpInstanceHandle     instanceHandle)
{
    MdpChatApp* app = (MdpChatApp*)MdpInstanceGetUserData(instanceHandle);
    if (MDP_SOCKET_INSTALL == cmd)
    {     
		if (!app->dispatcher.AddSocketInput((UdpSocket*)socketHandle)) 
			return false;
    }
    else
    {
		app->dispatcher.RemoveSocketInput((UdpSocket*)socketHandle);
    }
    return true;
}  // end MdpChatApp::MdpSocketInstaller()

bool MdpChatApp::MdpTimerInstaller(MdpTimerInstallCmd    cmd, 
                                   double                delay,
                                   MdpTimerHandle        timerHandle, 
                                   MdpInstanceHandle     instanceHandle)
{
	MdpChatApp* app = (MdpChatApp*)MdpInstanceGetUserData(instanceHandle);    
    if (MDP_TIMER_MODIFY == cmd)
    {
        ASSERT(app->mdp_timer.IsActive());        
        app->mdp_timer.SetInterval(delay);
        app->mdp_timer.Reset();
        app->mdp_timer_handle = timerHandle;
        return true;
    }
    else if (MDP_TIMER_INSTALL == cmd)
    {
        ASSERT(!app->mdp_timer.IsActive());
        app->mdp_timer_handle = timerHandle;
        app->mdp_timer.SetInterval(delay);
        app->dispatcher.InstallTimer(&app->mdp_timer);
        return true;
    }
    else  // PROTOCOL_TIMER_REMOVE
    {
        app->mdp_timer.Deactivate();
        return true;
    }
}  // end MdpChatApp::MdpTimerInstaller()


void MdpChatApp::MdpSocketHandler(EventDispatcher::Descriptor /*descriptor*/, 
                                  EventDispatcher::EventType  /*eventType*/, 
                                  const void*                 userData)
{
    MdpSocketOnDataReady((MdpSocketHandle)userData);
}  // end MdpChatApp::MdpSocketHandler()

bool MdpChatApp::OnMdpTimeout()
{
	mdp_timer.Deactivate();
    MdpTimerOnTimeout(mdp_timer_handle);
    return false;
} // end MdpChatApp::OnMdpTimeout()

void MdpChatApp::DoUserInput(EventDispatcher::Descriptor /*descriptor*/, 
                             EventDispatcher::EventType  /*eventType*/, 
                             const void*                 userData)
{
	((MdpChatApp *)userData)->OnUserInput();
 }  // end MdpChatApp::DoUserInput()


void MdpChatApp::OnUserInput()
{
	if (!current_input_buffer)
	{
		if(!(current_input_buffer = new char[256]))
		{
			fprintf(stderr, "mdpChat: Error allocating memory for new text!\n");
			return;
		}
		input_count = 0;
	}
    char c;
#ifndef WIN32
    c = getchar();
#else
	HANDLE hInput = GetStdHandle(STD_INPUT_HANDLE);
	INPUT_RECORD inputRecord;
	DWORD count;
	if (ReadConsoleInput(hInput, &inputRecord, 1, &count))
	{
		if ((KEY_EVENT == inputRecord.EventType) && 
			inputRecord.Event.KeyEvent.bKeyDown && 
			inputRecord.Event.KeyEvent.uChar.AsciiChar)
		{
			c = inputRecord.Event.KeyEvent.uChar.AsciiChar;
		}
		else
		{
			//TRACE("ReadConsole() uninteresting event ...\n");
            return;
		}
	}
	else
	{
		TRACE("ReadConsole() error ...\n");
        return;
	}
#endif // if/else !WIN32

    
    if ((c == '\n') || (c == '\r') || (input_count >= 255))
	{
		fprintf(stdout, "\n");
		current_input_buffer[input_count++] = '\0';
		MdpError error;
		const char* infoPtr = (user_name ? user_name : "UNKNOWN");
		unsigned short infoSize = strlen(infoPtr);
		MdpObjectHandle tx_object = MdpSessionQueueTxData(session_handle,
													  infoPtr, infoSize,
													  current_input_buffer, 
													  input_count,
													  &error);
		if (MDP_NULL_OBJECT == tx_object)
		{
			fprintf(stderr, "mdpChat: Error queueing tx data object (err=%d)\n", error);
		}
		else
		{
			current_input_buffer = NULL;
		}
	}
	else if ('\b' == c)
	{
		if (input_count) input_count--;
#ifdef WIN32
		fprintf(stdout, "\b \b");
#endif
	}
	else if (isprint(c))
	{
#ifdef WIN32
		fprintf(stdout, "%c", c);
#endif
		current_input_buffer[input_count++] = c;
	}
	else
	{
		//TRACE("Read non-printable character ...\n");
	} 
}  // end MdpChatApp::OnUserInput()

bool MdpChatApp::NotifyFunc(MdpNotifyCode       notifyCode,
                            MdpInstanceHandle   instanceHandle,
                        	MdpSessionHandle    sessionHandle,
                            MdpNodeHandle       nodeHandle,
                        	MdpObjectHandle     objectHandle,
                        	MdpError            errorCode)
{
    MdpChatApp* app = (MdpChatApp*)MdpInstanceGetUserData(instanceHandle);
	return (app->OnNotify(notifyCode, sessionHandle, 
                        nodeHandle, objectHandle, 
                        errorCode));
}  // end MdpChatApp::NotifyFunc()
        
bool MdpChatApp::OnNotify(MdpNotifyCode     notifyCode,
                          MdpSessionHandle  /*session_handle*/,
                          MdpNodeHandle     /*nodeHandle*/,
                          MdpObjectHandle   objectHandle,
                          MdpError         /*error_code*/)
{
	MdpError error;
	switch (notifyCode)
	{
		case MDP_NOTIFY_RX_OBJECT_START:
            if (MDP_OBJECT_DATA == MdpObjectGetType(objectHandle))
			{
                // Allocate memory for new receive object
				unsigned long size = MdpObjectGetSize(objectHandle);
				char* ptr = new char[size];
				if (ptr)
				{
					MdpObjectSetData(objectHandle, ptr, size);
				}
				else
				{
					fprintf(stderr, "mdpChat: Error allocating memory for recv data!\n");
				}
			}
			else
            {
				fprintf(stderr, "mdpChat: Invalid object type receive started!\n");
			}
            break;
			
			
		case MDP_NOTIFY_OBJECT_DELETE:
			if (MDP_OBJECT_DATA == MdpObjectGetType(objectHandle))
			{
                unsigned long size;
				char* ptr = MdpObjectGetData(objectHandle, &size, &error);
				if (ptr) delete []ptr;
			}
			else
			{
				fprintf(stderr, "mdpChat: Invalid object type closed?!\n");
			}
			break;
			
		case MDP_NOTIFY_RX_OBJECT_COMPLETE:
			if (MDP_OBJECT_DATA == MdpObjectGetType(objectHandle))
			{
                unsigned short infoSize = 511;
                char text[512];
				MdpObjectGetInfo(objectHandle, text, &infoSize);
				text[infoSize] = '\0';
				strcat(text, ": ");
				int len = strlen(text);
                unsigned long size;
				char* ptr = MdpObjectGetData(objectHandle, &size, &error);
				strncat(text, (ptr ? ptr:"\n"), MIN(512-len, (int)size));
				text[MIN(511,(len+size))] = '\0';
				printf("%s\n", text);
                fflush(stdout);
			}
			else
            {
				fprintf(stderr, "mdpChat: Invalid object type received?!\n");
			}
			break;
			
		default:
			break;
	}
	return true;
}


int main(int argc, char *argv[])
{
	MdpChatApp theApp;
	if (theApp.Init(argc, argv))
    {
		theApp.Run();
    }
	else
	{
		fprintf(stderr, "mdpChat: Error initing application!\n");
        Usage();
		exit(-1);
	}
	return 0;
}

bool MdpChatApp::Init(int argc, char *argv[])
{
	struct timeval currentTime;
	GetSystemTime(&currentTime);
	srand(currentTime.tv_usec);
	unsigned long  session_base_object_handle  =  rand() * (0xffffffff/RAND_MAX);
	session_name = "MDP Chat";
		
    // Parse command line arguments
    
    // Parse command line
    int i = 1;
    while(i < argc)
    {
        if (!strcmp("-A", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "mdpChat: Insufficient \"-A\" arguments!\n");
                Usage();
                return false;   
            }
            session_address = argv[i];
            char* ptr = strchr(session_address, '/');
            if (ptr)
            {
               *ptr++ = '\0';
               session_port = atoi(ptr); 
            }
            else
            {
                fprintf(stderr, "mdpChat: No port number given!\n");
                Usage();
                return false;
            }
            i++;
        }
        else if (!strcmp("-t", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "mdpChat: Insufficient \"-t\" arguments!\n");
                Usage();
                return false;   
            }
            session_ttl = atoi(argv[i]);
            i++;
        }
        else if (!strcmp("-d", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "mdpChat: Insufficient \"-d\" arguments!\n");
                Usage();
                return false;   
            }
            SetDebugLevel(atoi(argv[i]));
            i++;
        }
        else if (!strcmp("-N", argv[i]))
        {
            i++;
            if (i >= argc)
            {
                fprintf(stderr, "mdpChat: Insufficient \"-N\" arguments!\n");
                Usage();
                return false;   
            }
            user_name = argv[i];
            i++;
        }                        
        else
        {
            fprintf(stderr, "mdpChat: Unknown command!\n");
            Usage();
            return false; 
        }
    }  // end while (i < argc);
    
    
	if (!session_address || !session_port)
	{
		fprintf(stderr, "mdpChat: Session address/port is required!\n");
		return false;
	}

    EventDispatcher::Descriptor inputDescriptor;
#ifdef WIN32
	if (!dispatcher.Win32Init())
	{
		DMSG(0, "mdpChat: Error initializing Win32 messaging!\n");
		return false;
	}
	inputDescriptor = (EventDispatcher::Descriptor)GetStdHandle(STD_INPUT_HANDLE);
	SetConsoleMode((HANDLE)inputDescriptor, ENABLE_PROCESSED_INPUT);
#else
	inputDescriptor = (EventDispatcher::Descriptor)STDIN_FILENO;
    setvbuf(stdin, NULL, _IONBF, 0);
#endif  // if/else WIN32

	// Set up handler for console input
    if(!dispatcher.AddGenericInput(inputDescriptor, MdpChatApp::DoUserInput, (const void*)this)) 
    {
        fprintf(stderr, "mdpChat: Error installing user input handler!\n");
        return false;
    }
    
	// Init the MDPv2 protocol engine and install necessary callbacks
    instance = MdpInit();
	if (MDP_NULL_INSTANCE == instance)
    {
        fprintf(stderr, "mdpChat: Error initializing MDPv2 library!\n");
        return false;
    }	
    MdpInstanceSetUserData(instance, (const void*)this);
	
	MdpSetTimerInstallCallback(instance, MdpChatApp::MdpTimerInstaller);
    MdpSetSocketInstallCallback(instance, MdpChatApp::MdpSocketInstaller);
    MdpSetNotifyCallback(instance, MdpChatApp::NotifyFunc);
	
	// Create a reliable multicast session and init it
	MdpError err;
    session_handle = MdpNewSession(instance, session_address, session_port, 
                                   0, session_ttl, &err);	
	MdpSessionSetBaseObjectTransportId(session_handle, session_base_object_handle);    
    MdpSessionSetTxRate(session_handle, session_rate);
	
	// Even though this app currently doesn't deal with files
	MdpSessionSetArchiveMode(session_handle, false);  
    MdpSessionSetArchiveDirectory(session_handle, "/tmp");
	
	err = MdpSessionOpenClient(session_handle);
    if (MDP_ERROR_NONE != err)
    {
        fprintf(stderr, "mdpChat: Error opening client! (error = %d)\n",
                         err);
        return false;
    }    
	// Also open as server
	// Using small segment size and code rate since messages are small 
	// (FEC code rate si irrelevant for this application)
	int tx_segment_size = 256;
	int session_ndata = 2;    
	int session_nparity = 2;
    err = MdpSessionOpenServer(session_handle, tx_segment_size, 
                               session_ndata, session_nparity);
    if (MDP_ERROR_NONE != err)
    {
        fprintf(stderr, "mdpChat: Error opening server! (error = %d\n",
                         err);
        if (session_flags & MDP_SERVER) MdpSessionCloseClient(session_handle);
        return false;
    }     
	return true;
}  // end MdpChatApp::Init()
