/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include "mdpBitMask.h"
        
// Hamming weights for given one-byte bit masks
unsigned char WEIGHT[256] = 
{
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4, 
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 
    1, 2, 2, 3, 2, 3, 3, 4, 2, 3, 3, 4, 3, 4, 4, 5, 
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 
    2, 3, 3, 4, 3, 4, 4, 5, 3, 4, 4, 5, 4, 5, 5, 6, 
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 
    3, 4, 4, 5, 4, 5, 5, 6, 4, 5, 5, 6, 5, 6, 6, 7, 
    4, 5, 5, 6, 5, 6, 6, 7, 5, 6, 6, 7, 6, 7, 7, 8
};

// Erasure location vectors for given one-byte bit masks
unsigned char BITLOCS[256][8] = 
{
    {0, 0, 0, 0, 0, 0, 0, 0}, {7, 0, 0, 0, 0, 0, 0, 0}, 
    {6, 0, 0, 0, 0, 0, 0, 0}, {6, 7, 0, 0, 0, 0, 0, 0}, 
    {5, 0, 0, 0, 0, 0, 0, 0}, {5, 7, 0, 0, 0, 0, 0, 0}, 
    {5, 6, 0, 0, 0, 0, 0, 0}, {5, 6, 7, 0, 0, 0, 0, 0}, 
    {4, 0, 0, 0, 0, 0, 0, 0}, {4, 7, 0, 0, 0, 0, 0, 0}, 
    {4, 6, 0, 0, 0, 0, 0, 0}, {4, 6, 7, 0, 0, 0, 0, 0}, 
    {4, 5, 0, 0, 0, 0, 0, 0}, {4, 5, 7, 0, 0, 0, 0, 0}, 
    {4, 5, 6, 0, 0, 0, 0, 0}, {4, 5, 6, 7, 0, 0, 0, 0}, 
    {3, 0, 0, 0, 0, 0, 0, 0}, {3, 7, 0, 0, 0, 0, 0, 0}, 
    {3, 6, 0, 0, 0, 0, 0, 0}, {3, 6, 7, 0, 0, 0, 0, 0}, 
    {3, 5, 0, 0, 0, 0, 0, 0}, {3, 5, 7, 0, 0, 0, 0, 0}, 
    {3, 5, 6, 0, 0, 0, 0, 0}, {3, 5, 6, 7, 0, 0, 0, 0}, 
    {3, 4, 0, 0, 0, 0, 0, 0}, {3, 4, 7, 0, 0, 0, 0, 0}, 
    {3, 4, 6, 0, 0, 0, 0, 0}, {3, 4, 6, 7, 0, 0, 0, 0}, 
    {3, 4, 5, 0, 0, 0, 0, 0}, {3, 4, 5, 7, 0, 0, 0, 0}, 
    {3, 4, 5, 6, 0, 0, 0, 0}, {3, 4, 5, 6, 7, 0, 0, 0}, 
    {2, 0, 0, 0, 0, 0, 0, 0}, {2, 7, 0, 0, 0, 0, 0, 0}, 
    {2, 6, 0, 0, 0, 0, 0, 0}, {2, 6, 7, 0, 0, 0, 0, 0}, 
    {2, 5, 0, 0, 0, 0, 0, 0}, {2, 5, 7, 0, 0, 0, 0, 0}, 
    {2, 5, 6, 0, 0, 0, 0, 0}, {2, 5, 6, 7, 0, 0, 0, 0}, 
    {2, 4, 0, 0, 0, 0, 0, 0}, {2, 4, 7, 0, 0, 0, 0, 0}, 
    {2, 4, 6, 0, 0, 0, 0, 0}, {2, 4, 6, 7, 0, 0, 0, 0}, 
    {2, 4, 5, 0, 0, 0, 0, 0}, {2, 4, 5, 7, 0, 0, 0, 0}, 
    {2, 4, 5, 6, 0, 0, 0, 0}, {2, 4, 5, 6, 7, 0, 0, 0}, 
    {2, 3, 0, 0, 0, 0, 0, 0}, {2, 3, 7, 0, 0, 0, 0, 0}, 
    {2, 3, 6, 0, 0, 0, 0, 0}, {2, 3, 6, 7, 0, 0, 0, 0}, 
    {2, 3, 5, 0, 0, 0, 0, 0}, {2, 3, 5, 7, 0, 0, 0, 0}, 
    {2, 3, 5, 6, 0, 0, 0, 0}, {2, 3, 5, 6, 7, 0, 0, 0}, 
    {2, 3, 4, 0, 0, 0, 0, 0}, {2, 3, 4, 7, 0, 0, 0, 0}, 
    {2, 3, 4, 6, 0, 0, 0, 0}, {2, 3, 4, 6, 7, 0, 0, 0}, 
    {2, 3, 4, 5, 0, 0, 0, 0}, {2, 3, 4, 5, 7, 0, 0, 0}, 
    {2, 3, 4, 5, 6, 0, 0, 0}, {2, 3, 4, 5, 6, 7, 0, 0}, 
    {1, 0, 0, 0, 0, 0, 0, 0}, {1, 7, 0, 0, 0, 0, 0, 0}, 
    {1, 6, 0, 0, 0, 0, 0, 0}, {1, 6, 7, 0, 0, 0, 0, 0}, 
    {1, 5, 0, 0, 0, 0, 0, 0}, {1, 5, 7, 0, 0, 0, 0, 0}, 
    {1, 5, 6, 0, 0, 0, 0, 0}, {1, 5, 6, 7, 0, 0, 0, 0}, 
    {1, 4, 0, 0, 0, 0, 0, 0}, {1, 4, 7, 0, 0, 0, 0, 0}, 
    {1, 4, 6, 0, 0, 0, 0, 0}, {1, 4, 6, 7, 0, 0, 0, 0}, 
    {1, 4, 5, 0, 0, 0, 0, 0}, {1, 4, 5, 7, 0, 0, 0, 0}, 
    {1, 4, 5, 6, 0, 0, 0, 0}, {1, 4, 5, 6, 7, 0, 0, 0}, 
    {1, 3, 0, 0, 0, 0, 0, 0}, {1, 3, 7, 0, 0, 0, 0, 0}, 
    {1, 3, 6, 0, 0, 0, 0, 0}, {1, 3, 6, 7, 0, 0, 0, 0}, 
    {1, 3, 5, 0, 0, 0, 0, 0}, {1, 3, 5, 7, 0, 0, 0, 0}, 
    {1, 3, 5, 6, 0, 0, 0, 0}, {1, 3, 5, 6, 7, 0, 0, 0}, 
    {1, 3, 4, 0, 0, 0, 0, 0}, {1, 3, 4, 7, 0, 0, 0, 0}, 
    {1, 3, 4, 6, 0, 0, 0, 0}, {1, 3, 4, 6, 7, 0, 0, 0}, 
    {1, 3, 4, 5, 0, 0, 0, 0}, {1, 3, 4, 5, 7, 0, 0, 0}, 
    {1, 3, 4, 5, 6, 0, 0, 0}, {1, 3, 4, 5, 6, 7, 0, 0}, 
    {1, 2, 0, 0, 0, 0, 0, 0}, {1, 2, 7, 0, 0, 0, 0, 0}, 
    {1, 2, 6, 0, 0, 0, 0, 0}, {1, 2, 6, 7, 0, 0, 0, 0}, 
    {1, 2, 5, 0, 0, 0, 0, 0}, {1, 2, 5, 7, 0, 0, 0, 0}, 
    {1, 2, 5, 6, 0, 0, 0, 0}, {1, 2, 5, 6, 7, 0, 0, 0}, 
    {1, 2, 4, 0, 0, 0, 0, 0}, {1, 2, 4, 7, 0, 0, 0, 0}, 
    {1, 2, 4, 6, 0, 0, 0, 0}, {1, 2, 4, 6, 7, 0, 0, 0}, 
    {1, 2, 4, 5, 0, 0, 0, 0}, {1, 2, 4, 5, 7, 0, 0, 0}, 
    {1, 2, 4, 5, 6, 0, 0, 0}, {1, 2, 4, 5, 6, 7, 0, 0}, 
    {1, 2, 3, 0, 0, 0, 0, 0}, {1, 2, 3, 7, 0, 0, 0, 0}, 
    {1, 2, 3, 6, 0, 0, 0, 0}, {1, 2, 3, 6, 7, 0, 0, 0}, 
    {1, 2, 3, 5, 0, 0, 0, 0}, {1, 2, 3, 5, 7, 0, 0, 0}, 
    {1, 2, 3, 5, 6, 0, 0, 0}, {1, 2, 3, 5, 6, 7, 0, 0}, 
    {1, 2, 3, 4, 0, 0, 0, 0}, {1, 2, 3, 4, 7, 0, 0, 0}, 
    {1, 2, 3, 4, 6, 0, 0, 0}, {1, 2, 3, 4, 6, 7, 0, 0}, 
    {1, 2, 3, 4, 5, 0, 0, 0}, {1, 2, 3, 4, 5, 7, 0, 0}, 
    {1, 2, 3, 4, 5, 6, 0, 0}, {1, 2, 3, 4, 5, 6, 7, 0}, 
    {0, 0, 0, 0, 0, 0, 0, 0}, {0, 7, 0, 0, 0, 0, 0, 0}, 
    {0, 6, 0, 0, 0, 0, 0, 0}, {0, 6, 7, 0, 0, 0, 0, 0}, 
    {0, 5, 0, 0, 0, 0, 0, 0}, {0, 5, 7, 0, 0, 0, 0, 0}, 
    {0, 5, 6, 0, 0, 0, 0, 0}, {0, 5, 6, 7, 0, 0, 0, 0}, 
    {0, 4, 0, 0, 0, 0, 0, 0}, {0, 4, 7, 0, 0, 0, 0, 0}, 
    {0, 4, 6, 0, 0, 0, 0, 0}, {0, 4, 6, 7, 0, 0, 0, 0}, 
    {0, 4, 5, 0, 0, 0, 0, 0}, {0, 4, 5, 7, 0, 0, 0, 0}, 
    {0, 4, 5, 6, 0, 0, 0, 0}, {0, 4, 5, 6, 7, 0, 0, 0}, 
    {0, 3, 0, 0, 0, 0, 0, 0}, {0, 3, 7, 0, 0, 0, 0, 0}, 
    {0, 3, 6, 0, 0, 0, 0, 0}, {0, 3, 6, 7, 0, 0, 0, 0}, 
    {0, 3, 5, 0, 0, 0, 0, 0}, {0, 3, 5, 7, 0, 0, 0, 0}, 
    {0, 3, 5, 6, 0, 0, 0, 0}, {0, 3, 5, 6, 7, 0, 0, 0}, 
    {0, 3, 4, 0, 0, 0, 0, 0}, {0, 3, 4, 7, 0, 0, 0, 0}, 
    {0, 3, 4, 6, 0, 0, 0, 0}, {0, 3, 4, 6, 7, 0, 0, 0}, 
    {0, 3, 4, 5, 0, 0, 0, 0}, {0, 3, 4, 5, 7, 0, 0, 0}, 
    {0, 3, 4, 5, 6, 0, 0, 0}, {0, 3, 4, 5, 6, 7, 0, 0}, 
    {0, 2, 0, 0, 0, 0, 0, 0}, {0, 2, 7, 0, 0, 0, 0, 0}, 
    {0, 2, 6, 0, 0, 0, 0, 0}, {0, 2, 6, 7, 0, 0, 0, 0}, 
    {0, 2, 5, 0, 0, 0, 0, 0}, {0, 2, 5, 7, 0, 0, 0, 0}, 
    {0, 2, 5, 6, 0, 0, 0, 0}, {0, 2, 5, 6, 7, 0, 0, 0}, 
    {0, 2, 4, 0, 0, 0, 0, 0}, {0, 2, 4, 7, 0, 0, 0, 0}, 
    {0, 2, 4, 6, 0, 0, 0, 0}, {0, 2, 4, 6, 7, 0, 0, 0}, 
    {0, 2, 4, 5, 0, 0, 0, 0}, {0, 2, 4, 5, 7, 0, 0, 0}, 
    {0, 2, 4, 5, 6, 0, 0, 0}, {0, 2, 4, 5, 6, 7, 0, 0}, 
    {0, 2, 3, 0, 0, 0, 0, 0}, {0, 2, 3, 7, 0, 0, 0, 0}, 
    {0, 2, 3, 6, 0, 0, 0, 0}, {0, 2, 3, 6, 7, 0, 0, 0}, 
    {0, 2, 3, 5, 0, 0, 0, 0}, {0, 2, 3, 5, 7, 0, 0, 0}, 
    {0, 2, 3, 5, 6, 0, 0, 0}, {0, 2, 3, 5, 6, 7, 0, 0}, 
    {0, 2, 3, 4, 0, 0, 0, 0}, {0, 2, 3, 4, 7, 0, 0, 0}, 
    {0, 2, 3, 4, 6, 0, 0, 0}, {0, 2, 3, 4, 6, 7, 0, 0}, 
    {0, 2, 3, 4, 5, 0, 0, 0}, {0, 2, 3, 4, 5, 7, 0, 0}, 
    {0, 2, 3, 4, 5, 6, 0, 0}, {0, 2, 3, 4, 5, 6, 7, 0}, 
    {0, 1, 0, 0, 0, 0, 0, 0}, {0, 1, 7, 0, 0, 0, 0, 0}, 
    {0, 1, 6, 0, 0, 0, 0, 0}, {0, 1, 6, 7, 0, 0, 0, 0}, 
    {0, 1, 5, 0, 0, 0, 0, 0}, {0, 1, 5, 7, 0, 0, 0, 0}, 
    {0, 1, 5, 6, 0, 0, 0, 0}, {0, 1, 5, 6, 7, 0, 0, 0}, 
    {0, 1, 4, 0, 0, 0, 0, 0}, {0, 1, 4, 7, 0, 0, 0, 0}, 
    {0, 1, 4, 6, 0, 0, 0, 0}, {0, 1, 4, 6, 7, 0, 0, 0}, 
    {0, 1, 4, 5, 0, 0, 0, 0}, {0, 1, 4, 5, 7, 0, 0, 0}, 
    {0, 1, 4, 5, 6, 0, 0, 0}, {0, 1, 4, 5, 6, 7, 0, 0}, 
    {0, 1, 3, 0, 0, 0, 0, 0}, {0, 1, 3, 7, 0, 0, 0, 0}, 
    {0, 1, 3, 6, 0, 0, 0, 0}, {0, 1, 3, 6, 7, 0, 0, 0}, 
    {0, 1, 3, 5, 0, 0, 0, 0}, {0, 1, 3, 5, 7, 0, 0, 0}, 
    {0, 1, 3, 5, 6, 0, 0, 0}, {0, 1, 3, 5, 6, 7, 0, 0}, 
    {0, 1, 3, 4, 0, 0, 0, 0}, {0, 1, 3, 4, 7, 0, 0, 0}, 
    {0, 1, 3, 4, 6, 0, 0, 0}, {0, 1, 3, 4, 6, 7, 0, 0}, 
    {0, 1, 3, 4, 5, 0, 0, 0}, {0, 1, 3, 4, 5, 7, 0, 0}, 
    {0, 1, 3, 4, 5, 6, 0, 0}, {0, 1, 3, 4, 5, 6, 7, 0}, 
    {0, 1, 2, 0, 0, 0, 0, 0}, {0, 1, 2, 7, 0, 0, 0, 0}, 
    {0, 1, 2, 6, 0, 0, 0, 0}, {0, 1, 2, 6, 7, 0, 0, 0}, 
    {0, 1, 2, 5, 0, 0, 0, 0}, {0, 1, 2, 5, 7, 0, 0, 0}, 
    {0, 1, 2, 5, 6, 0, 0, 0}, {0, 1, 2, 5, 6, 7, 0, 0}, 
    {0, 1, 2, 4, 0, 0, 0, 0}, {0, 1, 2, 4, 7, 0, 0, 0}, 
    {0, 1, 2, 4, 6, 0, 0, 0}, {0, 1, 2, 4, 6, 7, 0, 0}, 
    {0, 1, 2, 4, 5, 0, 0, 0}, {0, 1, 2, 4, 5, 7, 0, 0}, 
    {0, 1, 2, 4, 5, 6, 0, 0}, {0, 1, 2, 4, 5, 6, 7, 0}, 
    {0, 1, 2, 3, 0, 0, 0, 0}, {0, 1, 2, 3, 7, 0, 0, 0}, 
    {0, 1, 2, 3, 6, 0, 0, 0}, {0, 1, 2, 3, 6, 7, 0, 0}, 
    {0, 1, 2, 3, 5, 0, 0, 0}, {0, 1, 2, 3, 5, 7, 0, 0}, 
    {0, 1, 2, 3, 5, 6, 0, 0}, {0, 1, 2, 3, 5, 6, 7, 0}, 
    {0, 1, 2, 3, 4, 0, 0, 0}, {0, 1, 2, 3, 4, 7, 0, 0}, 
    {0, 1, 2, 3, 4, 6, 0, 0}, {0, 1, 2, 3, 4, 6, 7, 0}, 
    {0, 1, 2, 3, 4, 5, 0, 0}, {0, 1, 2, 3, 4, 5, 7, 0}, 
    {0, 1, 2, 3, 4, 5, 6, 0}, {0, 1, 2, 3, 4, 5, 6, 7}
};
    
// This routine parses a bit mask with location bits set
// and returns the number of locations found and
// a vector ("locs") with the indices of the set locations
// The maximum location index for the regular MaskParse is 255
int MaskParse(const char *mask, int len, unsigned char *locs)
{
    int total_weight = 0;
    for (int i = 0; i < len; i++)
    {
        const unsigned char val = (unsigned char)*mask++;
        if (val)
        {
            const unsigned char *ptr = BITLOCS[val];
            int weight = WEIGHT[val];
            unsigned char base = i << 3;
            for (int j = 0; j < weight; j++)
                *locs++ = base + *ptr++;
            total_weight += weight;
        }
    }
    return total_weight;
}  // end MaskParse()

// This can parse longer masks
unsigned long MaskParseLong(const char* mask, unsigned long len,
                            unsigned long* locs, unsigned long locs_len)
{
    unsigned long total_weight = 0;
    for (unsigned long i = 0; i < len; i++)
    {
        const unsigned char val = (unsigned char)*mask++;
        if (val)
        {
            const unsigned char *ptr = BITLOCS[val];
            int weight = WEIGHT[val];
            unsigned long base = i << 3;
            if ((total_weight + weight) > locs_len)
                weight = locs_len - total_weight;
            for (int j = 0; j < weight; j++)
                *locs++ = base + *ptr++;
            total_weight += weight;
        }
    }
    return total_weight;
}  // end MaskParseLong()

unsigned long MaskFirstSet(const char* mask, unsigned long mask_len)
{
    unsigned long mask_index = 0;
    while (mask_index < mask_len)
    {
        if (*mask)
            return (( mask_index << 3) + BITLOCS[(unsigned char)*mask][0]);
        mask_index++;
        mask++;   
    }
    return mask_len << 3;
}  // end MaskFirstSet()

unsigned long MaskNextSet(unsigned long index, const char* mask, 
                          unsigned long maskLen)
{
    unsigned long mask_index = index >> 3;
    mask += mask_index;
    int w = WEIGHT[(unsigned char)*mask];
    int remainder = index & 0x07;
    for (int i = 0; i < w; i++)
    {
        int loc = BITLOCS[(unsigned char)*mask][i];
        if (loc >= remainder) return (mask_index << 3) + loc;
    }
    while(++mask_index < maskLen)
    {
        mask++;
        if ((unsigned char)*mask)
            return (mask_index << 3) + 
                   BITLOCS[(unsigned char)*mask][0];
    }
    return (maskLen << 3); 
}  // end MaskNextSet()

unsigned long MaskLastSet(const char* mask, unsigned long maskLen)
{
    unsigned long len = maskLen;
    unsigned char* ptr = (unsigned char*)mask + len - 1;
    while (len--)
    {
        if (*ptr)
        {
           return ((len << 3) + BITLOCS[*ptr][WEIGHT[*ptr]-1]);
        }
        else
        {
            ptr--;
        }
    }
    return (maskLen << 3);
}  // end MaskLastSet()

void MaskSetBits(char* mask, unsigned long index, unsigned long count)
{
    if (0 == count) return;
    unsigned long mask_index = index >> 3;
    
    // To set appropriate bits in first byte
    unsigned int bit_index = index & 0x07;
    unsigned int bit_remainder = 8 - bit_index;
    
    if (count <= bit_remainder)
    {
        mask[mask_index] |= (0x00ff >> bit_index) &
                            (0x00ff << (bit_remainder - count)); 
    }
    else
    {
        mask[mask_index] |= 0x00ff >> bit_index;
        count -= bit_remainder;
        unsigned long nbytes = count >> 3;  
        memset(&mask[++mask_index], 0xff, nbytes);
        count &= 0x07;  
        if (count) mask[mask_index+nbytes] |= 0xff << (8-count);
    }
}  // end MaskSetBits()

void MaskUnsetBits(char* mask, unsigned long index, unsigned long count)
{
    if (0 == count) return;
    unsigned long mask_index = index >> 3;
    
    // To unset appropriate bits in first byte
    unsigned int bit_index = index & 0x07;
    unsigned int bit_remainder = 8 - bit_index;
    
    if (count <= bit_remainder)
    {
        mask[mask_index] &= (0x00ff << bit_remainder) |
                            (0x00ff >> (bit_index + count)); 
    }
    else
    {
        mask[mask_index] &= 0x00ff << bit_remainder;
        count -= bit_remainder;
        unsigned long nbytes = count >> 3;  
        memset(&mask[++mask_index], 0, nbytes);
        count &= 0x07;  
        if (count) mask[mask_index+nbytes] &= 0xff >> count;
    }
}  // end MaskUnsetBits()

// Note masks are expected to be 8-bit aligned
// mask1 = mask1 + mask2 (mask1 | mask2)
void MaskAdd(char* mask1, const char* mask2, unsigned long maskLen)
{
    for(unsigned int i = 0; i < maskLen; i++)
        mask1[i] |= mask2[i];
}  // end MaskAdd()

// mask1 = mask1 - mask2 (mask1 & ~mask2)
void MaskSubtract(char* mask1, const char* mask2, unsigned long maskLen)
{
    for(unsigned int i = 0; i < maskLen; i++)
        mask1[i] &= (~mask2[i]);
}  // end MaskXCopy()

/****************************************************************
 * MdpBitMask implementation
 */

MdpBitMask::MdpBitMask()
    : mask(NULL)
{
}

MdpBitMask::~MdpBitMask()
{
    Destroy();
}

bool MdpBitMask::Init(unsigned long numBits)
{
    ASSERT(!mask);
    mask_len = (numBits + 7) >> 3; 
    first_set = numBits;
    num_bits = numBits;
    // Allocate memory for mask
    if((mask = new unsigned char[mask_len]))
    {
        Clear();
        return true;
    }
    else
    {
        return false;
    }
   
}  // end MdpBitMask::Init()

void MdpBitMask::Destroy()
{
    if (mask) delete []mask;
    mask = NULL;
}

void MdpBitMask::Set(unsigned long index)
{
    ASSERT(index < num_bits);
    mask[(index >> 3)] |= (0x80 >> (index & 0x07));
    if (index < first_set) first_set = index;
}  // end MdpBitMask::Set()


void MdpBitMask::Unset(unsigned long index)
{
    ASSERT(index < num_bits);
    mask[(index >> 3)] &= ~(0x80 >> (index & 0x07));
    if (index == first_set) first_set = NextSet(index);
}  // end MdpBitMask::Unset()


unsigned long MdpBitMask::NextSet(unsigned long index)
{
    ASSERT(index <=  num_bits);    
    if (index == num_bits) return num_bits;
    if (index < first_set) return first_set;
    unsigned long mask_index = index >> 3;
    if (mask[mask_index])
    {
        int w = WEIGHT[mask[mask_index]];
        int remainder = index & 0x07;
        for (int i = 0; i < w; i++)
        {
            int loc = BITLOCS[mask[mask_index]][i];
            if (loc >= remainder) return (mask_index << 3) + loc;
        }
    }
    while(++mask_index < mask_len)
    {
        if (mask[mask_index])
            return (mask_index << 3) + 
                   BITLOCS[mask[mask_index]][0];
    }
    return num_bits;
}  // end MdpBitMask::NextSet()

unsigned long MdpBitMask::NextUnset(unsigned long index)
{
    ASSERT(index <= num_bits);
    unsigned long maskIndex = index >> 3;
    unsigned char bit = 0x80 >> (index & 0x07);
    while (index < num_bits)
    {
        unsigned char val = mask[maskIndex];
        while (bit && (index < num_bits))
        {
            if (0 == (val & bit)) return index;
            index++;   
            bit >>= 0x01;
        }
        bit = 0x80;
        maskIndex++;
    }
    return index;
}  // end MdpBitMask::NextUnset()


void MdpBitMask::SetBits(unsigned long index, unsigned long count)
{
    if (0 == count) return;
    ASSERT((index+count) <= num_bits);
    unsigned long mask_index = index >> 3;
    // To set appropriate bits in first byte
    unsigned int bit_index = index & 0x07;
    unsigned int bit_remainder = 8 - bit_index;
    if (count <= bit_remainder)
    {
        mask[mask_index] |= (0x00ff >> bit_index) &
                            (0x00ff << (bit_remainder - count)); 
    }
    else
    {
        mask[mask_index] |= 0x00ff >> bit_index;
        count -= bit_remainder;
        unsigned long nbytes = count >> 3;  
        memset(&mask[++mask_index], 0xff, nbytes);
        count &= 0x07;  
        if (count)
            mask[mask_index+nbytes] |= 0xff << (8-count);
    }
    if (index < first_set) first_set = index;
}  // end MdpBitMask::SetBits()


void MdpBitMask::UnsetBits(unsigned long index, unsigned long count)
{
    if (0 == count) return;
    ASSERT((index+count) <= num_bits);
    unsigned long end = index + count;
    unsigned long mask_index = index >> 3;
    // To unset appropriate bits in first byte
    unsigned int bit_index = index & 0x07;
    unsigned int bit_remainder = 8 - bit_index;
    if (count <= bit_remainder)
    {
        mask[mask_index] &= (0x00ff << bit_remainder) |
                            (0x00ff >> (bit_index + count)); 
    }
    else
    {
        mask[mask_index] &= 0x00ff << bit_remainder;
        count -= bit_remainder;
        unsigned long nbytes = count >> 3;  
        memset(&mask[++mask_index], 0, nbytes);
        count &= 0x07;  
        if (count)
            mask[mask_index+nbytes] &= 0xff >> count;
    }
    if (first_set >= index) first_set = NextSet(end);
}  // end MdpBitMask::UnsetBits()

void MdpBitMask::Add(MdpBitMask *theMask)
{
    ASSERT(theMask->mask_len == mask_len);   
    for(unsigned int i = 0; i < mask_len; i++)
        mask[i] |= theMask->mask[i];
    if (theMask->first_set < first_set)
        first_set = theMask->first_set;
}  // end MdpBitMask::Add()

void MdpBitMask::XCopy(MdpBitMask& theMask)
{
    ASSERT(mask_len == theMask.mask_len);
    for (unsigned int i = 0; i < mask_len; i++)
        mask[i] = theMask.mask[i] & ~mask[i];
    if (theMask.first_set < first_set)
        first_set = theMask.first_set;
    else
        first_set = NextSet(theMask.first_set);
}  // end MdpBitMask::XCopy()

void MdpBitMask::And(MdpBitMask& theMask)
{
    ASSERT(mask_len == theMask.mask_len);
    for (unsigned int i = 0; i < mask_len; i++)
        mask[i] &= theMask.mask[i];
    first_set = NextSet(first_set);
}  // end MdpBitMask::And()

void MdpBitMask::AddRawMask(const char* theMask, unsigned long theLen)
{
    ASSERT(theLen <= mask_len);
    for(unsigned int i = 0; i < theLen; i++) mask[i] |= theMask[i];
    unsigned long low_bit = ::MaskFirstSet(theMask, theLen);
    if ((low_bit < first_set) && (low_bit < (theLen << 3)))
        first_set = low_bit;
}  // end MdpBitMask::AddRawMask()

void MdpBitMask::AddRawOffsetMask(const char* theMask, 
                                  unsigned long theLen,
                                  unsigned long theOffset)
{
    ASSERT(theOffset+theLen <= mask_len);
    char* ptr1 = (char*)mask + theOffset;
    const char* ptr2 =  theMask;
    for(unsigned int i = 0; i < theLen; i++)
       *ptr1++ |= *ptr2++;
    unsigned long low_bit = ::MaskFirstSet(theMask, theLen) + (theOffset << 3);
    if ((low_bit < first_set) && (low_bit < (theLen << 3)))
        first_set = low_bit;
}  // end MdpBitMask::AddRawOffsetMask()

#ifdef PROTO_DEBUG
void MdpBitMask::Display()
{
    for (unsigned long i = 0; i < mask_len; i++)
    {
        unsigned char bit = 0x80;
        for(int j = 0; j < 8; j++)
        {
           if ((num_bits == ((i << 3) + j))) break;
           if (mask[i] & bit) 
               DMSG(0, "1");
            else
               DMSG(0, "0");
            bit >>= 1;
        }
    } 
}  // end MdpBitMask::Display()


void MdpSlidingMask::Display()
{
    DMSG(0, "offset:0x%08x mask:", offset);
    bit_mask.Display();
}  // end MdpSlidingMask::Display()
#endif // PROTO_DEBUG


MdpSlidingMask::MdpSlidingMask()
    : offset(0)
{
    
}

// Test to see if the mask can be set for the corresponding index
// without allocating memory
bool MdpSlidingMask::CanFreeSet(unsigned long index)
{
    if (bit_mask.IsSet())
    {
        unsigned long diff = index - offset;
        // if (index < offset)
        if ((diff > 0x80000000) || ((0x80000000 == diff) && (index < offset))) 
        {
            if (((bit_mask.num_bits - MaskLastSet((char *)bit_mask.mask, bit_mask.mask_len) + 7) >> 3)
                < ((offset - index + 7) >> 3))
                return false;
            else
                return true;
        }
        else
        {
            unsigned long end = offset + bit_mask.num_bits;
            diff = index - end;
            if ((diff > 0x80000000) || ((0x80000000 == diff) && (index < end)))  // index < end
                return true;
            else
                return false;
        }   
    }
    else
    {
        return true;  // mask is free to be set for any offset
    }
}  // end MdpSlidingMask::CanFreeSet()

bool MdpSlidingMask::InCurrentRange(unsigned long index)
{
    if (bit_mask.IsSet())
    {
        // if (index < offset)
        unsigned long diff = index - offset;
        if ((diff > 0x80000000) || ((0x80000000 == diff) && (index < offset))) // index < offset
        {
            return false;
        }
        else
        {
            // if (index < end)
            unsigned long end = offset + bit_mask.num_bits;
            diff = index - end;
            if ((diff > 0x80000000) || ((0x80000000 == diff) && (index < end)))  // index < end
                return true;
            else
                return false;
        }   
    }
    else
    {
        return false;  // mask is free to be set for any offset
    }
}  // end MdpSlidingMask::InCurrentRange()

bool MdpSlidingMask::Set(unsigned long index)
{    
    if (bit_mask.IsSet())
    {
        unsigned long diff = index - offset;
        if ((diff > 0x80000000) || ((0x80000000 == diff) && (index < offset)))
        {
            unsigned long space_needed = (offset - index + 7) >> 3;
            unsigned long space_avail = 
                (bit_mask.num_bits - MaskLastSet((char *)bit_mask.mask, bit_mask.mask_len) + 7) >> 3;
            if (space_avail < space_needed)
            {
                unsigned long old_len = bit_mask.mask_len;
                unsigned long new_len = old_len * 2;
                while (new_len < (old_len + space_needed - space_avail)) new_len *= 2;
                if (new_len > 0x1fffffff) return false;
                unsigned char *new_mask = new unsigned char[new_len];
                if (!new_mask) return false;
                memset(new_mask, 0, space_needed);
                memcpy(&new_mask[space_needed], bit_mask.mask, old_len);
                memset(&new_mask[space_needed+old_len], 0, new_len - (space_needed+old_len));
                delete []bit_mask.mask;
                bit_mask.mask = new_mask;
                bit_mask.mask_len = new_len;
                bit_mask.num_bits = new_len << 3;
            }
            else
            {
                // Reverse slide
                unsigned char *mask = bit_mask.mask;
                memmove(&mask[space_needed], mask, bit_mask.mask_len - space_needed);   
                memset(mask, 0, space_needed);
            }
            bit_mask.first_set += space_needed << 3;
            offset = index & (~((unsigned long)0x07));
        }
        else 
        {
            unsigned long end = offset + bit_mask.num_bits;
            diff = index - end;
            if(!((diff > 0x80000000) || ((0x80000000 == diff) && (index < end))))
            {
                unsigned long space_needed = (index - end + 7) >> 3;
                unsigned long old_len = bit_mask.mask_len;
                unsigned long new_len = old_len * 2;
                while (new_len < (old_len + space_needed)) new_len *= 2;
                if (new_len > 0x1fffffff) return false;
                unsigned char *new_mask = new unsigned char[new_len];
                if(!new_mask) return false;
                memcpy(new_mask, bit_mask.mask, old_len);
                memset(&new_mask[old_len], 0, new_len - old_len);
                delete []bit_mask.mask;
                bit_mask.mask = new_mask;
                bit_mask.mask_len = new_len;
                bit_mask.num_bits = new_len << 3;
            }
        }
    }
    else
    {
        offset = index & (~((unsigned long)0x07)); 
    }
    bit_mask.Set(index - offset);
    return true;
}  // end MdpSlidingMask::Set()


bool MdpSlidingMask::NextSet(unsigned long *index)
{
    ASSERT(index);
    if (InCurrentRange(*index))
    {
        unsigned long next_set = bit_mask.NextSet(*index - offset);
        if (next_set < bit_mask.num_bits)
        {
            *index = next_set + offset;
            return true;
        }
        else
        {
            return false;
        }
    }
    else
    {
        return false;
    }
}  // end MdpSlidingMask::NextSet()

// Slide down unset mask prefix and update offset
void MdpSlidingMask::Slide()
{
    if (bit_mask.IsSet())
    {
        unsigned long i = 0;
        unsigned long mask_len = bit_mask.mask_len;
        unsigned char *mask = bit_mask.mask;
        while ((i < mask_len) && (0 == mask[i])) i++;     
        memmove(mask, &mask[i], mask_len - i);
        memset(&mask[mask_len-i], 0, i);
        offset += i << 3;
        if (bit_mask.first_set < bit_mask.num_bits)
            bit_mask.first_set -= i << 3;
    }
}  // end MdpSlidingMask::Compress()
