/*********************************************************************
 *
 * AUTHORIZATION TO USE AND DISTRIBUTE
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: 
 *
 * (1) source code distributions retain this paragraph in its entirety, 
 *  
 * (2) distributions including binary code include this paragraph in
 *     its entirety in the documentation or other materials provided 
 *     with the distribution, and 
 *
 * (3) all advertising materials mentioning features or use of this 
 *     software display the following acknowledgment:
 * 
 *      "This product includes software written and developed 
 *       by Brian Adamson and Joe Macker of the Naval Research 
 *       Laboratory (NRL)." 
 *         
 *  The name of NRL, the name(s) of NRL  employee(s), or any entity
 *  of the United States Government may not be used to endorse or
 *  promote  products derived from this software, nor does the 
 *  inclusion of the NRL written and developed software  directly or
 *  indirectly suggest NRL or United States  Government endorsement
 *  of this product.
 * 
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 ********************************************************************/
 

#include <stdio.h>
#include <errno.h>
#include <string.h>
               
#include "mdpApi.h"
#include "debug.h"
                
static bool ReadNextToken(FILE *inFile, char *delimiters, 
                             char *token, int max_len);
static bool DelimiterMatch(char value, char *delimiters);
    
MdpError ParseAckListFile(MdpSessionHandle sessionHandle, const char *path)
{
    FILE *inFile = fopen(path, "r");
    if (inFile)
    {
        // Scan file looking for host names/addresses delimited
        // by white space, line feeds, carriage returns, or commas
        char delimiters[] = "\n\r \t,";
        char host[256];
        int count = 0;
        while (ReadNextToken(inFile, delimiters, host, 256))
        {
            MdpError err = MdpSessionAddAckingClient(sessionHandle, host);
            if (MDP_ERROR_NONE != err) return err;
            count++;
        }
        if (count)  // Found at least one valid entry in the file
            return MDP_ERROR_NONE;
        else
            return MDP_ERROR_FILE_EMPTY;
    }
    else
    {
        DMSG(0, "mdp: Error opening access list file: %s\n", strerror(errno));
        return MDP_ERROR_FILE_OPEN;
    }
}  // end MdpSession::ParseAccessListFile()


static bool ReadNextToken(FILE *inFile, char *delimiters, char *token, int max_len)
{
    char *buf = token; 
    int count = 0;  
    while(fread(buf, sizeof(char), 1, inFile))
    {
        // Skip comment lines
        if ((0 == count) && ('#' == *buf))
        {
            while(fread(buf, sizeof(char), 1, inFile))
                if (('\n' == *buf) || ('\r' == *buf))
                    break;
            continue;  // comment skipped, continue reading the file
        }
        if (++count >= max_len) return 0;
        if (DelimiterMatch(*buf, delimiters))
        {
            *buf = '\0';
            if (strlen(token)) 
            {
                return true;
            }
            else
            {
                count = 0;
                buf = token;
            }
        }
        else
        {
            buf++;
        }
    }
    if (count) 
    {
        *buf = '\0';
        return true;
    }
    else
    {
        return false;
    }
}  // end ReadNextToken()    

static bool DelimiterMatch(char value, char *delimiters)
{
    while(*delimiters)
        if (value == *delimiters++) 
            return true;
    return false;
}  // end DelimiterMatch()


