/***********************************************************/
/* Pour cree une image a partir d'un fichier PBM (B&W)     */
/* image_from_pbm.c                                        */
/*                                                         */
/* Ecrit par : Daniel Lacroix (all rights reserved)        */
/*                                                         */
/***********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <string.h>
#include <image_from_pbm.h>

typedef enum {
  ASCII,
  BINARY
} ImgFormat;

/* Renvoi : 0 = Ok, != 0 = Erreur                                */
/* offset et positionne sur le premier caractere apres le nombre */
/* res contient la valeur lue.                                   */
static int give_uint32(char *buf, int *offset, int buf_size, uint32 *res)
{ uint32  last_res;
  boolean comment = FALSE;

  do {
    /* si on est a la fin c'est que le fichier est mal forme */
    if(*offset >= buf_size)
    { return(1); }
    
    if((buf[*offset] == '\n') && (comment))
    { /* fin d'un commentaire */
      comment = FALSE;
    } else if((buf[*offset] == '#') && (!comment))
    { /* debut d'un commentaire qui va jusqu'a la fin de la ligne */
      comment = TRUE;
    }
    /* si on est dans un commentaire ou qu'il ne s'agit pas */
    /* d'un nombre, alors on passe a caractere suivant.     */
    if(comment || (buf[*offset] < '0') || (buf[*offset] > '9'))
    {
      (*offset)++;
    } else {
      /* si on a trouve le debut du nombre, on sort de la boucle */
      break;
    }
  } while(1);
  last_res = *res = 0;
  do {
    *res = (*res)*10 + (buf[*offset]-'0');
    if(*res < last_res)
    { /* il y a debordement, le nombre est trop grand */
      return(1);
    }
    (*offset)++;
  } while((*offset < buf_size) && (buf[*offset] >= '0') && (buf[*offset] <= '9'));
  
  return(0);
}

/* Renvoi l'image ou NULL si impossible */
image *image_new_from_pbm(char *file_name)
{ int handle;
  int file_size;
  char *buf;
  image *vimage;
  int i,offset,x,y;
  int32 width,height;
  uint32 vuint32;
  ImgFormat mode;

  if((handle = open(file_name,O_RDONLY)) == -1) return(NULL);
  if((file_size = lseek(handle, 0, SEEK_END)) == -1)
  { close(handle); return(NULL); }
  if(lseek(handle, 0, SEEK_SET) == -1)
  { close(handle); return(NULL); }
  if((file_size > MAX_PBM_FILE_SIZE) ||
     (file_size < 2))
  { close(handle); return(NULL); }
  if((buf = (char *)malloc(file_size)) == NULL)
  { perror("malloc failed."); exit(1); }
  if(read(handle, buf, file_size) == -1)
  { free(buf); close(handle); return(NULL); }
  close(handle);

  if((buf[0] == 'P') && (buf[1] == '1'))
  { /* il s'agit du format pgm ASCI */
    mode = ASCII;
  } else if((buf[0] == 'P') && (buf[1] == '4'))
  { /* il s'agit du format pgm RAW */
    mode = BINARY;  
  } else {
    free(buf); return(NULL);
  }

  offset = 2;
  /* on lit la largeur */
  if(give_uint32(buf, &offset, file_size, &vuint32))
  { free(buf); return(NULL); }
  width = (int32)vuint32;
  if(width < vuint32)
  { free(buf); return(NULL); }
  /* on lit la hauteur */
  if(give_uint32(buf, &offset, file_size, &vuint32))
  { free(buf); return(NULL); }
  height = (int32)vuint32;
  if(height < vuint32)
  { free(buf); return(NULL); }

  /* on cree l'image qui contiendra le resultat */
  vimage = image_new(width,height);

  /* on lit le contenu de l'image */
  switch(mode)
  {
    case ASCII  :
      for(y=0;y<height;y++)
      {
        for(x=0;x<width;x++)
        {
          if(give_uint32(buf, &offset, file_size, &vuint32))
          { image_free(vimage); free(buf); return(NULL); }

          if(vuint32 == 1)
            vimage->buf[x+(y*width)] = BLACK;
          else if(vuint32 == 0)
            vimage->buf[x+(y*width)] = WHITE;
          else
          { image_free(vimage); free(buf); return(NULL); }
        }
      }
      break;
    case BINARY :
      offset++;
      /* il y a 8 pixels par octet. Chaque ligne est aligne sur 1 octet */
      if(((width+7)>>3)*height != (file_size-offset))
      { image_free(vimage); free(buf); return(NULL); }

      for(y=0;y<height;y++)
      {
        for(x=0,i=0;x<width;x++)
        {
          if(buf[offset] & (1<<(7-i)))
            vimage->buf[y*width+x] = BLACK;
          else
            vimage->buf[y*width+x] = WHITE;
          /* si i > 7 alors il n'y a plus de pixel a lire dans cet octet */
          if(++i >= 8)
          {
            i = 0;
            offset++;
          }
        }
        /* on traite l'alignement des lignes sur 1 octet */
        if(i > 0) offset++;
      }
      break;
  }
  free(buf);
  return(vimage);
}
