/****************************************************/
/* Bibliotheque d'image 3d                          */
/* image_3d_float.h                                 */
/*                                                  */
/* Ecrit par : Daniel Lacroix (all rights reserved) */
/*                                                  */
/****************************************************/

#ifndef __IMAGE_3D_FLOAT_H__
#define __IMAGE_3D_FLOAT_H__

#include <math.h>
#include <image.h>
#include <list.h>

/* une matrice est un tableau de 16 float, les matrices */
/* sont donc de 4 par 4. Les valeurs sont repartie a des*/
/* offset comme suit                                    */
/* | 0  1  2  3| */
/* | 4  5  6  7| */
/* | 8  9 10 11| */
/* |12 13 14 15| */

/* la 3d est definie d'apres un repere ou :            */
/* x est positif a droite                              */
/* y est positif en haut                               */
/* z est positif au loin (en s'enfoncant dans l'ecran) */

/* dfinition d'un point dans un repre 2d */
typedef struct {
  float x,y;
} dot_2d;

/* dfinition d'un point dans un repre 3d */
typedef struct {
  float x,y,z;
} dot_3d;

/* definition d'une couleur */
typedef struct {
  uint8 r,g,b;
} color;

/* definition d'un polygone */
typedef struct {
  /* indice dans le tableau dot des objets        */
  /* de la dfinition des trois point du triangle */
  /* l'ordre de dfinition des points doit tre   */
  /* tel que si je ramne le vecteur pt1-pt2 sur  */
  /* le vecteur pt1-pt3, pt1-pt2 soit sur la main */
  /* gauche et pt1-pt3 sur la main droite, et     */
  /* l'angle pt1-pt2,pt1-pt3 < 180 degrs. Dans   */
  /* Cette configuration, la normale doit sortir  */
  /* par la tte.                                 */
  uint32 pt1,pt2,pt3;
  /* coefficient de rflection diffuse 0.0 <= coef <= 1.0 */
  float  diffuse;
  /* coefficient de rflection spculaire 0.0 <= coef <= 1.0 */
  float  speculaire;
  /* coeff qui permet de rgler la taille des tches         */
  /* de la rflexion spculaire. (par exemple, 1.0 ... 20.0) */
  float  shine;
} face;

typedef enum{DOT_LIGHT,DIRECTIONNAL_LIGHT} LightType;

/* definition d'une lumire */
typedef struct {
  /* dfini le type de la lumire */
  LightType type;
  
  union {
    /* champs des dot lights */
    struct {
      /* position de la lumire dans l'espace */
      dot_3d pos;
      /* intensit de la lumire (... 0.6, 1.0 ...) */
      float  intensity;
    } dot_light;
    /* champs de directionnal lights */
    struct {
      /* direction de la lumire */
      dot_3d direction;
      /* intensit de la lumire (... 0.6, 1.0 ...) */
      float  intensity;
    } directionnal;
  } value;
} light;

/* pour render_type */
/*enum{FLAT,TEXTURED};*/

/* type de remplissage */
typedef enum{FLAT, GOURAUD, PHONG} FillMode;

/* dfinition d'un objet 3d */
typedef struct {
  /* type de remplissage */  
  FillMode fill_mode;
  /* numro d'identifiant de l'objet (utilise dans obj_buf) */
  uint32 id;
  /* nombre de polygone de cet objet */
  uint32 nb_face;
  /* dfinition de tout les polygones */
  face   *face;
  /* nombre de point de cet objet */
  uint32 nb_dot;
  /* dfinition de tout les points (utilis dans les polygones) */
  dot_3d *dot;
  /* tampon pour les calculs des points de l'objet apres transformation */
  dot_3d *dot_tmp;
  /* vecteur normaux (utilise dans les polygones) par face */
  dot_3d *normal;
  /* vecteur normaux en chaque sommet de chaque face */
  dot_3d *normal_dot;
  /* transformation  appliquer sur l'objet (null si aucune) */
  float  *obj_mat;
  /* temporaire, juste pour la dmo */
  color obj_color;
} obj_3d;

typedef struct _obj_list obj_list;

/* definition d'une liste d'objets 3d */
struct _obj_list {
  obj_3d   *obj;
  obj_list *next;
};

/*
#define DEFAULT_FOCAL_X 120
#define DEFAULT_FOCAL_Y 120
*/
#define DEFAULT_FOCAL_X 200
#define DEFAULT_FOCAL_Y 200

/* valeur la plus eloigne possible */
#define FAR_Z           -2E127

typedef struct {
  /* dimension de l'image 3d (utile pour la taille du zbuf et obj_buf) */
  int32 width,height;
  /* distance de la focal */
  float focal_x,focal_y;
  /* zbuffer pour dcider si un point est cach ou pas */
  float  *zbuf;
  /* indique les objects visibles en chaque point. NULL si non utilise */
  uint32 *obj_buf;
  /* matrice de transformation  appliquer sur tout les objets (NULL si aucune) */
  float  *global_mat;
  /* liste des objets */
  obj_list list;
  /* pour la lumire */
  float ambiante;
  /* liste des lumires utilises */
  List  *light_list;
} image_3d;

/* Cette fonction calcule la couleur  partir */
/* de la couleur d'origine est de la lumire  */
/* la couleur d'origine est dans src et la    */
/* couleur destination sera place dans dst.  */
/* L'intensite lumineuse est dans lum.       */
/* (lum = 0) => (dst = BLACK)                 */
/* (lum = 1) => (dst = src)                   */
void light_col(color *dst, color *src, float lum);

/* transforme le point psrc en pdst avec la matrice pmat */
/* il est possible d'utiliser le meme point pour pdst et */
/* psrc (il et alors ecrit sur lui meme).                */
void trans_pix(dot_3d *pdst, dot_3d *psrc, float *pmat);

/* transforme la normale psrc en pdst avec la matrice pmat */
/* il est possible d'utiliser le meme point pour pdst et   */
/* psrc (il et alors ecrit sur lui meme).                  */
void trans_normal(dot_3d *pdst, dot_3d *psrc, float *pmat);

/* multipile la matrice p1 par p2, la matrice resultante est pdst         */
/* ATTENTION, pdst ne doit pas pointer sur les meme matrices que p1 et p2 */
void mul_mat(float *pdst, float *p1, float *p2);

/* construit la matrice pdst comme tant la transpose de la matrice psrc */
/* les lignes de pdst sont les colonnes de psrc...                        */
/* ATTENTION, pdst ne doit pas pointer sur la mme matrice que psrc       */
void transpose_mat(float *pdst, float *psrc);

/* construit la matrice pdst comme tant la matrice inverse de la matrice */
/* psrc.                                                                  */
/* ATTENTION, pdst ne doit pas pointer sur la mme matrice que psrc       */
void invert_mat(float *pdst, float *psrc);

/* calcul le vecteur normal pdst au plan forme de p1,p2 et p3 */
/* pdst est normalise.                                        */
void normal(dot_3d *pdst, dot_3d *p1, dot_3d *p2, dot_3d *p3);

/* genere une matrice de translation dans pmat pour translater */
/* du vecteur (0,0,0) vers vecteur                             */
void move_mat(dot_3d *vecteur, float *pmat);

/* genere une matrice de zoom de factor. factor = 1.0 => aucun changement */
void zoom_mat(float factor, float *pmat);

/* genere une matrice de rotation autour de l'axe x          */
/* angle est compris entre 0 (0 degres) et 1023 (359 degres) */
void x_rot_mat(int32 angle, float *pmat);

/* genere une matrice de rotation autour de l'axe y          */
/* angle est compris entre 0 (0 degres) et 1023 (359 degres) */
void y_rot_mat(int32 angle, float *pmat);

/* genere une matrice de rotation autour de l'axe z          */
/* angle est compris entre 0 (0 degres) et 1023 (359 degres) */
void z_rot_mat(int32 angle, float *pmat);

/* cree une nouvelle image_3d */
image_3d *image_3d_new(int32 width, int32 height);

/* libere une image_3d */
void image_3d_free(image_3d *pimage_3d);

/* choisi la matrice de transformation qui s'applique a tout les objets */
/* la matrice n'est pas recopie, toute modification de l'original et    */
/* donc repercute. Sa deallocation et aussi a la charge de l'appelant   */
void image_3d_set_global_mat(image_3d *pimage_3d, float *pmat);

/* ajoute un objet 3d a l'image 3d, l'objet n'est pas recopie, toute */
/* modification sur l'original et donc repercute. L'objet et donc    */
/* sa deallocation son a la charge de l'appelant.                    */
void image_3d_prepend_obj(image_3d *pimage_3d, obj_3d *obj);

/* Rajoute une "dot light"  l'image 3d pimage    */
/* pos dfinie l'emplacement de la lumire        */
/* et intensity dfinie sont intensit (ex : 1.0) */
void image_3d_append_dot_light(image_3d *pimage, dot_3d pos, float intensity);

/* si wanted == TRUE alors on active le buffer objet, */
/* sinon on le deactive (l'obj_buf sert a savoir quel */
/* objet se trouve a un point de l'image grace a son  */
/* numero d'identifiant id).                          */
void image_3d_set_obj_buffer(image_3d *pimage_3d, int wanted);

/* libere un obj_3d */
void obj_3d_free(obj_3d *pobj_3d);

/* recalcule l'image 3d. Le rendu est place dans pimage. */
/* pimage DOIT avoir la meme taille que l'image 3d.      */
void image_3d_render(image *pimage, image_3d *pimage_3d);

/* Renvoie la norme du vecteur pdot */
float norme(dot_3d *pdot);

/* Renvoi dans res la symtrie de vect par axis */
void vect_mirror(dot_3d *vect, dot_3d *axis, dot_3d *res);

/* Renvoi le produit scalaire. Si les 2 vecteurs sont normalise cela */
/* revient a renvoyer le cosinus de l'angle entre les 2 vecteurs.    */
/* En effet scalaire(p1,p2) = cos(p1,p2)*norme(p1)*norme(p2)         */
float dot_product(dot_3d *p1, dot_3d *p2);

/* Cette fonction ramne l'objet obj dans un cube unitaire centr       */
/* autour de (0,0,0). Pour cela, l'objet est translat et redimentionn */
void obj_3d_unitize(obj_3d *obj);

/* Cette fonction fusionne ensemble les points d'un objet    */
/* qui  une distance infrieur  epsilon. Une bonne valeur  */
/* pour epsilon est : 0.000001. Les faces dgnrer en point */
/* ou en ligne sont supprimes.                              */
void obj_3d_fusion_dot(obj_3d *obj, float epsilon);

/* Change les proprits de rflexion de l'objet obj  */
void obj_3d_change_properties(obj_3d *obj,
  float diffuse, float speculaire, float shininess);

/* quelques objets 3d pour faire des tests */
extern obj_3d test_obj;
extern obj_3d test_obj2;

#endif /* __IMAGE_3D_FLOAT_H__ */
