/****************************************************/
/* Bibliotheque d'image 3d                          */
/* image_3d_float.c                                 */
/*                                                  */
/* Ecrit par : Daniel Lacroix (all rights reserved) */
/*                                                  */
/****************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <image_3d_float.h>

/**************************************************/
/* Ce moteur 3D souffre de beaucoup de limitation */
/* - ne supporte pas le clipping en Z.            */
/* - ne supporte pas les eclairages.              */
/* - a finir le texturage.                        */
/* - un gros travail a faire pour optimiser.      */
/**************************************************/

/* table des cosinus et sinus pour accelerer */
float *cos_table = NULL;

#define cos_func(x) cos_table[x&0x3FF]
#define sin_func(x) cos_table[(x-256)&0x3FF]

/**********************************************/
/* Cette fonction calcule la couleur  partir */
/* de la couleur d'origine est de la lumire  */
/* la couleur d'origine est dans src et la    */
/* couleur destination sera place dans dst.  */
/* L'intensite lumineuse est dans lum.       */
/* (lum = 0) => (dst = BLACK)                 */
/* (lum = 1) => (dst = src)                   */
void light_col(color *dst, color *src, float lum)
{ float col;

  if(lum < 0.0) { dst->r = 0; dst->g = 0; dst->b = 0; return; }
  /* on calcul la valeur pour chaque composantes */
  /* et on sature le rsultat.                   */
  col = ((float)src->r)*lum;
  if(col > 255.0) col = 255.0;
  dst->r = (uint8)col;
  col = ((float)src->g)*lum;
  if(col > 255.0) col = 255.0;
  dst->g = (uint8)col;
  col = ((float)src->b)*lum;
  if(col > 255.0) col = 255.0;
  dst->b = (uint8)col;
}
/**********************************************/

/************************************/
/* Renvoie la norme du vecteur pdot */
float norme(dot_3d *pdot)
{
  return(sqrt(pdot->x*pdot->x + pdot->y*pdot->y + pdot->z*pdot->z));
}
/************************************/

/*****************************/
/* Normalise le vecteur pdot */
void normalize(dot_3d *pdot)
{ float length;

  length = norme(pdot);

  /* on test pour viter les divisions par 0         */
  /* si la norme vaut 0 alors x,y et z valent dj 0 */
  if(length != 0.0)
  {
    pdot->x /= length;
    pdot->y /= length;
    pdot->z /= length;
  }
}
/*****************************/


/*********************************************************/
/* transforme le point psrc en pdst avec la matrice pmat */
/* psrc est pdst ne DOIVENT pas pointer sur le m endroit */
/* pdst = pmat * psrc                                    */
void trans_pix(dot_3d *pdst, dot_3d *psrc, float *pmat)
{ float w;

  /* si il n'y a pas de matrice pour la transformation, */
  /* on considre qu'il s'agit d'une identite.         */
  if(pmat == NULL)
  {
    *pdst = *psrc;
    return;
  }

  /* c'est le produit ligne colonne */
  w       = pmat[12]*psrc->x + pmat[13]*psrc->y + pmat[14]*psrc->z + pmat[15];
  pdst->x = pmat[ 0]*psrc->x + pmat[ 1]*psrc->y + pmat[ 2]*psrc->z + pmat[ 3];
  pdst->y = pmat[ 4]*psrc->x + pmat[ 5]*psrc->y + pmat[ 6]*psrc->z + pmat[ 7];
  pdst->z = pmat[ 8]*psrc->x + pmat[ 9]*psrc->y + pmat[10]*psrc->z + pmat[11];
  /* on supprime le facteur d'chelle */
  if(w != 0.0)
  {
    pdst->x /= w;
    pdst->y /= w;
    pdst->z /= w;
  }
}
/*********************************************************/

/***********************************************************/
/* transforme la normale psrc en pdst avec la matrice pmat */
/* psrc est pdst ne DOIVENT pas pointer sur le mme endroit*/
/* pdst = pmat * psrc                                      */
void trans_normal(dot_3d *pdst, dot_3d *psrc, float *pmat)
{ float w;

  /* si il n'y a pas de matrice pour la transformation, */
  /* on considre qu'il s'agit d'une identite.         */
  if(pmat == NULL)
  {
    *pdst = *psrc;
    return;
  }

  /* c'est le produit ligne colonne */
  w       = pmat[12]*psrc->x + pmat[13]*psrc->y + pmat[14]*psrc->z + pmat[15];
  pdst->x = pmat[ 0]*psrc->x + pmat[ 1]*psrc->y + pmat[ 2]*psrc->z;
  pdst->y = pmat[ 4]*psrc->x + pmat[ 5]*psrc->y + pmat[ 6]*psrc->z;
  pdst->z = pmat[ 8]*psrc->x + pmat[ 9]*psrc->y + pmat[10]*psrc->z;
  /* on supprime le facteur d'chelle */
  if(w != 0.0)
  {
    pdst->x /= w;
    pdst->y /= w;
    pdst->z /= w;
  }

  /* on normalise le vecteur */
  normalize(pdst);
}
/***********************************************************/

/************************************************/
/* Renvoi dans res la symtrie de vect par axis */
void vect_mirror(dot_3d *vect, dot_3d *axis, dot_3d *res)
{ float dot_cross;

  /* res = 2*scalaire(vect,axis)*axis-vect */
  dot_cross = dot_product(vect,axis);

  res->x = (2.0 * dot_cross * axis->x) - vect->x;
  res->y = (2.0 * dot_cross * axis->y) - vect->y;
  res->z = (2.0 * dot_cross * axis->z) - vect->z;
}
/************************************************/


void mul_mat_3x3(float *pdst, float *p1, float *p2)
{
  /* premire colonne */
  pdst[ 0] = (p1[ 0]*p2[ 0])+(p1[ 1]*p2[ 3])+(p1[ 2]*p2[ 6]);
  pdst[ 3] = (p1[ 3]*p2[ 0])+(p1[ 4]*p2[ 3])+(p1[ 5]*p2[ 6]);
  pdst[ 6] = (p1[ 6]*p2[ 0])+(p1[ 7]*p2[ 3])+(p1[ 8]*p2[ 6]);
  /* deuxime colonne */
  pdst[ 1] = (p1[ 0]*p2[ 1])+(p1[ 1]*p2[ 4])+(p1[ 2]*p2[ 7]);
  pdst[ 4] = (p1[ 3]*p2[ 1])+(p1[ 4]*p2[ 4])+(p1[ 5]*p2[ 7]);
  pdst[ 7] = (p1[ 6]*p2[ 1])+(p1[ 7]*p2[ 4])+(p1[ 8]*p2[ 7]);
  /* troisime colonne */
  pdst[ 2] = (p1[ 0]*p2[ 2])+(p1[ 1]*p2[ 5])+(p1[ 2]*p2[ 8]);
  pdst[ 5] = (p1[ 3]*p2[ 2])+(p1[ 4]*p2[ 5])+(p1[ 5]*p2[ 8]);
  pdst[ 8] = (p1[ 6]*p2[ 2])+(p1[ 7]*p2[ 5])+(p1[ 8]*p2[ 8]);
}

/**************************************************************************/
/* construit la matrice pdst comme tant la transpose de la matrice psrc */
/* les lignes de pdst sont les colonnes de psrc...                        */
/* ATTENTION, pdst ne doit pas pointer sur la mme matrice que psrc       */
void transpose_mat_3x3(float *pdst, float *psrc)
{
  /* premiere ligne */
  pdst[ 0] = psrc[ 0];
  pdst[ 1] = psrc[ 3];
  pdst[ 2] = psrc[ 6];
  /* deuxieme ligne */
  pdst[ 3] = psrc[ 1];
  pdst[ 4] = psrc[ 4];
  pdst[ 5] = psrc[ 7];
  /* troisieme ligne */  
  pdst[ 6] = psrc[ 2];
  pdst[ 7] = psrc[ 5];
  pdst[ 8] = psrc[ 8];
}
/**************************************************************************/

/**************************************************************************/
/* construit la matrice pdst comme tant la matrice inverse de la matrice */
/* psrc.                                                                  */
/* Renvoi 0 si l'inversion est faite, -1 si la matrice psrc est           */
/* dgnr et donc que l'on ne peut pas l'inverser.                      */
/* ATTENTION, pdst ne doit pas pointer sur la mme matrice que psrc       */
int invert_mat_3x3(float *pdst, float *psrc)
{ float t4, t6, t8, t10, t12, t14, t1;
  
  t4  = psrc[0*3+0]*psrc[1*3+1];
  t6  = psrc[0*3+0]*psrc[1*3+2];
  t8  = psrc[0*3+1]*psrc[1*3+0];
  t10 = psrc[0*3+2]*psrc[1*3+0];
  t12 = psrc[0*3+1]*psrc[2*3+0];
  t14 = psrc[0*3+2]*psrc[2*3+0];
  t1  = ( t4*psrc[2*3+2]- t6*psrc[2*3+1]- t8*psrc[2*3+2]+
	 t10*psrc[2*3+1]+t12*psrc[1*3+2]-t14*psrc[1*3+1]);

  /* si la matrice est dgnr, on s'arrte */
  if(t1 == 0)
    return(-1);

  pdst[0*3+0] =  (psrc[1*3+1]*psrc[2*3+2]-psrc[1*3+2]*psrc[2*3+1])/t1;
  pdst[0*3+1] = -(psrc[0*3+1]*psrc[2*3+2]-psrc[0*3+2]*psrc[2*3+1])/t1;
  pdst[0*3+2] =  (psrc[0*3+1]*psrc[1*3+2]-psrc[0*3+2]*psrc[1*3+1])/t1;
  pdst[1*3+0] = -(psrc[1*3+0]*psrc[2*3+2]-psrc[1*3+2]*psrc[2*3+0])/t1;
  pdst[1*3+1] =  (psrc[0*3+0]*psrc[2*3+2]-t14)/t1;
  pdst[1*3+2] = -(t6-t10)/t1;
  pdst[2*3+0] =  (psrc[1*3+0]*psrc[2*3+1]-psrc[1*3+1]*psrc[2*3+0])/t1;
  pdst[2*3+1] = -(psrc[0*3+0]*psrc[2*3+1]-t12)/t1;
  pdst[2*3+2] =  (t4-t8)/t1;
  return(0);
}
/**************************************************************************/


void invert_project(image_3d *pimage_3d, dot_3d *p1, dot_3d *p2, dot_3d *p3,
  dot_2d *src, dot_3d *dst, float *r2, float *r3)
{ float mat[9], mat_inv[9];
  float x,y,focal;
  float k,a1,a2;

  x = src->x - (pimage_3d->width  >> 1);
  y = (pimage_3d->height >> 1) - src->y;

  focal = -pimage_3d->focal_x;
  mat[3*0+0] =     x; mat[3*0+1] = -(p2->x-p1->x); mat[3*0+2] = -(p3->x-p1->x);
  mat[3*1+0] =     y; mat[3*1+1] = -(p2->y-p1->y); mat[3*1+2] = -(p3->y-p1->y);
  mat[3*2+0] = focal; mat[3*2+1] = -(p2->z-p1->z); mat[3*2+2] = -(p3->z-p1->z);

  invert_mat_3x3(mat_inv, mat);

  /* produit vecteur matrice */
  k  = mat_inv[0]*p1->x + mat_inv[1]*p1->y + mat_inv[2]*p1->z;
  a1 = mat_inv[3]*p1->x + mat_inv[4]*p1->y + mat_inv[5]*p1->z;
  a2 = mat_inv[6]*p1->x + mat_inv[7]*p1->y + mat_inv[8]*p1->z;
  
  dst->x = k*x;
  dst->y = k*y;
  dst->z = k*focal;
  
  *r2 = a1;
  *r3 = a2;
}

/**************************************************************************/
/* construit la matrice pdst comme tant la transpose de la matrice psrc */
/* les lignes de pdst sont les colonnes de psrc...                        */
/* ATTENTION, pdst ne doit pas pointer sur la mme matrice que psrc       */
void transpose_mat(float *pdst, float *psrc)
{
  /* premire ligne */
  pdst[ 0] = psrc[ 0];
  pdst[ 1] = psrc[ 4];
  pdst[ 2] = psrc[ 8];
  pdst[ 3] = psrc[12];
  /* deuxime ligne */
  pdst[ 4] = psrc[ 1];
  pdst[ 5] = psrc[ 5];
  pdst[ 6] = psrc[ 9];
  pdst[ 7] = psrc[13];
  /* troisime ligne */
  pdst[ 8] = psrc[ 2];
  pdst[ 9] = psrc[ 6];
  pdst[10] = psrc[10];
  pdst[11] = psrc[14];
  /* quatrime ligne */
  pdst[12] = psrc[ 3];
  pdst[13] = psrc[ 7];
  pdst[14] = psrc[11];
  pdst[15] = psrc[15];
}
/**************************************************************************/

/**************************************************************************/
/* construit la matrice pdst comme tant la matrice inverse de la matrice */
/* psrc.                                                                  */
/* ATTENTION, pdst ne doit pas pointer sur la mme matrice que psrc       */
void invert_mat(float *pdst, float *psrc)
{ float det_mat, det_tmp;
  float det_pos, det_neg;
  int   x,y,x1,y1,x2,y2;
  float *tmp_mat;
#define MAT_SIZE 4

  /* on calcul le dterminant de la matrice source */
  det_mat = 0.0;
  for(y=0; y<MAT_SIZE; y++)
  {
    det_pos = 1.0;
    det_neg = 1.0;
    for(x=0; x<MAT_SIZE; x++)
    {
      x2 = x;
      y2 = (y+x) % MAT_SIZE;
      det_pos *= psrc[(y2*MAT_SIZE)+x2];

      x2 = MAT_SIZE-1-x;
      y2 = (y+x) % MAT_SIZE;
      det_neg *= psrc[(y2*MAT_SIZE)+x2];
    }
    det_mat += det_pos;
    det_mat -= det_neg;
  }

  /* si le dterminant de la matrice source est 0 alors la     */
  /* matrice source est dgnre et ne peut pas tre inverse */
  if(det_mat == 0.0)
  { /* retourne la matrice identite (purement arbitraire) */
    zoom_mat(1.0, pdst);
    return;
  }

  /* alloue une matrice temporaire pour les calculs */
  if((tmp_mat = (float *)malloc(MAT_SIZE*MAT_SIZE*sizeof(float))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* on transpose la matrice */
  transpose_mat(pdst, psrc);

  /* calcul la matrice des dterminants des "sous-matrices" */
  for(y=0; y<MAT_SIZE; y++)
  {
    for(x=0; x<MAT_SIZE; x++)
    {
      /* on calcul le dterminant de la sous-matrice */
      det_tmp = 0.0;
      for(y1=0; y1<MAT_SIZE-1; y1++)
      {
        det_pos = 1.0;
        det_neg = 1.0;
        for(x1=0; x1<MAT_SIZE-1; x1++)
        {
          x2 = x1;
          y2 = (y1+x1) % (MAT_SIZE-1);
          /* retrouve les coordonnes du point de la sous-matrice */
          /* dans la matrice.                                     */
          if(x2 >= x) x2++;
          if(y2 >= y) y2++;
          det_pos *= pdst[(y2*MAT_SIZE)+x2];

          x2 = MAT_SIZE-2-x1;
          y2 = (y1+x1) % (MAT_SIZE-1);
          /* retrouve les coordonnes du point de la sous-matrice */
          /* dans la matrice.                                     */
          if(x2 >= x) x2++;
          if(y2 >= y) y2++;
          det_neg *= pdst[(y2*MAT_SIZE)+x2];
        }
        det_tmp += det_pos;
        det_tmp -= det_neg;
      }
      /* range le rsultat */
      tmp_mat[(y*MAT_SIZE)+x] = det_tmp;      
    }
  }

  /* matrice des cofacteurs. On change le signe */
  /* lorsque la somme de indice est impaire     */
  for(y=0; y<MAT_SIZE; y++)
    for(x=0; x<MAT_SIZE; x++)
      if((x+y)&1) tmp_mat[(y*MAT_SIZE)+x] = -tmp_mat[(y*MAT_SIZE)+x];

  /* on divise la matrice par le dterminant de la matrice psrc */
  /* et on obtient la matrice inverse de psrc                   */
  for(y=0; y<MAT_SIZE; y++)
    for(x=0; x<MAT_SIZE; x++)
      pdst[(y*MAT_SIZE)+x] = tmp_mat[(y*MAT_SIZE)+x] / det_mat;

  /* libere la matrice temporaire */
  free(tmp_mat);
}
/**************************************************************************/

/**************************************************************************/
/* multiplie la matrice p1 par p2, la matrice rsultante est pdst         */
/* ATTENTION, pdst ne doit pas pointer sur les mme matrices que p1 et p2 */
/* pdst = p1 * p2                                                         */
void mul_mat(float *pdst, float *p1, float *p2)
{
  /* premire colonne */
  pdst[ 0] = (p1[ 0]*p2[ 0])+(p1[ 1]*p2[ 4])+(p1[ 2]*p2[ 8])+(p1[ 3]*p2[12]);
  pdst[ 4] = (p1[ 4]*p2[ 0])+(p1[ 5]*p2[ 4])+(p1[ 6]*p2[ 8])+(p1[ 7]*p2[12]);
  pdst[ 8] = (p1[ 8]*p2[ 0])+(p1[ 9]*p2[ 4])+(p1[10]*p2[ 8])+(p1[11]*p2[12]);
  pdst[12] = (p1[12]*p2[ 0])+(p1[13]*p2[ 4])+(p1[14]*p2[ 8])+(p1[15]*p2[12]);
  /* deuxime colonne */
  pdst[ 1] = (p1[ 0]*p2[ 1])+(p1[ 1]*p2[ 5])+(p1[ 2]*p2[ 9])+(p1[ 3]*p2[13]);
  pdst[ 5] = (p1[ 4]*p2[ 1])+(p1[ 5]*p2[ 5])+(p1[ 6]*p2[ 9])+(p1[ 7]*p2[13]);
  pdst[ 9] = (p1[ 8]*p2[ 1])+(p1[ 9]*p2[ 5])+(p1[10]*p2[ 9])+(p1[11]*p2[13]);
  pdst[13] = (p1[12]*p2[ 1])+(p1[13]*p2[ 5])+(p1[14]*p2[ 9])+(p1[15]*p2[13]);
  /* troisime colonne */
  pdst[ 2] = (p1[ 0]*p2[ 2])+(p1[ 1]*p2[ 6])+(p1[ 2]*p2[10])+(p1[ 3]*p2[14]);
  pdst[ 6] = (p1[ 4]*p2[ 2])+(p1[ 5]*p2[ 6])+(p1[ 6]*p2[10])+(p1[ 7]*p2[14]);
  pdst[10] = (p1[ 8]*p2[ 2])+(p1[ 9]*p2[ 6])+(p1[10]*p2[10])+(p1[11]*p2[14]);
  pdst[14] = (p1[12]*p2[ 2])+(p1[13]*p2[ 6])+(p1[14]*p2[10])+(p1[15]*p2[14]);
  /* quatrime colonne */
  pdst[ 3] = (p1[ 0]*p2[ 3])+(p1[ 1]*p2[ 7])+(p1[ 2]*p2[11])+(p1[ 3]*p2[15]);
  pdst[ 7] = (p1[ 4]*p2[ 3])+(p1[ 5]*p2[ 7])+(p1[ 6]*p2[11])+(p1[ 7]*p2[15]);
  pdst[11] = (p1[ 8]*p2[ 3])+(p1[ 9]*p2[ 7])+(p1[10]*p2[11])+(p1[11]*p2[15]);
  pdst[15] = (p1[12]*p2[ 3])+(p1[13]*p2[ 7])+(p1[14]*p2[11])+(p1[15]*p2[15]);
}
/**************************************************************************/

/*********************************************************************/
/* Renvoi le produit scalaire. Si les 2 vecteurs sont normalise cela */
/* revient a renvoyer le cosinus de l'angle entre les 2 vecteurs.    */
/* En effet scalaire(p1,p2) = cos(p1,p2)*norme(p1)*norme(p2)         */
float dot_product(dot_3d *p1, dot_3d *p2)
{ return(p1->x*p2->x + p1->y*p2->y + p1->z*p2->z); }
/******************************************************************/

/**************************************************************/
/* calcul le vecteur normal pdst au plan forme de p1,p2 et p3 */
/* pdst est normalis                                         */
void normal(dot_3d *pdst, dot_3d *p1, dot_3d *p2, dot_3d *p3)
{
  /* c'est le calcul du produit vectoriel (Cross Product)           */
  /* vect2 x vect3 = vectD. normeD = norme2*norme3*sin(vect2,vect3) */
  /* C perpendiculaire au plan A,B                                  */
  pdst->x = (p2->y - p1->y) * (p3->z - p1->z) - (p2->z - p1->z) * (p3->y - p1->y);
  pdst->y = (p2->z - p1->z) * (p3->x - p1->x) - (p2->x - p1->x) * (p3->z - p1->z);
  pdst->z = (p2->x - p1->x) * (p3->y - p1->y) - (p2->y - p1->y) * (p3->x - p1->x);

  /* on normalize la normale */
  normalize(pdst);
}
/**************************************************************/

/***************************************************************/
/* gnre une matrice de translation dans pmat pour translater */
/* du vecteur (0.0,0.0,0.0) vers vecteur                       */
void move_mat(dot_3d *vecteur, float *pmat)
{
  pmat[ 0] = 1.0; pmat[ 5] = 1.0; pmat[10] = 1.0; pmat[15] = 1.0;
  pmat[ 3] = vecteur->x;
  pmat[ 7] = vecteur->y;
  pmat[11] = vecteur->z;
  /* tout le reste a 0 */
  pmat[ 1] = 0.0; pmat[ 2] = 0.0; pmat[ 4] = 0.0; pmat[ 6] = 0.0; pmat[ 8] = 0.0;
  pmat[ 9] = 0.0; pmat[12] = 0.0; pmat[13] = 0.0; pmat[14] = 0.0;
}
/***************************************************************/

/***************************************************************************/
/* gnre une matrice de zoom de factor. factor = 1.0 => aucun changement  */
void zoom_mat(float factor, float *pmat)
{
  pmat[ 0] = factor; pmat[5] = factor; pmat[10] = factor;
  pmat[15] = 1.0;
  /* tout le reste a 0 */
  pmat[ 1] = 0.0; pmat[ 2] = 0.0; pmat[ 3] = 0.0; pmat[ 4] = 0.0;
  pmat[ 6] = 0.0; pmat[ 7] = 0.0; pmat[ 8] = 0.0; pmat[ 9] = 0.0;
  pmat[11] = 0.0; pmat[12] = 0.0; pmat[13] = 0.0; pmat[14] = 0.0;
}
/***************************************************************************/

/*************************************************************/
/* gnre une matrice de rotation autour de l'axe x          */
/* angle est compris entre 0 (0 degrs) et 1023 (359 degrs) */
void x_rot_mat(int32 angle, float *pmat)
{
  pmat[ 5] =  cos_func(angle);
  pmat[ 6] = -sin_func(angle);
  pmat[ 9] =  sin_func(angle);
  pmat[10] =  cos_func(angle);
  pmat[ 0] = 1.0; pmat[15] = 1.0;
  /* tout le reste avec des 0 */
  pmat[ 1] = 0.0; pmat[ 2] = 0.0; pmat[ 3] = 0.0; pmat[ 4] = 0.0;
  pmat[ 7] = 0.0; pmat[ 8] = 0.0; pmat[11] = 0.0; pmat[12] = 0.0;
  pmat[13] = 0.0; pmat[14] = 0.0;
}
/*************************************************************/

/*************************************************************/
/* gnre une matrice de rotation autour de l'axe y          */
/* angle est compris entre 0 (0 degrs) et 1023 (359 degrs) */
void y_rot_mat(int32 angle, float *pmat)
{
  pmat[ 0] =  cos_func(angle);
  pmat[ 8] = -sin_func(angle);
  pmat[ 2] =  sin_func(angle);
  pmat[10] =  cos_func(angle);
  pmat[ 5] = 1.0; pmat[15] = 1.0;
  /* tout le reste avec des 0 */
  pmat[ 1] = 0.0; pmat[ 3] = 0.0; pmat[ 4] = 0.0; pmat[ 6] = 0.0;
  pmat[ 7] = 0.0; pmat[ 9] = 0.0; pmat[11] = 0.0; pmat[12] = 0.0;
  pmat[13] = 0.0; pmat[14] = 0.0;
}
/*************************************************************/

/*************************************************************/
/* gnre une matrice de rotation autour de l'axe z          */
/* angle est compris entre 0 (0 degrs) et 1023 (359 degrs) */
void z_rot_mat(int32 angle, float *pmat)
{
  pmat[ 0] =  cos_func(angle);
  pmat[ 1] = -sin_func(angle);
  pmat[ 4] =  sin_func(angle);
  pmat[ 5] =  cos_func(angle);
  pmat[10] = 1.0; pmat[15] = 1.0;
  /* tout le reste avec des 0 */
  pmat[ 2] = 0.0; pmat[ 3] = 0.0; pmat[ 6] = 0.0; pmat[ 7] = 0.0;
  pmat[ 8] = 0.0; pmat[ 9] = 0.0; pmat[11] = 0.0; pmat[12] = 0.0;
  pmat[13] = 0.0; pmat[14] = 0.0;  
}
/*************************************************************/

/*****************************************************************/
/* Calcul le point qui est au centre de la face de l'objet obj.  */
/* face correnspond  l'indice dans obj->face. Le point resultat */
/* est plac dans dot_res.                                       */
void center_dot(obj_3d *obj, uint32 face, dot_3d *dot_res)
{
  dot_res->x = 
    (obj->dot[obj->face[face].pt1].x+
     obj->dot[obj->face[face].pt2].x+
     obj->dot[obj->face[face].pt3].x)/3.0;

  dot_res->y = 
    (obj->dot[obj->face[face].pt1].y+
     obj->dot[obj->face[face].pt2].y+
     obj->dot[obj->face[face].pt3].y)/3.0;

  dot_res->z = 
    (obj->dot[obj->face[face].pt1].z+
     obj->dot[obj->face[face].pt2].z+
     obj->dot[obj->face[face].pt3].z)/3.0;
}
/*****************************************************************/

/****************************************/
/* Calcul la distance entre deux points */
float dot_length(dot_3d *dot1, dot_3d *dot2)
{
  return(sqrt((dot1->x-dot2->x)*(dot1->x-dot2->x)+
              (dot1->y-dot2->y)*(dot1->y-dot2->y)+
              (dot1->z-dot2->z)*(dot1->z-dot2->z)));
}
/****************************************/

/*********************************************************************/
/* ajoute un objet 3d a l'image 3d, l'objet n'est pas recopie, toute */
/* modification sur l'original et donc repercute. L'objet et donc    */
/* sa deallocation sont a la charge de l'appelant.                   */
void image_3d_prepend_obj(image_3d *pimage_3d, obj_3d *obj)
{ obj_list *list_new;
  uint32 compt;
  uint32 i;
  float  length_sum;
  int    around_size;
  int    around_index;
  int    *around_normal;
  float  *around_length;
  float  cos_angle;

  /* dfinie un angle au dessus duquel, on ne prend     */  
  /* pas la normale d'une face pour calculer la normale */
  /* liss aux sommets de chaque face.                  */
  cos_angle = cos(90 * M_PI / 180.0);

  if((list_new = (obj_list *)malloc(sizeof(obj_list))) == NULL)
  { perror("malloc failed\n"); exit(1); }

  list_new->obj  = obj;
  list_new->next = pimage_3d->list.next;
  pimage_3d->list.next = list_new;
  
  /* calcul les normales aux faces si ce n'est pas fait */
  if(obj->normal == NULL)
  {
    /* alloue la place pour les vecteur normaux */
    if((obj->normal = (dot_3d *)malloc(sizeof(dot_3d)*obj->nb_face)) == NULL)
    { perror("malloc failed\n"); exit(1); }
    
    /* fait les calculs pour chaque face */
    for(compt = 0; compt < obj->nb_face; compt++)
    {
      normal(&(obj->normal[compt]),&(obj->dot[obj->face[compt].pt1]),
        &(obj->dot[obj->face[compt].pt2]),&(obj->dot[obj->face[compt].pt3]));
    }
  }

  /* calcul les normales en chaque sommet */
  if(obj->normal_dot == NULL)
  {
    /* alloue la place pour les vecteur normaux */
    if((obj->normal_dot = (dot_3d *)malloc(sizeof(dot_3d)*obj->nb_face*3)) == NULL)
    { perror("malloc failed\n"); exit(1); }

    /* definie la taille de tableau qui vont         */
    /* conserver les normales autour de notre sommet */
    around_size = 5;

    if((around_normal = (int *)malloc(sizeof(int)*around_size)) == NULL)
    { perror("malloc failed\n"); exit(1); }

    if((around_length = (float *)malloc(sizeof(float)*around_size)) == NULL)
    { perror("malloc failed\n"); exit(1); }

    /* fait les calculs pour chaque sommet de chaque face */
    for(compt = 0; compt < obj->nb_face; compt++)
    {
      /* pour le sommet 1 */
      length_sum = 0.0;
      around_index = 0;

      /* on recherche toutes les faces qui utilisent ce point */
      for(i = 0; i< obj->nb_face; i++)
      { dot_3d center;
        float  length;
      
        if(((obj->face[i].pt1 == obj->face[compt].pt1) ||
            (obj->face[i].pt2 == obj->face[compt].pt1) ||
            (obj->face[i].pt3 == obj->face[compt].pt1)) &&
           (dot_product(&(obj->normal[compt]),&(obj->normal[i])) > cos_angle))
        { /* on calcul le point au centre de la face */
          center_dot(obj, i, &center);
          /* calcul la distance entre le centre de la face (point prsum */
          /* de la normal au plan) et notre sommet                        */
          length = dot_length(&center, &obj->dot[obj->face[compt].pt1]);
          /* met a jour la somme des distances */
          length_sum += length;
          /* plus assez de place pour conserver les normales autour de notre sommet */
          if(around_index >= around_size)
          {
            around_size += 5;
            if((around_normal = (int *)realloc(around_normal,sizeof(int)*around_size)) == NULL)
            { perror("realloc failed\n"); exit(1); }
            if((around_length = (float *)realloc(around_length,sizeof(float)*around_size)) == NULL)
            { perror("realloc failed\n"); exit(1); }
          }
          /* on conserve les valeurs */
          around_length[around_index] = length;
          around_normal[around_index] = i;
          around_index++;
        }
      }
      /* calcul la normale au sommet par interpolation linaire */
      /* des normales aux faces autour du sommet.               */
      obj->normal_dot[compt*3].x = 0.0;
      obj->normal_dot[compt*3].y = 0.0;
      obj->normal_dot[compt*3].z = 0.0;
      for(i = 0; i< around_index; i++)
      {
/*        obj->normal_dot[compt*3].x += 
          obj->normal[around_normal[i]].x*(length_sum-around_length[i]);
        obj->normal_dot[compt*3].y += 
          obj->normal[around_normal[i]].y*(length_sum-around_length[i]);
        obj->normal_dot[compt*3].z += 
          obj->normal[around_normal[i]].z*(length_sum-around_length[i]);*/

/*        obj->normal_dot[compt*3].x += 
          obj->normal[around_normal[i]].x/around_length[i];
        obj->normal_dot[compt*3].y += 
          obj->normal[around_normal[i]].y/around_length[i];
        obj->normal_dot[compt*3].z += 
          obj->normal[around_normal[i]].z/around_length[i];*/

        obj->normal_dot[compt*3].x += 
          obj->normal[around_normal[i]].x;
        obj->normal_dot[compt*3].y += 
          obj->normal[around_normal[i]].y;
        obj->normal_dot[compt*3].z += 
          obj->normal[around_normal[i]].z;
      }
      normalize(&(obj->normal_dot[compt*3]));
/*      printf("around = %d, res (%.2f,%.2f,%.2f)\n",around_index,
        obj->normal_dot[compt*3].x,obj->normal_dot[compt*3].y,obj->normal_dot[compt*3].z);
*/
      /* pour le sommet 2 */
      length_sum = 0.0;
      around_index = 0;

      /* on recherche toutes les faces qui utilisent ce point */
      for(i = 0; i< obj->nb_face; i++)
      { dot_3d center;
        float  length;
      
        if(((obj->face[i].pt1 == obj->face[compt].pt2) ||
            (obj->face[i].pt2 == obj->face[compt].pt2) ||
            (obj->face[i].pt3 == obj->face[compt].pt2)) &&
           (dot_product(&(obj->normal[compt]),&(obj->normal[i])) > cos_angle))
        { /* on calcul le point au centre de la face */
          center_dot(obj, i, &center);
          /* calcul la distance entre le centre de la face (point prsum */
          /* de la normal au plan) et notre sommet                        */
          length = dot_length(&center, &obj->dot[obj->face[compt].pt2]);
          /* met a jour la somme des distances */
          length_sum += length;
          /* plus assez de place pour conserver les normales autour de notre sommet */
          if(around_index >= around_size)
          {
            around_size += 5;
            if((around_normal = (int *)realloc(around_normal,sizeof(int)*around_size)) == NULL)
            { perror("realloc failed\n"); exit(1); }
            if((around_length = (float *)realloc(around_length,sizeof(float)*around_size)) == NULL)
            { perror("realloc failed\n"); exit(1); }
          }
          /* on conserve les valeurs */
          around_length[around_index] = length;
          around_normal[around_index] = i;
          around_index++;
        }
      }
      /* calcul la normale au sommet par interpolation linaire */
      /* des normales aux faces autour du sommet.              */
      obj->normal_dot[(compt*3)+1].x = 0.0;
      obj->normal_dot[(compt*3)+1].y = 0.0;
      obj->normal_dot[(compt*3)+1].z = 0.0;
      for(i = 0; i< around_index; i++)
      {
/*        obj->normal_dot[(compt*3)+1].x += 
          obj->normal[around_normal[i]].x*(length_sum-around_length[i]);
        obj->normal_dot[(compt*3)+1].y += 
          obj->normal[around_normal[i]].y*(length_sum-around_length[i]);
        obj->normal_dot[(compt*3)+1].z += 
          obj->normal[around_normal[i]].z*(length_sum-around_length[i]);*/
        obj->normal_dot[(compt*3)+1].x += 
          obj->normal[around_normal[i]].x;
        obj->normal_dot[(compt*3)+1].y += 
          obj->normal[around_normal[i]].y;
        obj->normal_dot[(compt*3)+1].z += 
          obj->normal[around_normal[i]].z;
      }
      normalize(&(obj->normal_dot[(compt*3)+1]));

      /* pour le sommet 3 */
      length_sum = 0.0;
      around_index = 0;

      /* on recherche toutes les faces qui utilisent ce point */
      for(i = 0; i< obj->nb_face; i++)
      { dot_3d center;
        float  length;
      
        if(((obj->face[i].pt1 == obj->face[compt].pt3) ||
            (obj->face[i].pt2 == obj->face[compt].pt3) ||
            (obj->face[i].pt3 == obj->face[compt].pt3)) &&
           (dot_product(&(obj->normal[compt]),&(obj->normal[i])) > cos_angle))
        { /* on calcul le point au centre de la face */
          center_dot(obj, i, &center);
          /* calcul la distance entre le centre de la face (point prsum */
          /* de la normal au plan) et notre sommet                        */
          length = dot_length(&center, &obj->dot[obj->face[compt].pt3]);
          /* met a jour la somme des distances */
          length_sum += length;
          /* plus assez de place pour conserver les normales autour de notre sommet */
          if(around_index >= around_size)
          {
            around_size += 5;
            if((around_normal = (int *)realloc(around_normal,sizeof(int)*around_size)) == NULL)
            { perror("realloc failed\n"); exit(1); }
            if((around_length = (float *)realloc(around_length,sizeof(float)*around_size)) == NULL)
            { perror("realloc failed\n"); exit(1); }
          }
          /* on conserve les valeurs */
          around_length[around_index] = length;
          around_normal[around_index] = i;
          around_index++;
        }
      }
      /* calcul la normale au sommet par interpolation linaire */
      /* des normales aux faces autour du sommet.               */
      obj->normal_dot[compt*3+2].x = 0.0;
      obj->normal_dot[compt*3+2].y = 0.0;
      obj->normal_dot[compt*3+2].z = 0.0;
      for(i = 0; i< around_index; i++)
      {
/*        obj->normal_dot[(compt*3)+2].x += 
          obj->normal[around_normal[i]].x*(length_sum-around_length[i]);
        obj->normal_dot[(compt*3)+2].y += 
          obj->normal[around_normal[i]].y*(length_sum-around_length[i]);
        obj->normal_dot[(compt*3)+2].z += 
          obj->normal[around_normal[i]].z*(length_sum-around_length[i]);*/
        obj->normal_dot[(compt*3)+2].x += 
          obj->normal[around_normal[i]].x;
        obj->normal_dot[(compt*3)+2].y += 
          obj->normal[around_normal[i]].y;
        obj->normal_dot[(compt*3)+2].z += 
          obj->normal[around_normal[i]].z;
      }
      normalize(&(obj->normal_dot[(compt*3)+2]));
    }
    free(around_normal);
    free(around_length);
  }

  /* si la memoire pour les points une fois transformes n'est pas alloue, 
     on l'alloue ici */
  if(obj->dot_tmp == NULL)
  {
    if((obj->dot_tmp = (dot_3d *)malloc(obj->nb_dot*sizeof(dot_3d))) == NULL)
    { perror("malloc failed "); exit(1); }
  }
}
/*********************************************************************/

/******************************************************/
/* Change les proprits de rflexion de l'objet obj  */
void obj_3d_change_properties(obj_3d *obj, float diffuse, float speculaire, float shininess)
{ uint32 i;

  for(i = 0; i<obj->nb_face; i++)
  {
    obj->face[i].diffuse    = diffuse;
    obj->face[i].speculaire = speculaire;
    obj->face[i].shine      = shininess;
  }
}
/******************************************************/

/**************************************************/
/* Rajoute une "dot light"  l'image 3d pimage    */
/* pos dfinie l'emplacement de la lumire        */
/* et intensity dfinie sont intensit (ex : 1.0) */
void image_3d_append_dot_light(image_3d *pimage, dot_3d pos, float intensity)
{ light *vlight;

  /* on alloue la mmoire pour la lumire */
  if((vlight = (light *)malloc(sizeof(light))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* fixe le type de la lumire */
  vlight->type = DOT_LIGHT;

  /* recope la dfinition de l'emplacement de la lumire */
  memcpy(&(vlight->value.dot_light.pos),&pos,sizeof(dot_3d));

  /* recope l'intensite de la lumire */
  vlight->value.dot_light.intensity = intensity;

  /* insre la lumire dans la liste des lumires */  
  pimage->light_list = list_append(pimage->light_list, vlight);
}
/**************************************************/

/************************************************************************/
/* choisi la matrice de transformation qui s'applique a tout les objets */
/* la matrice n'est pas recopie, toute modification de l'original et    */
/* donc repercute. Sa deallocation et aussi a la charge de l'appelant   */
void image_3d_set_global_mat(image_3d *pimage_3d, float *pmat)
{ pimage_3d->global_mat = pmat; }
/************************************************************************/

/******************************************/
/* rempli les tables de cosinus et sinus  */
void init_cos_sin(void)
{ int32  pos;

  if((cos_table=(float *)malloc(sizeof(float)*1024)) == NULL)
  { perror("malloc failed\n"); exit(1); }
  for(pos = 0; pos < 1024; pos++)
  { cos_table[pos] = cos((pos*2.0*M_PI)/1024.0); }
}
/******************************************/

/******************************/
/* cree une nouvelle image 3d */
image_3d *image_3d_new(int32 width, int32 height)
{ image_3d *vimage_3d;
  
  if(cos_table == NULL) init_cos_sin();
  
  /* allocation de la structure elle meme */ 
  if((vimage_3d=(image_3d *)malloc(sizeof(image_3d))) == NULL)
  { perror("malloc failed\n"); exit(1); }
  /* choisi la distance focal */
  vimage_3d->focal_x = DEFAULT_FOCAL_X; vimage_3d->focal_y = DEFAULT_FOCAL_Y;
  /* rempli les dimensions de l'image */
  vimage_3d->width = width; vimage_3d->height = height;
  /* allocation du zbuffer correspondant a l'image */
  if((vimage_3d->zbuf=(float *)malloc(width*height*sizeof(float))) == NULL)
  { perror("malloc failed\n"); exit(1); }
  /* pas de buffer d'objet au debut */
  vimage_3d->obj_buf = NULL;
  /* pas d'objet au debut */
  vimage_3d->list.next = NULL;
  /* pas de transformation global au debut */
  vimage_3d->global_mat = NULL;
  /* lumiere ambiante par defaut (il faut bien que l'on voit qq chose) */
  vimage_3d->ambiante = 0.25;
  /* cre la liste qui va contenir les lumires */
  vimage_3d->light_list = list_new();
  
  /* POUR LE TEST */
  { dot_3d pos = {100.0, 200.0, 40.0};
  
    image_3d_append_dot_light(vimage_3d, pos, 0.9);
  }
  
  return(vimage_3d);
}
/******************************/

/******************************************************/
/* si wanted == TRUE alors on active le buffer objet, */
/* sinon on le deactive (l'obj_buf sert a savoir quel */
/* objet se trouve a un point de l'image grace a son  */
/* numero d'identifiant id).                          */
void image_3d_set_obj_buffer(image_3d *pimage_3d, int wanted)
{
  if(wanted == TRUE)
  {
    /* si on en veut un et qu'il n'y en a pas deja, on le cree */
    if(pimage_3d->obj_buf == NULL)
    {
      /* allocation du buffer d'objects correspondant a l'image */
      if((pimage_3d->obj_buf=
        (uint32 *)malloc(pimage_3d->width*pimage_3d->height*sizeof(uint32))) == NULL)
      { perror("malloc failed\n"); exit(1); }
    }
  } else {
    /* si il y en avait deja un et que l'on en veut plus, on le libere */
    if(pimage_3d->obj_buf != NULL)
    {
      free(pimage_3d->obj_buf);
      pimage_3d->obj_buf = NULL;
    }
  }
}
/******************************************************/

/***********************/
/* libere une image_3d */
void image_3d_free(image_3d *pimage_3d)
{ obj_list *list_pos,*list_pos_last;

  /* libere le zbuffer si il existe */
  if(pimage_3d->zbuf != NULL)    free(pimage_3d->zbuf);
  /* libere le buffer objet si il existe */
  if(pimage_3d->obj_buf != NULL) free(pimage_3d->obj_buf);
  /* libere la liste des objets */
  list_pos = pimage_3d->list.next;
  while(list_pos != NULL)
  {
    list_pos_last = list_pos;
    list_pos = list_pos->next;
    free(list_pos_last);
  }
  /* libere la liste des lumires. La fonction free */
  /* est appele pour chaque lumire.               */
  list_free_full_simple(pimage_3d->light_list);
  /* libere la structure meme de l'image 3d */
  free(pimage_3d);
}
/***********************/

/***************************/
/* reinitialise le obj_buf */
void clear_obj_buf(image_3d *pimage_3d)
{ uint32 compt;
  /* si il n'y a pas de buffer objet, on quitte */
  if(pimage_3d->obj_buf == NULL) return;
  /* place la valeur qui represent aucun objet dans tout l'obj buffer */
  for(compt=0;compt<pimage_3d->width*pimage_3d->height;compt++)
    pimage_3d->obj_buf[compt] = -1;
}
/***************************/

/***************************/
/* rinitialise le zbuffer */
void clear_zbuf(image_3d *pimage_3d)
{ uint32 compt;
  /* si il n'y a pas de zbuffer, on quitte */
  if(pimage_3d->zbuf == NULL) return;
  /* place la valeur la plus eloigne possible dans tout le zbuffer */
  for(compt=0;compt<pimage_3d->width*pimage_3d->height;compt++)
    pimage_3d->zbuf[compt] = FAR_Z;
}
/***************************/

/********************/
/* libere un obj_3d */
void obj_3d_free(obj_3d *pobj_3d)
{ 
  /* libere les definitions des polygones */
  if(pobj_3d->face != NULL) free(pobj_3d->face);
  /* libere les points qui definissent les polygones */
  if(pobj_3d->dot != NULL) free(pobj_3d->dot);
  /* libere les points qui definissent les polygones apres transformation */
  if(pobj_3d->dot_tmp != NULL) free(pobj_3d->dot_tmp);
  /* libere les vecteurs normaux attache aux polygones */
  if(pobj_3d->normal != NULL) free(pobj_3d->normal);
  /* libere les vecteurs normaux attache aux sommets */
  if(pobj_3d->normal_dot != NULL) free(pobj_3d->normal_dot);
  /* libere l'objet lui meme */
  free(pobj_3d);
}
/********************/

/************************************************************************/
/* Cette fonction ramne l'objet obj dans un cube unitaire centr       */
/* autour de (0,0,0). Pour cela, l'objet est translat et redimentionn */
void obj_3d_unitize(obj_3d *obj)
{ uint32 i;
  float minx, maxx;
  float miny, maxy;
  float minz, maxz;
  float width, height, depth;
  float centerx, centery, centerz;
  float scale;

  /* si l'objet 3D n'existe pas on si il n'y       */
  /* a aucun point pour dfinir l'objet, on quitte */
  if((obj == NULL) || (obj->nb_dot == 0)) return;

  /* on recherche les valeurs extrmes */
  minx = maxx = obj->dot[0].x;
  miny = maxy = obj->dot[0].y;
  minz = maxz = obj->dot[0].z;
  for(i = 1; i < obj->nb_dot; i++)
  {
    if(obj->dot[i].x < minx) minx = obj->dot[i].x;
    if(obj->dot[i].x > maxx) maxx = obj->dot[i].x;

    if(obj->dot[i].y < miny) miny = obj->dot[i].y;
    if(obj->dot[i].y > maxy) maxy = obj->dot[i].y;

    if(obj->dot[i].z < minz) minz = obj->dot[i].z;
    if(obj->dot[i].z > maxz) maxz = obj->dot[i].z;
  }
  /* calcule la largeur, la hauteur et la profondeur de l'objet */
  width  = maxx - minx;
  height = maxy - miny;
  depth  = maxz - minz;

  /* calcule le centre de l'objet */
  centerx = (maxx + minx) / 2.0;
  centery = (maxy + miny) / 2.0;
  centerz = (maxz + minz) / 2.0;

  /* si l'objet est ponctuelle, on quitte */
  if((width == 0) && (height == 0) && (depth == 0)) return;

  /* calcule le facteur de mise  l'echelle  appliquer */
  scale = 2.0 / MAX(MAX(width, height), depth);

  /* on translate le centre et on redimentionne */
  for(i = 0; i < obj->nb_dot; i++)
  { /* on translate */
    obj->dot[i].x -= centerx;
    obj->dot[i].y -= centery;
    obj->dot[i].z -= centerz;
    /* on redimentionne */
    obj->dot[i].x *= scale;
    obj->dot[i].y *= scale;
    obj->dot[i].z *= scale;
  }
}
/************************************************************************/

/*************************************************************/
/* Cette fonction fusionne ensemble les points d'un objet    */
/* qui  une distance infrieur  epsilon. Une bonne valeur  */
/* pour epsilon est : 0.000001. Les faces dgnrer en point */
/* ou en ligne sont supprimes.                              */
void obj_3d_fusion_dot(obj_3d *obj, float epsilon)
{ uint32 i,i2;
  int32  *reloc_tab;
  dot_3d *reloc_dot;
  dot_3d *reloc_normal_dot;
  uint32 reloc_pos = 0;
  face   *reloc_face;

  /* alloue une table pour indiquer les relocations */
  if((reloc_tab = (int32 *)malloc(obj->nb_dot*sizeof(int32))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* alloue le tableau des points relogs */
  if((reloc_dot = (dot_3d *)malloc(obj->nb_dot*sizeof(dot_3d))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* initialise le tableau de relocation avec -1 = point non relog */
  for(i=0; i<obj->nb_dot; i++)
    reloc_tab[i] = -1;

  for(i=0; i<obj->nb_dot; i++)
  {
    /* si le point n'est pas dj relog, on le traite */
    if(reloc_tab[i] == -1)
    {
      reloc_tab[i] = reloc_pos;
      reloc_dot[reloc_pos] = obj->dot[i];

      for(i2=i+1; i2<obj->nb_dot; i2++)
      {
        /* si le point n'est pas relog et qu'il est gal au point */
        /* que l'on est entrain de fusionner, alors on reloge.     */
        if((reloc_tab[i2] == -1) && 
           (ABS(obj->dot[i].x - obj->dot[i2].x) < epsilon) &&
           (ABS(obj->dot[i].y - obj->dot[i2].y) < epsilon) &&
           (ABS(obj->dot[i].z - obj->dot[i2].z) < epsilon))
        {
          reloc_tab[i2] = reloc_pos;
        }
      }

      reloc_pos++;
    }
  }

  /* retaille le tableau des points relogs */
  if(reloc_pos < obj->nb_dot)
    if((reloc_dot = (dot_3d *)realloc(reloc_dot,reloc_pos*sizeof(dot_3d))) == NULL)
    { perror("realloc failed "); exit(1); }

/*  printf("nb_dot before = %d, after = %d\n",obj->nb_dot,reloc_pos);*/

  /* on met a jour les champs de l'objet */
  free(obj->dot);
  obj->nb_dot = reloc_pos;
  obj->dot    = reloc_dot;

  /* alloue le tableau des faces reloges */
  if((reloc_face = (face *)malloc(obj->nb_face*sizeof(face))) == NULL)
  { perror("malloc failed "); exit(1); }

  /* alloue le tableau des normales aux sommets reloges */
  if((reloc_normal_dot = (dot_3d *)malloc(obj->nb_face*3*sizeof(dot_3d))) == NULL)
  { perror("malloc failed "); exit(1); }

  reloc_pos = 0;
  /* on met  jour les faces et on ne garde que les faces    */
  /* qui n'ont pas t fusionnes en un seul point ou ligne. */
  for(i=0; i<obj->nb_face; i++)
  {
    if((reloc_tab[obj->face[i].pt1] != reloc_tab[obj->face[i].pt2]) &&
       (reloc_tab[obj->face[i].pt1] != reloc_tab[obj->face[i].pt3]) &&
       (reloc_tab[obj->face[i].pt2] != reloc_tab[obj->face[i].pt3]))
    {
      reloc_face[reloc_pos] = obj->face[i];
      reloc_face[reloc_pos].pt1 = reloc_tab[obj->face[i].pt1];
      reloc_face[reloc_pos].pt2 = reloc_tab[obj->face[i].pt2];
      reloc_face[reloc_pos].pt3 = reloc_tab[obj->face[i].pt3];
      
      /* si il y avait une normale associe au point, on la reloge aussi */
      if(obj->normal_dot)
      {
        reloc_normal_dot[reloc_pos*3  ] = obj->normal_dot[i*3  ];
        reloc_normal_dot[reloc_pos*3+1] = obj->normal_dot[i*3+1];
        reloc_normal_dot[reloc_pos*3+2] = obj->normal_dot[i*3+2];
      }
      reloc_pos++;
    }
  }

/*  printf("nb_face before = %d, after = %d\n",obj->nb_face,reloc_pos);*/

  /* retaille le tableau des faces reloges */
  if(reloc_pos < obj->nb_face)
    if((reloc_face = (face *)realloc(reloc_face,reloc_pos*sizeof(face))) == NULL)
    { perror("realloc failed "); exit(1); }

  /* retaille le tableau des normales aux sommets reloges */
  if(reloc_pos < obj->nb_face)
    if((reloc_normal_dot = (dot_3d *)realloc(reloc_normal_dot,reloc_pos*3*sizeof(dot_3d))) == NULL)
    { perror("realloc failed "); exit(1); }

  /* on met a jour les champs de l'objet */
  free(obj->face);
  obj->nb_face    = reloc_pos;
  obj->face       = reloc_face;
  if(obj->normal_dot)
  {
    free(obj->normal_dot);
    obj->normal_dot = reloc_normal_dot;
  }

  /* on doit encore traiter les normales au plan si elles existaient */
  /* A FAIRE */

  /* on libre la table de relocation */
  free(reloc_tab);

}
/*************************************************************/

/**************************************************************/
/* projection du point src sur l'cran (coordonnes dans dst) */
void project(image_3d *pimage_3d, dot_2d *dst, dot_3d *src)
{
  if(src->z != 0.0)
  {
    dst->x = ((pimage_3d->focal_x * src->x) / (-src->z)) + (pimage_3d->width  >> 1);
    dst->y = (pimage_3d->height >> 1) - ((pimage_3d->focal_y * src->y) / (-src->z));
  } else {
    /* pour viter une division par 0, on considre que z = 1 */
    dst->x = (pimage_3d->focal_x * src->x) + (pimage_3d->width  >> 1);
    dst->y = (pimage_3d->height >> 1) - (pimage_3d->focal_y * src->y);
  }
}
/**************************************************************/

/* quelques variables d'environnement utilises comme des parametres */
pix    *buf  = NULL;      /* l'image           */
float  *zbuf = NULL;      /* le zbuffer        */
uint32 *obj_buf = NULL;   /* le buffer d'objet */
uint32 current_obj = 0;   /* le numero d'identifiant de l'objet en cours de dessin */
int32  current_width;     /* la largeur de l'image */
int32  current_height;    /* la hauteur de l'image */
int32  current_width_mul_height; /* largeur*hauteur */
float  *current_matrice;  /* la matrice de transformation a appliquer */
float  current_focal_x,current_focal_y; /* distance focal */
image_3d *current_image_3d;
image    *current_image;
/* pour les equations de plan */
float A,B,C,D;
/* pour le texturage */
dot_3d O,V,H;

static void draw_obj(obj_3d *pobj_3d);

/*********************************************************/
/* recalcule l'image 3d. Le rendu est place dans pimage. */
/* pimage DOIT avoir la meme taille que l'image 3d.      */
void image_3d_render(image *pimage, image_3d *pimage_3d)
{ obj_list *list_pos;
  float vmat[16];

  /* reinitialisation du zbuffer */
  clear_zbuf(pimage_3d);
  /* reinitialisation du buffer objets */
  clear_obj_buf(pimage_3d);

  /* mise en place de variable d'environnement */
  buf  = pimage->buf;
  zbuf = pimage_3d->zbuf;
  obj_buf = pimage_3d->obj_buf;
  current_width  = pimage->width;
  current_height = pimage->height;
  current_width_mul_height = current_width * current_height;
  current_focal_x = pimage_3d->focal_x;
  current_focal_y = pimage_3d->focal_y;

  current_image_3d = pimage_3d;
  current_image = pimage;

  list_pos = pimage_3d->list.next;
  while(list_pos != NULL)
  {
    /* calcul de la matrice de transformation pour cet objet */
    if(pimage_3d->global_mat != NULL)
    {
      if(list_pos->obj->obj_mat != NULL)
      {
        mul_mat(vmat, pimage_3d->global_mat, list_pos->obj->obj_mat);
        current_matrice = vmat;
      } else { current_matrice = pimage_3d->global_mat; }
    } else {
      if(list_pos->obj->obj_mat != NULL)
        current_matrice = list_pos->obj->obj_mat;
      else current_matrice =  NULL;
    }
    /* identifiant de l'objet pour l'obj_buf */
    current_obj = list_pos->obj->id;
    /* on dessine l'objet */
    draw_obj(list_pos->obj);
    /* passe a l'objet suivant */
    list_pos = list_pos->next;
  }
}
/*********************************************************/

static void triangle_flat(dot_3d p1, dot_3d p2, dot_3d p3, dot_3d n, color pcol);
static void triangle_texture(dot_3d p1, dot_3d p2, dot_3d p3, dot_3d n);

obj_3d *current_obj_3d;
uint32 current_face;
dot_3d current_face_center;

/*********************************************************/
/* dessine un objet 3d. Se sert des variables globales.  */
/* Cette procedure ne doit pas etre appele a l'exterieur */
/* de ce paquage.                                        */
static void draw_obj(obj_3d *pobj_3d)
{ uint32 xinc;
  dot_3d n;

  /* on fixe l'objet en cours de rendu pour que */
  /* les autres fonctions puisse l'utiliser.    */
  current_obj_3d = pobj_3d;

  /* calcul tout les points apres transformation */
  for(xinc = 0; xinc < pobj_3d->nb_dot; xinc++)
  {
    trans_pix(&pobj_3d->dot_tmp[xinc], &pobj_3d->dot[xinc], current_matrice);
  }

  /* dessine tout les polygones de cet objet */
  for(xinc = 0; xinc < pobj_3d->nb_face; xinc++)
  {
    /* transforme la normale de la face dans le repre de la camra */
    trans_normal(&n,  &pobj_3d->normal[xinc], current_matrice);

    /* on peut aussi recalculer la normale  partir des points transforms */
    /*
    normal(&n,&pobj_3d->dot_tmp[pobj_3d->face[xinc].pt1],
      &pobj_3d->dot_tmp[pobj_3d->face[xinc].pt2],
      &pobj_3d->dot_tmp[pobj_3d->face[xinc].pt3]);
    */

    /* calcul le centre de la face */
    current_face_center.x = (pobj_3d->dot_tmp[pobj_3d->face[xinc].pt1].x+
                             pobj_3d->dot_tmp[pobj_3d->face[xinc].pt2].x+
                             pobj_3d->dot_tmp[pobj_3d->face[xinc].pt3].x)/3.0;
    current_face_center.y = (pobj_3d->dot_tmp[pobj_3d->face[xinc].pt1].y+
                             pobj_3d->dot_tmp[pobj_3d->face[xinc].pt2].y+
                             pobj_3d->dot_tmp[pobj_3d->face[xinc].pt3].y)/3.0;
    current_face_center.z = (pobj_3d->dot_tmp[pobj_3d->face[xinc].pt1].z+
                             pobj_3d->dot_tmp[pobj_3d->face[xinc].pt2].z+
                             pobj_3d->dot_tmp[pobj_3d->face[xinc].pt3].z)/3.0;

    /* si la face nous tourne le dos, on ne l'affiche pas */
    if(dot_product(&current_face_center,&n) > 0.0) continue;
    
    /* on indique dans une variable globale, quelle face */
    /* on est entrain de calculer.                       */
    current_face = xinc;
    
    /* choisi le bon rendu en fonction de l'objet */
/*    switch(pobj_3d->render_type)
    {
      case FLAT :*/
        triangle_flat(pobj_3d->dot_tmp[pobj_3d->face[xinc].pt1],
          pobj_3d->dot_tmp[pobj_3d->face[xinc].pt2],
          pobj_3d->dot_tmp[pobj_3d->face[xinc].pt3],n,
/*          (color){140,200,190}*/
/*          (color){135,140,190}*/
            current_obj_3d->obj_color);
/*          pobj_3d->face[xinc].col1);*/
/*        break;*/
/*      case TEXTURED :
*/        /* ESSAI TEMPORAIRE */
/*        triangle_texture(pobj_3d->dot_tmp[pobj_3d->face[xinc].pt1],
          pobj_3d->dot_tmp[pobj_3d->face[xinc].pt2],
          pobj_3d->dot_tmp[pobj_3d->face[xinc].pt3],n);
        break;*/
/*      default :
        printf("Mode de rendu 3D inconnu\n");
        exit(2);
    }*/
  }
}
/*********************************************************/

static void hline_flat(int32 x1, int32 x2, int32 y, color pcol,
  dot_3d normal_start, dot_3d normal_end, dot_3d dot_start, dot_3d dot_end);
dot_2d d1,d2,d3;

/***********************************************************/
/* Dessine un triangle 3D rempli d'une couleur uni (pcol). */
/* Se sert des variables globales.                         */
static void triangle_flat(dot_3d p1, dot_3d p2, dot_3d p3, dot_3d n, color pcol)
{
  dot_2d dtmp;
/*  int xd1,yd1,xd2,yd2,i;*/
/*  float xd1,yd1,xd2,yd2,i;*/
  int32 start,end;
  int   pos1, pos2, pos3, pos_tmp;
  dot_3d inter_normal_start,inter_normal_end;
  dot_3d inter_normal_start2,inter_normal_end2;
  dot_3d inter_dot_start,inter_dot_end;
  dot_3d nd1,nd2,nd3;

  int32 d1_x, d1_y, d2_x, d2_y, d3_x, d3_y, d_x, d_y;
  int32 xd1,yd1,xd2,yd2,i;

  /* calcul l'equation du plan (Ax + By + Cz + D = 0) */
  D = -p1.x*(p2.y*p3.z-p3.y*p2.z) - p2.x*(p3.y*p1.z-p1.y*p3.z) -
      p3.x*(p1.y*p2.z-p2.y*p1.z);

  /* permet de savoir si la face nous tourne le dos.             */
  /* pour cela il faut que les points des polygones soit         */
  /* definie dans le sens de rotation des aiguilles d'une montre */
/*  if(D < 0.0) return;*/
  
  /* on termine le calcul de l'equation du plan */      
  A = p1.y*(p2.z-p3.z) + p2.y*(p3.z-p1.z) + p3.y*(p1.z-p2.z);
  B = p1.z*(p2.x-p3.x) + p2.z*(p3.x-p1.x) + p3.z*(p1.x-p2.x);
  C = p1.x*(p2.y-p3.y) + p2.x*(p3.y-p1.y) + p3.x*(p1.y-p2.y);
	
  /* on projete les points 3d sur l'ecran */
  project(current_image_3d, &d1, &p1);
  project(current_image_3d, &d2, &p2);
  project(current_image_3d, &d3, &p3);
  
  pos1 = 0; pos2 = 1; pos3 = 2;

  d1_x = d1.x; d1_y = d1.y; d2_x = d2.x; d2_y = d2.y; d3_x = d3.x; d3_y = d3.y;
  
  /* on tri les 3 points par ordre de y */
/*  if(d2.y < d1.y){ dtmp = d1; d1 = d2; d2 = dtmp; pos_tmp = pos1; pos1 = pos2; pos2 = pos_tmp; }
  if(d3.y < d1.y){ dtmp = d1; d1 = d3; d3 = dtmp; pos_tmp = pos1; pos1 = pos3; pos3 = pos_tmp; }
  if(d3.y < d2.y){ dtmp = d2; d2 = d3; d3 = dtmp; pos_tmp = pos2; pos2 = pos3; pos3 = pos_tmp; }
*/
  /* on tri les 3 points par ordre de y */
  if(d2_y < d1_y)
  {
    d_x = d1_x; d_y = d1_y;
    d1_x = d2_x; d1_y = d2_y;
    d2_x = d_x; d2_y = d_y;
    pos_tmp = pos1; pos1 = pos2; pos2 = pos_tmp;
  }
  if(d3_y < d1_y)
  {
    d_x = d1_x; d_y = d1_y;
    d1_x = d3_x; d1_y = d3_y;
    d3_x = d_x; d3_y = d_y;
    pos_tmp = pos1; pos1 = pos3; pos3 = pos_tmp;
  }
  if(d3_y < d2_y)
  {
    d_x = d2_x; d_y = d2_y;
    d2_x = d3_x; d2_y = d3_y;
    d3_x = d_x; d3_y = d_y;
    pos_tmp = pos2; pos2 = pos3; pos3 = pos_tmp;
  }

  xd1 = d2_x - d1_x;
  yd1 = d2_y - d1_y;
  xd2 = d3_x - d1_x;
  yd2 = d3_y - d1_y;


/*  printf("pos1 = %d, pos2 = %d, pos3 = %d, face = %d\n",pos1,pos2,pos3,current_face);
  printf("p1 (%.2f,%.2f,%.2f), p2 (%.2f,%.2f,%.2f), p3 (%.2f,%.2f,%.2f)\n",
    p1.x,p1.y,p1.z, p2.x,p2.y,p2.z, p3.x,p3.y,p3.z);
  printf("normal 1 (%.2f,%.2f,%.2f), normal 2 (%.2f,%.2f,%.2f), normal 3 (%.2f,%.2f,%.2f)\n",
    current_obj_3d->normal_dot[current_face*3+pos1].x,
    current_obj_3d->normal_dot[current_face*3+pos1].y,
    current_obj_3d->normal_dot[current_face*3+pos1].z,
    current_obj_3d->normal_dot[current_face*3+pos2].x,
    current_obj_3d->normal_dot[current_face*3+pos2].y,
    current_obj_3d->normal_dot[current_face*3+pos2].z,
    current_obj_3d->normal_dot[current_face*3+pos3].x,
    current_obj_3d->normal_dot[current_face*3+pos3].y,
    current_obj_3d->normal_dot[current_face*3+pos3].z);
*/

  /* premier cas particulier, une ligne horizontale */
  if((d1_y == d2_y) && (d2_y == d3_y))
  { int pos_sx, pos_ex;

    if(d1_y >= current_height) return;

    if(d1_x < d2_x)
      if(d2_x < d3_x)
      { start = d1_x; pos_sx = pos1; }
      else
        if(d1_x < d3_x)
        { start = d1_x; pos_sx = pos1; }
        else
        { start = d3_x; pos_sx = pos3; }
    else
      if(d1_x < d3_x)
      { start = d2_x; pos_sx = pos2; }
      else
        if(d2_x < d3_x)
        { start = d2_x; pos_sx = pos2; }
        else
        { start = d3_x; pos_sx = pos3; }

    if(d1_x > d2_x)
      if(d2_x > d3_x)
      { end = d1_x; pos_ex = pos1; }
      else
        if(d1_x > d3_x)
        { end = d1_x; pos_ex = pos1; }
        else
        { end = d3_x; pos_ex = pos3; }
    else
      if(d1_x > d3_x)
      { end = d2_x; pos_ex = pos2; }
      else
        if(d2_x > d3_x)
        { end = d2_x; pos_ex = pos2; }
        else
        { end = d3_x; pos_ex = pos3; }

    if(pos_sx == 0)
      inter_dot_start = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
    else if(pos_sx == 1)
      inter_dot_start = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
    else if(pos_sx == 2)
      inter_dot_start = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

    if(pos_ex == 0)
      inter_dot_end = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
    else if(pos_ex == 1)
      inter_dot_end = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
    else if(pos_ex == 2)
      inter_dot_end = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

    inter_normal_start = current_obj_3d->normal_dot[current_face*3+pos_sx];
    inter_normal_end = current_obj_3d->normal_dot[current_face*3+pos_ex];

    trans_normal(&inter_normal_start2,&inter_normal_start,current_matrice);
    trans_normal(&inter_normal_end2,&inter_normal_end,current_matrice);
    
    /* cas particulier, il s'agit d'un point */
    if(start == end)
      hline_flat(start,start,d1_y,pcol,
        inter_normal_start2,
        inter_normal_end2,
        inter_dot_start,inter_dot_end);

    /* cas d'une ligne horizontale > 1 en taille */
    hline_flat(start,end,d1_y,pcol,
      inter_normal_start2,
      inter_normal_end2,
      inter_dot_start,inter_dot_end);
    /* on quitte */
    return;
  }

  if(pos1 == 0)
    nd1 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
  else if(pos1 == 1)
    nd1 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
  else if(pos1 == 2)
    nd1 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

  if(pos2 == 0)
    nd2 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
  else if(pos2 == 1)
    nd2 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
  else if(pos2 == 2)
    nd2 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

  if(pos3 == 0)
    nd3 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
  else if(pos3 == 1)
    nd3 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
  else if(pos3 == 2)
    nd3 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];


  /* cas particulier d1 et d2 sont sur la mme ligne horizontale */
  if(d1_y == d2_y)
  { int pos_sx, pos_ex;
    if(d1_y < current_height)
    {

      if(d1_x < d2_x)
      { start = d1_x; pos_sx = pos1; end = d2_x; pos_ex = pos2; }
      else
      { start = d2_x; pos_sx = pos2; end = d1_x; pos_ex = pos1; }

      /* cas particulier, il s'agit d'un point */
/*      if(start == end)
        hline_flat(start,start,i,pcol,
          current_obj_3d->normal_dot[current_face*3+pos1],
          current_obj_3d->normal_dot[current_face*3+pos1]);*/

      if(pos_sx == 0)
        inter_dot_start = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
      else if(pos_sx == 1)
        inter_dot_start = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
      else if(pos_sx == 2)
        inter_dot_start = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

      if(pos_ex == 0)
        inter_dot_end = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
      else if(pos_ex == 1)
        inter_dot_end = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
      else if(pos_ex == 2)
        inter_dot_end = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

      inter_normal_start = current_obj_3d->normal_dot[current_face*3+pos_sx];
      inter_normal_end = current_obj_3d->normal_dot[current_face*3+pos_ex];

      trans_normal(&inter_normal_start2,&inter_normal_start,current_matrice);
      trans_normal(&inter_normal_end2,&inter_normal_end,current_matrice);

      /* cas d'une ligne horizontale > 1 en taille */
      hline_flat(start,end,d1_y,pcol,
        inter_normal_start2,inter_normal_end2,
        inter_dot_start,inter_dot_end);
    }
  } else {
    for(i = d1_y; i <= MIN(d2_y,current_height-1); i++)
    {
      start = d1_x + ((i - d1_y) * xd1) / yd1;
      end   = d1_x + ((i - d1_y) * xd2) / yd2;

/*      if(start == end) continue;*/

      inter_normal_start.x = 
        current_obj_3d->normal_dot[current_face*3+pos1].x + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos2].x-
         current_obj_3d->normal_dot[current_face*3+pos1].x)) / yd1;
      inter_normal_start.y = 
        current_obj_3d->normal_dot[current_face*3+pos1].y + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos2].y-
         current_obj_3d->normal_dot[current_face*3+pos1].y)) / yd1;
      inter_normal_start.z = 
        current_obj_3d->normal_dot[current_face*3+pos1].z + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos2].z-
         current_obj_3d->normal_dot[current_face*3+pos1].z)) / yd1;

      inter_dot_start.x = nd1.x + ((i - d1_y)*(nd2.x-nd1.x)) / yd1;
      inter_dot_start.y = nd1.y + ((i - d1_y)*(nd2.y-nd1.y)) / yd1;
      inter_dot_start.z = nd1.z + ((i - d1_y)*(nd2.z-nd1.z)) / yd1;

      inter_normal_end.x = 
        current_obj_3d->normal_dot[current_face*3+pos1].x + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos3].x-
         current_obj_3d->normal_dot[current_face*3+pos1].x)) / yd2;
      inter_normal_end.y = 
        current_obj_3d->normal_dot[current_face*3+pos1].y + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos3].y-
         current_obj_3d->normal_dot[current_face*3+pos1].y)) / yd2;
      inter_normal_end.z = 
        current_obj_3d->normal_dot[current_face*3+pos1].z + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos3].z-
         current_obj_3d->normal_dot[current_face*3+pos1].z)) / yd2;

      inter_dot_end.x = nd1.x + ((i - d1_y)*(nd3.x-nd1.x)) / yd2;
      inter_dot_end.y = nd1.y + ((i - d1_y)*(nd3.y-nd1.y)) / yd2;
      inter_dot_end.z = nd1.z + ((i - d1_y)*(nd3.z-nd1.z)) / yd2;

      trans_normal(&inter_normal_start2,&inter_normal_start,current_matrice);
      trans_normal(&inter_normal_end2,&inter_normal_end,current_matrice);

      if(start < end)
        hline_flat(start,end,i,pcol,inter_normal_start2,inter_normal_end2,inter_dot_start,inter_dot_end);
      else
        hline_flat(end,start,i,pcol,inter_normal_end2,inter_normal_start2,inter_dot_end,inter_dot_start);
    }
  }

  xd1 = d3_x - d2_x;
  yd1 = d3_y - d2_y;
  
  if(yd1)
  {
    for(i = d2_y+1; i <= MIN(d3_y,current_height-1); i++)
    {
      start = d1_x + ((i - d1_y) * xd2) / yd2;
      end   = d2_x + ((i - d2_y) * xd1) / yd1;

      inter_normal_start.x = 
        current_obj_3d->normal_dot[current_face*3+pos1].x + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos3].x-
         current_obj_3d->normal_dot[current_face*3+pos1].x)) / yd2;
      inter_normal_start.y = 
        current_obj_3d->normal_dot[current_face*3+pos1].y + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos3].y-
         current_obj_3d->normal_dot[current_face*3+pos1].y)) / yd2;
      inter_normal_start.z = 
        current_obj_3d->normal_dot[current_face*3+pos1].z + 
        ((i - d1_y)*(current_obj_3d->normal_dot[current_face*3+pos3].z-
         current_obj_3d->normal_dot[current_face*3+pos1].z)) / yd2;

      inter_dot_start.x = nd1.x + ((i - d1_y)*(nd3.x-nd1.x)) / yd2;
      inter_dot_start.y = nd1.y + ((i - d1_y)*(nd3.y-nd1.y)) / yd2;
      inter_dot_start.z = nd1.z + ((i - d1_y)*(nd3.z-nd1.z)) / yd2;

      inter_normal_end.x = 
        current_obj_3d->normal_dot[current_face*3+pos2].x + 
        ((i - d2_y)*(current_obj_3d->normal_dot[current_face*3+pos3].x-
         current_obj_3d->normal_dot[current_face*3+pos2].x)) / yd1;
      inter_normal_end.y = 
        current_obj_3d->normal_dot[current_face*3+pos2].y + 
        ((i - d2_y)*(current_obj_3d->normal_dot[current_face*3+pos3].y-
         current_obj_3d->normal_dot[current_face*3+pos2].y)) / yd1;
      inter_normal_end.z = 
        current_obj_3d->normal_dot[current_face*3+pos2].z + 
        ((i - d2_y)*(current_obj_3d->normal_dot[current_face*3+pos3].z-
         current_obj_3d->normal_dot[current_face*3+pos2].z)) / yd1;

      inter_dot_end.x = nd2.x + ((i - d2_y)*(nd3.x-nd2.x)) / yd1;
      inter_dot_end.y = nd2.y + ((i - d2_y)*(nd3.y-nd2.y)) / yd1;
      inter_dot_end.z = nd2.z + ((i - d2_y)*(nd3.z-nd2.z)) / yd1;

/*      if(start == end) continue;*/

      trans_normal(&inter_normal_start2,&inter_normal_start,current_matrice);
      trans_normal(&inter_normal_end2,&inter_normal_end,current_matrice);

      if(start < end)
        hline_flat(start,end,i,pcol,inter_normal_start2,inter_normal_end2,inter_dot_start,inter_dot_end);
      else
        hline_flat(end,start,i,pcol,inter_normal_end2,inter_normal_start2,inter_dot_end,inter_dot_start);
    }
  }
}
/***********************************************************/

#if 0
/***********************************************************/
/* Dessine un triangle 3D rempli d'une couleur uni (pcol). */
/* Se sert des variables globales.                         */
static void triangle_flat(dot_3d p1, dot_3d p2, dot_3d p3, dot_3d n, color pcol)
{
  dot_2d dtmp;
/*  int xd1,yd1,xd2,yd2,i;*/
  float xd1,yd1,xd2,yd2,i;
  int32 start,end;
  int   pos1, pos2, pos3, pos_tmp;
  dot_3d inter_normal_start,inter_normal_end;

  /* calcul l'equation du plan (Ax + By + Cz + D = 0) */
  D = -p1.x*(p2.y*p3.z-p3.y*p2.z) - p2.x*(p3.y*p1.z-p1.y*p3.z) -
      p3.x*(p1.y*p2.z-p2.y*p1.z);

  /* permet de savoir si la face nous tourne le dos.             */
  /* pour cela il faut que les points des polygones soit         */
  /* definie dans le sens de rotation des aiguilles d'une montre */
/*  if(D < 0.0) return;*/
  
  /* on termine le calcul de l'equation du plan */      
  A = p1.y*(p2.z-p3.z) + p2.y*(p3.z-p1.z) + p3.y*(p1.z-p2.z);
  B = p1.z*(p2.x-p3.x) + p2.z*(p3.x-p1.x) + p3.z*(p1.x-p2.x);
  C = p1.x*(p2.y-p3.y) + p2.x*(p3.y-p1.y) + p3.x*(p1.y-p2.y);
	
  /* on projete les points 3d sur l'ecran */
  project(current_image_3d, &d1, &p1);
  project(current_image_3d, &d2, &p2);
  project(current_image_3d, &d3, &p3);
  
  pos1 = 0; pos2 = 1; pos3 = 2;
  
  /* on tri les 3 points par ordre de y */
  if(d2.y < d1.y){ dtmp = d1; d1 = d2; d2 = dtmp; pos_tmp = pos1; pos1 = pos2; pos2 = pos_tmp; }
  if(d3.y < d1.y){ dtmp = d1; d1 = d3; d3 = dtmp; pos_tmp = pos1; pos1 = pos3; pos3 = pos_tmp; }
  if(d3.y < d2.y){ dtmp = d2; d2 = d3; d3 = dtmp; pos_tmp = pos2; pos2 = pos3; pos3 = pos_tmp; }

  xd1 = d2.x - d1.x;
  yd1 = d2.y - d1.y;
  xd2 = d3.x - d1.x;
  yd2 = d3.y - d1.y;

/*  printf("pos1 = %d, pos2 = %d, pos3 = %d, face = %d\n",pos1,pos2,pos3,current_face);
  printf("p1 (%.2f,%.2f,%.2f), p2 (%.2f,%.2f,%.2f), p3 (%.2f,%.2f,%.2f)\n",
    p1.x,p1.y,p1.z, p2.x,p2.y,p2.z, p3.x,p3.y,p3.z);
  printf("normal 1 (%.2f,%.2f,%.2f), normal 2 (%.2f,%.2f,%.2f), normal 3 (%.2f,%.2f,%.2f)\n",
    current_obj_3d->normal_dot[current_face*3+pos1].x,
    current_obj_3d->normal_dot[current_face*3+pos1].y,
    current_obj_3d->normal_dot[current_face*3+pos1].z,
    current_obj_3d->normal_dot[current_face*3+pos2].x,
    current_obj_3d->normal_dot[current_face*3+pos2].y,
    current_obj_3d->normal_dot[current_face*3+pos2].z,
    current_obj_3d->normal_dot[current_face*3+pos3].x,
    current_obj_3d->normal_dot[current_face*3+pos3].y,
    current_obj_3d->normal_dot[current_face*3+pos3].z);
*/
  if(yd1 > 0.0)
  {
    for(i = d1.y; i <= MIN(d2.y,current_height); i++)
    {
      start = d1.x + ((i - d1.y) * xd1) / yd1;
      end   = d1.x + ((i - d1.y) * xd2) / yd2;

      inter_normal_start.x = 
        current_obj_3d->normal_dot[current_face*3+pos1].x + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos2].x-
         current_obj_3d->normal_dot[current_face*3+pos1].x)) / yd1;
      inter_normal_start.y = 
        current_obj_3d->normal_dot[current_face*3+pos1].y + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos2].y-
         current_obj_3d->normal_dot[current_face*3+pos1].y)) / yd1;
      inter_normal_start.z = 
        current_obj_3d->normal_dot[current_face*3+pos1].z + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos2].z-
         current_obj_3d->normal_dot[current_face*3+pos1].z)) / yd1;

      inter_normal_end.x = 
        current_obj_3d->normal_dot[current_face*3+pos1].x + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos3].x-
         current_obj_3d->normal_dot[current_face*3+pos1].x)) / yd2;
      inter_normal_end.y = 
        current_obj_3d->normal_dot[current_face*3+pos1].y + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos3].y-
         current_obj_3d->normal_dot[current_face*3+pos1].y)) / yd2;
      inter_normal_end.z = 
        current_obj_3d->normal_dot[current_face*3+pos1].z + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos3].z-
         current_obj_3d->normal_dot[current_face*3+pos1].z)) / yd2;
        
/*      if(start == end) continue;*/

      if(start < end)
        hline_flat(start,end,i,pcol,inter_normal_start,inter_normal_end);
      else
        hline_flat(end,start,i,pcol,inter_normal_end,inter_normal_start);
    }
  }

  xd1 = d3.x - d2.x;
  yd1 = d3.y - d2.y;
  
  if(yd1 > 0.0)
  {
    for(i = d2.y; i <= MIN(d3.y,current_height); i++)
    {
      start = d1.x + ((i - d1.y) * xd2) / yd2;
      end   = d2.x + ((i - d2.y) * xd1) / yd1;

      inter_normal_start.x = 
        current_obj_3d->normal_dot[current_face*3+pos1].x + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos3].x-
         current_obj_3d->normal_dot[current_face*3+pos1].x)) / yd2;
      inter_normal_start.y = 
        current_obj_3d->normal_dot[current_face*3+pos1].y + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos3].y-
         current_obj_3d->normal_dot[current_face*3+pos1].y)) / yd2;
      inter_normal_start.z = 
        current_obj_3d->normal_dot[current_face*3+pos1].z + 
        ((i - d1.y)*(current_obj_3d->normal_dot[current_face*3+pos3].z-
         current_obj_3d->normal_dot[current_face*3+pos1].z)) / yd2;

      inter_normal_end.x = 
        current_obj_3d->normal_dot[current_face*3+pos2].x + 
        ((i - d2.y)*(current_obj_3d->normal_dot[current_face*3+pos3].x-
         current_obj_3d->normal_dot[current_face*3+pos2].x)) / yd1;
      inter_normal_end.y = 
        current_obj_3d->normal_dot[current_face*3+pos2].y + 
        ((i - d2.y)*(current_obj_3d->normal_dot[current_face*3+pos3].y-
         current_obj_3d->normal_dot[current_face*3+pos2].y)) / yd1;
      inter_normal_end.z = 
        current_obj_3d->normal_dot[current_face*3+pos2].z + 
        ((i - d2.y)*(current_obj_3d->normal_dot[current_face*3+pos3].z-
         current_obj_3d->normal_dot[current_face*3+pos2].z)) / yd1;

/*      if(start == end) continue;*/

      if(start < end)
        hline_flat(start,end,i,pcol,inter_normal_start,inter_normal_end);
      else
        hline_flat(end,start,i,pcol,inter_normal_end,inter_normal_start);
    }
  }
}
/***********************************************************/
#endif

/*********************************************************/
/* Dessine une ligne horizontale entre (x1,y) et (x2,y). */
/* x1 DOIT etre inferieur ou egal a x2.                  */
/* Se sert des variables globales.                       */
static void hline_flat(int32 x1, int32 x2, int32 y, color pcol, 
  dot_3d normal_start, dot_3d normal_end,dot_3d dot_start, dot_3d dot_end)
{
  int32 x;
  int32 ligne;
  float z;
  int32 rx1,rx2;
  float lum, tmp_lum;
  light *vlight;
  List  *vListMove;
  color res_col;
  dot_3d light_vect;
  dot_3d reflect_vect;
  dot_3d view_vect;
  dot_3d inter_normal;
  dot_3d dot_pos;
  dot_2d d2d;
  float a2,a3;
  float d1,d2,d3,sum;
  dot_3d dot1,dot2,dot3;
  dot_3d nor1,nor2,nor3;
  
  if((y < 0) || (x2 < 0)) return;

  ligne = y * current_width;

  /* calcul les vrai x1 et x2 a cause du clipping (debordement = on recoupe)
     de l'image destination */
  rx1 = MAX(0,x1);
  rx2 = MIN(current_width-1,x2);
  
  for(x = rx1; x <= rx2; x++)
  {
    z = -D / ( ((A*(x-(current_width>>1))) / current_image_3d->focal_x) +
               ((B*((current_height>>1)+y)) / current_image_3d->focal_y) + C);

    d2d.x = (float)x;
    d2d.y = (float)y;
    invert_project(current_image_3d,
      &(current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1]),
      &(current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2]),
      &(current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3]),
      &d2d, &dot_pos,&a2,&a3);

    z = dot_pos.z;

    /* on z-buffer pas cher pour dpanner les conneries du prcdent */
/*    z = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1].z;*/

    if(z >= zbuf[ligne + x])
    {
      /* on met  jour le Z-Buffer */
      zbuf[ligne + x] = z;
      
      /* on calcul le niveau de la lumire */
      lum = current_image_3d->ambiante;
      vListMove = current_image_3d->light_list;
      while(!list_is_end(current_image_3d->light_list,vListMove))
      { /* rcupre la dfinition de la lumire courante */
        vlight = list_data(vListMove);

        /* on traite la cas de chaque type lumire */
        switch(vlight->type)
        {
          case DOT_LIGHT          :
/*            printf("start (%.2f,%.2f,%.2f), end (%.2f,%.2f,%.2f)\n",
              normal_start.x,normal_start.y,normal_start.z,
              normal_end.x,normal_end.y,normal_end.z);*/

            if(x1 == x2)
            {
              inter_normal.x = (normal_start.x + normal_end.x)/2.0;
              inter_normal.y = (normal_start.y + normal_end.y)/2.0;
              inter_normal.z = (normal_start.z + normal_end.z)/2.0;

/*              dot_pos = dot_start;*/
            } else {
              inter_normal.x = normal_start.x +
                (normal_end.x - normal_start.x)*(x-x1)/(x2-x1);
              inter_normal.y = normal_start.y +
                (normal_end.y - normal_start.y)*(x-x1)/(x2-x1);
              inter_normal.z = normal_start.z +
                (normal_end.z - normal_start.z)*(x-x1)/(x2-x1);

/*              dot_pos.x = dot_start.x +
                (dot_end.x - dot_start.x)*(x-x1)/(x2-x1);
              dot_pos.y = dot_start.y +
                (dot_end.y - dot_start.y)*(x-x1)/(x2-x1);
              dot_pos.z = dot_start.z +
                (dot_end.z - dot_start.z)*(x-x1)/(x2-x1);*/
            }

/*            dot1 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt1];
            dot2 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt2];
            dot3 = current_obj_3d->dot_tmp[current_obj_3d->face[current_face].pt3];

            d1 = dot_length(&dot_pos, &dot1);
            d2 = dot_length(&dot_pos, &dot2);
            d3 = dot_length(&dot_pos, &dot3);
            sum = d1+d2+d3;

            inter_normal.x = dot1.x*(sum-d1)/sum+dot2.x*(sum-d2)/sum+dot3.x*(sum-d3)/sum;
            inter_normal.y = dot1.y*(sum-d1)/sum+dot2.y*(sum-d2)/sum+dot3.y*(sum-d3)/sum;
            inter_normal.z = dot1.z*(sum-d1)/sum+dot2.z*(sum-d2)/sum+dot3.z*(sum-d3)/sum;
*/
/*            printf("a2 : %.5f,a3 : %.5f\n",a2,a3);*/
/*
            dot1 = current_obj_3d->normal_dot[current_face*3];
            dot2 = current_obj_3d->normal_dot[current_face*3+1];
            dot3 = current_obj_3d->normal_dot[current_face*3+2];
            
            trans_normal(&nor1,&dot1,current_matrice);
            trans_normal(&nor2,&dot2,current_matrice);
            trans_normal(&nor3,&dot3,current_matrice);            

            inter_normal.x = nor1.x*(1.0-a2)*(1.0-a3)+nor2.x*a2+nor3.x*a3;
            inter_normal.y = nor1.y*(1.0-a2)*(1.0-a3)+nor2.y*a2+nor3.y*a3;
            inter_normal.z = nor1.z*(1.0-a2)*(1.0-a3)+nor2.z*a2+nor3.z*a3;

            normalize(&inter_normal);
*/
            if(current_obj_3d->fill_mode == FLAT)
            {
              /* on simule du flat en prenant une normale constante */
              /* et un point constant.                              */
              { dot_3d tmp_normal;
                tmp_normal = current_obj_3d->normal[current_face];
                trans_normal(&inter_normal,&tmp_normal,current_matrice);
                dot_pos = current_face_center;
              }
            }

            /* on calcul le vecteur de la lumire avec la position de la lumire */
            /* et le centre de la face. C'est une appoximation car il faudrait   */
            /* prendre le point exact sur la face mais cet approximation est     */
            /* bonne si la distance entre la lumire est la face est importante  */
            /* devant la taille de la face                                       */
/*            light_vect.x = vlight->value.dot_light.pos.x - current_face_center.x;
            light_vect.y = vlight->value.dot_light.pos.y - current_face_center.y;
            light_vect.z = vlight->value.dot_light.pos.z - current_face_center.z;*/

            light_vect.x = vlight->value.dot_light.pos.x - dot_pos.x;
            light_vect.y = vlight->value.dot_light.pos.y - dot_pos.y;
            light_vect.z = vlight->value.dot_light.pos.z - dot_pos.z;
            
            /* on normalise le vecteur de la lumire */
            normalize(&light_vect);

/*            printf("pos %d <= %d <= %d, inter (%.2f,%.2f,%.2f)\n",
              x1,x,x2,
              inter_normal.x,inter_normal.y,inter_normal.z);*/

/*            printf("light vect (%.2f,%.2f,%.2f)\n",
              light_vect.x,light_vect.y,light_vect.z);*/


/*            inter_normal = current_obj_3d->normal[current_face];*/


/*            tmp_lum = dot_product(&current_obj_3d->normal[current_face],
              &light_vect);*/

/*            tmp_lum = dot_product(&current_obj_3d->normal_dot[current_face*3],
              &light_vect);*/

            tmp_lum = dot_product(&inter_normal,
              &light_vect);
/*
            printf("pos %d <= %d <= %d, inter (%.2f,%.2f,%.2f), vect_light (%.2f,%.2f,%.2f), dot (%.2f,%.2f,%.2f), light (%.2f,%.2f,%.2f), lum %.3f\n",
              x1,x,x2,
              inter_normal.x,inter_normal.y,inter_normal.z,
              light_vect.x,light_vect.y,light_vect.z,
              dot_pos.x,dot_pos.y,dot_pos.z,
              vlight->value.dot_light.pos.x,vlight->value.dot_light.pos.y,vlight->value.dot_light.pos.z,
              tmp_lum);*/

/*            printf("normal (%.2f,%.2f,%.2f), light (%.2f,%.2f,%.2f)\n",
              inter_normal.x,inter_normal.y,inter_normal.z,
              light_vect.x,light_vect.y,light_vect.z);
              
            printf("lum %.3f\n",tmp_lum);
*/
            /* if (tmp_lum < 0) la lumiere n'claire pas la face            */
            /* on multiplie la lumire par le coeff d'mission              */
            /* de la surface (1-absortion). Il s'agit de la lumire diffuse */
            lum += MAX(tmp_lum,0.0)*current_obj_3d->face[current_face].diffuse;

            /* calcul la lumire spculaire */
            
            /* calcul le rayon rflchie = symtrique du vecteur lumire */
            /* par la normale  la face.                                 */
/*            vect_mirror(&light_vect,&current_obj_3d->normal[current_face],&reflect_vect);*/

/*            vect_mirror(&light_vect,&current_obj_3d->normal_dot[current_face*3],&reflect_vect);*/

            vect_mirror(&light_vect,&inter_normal,&reflect_vect);

/*            printf("reflect (%.2f,%.2f,%.2f)\n",
              reflect_vect.x,reflect_vect.y,reflect_vect.z);
*/
            /* dfini le vecteur du point de vue = - coord du point */
/*            view_vect.x = -current_face_center.x;
            view_vect.y = -current_face_center.y;
            view_vect.z = -current_face_center.z;*/

            view_vect.x = -dot_pos.x;
            view_vect.y = -dot_pos.y;
            view_vect.z = -dot_pos.z;

            /* on normalise le vecteur du point de vue */
            normalize(&view_vect);

            /* calcul le produit scalaire du rayon rflchi            */
            /* et du point de vue. Comme les 2 vecteurs sont           */
            /* normaliss, on rcupre cos(angle) entre les 2 vecteurs */
            tmp_lum = dot_product(&view_vect, &reflect_vect);

/*            printf("lum spec %.3f\n",tmp_lum);
*/
            /* if (tmp_lum < 0) la lumiere n'claire pas la face */
            tmp_lum = MAX(tmp_lum, 0.0);

            /* applique le coefficiant de brillance */
            tmp_lum = pow(tmp_lum,current_obj_3d->face[current_face].shine);
/*
            printf("pos %d <= %d <= %d, inter (%.2f,%.2f,%.2f), view (%.2f,%.2f,%.2f), reflect (%.2f,%.2f,%.2f) lum %.3f\n",
              x1,x,x2,
              inter_normal.x,inter_normal.y,inter_normal.z,
              view_vect.x,view_vect.y,view_vect.z,
              reflect_vect.x,reflect_vect.y,reflect_vect.z,
              tmp_lum);*/

            /* on rajoute la contribution spculaire */
            lum += tmp_lum*current_obj_3d->face[current_face].speculaire;
            
            break;
          case DIRECTIONNAL_LIGHT :
            break;
        }

        /* on passe  la lumire suivante */
        list_move_next(current_image_3d->light_list,vListMove);
      }
      /* on calcul la couleur du point en tenant compte */
      /* de la lumire.                                 */
      light_col(&res_col,&pcol,lum);

      /* on place le point rsultat dans l'image */      
      buf[ligne + x] = COL(res_col.r, res_col.g, res_col.b);
    }
  }
}
/*********************************************************/

static void hline_texture(int32 x1, int32 x2, int32 y);

dot_3d ps1,ps2,ps3,ns;
dot_3d vec_light = {0,0,-1};
  
/***********************************************************/
/* Dessine un triangle 3D texture.                         */
/* Se sert des variables globales.                         */
static void triangle_texture(dot_3d p1, dot_3d p2, dot_3d p3, dot_3d n)
{
  dot_2d d1,d2,d3,dtmp;
  int32 xd1,yd1,xd2,yd2,i;
  int32 x1,x2,x3,y1,y2,y3;
  int32 start,end;

  ps1 = p1; ps2 = p2; ps3 = p3; ns = n;

  /* calcul l'equation du plan (Ax + By + Cz + D = 0) */
  D = -p1.x*(p2.y*p3.z-p3.y*p2.z) - p2.x*(p3.y*p1.z-p1.y*p3.z) -
      p3.x*(p1.y*p2.z-p2.y*p1.z);

  /* permet de savoir si la face nous tourne le dos.             */
  /* pour cela il faut que les points des polygones soit         */
  /* definie dans le sens de rotation des aiguilles d'une montre */
  if(D < 0.0) return;
  
  /* on termine le calcul de l'equation du plan */      
  A = p1.y*(p2.z-p3.z) + p2.y*(p3.z-p1.z) + p3.y*(p1.z-p2.z);
  B = p1.z*(p2.x-p3.x) + p2.z*(p3.x-p1.x) + p3.z*(p1.x-p2.x);
  C = p1.x*(p2.y-p3.y) + p2.x*(p3.y-p1.y) + p3.x*(p1.y-p2.y);

/*
  printf("triangle (%f,%f,%f) (%f,%f,%f) (%f,%f,%f)\n",p1.x,p1.y,p1.z,
    p2.x,p2.y,p2.z,p3.x,p3.y,p3.z);
  printf("equation du plan : (%f)*x + (%f)*y + (%f)*z + (%f) = 0\n",A,B,C,D);
  
  printf("(%f,%f,%f).(%f,%f,%f) => light = %f\n",
    n.x,n.y,n.z,vec_light.x,vec_light.y,vec_light.z,dot_product(&n, &vec_light));
*/    
  /* on projete les points 3d sur l'ecran */
  project(current_image_3d, &d1, &p1);
  project(current_image_3d, &d2, &p2);
  project(current_image_3d, &d3, &p3);
  
  /* on tri les 3 points par ordre de y */
  if(d3.y < d2.y){ dtmp = d2; d2 = d3; d3 = dtmp; }
  if(d2.y < d1.y){ dtmp = d1; d1 = d2; d2 = dtmp; }
  if(d3.y < d2.y){ dtmp = d2; d2 = d3; d3 = dtmp; }

  x1 = d1.x; x2 = d2.x; x3 = d3.x;
  y1 = d1.y; y2 = d2.y; y3 = d3.y;

  xd1 = x2 - x1;
  yd1 = y2 - y1;
  xd2 = x3 - x1;
  yd2 = y3 - y1;
/*
  xd1 = d2.x - d1.x;
  yd1 = d2.y - d1.y;
  xd2 = d3.x - d1.x;
  yd2 = d3.y - d1.y;
*/

/*  if((int)(d3.y - d1.y) == 0) 
  {
    hline_texture(MIN3(d1.x,d2.x,d3.x),MAX3(d1.x,d2.x,d3.x),d1.y); return;
  }

  x2 = d1.x;
  step2 = (d3.x - d1.x)/(d3.y - d1.y);
  if((int)(d2.y - d1.y))
  {
    x1 = d1.x;
    step1 = (d2.x - d1.x)/(d2.y - d1.y);

    for(i = d1.y; i <= d2.y; i++)
    { start = x1; end = x2;
      x1 += step1; x2 += step2;
      if(start < end)
        hline_texture(start,end,i);
      else
        hline_texture(end,start,i);
    }
  } else {
    hline_texture(MIN(d1.x,d2.x),MAX(d1.x,d2.x),d1.y); return;
  }

  if((int)(d3.y - d2.y))
  {
    x1 = d2.x;  
    step1 = (d3.x - d2.x)/(d3.y - d2.y);

    for(i = d2.y; i <= d3.y; i++)
    { start = x1; end = x2;
      x1 += step1; x2 += step2;
      if(start < end)
        hline_texture(start,end,i);
      else
        hline_texture(end,start,i);
    }
  } else {
    hline_texture(MIN(d2.x,d3.x),MAX(d2.x,d3.x),d1.y); return;
  }
*/   
/*  if(yd1)
  {
    for(i = d1.y; i <= d2.y; i++)
    {
      start = d1.x + ((i - d1.y) * xd1) / yd1;
      end   = d1.x + ((i - d1.y) * xd2) / yd2;

      if(start < end)
        hline_texture(start,end,i);
      else
        hline_texture(end,start,i);
    }
  }
      
  xd1 = d3.x - d2.x;
  yd1 = d3.y - d2.y;
  
  if(yd1)
  {
    for(i = d2.y + 1; i <= d3.y; i++)
    {
      start = d1.x + ((i - d1.y) * xd2) / yd2;
      end   = d2.x + ((i - d2.y) * xd1) / yd1;

      if(start < end)
        hline_texture(start,end,i);
      else
        hline_texture(end,start,i);
    }
  }*/
  if(yd1)
  {
    for(i = y1; i < y2; i++)
    {
      start = x1 + ((i - y1) * xd1) / yd1;
      end   = x1 + ((i - y1) * xd2) / yd2;

      if(start < end)
        hline_texture(start,end,i);
      else
        hline_texture(end,start,i);
    }
  } /*else { hline_texture(MIN3(x1,x2,x3),MAX3(x1,x2,x3),y1); }*/
      
  xd1 = x3 - x2;
  yd1 = y3 - y2;
  
  if(yd1)
  {
    for(i = y2; i <= y3; i++)
    {
      start = x1 + ((i - y1) * xd2) / yd2;
      end   = x2 + ((i - y2) * xd1) / yd1;

      if(start < end)
        hline_texture(start,end,i);
      else
        hline_texture(end,start,i);
    }
  } /*else { hline_texture(MIN3(x1,x2,x3),MAX3(x1,x2,x3),y2); }*/
  
  if((y1 < current_height) && (y1 > 0) && (x1 < current_width) && (x1 > 0))
  {
    buf[y1*current_width+x1] = RED;
  }

  if((y2 < current_height) && (y2 > 0) && (x2 < current_width) && (x2 > 0))
  {  
    buf[y2*current_width+x2] = RED;
  }
  
  if((y3 < current_height) && (y3 > 0) && (x3 < current_width) && (x3 > 0))
  {
    buf[y3*current_width+x3] = RED;
  }
}
/***********************************************************/

pix texture[16] = {0xFFD0D0FF,0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,
                   0xFFFFFFFF,0xFFD0D0FF,0xFFFFFFFF,0xFFFFFFFF,
                   0xFFFFFFFF,0xFFFFFFFF,0x00FFFFFF,0xFFFFFFFF,
                   0xFFFFFFFF,0xFFFFFFFF,0xFFFFFFFF,0x00FFFFFF};

/*********************************************************/
/* Dessine une ligne horizontale entre (x1,y) et (x2,y). */
/* x1 DOIT etre inferieur ou egal a x2.                  */
/* Se sert des variables globales.                       */
static void hline_texture(int32 x1, int32 x2, int32 y)
{
  int32 x;
  int32 ligne;
  float z,u,v;
  int32 rx1,rx2;
  dot_3d cdot;
  int ui,vi;
  float xr,yr;
  float lum;
  List  *vListMove;
  light *vlight;

  /* quitte si la ligne n'est pas du tout visible */
  if((y < 0) || (y >= current_height) || (x2 < 0) || (x1 >= current_width)) return;

  ligne = y * current_width;

  /* calcul les vrai x1 et x2 a cause du clipping (debordement = on recoupe)
     de l'image destination */
  rx1 = MAX(0,x1);
  rx2 = MIN(current_width-1,x2);

  xr = x - (current_width>>1);
  yr = (current_height>>1) - y;
  
  for(x = rx1; x <= rx2; x++)
  {
    xr = x - (current_width>>1);

    z = -D / ( ((A*xr) / current_image_3d->focal_x) +
               ((B*yr) / current_image_3d->focal_y) + C);

    cdot.z = z;
    cdot.x = (xr * cdot.z) / current_image_3d->focal_x; 
    cdot.y = (yr * cdot.z) / current_image_3d->focal_y;
    
    /*printf("(%f,%f,%f)",cdot.x,cdot.y,cdot.z);*/
    
    /* u = (2xryjzj-xjyrzj-xjyjzr)/(2xiyjzj-xjyizj-xjyjzi) */
    u = (2*(cdot.x-ps1.x)*(ps3.y-ps1.y)*(ps3.z-ps1.z) -
          (ps3.x-ps1.x)*(cdot.y-ps1.y)*(ps3.z-ps1.z)-
          (ps3.x-ps1.x)*(ps3.y-ps1.y)*(cdot.z-ps1.z))/
	      (2*(ps2.x-ps1.x)*(ps3.y-ps1.y)*(ps3.z-ps1.z) -
      	  (ps3.x-ps1.x)*(ps2.y-ps1.y)*(ps3.z-ps1.z) -
          (ps3.x-ps1.x)*(ps3.y-ps1.y)*(ps2.z-ps1.z));

    /* v = (2xryizi-xiyrzi-xiyizr)/(2xjyizi-xiyjzi-xiyizj) */
    v = (2*(cdot.x-ps1.x)*(ps2.y-ps1.y)*(ps2.z-ps1.z) -
          (ps2.x-ps1.x)*(cdot.y-ps1.y)*(ps2.z-ps1.z)-
          (ps2.x-ps1.x)*(ps2.y-ps1.y)*(cdot.z-ps1.z))/
        (2*(ps3.x-ps1.x)*(ps2.y-ps1.y)*(ps2.z-ps1.z) -
          (ps2.x-ps1.x)*(ps3.y-ps1.y)*(ps2.z-ps1.z) -
          (ps2.x-ps1.x)*(ps2.y-ps1.y)*(ps3.z-ps1.z));

    u *= 4; v *= 4;

    /* on calcul la contribution de chaque lumire */
/*    light = 0.0;
    vListMove = current_image_3d->light_list;
    while(!list_is_end(current_image_3d->light_list,vListMove))
    {
      vlight = list_data(vListMove);
      
      switch(vlight->type)
      {
        case DOT_LIGHT          :
          
        
          light = dot_product(&ns, &vec_light);
          vlight->pos
          break;
        case DIRECTIONNAL_LIGHT :
          break;
      }
*/
      /* on passe  la lumire suivante */
/*      list_move_next(current_image_3d->light_list,vListMove);
    }*/

    lum = dot_product(&ns, &vec_light);
    /* on y ajoute la lumiere ambiante qui baigne tout les objets */
    lum += current_image_3d->ambiante;
    lum *= 0.7;
    
    if(lum < 0.0) lum = 0.0;
    
    /*printf("(%f)",light);*/
    
    /*printf(">(%f,%f)",u,v);*/
    
    ui = ((int)u) & 3;
    vi = ((int)v) & 3;
    /*u = u % 4; v = v % 4;*/
    if(z < zbuf[ligne + x])
    {
      zbuf[ligne + x] = z;
      buf[ligne + x] = COL(
        (((float)COL_RED(texture[(vi*4) + ui]))*lum < 256.0)?(uint8)((float)COL_RED(texture[(vi*4) + ui])*lum) : 255,
        (((float)COL_GREEN(texture[(vi*4) + ui]))*lum < 256.0)?(uint8)((float)COL_GREEN(texture[(vi*4) + ui])*lum) : 255,
        (((float)COL_BLUE(texture[(vi*4) + ui]))*lum < 256.0)?(uint8)((float)COL_BLUE(texture[(vi*4) + ui])*lum) : 255);
    }
  }
}
/*********************************************************/

/* un tetraedre pour tester le moteur 3D */
dot_3d test_dot[4]  = {{ 0.0, 60.0,0.0}, {-40.0,-40.0, 0.0},
                       {40.0,-40.0,0.0}, {  0.0,  0.0,60.0}};
face   test_face[4] = {
   {0,2,1, 0.4, 0.5, 5.0},
   {0,1,3, 0.4, 0.5, 5.0},
   {0,3,2, 0.4, 0.5, 5.0},
   {3,1,2, 0.4, 0.5, 5.0}};
   
obj_3d test_obj = {
  /* mode de rendu */
  /*TEXTURED,*/ FLAT,
  /* numero d'identifiant de l'objet (utilise dans obj_buf) */
  1,
  /* nombre de polygone de cet objet */
  4,
  /* definition de tout les polygones */
  test_face,
  /* nombre de point de cet objet */
  4,
  /* definition de tout les points (utilise dans les polygones) */
  test_dot,
  /* tampon pour les calculs des points de l'objet apres transformation */
  NULL,
  /* vecteur normaux (utilise dans les polygones) */
  NULL,
  /* vecteur normaux en chaque sommet */
  NULL,
  /* transformation a appliquer sur l'objet (null si aucune) */
  NULL,
  /* temporaire, juste pour la demo */
  {120,120,120}
};

/* un triangle pour tester le moteur 3D */
dot_3d test_dot2[3]  = {{ 0.0, 60.0,0.0}, {-40.0,-40.0, 0.0},
                        {40.0,-40.0,0.0}};

/*dot_3d test_normal_dot2[3]  = {{ 0.0, 0.7, 0.7}, {-0.5,-0.5, 0.7},
                               { 0.5,-0.5, 0.7}};*/
dot_3d test_normal_dot2[3]  = {{ 0.0, 0.31225, 0.95}, {0.0, -0.31225, 0.95},
                               { 0.0, -0.31225, 0.95}};

                       
face   test_face2[1] = {
   {0,1,2, 0.5, 2.0, 15.0}};
   
obj_3d test_obj2 = {
  /* mode de rendu */
  /*TEXTURED,*/ PHONG,
  /* numero d'identifiant de l'objet (utilise dans obj_buf) */
  2,
  /* nombre de polygone de cet objet */
  1,
  /* definition de tout les polygones */
  test_face2,
  /* nombre de point de cet objet */
  3,
  /* definition de tout les points (utilise dans les polygones) */
  test_dot2,
  /* tampon pour les calculs des points de l'objet apres transformation */
  NULL,
  /* vecteur normaux (utilise dans les polygones) */
  NULL,
  /* vecteur normaux en chaque sommet */
  test_normal_dot2,
  /* transformation a appliquer sur l'objet (null si aucune) */
  NULL,
  /* temporaire, juste pour la demo */
  {120,120,120}
};

