/*
	File:		Resource.cc

	Function:	Read mac resource forks under unix

	Author:		Andrew Willmott

	Notes:		
*/

#include "Resource.h"
#include <netinet/in.h>

#define dprintf if (0) printf

Void ResourceFile::ReadID(ResID id)
{
	fread(id, 4, 1, resFork);
	id[4] = 0;
}

ResLong ResourceFile::ReadLong()
{
	ResLong len;
	
	fread(&len, 4, 1, resFork);

	return(ntohl(len));
}

ResShort ResourceFile::ReadShort()
{
	ResShort result;
	
	fread(&result, 2, 1, resFork);

	return(ntohs(result));
}

Void *ResourceFile::ReadResource(ResLong *size)
{
	ResLong len = ReadLong();
	char *result;
	
	result = new char[len];
	fread(result, len, 1, resFork);
	if (size)
		*size = len;

	return(result);
}

Void *ResourceFile::ReadResource(Int i, Int j, ResLong *size)
{
	fseek(resFork, resHeader.dataOffset + resLists[i][j].dataOffset, SEEK_SET);
	return(ReadResource(size));
}

Int ResourceFile::GetNumTypes()
{
	return(resMap.numTypes);
}

Int ResourceFile::GetNumResources(Int i)
{
	return(resTypes[i].items);
}

Int ResourceFile::GetTypeNumber(ResIDParam typeID)
{
	Int		i;
	
	for (i = 0; i < resMap.numTypes; i++)
		if (strcmp(resTypes[i].id, typeID) == 0)
			return(i);

	return(-1);
}

Int ResourceFile::GetResourceNumber(Int i, ResShort resID)
{
	Int		j;

	for (j = 0; j < resTypes[i].items; j++)
		if (resLists[i][j].id == resID)
			return(j);

	return(-1);
}

Void *ResourceFile::GetResource(ResIDParam typeID, ResShort resID, ResLong *size)
{
	Int		i, j;
	
	if ((i = GetTypeNumber(typeID)) >= 0 
			&& (j = GetResourceNumber(i, resID)) >= 0)
		return(ReadResource(i, j, size));
	else
		return(0);
}

const Char *ResourceFile::GetResourceName(ResIDParam typeID, ResShort resID)
{
	Int		i, j;
	
	if ((i = GetTypeNumber(typeID)) >= 0 
			&& (j = GetResourceNumber(i, resID)) >= 0)
		return(resLists[i][j].name);
	else
		return(0);
}

Void ResourceFile::ReadMap()
{
	Int			i, j, len;
	
	fseek(resFork, resHeader.mapOffset + 22, SEEK_SET);

	resMap.resAttr = ReadShort();
	resMap.typeOffset = ReadShort();
	resMap.nameOffset = ReadShort();
	resMap.numTypes = ReadShort();
	resMap.numTypes++;

	dprintf("Read %d types, type offset %d, name offset %d\n", resMap.numTypes,
		resMap.typeOffset, resMap.nameOffset);

	fseek(resFork, resHeader.mapOffset + resMap.typeOffset + 2, SEEK_SET);
	resTypes = new ResType[resMap.numTypes];

	for (i = 0; i < resMap.numTypes; i++)
	{
		ReadID(resTypes[i].id);
		resTypes[i].items = ReadShort();
		resTypes[i].offset = ReadShort();
		resTypes[i].items++;
	}
	
	for (i = 0; i < resMap.numTypes; i++)
	{
		dprintf("resource '%c%c%c%c': items %d, offset %d\n", 
			resTypes[i].id[0], resTypes[i].id[1], 
			resTypes[i].id[2], resTypes[i].id[3],
			resTypes[i].items, resTypes[i].offset);
	}	

	resLists = new ResPtr[resMap.numTypes];
	
	for (i = 0; i < resMap.numTypes; i++)
	{
		resLists[i] = new Resource[resTypes[i].items];
		fseek(resFork, 
			resTypes[i].offset + resHeader.mapOffset + resMap.typeOffset, SEEK_SET);

		for (j = 0; j < resTypes[i].items; j++)
		{
			ResPtr resPtr = resLists[i] + j;
			
			resPtr->id = ReadShort();
			resPtr->nameOffset = ReadShort();
			resPtr->dataOffset = ReadLong();
			ReadLong();
			resPtr->name = 0;
			
			resPtr->attr = resPtr->dataOffset >> 24;
			resPtr->dataOffset &= 0xFFFFFF;

			dprintf("resource '%c%c%c%c' %d: name %d, data %d, attr %d\n", 
				resTypes[i].id[0], resTypes[i].id[1], 
				resTypes[i].id[2], resTypes[i].id[3],
				resPtr->id, resPtr->nameOffset, resPtr->dataOffset, resPtr->attr);
		}

		for (j = 0; j < resTypes[i].items; j++)
		{
			if (resLists[i][j].nameOffset != -1)
			{
				fseek(resFork, 
					resLists[i][j].nameOffset + resHeader.mapOffset + resMap.nameOffset,
					SEEK_SET
				);
				
				len = fgetc(resFork);
				resLists[i][j].name = new Char[len + 1];
				resLists[i][j].name[len] = 0;
				fread(resLists[i][j].name, len, 1, resFork);
				dprintf("name of %d: %s\n", resLists[i][j].id, resLists[i][j].name);
			}
		}
	}
}

Void ResourceFile::Open(const Char *filename)
{
	
	resFork = fopen(filename, "rb");

	resHeader.dataOffset = ReadLong();
	resHeader.mapOffset = ReadLong();
	resHeader.dataLength = ReadLong();
	resHeader.mapLength = ReadLong();

	dprintf("got header: data %d [%d] map %d [%d]\n", 
		resHeader.dataOffset,
		resHeader.dataLength,
		resHeader.mapOffset,
		resHeader.mapLength
	);

	ReadMap();
}

Void ResourceFile::PrintAllResources()
{
	Int		i;
	
	for (i = 0; i < resMap.numTypes; i++)
	{
		printf("Type '%c%c%c%c'\n", 
			resTypes[i].id[0], resTypes[i].id[1], 
			resTypes[i].id[2], resTypes[i].id[3]);

		PrintResources(i);
	}
}

Void ResourceFile::PrintResources(Int i)
{
	Int		j, len;
	
	for (j = 0; j < resTypes[i].items; j++)
	{
		fseek(resFork, resHeader.dataOffset + resLists[i][j].dataOffset, SEEK_SET);
		len = ReadLong();
		
		printf("    %6d   %8d", resLists[i][j].id, len);
		if (resLists[i][j].name)
			printf("      \"%s\"\n", resLists[i][j].name);
		else
			printf("\n");
	}
}

Void ResourceFile::Close()
{
	fclose(resFork);
}
