/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <stddef.h>
#include <stdlib.h>

#include "xen_internal.h"
#include <xen/api/xen_common.h>
#include <xen/api/xen_role.h>


XEN_FREE(xen_role)
XEN_SET_ALLOC_FREE(xen_role)
XEN_ALLOC(xen_role_record)
XEN_SET_ALLOC_FREE(xen_role_record)
XEN_ALLOC(xen_role_record_opt)
XEN_RECORD_OPT_FREE(xen_role)
XEN_SET_ALLOC_FREE(xen_role_record_opt)


static const struct_member xen_role_record_struct_members[] =
    {
        { .key = "uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_role_record, uuid) },
        { .key = "name_label",
          .type = &abstract_type_string,
          .offset = offsetof(xen_role_record, name_label) },
        { .key = "name_description",
          .type = &abstract_type_string,
          .offset = offsetof(xen_role_record, name_description) },
        { .key = "subroles",
          .type = &abstract_type_ref_set,
          .offset = offsetof(xen_role_record, subroles) }
    };

const abstract_type xen_role_record_abstract_type_ =
    {
       .typename = STRUCT,
       .struct_size = sizeof(xen_role_record),
       .member_count =
           sizeof(xen_role_record_struct_members) / sizeof(struct_member),
       .members = xen_role_record_struct_members
    };


void
xen_role_record_free(xen_role_record *record)
{
    if (record == NULL)
    {
        return;
    }
    free(record->handle);
    free(record->uuid);
    free(record->name_label);
    free(record->name_description);
    xen_role_record_opt_set_free(record->subroles);
    free(record);
}


bool
xen_role_get_record(xen_session *session, xen_role_record **result, xen_role role)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = role }
        };

    abstract_type result_type = xen_role_record_abstract_type_;

    *result = NULL;
    XEN_CALL_("role.get_record");

    if (session->ok)
    {
       (*result)->handle = xen_strdup_((*result)->uuid);
    }

    return session->ok;
}


bool
xen_role_get_by_uuid(xen_session *session, xen_role *result, char *uuid)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("role.get_by_uuid");
    return session->ok;
}


bool
xen_role_get_by_name_label(xen_session *session, struct xen_role_set **result, char *label)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = label }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("role.get_by_name_label");
    return session->ok;
}


bool
xen_role_get_name_label(xen_session *session, char **result, xen_role role)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = role }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("role.get_name_label");
    return session->ok;
}


bool
xen_role_get_name_description(xen_session *session, char **result, xen_role role)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = role }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("role.get_name_description");
    return session->ok;
}


bool
xen_role_get_subroles(xen_session *session, struct xen_role_set **result, xen_role role)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = role }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("role.get_subroles");
    return session->ok;
}


bool
xen_role_get_permissions(xen_session *session, struct xen_role_set **result, xen_role self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("role.get_permissions");
    return session->ok;
}


bool
xen_role_get_permissions_name_label(xen_session *session, struct xen_string_set **result, xen_role self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("role.get_permissions_name_label");
    return session->ok;
}


bool
xen_role_get_by_permission(xen_session *session, struct xen_role_set **result, xen_role permission)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = permission }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("role.get_by_permission");
    return session->ok;
}


bool
xen_role_get_by_permission_name_label(xen_session *session, struct xen_role_set **result, char *label)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = label }
        };

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    XEN_CALL_("role.get_by_permission_name_label");
    return session->ok;
}


bool
xen_role_get_all(xen_session *session, struct xen_role_set **result)
{

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    xen_call_(session, "role.get_all", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_role_get_uuid(xen_session *session, char **result, xen_role role)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = role }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("role.get_uuid");
    return session->ok;
}
