/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */


#include <stddef.h>
#include <stdlib.h>

#include "xen_cls_internal.h"
#include "xen_internal.h"
#include <xen/api/xen_common.h>
#include <xen/api/xen_message.h>


XEN_FREE(xen_message)
XEN_SET_ALLOC_FREE(xen_message)
XEN_ALLOC(xen_message_record)
XEN_SET_ALLOC_FREE(xen_message_record)
XEN_ALLOC(xen_message_record_opt)
XEN_RECORD_OPT_FREE(xen_message)
XEN_SET_ALLOC_FREE(xen_message_record_opt)


static const struct_member xen_message_record_struct_members[] =
    {
        { .key = "uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_message_record, uuid) },
        { .key = "name",
          .type = &abstract_type_string,
          .offset = offsetof(xen_message_record, name) },
        { .key = "priority",
          .type = &abstract_type_int,
          .offset = offsetof(xen_message_record, priority) },
        { .key = "cls",
          .type = &xen_cls_abstract_type_,
          .offset = offsetof(xen_message_record, cls) },
        { .key = "obj_uuid",
          .type = &abstract_type_string,
          .offset = offsetof(xen_message_record, obj_uuid) },
        { .key = "timestamp",
          .type = &abstract_type_datetime,
          .offset = offsetof(xen_message_record, timestamp) },
        { .key = "body",
          .type = &abstract_type_string,
          .offset = offsetof(xen_message_record, body) }
    };

const abstract_type xen_message_record_abstract_type_ =
    {
       .typename = STRUCT,
       .struct_size = sizeof(xen_message_record),
       .member_count =
           sizeof(xen_message_record_struct_members) / sizeof(struct_member),
       .members = xen_message_record_struct_members
    };


void
xen_message_record_free(xen_message_record *record)
{
    if (record == NULL)
    {
        return;
    }
    free(record->handle);
    free(record->uuid);
    free(record->name);
    free(record->obj_uuid);
    free(record->body);
    free(record);
}


bool
xen_message_create(xen_session *session, xen_message *result, char *name, int64_t priority, enum xen_cls cls, char *obj_uuid, char *body)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = name },
            { .type = &abstract_type_int,
              .u.int_val = priority },
            { .type = &xen_cls_abstract_type_,
              .u.enum_val = cls },
            { .type = &abstract_type_string,
              .u.string_val = obj_uuid },
            { .type = &abstract_type_string,
              .u.string_val = body }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("message.create");
    return session->ok;
}


bool
xen_message_destroy(xen_session *session, xen_message self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    xen_call_(session, "message.destroy", param_values, 1, NULL, NULL);
    return session->ok;
}


bool
xen_message_get_all(xen_session *session, struct xen_message_set **result)
{

    abstract_type result_type = abstract_type_string_set;

    *result = NULL;
    xen_call_(session, "message.get_all", NULL, 0, &result_type, result);
    return session->ok;
}


bool
xen_message_get_record(xen_session *session, xen_message_record **result, xen_message self)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = self }
        };

    abstract_type result_type = xen_message_record_abstract_type_;

    *result = NULL;
    XEN_CALL_("message.get_record");

    if (session->ok)
    {
       (*result)->handle = xen_strdup_((*result)->uuid);
    }

    return session->ok;
}


bool
xen_message_get_by_uuid(xen_session *session, xen_message *result, char *uuid)
{
    abstract_value param_values[] =
        {
            { .type = &abstract_type_string,
              .u.string_val = uuid }
        };

    abstract_type result_type = abstract_type_string;

    *result = NULL;
    XEN_CALL_("message.get_by_uuid");
    return session->ok;
}
