/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef XEN_SECRET_H
#define XEN_SECRET_H

#include <xen/api/xen_common.h>
#include <xen/api/xen_secret_decl.h>


/*
 * The secret class.
 * 
 * A secret.
 */


/**
 * Free the given xen_secret.  The given handle must have been
 * allocated by this library.
 */
extern void
xen_secret_free(xen_secret secret);


typedef struct xen_secret_set
{
    size_t size;
    xen_secret *contents[];
} xen_secret_set;

/**
 * Allocate a xen_secret_set of the given size.
 */
extern xen_secret_set *
xen_secret_set_alloc(size_t size);

/**
 * Free the given xen_secret_set.  The given set must have been
 * allocated by this library.
 */
extern void
xen_secret_set_free(xen_secret_set *set);


typedef struct xen_secret_record
{
    xen_secret handle;
    char *uuid;
    char *value;
} xen_secret_record;

/**
 * Allocate a xen_secret_record.
 */
extern xen_secret_record *
xen_secret_record_alloc(void);

/**
 * Free the given xen_secret_record, and all referenced values.  The
 * given record must have been allocated by this library.
 */
extern void
xen_secret_record_free(xen_secret_record *record);


typedef struct xen_secret_record_opt
{
    bool is_record;
    union
    {
        xen_secret handle;
        xen_secret_record *record;
    } u;
} xen_secret_record_opt;

/**
 * Allocate a xen_secret_record_opt.
 */
extern xen_secret_record_opt *
xen_secret_record_opt_alloc(void);

/**
 * Free the given xen_secret_record_opt, and all referenced values. 
 * The given record_opt must have been allocated by this library.
 */
extern void
xen_secret_record_opt_free(xen_secret_record_opt *record_opt);


typedef struct xen_secret_record_set
{
    size_t size;
    xen_secret_record *contents[];
} xen_secret_record_set;

/**
 * Allocate a xen_secret_record_set of the given size.
 */
extern xen_secret_record_set *
xen_secret_record_set_alloc(size_t size);

/**
 * Free the given xen_secret_record_set, and all referenced values. 
 * The given set must have been allocated by this library.
 */
extern void
xen_secret_record_set_free(xen_secret_record_set *set);



typedef struct xen_secret_record_opt_set
{
    size_t size;
    xen_secret_record_opt *contents[];
} xen_secret_record_opt_set;

/**
 * Allocate a xen_secret_record_opt_set of the given size.
 */
extern xen_secret_record_opt_set *
xen_secret_record_opt_set_alloc(size_t size);

/**
 * Free the given xen_secret_record_opt_set, and all referenced values.
 *  The given set must have been allocated by this library.
 */
extern void
xen_secret_record_opt_set_free(xen_secret_record_opt_set *set);


/**
 * Get a record containing the current state of the given secret.
 */
extern bool
xen_secret_get_record(xen_session *session, xen_secret_record **result, xen_secret secret);


/**
 * Get a reference to the secret instance with the specified UUID.
 */
extern bool
xen_secret_get_by_uuid(xen_session *session, xen_secret *result, char *uuid);


/**
 * Create a new secret instance, and return its handle.
 */
extern bool
xen_secret_create(xen_session *session, xen_secret *result, xen_secret_record *record);


/**
 * Destroy the specified secret instance.
 */
extern bool
xen_secret_destroy(xen_session *session, xen_secret secret);


/**
 * Get the uuid field of the given secret.
 */
extern bool
xen_secret_get_uuid(xen_session *session, char **result, xen_secret secret);


/**
 * Get the value field of the given secret.
 */
extern bool
xen_secret_get_value(xen_session *session, char **result, xen_secret secret);


/**
 * Set the value field of the given secret.
 */
extern bool
xen_secret_set_value(xen_session *session, xen_secret secret, char *value);


/**
 * .
 */
extern bool
xen_secret_introduce(xen_session *session, xen_secret *result, char *uuid, char *value);


/**
 * Return a list of all the secrets known to the system.
 */
extern bool
xen_secret_get_all(xen_session *session, struct xen_secret_set **result);


#endif
