/*
 * Copyright (c) 2006-2010 Citrix Systems, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef XEN_BOND_H
#define XEN_BOND_H

#include <xen/api/xen_bond_decl.h>
#include <xen/api/xen_common.h>
#include <xen/api/xen_network_decl.h>
#include <xen/api/xen_pif_decl.h>
#include <xen/api/xen_string_string_map.h>


/*
 * The Bond class.
 * 
 * .
 */


/**
 * Free the given xen_bond.  The given handle must have been allocated
 * by this library.
 */
extern void
xen_bond_free(xen_bond bond);


typedef struct xen_bond_set
{
    size_t size;
    xen_bond *contents[];
} xen_bond_set;

/**
 * Allocate a xen_bond_set of the given size.
 */
extern xen_bond_set *
xen_bond_set_alloc(size_t size);

/**
 * Free the given xen_bond_set.  The given set must have been allocated
 * by this library.
 */
extern void
xen_bond_set_free(xen_bond_set *set);


typedef struct xen_bond_record
{
    xen_bond handle;
    char *uuid;
    struct xen_pif_record_opt *master;
    struct xen_pif_record_opt_set *slaves;
    xen_string_string_map *other_config;
} xen_bond_record;

/**
 * Allocate a xen_bond_record.
 */
extern xen_bond_record *
xen_bond_record_alloc(void);

/**
 * Free the given xen_bond_record, and all referenced values.  The
 * given record must have been allocated by this library.
 */
extern void
xen_bond_record_free(xen_bond_record *record);


typedef struct xen_bond_record_opt
{
    bool is_record;
    union
    {
        xen_bond handle;
        xen_bond_record *record;
    } u;
} xen_bond_record_opt;

/**
 * Allocate a xen_bond_record_opt.
 */
extern xen_bond_record_opt *
xen_bond_record_opt_alloc(void);

/**
 * Free the given xen_bond_record_opt, and all referenced values.  The
 * given record_opt must have been allocated by this library.
 */
extern void
xen_bond_record_opt_free(xen_bond_record_opt *record_opt);


typedef struct xen_bond_record_set
{
    size_t size;
    xen_bond_record *contents[];
} xen_bond_record_set;

/**
 * Allocate a xen_bond_record_set of the given size.
 */
extern xen_bond_record_set *
xen_bond_record_set_alloc(size_t size);

/**
 * Free the given xen_bond_record_set, and all referenced values.  The
 * given set must have been allocated by this library.
 */
extern void
xen_bond_record_set_free(xen_bond_record_set *set);



typedef struct xen_bond_record_opt_set
{
    size_t size;
    xen_bond_record_opt *contents[];
} xen_bond_record_opt_set;

/**
 * Allocate a xen_bond_record_opt_set of the given size.
 */
extern xen_bond_record_opt_set *
xen_bond_record_opt_set_alloc(size_t size);

/**
 * Free the given xen_bond_record_opt_set, and all referenced values. 
 * The given set must have been allocated by this library.
 */
extern void
xen_bond_record_opt_set_free(xen_bond_record_opt_set *set);


/**
 * Get a record containing the current state of the given Bond.
 */
extern bool
xen_bond_get_record(xen_session *session, xen_bond_record **result, xen_bond bond);


/**
 * Get a reference to the Bond instance with the specified UUID.
 */
extern bool
xen_bond_get_by_uuid(xen_session *session, xen_bond *result, char *uuid);


/**
 * Get the uuid field of the given Bond.
 */
extern bool
xen_bond_get_uuid(xen_session *session, char **result, xen_bond bond);


/**
 * Get the master field of the given Bond.
 */
extern bool
xen_bond_get_master(xen_session *session, xen_pif *result, xen_bond bond);


/**
 * Get the slaves field of the given Bond.
 */
extern bool
xen_bond_get_slaves(xen_session *session, struct xen_pif_set **result, xen_bond bond);


/**
 * Get the other_config field of the given Bond.
 */
extern bool
xen_bond_get_other_config(xen_session *session, xen_string_string_map **result, xen_bond bond);


/**
 * Set the other_config field of the given Bond.
 */
extern bool
xen_bond_set_other_config(xen_session *session, xen_bond bond, xen_string_string_map *other_config);


/**
 * Add the given key-value pair to the other_config field of the given
 * Bond.
 */
extern bool
xen_bond_add_to_other_config(xen_session *session, xen_bond bond, char *key, char *value);


/**
 * Remove the given key and its corresponding value from the
 * other_config field of the given Bond.  If the key is not in that Map, then
 * do nothing.
 */
extern bool
xen_bond_remove_from_other_config(xen_session *session, xen_bond bond, char *key);


/**
 * Create an interface bond.
 */
extern bool
xen_bond_create(xen_session *session, xen_bond *result, xen_network network, struct xen_pif_set *members, char *mac);


/**
 * Destroy an interface bond.
 */
extern bool
xen_bond_destroy(xen_session *session, xen_bond self);


/**
 * Return a list of all the Bonds known to the system.
 */
extern bool
xen_bond_get_all(xen_session *session, struct xen_bond_set **result);


#endif
