/*
 * Copyright (C) 1995-2008 University of Karlsruhe.  All right reserved.
 *
 * This file is part of libFirm.
 *
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file LICENSE.GPL included in the
 * packaging of this file.
 *
 * Licensees holding valid libFirm Professional Edition licenses may use
 * this file in accordance with the libFirm Commercial License.
 * Agreement provided with the Software.
 *
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

/**
 * @file
 * @brief   The codegenerator (transform FIRM into arm FIRM)
 * @author  Oliver Richter, Tobias Gneist, Michael Beck
 * @version $Id: arm_transform.c 25572 2009-02-27 12:55:06Z beck $
 */
#include "config.h"

#include "irnode_t.h"
#include "irgraph_t.h"
#include "irmode_t.h"
#include "irgmod.h"
#include "iredges.h"
#include "irvrfy.h"
#include "ircons.h"
#include "irprintf.h"
#include "dbginfo.h"
#include "iropt_t.h"
#include "debug.h"
#include "error.h"

#include "../benode_t.h"
#include "../beirg_t.h"
#include "../beutil.h"
#include "../betranshlp.h"
#include "bearch_arm_t.h"

#include "arm_nodes_attr.h"
#include "arm_transform.h"
#include "arm_optimize.h"
#include "arm_new_nodes.h"
#include "arm_map_regs.h"

#include "gen_arm_regalloc_if.h"

#include <limits.h>


/** hold the current code generator during transformation */
static arm_code_gen_t *env_cg;

extern ir_op *get_op_Mulh(void);


/****************************************************************************************************
 *                  _        _                        __                           _   _
 *                 | |      | |                      / _|                         | | (_)
 *  _ __   ___   __| | ___  | |_ _ __ __ _ _ __  ___| |_ ___  _ __ _ __ ___   __ _| |_ _  ___  _ __
 * | '_ \ / _ \ / _` |/ _ \ | __| '__/ _` | '_ \/ __|  _/ _ \| '__| '_ ` _ \ / _` | __| |/ _ \| '_ \
 * | | | | (_) | (_| |  __/ | |_| | | (_| | | | \__ \ || (_) | |  | | | | | | (_| | |_| | (_) | | | |
 * |_| |_|\___/ \__,_|\___|  \__|_|  \__,_|_| |_|___/_| \___/|_|  |_| |_| |_|\__,_|\__|_|\___/|_| |_|
 *
 ****************************************************************************************************/

static inline int mode_needs_gp_reg(ir_mode *mode) {
	return mode_is_int(mode) || mode_is_reference(mode);
}

/**
 * Creates a arm_Const node.
 */
static ir_node *create_mov_node(dbg_info *dbg, ir_node *block, long value) {
	ir_mode *mode  = mode_Iu;
	ir_node *res;

	if (mode_needs_gp_reg(mode))
		mode = mode_Iu;
	res = new_bd_arm_Mov_i(dbg, block, mode, value);
	be_dep_on_frame(res);
	return res;
}

/**
 * Creates a arm_Const_Neg node.
 */
static ir_node *create_mvn_node(dbg_info *dbg, ir_node *block, long value) {
	ir_mode *mode = mode_Iu;
	ir_node *res;

	if (mode_needs_gp_reg(mode))
		mode = mode_Iu;
	res = new_bd_arm_Mvn_i(dbg, block, mode, value);
	be_dep_on_frame(res);
	return res;
}

#define NEW_BINOP_NODE(opname, env, op1, op2) new_bd_arm_##opname(env->dbg, current_ir_graph, env->block, op1, op2, env->mode)

/**
 * Creates a possible DAG for an constant.
 */
static ir_node *create_const_graph_value(dbg_info *dbg, ir_node *block, unsigned int value) {
	ir_node *result;
	arm_vals v, vn;
	int cnt;
	ir_mode *mode = mode_Iu;

	arm_gen_vals_from_word(value, &v);
	arm_gen_vals_from_word(~value, &vn);

	if (vn.ops < v.ops) {
		/* remove bits */
		result = create_mvn_node(dbg, block, arm_encode_imm_w_shift(vn.shifts[0], vn.values[0]));

		for (cnt = 1; cnt < vn.ops; ++cnt) {
			long value = arm_encode_imm_w_shift(vn.shifts[cnt], vn.values[cnt]);
			ir_node *bic_i_node = new_bd_arm_Bic_i(dbg, block, result, mode, value);
			result = bic_i_node;
		}
	}
	else {
		/* add bits */
		result = create_mov_node(dbg, block, arm_encode_imm_w_shift(v.shifts[0], v.values[0]));

		for (cnt = 1; cnt < v.ops; ++cnt) {
			long value = arm_encode_imm_w_shift(v.shifts[cnt], v.values[cnt]);
			ir_node *orr_i_node = new_bd_arm_Or_i(dbg, block, result, mode, value);
			result = orr_i_node;
		}
	}
	return result;
}

/**
 * Create a DAG constructing a given Const.
 *
 * @param irn  a Firm const
 */
static ir_node *create_const_graph(ir_node *irn, ir_node *block) {
	tarval  *tv = get_Const_tarval(irn);
	ir_mode *mode = get_tarval_mode(tv);
	int     value;

	if (mode_is_reference(mode)) {
		/* ARM is 32bit, so we can safely convert a reference tarval into Iu */
		assert(get_mode_size_bits(mode) == get_mode_size_bits(mode_Iu));
		tv = tarval_convert_to(tv, mode_Iu);
	}
	value = get_tarval_long(tv);
	return create_const_graph_value(get_irn_dbg_info(irn), block, value);
}

/**
 * Create an And that will mask all upper bits
 */
static ir_node *gen_zero_extension(dbg_info *dbg, ir_node *block, ir_node *op, int result_bits) {
	unsigned mask_bits = (1 << result_bits) - 1;
	ir_node *mask_node = create_const_graph_value(dbg, block, mask_bits);
	return new_bd_arm_And(dbg, block, op, mask_node, mode_Iu, ARM_SHF_NONE, 0);
}

/**
 * Generate code for a sign extension.
 */
static ir_node *gen_sign_extension(dbg_info *dbg, ir_node *block, ir_node *op, int result_bits) {
	int shift_width = 32 - result_bits;
	ir_node *shift_const_node = create_const_graph_value(dbg, block, shift_width);
	ir_node *lshift_node = new_bd_arm_Shl(dbg, block, op, shift_const_node, mode_Iu);
	ir_node *rshift_node = new_bd_arm_Shrs(dbg, block, lshift_node, shift_const_node, mode_Iu);
	return rshift_node;
}

/**
 * Transforms a Conv node.
 *
 * @return The created ia32 Conv node
 */
static ir_node *gen_Conv(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *op       = get_Conv_op(node);
	ir_node  *new_op   = be_transform_node(op);
	ir_mode  *src_mode = get_irn_mode(op);
	ir_mode  *dst_mode = get_irn_mode(node);
	dbg_info *dbg      = get_irn_dbg_info(node);

	if (src_mode == dst_mode)
		return new_op;

	if (mode_is_float(src_mode) || mode_is_float(dst_mode)) {
		env_cg->have_fp_insn = 1;

		if (USE_FPA(env_cg->isa)) {
			if (mode_is_float(src_mode)) {
				if (mode_is_float(dst_mode)) {
					/* from float to float */
					return new_bd_arm_fpaMvf(dbg, block, new_op, dst_mode);
				}
				else {
					/* from float to int */
					return new_bd_arm_fpaFix(dbg, block, new_op, dst_mode);
				}
			}
			else {
				/* from int to float */
				return new_bd_arm_fpaFlt(dbg, block, new_op, dst_mode);
			}
		}
		else if (USE_VFP(env_cg->isa)) {
			panic("VFP not supported yet");
			return NULL;
		}
		else {
			panic("Softfloat not supported yet");
			return NULL;
		}
	}
	else { /* complete in gp registers */
		int src_bits = get_mode_size_bits(src_mode);
		int dst_bits = get_mode_size_bits(dst_mode);
		int min_bits;
		ir_mode *min_mode;

		if (is_Load(skip_Proj(op))) {
			if (src_bits == dst_bits) {
				/* kill unneccessary conv */
				return new_op;
			}
			/* after a load, the bit size is already converted */
			src_bits = 32;
		}

		if (src_bits == dst_bits) {
			/* kill unneccessary conv */
			return new_op;
		} else if (dst_bits <= 32 && src_bits <= 32) {
			if (src_bits < dst_bits) {
				min_bits = src_bits;
				min_mode = src_mode;
			} else {
				min_bits = dst_bits;
				min_mode = dst_mode;
			}
			if (mode_is_signed(min_mode)) {
				return gen_sign_extension(dbg, block, new_op, min_bits);
			} else {
				return gen_zero_extension(dbg, block, new_op, min_bits);
			}
		} else {
			panic("Cannot handle Conv %+F->%+F with %d->%d bits", src_mode, dst_mode,
				src_bits, dst_bits);
			return NULL;
		}
	}
}

/**
 * Return true if an operand is a shifter operand
 */
static int is_shifter_operand(ir_node *n, arm_shift_modifier *pmod) {
	arm_shift_modifier mod = ARM_SHF_NONE;

	if (is_arm_Mov(n))
		mod = get_arm_shift_modifier(n);

	*pmod = mod;
	if (mod != ARM_SHF_NONE) {
		long v = get_arm_imm_value(n);
		if (v < 32)
			return (int)v;
	}
	return 0;
}

/**
 * Creates an ARM Add.
 *
 * @return the created arm Add node
 */
static ir_node *gen_Add(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Add_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Add_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = get_irn_mode(node);
	ir_node  *new_op3;
	int v;
	arm_shift_modifier mod;
	dbg_info *dbg = get_irn_dbg_info(node);

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa)) {
			if (is_arm_fpaMvf_i(new_op1))
				return new_bd_arm_fpaAdf_i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1));
			if (is_arm_fpaMvf_i(new_op2))
				return new_bd_arm_fpaAdf_i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2));
			return new_bd_arm_fpaAdf(dbg, block, new_op1, new_op2, mode);
		} else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
			return NULL;
		}
		else {
			panic("Softfloat not supported yet");
			return NULL;
		}
	} else {
		assert(mode_is_data(mode));
		mode = mode_Iu;

		if (is_arm_Mov_i(new_op1))
			return new_bd_arm_Add_i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1));
		if (is_arm_Mov_i(new_op2))
			return new_bd_arm_Add_i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2));

		/* check for MLA */
		if (is_arm_Mul(new_op1) && get_irn_n_edges(op1) == 1) {
			new_op3 = new_op2;
			new_op2 = get_irn_n(new_op1, 1);
			new_op1 = get_irn_n(new_op1, 0);

			return new_bd_arm_Mla(dbg, block, new_op1, new_op2, new_op3, mode);
		}
		if (is_arm_Mul(new_op2) && get_irn_n_edges(op2) == 1) {
			new_op3 = new_op1;
			new_op1 = get_irn_n(new_op2, 0);
			new_op2 = get_irn_n(new_op2, 1);

			return new_bd_arm_Mla(dbg, block, new_op1, new_op2, new_op3, mode);
		}

		/* is the first a shifter */
		v = is_shifter_operand(new_op1, &mod);
		if (v) {
			new_op1 = get_irn_n(new_op1, 0);
			return new_bd_arm_Add(dbg, block, new_op2, new_op1, mode, mod, v);
		}
		/* is the second a shifter */
		v = is_shifter_operand(new_op2, &mod);
		if (v) {
			new_op2 = get_irn_n(new_op2, 0);
			return new_bd_arm_Add(dbg, block, new_op1, new_op2, mode, mod, v);
		}

		/* normal ADD */
		return new_bd_arm_Add(dbg, block, new_op1, new_op2, mode, ARM_SHF_NONE, 0);
	}
}

/**
 * Creates an ARM Mul.
 *
 * @return the created arm Mul node
 */
static ir_node *gen_Mul(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Mul_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Mul_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = get_irn_mode(node);
	dbg_info *dbg     = get_irn_dbg_info(node);

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa)) {
			if (is_arm_Mov_i(new_op1))
				return new_bd_arm_fpaMuf_i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1));
			if (is_arm_Mov_i(new_op2))
				return new_bd_arm_fpaMuf_i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2));
			return new_bd_arm_fpaMuf(dbg, block, new_op1, new_op2, mode);
		}
		else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
			return NULL;
		}
		else {
			panic("Softfloat not supported yet");
			return NULL;
		}
	}
	assert(mode_is_data(mode));
	mode = mode_Iu;
	return new_bd_arm_Mul(dbg, block, new_op1, new_op2, mode);
}

/**
 * Creates an ARM floating point Div.
 *
 * @param env   The transformation environment
 * @return the created arm fDiv node
 */
static ir_node *gen_Quot(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Quot_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Quot_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = get_irn_mode(node);
	dbg_info *dbg     = get_irn_dbg_info(node);

	assert(mode != mode_E && "IEEE Extended FP not supported");

	env_cg->have_fp_insn = 1;
	if (USE_FPA(env_cg->isa)) {
		if (is_arm_Mov_i(new_op1))
			return new_bd_arm_fpaRdf_i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1));
		if (is_arm_Mov_i(new_op2))
			return new_bd_arm_fpaDvf_i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2));
		return new_bd_arm_fpaDvf(dbg, block, new_op1, new_op2, mode);
	} else if (USE_VFP(env_cg->isa)) {
		assert(mode != mode_E && "IEEE Extended FP not supported");
		panic("VFP not supported yet");
	}
	else {
		panic("Softfloat not supported yet");
		return NULL;
	}
}

#define GEN_INT_OP(op) \
	ir_node  *block   = be_transform_node(get_nodes_block(node)); \
	ir_node  *op1     = get_ ## op ## _left(node); \
	ir_node  *new_op1 = be_transform_node(op1); \
	ir_node  *op2     = get_ ## op ## _right(node); \
	ir_node  *new_op2 = be_transform_node(op2); \
	ir_mode  *mode    = mode_Iu; \
	dbg_info *dbg     = get_irn_dbg_info(node); \
	int      v; \
	arm_shift_modifier mod; \
 \
	if (is_arm_Mov_i(new_op1)) \
		return new_bd_arm_ ## op ## _i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1)); \
	if (is_arm_Mov_i(new_op2)) \
		return new_bd_arm_ ## op ## _i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2)); \
	/* is the first a shifter */ \
	v = is_shifter_operand(new_op1, &mod); \
	if (v) { \
		new_op1 = get_irn_n(new_op1, 0); \
		return new_bd_arm_ ## op(dbg, block, new_op2, new_op1, mode, mod, v); \
	} \
	/* is the second a shifter */ \
	v = is_shifter_operand(new_op2, &mod); \
	if (v) { \
		new_op2 = get_irn_n(new_op2, 0); \
		return new_bd_arm_ ## op(dbg, block, new_op1, new_op2, mode, mod, v); \
	} \
	/* Normal op */ \
	return new_bd_arm_ ## op(dbg, block, new_op1, new_op2, mode, ARM_SHF_NONE, 0) \

/**
 * Creates an ARM And.
 *
 * @return the created arm And node
 */
static ir_node *gen_And(ir_node *node) {
	GEN_INT_OP(And);
}

/**
 * Creates an ARM Orr.
 *
 * @param env   The transformation environment
 * @return the created arm Or node
 */
static ir_node *gen_Or(ir_node *node) {
	GEN_INT_OP(Or);
}

/**
 * Creates an ARM Eor.
 *
 * @return the created arm Eor node
 */
static ir_node *gen_Eor(ir_node *node) {
	GEN_INT_OP(Eor);
}

/**
 * Creates an ARM Sub.
 *
 * @return the created arm Sub node
 */
static ir_node *gen_Sub(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Sub_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Sub_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = get_irn_mode(node);
	dbg_info *dbg     = get_irn_dbg_info(node);
	int      v;
	arm_shift_modifier mod;

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa)) {
			if (is_arm_Mov_i(new_op1))
				return new_bd_arm_fpaRsf_i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1));
			if (is_arm_Mov_i(new_op2))
				return new_bd_arm_fpaSuf_i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2));
			return new_bd_arm_fpaSuf(dbg, block, new_op1, new_op2, mode);
		} else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
			return NULL;
		}
		else {
			panic("Softfloat not supported yet");
			return NULL;
		}
	}
	else {
		assert(mode_is_data(mode) && "unknown mode for Sub");
		mode = mode_Iu;

		if (is_arm_Mov_i(new_op1))
			return new_bd_arm_Rsb_i(dbg, block, new_op2, mode, get_arm_imm_value(new_op1));
		if (is_arm_Mov_i(new_op2))
			return new_bd_arm_Sub_i(dbg, block, new_op1, mode, get_arm_imm_value(new_op2));

		/* is the first a shifter */
		v = is_shifter_operand(new_op1, &mod);
		if (v) {
			new_op1 = get_irn_n(new_op1, 0);
			return new_bd_arm_Rsb(dbg, block, new_op2, new_op1, mode, mod, v);
		}
		/* is the second a shifter */
		v = is_shifter_operand(new_op2, &mod);
		if (v) {
			new_op2 = get_irn_n(new_op2, 0);
			return new_bd_arm_Sub(dbg, block, new_op1, new_op2, mode, mod, v);
		}
		/* normal sub */
		return new_bd_arm_Sub(dbg, block, new_op1, new_op2, mode, ARM_SHF_NONE, 0);
	}
}

/**
 * Creates an ARM Shl.
 *
 * @return the created ARM Shl node
 */
static ir_node *gen_Shl(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Shl_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Shl_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = mode_Iu;
	dbg_info *dbg     = get_irn_dbg_info(node);

	if (is_arm_Mov_i(new_op2)) {
		return new_bd_arm_Mov(dbg, block, new_op1, mode, ARM_SHF_LSL, get_arm_imm_value(new_op2));
	}
	return new_bd_arm_Shl(dbg, block, new_op1, new_op2, mode);
}

/**
 * Creates an ARM Shr.
 *
 * @return the created ARM Shr node
 */
static ir_node *gen_Shr(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Shr_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Shr_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = mode_Iu;
	dbg_info *dbg     = get_irn_dbg_info(node);

	if (is_arm_Mov_i(new_op2)) {
		return new_bd_arm_Mov(dbg, block, new_op1, mode, ARM_SHF_LSR, get_arm_imm_value(new_op2));
	}
	return new_bd_arm_Shr(dbg, block, new_op1, new_op2, mode);
}

/**
 * Creates an ARM Shrs.
 *
 * @return the created ARM Shrs node
 */
static ir_node *gen_Shrs(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op1     = get_Shrs_left(node);
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *op2     = get_Shrs_right(node);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = mode_Iu;
	dbg_info *dbg     = get_irn_dbg_info(node);

	if (is_arm_Mov_i(new_op2)) {
		return new_bd_arm_Mov(dbg, block, new_op1, mode, ARM_SHF_ASR, get_arm_imm_value(new_op2));
	}
	return new_bd_arm_Shrs(dbg, block, new_op1, new_op2, mode);
}

/**
 * Creates an ARM Ror.
 *
 * @return the created ARM Ror node
 */
static ir_node *gen_Ror(ir_node *node, ir_node *op1, ir_node *op2) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *new_op1 = be_transform_node(op1);
	ir_node  *new_op2 = be_transform_node(op2);
	ir_mode  *mode    = mode_Iu;
	dbg_info *dbg     = get_irn_dbg_info(node);

	if (is_arm_Mov_i(new_op2)) {
		return new_bd_arm_Mov(dbg, block, new_op1, mode, ARM_SHF_ROR, get_arm_imm_value(new_op2));
	}
	return new_bd_arm_Ror(dbg, block, new_op1, new_op2, mode);
}

/**
 * Creates an ARM Rol.
 *
 * @return the created ARM Rol node
 *
 * Note: there is no Rol on arm, we have to use Ror
 */
static ir_node *gen_Rol(ir_node *node, ir_node *op1, ir_node *op2) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *new_op1 = be_transform_node(op1);
	ir_mode  *mode    = mode_Iu;
	dbg_info *dbg     = get_irn_dbg_info(node);
	ir_node  *new_op2 = be_transform_node(op2);
	
	new_op2 = new_bd_arm_Rsb_i(dbg, block, new_op2, mode, 32);
	return new_bd_arm_Ror(dbg, block, new_op1, new_op2, mode);
}

/**
 * Creates an ARM ROR from a Firm Rotl.
 *
 * @return the created ARM Ror node
 */
static ir_node *gen_Rotl(ir_node *node) {
	ir_node *rotate = NULL;
	ir_node *op1    = get_Rotl_left(node);
	ir_node *op2    = get_Rotl_right(node);

	/* Firm has only RotL, so we are looking for a right (op2)
	   operand "-e+mode_size_bits" (it's an already modified "mode_size_bits-e",
	   that means we can create a RotR. */

	if (is_Add(op2)) {
		ir_node *right = get_Add_right(op2);
		if (is_Const(right)) {
			tarval  *tv   = get_Const_tarval(right);
			ir_mode *mode = get_irn_mode(node);
			long     bits = get_mode_size_bits(mode);
			ir_node *left = get_Add_left(op2);

			if (is_Minus(left) &&
			    tarval_is_long(tv)          &&
			    get_tarval_long(tv) == bits &&
			    bits                == 32)
				rotate = gen_Ror(node, op1, get_Minus_op(left));
		}
	} else if (is_Sub(op2)) {
		ir_node *left = get_Sub_left(op2);
		if (is_Const(left)) {
			tarval  *tv   = get_Const_tarval(left);
			ir_mode *mode = get_irn_mode(node);
			long     bits = get_mode_size_bits(mode);
			ir_node *right = get_Sub_right(op2);

			if (tarval_is_long(tv)          &&
			    get_tarval_long(tv) == bits &&
			    bits                == 32)
				rotate = gen_Ror(node, op1, right);
		}
	} else if (is_Const(op2)) {
			tarval  *tv   = get_Const_tarval(op2);
			ir_mode *mode = get_irn_mode(node);
			long     bits = get_mode_size_bits(mode);

			if (tarval_is_long(tv) && bits == 32) {
				ir_node  *block   = be_transform_node(get_nodes_block(node));
				ir_node  *new_op1 = be_transform_node(op1);
				ir_mode  *mode    = mode_Iu;
				dbg_info *dbg     = get_irn_dbg_info(node);

				bits = (bits - get_tarval_long(tv)) & 31;
				rotate = new_bd_arm_Mov(dbg, block, new_op1, mode, ARM_SHF_ROR, bits);
			}
	}

	if (rotate == NULL) {
		rotate = gen_Rol(node, op1, op2);
	}

	return rotate;
}

/**
 * Transforms a Not node.
 *
 * @return the created ARM Not node
 */
static ir_node *gen_Not(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op      = get_Not_op(node);
	ir_node  *new_op  = be_transform_node(op);
	dbg_info *dbg     = get_irn_dbg_info(node);
	ir_mode  *mode    = mode_Iu;
	arm_shift_modifier mod = ARM_SHF_NONE;
	int      v        = is_shifter_operand(new_op, &mod);

	if (v) {
		new_op = get_irn_n(new_op, 0);
	}
	return new_bd_arm_Mvn(dbg, block, new_op, mode, mod, v);
}

/**
 * Transforms an Abs node.
 *
 * @param env   The transformation environment
 * @return the created ARM Abs node
 */
static ir_node *gen_Abs(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op      = get_Abs_op(node);
	ir_node  *new_op  = be_transform_node(op);
	dbg_info *dbg     = get_irn_dbg_info(node);
	ir_mode  *mode    = get_irn_mode(node);

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa))
			return new_bd_arm_fpaAbs(dbg, block, new_op, mode);
		else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
		}
		else {
			panic("Softfloat not supported yet");
		}
	}
	assert(mode_is_data(mode));
	mode = mode_Iu;
	return new_bd_arm_Abs(dbg, block, new_op, mode);
}

/**
 * Transforms a Minus node.
 *
 * @return the created ARM Minus node
 */
static ir_node *gen_Minus(ir_node *node) {
	ir_node  *block   = be_transform_node(get_nodes_block(node));
	ir_node  *op      = get_Minus_op(node);
	ir_node  *new_op  = be_transform_node(op);
	dbg_info *dbg     = get_irn_dbg_info(node);
	ir_mode  *mode    = get_irn_mode(node);

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa))
			return new_bd_arm_fpaMvf(dbg, block, op, mode);
		else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
		}
		else {
			panic("Softfloat not supported yet");
		}
	}
	assert(mode_is_data(mode));
	mode = mode_Iu;
	return new_bd_arm_Rsb_i(dbg, block, new_op, mode, 0);
}

/**
 * Transforms a Load.
 *
 * @return the created ARM Load node
 */
static ir_node *gen_Load(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *ptr      = get_Load_ptr(node);
	ir_node  *new_ptr  = be_transform_node(ptr);
	ir_node  *mem      = get_Load_mem(node);
	ir_node  *new_mem  = be_transform_node(mem);
	ir_mode  *mode     = get_Load_mode(node);
	dbg_info *dbg      = get_irn_dbg_info(node);
	ir_node  *new_load = NULL;

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa))
			new_load = new_bd_arm_fpaLdf(dbg, block, new_ptr, new_mem, mode);
		else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
		}
		else {
			panic("Softfloat not supported yet");
		}
	}
	else {
		assert(mode_is_data(mode) && "unsupported mode for Load");

		if (mode_is_signed(mode)) {
			/* sign extended loads */
			switch (get_mode_size_bits(mode)) {
			case 8:
				new_load = new_bd_arm_Loadbs(dbg, block, new_ptr, new_mem);
				break;
			case 16:
				new_load = new_bd_arm_Loadhs(dbg, block, new_ptr, new_mem);
				break;
			case 32:
				new_load = new_bd_arm_Load(dbg, block, new_ptr, new_mem);
				break;
			default:
				panic("mode size not supported");
			}
		} else {
			/* zero extended loads */
			switch (get_mode_size_bits(mode)) {
			case 8:
				new_load = new_bd_arm_Loadb(dbg, block, new_ptr, new_mem);
				break;
			case 16:
				new_load = new_bd_arm_Loadh(dbg, block, new_ptr, new_mem);
				break;
			case 32:
				new_load = new_bd_arm_Load(dbg, block, new_ptr, new_mem);
				break;
			default:
				panic("mode size not supported");
			}
		}
	}
	set_irn_pinned(new_load, get_irn_pinned(node));

	/* check for special case: the loaded value might not be used */
	if (be_get_Proj_for_pn(node, pn_Load_res) == NULL) {
		ir_graph *irg = current_ir_graph;

		/* add a result proj and a Keep to produce a pseudo use */
		ir_node *proj = new_r_Proj(irg, block, new_load, mode_Iu, pn_arm_Load_res);
		be_new_Keep(arch_get_irn_reg_class_out(proj), irg, block, 1, &proj);
	}

	return new_load;
}

/**
 * Transforms a Store.
 *
 * @return the created ARM Store node
 */
static ir_node *gen_Store(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *ptr      = get_Store_ptr(node);
	ir_node  *new_ptr  = be_transform_node(ptr);
	ir_node  *mem      = get_Store_mem(node);
	ir_node  *new_mem  = be_transform_node(mem);
	ir_node  *val      = get_Store_value(node);
	ir_node  *new_val  = be_transform_node(val);
	ir_mode  *mode     = get_irn_mode(val);
	dbg_info *dbg      = get_irn_dbg_info(node);
	ir_node *new_store = NULL;

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa))
			new_store = new_bd_arm_fpaStf(dbg, block, new_ptr, new_val, new_mem, mode);
		else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
		} else {
			panic("Softfloat not supported yet");
		}
	} else {
		assert(mode_is_data(mode) && "unsupported mode for Store");
		switch (get_mode_size_bits(mode)) {
		case 8:
			new_store = new_bd_arm_Storeb(dbg, block, new_ptr, new_val, new_mem);
		case 16:
			new_store = new_bd_arm_Storeh(dbg, block, new_ptr, new_val, new_mem);
		default:
			new_store = new_bd_arm_Store(dbg, block, new_ptr, new_val, new_mem);
		}
	}
	set_irn_pinned(new_store, get_irn_pinned(node));
	return new_store;
}

/**
 * Transforms a Cond.
 *
 * @return the created ARM Cond node
 */
static ir_node *gen_Cond(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *selector = get_Cond_selector(node);
	dbg_info *dbg      = get_irn_dbg_info(node);
	ir_mode  *mode     = get_irn_mode(selector);

	if (mode == mode_b) {
		/* an conditional jump */
		ir_node *cmp_node = get_Proj_pred(selector);
		ir_node *op1      = get_Cmp_left(cmp_node);
		ir_node *new_op1  = be_transform_node(op1);
		ir_node *op2      = get_Cmp_right(cmp_node);

		if (mode_is_float(get_irn_mode(op1))) {
			ir_node *new_op2  = be_transform_node(op2);
			/* floating point compare */
			pn_Cmp pnc = get_Proj_proj(selector);

			if (pnc & pn_Cmp_Uo) {
				/* check for unordered, need cmf */
				return new_bd_arm_fpaCmfBra(dbg, block, new_op1, new_op2, pnc);
			}
			/* Hmm: use need cmfe */
			return new_bd_arm_fpaCmfeBra(dbg, block, new_op1, new_op2, pnc);
		} else if (is_Const(op2) && tarval_is_null(get_Const_tarval(op2))) {
			/* compare with 0 */
			return new_bd_arm_TstBra(dbg, block, new_op1, new_op1, get_Proj_proj(selector));
		} else {
			/* integer compare */
			ir_node *new_op2  = be_transform_node(op2);
			return new_bd_arm_CmpBra(dbg, block, new_op1, new_op2, get_Proj_proj(selector));
		}
	} else {
		/* SwitchJmp */
		ir_node *new_op = be_transform_node(selector);
		ir_node *const_graph;
		ir_node *sub;

		ir_node *proj;
		const ir_edge_t *edge;
		int min = INT_MAX;
		int max = INT_MIN;
		int translation;
		int pn;
		int n_projs;

		foreach_out_edge(node, edge) {
			proj = get_edge_src_irn(edge);
			assert(is_Proj(proj) && "Only proj allowed at SwitchJmp");

			pn = get_Proj_proj(proj);

			min = pn<min ? pn : min;
			max = pn>max ? pn : max;
		}
		translation = min;
		n_projs = max - translation + 1;

		foreach_out_edge(node, edge) {
			proj = get_edge_src_irn(edge);
			assert(is_Proj(proj) && "Only proj allowed at SwitchJmp");

			pn = get_Proj_proj(proj) - translation;
			set_Proj_proj(proj, pn);
		}

		const_graph = create_const_graph_value(dbg, block, translation);
		sub = new_bd_arm_Sub(dbg, block, new_op, const_graph, mode, ARM_SHF_NONE, 0);
		return new_bd_arm_SwitchJmp(dbg, block, sub, n_projs, get_Cond_default_proj(node) - translation);
	}
}

/**
 * Returns the name of a SymConst.
 * @param symc  the SymConst
 * @return name of the SymConst
 */
static ident *get_sc_ident(ir_node *symc) {
	ir_entity *ent;

	switch (get_SymConst_kind(symc)) {
		case symconst_addr_name:
			return get_SymConst_name(symc);

		case symconst_addr_ent:
			ent = get_SymConst_entity(symc);
			set_entity_backend_marked(ent, 1);
			return get_entity_ld_ident(ent);

		default:
			assert(0 && "Unsupported SymConst");
	}

	return NULL;
}

static tarval *fpa_imm[3][fpa_max];

/**
 * Check, if a floating point tarval is an fpa immediate, i.e.
 * one of 0, 1, 2, 3, 4, 5, 10, or 0.5.
 */
static int is_fpa_immediate(tarval *tv) {
	ir_mode *mode = get_tarval_mode(tv);
	int i, j, res = 1;

	switch (get_mode_size_bits(mode)) {
	case 32:
		i = 0;
		break;
	case 64:
		i = 1;
		break;
	default:
		i = 2;
	}

	if (tarval_cmp(tv, get_tarval_null(mode)) & pn_Cmp_Lt) {
		tv = tarval_neg(tv);
		res = -1;
	}

	for (j = 0; j < fpa_max; ++j) {
		if (tv == fpa_imm[i][j])
			return res * j;
	}
	return fpa_max;
}

/**
 * Transforms a Const node.
 *
 * @return The transformed ARM node.
 */
static ir_node *gen_Const(ir_node *node) {
	ir_node  *block = be_transform_node(get_nodes_block(node));
	ir_mode *mode = get_irn_mode(node);
	dbg_info *dbg = get_irn_dbg_info(node);

	if (mode_is_float(mode)) {
		env_cg->have_fp_insn = 1;
		if (USE_FPA(env_cg->isa)) {
			tarval *tv = get_Const_tarval(node);
			int imm = is_fpa_immediate(tv);

			if (imm != fpa_max) {
				if (imm > 0)
					node = new_bd_arm_fpaMvf_i(dbg, block, mode, imm);
				else
					node = new_bd_arm_fpaMnf_i(dbg, block, mode, -imm);
			} else {
				node = new_bd_arm_fpaConst(dbg, block, tv);
			}
			be_dep_on_frame(node);
			return node;
		}
		else if (USE_VFP(env_cg->isa)) {
			assert(mode != mode_E && "IEEE Extended FP not supported");
			panic("VFP not supported yet");
		}
		else {
			panic("Softfloat not supported yet");
		}
	}
	return create_const_graph(node, block);
}

/**
 * Transforms a SymConst node.
 *
 * @return The transformed ARM node.
 */
static ir_node *gen_SymConst(ir_node *node) {
	ir_node  *block = be_transform_node(get_nodes_block(node));
	ir_mode  *mode  = mode_Iu;
	dbg_info *dbg   = get_irn_dbg_info(node);
	ir_node  *res;

	res = new_bd_arm_SymConst(dbg, block, mode, get_sc_ident(node));
	be_dep_on_frame(res);
	return res;
}

/**
 * Transforms a CopyB node.
 *
 * @return The transformed ARM node.
 */
static ir_node *gen_CopyB(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *src      = get_CopyB_src(node);
	ir_node  *new_src  = be_transform_node(src);
	ir_node  *dst      = get_CopyB_dst(node);
	ir_node  *new_dst  = be_transform_node(dst);
	ir_node  *mem      = get_CopyB_mem(node);
	ir_node  *new_mem  = be_transform_node(mem);
	ir_graph *irg      = current_ir_graph;
	dbg_info *dbg      = get_irn_dbg_info(node);
	int      size      = get_type_size_bytes(get_CopyB_type(node));
	ir_node  *src_copy;
	ir_node  *dst_copy;

	src_copy = be_new_Copy(&arm_reg_classes[CLASS_arm_gp], irg, block, new_src);
	dst_copy = be_new_Copy(&arm_reg_classes[CLASS_arm_gp], irg, block, new_dst);

 	return new_bd_arm_CopyB(dbg, block, dst_copy, src_copy,
			new_bd_arm_EmptyReg(dbg, block, mode_Iu),
			new_bd_arm_EmptyReg(dbg, block, mode_Iu),
			new_bd_arm_EmptyReg(dbg, block, mode_Iu),
			new_mem, size);
}


/********************************************
 *  _                          _
 * | |                        | |
 * | |__   ___ _ __   ___   __| | ___  ___
 * | '_ \ / _ \ '_ \ / _ \ / _` |/ _ \/ __|
 * | |_) |  __/ | | | (_) | (_| |  __/\__ \
 * |_.__/ \___|_| |_|\___/ \__,_|\___||___/
 *
 ********************************************/

/**
 * Return an expanding stack offset.
 * Note that function is called in the transform phase
 * where the stack offsets are still relative regarding
 * the first (frame allocating) IncSP.
 * However this is exactly what we want because frame
 * access must be done relative the the fist IncSP ...
 */
static int get_sp_expand_offset(ir_node *inc_sp) {
	int offset = be_get_IncSP_offset(inc_sp);

	if (offset == BE_STACK_FRAME_SIZE_EXPAND)
		return 0;

	return offset;
}

#if 0
static ir_node *gen_StackParam(ir_node *irn) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node   *new_op = NULL;
	ir_node   *noreg  = ia32_new_NoReg_gp(env->cg);
	ir_node   *mem    = new_NoMem();
	ir_node   *ptr    = get_irn_n(irn, 0);
	ir_entity *ent    = be_get_frame_entity(irn);
	ir_mode   *mode   = env->mode;

//	/* If the StackParam has only one user ->     */
//	/* put it in the Block where the user resides */
//	if (get_irn_n_edges(node) == 1) {
//		env->block = get_nodes_block(get_edge_src_irn(get_irn_out_edge_first(node)));
//	}

	if (mode_is_float(mode)) {
		if (USE_SSE2(env->cg))
			new_op = new_rd_ia32_fLoad(env->dbg, env->irg, block, ptr, noreg, mem, mode_T);
		else {
			env->cg->used_x87 = 1;
			new_op = new_rd_ia32_vfld(env->dbg, env->irg, block, ptr, noreg, mem, mode_T);
		}
	}
	else {
		new_op = new_rd_ia32_Load(env->dbg, env->irg, block, ptr, noreg, mem, mode_T);
	}

	set_ia32_frame_ent(new_op, ent);
	set_ia32_use_frame(new_op);

	set_ia32_am_support(new_op, ia32_am_Source);
	set_ia32_op_type(new_op, ia32_AddrModeS);
	set_ia32_am_flavour(new_op, ia32_B);
	set_ia32_ls_mode(new_op, mode);

	SET_IA32_ORIG_NODE(new_op, ia32_get_old_node_name(env->cg, env->irn));

	return new_rd_Proj(env->dbg, env->irg, block, new_op, mode, 0);
}
#endif

/**
 * Transforms a FrameAddr into an ARM Add.
 */
static ir_node *gen_be_FrameAddr(ir_node *node) {
	ir_node   *block  = be_transform_node(get_nodes_block(node));
	ir_entity *ent    = be_get_frame_entity(node);
	int       offset  = get_entity_offset(ent);
	ir_node   *op     = be_get_FrameAddr_frame(node);
	ir_node   *new_op = be_transform_node(op);
	dbg_info  *dbg    = get_irn_dbg_info(node);
	ir_mode   *mode   = mode_Iu;
	ir_node   *cnst;

	if (be_is_IncSP(op)) {
		/* BEWARE: we get an offset which is absolute from an offset that
		   is relative. Both must be merged */
		offset += get_sp_expand_offset(op);
	}
	cnst = create_const_graph_value(dbg, block, (unsigned)offset);
	if (is_arm_Mov_i(cnst))
		return new_bd_arm_Add_i(dbg, block, new_op, mode, get_arm_imm_value(cnst));
	return new_bd_arm_Add(dbg, block, new_op, cnst, mode, ARM_SHF_NONE, 0);
}

/**
 * Transform a be_AddSP into an arm_AddSP. Eat up const sizes.
 */
static ir_node *gen_be_AddSP(ir_node *node) {
	ir_node  *block  = be_transform_node(get_nodes_block(node));
	ir_node  *sz     = get_irn_n(node, be_pos_AddSP_size);
	ir_node  *new_sz = be_transform_node(sz);
	ir_node  *sp     = get_irn_n(node, be_pos_AddSP_old_sp);
	ir_node  *new_sp = be_transform_node(sp);
	dbg_info *dbgi   = get_irn_dbg_info(node);
	ir_node  *nomem  = new_NoMem();
	ir_node  *new_op;

	/* ARM stack grows in reverse direction, make a SubSPandCopy */
	new_op = new_bd_arm_SubSPandCopy(dbgi, block, new_sp, new_sz, nomem);

	return new_op;
}

/**
 * Transform a be_SubSP into an arm_SubSP. Eat up const sizes.
 */
static ir_node *gen_be_SubSP(ir_node *node) {
	ir_node  *block  = be_transform_node(get_nodes_block(node));
	ir_node  *sz     = get_irn_n(node, be_pos_SubSP_size);
	ir_node  *new_sz = be_transform_node(sz);
	ir_node  *sp     = get_irn_n(node, be_pos_SubSP_old_sp);
	ir_node  *new_sp = be_transform_node(sp);
	dbg_info *dbgi   = get_irn_dbg_info(node);
	ir_node  *nomem  = new_NoMem();
	ir_node  *new_op;

	/* ARM stack grows in reverse direction, make an AddSP */
	new_op = new_bd_arm_AddSP(dbgi, block, new_sp, new_sz, nomem);

	return new_op;
}

/**
 * Transform a be_Copy.
 */
static ir_node *gen_be_Copy(ir_node *node) {
	ir_node *result = be_duplicate_node(node);
	ir_mode *mode   = get_irn_mode(result);

	if (mode_needs_gp_reg(mode)) {
		set_irn_mode(node, mode_Iu);
	}

	return result;
}

/**
 * Transform a Proj from a Load.
 */
static ir_node *gen_Proj_Load(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *load     = get_Proj_pred(node);
	ir_node  *new_load = be_transform_node(load);
	ir_graph *irg      = current_ir_graph;
	dbg_info *dbgi     = get_irn_dbg_info(node);
	long     proj      = get_Proj_proj(node);

	/* renumber the proj */
	switch (get_arm_irn_opcode(new_load)) {
	case iro_arm_Load:
	case iro_arm_Loadb:
	case iro_arm_Loadbs:
	case iro_arm_Loadh:
	case iro_arm_Loadhs:
		/* handle all gp loads equal: they have the same proj numbers. */
		if (proj == pn_Load_res) {
			return new_rd_Proj(dbgi, irg, block, new_load, mode_Iu, pn_arm_Load_res);
		} else if (proj == pn_Load_M) {
			return new_rd_Proj(dbgi, irg, block, new_load, mode_M, pn_arm_Load_M);
		}
		break;
	case iro_arm_fpaLdf:
		if (proj == pn_Load_res) {
			ir_mode *mode = get_Load_mode(load);
			return new_rd_Proj(dbgi, irg, block, new_load, mode, pn_arm_fpaLdf_res);
		} else if (proj == pn_Load_M) {
			return new_rd_Proj(dbgi, irg, block, new_load, mode_M, pn_arm_fpaLdf_M);
		}
		break;
	default:
		break;
	}
	panic("Unsupported Proj from Load");
}

/**
 * Transform and renumber the Projs from a CopyB.
 */
static ir_node *gen_Proj_CopyB(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *pred     = get_Proj_pred(node);
	ir_node  *new_pred = be_transform_node(pred);
	ir_graph *irg      = current_ir_graph;
	dbg_info *dbgi     = get_irn_dbg_info(node);
	long     proj      = get_Proj_proj(node);

	switch(proj) {
	case pn_CopyB_M_regular:
		if (is_arm_CopyB(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_CopyB_M);
		}
		break;
	default:
		break;
	}
	panic("Unsupported Proj from CopyB");
}

/**
 * Transform and renumber the Projs from a Quot.
 */
static ir_node *gen_Proj_Quot(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *pred     = get_Proj_pred(node);
	ir_node  *new_pred = be_transform_node(pred);
	ir_graph *irg      = current_ir_graph;
	dbg_info *dbgi     = get_irn_dbg_info(node);
	ir_mode  *mode     = get_irn_mode(node);
	long     proj      = get_Proj_proj(node);

	switch (proj) {
	case pn_Quot_M:
		if (is_arm_fpaDvf(new_pred) || is_arm_fpaDvf_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_fpaDvf_M);
		} else if (is_arm_fpaRdf(new_pred) || is_arm_fpaRdf_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_fpaRdf_M);
		} else if (is_arm_fpaFdv(new_pred) || is_arm_fpaFdv_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_fpaFdv_M);
		} else if (is_arm_fpaFrd(new_pred) || is_arm_fpaFrd_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_fpaFrd_M);
		}
		break;
	case pn_Quot_res:
		if (is_arm_fpaDvf(new_pred) || is_arm_fpaDvf_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode, pn_arm_fpaDvf_res);
		} else if (is_arm_fpaRdf(new_pred) || is_arm_fpaRdf_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode, pn_arm_fpaRdf_res);
		} else if (is_arm_fpaFdv(new_pred) || is_arm_fpaFdv_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode, pn_arm_fpaFdv_res);
		} else if (is_arm_fpaFrd(new_pred) || is_arm_fpaFrd_i(new_pred)) {
			return new_rd_Proj(dbgi, irg, block, new_pred, mode, pn_arm_fpaFrd_res);
		}
		break;
	default:
		break;
	}
	panic("Unsupported Proj from Quot");
}

/**
 * Transform the Projs of a be_AddSP.
 */
static ir_node *gen_Proj_be_AddSP(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *pred     = get_Proj_pred(node);
	ir_node  *new_pred = be_transform_node(pred);
	ir_graph *irg      = current_ir_graph;
	dbg_info *dbgi     = get_irn_dbg_info(node);
	long     proj      = get_Proj_proj(node);

	if (proj == pn_be_AddSP_sp) {
		ir_node *res = new_rd_Proj(dbgi, irg, block, new_pred, mode_Iu,
		                           pn_arm_SubSPandCopy_stack);
		arch_set_irn_register(res, &arm_gp_regs[REG_SP]);
		return res;
	} else if(proj == pn_be_AddSP_res) {
		return new_rd_Proj(dbgi, irg, block, new_pred, mode_Iu,
		                   pn_arm_SubSPandCopy_addr);
	} else if (proj == pn_be_AddSP_M) {
		return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_SubSPandCopy_M);
	}
	panic("Unsupported Proj from AddSP");
}

/**
 * Transform the Projs of a be_SubSP.
 */
static ir_node *gen_Proj_be_SubSP(ir_node *node) {
	ir_node  *block    = be_transform_node(get_nodes_block(node));
	ir_node  *pred     = get_Proj_pred(node);
	ir_node  *new_pred = be_transform_node(pred);
	ir_graph *irg      = current_ir_graph;
	dbg_info *dbgi     = get_irn_dbg_info(node);
	long     proj      = get_Proj_proj(node);

	if (proj == pn_be_SubSP_sp) {
		ir_node *res = new_rd_Proj(dbgi, irg, block, new_pred, mode_Iu,
		                           pn_arm_AddSP_stack);
		arch_set_irn_register(res, &arm_gp_regs[REG_SP]);
		return res;
	} else if (proj == pn_be_SubSP_M) {
		return new_rd_Proj(dbgi, irg, block, new_pred, mode_M, pn_arm_AddSP_M);
	}
	panic("Unsupported Proj from SubSP");
}

/**
 * Transform the Projs from a Cmp.
 */
static ir_node *gen_Proj_Cmp(ir_node *node) {
	(void) node;
	panic("Mux NYI");
}


/**
 * Transform the Thread Local Storage Proj.
 */
static ir_node *gen_Proj_tls(ir_node *node) {
	ir_node  *block = be_transform_node(get_nodes_block(node));
	dbg_info *dbgi  = NULL;

	return new_bd_arm_LdTls(dbgi, block, mode_Iu);
}

/**
 * Transform a Proj node.
 */
static ir_node *gen_Proj(ir_node *node) {
	ir_graph *irg  = current_ir_graph;
	dbg_info *dbgi = get_irn_dbg_info(node);
	ir_node  *pred = get_Proj_pred(node);
	long     proj  = get_Proj_proj(node);

	if (is_Store(pred)) {
		if (proj == pn_Store_M) {
			return be_transform_node(pred);
		} else {
			panic("Unsupported Proj from Store");
		}
	} else if (is_Load(pred)) {
		return gen_Proj_Load(node);
	} else if (is_CopyB(pred)) {
		return gen_Proj_CopyB(node);
	} else if (is_Quot(pred)) {
		return gen_Proj_Quot(node);
	} else if (be_is_SubSP(pred)) {
		return gen_Proj_be_SubSP(node);
	} else if (be_is_AddSP(pred)) {
		return gen_Proj_be_AddSP(node);
	} else if (is_Cmp(pred)) {
		return gen_Proj_Cmp(node);
	} else if (is_Start(pred)) {
		if (proj == pn_Start_X_initial_exec) {
			ir_node *block = get_nodes_block(pred);
			ir_node *jump;

			/* we exchange the ProjX with a jump */
			block = be_transform_node(block);
			jump  = new_rd_Jmp(dbgi, irg, block);
			return jump;
		}
		if (node == get_irg_anchor(irg, anchor_tls)) {
			return gen_Proj_tls(node);
		}
	} else {
		ir_node *new_pred = be_transform_node(pred);
		ir_mode *mode     = get_irn_mode(node);
		if (mode_needs_gp_reg(mode)) {
			ir_node *block    = be_transform_node(get_nodes_block(node));
			ir_node *new_proj = new_r_Proj(irg, block, new_pred, mode_Iu,
			                               get_Proj_proj(node));
			new_proj->node_nr = node->node_nr;
			return new_proj;
		}
	}

	return be_duplicate_node(node);
}

typedef ir_node *(*create_const_node_func)(dbg_info *db, ir_node *block);

static inline ir_node *create_const(ir_node **place,
                                    create_const_node_func func,
                                    const arch_register_t* reg)
{
	ir_node *block, *res;

	if (*place != NULL)
		return *place;

	block = get_irg_start_block(env_cg->irg);
	res = func(NULL, block);
	arch_set_irn_register(res, reg);
	*place = res;
	return res;
}

static ir_node *arm_new_Unknown_gp(void) {
	return create_const(&env_cg->unknown_gp, new_bd_arm_Unknown_GP,
	                    &arm_gp_regs[REG_GP_UKNWN]);
}

static ir_node *arm_new_Unknown_fpa(void) {
	return create_const(&env_cg->unknown_fpa, new_bd_arm_Unknown_FPA,
	                    &arm_fpa_regs[REG_FPA_UKNWN]);
}

/**
 * This function just sets the register for the Unknown node
 * as this is not done during register allocation because Unknown
 * is an "ignore" node.
 */
static ir_node *gen_Unknown(ir_node *node) {
	ir_mode *mode = get_irn_mode(node);
	if (mode_is_float(mode)) {
		if (USE_FPA(env_cg->isa))
			return arm_new_Unknown_fpa();
		else if (USE_VFP(env_cg->isa))
			panic("VFP not supported yet");
		else
			panic("Softfloat not supported yet");
	} else if (mode_needs_gp_reg(mode)) {
		return arm_new_Unknown_gp();
	} else {
		assert(0 && "unsupported Unknown-Mode");
	}

	return NULL;
}

/**
 * Change some phi modes
 */
static ir_node *gen_Phi(ir_node *node) {
	ir_node  *block = be_transform_node(get_nodes_block(node));
	ir_graph *irg   = current_ir_graph;
	dbg_info *dbgi  = get_irn_dbg_info(node);
	ir_mode  *mode  = get_irn_mode(node);
	ir_node  *phi;

	if (mode_needs_gp_reg(mode)) {
		/* we shouldn't have any 64bit stuff around anymore */
		assert(get_mode_size_bits(mode) <= 32);
		/* all integer operations are on 32bit registers now */
		mode = mode_Iu;
	}

	/* phi nodes allow loops, so we use the old arguments for now
	 * and fix this later */
	phi = new_ir_node(dbgi, irg, block, op_Phi, mode, get_irn_arity(node), get_irn_in(node) + 1);
	copy_node_attr(node, phi);
	be_duplicate_deps(node, phi);

	be_enqueue_preds(node);

	return phi;
}

/*********************************************************
 *                  _             _      _
 *                 (_)           | |    (_)
 *  _ __ ___   __ _ _ _ __     __| |_ __ ___   _____ _ __
 * | '_ ` _ \ / _` | | '_ \   / _` | '__| \ \ / / _ \ '__|
 * | | | | | | (_| | | | | | | (_| | |  | |\ V /  __/ |
 * |_| |_| |_|\__,_|_|_| |_|  \__,_|_|  |_| \_/ \___|_|
 *
 *********************************************************/

/**
 * the BAD transformer.
 */
static ir_node *bad_transform(ir_node *irn) {
	panic("ARM backend: Not implemented: %+F", irn);
	return irn;
}

/**
 * Set a node emitter. Make it a bit more type safe.
 */
static inline void set_transformer(ir_op *op, be_transform_func arm_transform_func) {
	op->ops.generic = (op_func)arm_transform_func;
}

/**
 * Enters all transform functions into the generic pointer
 */
static void arm_register_transformers(void) {
	/* first clear the generic function pointer for all ops */
	clear_irp_opcodes_generic_func();

#define GEN(a)     set_transformer(op_##a, gen_##a)
#define BAD(a)     set_transformer(op_##a, bad_transform)

	GEN(Add);
	GEN(Sub);
	GEN(Mul);
	BAD(Mulh);	/* unsupported yet */
	GEN(And);
	GEN(Or);
	GEN(Eor);

	GEN(Shl);
	GEN(Shr);
	GEN(Shrs);
	GEN(Rotl);

	GEN(Quot);

	/* should be lowered */
	BAD(Div);
	BAD(Mod);
	BAD(DivMod);

	GEN(Minus);
	GEN(Conv);
	GEN(Abs);
	GEN(Not);

	GEN(Load);
	GEN(Store);
	GEN(Cond);

	BAD(ASM);	/* unsupported yet */
	GEN(CopyB);
	BAD(Mux);	/* unsupported yet */
	GEN(Proj);
	GEN(Phi);

	GEN(Const);
	GEN(SymConst);

	/* we should never see these nodes */
	BAD(Raise);
	BAD(Sel);
	BAD(InstOf);
	BAD(Cast);
	BAD(Free);
	BAD(Tuple);
	BAD(Id);
	//BAD(Bad);
	BAD(Confirm);
	BAD(Filter);
	BAD(CallBegin);
	BAD(EndReg);
	BAD(EndExcept);

	/* handle builtins */
	BAD(Builtin);

	/* handle generic backend nodes */
	GEN(be_FrameAddr);
	//GEN(be_Call);
	//GEN(be_Return);
	GEN(be_AddSP);
	GEN(be_SubSP);
	GEN(be_Copy);

	/* set the register for all Unknown nodes */
	GEN(Unknown);

#undef GEN
#undef BAD
}

/**
 * Pre-transform all unknown nodes.
 */
static void arm_pretransform_node(void)
{
	arm_code_gen_t *cg = env_cg;

	cg->unknown_gp  = be_pre_transform_node(cg->unknown_gp);
	cg->unknown_fpa = be_pre_transform_node(cg->unknown_fpa);
}

/**
 * Initialize fpa Immediate support.
 */
static void arm_init_fpa_immediate(void) {
	/* 0, 1, 2, 3, 4, 5, 10, or 0.5. */
	fpa_imm[0][fpa_null]  = get_tarval_null(mode_F);
	fpa_imm[0][fpa_one]   = get_tarval_one(mode_F);
	fpa_imm[0][fpa_two]   = new_tarval_from_str("2", 1, mode_F);
	fpa_imm[0][fpa_three] = new_tarval_from_str("3", 1, mode_F);
	fpa_imm[0][fpa_four]  = new_tarval_from_str("4", 1, mode_F);
	fpa_imm[0][fpa_five]  = new_tarval_from_str("5", 1, mode_F);
	fpa_imm[0][fpa_ten]   = new_tarval_from_str("10", 2, mode_F);
	fpa_imm[0][fpa_half]  = new_tarval_from_str("0.5", 3, mode_F);

	fpa_imm[1][fpa_null]  = get_tarval_null(mode_D);
	fpa_imm[1][fpa_one]   = get_tarval_one(mode_D);
	fpa_imm[1][fpa_two]   = new_tarval_from_str("2", 1, mode_D);
	fpa_imm[1][fpa_three] = new_tarval_from_str("3", 1, mode_D);
	fpa_imm[1][fpa_four]  = new_tarval_from_str("4", 1, mode_D);
	fpa_imm[1][fpa_five]  = new_tarval_from_str("5", 1, mode_D);
	fpa_imm[1][fpa_ten]   = new_tarval_from_str("10", 2, mode_D);
	fpa_imm[1][fpa_half]  = new_tarval_from_str("0.5", 3, mode_D);

	fpa_imm[2][fpa_null]  = get_tarval_null(mode_E);
	fpa_imm[2][fpa_one]   = get_tarval_one(mode_E);
	fpa_imm[2][fpa_two]   = new_tarval_from_str("2", 1, mode_E);
	fpa_imm[2][fpa_three] = new_tarval_from_str("3", 1, mode_E);
	fpa_imm[2][fpa_four]  = new_tarval_from_str("4", 1, mode_E);
	fpa_imm[2][fpa_five]  = new_tarval_from_str("5", 1, mode_E);
	fpa_imm[2][fpa_ten]   = new_tarval_from_str("10", 2, mode_E);
	fpa_imm[2][fpa_half]  = new_tarval_from_str("0.5", 3, mode_E);
}

/**
 * Transform a Firm graph into an ARM graph.
 */
void arm_transform_graph(arm_code_gen_t *cg) {
	static int imm_initialized = 0;

	if (! imm_initialized) {
		arm_init_fpa_immediate();
		imm_initialized = 1;
	}
	arm_register_transformers();
	env_cg = cg;
	be_transform_graph(cg->birg, arm_pretransform_node);
}

void arm_init_transform(void) {
	// FIRM_DBG_REGISTER(dbg, "firm.be.arm.transform");
}
