/**
 * @file libcomprex/utils.c Utility functions
 * 
 * $Id: utils.c,v 1.21 2003/01/01 06:22:36 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>
#include <libcomprex/utils.h>

void
cxSplitPath(const char *path, char **dir, char **basename)
{
	char *c;

	if (path == NULL || (dir == NULL && basename == NULL))
		return;

	if (dir      != NULL) *dir      = NULL;
	if (basename != NULL) *basename = NULL;

	c = strrchr(path, '/');

	if (c == NULL)
	{
		if (basename != NULL)
			*basename = strdup(path);
	}
	else
	{
		if (dir != NULL)
		{
			if (c - path == 0)
			{
				*dir = strdup("/");
			}
			else
			{
				MEM_CHECK(*dir = (char *)malloc(c - path + 1));

				strncpy(*dir, path, c - path);

				(*dir)[c - path] = '\0';
			}
		}

		if (basename != NULL)
		{
			if (*(c + 1) != '\0')
				*basename = strdup(c + 1);
		}
	}
}

char *
cxGetBaseName(const char *path)
{
	char *basename;

	cxSplitPath(path, NULL, &basename);

	return basename;
}

char *
cxGetBasePath(const char *path)
{
	char *dir;

	cxSplitPath(path, &dir, NULL);

	return dir;
}

char *
cxFixPath(const char *path)
{
	char *buffer, *result;
	const char *c;
	char *d;

	/* The buffer will never be any larger than the path. */
	MEM_CHECK(buffer = (char *)malloc(strlen(path) + 1));

	*buffer = '\0';

	for (c = path, d = buffer; *c != '\0'; c++, d++)
	{
		if (*c == '.')
		{
			if (*(c + 1) == '.' && *(c + 2) == '/')
			{
				if (d != buffer)
				{
					*(d - 1) = '\0';
					d = strrchr(buffer, '/');
				}

				c += 2;

				continue;
			}
			else if (*(c + 1) == '/')
			{
				c++;
				
				continue;
			}
		}
		else if (*c == '/')
		{
			while (*(c + 1) == '/')
				c++;
		}

		*d = *c;
	}

	*d = '\0';

	result = strdup(buffer);

	free(buffer);
	
	return result;
}

/* XXX */
#ifndef PATH_MAX
# define PATH_MAX 4096
#endif

char *
cxGetFullFilePath(const char *filename)
{
	char *result;

	if (filename == NULL || *filename == '\0')
		return NULL;

	if (*filename == '/')
	{
		result = cxFixPath(filename);
	}
	else
	{
		char *buffer;
		char cwd[PATH_MAX];

		/* Get the current working directory. */
		getcwd(cwd, PATH_MAX);

		/* Buffer will never be any larger than filename + cwd + 2 */
		MEM_CHECK(buffer = (char *)malloc(strlen(cwd) + 1 +
										  strlen(filename) + 1));

		strcpy(buffer, cwd);
		strcat(buffer, "/");
		strcat(buffer, filename);

		result = cxFixPath(buffer);

		free(buffer);
	}

	return result;
}

void
cxMakePhysDirs(const char *path, CxArchive *archive)
{
	char *c;
	char *buffer;
	struct stat sb;
	char createDir;
	CxDirectory *lastDir = NULL;
	
	if (path == NULL || *path == '\0')
		return;

	if (archive != NULL)
		lastDir = cxGetArchiveRoot(archive);
	
	buffer = strdup(path);
	c = buffer;

	for (;;)
	{
		if (*c == '/' || *c == '\0')
		{
			char endPath = (*c == '\0');

			createDir = 0;
			*c = '\0';
			
			if (stat(buffer, &sb) == 0)
			{
				/*
				 * XXX This will nuke a file if it exists with the same name
				 *     as the directory! Tar has this behavior, so I'm just
				 *     doing the same. Blame tar. :)
				 */
				if (!S_ISDIR(sb.st_mode))
				{
					unlink(buffer);

					createDir = 1;
				}
			}
			else
				createDir = 1;

			if (createDir)
			{
				mkdir(buffer, 0755);

				if (lastDir != NULL)
				{
					char *basename = cxGetBaseName(buffer);
					lastDir = cxGetDirectory(lastDir, basename);
					free(basename);

					cxApplyFsNodeInfo(lastDir, buffer);
				}
			}

			if (endPath)
				break;

			*c = '/';
		}

		c++;
	}
}

void
cxApplyFsNodeInfo(CxFsNode *node, const char *path)
{
	struct utimbuf stamp;
	mode_t mode;

	if (node == NULL || path == NULL)
		return;

	mode = cxGetFsNodeMode(node);

	stamp.actime = cxGetFsNodeDate(node); /* XXX */
	stamp.modtime= cxGetFsNodeDate(node);

	if (mode != 0)
		chmod(path, mode);

	chown(path, cxGetFsNodeUid(node), cxGetFsNodeGid(node));
	utime(path, &stamp);
}
