/**
 * @file libcomprex/internal.c Internal functions
 * 
 * $Id: internal.c,v 1.28 2003/01/01 06:22:36 chipx86 Exp $
 *
 * @Copyright (C) 2001-2003 The GNUpdate Project.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libcomprex/internal.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
# ifdef HAVE_PWD_H
#  include <pwd.h>
# endif
#endif

#ifdef HAVE_DIRENT_H
# include <dirent.h>
# define NAMELEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMELEN(dirent) (dirent)->d_namelen
# ifdef HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# ifdef HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# ifdef HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

static char *__tempDir  = NULL;
static char *__homeDir  = NULL;
static char *__userName = NULL;
static char *__realName = NULL;

/*
 * This scans for the system's temp directory, the user's home directory,
 * user name, and user's real name.
 * 
 * This was borrowed from glib/gutils.c. Thanks guys! :)
 */
static void
__getEnvInfo()
{
	char *temp;
	
	if (__tempDir == NULL)
	{
		temp = getenv("TMPDIR");

		if (temp == NULL)
			temp = getenv("TMP");

		if (temp == NULL)
			temp = getenv("TEMP");

		if (temp != NULL)
		{
			__tempDir = strdup(temp);
		}
		else
		{
#ifdef NATIVE_WIN32
			__tempDir = strdup("C:\\");
#else
			__tempDir = strdup("/tmp");
#endif
		}
	}

	if (__homeDir == NULL)
	{
		temp = getenv("HOME");

#ifdef NATIVE_WIN32
		if (temp == NULL)
		{
			/*
			 * (Taken from glib/gutils.c:)
			 *
			 * The official way to specify a home directory on NT is the
			 * HOMEDRIVE and HOMEPATH environment variables.
			 *
			 * This is inside #ifdef NATIVE_WIN32 because with the cygwin
			 * dll, HOME should be a POSIX style pathname.
			 */
			if (getenv("HOMEDRIVE") != NULL && getenv("HOMEPATH") != NULL)
			{
				char *homedrive, *homepath;

				homedrive = strdup(getenv("HOMEDRIVE"));
				homepath  = strdup(getenv("HOMEPATH"));

				MEM_CHECK(__homeDir = (char *)malloc((strlen(homedrive) +
													  strlen(homepath) + 1) *
													 sizeof(char)));

				strcpy(__homeDir, homedrive);
				strcat(__homeDir, homepath);

				free(homedrive);
				free(homepath);
			}
		}
#endif /* NATIVE_WIN32 */

#ifdef HAVE_PWD_H
		{
			struct passwd *pw = NULL;
			void *buffer = NULL;

#ifdef HAVE_GETPWUID_R
			struct passwd pwd;
			
#ifdef _SC_GETPW_R_SIZE_MAX
			/* This returns the maximum length */
			unsigned int bufsize = sysconf(_SC_GETPW_R_SIZE_MAX);
#else
			unsigned int bufsize = 64;
#endif
			int error;

			do
			{
				if (buffer != NULL)
					free(buffer);
				
				MEM_CHECK(buffer = malloc(bufsize));

				errno = 0;

#if 0
#ifdef HAVE_GETPWUID_R_POSIX
				error = getpwuid_r(getuid(), &pwd, buffer, bufsize, &pw);
				error = (error < 0 ? errno : error);
#else
# ifdef _AIX
				error = getpwuid_r(getuid(), &pwd, buffer, bufsize);
				pw = (error == 0 ? &pwd : NULL);
# else
				pw = getpwuid_r(getuid(), &pwd, buffer, bufsize);
				error = (pw ? 0 : errno);
# endif
#endif
#endif
				
				if (pw == NULL)
				{
					/*
					 * We bail out prematurely if the user ID can't be found
					 * (should be pretty rare case actually), or if the
					 * buffer should be sufficiently big and lookups are
					 * still not successfull.
					 */
					if (error == 0 || error == ENOENT)
					{
						fprintf(stderr,
								_("libcomprex: __getEnvInfo(): "
								  "No such user %d."), getuid());
						break;
					}

					if (bufsize > (32 * 1024))
					{
						fprintf(stderr, _("libcomprex: __getEnvInfo(): %s."),
								strerror(error));
						break;
					}

					bufsize *= 2;
				}
			}
			while (pw == NULL);
#endif /* !HAVE_GETPWUID_R */
			
			if (pw == NULL)
			{
				setpwent();
				pw = getpwuid(getuid());
				endpwent();
			}

			if (pw != NULL)
			{
				__userName = strdup(pw->pw_name);
				__realName = strdup(pw->pw_gecos);

				if (__homeDir == NULL)
					__homeDir = strdup(pw->pw_dir);
			}

			if (buffer != NULL)
				free(buffer);
		}
#else /* !HAVE_PWD_H */

#ifdef NATIVE_WIN32
		{
			unsigned int len = 17;
			char buffer[17];

			if (GetUserName(buffer, &len))
			{
				__userName = strdup(buffer);
				__realName = strdup(buffer);
			}
		}
#endif /* NATIVE_WIN32 */
#endif /* !HAVE_PWD_H_ */

		if (__userName == NULL)
			__userName = strdup("somebody");

		if (__realName == NULL)
			__realName = strdup("Unknown");
		else
		{
			char *p;

			for (p = __realName; *p != '\0'; p++)
			{
				if (*p == ',')
				{
					*p = 0;
					p = strdup(__realName);
					free(__realName);
					__realName = p;
					
					break;
				}
			}
		}
	}
}

const char *
cxGetUserName()
{
	if (__tempDir == NULL)
		__getEnvInfo();

	return __userName;
}

const char *
cxGetRealName()
{
	if (__tempDir == NULL)
		__getEnvInfo();

	return __realName;
}

const char *
cxGetHomeDir()
{
	if (__tempDir == NULL)
		__getEnvInfo();

	return __homeDir;
}

const char *
cxGetTempDir()
{
	if (__tempDir == NULL)
		__getEnvInfo();

	return __tempDir;
}

void
cxCleanupEnvInfo()
{
	if (__tempDir  != NULL) free(__tempDir);
	if (__homeDir  != NULL) free(__homeDir);
	if (__userName != NULL) free(__userName);
	if (__realName != NULL) free(__realName);

	__tempDir  = NULL;
	__homeDir  = NULL;
	__userName = NULL;
	__realName = NULL;
}

/*
 * Borrowed from Imlib2's __imlib_FileDir()
 */
char **
cxListDir(const char *dir, int *fileCount, const char *prefix)
{
	struct dirent *dp;
	int            i, dirLength;
	int            done = 0;
	DIR           *dirp;
	char         **names;

	if (dir == NULL)
		return NULL;

	dirp = opendir(dir);

	if (dirp == NULL)
	{
		*fileCount = 0;
		return NULL;
	}

	/* Count number of entries in the dir (worst case) */
	for (dirLength = 0; (dp = readdir(dirp)) != NULL; dirLength++);

	if (dirLength == 0)
	{
		closedir(dirp);
		*fileCount = dirLength;
		return NULL;
	}

	names = (char **)malloc(dirLength * sizeof(char *));

	if (names == NULL)
		return NULL;

	rewinddir(dirp);

	for (i = 0; i < dirLength;)
	{
		dp = readdir(dirp);

		if (dp == NULL)
			break;

		if (strcmp(dp->d_name, ".") && strcmp(dp->d_name, "..") &&
			(prefix == NULL || !strncmp(dp->d_name, prefix, strlen(prefix))))
		{
			names[i] = strdup(dp->d_name);
			i++;
		}
	}

	if (i < dirLength)
		dirLength = i;  /* dir got shorter... */

	closedir(dirp);
	*fileCount = dirLength;

	/* Do a simple bubble sort here to alphanumberic it */
	/* TODO: Should we just use bsort() here? */
	while (!done)
	{
		done = 1;

		for (i = 0; i < dirLength - 1; i++)
		{
			if (strcmp(names[i], names[i + 1]) > 0)
			{
				char *temp;
				temp = names[i];
				names[i] = names[i + 1];
				names[i + 1] = temp;
				done = 0;
			}
		}
	}

	return names;
}

/*
 * Borrowed from Imlib2's __imlib_FileFreeDirList()
 */
void
cxFreeDirList(char **files, int fileCount)
{
	if (files == NULL)
		return;

	while (fileCount--)
	{
		if (files[fileCount] != NULL)
			free(files[fileCount]);
	}

	free(files);
}

void
cxProcessUri(const char *uri, char **scheme, char **path)
{
	const char *sep;
	int len, n;

	len = strlen(uri);
	sep = strchr(uri, ':');

	if (sep == NULL)
	{
		sep = uri;
		*scheme = strdup("file");
	}
	else
	{
		n = sep - uri;

		MEM_CHECK(*scheme = (char *)malloc((n + 1) * sizeof(char)));

		strncpy(*scheme, uri, n);
		(*scheme)[n] = '\0';

		uri  = sep + 1;
		len -= n + 1;
	}
	
	if (len <= 0)
		*path = NULL;
	else
	{
		MEM_CHECK(*path = (char *)malloc((len + 1) * sizeof(char)));

		strcpy(*path, uri);
	}
}

char *
cxMakeTempFilename(void)
{
	char *filename;
	const char *tempDir = cxGetTempDir();
	int i;

	MEM_CHECK(filename = (char *)malloc((strlen(tempDir) + 19) * sizeof(char)));
	
	strcpy(filename, tempDir);
	strcat(filename, "/libcomprex-XXXXXX");
	
	i = mkstemp(filename);

	if (i == -1)
	{
		/*
		 * TODO: Since our format is correct, any error would be
		 *       EEXIST. If this becomes a problem, we'll need to
		 *       to update this.
		 */
		free(filename);
		
		return NULL;
	}
	
	/*
	 * We don't want to have this open right now.
	 * We'll open it manually later.
	 */
	close(i);

	return filename;
}

CxModule *
cxFindOwnerModule(CxArchive *archive, CxFP *fp)
{
	CxModule *module = NULL;
	const char *ext = NULL;
	
	if (archive == NULL || fp == NULL)
		return NULL;

	if (cxGetArchiveFileName(archive) != NULL)
		ext = strrchr(cxGetArchiveFileName(archive), '.');

	if (ext != NULL)
	{
		ext++;

		/* Let's find a module that can open it. */
		for (module = cxGetFirstModule(CX_MODULE_ARCHIVE);
			 module != NULL;
			 module = module->next)
		{
			if (module->ops.archive->supportsExtension(ext))
			{
				/* Try to open it. */
				CxStatus status;

				cxRewind(fp);

				status = module->ops.archive->readArchive(archive, fp);

				if (status == CX_SUCCESS)
				{
					cxSetArchiveModule(archive, module);

					return module;
				}
			}
		}
	}

	if (module == NULL)
	{
		/* Loop through all modules and see if any are able to support it. */
		for (module = cxGetFirstModule(CX_MODULE_ARCHIVE);
			 module != NULL;
			 module = module->next)
		{
			/* Try to open it. */
			CxStatus status;

			cxRewind(fp);

			status = module->ops.archive->readArchive(archive, fp);

			if (status == CX_SUCCESS)
			{
				cxSetArchiveModule(archive, module);

				return module;
			}
		}
	}

	return NULL;
}

