/*-
 * Copyright (c) 2005 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipastat_thresholds.c,v 1.2.2.1 2011/11/15 18:12:29 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <limits.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"

#include "ipastat_log.h"
#include "ipastat_main.h"
#include "ipastat_rules.h"
#include "ipastat_st.h"

char		has_opt_thresholds = 0;	/* Set if -q ... -t ... */

#ifdef WITH_THRESHOLDS

signed char	dynamic_thresholds;	/* dynamic_thresholds parameter. */

ipa_mzone	*threshold_mzone;	/* Mzone for struct threshold{}. */

struct threshold *
alloc_threshold(void)
{
	struct threshold *threshold;

	threshold = mzone_alloc(threshold_mzone);
	if (threshold == NULL) {
		xlogmsgx(IPA_LOG_ERR, "alloc_threshold: mzone_alloc failed");
		return (NULL);
	}
	threshold->info = NULL;
	threshold->st_list = NULL;
	threshold->inited = 0;
	return (threshold);
}

/*
 * Return pointer to threshold with the give name.
 */
struct threshold *
threshold_by_name(const struct rule *rule, const char *name)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (strcmp(name, threshold->name) == 0)
			break;
	return (threshold);
}

static void
get_min_max_no(	const struct opt_thresholds *opt_thresholds,
    unsigned int *minno, unsigned int *maxno)
{
	const struct threshold *threshold;
	const struct opt_threshold *opt_threshold;

	*minno = UINT_MAX;
	*maxno = 0;
	STAILQ_FOREACH(opt_threshold, opt_thresholds, link) {
		threshold = opt_threshold->threshold;
		if (*minno > threshold->no)
			*minno = threshold->no;
		if (*maxno < threshold->no)
			*maxno = threshold->no;
	}
}

/*
 * Initialize all thresholds from one rule in modules.
 */
int
init_thresholds(const struct opt_rule *opt_rule)
{
	const struct rule *rule;
	const struct opt_threshold *opt_threshold;
	const struct opt_thresholds *opt_thresholds;
	struct threshold *threshold;
	unsigned int minno, maxno, no;

	rule = opt_rule->rule;
	opt_thresholds = &opt_rule->opt_thresholds;

	get_min_max_no(opt_thresholds, &minno, &maxno);

	for (no = minno; no <= maxno; ++no)
		STAILQ_FOREACH(opt_threshold, opt_thresholds, link) {
			threshold = opt_threshold->threshold;
			if (threshold->no == no) {
				if (!threshold->inited) {
					threshold->inited = 1;
					if (st_init_threshold(rule,
					    threshold) < 0) {
						logbt("init_thresholds");
						return (-1);
					}
				}
				break;
			}
		}

	return (0);
}

int
deinit_thresholds(const struct opt_rule *opt_rule)
{
	const struct rule *rule;
	const struct opt_threshold *opt_threshold;
	const struct opt_thresholds *opt_thresholds;
	struct threshold *threshold;
	unsigned int minno, maxno, no;
	int rv;

	rule = opt_rule->rule;
	opt_thresholds = &opt_rule->opt_thresholds;

	get_min_max_no(opt_thresholds, &minno, &maxno);

	rv = 0;
	for (no = minno; no <= maxno; ++no)
		STAILQ_FOREACH(opt_threshold, &opt_rule->opt_thresholds, link) {
			threshold = opt_threshold->threshold;
			if (threshold->no == no) {
				if (threshold->inited) {
					threshold->inited = 0;
					if (st_deinit_threshold(rule,
					    threshold) < 0) {
						logbt("deinit_thresholds");
						rv = -1;
					}
				}
				break;
			}
		}

	return (rv);
}

/*
 * Copy all thresholds from list_src to rule.
 */
int
copy_thresholds(struct rule *rule, const struct thresholds_list *list)
{
	const struct threshold *tsrc;
	struct threshold *tdst;

	STAILQ_FOREACH(tsrc, list, link) {
		tdst = mzone_alloc(threshold_mzone);
		if (tdst == NULL) {
			xlogmsgx(IPA_LOG_ERR, "copy_thresholds: "
			    "mzone_alloc failed");
			return (-1);
		}

		/* Copy settings from source threshold. */
		*tdst = *tsrc;

		tdst->free_mask = 0;

		/* Link just allocated threshold to rule. */
		STAILQ_INSERT_TAIL(&rule->thresholds, tdst, link);
	}
	return (0);
}

/*
 * Release memory used by a list of thresholds.
 */
void
free_thresholds(struct thresholds_list *thresholds)
{
	struct threshold *threshold, *threshold_next;

	STAILQ_FOREACH_SAFE(threshold, thresholds, link, threshold_next) {
		if (threshold->free_mask & THRESHOLD_FREE_NAME)
			mem_free(threshold->name, m_anon);
		mem_free(threshold->info, m_result);
		mzone_free(threshold_mzone, threshold);
	}
}

/*
 * Add one optional threshold given in the -q -r option.
 */
int
opt_threshold_add(const char *name)
{
	struct opt_threshold *opt_threshold;

	opt_threshold = mem_malloc(sizeof(*opt_threshold), m_anon);
	if (opt_threshold == NULL) {
		logmsgx(IPA_LOG_ERR, "opt_threshold_add: mem_malloc failed");
		return (-1);
	}

	opt_threshold->name = name;
	opt_threshold->opt_st = cur_opt_st;

	cur_opt_rule->type = OPT_RULE_THRESHOLD;

	STAILQ_INSERT_TAIL(&cur_opt_rule->opt_thresholds, opt_threshold, link);

	has_opt_thresholds = 1;
	return (0);
}

int
opt_thresholds_parse(const struct opt_rule *opt_rule)
{
	struct rule *rule;
	struct threshold *threshold;
	struct opt_threshold *opt_threshold;
	unsigned int thresholdno;

	if (STAILQ_EMPTY(&opt_rule->opt_thresholds))
		return (0);

	rule = opt_rule->rule;
	if (STAILQ_EMPTY(&rule->thresholds))
		thresholdno = 0;
	else {
		threshold = STAILQ_LAST(&rule->thresholds, threshold, link);
		thresholdno = threshold->no + 1;
	}

	STAILQ_FOREACH(opt_threshold, &opt_rule->opt_thresholds, link) {
		threshold = threshold_by_name(rule, opt_threshold->name);
		if (threshold == NULL) {
			/*
			 * This threshold is not given in the configuration
			 * file.
			 */
			if (!dynamic_thresholds) {
				logmsgx(IPA_LOG_ERR, "opt_thresholds_parse: "
				    "unknown rule %s, threshold %s",
				    rule->name, opt_threshold->name);
				return (-1);
			}
			threshold = alloc_threshold();
			if (threshold == NULL) {
				logbt("opt_thresholds_parse");
				return (-1);
			}
			STAILQ_INSERT_TAIL(&rule->thresholds, threshold, link);
			threshold->no = thresholdno++;
			threshold->name = (char *)opt_threshold->name;
			threshold->free_mask = 0;
		}
		opt_threshold->threshold = threshold;
		if (opt_threshold->opt_st != NULL)
			threshold->st_list = opt_threshold->opt_st->st_list;
		if (threshold->st_list == NULL)
			threshold->st_list = rule->st_list;
	}

	return (0);
}

void
opt_thresholds_free(const struct opt_thresholds *opt_thresholds)
{
	struct opt_threshold *opt_threshold, *opt_threshold_next;

	STAILQ_FOREACH_SAFE(opt_threshold, opt_thresholds, link,
	    opt_threshold_next)
		mem_free(opt_threshold, m_anon);
}
#endif /* WITH_THRESHOLDS */
