/*-
 * Copyright (c) 2003 Andrey Simonenko
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include "config.h"

#ifndef lint
static const char rcsid[] ATTR_UNUSED =
  "@(#)$Id: ipa_thresholds.c,v 1.2 2011/01/23 18:42:35 simon Exp $";
#endif /* !lint */

#include <sys/types.h>

#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ipa_mod.h"

#include "queue.h"

#include "dlapi.h"
#include "confcommon.h"
#include "memfunc.h"
#include "parser.h"

#include "ipa_ac.h"
#include "ipa_db.h"
#include "ipa_ctl.h"
#include "ipa_cmd.h"
#include "ipa_time.h"

#include "ipa_conf.h"
#include "ipa_log.h"
#include "ipa_main.h"
#include "ipa_rules.h"
#include "ipa_autorules.h"

unsigned int	nstatthresholds;	/* Number of static thresholds. */
unsigned int	ndynthresholds;		/* Number of dynamic thresholds. */

#ifdef WITH_THRESHOLDS

/* Parameters in global{} section. */
signed char	global_debug_threshold;
signed char	global_debug_threshold_init;
signed char	global_threshold_type;
signed char	global_load_threshold;
unsigned int	global_threshold_time_width;
const struct tevent *global_threshold_time_slice;

/* Parts of global { threshold_balance } parameter. */
unsigned int	global_threshold_below_lim;	/* X:-:- */
unsigned int	global_threshold_equal_lim;	/* -:X:- */
unsigned int	global_threshold_above_lim;	/* -:-:X */

/* Mzone for all struct threshold{}. */
ipa_mzone	*threshold_mzone;

const char *const threshold_event_msg[] = {
	"BELOW",	/* 0 */
	"EQUAL",	/* 1 */
	"ABOVE"		/* 2 */
};

#define EVENT(x) IPA_THRESHOLD_EVENT_ ## x

void
set_thr_min_max(struct threshold *threshold)
{
	uint64_t thr, dev;

	thr = threshold->thr;
	dev = threshold->thr_dev_pc != 0 ?
	    uint64_per_cent(&thr, threshold->thr_dev_pc) : threshold->thr_dev;
	threshold->thr_max = (thr > UINT64_MAX - dev) ?
	    UINT64_MAX : thr + dev;
	threshold->thr_min = thr < dev ? 0 : thr - dev;
}

/*
 * Remove statistics from the latest slice, if positive counter or
 * negative counter overflows, then return -1, since this means
 * incorrect configuration.
 */
static int
threshold_flush_slice(const struct rule *rule, struct threshold *threshold)
{
	uint64_t chunk;
	BITMAP_TYPE *bitword, bitmask;
	unsigned int i;

	i = threshold->cnt_slice_i;
	chunk = threshold->cnt_slice[i];
	threshold->cnt_slice[i] = 0;

	bitword = BIT_WORD(threshold->cnt_slice_sign, i);
	if (*bitword != 0) {
		/* Have to check bit. */
		bitmask = BIT_MASK(i);
		if (BIT_TEST(bitword, bitmask)) {
			BIT_CLEAR(bitword, bitmask);
			goto add_chunk;
		}
	}

	/* Subtract positive slice counter. */
	if (rule->debug_threshold)
		logdbg("rule %s, threshold %s: threshold_flush_slice: "
		    "remove chunk %"PRIu64" from slice #%u", rule->name,
		    threshold->name, chunk, i);
	if (threshold->cnt >= chunk)
		threshold->cnt -= chunk;
	else {
		chunk -= threshold->cnt;
		if (threshold->cnt_neg > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "threshold_flush_slice: negative counter "
			    "overflowed", rule->name, threshold->name);
			logmsgx(IPA_LOG_ERR, "this means that something "
			    "is wrong in configuration");
			return (-1);
		}
		threshold->cnt_neg += chunk;
		threshold->cnt = 0;
	}

	return (0);

add_chunk:
	/* Add absolute value of negative slice counter. */
	if (rule->debug_threshold)
		logdbg("rule %s, threshold %s: threshold_flush_slice: "
		    "remove chunk -%"PRIu64" from slice #%u", rule->name,
		    threshold->name, chunk, i);
	if (threshold->cnt_neg >= chunk)
		threshold->cnt_neg -= chunk;
	else {
		chunk -= threshold->cnt_neg;
		if (threshold->cnt > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "threshold_flush_slice: positive counter "
			    "overflowed", rule->name, threshold->name);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		threshold->cnt += chunk;
		threshold->cnt_neg = 0;
	}

	return (0);
}

/*
 * Set threshold active or inactive in modules it uses.
 */
int
mod_set_threshold_active(const struct rule *rule, struct threshold *threshold,
    int active)
{
	if ((threshold->thr_flags & THRESHOLD_FLAG_ACTIVE) ==
	    (active ? THRESHOLD_FLAG_ACTIVE : 0)) {
		logmsgx(IPA_LOG_ERR, "internal error: "
		    "mod_set_threshold_active(%s, %s, %d): threshold is "
		    "already %s", rule->name, threshold->name, active,
		    active_msg[active]);
		return (-1);
	}

	if (debug_worktime)
		logdbg("rule %s, threshold %s: set threshold %s",
		    rule->name, threshold->name, active_msg[active]);

	if (active)
		THRESHOLD_SET_ACTIVE(threshold);
	else
		THRESHOLD_SET_INACTIVE(threshold);

	if (ac_set_threshold_active(rule, threshold, active) < 0)
		goto failed;
	if (db_set_threshold_active(rule, threshold, active) < 0)
		goto failed;

	return (0);

failed:
	logbt("mod_set_threshold_active");
	return (-1);
}

static int
set_threshold_active(const struct rule *rule, struct threshold *threshold)
{
	unsigned int i, n;

	if (threshold->thr_type & THRESHOLD_JUMP_OVER_INACTIVE) {
		i = ipa_tm_diff(&threshold->tm_updated, &threshold->tm_started);
		threshold->tm_started = curdate;
		ipa_tm_sub(&threshold->tm_started, i);
		if (rule->debug_threshold)
			logdbg("rule %s, threshold %s: set_threshold_active: "
			    "jumping over inactive time interval", rule->name,
			    threshold->name);
	} else {
		n = ipa_tm_diff(&curdate, &threshold->tm_updated) /
		    threshold->time_slice->event_step;
		if (n >= threshold->cnt_slice_n)
			n = threshold->cnt_slice_n;

		if (rule->debug_threshold)
			logdbg("rule %s, threshold %s: set_threshold_active: "
			    "flushing %u slice(s), starting from slice #%u",
			    rule->name, threshold->name, n,
			    threshold->cnt_slice_i);

		for (i = 0; i < n; ++i) {
			threshold->cnt_slice_i++;
			if (threshold->cnt_slice_i == threshold->cnt_slice_n)
				threshold->cnt_slice_i = 0;
			if (threshold_flush_slice(rule, threshold) < 0) {
				logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
				    "set_threshold_active: "
				    "threshold_flush_slice failed",
				    rule->name, threshold->name);
				return (-1);
			}
		}
	}

	/* Set new time when to check this threshold again. */
	threshold->check_sec = threshold->time_slice->event_sec;

	return (mod_set_threshold_active(rule, threshold, 1));
}

/*
 * Add chunk to one threshold, if positive counter or slice counter
 * overflows, then return -1, since this means incorrect configuration.
 */
int
threshold_add_chunk(const struct rule *rule, struct threshold *threshold,
    const uint64_t *chunk_ptr)
{
	uint64_t chunk;
	BITMAP_TYPE *bitword, bitmask;
	unsigned int i;

	chunk = *chunk_ptr;

	/* Add chunk to counter. */
	if (threshold->cnt_neg >= chunk)
		threshold->cnt_neg -= chunk;
	else {
		chunk -= threshold->cnt_neg;
		if (threshold->cnt > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "threshold_add_chunk: positive counter overflowed",
			    rule->name, threshold->name);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		threshold->cnt += chunk;
		threshold->cnt_neg = 0;
		/* Restore chunk. */
		chunk = *chunk_ptr;
	}

	/* Add chunk to slice counter. */
	i = threshold->cnt_slice_i;
	if (rule->debug_threshold)
		logdbg("rule %s, threshold %s: threshold_add_chunk: add "
		    "chunk %"PRIu64" to slice #%u", rule->name,
		    threshold->name, chunk, i);

	/*
	 * Usually statistics is positive, so let's make some
	 * optimization for usual situation.
	 */
	bitword = BIT_WORD(threshold->cnt_slice_sign, i);
	if (*bitword != 0) {
		/* Have to check bit. */
		bitmask = BIT_MASK(i);
		if (BIT_TEST(bitword, bitmask)) {
			/* Slice has negative statistics. */
			if (threshold->cnt_slice[i] > chunk)
				threshold->cnt_slice[i] -= chunk;
			else {
				threshold->cnt_slice[i] = chunk -
				    threshold->cnt_slice[i];
				BIT_CLEAR(bitword, bitmask);
			}
			return (0);
		}
	}

	/* Slice has positive statistics. */
	if (threshold->cnt_slice[i] > UINT64_MAX - chunk) {
		logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
		    "threshold_add_chunk: positive slice counter #%u "
		    "overflowed", rule->name, threshold->name, i);
		logmsgx(IPA_LOG_ERR, "this means that something is "
		    "wrong in configuration");
		return (-1);
	}
	threshold->cnt_slice[i] += chunk;

	return (0);
}

/*
 * Add chunk to every rule's threshold.
 */
int
thresholds_add_chunk(const struct rule *rule, const uint64_t *chunk_ptr)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link) {
		if (THRESHOLD_IS_INACTIVE(threshold))
			continue;
		if (threshold_add_chunk(rule, threshold, chunk_ptr) < 0) {
			logbt("thresholds_add_chunk");
			return (-1);
		}
	}
	return (0);
}

/*
 * Subtract chunk from one threshold.
 */
int
threshold_sub_chunk(const struct rule *rule, struct threshold *threshold,
    const uint64_t *chunk_ptr)
{
	uint64_t chunk;
	BITMAP_TYPE *bitword, bitmask;
	unsigned int i;

	chunk = *chunk_ptr;

	/* Subtract chunk from counter. */
	if (threshold->cnt >= chunk)
		threshold->cnt -= chunk;
	else {
		chunk -= threshold->cnt;
		if (threshold->cnt_neg > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "threshold_sub_chunk: negative counter overflowed",
			    rule->name, threshold->name);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		threshold->cnt_neg += chunk;
		threshold->cnt = 0;
		/* Restore chunk. */
		chunk = *chunk_ptr;
	}

	/* Subtract chunk from slice counter. */
	i = threshold->cnt_slice_i;
	if (rule->debug_threshold)
		logdbg("rule %s, threshold %s: threshold_sub_chunk: subtract "
		    "chunk %"PRIu64" from slice #%u", rule->name,
		    threshold->name, chunk, i);

	bitword = BIT_WORD(threshold->cnt_slice_sign, i);
	bitmask = BIT_MASK(i);
	if (BIT_TEST(bitword, bitmask)) {
		/* Slice has negative statistics. */
		if (threshold->cnt_slice[i] > UINT64_MAX - chunk) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "threshold_sub_chunk: negative slice counter "
			    "%u overflowed", rule->name, threshold->name, i);
			logmsgx(IPA_LOG_ERR, "this means that something is "
			    "wrong in configuration");
			return (-1);
		}
		threshold->cnt_slice[i] += chunk;
	} else {
		/* Slice has positive statistics. */
		if (threshold->cnt_slice[i] >= chunk)
			threshold->cnt_slice[i] -= chunk;
		else {
			threshold->cnt_slice[i] = chunk -
			    threshold->cnt_slice[i];
			BIT_SET(bitword, bitmask);
		}
	}

	return (0);
}

/*
 * Subtract chunk from every rule's threshold.
 */
int
thresholds_sub_chunk(const struct rule *rule, const uint64_t *chunk_ptr)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link) {
		if (THRESHOLD_IS_INACTIVE(threshold))
			continue;
		if (threshold_sub_chunk(rule, threshold, chunk_ptr) < 0) {
			logbt("thresholds_sub_chunk");
			return (-1);
		}
	}
	return (0);
}

static int
above_threshold(const struct rule *rule, struct threshold *threshold)
{
	if (rule->debug_threshold)
		logdbg("rule %s, threshold %s: above threshold cnt %"PRIu64", "
		    "rest %u invocations", rule->name, threshold->name,
		    threshold->cnt, threshold->above_cnt);

	if (threshold->above_cnt == 0)
		return (0);

	/* Adjust number of invocations. */
	threshold->above_cnt--;
	threshold->equal_cnt = threshold->equal_lim;
	threshold->below_cnt = threshold->below_lim;

	/* Register event in modules. */
	if (ac_threshold_event(rule, threshold, EVENT(ABOVE)) < 0)
		goto failed;

	/* Run commands. */
	if (threshold->above_thr.has_cmd)
		if (run_cmds(rule, &threshold->wpid, &threshold->above_thr,
		    "rule %s { threshold %s { above_threshold {}}}",
		    rule->name, threshold->name) < 0)
			goto failed;

	return (0);

failed:
	logbt("above_threshold");
	return (-1);
}

static int
below_threshold(const struct rule *rule, struct threshold *threshold)
{
	if (rule->debug_threshold) {
		uint64_t cnt;
		const char *sign;

		if (threshold->cnt_neg == 0) {
			cnt = threshold->cnt;
			sign = "";
		} else {
			cnt = threshold->cnt_neg;
			sign = "-";
		}
		logdbg("rule %s, threshold %s: below threshold, "
		    "cnt %s%"PRIu64", rest %u invocations", rule->name,
		    threshold->name, sign, cnt, threshold->below_cnt);
	}
	if (threshold->below_cnt == 0)
		return (0);

	/* Adjust number of invocations. */
	threshold->below_cnt--;
	threshold->equal_cnt = threshold->equal_lim;
	threshold->above_cnt = threshold->above_lim;

	/* Register event in modules. */
	if (ac_threshold_event(rule, threshold, EVENT(BELOW)) < 0)
		goto failed;

	/* Run commands. */
	if (threshold->below_thr.has_cmd)
		if (run_cmds(rule, &threshold->wpid, &threshold->below_thr,
		    "rule %s { threshold %s { below_threshold {}}}",
		    rule->name, threshold->name) < 0)
			goto failed;

	return (0);

failed:
	logbt("below_threshold");
	return (-1);
}

static int
equal_threshold(const struct rule *rule, struct threshold *threshold)
{
	if (rule->debug_threshold)
		logdbg("rule %s, threshold %s: equal threshold, "
		    "cnt %"PRIu64", rest %u invocations", rule->name,
		    threshold->name, threshold->cnt, threshold->equal_cnt);

	if (threshold->equal_cnt == 0)
		return (0);

	/* Adjust number of invocations. */
	threshold->equal_cnt--;
	threshold->below_cnt = threshold->below_lim;
	threshold->above_cnt = threshold->above_lim;

	/* Register event in modules. */
	if (ac_threshold_event(rule, threshold, EVENT(EQUAL)) < 0)
		goto failed;

	/* Run commands. */
	if (threshold->equal_thr.has_cmd)
		if (run_cmds(rule, &threshold->wpid, &threshold->equal_thr,
		    "rule %s { threshold %s { equal_threshold {}}}",
		    rule->name, threshold->name) < 0)
			goto failed;

	return (0);

failed:
	logbt("equal_threshold");
	return (-1);
}

static int
check_threshold_cnt(const struct rule *rule, struct threshold *threshold)
{
	if (threshold->cnt > threshold->thr_max) {
		/* [  ] x */
		if (above_threshold(rule, threshold) < 0)
			goto failed;
	} else if (threshold->cnt < threshold->thr_min) {
		/* x [  ] */
		if (below_threshold(rule, threshold) < 0)
			goto failed;
	} else {
		/* [ x ] */
		if (equal_threshold(rule, threshold) < 0)
			goto failed;
	}
	return (0);

failed:
	logbt("check_threshold_cnt");
	return (-1);
}

/*
 * This routine implements "below_threshold", "equal_threshold",
 * "above_threshold" and part of "worktime" in "threshold" section.
 */
int
check_thresholds(const struct rule *rule, unsigned int *check_sec_ptr)
{
	ipa_tm tm_started;
	const struct worktime *wt;
	struct threshold *threshold;
	unsigned int i, check_sec;

	check_sec = EVENT_NOT_SCHEDULED;

	STAILQ_FOREACH(threshold, &rule->thresholds, link) {
		wt = threshold->worktime;
		if (THRESHOLD_IS_INACTIVE(threshold)) {
			/* Threshold is inactive. */
			if (wt->active_sec <= cursec) {
				/* It's time to make threshold active. */
				if (set_threshold_active(rule, threshold) < 0)
					goto failed;
			} else {
				if (check_sec > wt->active_sec)
					check_sec = wt->active_sec;
				continue; /* do not check any time events. */
			}
		}

		/* Here threshold is active. */
		if (threshold->check_sec <= cursec) {
			/* It's time to check threshold. */
			if (check_threshold_cnt(rule, threshold) < 0)
				goto failed;

			/* Calculate next slice number. */
			i = threshold->cnt_slice_i + 1;
			if (i == threshold->cnt_slice_n)
				i = 0;
			threshold->cnt_slice_i = i;

			/* Remove statistics from the latest slice. */
			if (threshold_flush_slice(rule, threshold) < 0)
				goto failed;

			/* Set new time when to check this threshold again. */
			threshold->check_sec = threshold->time_slice->event_sec;

			/* Update threshold in the database. */
			tm_started = curdate;
			if (newday_flag)
				fix_240000(&tm_started);
			ipa_tm_sub(&tm_started, threshold->time_width);
			if (threshold->shift_window)
				threshold->tm_started = tm_started;
			else if (cmp_ipa_tm(&tm_started,
			    &threshold->tm_started) > 0) {
				threshold->shift_window = 1;
				threshold->tm_started = tm_started;
			}

			threshold->tm_updated = curdate;
			if (db_update_threshold(rule, threshold) < 0)
				goto failed;
		} else if (need_check_flag) {
			if (db_update_threshold(rule, threshold) < 0)
				goto failed;
		}

		if (WT_IS_INACTIVE(wt)) {
			/* Threshold became inactive. */
			if (!newday_flag) {
				if (set_threshold_inactive(rule, threshold) < 0)
					goto failed;
				if (check_sec > wt->active_sec)
					check_sec = wt->active_sec;
			}
		} else {
			/* Threshold is still active. */
			if (check_sec > wt->inactive_sec)
				check_sec = wt->inactive_sec;
			if (check_sec > threshold->check_sec)
				check_sec = threshold->check_sec;
		}
	}

	*check_sec_ptr = check_sec;
	return (0);

failed:
	logbt("check_thresholds");
	return (-1);
}

/*
 * Initialize one threshold.
 */
static int
init_threshold(const struct rule *rule, struct threshold *threshold)
{
	struct ipa_threshold_state ostat, nstat;
	uint64_t chunk, cnt;
	size_t size;
	const char *rule_name, *threshold_name;
	unsigned int i, n, old_slice_n, old_time_width;
	signed char debug_threshold_init;

	/*
	 * If init_threshold() is called second, third... time,
	 * then threshold can be inactive.
	 */
	if (THRESHOLD_IS_INACTIVE(threshold))
		if (set_threshold_active(rule, threshold) < 0)
			goto failed;

	rule_name = rule->name;
	threshold_name = threshold->name;

	threshold->cnt_neg = 0;
	threshold->shift_window = 0;
	threshold->wpid.debug_exec = rule->debug_exec;

	/* Get number of statistics slices. */
	n = threshold->cnt_slice_n =
	    threshold->time_width / threshold->time_slice->event_step;

	/* Initialize statistics slices. */
	size = n * sizeof(*threshold->cnt_slice);
	if (threshold->cnt_slice == NULL) {
		threshold->cnt_slice = mem_malloc(size, m_anon);
		if (threshold->cnt_slice == NULL) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "init_threshold: mem_malloc for slices failed",
			    rule_name, threshold_name);
			return (-1);
		}
	}
	memset(threshold->cnt_slice, 0, size);

	/* Initialize statistics slices signs. */
	size = BITMAP_SIZE(n);
	if (threshold->cnt_slice_sign == NULL) {
		threshold->cnt_slice_sign = mem_malloc(size, m_anon);
		if (threshold->cnt_slice_sign == NULL) {
			logmsgx(IPA_LOG_ERR, "rule %s, threshold %s: "
			    "init_threshold: mem_malloc for slices signs "
			    "failed", rule_name, threshold_name);
			return (-1);
		}
	}
	memset(threshold->cnt_slice_sign, 0, size);

	debug_threshold_init = rule->debug_threshold_init;

	if (debug_threshold_init)
		logdbg("rule %s, threshold %s: init_threshold: allocated %u "
		    "slices", rule_name, threshold_name, n);

	/*
	 * If we cannot get current threshold state from the database,
	 * then let's use current "threshold" value.
	 */
	nstat.thr = threshold->thr;

	if (STAILQ_EMPTY(threshold->db_list)) {
		/* "null" database is used. */
		if (THRESHOLD_IS_INITED(threshold)) {
			/* Imitate that old state is known. */
			ostat.thr = threshold->thr;
			ostat.cnt = threshold->cnt;
			ostat.tm_from = threshold->tm_started;
			ostat.tm_updated = threshold->tm_updated;
			if (debug_threshold_init)
				logdbg("rule %s, threshold %s: init_threshold: "
				    "continue to use previous threshold state",
				    rule_name, threshold_name);
		} else {
			if (debug_threshold_init)
				logdbg("rule %s, threshold %s: init_threshold: "
				    "create threshold with no database",
				    rule_name, threshold_name);
			goto new_state;
		}
	} else {
		const char *db_name;
		int rv;

		/* Get threshold state from databases. */
		rv = db_get_threshold_state(rule, threshold, &ostat, &db_name);
		if (rv < 0)
			goto failed;
		if (db_name == NULL) {
			logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
			    "init_threshold: no database used for this "
			    "threshold supports db_get_threshold_state "
			    "function", rule_name, threshold_name);
			goto new_state;
		}
		if (rv > 0) {
			if (debug_threshold_init)
				logdbg("rule %s, threshold %s: init_threshold: "
				    "got state from database %s",
				    rule_name, threshold_name, db_name);
		} else {
			if (debug_threshold_init)
				logdbg("rule %s, threshold %s: init_threshold: "
				    "database %s does not have threshold state "
				    "(a new threshold)", rule_name,
				    threshold_name, db_name);
			goto new_state;
		}
	}

	/* Check and initialize "threshold" parameter. */
	if (debug_threshold_init && threshold->thr != ostat.thr) {
		logdbg("rule %s, threshold %s: init_threshold: configuration "
		    "threshold %"PRIu64", database threshold %"PRIu64" "
		    "(will use %s value)", rule_name, threshold_name,
		    threshold->thr, ostat.thr,
		    threshold->load_thr ? "database" : "configuration");
	}
	if (threshold->load_thr)
		nstat.thr = threshold->thr = ostat.thr;

	/* Check tm_from and tm_updated (it can be 24:00:00). */
	if (check_ipa_tm(&ostat.tm_from) < 0) {
		logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
		    "init_threshold: wrong value of tm_from: %s",
		    rule_name, threshold_name, tm_str(&ostat.tm_from));
		goto new_state_error;
	}
	fix_240000(&ostat.tm_updated);
	if (check_ipa_tm(&ostat.tm_updated) < 0) {
		logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
		    "init_threshold: wrong value of tm_updated: %s",
		    rule_name, threshold_name, tm_str(&ostat.tm_updated));
		goto new_state_error;
	}

	/* tm_from should be <= tm_updated. */
	if (cmp_ipa_tm(&ostat.tm_updated, &ostat.tm_from) < 0) {
		logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
		    "init_threshold: tm_updated is less than tm_from: %s < %s",
		    rule_name, threshold_name, tm_str(&ostat.tm_updated),
		    tm_str2(&ostat.tm_from));
		goto new_state_error;
	}

	/* Check whether time goes back. */
	if (cmp_ipa_tm(&ostat.tm_updated, &curdate) > 0) {
		logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
		    "init_threshold: tm_updated %s is greater than "
		    "current time", rule_name, threshold_name,
		    tm_str(&ostat.tm_updated));
		/*
		 * If already initialized threshold has update time greater
		 * than current time, then change tm_from and tm_updated.
		 */
		if (THRESHOLD_IS_INITED(threshold)) {
			const char *cp;
			unsigned int d;

			d = ipa_tm_diff(&ostat.tm_updated, &ostat.tm_from);
			cp = tm_str(&ostat.tm_from);
			ostat.tm_from = ostat.tm_updated = curdate;
			ipa_tm_sub(&ostat.tm_from, d);
			logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: "
			    "init_threshold: tm_from time changed: %s -> %s",
			    rule_name, threshold_name, cp,
			    tm_str2(&ostat.tm_from));
		} else
			goto new_state_error;
	}

	old_time_width = ipa_tm_diff(&ostat.tm_updated, &ostat.tm_from);

	if (old_time_width != threshold->time_width) {
		char buf[SEC_STR_SIZE];

		strncpy(buf, time_str(old_time_width), sizeof(buf) - 1);
		buf[sizeof(buf) - 1] = '\0';
		logmsgx(IPA_LOG_INFO, "rule %s, threshold %s: init_threshold: "
		    "it seems that threshold_time_width is changed: %s -> %s",
		    rule_name, threshold_name, buf,
		    time_str(threshold->time_width));
	}

	/* Old number of slices between tm_from and tm_updated. */
	old_slice_n = old_time_width / threshold->time_slice->event_step;

	/*
	 * Let it be one slice between tm_from and tm_updated,
	 * it is better, than simply forget about that statistics.
	 */
	if (old_slice_n == 0)
		old_slice_n = 1;

	/* One chunk between tm_from and tm_updated. */
	chunk = ostat.cnt / old_slice_n;

	if (threshold->thr_type & THRESHOLD_JUMP_OVER_STOPPED) {
		/* Jump over time interval when ipa was stopped. */
		if (threshold->cnt_slice_n <= old_slice_n) {
			threshold->cnt_slice_i = 0;
			n = threshold->cnt_slice_n;
		} else {
			threshold->cnt_slice_i = old_slice_n;
			n = old_slice_n;
		}

		/* Calculate tm_from. */
		nstat.tm_from = curdate;
		if (old_time_width < threshold->time_width)
			ipa_tm_sub(&nstat.tm_from, old_time_width);
		else
			ipa_tm_sub(&nstat.tm_from, threshold->time_width);
	} else {
		/* Slide threshold's statistics over time interval. */
		if (ipa_tm_diff(&curdate, &ostat.tm_updated) <
		    threshold->time_width) {
			/* tm_from tm_updated curdate */

			/*
			 * Get number of slices between curdate and tm_updated.
			 * n is always less than cnt_slice_n, since curdate -
			 * ostat.tm_updated is less than time_width.
			 */
			n = ipa_tm_diff(&curdate, &ostat.tm_updated) /
			    threshold->time_slice->event_step;

			/*
			 * Remember number of slices between curdate and
			 * tm_updated.
			 */
			threshold->cnt_slice_i = n;

			/*
			 * Max possible number of slices between tm_from and
			 * tm_updated.
			 */
			n = threshold->cnt_slice_n - n;

			/* n = min(n, old_slice_n). */
			if (n > old_slice_n)
				n = old_slice_n;

			/* Get number of next slice to use. */
			threshold->cnt_slice_i += n;
			if (threshold->cnt_slice_i == threshold->cnt_slice_n)
				threshold->cnt_slice_i = 0;

			/* Calculate tm_from. */
			nstat.tm_from = curdate;
			ipa_tm_sub(&nstat.tm_from, threshold->time_width);
			if (cmp_ipa_tm(&nstat.tm_from, &ostat.tm_from) < 0)
				nstat.tm_from = ostat.tm_from;
		} else {
			/*
			 * Do not use previous threshold state, because
			 * it is stale.
			 */
			nstat.tm_from = curdate;
			threshold->cnt_slice_i = n = 0;
			cnt = 0;
			goto skip_chunk_fill;
		}
	}

	/* Fill slices. */
	if (chunk != 0) {
		/* Fill equal chunks. */
		for (i = 0; i < n; ++i)
			threshold->cnt_slice[i] = chunk;
		cnt = n * chunk;
		chunk = ostat.cnt - cnt;
	} else {
		/* Not enough statistics for equal chunks. */
		cnt = 0;
		chunk = ostat.cnt;
	}

	if (chunk != 0) {
		/*
		 * Fill the rest of statistics, this is very approximately
		 * as all threshold's initialization.
		 */
		for (i = 0; i < n; ++i) {
			threshold->cnt_slice[i] += 1;
			cnt++;
			if (--chunk == 0)
				break;
		}
	}

skip_chunk_fill:
	set_thr_min_max(threshold);

	if (debug_threshold_init) {
		logdbg("rule %s, threshold %s: init_threshold: %u slice(s), "
		    "current slice #%u, cnt %"PRIu64, rule_name,
		    threshold_name, n, threshold->cnt_slice_i, cnt);
		logdbg("rule %s, threshold %s: init_threshold: "
		    "thr %"PRIu64", thr_min %"PRIu64", thr_max %"PRIu64,
		    rule_name, threshold_name, threshold->thr,
		    threshold->thr_min, threshold->thr_max);
		logdbg("rule %s, threshold %s: init_threshold: set "
		    "threshold state", rule_name, threshold_name);
	}

	threshold->cnt = nstat.cnt = cnt;
	threshold->tm_started = nstat.tm_from;
	threshold->tm_updated = nstat.tm_updated = curdate;

	/* Remove statistics from the latest slice. */
	threshold_flush_slice(rule, threshold);

	if (db_set_threshold_state(rule, threshold, &nstat) < 0)
		goto failed;

	THRESHOLD_SET_INITED(threshold);
	return (0);

new_state_error:
	logmsgx(IPA_LOG_WARNING, "rule %s, threshold %s: init_threshold: "
	    "set new threshold state, correcting errors found in database",
	    rule_name, threshold_name);

new_state:
	set_thr_min_max(threshold);

	threshold->tm_started = threshold->tm_updated =
	    nstat.tm_from = nstat.tm_updated = curdate;
	threshold->cnt = nstat.cnt = 0;

	threshold->cnt_slice_i = 0;

	if (debug_threshold_init) {
		logdbg("rule %s, threshold %s: init_threshold: "
		    "thr %"PRIu64", thr_min %"PRIu64", thr_max %"PRIu64,
		    rule_name, threshold_name, threshold->thr,
		    threshold->thr_min, threshold->thr_max);
		logdbg("rule %s, threshold %s: init_threshold: set new "
		    "threshold state", rule_name, threshold_name);
	}

	if (db_set_threshold_state(rule, threshold, &nstat) < 0)
		goto failed;

	THRESHOLD_SET_INITED(threshold);
	return (0);

failed:
	logbt("init_threshold");
	return (-1);
}

/*
 * Initialize all thresholds in one rule.
 */
int
init_thresholds(const struct rule *rule)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (init_threshold(rule, threshold) < 0) {
			logbt("init_thresholds");
			return (-1);
		}
	return (0);
}

/*
 * Return pointer to threshold with the give name.
 */
struct threshold *
threshold_by_name(const struct rule *rule, const char *name)
{
	struct threshold *threshold;

	STAILQ_FOREACH(threshold, &rule->thresholds, link)
		if (strcmp(name, threshold->name) == 0)
			break;
	return (threshold);
}

/*
 * This function is called from newday().
 */
int
thresholds_newday(struct rule *rule)
{
	struct threshold *threshold;
	unsigned int check_sec;

	check_sec = EVENT_NOT_SCHEDULED;

	STAILQ_FOREACH(threshold, &rule->thresholds, link) {
		if (WT_IS_INACTIVE(threshold->worktime)) {
			if (THRESHOLD_IS_ACTIVE(threshold))
				if (set_threshold_inactive(rule, threshold) < 0)
					goto failed;
			if (check_sec > threshold->worktime->active_sec)
				check_sec = threshold->worktime->active_sec;
		} else {
			if (THRESHOLD_IS_INACTIVE(threshold)) {
				if (set_threshold_active(rule, threshold) < 0)
					goto failed;
				/*
				 * threshold->check_sec was updated in
				 * set_threshold_active().
				 */
			} else
				threshold->check_sec =
				    threshold->time_slice->event_sec;
			if (check_sec > threshold->worktime->inactive_sec)
				check_sec = threshold->worktime->inactive_sec;
			if (check_sec > threshold->check_sec)
				check_sec = threshold->check_sec;
		}
		if (rule->check_sec > check_sec)
			rule->check_sec = check_sec;
	}

	return (0);

failed:
	logbt("thresholds_newday");
	return (-1);
}

/*
 * Copy all thresholds from the given threshold list to rule.
 */
int
copy_thresholds(struct rule *rule, const struct thresholds_list *list)
{
	const struct threshold *tsrc;
	struct threshold *tdst;
	unsigned int count;
	int rv;

	count = 0;
	STAILQ_FOREACH(tsrc, list, link) {
		tdst = mzone_alloc(threshold_mzone);
		if (tdst == NULL) {
			xlogmsgx(IPA_LOG_ERR, "copy_thresholds: "
			    "mzone_alloc failed");
			return (-1);
		}

		/* Copy settings from source threshold. */
		*tdst = *tsrc;

		/*
		 * Initialize fields which are pointers to memory,
		 * which cannot be shared.
		 */
		threshold_init_cmds(tdst);
		tdst->rule = rule;
		tdst->wpid.u.threshold = tdst;

		/* Link just allocated threshold to rule. */
		STAILQ_INSERT_TAIL(&rule->thresholds, tdst, link);

		/*
		 * If something goes wrong with memory allocation,
		 * previous settings will allow to deinitialize current rule.
		 */
		rv =
		    cmds_copy(rule, &tdst->below_thr, &tsrc->below_thr) +
		    cmds_copy(rule, &tdst->equal_thr, &tsrc->equal_thr) +
		    cmds_copy(rule, &tdst->above_thr, &tsrc->above_thr) +
		    cmds_copy(rule, &tdst->rc[0], &tsrc->rc[0]) +
		    cmds_copy(rule, &tdst->rc[1], &tsrc->rc[1]);
		if (rv < 0)
			return (-1);
		++count;
	}
	if (RULE_IS_DYNAMIC(rule))
		ndynthresholds += count;
	else
		nstatthresholds += count;
	return (0);
}

/*
 * Release memory used by a list of thresholds, rule_flags determines
 * which part of threshold{} structure to free.
 */
void
free_thresholds(unsigned int rule_flags, struct thresholds_list *thresholds,
    int dyn_flag)
{
	struct threshold *threshold, *threshold_next;
	unsigned int count;

	count = 0;
	STAILQ_FOREACH_SAFE(threshold, thresholds, link, threshold_next) {
		if (rule_flags & RULE_FLAG_FREE_THRESHOLDS) {
			mem_free(threshold->name, m_anon);
			mem_free(threshold->info, m_parser);
		}
		mem_free(threshold->cnt_slice, m_anon);
		mem_free(threshold->cnt_slice_sign, m_anon);
		cmds_free(&threshold->below_thr);
		cmds_free(&threshold->equal_thr);
		cmds_free(&threshold->above_thr);
		cmds_free(&threshold->rc[RC_STARTUP]);
		cmds_free(&threshold->rc[RC_SHUTDOWN]);
		mzone_free(threshold_mzone, threshold);
		++count;
	}
	if (dyn_flag)
		ndynthresholds -= count;
}

void
threshold_init_cmds(struct threshold *threshold)
{
	cmds_init(&threshold->below_thr);
	cmds_init(&threshold->equal_thr);
	cmds_init(&threshold->above_thr);
	cmds_init(&threshold->rc[RC_STARTUP]);
	cmds_init(&threshold->rc[RC_SHUTDOWN]);
}

/*
 * Set default settings and inherit settings from global{} in threshold{}.
 */
void
threshold_inherit(struct threshold *threshold)
{
	if (threshold->time_width == 0) {
		threshold->time_width = global_threshold_time_width;
		threshold->time_slice = global_threshold_time_slice;
	}
	if (threshold->load_thr < 0)
		threshold->load_thr = global_load_threshold;
	if (threshold->thr_type < 0)
		threshold->thr_type = global_threshold_type;
	if (threshold->below_lim == 0) {
		threshold->below_lim = threshold->below_cnt =
		    global_threshold_below_lim;
		threshold->equal_lim = threshold->equal_cnt =
		    global_threshold_equal_lim;
		threshold->above_lim = threshold->above_cnt =
		    global_threshold_above_lim;
	}
	if (threshold->below_thr.sync < 0)
		threshold->below_thr.sync = 0;
	if (threshold->equal_thr.sync < 0)
		threshold->equal_thr.sync = 0;
	if (threshold->above_thr.sync < 0)
		threshold->above_thr.sync = 0;
	if (threshold->rc[RC_STARTUP].sync < 0)
		threshold->rc[RC_STARTUP].sync = 1;
	if (threshold->rc[RC_SHUTDOWN].sync < 0)
		threshold->rc[RC_SHUTDOWN].sync = 1;
}
#endif /* WITH_THRESHOLDS */
